/****************************************************************************
**
** Copyright (C) 1992-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://www.trolltech.com/products/qt/opensource.html
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://www.trolltech.com/products/qt/licensing.html or contact the
** sales department at sales@trolltech.com.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#include <private/qwindowsurface_p.h>
#include <qwidget.h>
#include <private/qbackingstore_p.h>

class QWindowSurfacePrivate
{
public:
    QWindowSurfacePrivate(QWidget *w) : window(w) {}

    QWidget *window;
    QRect geometry;
    QList<QImage*> bufferImages;
};

/*!
    \class QWindowSurface
    \since 4.3
    \preliminary

    \brief The QWindowSurface class provides the drawing area for top-level
    windows.
*/


/*!
    \fn void QWindowSurface::beginPaint(const QRegion &region)

    This function is called before painting onto the surface begins,
    with the \a region in which the painting will occur.

    \sa endPaint(), paintDevice()
*/

/*!
    \fn void QWindowSurface::endPaint(const QRegion &region)

    This function is called after painting onto the surface has ended,
    with the \a region in which the painting was performed.

    \sa beginPaint(), paintDevice()
*/

/*!
    \fn void QWindowSurface::flush(QWidget *widget, const QRegion &region,
                                  const QPoint &offset)

    Flushes the given \a region from the specified \a widget onto the
    screen.

    Note that the \a offset parameter is currently unused.

    \sa painterOffset()
*/

/*!
    \fn QPaintDevice* QWindowSurface::paintDevice()

    Implement this function to return the appropriate paint device.

    \sa painterOffset()
*/

/*!
    Constructs an empty surface for the given top-level \a window.
*/
QWindowSurface::QWindowSurface(QWidget *window)
    : d_ptr(new QWindowSurfacePrivate(window))
{
    if (window)
        window->setWindowSurface(this);
}

/*!
    Destroys this surface.
*/
QWindowSurface::~QWindowSurface()
{
    if (d_ptr->window)
        d_ptr->window->setWindowSurface(0);
    delete d_ptr;
}

/*!
    Returns a pointer to the top-level window associated with this
    surface.

    \sa isValid(), key()
*/
QWidget* QWindowSurface::window() const
{
    return d_ptr->window;
}

/*!
    This function is called before painting onto the surface begins,
    with the \a region in which the painting will occur.

    \sa endPaint()
*/
void QWindowSurface::beginPaint(const QRegion &)
{
}

/*!
    This function is called after painting onto the surface has ended,
    with the \a region in which the painting was performed.

    \sa beginPaint()
*/
void QWindowSurface::endPaint(const QRegion &)
{
    qDeleteAll(d_ptr->bufferImages);
}

/*!
    Sets the currently allocated area to be the given \a rectangle.

    This function is called whenever area covered by the top-level
    window changes.

    \sa geometry()
*/
void QWindowSurface::setGeometry(const QRect &rect)
{
    d_ptr->geometry = rect;
}

/*!
    Returns the currently allocated area on the screen.
*/
QRect QWindowSurface::geometry() const
{
    return d_ptr->geometry;
}

/*!
    Scrolls the given \a region \a dx pixels to the right and \a dy
    downward; both \a dx and \a dy may be negative.

    Returns true if the area was scrolled successfully; false otherwise.
*/
bool QWindowSurface::scroll(const QRegion &area, int dx, int dy)
{
    Q_UNUSED(area);
    Q_UNUSED(dx);
    Q_UNUSED(dy);

    return false;
}

/*!
    Returns a QImage pointer which represents the actual buffer the \a widget
    is drawn into or 0 if this is unavailable.

    You must call beginPaint() before you call this function and the returned
    pointer is only valid until endPaint() is called.
*/
QImage* QWindowSurface::buffer(const QWidget *widget)
{
    if (widget->window() != window())
        return 0;

    QPaintDevice *pdev = paintDevice();
    if (!pdev || pdev->devType() != QInternal::Image)
        return 0;

    const QPoint off = offset(widget);
    QImage *img = static_cast<QImage*>(pdev);
    img = new QImage(img->scanLine(off.y()) + off.x() * img->depth() / 8,
                     widget->width(), widget->height(),
                     img->bytesPerLine(), img->format());
    d_ptr->bufferImages.append(img);

    return img;
}

QPoint QWindowSurface::offset(const QWidget *widget) const
{
    QWidget *window = d_ptr->window;
    QPoint offset = widget->mapTo(window, QPoint());
#ifdef Q_WS_QWS
    offset += window->geometry().topLeft() - window->frameGeometry().topLeft();
#endif
    return offset;
}

