/* Copyright (C) 2003 Nikos Chantziaras.
 *
 * This file is part of the QTads program.  This program is free
 * software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Classes that provide a mechanism for saving and restoring the
 * application's parameters (themes and other preferences).
 */

#ifndef QTADSSETTINGS_H
#define QTADSSETTINGS_H

#include "config.h"

#include <vector>
#include <map>

#include <qapplication.h>
#include <qcolor.h>
#include <qfont.h>


/* A "Theme" is a set of user-preferences that defines a certain "look
 * and feel", like fonts and colors.  Preferences that don't affect the
 * look of the application are not part of a theme.
 */
class QTadsTheme {
  private:
	// The preferences that are stored by a theme.  It should be
	// clear what they do since they have obvious names.
	QColor             fGameBgColor;
	QColor             fGameTextColor;
	QColor             fStatusBgColor;
	QColor             fStatusTextColor;
	QFont              fGameFont;
	QFont              fStatusFont;
	Qt::AlignmentFlags fAlignment;
	int                fLeftMargin;
	int                fRightMargin;
	bool               fDoubleSpace;
	bool               fBoldInput;
	bool               fItalicInput;
	bool               fUnderlinedInput;
	bool               fCurlyQuotes;
	bool               fCurlyApostrophes;
	bool               fDashConversion;

  public:
	QTadsTheme( const QColor& gameBg = Qt::white, const QColor& gameText = Qt::black,
		    const QColor& statusBg = Qt::white, const QColor& statusText = Qt::black,
		    const QFont& gameFnt = QApplication::font(),
		    const QFont& statusFnt = QApplication::font(),
		    const Qt::AlignmentFlags& alignFlags = Qt::AlignLeft, int leftMarg = 5,
                    int rightMarg = 5, bool doubleSpc = false, bool boldInp = false,
		    bool italicInp = true, bool underlinedInp = false,
		    bool curlyQuot = true, bool curlyApostr = true, bool dashConv = true );

	// The methods below provide access to the preferences stored
	// by the theme.  They are pairs of overloaded methods; one for
	// reading the value, one for setting it.  They are one-liners
	// with a single instruction and will be inlined.
	QColor
	gameBgColor() const;

	void
	gameBgColor( const QColor& color );

	QColor
	gameTextColor() const;

	void
	gameTextColor( const QColor& color );

	QColor
	statusBgColor() const;

	void
	statusBgColor( const QColor& color );

	QColor
	statusTextColor() const;

	void
	statusTextColor( const QColor& color );

	QFont
	gameFont() const;

	void
	gameFont( const QFont& font );

	QFont
	statusFont() const;

	void
	statusFont( const QFont& font );

	Qt::AlignmentFlags
	alignment() const;

	void
	alignment( const Qt::AlignmentFlags& flags );

	int
	leftMargin() const;

	void
	leftMargin( int margin );

	int
	rightMargin() const;

	void
	rightMargin( int margin );

	bool
	doubleSpace() const;

	void
	doubleSpace( bool b );

	bool
	boldInput() const;

	void
	boldInput( bool b );

	bool
	italicInput() const;

	void
	italicInput( bool b );

	bool
	underlinedInput() const;

	void
	underlinedInput( bool b );

	bool
	curlyQuotes() const;

	void
	curlyQuotes( bool b );

	bool
	curlyApostrophes() const;

	void
	curlyApostrophes( bool b );

	bool
	dashConversion() const;

	void
	dashConversion( bool b );
};


/* This class manages the themes and also contains the settings that
 * aren't managed by themes.
 */
class QTadsSettings {
  public:
	// Allowed sizes for the scrollback buffer.  Using enums with
	// explicit values provides type-safety without a need to use
	// plain integer constants.
	enum ScrollBufferSize {
		BufferSize1 =   65536, //  64K
		BufferSize2 =  131072, // 128K
		BufferSize3 =  262144, // 256K
		BufferSize4 =  524288, // 512K
		BufferSize5 = 1048576, //   1M
		BufferSize6 = 2097152, //   2M
		BufferSize7 = 4194304, //   4M
		BufferSize8 = 8388608  //   8M
	};

  private:
	ScrollBufferSize fScrollBufferSize;
	bool fImmediateQuit;
	bool fImmediateRestart;

	typedef std::map<QString,QTadsTheme> ThemeMap;
	ThemeMap fThemes;
	QString fCurrentTheme;

	std::vector<QString> fRecentGames;
	int fRecentGamesCount;
	int fRecentGamesCapacity;

  public:
	// The ctor tries to load the settings from disk.  The
	// operation never really fails, since defaults will be used if
	// something goes wrong.
	QTadsSettings();

	// The dtor saves the settings to disk.
	// TODO: What if an error occurs?
	~QTadsSettings();

	// Returns the name of the current theme.
	QString
	nameOfCurrentTheme() const;

	// Returns the currently active theme.
	// This is a convenience method; it's the same as
	// theme(nameOfCurrentTheme()).
	QTadsTheme&
	currentTheme();

	// Changes the active theme.  Returns true on success, false if
	// there is no theme with name `themeName'.
	bool
	currentTheme( const QString& themeName );

	// Returns the theme with name `themeName'.  If there is no
	// such theme, the default theme will be returned.
	QTadsTheme&
	theme( const QString& themeName );

	// Adds a theme and names it `themeName'.  If a theme with the
	// same name already exists, it will be replaced.
	void
	addTheme( const QTadsTheme& theme, const QString& themeName );

	// Removes the theme with name `themeName'.  Returns true on
	// success, false if there was no such theme.  If the theme to
	// be removed is the currently active one, then the default
	// theme will be activated before removing it.  Note that the
	// default theme (with name "Default") can't be removed; false
	// will be returned in this case.
	bool
	removeTheme( const QString& themeName );

	// Returns the names of all the themes.
	std::vector<QString>
	themes() const;

	// Returns the filenames of the most recently played games.
	std::vector<QString>
	recentGames() const;

	// This returns the maximum size of the recently played games
	// list.
	std::vector<QString>::size_type
	recentGamesCapacity() const;

	// Change the capacity of the recently played games list.
	void
	recentGamesCapacity( std::vector<QString>::size_type newCapacity );

	// Adds a filename to the "recent games" list.  If the same
	// entry already exists, it is moved to the top as the most
	// recent one.
	void
	addRecentGame( const QString& filename );

	// The methods below provide access to the user preferences.
	// They are pairs of overloaded methods; one for reading the
	// value, one for setting it.

	// Size of the game window's scrollback buffer.
	ScrollBufferSize
	scrollBufferSize() const;

	void
	scrollBufferSize( ScrollBufferSize size );

	// Specifies how a call to QTadsIO::quit() should be treated.
	// If immediate quit is enabled, the game will be terminated
	// immediately.  Otherwise, a "quit" command will be generated.
	bool
	immediateQuit() const;

	void
	immediateQuit( bool b );

	// Specifies how a call to QTadsIO::restart() should be
	// treated.  If immediate restarting is enabled, the game will
	// be terminated immediately and reopened.  Otherwise, a
	// "restart" command will be generated.
	bool
	immediateRestart() const;

	void
	immediateRestart( bool b );
};


/* --------------------------------------------------------------------
 * QTadsTheme inline implementations.
 */
inline
QColor
QTadsTheme::gameBgColor() const
{
	return this->fGameBgColor;
}


inline
void
QTadsTheme::gameBgColor( const QColor& color )
{
	this->fGameBgColor = color;
}


inline
QColor
QTadsTheme::gameTextColor() const
{
	return this->fGameTextColor;
}


inline
void
QTadsTheme::gameTextColor( const QColor& color )
{
	this->fGameTextColor = color;
}


inline
QColor
QTadsTheme::statusBgColor() const
{
	return this->fStatusBgColor;
}


inline
void
QTadsTheme::statusBgColor( const QColor& color )
{
	this->fStatusBgColor = color;
}


inline
QColor
QTadsTheme::statusTextColor() const
{
	return this->fStatusTextColor;
}


inline
void
QTadsTheme::statusTextColor( const QColor& color )
{
	this->fStatusTextColor = color;
}


inline
QFont
QTadsTheme::gameFont() const
{
	return this->fGameFont;
}


inline
void
QTadsTheme::gameFont( const QFont& font )
{
	this->fGameFont = font;
}


inline
QFont
QTadsTheme::statusFont() const
{
	return this->fStatusFont;
}


inline
void
QTadsTheme::statusFont( const QFont& font )
{
	this->fStatusFont = font;
}


inline
Qt::AlignmentFlags
QTadsTheme::alignment() const
{
	return this->fAlignment;
}


inline
void
QTadsTheme::alignment( const Qt::AlignmentFlags& flags )
{
	this->fAlignment = flags;
}


inline
int
QTadsTheme::leftMargin() const
{
	return this->fLeftMargin;
}


inline
void
QTadsTheme::leftMargin( int margin )
{
	this->fLeftMargin = margin;
}


inline
int
QTadsTheme::rightMargin() const
{
	return this->fRightMargin;
}


inline
void
QTadsTheme::rightMargin( int margin )
{
	this->fRightMargin = margin;
}


inline
bool
QTadsTheme::doubleSpace() const
{
	return this->fDoubleSpace;
}


inline
void
QTadsTheme::doubleSpace( bool b )
{
	this->fDoubleSpace = b;
}


inline
bool
QTadsTheme::boldInput() const
{
	return this->fBoldInput;
}


inline
void
QTadsTheme::boldInput( bool b )
{
	this->fBoldInput = b;
}


inline
bool
QTadsTheme::italicInput() const
{
	return this->fItalicInput;
}


inline
void
QTadsTheme::italicInput( bool b )
{
	this->fItalicInput = b;
}


inline
bool
QTadsTheme::underlinedInput() const
{
	return this->fUnderlinedInput;
}


inline
void
QTadsTheme::underlinedInput( bool b )
{
	this->fUnderlinedInput = b;
}


inline
bool
QTadsTheme::curlyQuotes() const
{
	return this->fCurlyQuotes;
}


inline
void
QTadsTheme::curlyQuotes( bool b )
{
	this->fCurlyQuotes = b;
}


inline
bool
QTadsTheme::curlyApostrophes() const
{
	return this->fCurlyApostrophes;
}


inline
void
QTadsTheme::curlyApostrophes( bool b )
{
	this->fCurlyApostrophes = b;
}


inline
bool
QTadsTheme::dashConversion() const
{
	return this->fDashConversion;
}


inline
void
QTadsTheme::dashConversion( bool b )
{
	this->fDashConversion = b;
}


/* --------------------------------------------------------------------
 * QTadsSettings inline implementations.
 */
inline
QString
QTadsSettings::nameOfCurrentTheme() const
{
	return this->fCurrentTheme;
}


inline
QTadsTheme&
QTadsSettings::currentTheme()
{
	return this->fThemes[this->fCurrentTheme];
}


inline
void
QTadsSettings::addTheme( const QTadsTheme& theme, const QString& themeName )
{
	this->fThemes[themeName] = theme;
}


inline
std::vector<QString>::size_type
QTadsSettings::recentGamesCapacity() const
{
	return this->fRecentGamesCapacity;
}


inline
void
QTadsSettings::recentGamesCapacity( std::vector<QString>::size_type newCapacity )
{
	this->fRecentGames.resize(newCapacity);
	this->fRecentGamesCapacity = newCapacity;
}


inline
void
QTadsSettings::scrollBufferSize( QTadsSettings::ScrollBufferSize size )
{
	this->fScrollBufferSize = size;
}


inline
bool
QTadsSettings::immediateQuit() const
{
	return this->fImmediateQuit;
}


inline
void
QTadsSettings::immediateQuit( bool b )
{
	this->fImmediateQuit = b;
}


inline
bool
QTadsSettings::immediateRestart() const
{
	return this->fImmediateRestart;
}


inline
void
QTadsSettings::immediateRestart( bool b )
{
	this->fImmediateRestart = b;
}


#endif // QTADSSETTINGS_H
