/************************************************************************
**qprocess_unix.cpp  23 Jan 2001
**splitted qprocess_unix.cpp in qprocess_unix.h and qprocess_unix.cpp
**Michael Herder crapsite@gmx.net
************************************************************************/
/****************************************************************************
** $Id: qprocess_unix.cpp,v 1.4 2002/11/27 12:59:17 dermichel Exp $
**
** Implementation of QProcess class for Unix
**
** Created : 20000905
**
** Copyright (C) 1992-2000 Trolltech AS.  All rights reserved.
**
** This file is part of the kernel module of the Qt GUI Toolkit.
**
** This file may be distributed under the terms of the Q Public License
** as defined by Trolltech AS of Norway and appearing in the file
** LICENSE.QPL included in the packaging of this file.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** Licensees holding valid Qt Enterprise Edition or Qt Professional Edition
** licenses may use this file in accordance with the Qt Commercial License
** Agreement provided with the Software.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/pricing.html or email sales@trolltech.com for
**   information about Qt Commercial License Agreements.
** See http://www.trolltech.com/qpl/ for QPL licensing information.
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

#include "qprocess.h"
#include "qprocess_unix.h"

QCleanupHandler<QProcessBackportManager> qprocess_cleanup_procmanager;

QProcessBackportManager::QProcessBackportManager()
{
    procList = new QList<QProcBackport>;
    procList->setAutoDelete( TRUE );

    // The SIGCHLD handler writes to a socket to tell the manager that
    // something happened. This is done to get the processing in sync with the
    // event reporting.
    if ( ::socketpair( AF_UNIX, SOCK_STREAM, 0, sigchldFd ) ) {
	sigchldFd[0] = 0;
	sigchldFd[1] = 0;
    } else {
	QSocketNotifier *sn = new QSocketNotifier( sigchldFd[1],
		QSocketNotifier::Read, this );
	connect( sn, SIGNAL(activated(int)),
		this, SLOT(sigchldHnd(int)) );
	sn->setEnabled( TRUE );
    }

    // install a SIGCHLD handler and ignore SIGPIPE
    struct sigaction act;

    act.sa_handler = qt_C_sigchldHnd;
    sigemptyset( &(act.sa_mask) );
    sigaddset( &(act.sa_mask), SIGCHLD );
    act.sa_flags = SA_NOCLDSTOP;
#if defined(SA_RESTART)
    act.sa_flags |= SA_RESTART;
#endif
    if ( sigaction( SIGCHLD, &act, &oldactChld ) != 0 )
	qWarning( "Error installing SIGCHLD handler" );

    act.sa_handler = qt_C_sigign;
    sigemptyset( &(act.sa_mask) );
    sigaddset( &(act.sa_mask), SIGPIPE );
    act.sa_flags = 0;
    if ( sigaction( SIGPIPE, &act, &oldactPipe ) != 0 )
	qWarning( "Error installing SIGPIPE handler" );

    qprocess_cleanup_procmanager.add( this );
}

QProcessBackportManager::~QProcessBackportManager()
{
    delete procList;

    if ( sigchldFd[0] != 0 )
	::close( sigchldFd[0] );
    if ( sigchldFd[1] != 0 )
	::close( sigchldFd[1] );

    // restore SIGCHLD handler
    if ( sigaction( SIGCHLD, &oldactChld, 0 ) != 0 )
	qWarning( "Error restoring SIGCHLD handler" );

    if ( sigaction( SIGPIPE, &oldactPipe, 0 ) != 0 )
	qWarning( "Error restoring SIGPIPE handler" );
}

void QProcessBackportManager::append( QProcBackport *p )
{
    procList->append( p );
}

void QProcessBackportManager::remove( QProcBackport *p )
{
    procList->remove( p );
    if ( procList->count() == 0 ) {
	QTimer::singleShot( 0, this, SLOT(removeMe()) );
    }
}

void QProcessBackportManager::removeMe()
{
    QProcessBackportPrivate::procManager = 0;
    qprocess_cleanup_procmanager.remove( this );
    delete this;
}

void QProcessBackportManager::sigchldHnd( int fd )
{
    char tmp;
    ::read( fd, &tmp, sizeof(tmp) );
    QProcBackport *proc;
    QProcessBackport *process;
    bool removeProc;
    proc = procList->first();
    while ( proc != 0 ) {
	removeProc = FALSE;
	process = proc->process;
	if ( process != 0 ) {
	    if ( !process->isRunning() ) {
		// read pending data
		process->socketRead( process->d->socketStdout[0] );
		process->socketRead( process->d->socketStderr[0] );

		if ( process->notifyOnExit )
		    emit process->processExited();

		removeProc = TRUE;
	    }
	} else {
	    int status;
	    if ( ::waitpid( proc->pid, &status, WNOHANG ) == proc->pid ) {
		removeProc = TRUE;
	    }
	}
	if ( removeProc ) {
	    QProcBackport *oldproc = proc;
	    proc = procList->next();
	    remove( oldproc );
	} else {
	    proc = procList->next();
	}
    }
}

QProcessBackportManager *QProcessBackportPrivate::procManager = 0;

QProcessBackportPrivate::QProcessBackportPrivate()
{
    stdinBufRead = 0;

    notifierStdin = 0;
    notifierStdout = 0;
    notifierStderr = 0;

    socketStdin[0] = 0;
    socketStdin[1] = 0;
    socketStdout[0] = 0;
    socketStdout[1] = 0;
    socketStderr[0] = 0;
    socketStderr[1] = 0;

    exitValuesCalculated = FALSE;

    proc = 0;
}

QProcessBackportPrivate::~QProcessBackportPrivate()
{
    if ( proc != 0 )
	proc->process = 0;

    while ( !stdinBuf.isEmpty() ) {
	delete stdinBuf.dequeue();
    }
    if ( notifierStdin ) {
	delete notifierStdin;
    }
    if ( notifierStdout ) {
	delete notifierStdout;
    }
    if ( notifierStderr ) {
	delete notifierStderr;
    }
    if( socketStdin[1] != 0 )
	::close( socketStdin[1] );
    if( socketStdout[0] != 0 )
	::close( socketStdout[0] );
    if( socketStderr[0] != 0 )
	::close( socketStderr[0] );
}

void QProcessBackportPrivate::closeOpenSocketsForChild()
{
    ::close( socketStdin[1] );
    ::close( socketStdout[0] );
    ::close( socketStderr[0] );

    if ( procManager != 0 ) {
	if ( procManager->sigchldFd[0] != 0 )
	    ::close( procManager->sigchldFd[0] );
	if ( procManager->sigchldFd[1] != 0 )
	    ::close( procManager->sigchldFd[1] );

	// delete also the sockets from other QProcess instances
	QProcBackport *proc;
	QProcessBackport *process;
	for ( proc=procManager->procList->first(); proc!=0; proc=procManager->procList->next() ) {
	    process = proc->process;
	    if ( process != 0 ) {
		::close( process->d->socketStdin[1] );
		::close( process->d->socketStdout[0] );
		::close( process->d->socketStderr[0] );
	    }
	}
    }
}

void QProcessBackportPrivate::newProc( pid_t pid, QProcessBackport *process )
{
    proc = new QProcBackport( pid, process );
    if ( procManager == 0 ) {
	procManager = new QProcessBackportManager;
    }
    // the QProcessManager takes care of deleting the QProc instances
    procManager->append( proc );
}

//#if defined(SIGNAL_HACK)
//void qt_C_sigchldHnd()
//#else
void qt_C_sigchldHnd( int )
//#endif
{
    if ( QProcessBackportPrivate::procManager == 0 )
	return;
    if ( QProcessBackportPrivate::procManager->sigchldFd[0] == 0 )
	return;

    char a = 1;
    ::write( QProcessBackportPrivate::procManager->sigchldFd[0], &a, sizeof(a) );
}

void QProcessBackport::init()
{
    d = new QProcessBackportPrivate();
    exitStat = 0;
    exitNormal = FALSE;
}

void QProcessBackport::reset()
{
    delete d;
    d = new QProcessBackportPrivate();
    exitStat = 0;
    exitNormal = FALSE;
    bufStdout.resize( 0 );
    bufStderr.resize( 0 );
}

QProcessBackport::~QProcessBackport()
{
    delete d;
}

bool QProcessBackport::start()
{
    reset();

    // open sockets for piping
    if ( ::socketpair( AF_UNIX, SOCK_STREAM, 0, d->socketStdin ) ) {
	return FALSE;
    }
    if ( ::socketpair( AF_UNIX, SOCK_STREAM, 0, d->socketStdout ) ) {
	return FALSE;
    }
    if ( ::socketpair( AF_UNIX, SOCK_STREAM, 0, d->socketStderr ) ) {
	return FALSE;
    }

    // the following pipe is only used to determine if the process could be
    // started
    int fd[2];
    if ( pipe( fd ) < 0 ) {
	// non critical error, go on
	fd[0] = 0;
	fd[1] = 0;
    }

    // construct the arguments for exec
    QCString *arglistQ = new QCString[ _arguments.count() + 1 ];
    const char** arglist = new const char*[ _arguments.count() + 1 ];
    int i = 0;
    for ( QStringList::Iterator it = _arguments.begin(); it != _arguments.end(); ++it ) {
	arglistQ[i] = (*it).local8Bit();
	arglist[i] = arglistQ[i];
#if defined(QT_QPROCESS_DEBUG)
	qDebug( "QProcess::start(): arg %d = %s", i, arglist[i] );
#endif
	i++;
    }
    arglist[i] = 0;

    // fork and exec
    QApplication::flushX();
    pid_t pid = fork();
    if ( pid == 0 ) {
	// child
	d->closeOpenSocketsForChild();
	::dup2( d->socketStdin[0], STDIN_FILENO );
	::dup2( d->socketStdout[1], STDOUT_FILENO );
	::dup2( d->socketStderr[1], STDERR_FILENO );
	::chdir( workingDir.absPath().latin1() );
	if ( fd[0] )
	    ::close( fd[0] );
	if ( fd[1] )
	    ::fcntl( fd[1], F_SETFD, FD_CLOEXEC ); // close on exec shows sucess
	::execvp( arglist[0], (char*const*)arglist ); // ### cast not nice
	if ( fd[1] ) {
	    char buf = 0;
	    ::write( fd[1], &buf, 1 );
	    ::close( fd[1] );
	}
	::exit( -1 );
    } else if ( pid == -1 ) {
	// error forking
	goto error;
    }
    // test if exec was successful
    if ( fd[1] )
	close( fd[1] );
    if ( fd[0] ) {
	char buf;
	while ( TRUE ) {
	    int n = ::read( fd[0], &buf, 1 );
	    if ( n==1 ) {
		// socket was not closed => error
		goto error;
	    } else if ( n==-1 ) {
		if ( errno==EAGAIN || errno==EINTR )
		    // try it again
		    continue;
	    }
	    break;
	}
    }

    d->newProc( pid, this );

    ::close( d->socketStdin[0] );
    ::close( d->socketStdout[1] );
    ::close( d->socketStderr[1] );

    // setup notifiers for the sockets
    d->notifierStdin = new QSocketNotifier( d->socketStdin[1],
	    QSocketNotifier::Write );
    d->notifierStdout = new QSocketNotifier( d->socketStdout[0],
	    QSocketNotifier::Read );
    d->notifierStderr = new QSocketNotifier( d->socketStderr[0],
	    QSocketNotifier::Read );
    connect( d->notifierStdin, SIGNAL(activated(int)),
	    this, SLOT(socketWrite(int)) );
    connect( d->notifierStdout, SIGNAL(activated(int)),
	    this, SLOT(socketRead(int)) );
    connect( d->notifierStderr, SIGNAL(activated(int)),
	    this, SLOT(socketRead(int)) );
    if ( !d->stdinBuf.isEmpty() ) {
	d->notifierStdin->setEnabled( TRUE );
    }
    if ( ioRedirection ) {
	d->notifierStdout->setEnabled( TRUE );
	d->notifierStderr->setEnabled( TRUE );
    }

    // cleanup and return
    delete[] arglistQ;
    delete[] arglist;
    return TRUE;

error:
    ::close( d->socketStdin[1] );
    ::close( d->socketStdout[0] );
    ::close( d->socketStderr[0] );
    ::close( d->socketStdin[0] );
    ::close( d->socketStdout[1] );
    ::close( d->socketStderr[1] );
    d->socketStdin[0] = 0;
    d->socketStdin[1] = 0;
    d->socketStdout[0] = 0;
    d->socketStdout[1] = 0;
    d->socketStderr[0] = 0;
    d->socketStderr[1] = 0;
    ::close( fd[0] );
    ::close( fd[1] );
    delete[] arglistQ;
    delete[] arglist;
    return FALSE;
}

void QProcessBackport::hangUp() const
{
    if ( d->proc != 0 )
	::kill( d->proc->pid, SIGHUP );
}

void QProcessBackport::kill() const
{
    if ( d->proc != 0 )
	::kill( d->proc->pid, SIGKILL );
}

bool QProcessBackport::isRunning() const
{
    if ( d->exitValuesCalculated ) {
	return FALSE;
    }
    if ( d->proc == 0 )
	return FALSE;
    int status;
    if ( ::waitpid( d->proc->pid, &status, WNOHANG ) == d->proc->pid )
    {
	// compute the exit values
	QProcessBackport *that = (QProcessBackport*)this; // mutable
	that->exitNormal = WIFEXITED( status ) != 0;
	if ( exitNormal ) {
	    that->exitStat = WEXITSTATUS( status );
	}
	d->exitValuesCalculated = TRUE;
	return FALSE;
    }
    return TRUE;
}

void QProcessBackport::writeToStdin( const QByteArray& buf )
{
    d->stdinBuf.enqueue( new QByteArray(buf) );
    if ( d->notifierStdin != 0 )
        d->notifierStdin->setEnabled( TRUE );
}

void QProcessBackport::closeStdin()
{
    if ( d->socketStdin[1] !=0 ) {
	while ( !d->stdinBuf.isEmpty() ) {
	    delete d->stdinBuf.dequeue();
	}
	if ( d->notifierStdin ) {
	    delete d->notifierStdin;
	    d->notifierStdin = 0;
	}
	if ( ::close( d->socketStdin[1] ) != 0 ) {
	    qWarning( "Could not close stdin of child process" );
	}
	d->socketStdin[1] = 0;
    }
}

void QProcessBackport::socketRead( int fd )
{
    if ( fd == 0 )
	return;
    const int bufsize = 4096;
    QByteArray buffer;
    uint oldSize;
    int n;
    if ( fd == d->socketStdout[0] ) {
	buffer = bufStdout;
    } else {
	buffer = bufStderr;
    }

    // read data
    oldSize = buffer.size();
    buffer.resize( oldSize + 4096 );
    n = ::read( fd, buffer.data()+oldSize, bufsize );
    if ( n > 0 )
	buffer.resize( oldSize + n );
    // eof or error?
    if ( n == 0 || n == -1 ) {
	if ( fd == d->socketStdout[0] ) {
	    d->notifierStdout->setEnabled( FALSE );
	    delete d->notifierStdout;
	    d->notifierStdout = 0;
	    ::close( d->socketStdout[0] );
	    d->socketStdout[0] = 0;
	    return;
	} else {
	    d->notifierStderr->setEnabled( FALSE );
	    delete d->notifierStderr;
	    d->notifierStderr = 0;
	    ::close( d->socketStderr[0] );
	    d->socketStderr[0] = 0;
	    return;
	}
    }
    // read all data that is available
    while ( n == bufsize ) {
	oldSize = buffer.size();
	buffer.resize( oldSize + 4096 );
	n = ::read( fd, buffer.data()+oldSize, bufsize );
	if ( n > 0 )
	    buffer.resize( oldSize + n );
    }

    if ( fd == d->socketStdout[0] ) {
	emit readyReadStdout();
    } else {
	emit readyReadStderr();
    }
}

void QProcessBackport::socketWrite( int fd )
{
    if ( fd != d->socketStdin[1] || d->socketStdin[1] == 0 )
	return;
    if ( d->stdinBuf.isEmpty() ) {
	d->notifierStdin->setEnabled( FALSE );
	return;
    }
    ssize_t ret = ::write( fd,
	    d->stdinBuf.head()->data() + d->stdinBufRead,
	    d->stdinBuf.head()->size() - d->stdinBufRead );
    if ( ret > 0 )
	d->stdinBufRead += ret;
    if ( d->stdinBufRead == (ssize_t)d->stdinBuf.head()->size() ) {
	d->stdinBufRead = 0;
	delete d->stdinBuf.dequeue();
	if ( wroteToStdinConnected && d->stdinBuf.isEmpty() )
	    emit wroteToStdin();
	socketWrite( fd );
    }
}

void QProcessBackport::timeout()
{
}

void QProcessBackport::setIoRedirection( bool value )
{
    ioRedirection = value;
    if ( ioRedirection ) {
	if ( d->notifierStdout )
	    d->notifierStdout->setEnabled( TRUE );
	if ( d->notifierStderr )
	    d->notifierStderr->setEnabled( TRUE );
    } else {
	if ( d->notifierStdout )
	    d->notifierStdout->setEnabled( FALSE );
	if ( d->notifierStderr )
	    d->notifierStderr->setEnabled( FALSE );
    }
}

void QProcessBackport::setNotifyOnExit( bool value )
{
    notifyOnExit = value;
}

void QProcessBackport::setWroteStdinConnected( bool value )
{
    wroteToStdinConnected = value;
}

