/*
 * QuteCom, a voice over Internet phone
 * Copyright (C) 2010 Mbdsys
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "QtToolBar.h"

#include "ui_QuteComWindow.h"

#include <cutil/global.h>

#include "QtQuteCom.h"
#include "QtAbout.h"
#include "QtNoQuteComAlert.h"
#include "profile/QtProfileDetails.h"
#include "profile/QtUserProfileHandler.h"
#include "config/QtQuteComConfigDialog.h"
#include "webservices/sms/QtSms.h"
#include "phonecall/QtContactCallListWidget.h"
#include "phonecall/QtPhoneCall.h"
#include "contactlist/QtContactList.h"
#include "filetransfer/QtFileTransfer.h"
#include "filetransfer/QtFileTransferWidget.h"
#include "conference/QtConferenceCallWidget.h"

#include "webdirectory/QtWebDirectory.h"

#include <control/CQuteCom.h>
#include <control/history/CHistory.h>
#include <control/phonecall/CPhoneCall.h>
#include <control/phoneline/CPhoneLine.h>
#include <control/profile/CUserProfile.h>
#include <control/profile/CUserProfileHandler.h>

#include <model/contactlist/ContactProfile.h>
#include <model/phonecall/ConferenceCall.h>
#include <model/account/qutecom/QuteComAccount.h>
#include <model/profile/UserProfile.h>
#include <model/history/History.h>
#include <model/config/ConfigManager.h>
#include <model/config/Config.h>
#include <model/config/EnumToolBarMode.h>

#include <util/Logger.h>

#include <qtutil/WidgetBackgroundImage.h>
#include <qtutil/SafeConnect.h>
#include <qtutil/Object.h>
#include <qtutil/WidgetFactory.h>

#include <QtGui/QMessageBox>
//#include <QtGui/QToolButton>

static const int TOOLBAR_BUTTON_MINIMUM_WIDTH = 60;

void QtToolBar::connectOrHide(bool condition, QAction* action, QObject* receiver, const char* slot) {
	if (condition) {
		SAFE_CONNECT_RECEIVER(action, SIGNAL(triggered()), receiver, slot);
	} else {
#if QT_VERSION >= 0x040200
		Q_FOREACH(QWidget* widget, action->associatedWidgets()) {
			widget->removeAction(action);
		}
#else
		Q_FOREACH(QMenu* menu, _ui->menuBar->findChildren<QMenu*>()) {
			menu->removeAction(action);
		}
		_ui->toolBar->removeAction(action);
#endif
	}
}


void QtToolBar::connectOrHide(WsUrl::Page page, QAction* action, QObject* receiver, const char* slot) {
	connectOrHide(WsUrl::hasPage(page), action, receiver, slot);
}


QtToolBar::QtToolBar(QtQuteCom & qtQuteCom, Ui::QuteComWindow * qtQuteComUi, QWidget * parent)
	: QObject(parent),
	_qtQuteCom(qtQuteCom),
	_cQuteCom(_qtQuteCom.getCQuteCom()) {

	_ui = qtQuteComUi;

	Config & config = ConfigManager::getInstance().getCurrentConfig();
	bool canSendSms = config.getSmsFeatureEnabled();
	bool canDoConference = config.getAudioConferenceFeatureEnabled();

	SAFE_CONNECT(&_qtQuteCom, SIGNAL(userProfileDeleted()), SLOT(userProfileDeleted()));

	// actions initialization
	_userIsLogged = false;
	updateActionsAccordingToUserStatus();
	////

	//menuQuteCom
	connectOrHide(WsUrl::Account, _ui->actionShowQuteComAccount, this, SLOT(showQuteComAccount()) );
	SAFE_CONNECT(_ui->actionEditMyProfile, SIGNAL(triggered()), SLOT(editMyProfile()));
	SAFE_CONNECT(_ui->actionLogOff, SIGNAL(triggered()), SLOT(logOff()));
	connectOrHide(WsUrl::CallOut, _ui->actionQuteComServices, this, SLOT(showQuteComServices()) );
	SAFE_CONNECT_RECEIVER(_ui->actionClose, SIGNAL(triggered()), &_qtQuteCom, SLOT(closeWindow()));
	SAFE_CONNECT_RECEIVER(_ui->actionQuit, SIGNAL(triggered()), &_qtQuteCom, SLOT(prepareToExitApplication()));

	//menuContacts
	SAFE_CONNECT(_ui->actionAddContact, SIGNAL(triggered()), SLOT(addContact()));
	connectOrHide(WsUrl::Directory, _ui->actionSearchQuteComContact, this, SLOT(searchQuteComContact()) );
	SAFE_CONNECT(_ui->actionShowHideOfflineContacts, SIGNAL(triggered()), SLOT(showHideOfflineContacts()));
	SAFE_CONNECT(_ui->actionShowHideContactGroups, SIGNAL(triggered()), SLOT(showHideContactGroups()));

	//menuActions
	connectOrHide(canDoConference, _ui->actionCreateConferenceCall, this, SLOT(createConferenceCall()));
	connectOrHide(canSendSms, _ui->actionSendSms, this, SLOT(sendSms()));
	SAFE_CONNECT(_ui->actionAcceptCall, SIGNAL(triggered()), SLOT(acceptCall()));
	SAFE_CONNECT(_ui->actionHangUpCall, SIGNAL(triggered()), SLOT(hangUpCall()));
	SAFE_CONNECT(_ui->actionHoldResumeCall, SIGNAL(triggered()), SLOT(holdResumeCall()));

	//menuHelp
	connectOrHide(WsUrl::Forum, _ui->actionShowQuteComForum, this, SLOT(showQuteComForum()));
	connectOrHide(WsUrl::FAQ, _ui->actionQuteComFAQ, this, SLOT(showQuteComFAQ()));
	SAFE_CONNECT(_ui->actionShowAbout, SIGNAL(triggered()), SLOT(showAbout()));

	//menuTools
	SAFE_CONNECT(_ui->actionShowConfig, SIGNAL(triggered()), SLOT(showConfig()));
	SAFE_CONNECT(_ui->actionShowFileTransfer, SIGNAL(triggered()), SLOT(showFileTransferWindow()));
	SAFE_CONNECT(_ui->actionToggleVideo, SIGNAL(triggered()), SLOT(toggleVideo()));

	//menuClearHistory
	SAFE_CONNECT(_ui->actionClearOutgoingCalls, SIGNAL(triggered()), SLOT(clearHistoryOutgoingCalls()));
	SAFE_CONNECT(_ui->actionClearIncomingCalls, SIGNAL(triggered()), SLOT(clearHistoryIncomingCalls()));
	SAFE_CONNECT(_ui->actionClearMissedCalls, SIGNAL(triggered()), SLOT(clearHistoryMissedCalls()));
	SAFE_CONNECT(_ui->actionClearRejectedCalls, SIGNAL(triggered()), SLOT(clearHistoryRejectedCalls()));
	SAFE_CONNECT(_ui->actionClearChatSessions, SIGNAL(triggered()), SLOT(clearHistoryChatSessions()));
	connectOrHide(canSendSms, _ui->actionClearSms, this, SLOT(clearHistorySms()));
	SAFE_CONNECT(_ui->actionClearAll, SIGNAL(triggered()), SLOT(clearHistoryAll()));

	//Open chat window button
	_ui->actionOpenChatWindow->setEnabled(false);
	SAFE_CONNECT(_ui->actionOpenChatWindow, SIGNAL(triggered()), SLOT(showChatWindow()));

	//Get notified of config changes
	config.valueChangedEvent += boost::bind(&QtToolBar::configChangedEventHandler, this, _1);
	SAFE_CONNECT(this, SIGNAL(configChanged(QString)), SLOT(slotConfigChanged(QString)) );

	// Update toolbar mode
	updateToolBar();
	
	//a menu is about to be displayed
	SAFE_CONNECT(_ui->menuActions, SIGNAL(aboutToShow()), SLOT(updateMenuActions()));

#ifndef CUSTOM_ACCOUNT
	_ui->actionSearchQuteComContact->setVisible(false);
#endif

#ifdef DISABLE_SMS
	_ui->actionClearSms->setVisible(false);
	_ui->actionSendSms->setVisible(false);
#endif

#ifdef DISABLE_FILETRANSFER
 	_ui->actionShowFileTransfer->setVisible(false);
#endif

}

QtToolBar::~QtToolBar() {
}

QWidget * QtToolBar::getWidget() const {
	return _qtQuteCom.getWidget();
}

void QtToolBar::showQuteComAccount() {
	if (_cQuteCom.getCUserProfileHandler().getCUserProfile()) {
		//WARNING should not be called when no UserProfile set
		WsUrl::showQuteComAccount();
	}
}

void QtToolBar::editMyProfile() {
	if (_cQuteCom.getCUserProfileHandler().getCUserProfile()) {
		QWidget * parent = qobject_cast<QWidget *>(sender()->parent());

		//FIXME this method should not be called if no UserProfile has been set
		QtProfileDetails qtProfileDetails(*_cQuteCom.getCUserProfileHandler().getCUserProfile(),
			_cQuteCom.getCUserProfileHandler().getCUserProfile()->getUserProfile(),
			parent, tr("Edit My Profile"));

		//TODO UserProfile must be updated if QtProfileDetails was accepted
		qtProfileDetails.show();
	}
}

void QtToolBar::addContact() {
	if (_cQuteCom.getCUserProfileHandler().getCUserProfile()) {
		QWidget * parent = qobject_cast<QWidget *>(sender()->parent());

		//FIXME this method should not be called if no UserProfile has been set
		ContactProfile contactProfile;
		QtProfileDetails qtProfileDetails(*_cQuteCom.getCUserProfileHandler().getCUserProfile(),
			contactProfile, parent, tr("Add a Contact"));
		if (qtProfileDetails.show()) {
			_cQuteCom.getCUserProfileHandler().getCUserProfile()->getCContactList().addContact(contactProfile);
		}
	}
}

void QtToolBar::showConfig() {
	QtQuteComConfigDialog::showInstance(_cQuteCom);
}

void QtToolBar::updateShowHideOfflineContactsAction() {
	Config & config = ConfigManager::getInstance().getCurrentConfig();
	QString text;
	if (config.getShowOfflineContacts()) {
		text = tr("Hide Unavailable Contacts");
	} else {
		text = tr("Show Unavailable Contacts");
	}

	_ui->actionShowHideOfflineContacts->setText(text);
}

void QtToolBar::updateShowHideContactGroupsAction() {
	Config & config = ConfigManager::getInstance().getCurrentConfig();
	QString text;
	if (config.getShowGroups()) {
		text = tr("Hide Groups");
	} else {
		text = tr("Show Groups");
	}

	_ui->actionShowHideContactGroups->setText(text);
}

void QtToolBar::updateToggleVideoAction() {
	Config & config = ConfigManager::getInstance().getCurrentConfig();

	bool videoEnabled = config.getVideoEnable();
	QString text;
	if (videoEnabled) {
		text = tr("Disable &Video Calls");
	} else {
		text = tr("Enable &Video Calls");
	}

	_ui->actionToggleVideo->setText(text);
	_ui->actionToggleVideo->setEnabled(!config.getVideoWebcamDevice().empty());
}

void QtToolBar::showQuteComForum() {
	WsUrl::showQuteComForum();
}

void QtToolBar::showQuteComServices() {
	WsUrl::showQuteComCallOut();
}

void QtToolBar::searchQuteComContact() {

	QtWebDirectory * qtWebDirectory = _qtQuteCom.getQtWebDirectory();
	if (qtWebDirectory) {
		qtWebDirectory->raise();
	}
}

void QtToolBar::showAbout() {
	QWidget * parent = qobject_cast<QWidget *>(sender()->parent());
	static QtAbout * qtAbout = new QtAbout(parent);
	qtAbout->getWidget()->show();
}

void QtToolBar::sendSms() {

	CUserProfile * cuserprofile = _cQuteCom.getCUserProfileHandler().getCUserProfile();
	if (cuserprofile) {
		if (cuserprofile->getUserProfile().hasQuteComAccount()) {
			QtSms * qtSms = _qtQuteCom.getQtSms();
			if (qtSms) {
				QWidget * parent = qobject_cast<QWidget *>(sender()->parent());
				qtSms->getWidget()->setParent(parent, Qt::Dialog);
				qtSms->getWidget()->show();
			}
			return;
		} else {
			QtNoQuteComAlert noQuteComAlert(NULL, _qtQuteCom);
			noQuteComAlert.getQDialog()->exec();
		}
	}
}

void QtToolBar::createConferenceCall() {

	CUserProfile * cUserProfile = _cQuteCom.getCUserProfileHandler().getCUserProfile();
	if (cUserProfile) {

		IPhoneLine * phoneLine = cUserProfile->getUserProfile().getActivePhoneLine();
		if (phoneLine) {
			if (phoneLine->getActivePhoneCall()) {
				QMessageBox::information(
					_qtQuteCom.getWidget(),
					tr("@product@ - Proceed Conference"), 
					tr("<b>Conference can't be started while there are active phone calls.</b><br><br>"
					"Finish all phone calls and try again.")
					);
			} else {
				QtConferenceCallWidget conferenceDialog(_qtQuteCom.getWidget(), _cQuteCom, phoneLine);
				conferenceDialog.exec();
			}
		}
	}
}

void QtToolBar::showQuteComFAQ() {
	WsUrl::showQuteComFAQ();
}

void QtToolBar::showHideOfflineContacts() {
	QtContactList * qtContactList = _qtQuteCom.getQtContactList();
	if (qtContactList) {
		Config & config = ConfigManager::getInstance().getCurrentConfig();
		config.set(Config::GENERAL_SHOW_OFFLINE_CONTACTS_KEY, !config.getShowOfflineContacts());
		qtContactList->updatePresentation();

		updateShowHideOfflineContactsAction();
	}
}

void QtToolBar::showHideContactGroups() {
	QtContactList * qtContactList = _qtQuteCom.getQtContactList();
	if (qtContactList) {
		Config & config = ConfigManager::getInstance().getCurrentConfig();
		config.set(Config::GENERAL_SHOW_GROUPS_KEY, !config.getShowGroups());
		qtContactList->updatePresentation();

		updateShowHideContactGroupsAction();
	}
}

void QtToolBar::toggleVideo() {
	Config & config = ConfigManager::getInstance().getCurrentConfig();
	config.set(Config::VIDEO_ENABLE_KEY, !config.getVideoEnable());
}

void QtToolBar::clearHistoryOutgoingCalls() {
	if (_cQuteCom.getCUserProfileHandler().getCUserProfile() &&
		_cQuteCom.getCUserProfileHandler().getCUserProfile()->getCHistory()) {
		_cQuteCom.getCUserProfileHandler().getCUserProfile()->getCHistory()->clear(HistoryMemento::OutgoingCall);
	}
}

void QtToolBar::clearHistoryIncomingCalls() {
	if (_cQuteCom.getCUserProfileHandler().getCUserProfile() &&
		_cQuteCom.getCUserProfileHandler().getCUserProfile()->getCHistory()) {
		_cQuteCom.getCUserProfileHandler().getCUserProfile()->getCHistory()->clear(HistoryMemento::IncomingCall);
	}
}

void QtToolBar::clearHistoryMissedCalls() {
	if (_cQuteCom.getCUserProfileHandler().getCUserProfile() &&
		_cQuteCom.getCUserProfileHandler().getCUserProfile()->getCHistory()) {
		_cQuteCom.getCUserProfileHandler().getCUserProfile()->getCHistory()->clear(HistoryMemento::MissedCall);
	}
}

void QtToolBar::clearHistoryRejectedCalls() {
	if (_cQuteCom.getCUserProfileHandler().getCUserProfile() &&
		_cQuteCom.getCUserProfileHandler().getCUserProfile()->getCHistory()) {
		_cQuteCom.getCUserProfileHandler().getCUserProfile()->getCHistory()->clear(HistoryMemento::RejectedCall);
	}
}

void QtToolBar::clearHistoryChatSessions() {
	if (_cQuteCom.getCUserProfileHandler().getCUserProfile() &&
		_cQuteCom.getCUserProfileHandler().getCUserProfile()->getCHistory()) {
		_cQuteCom.getCUserProfileHandler().getCUserProfile()->getCHistory()->clear(HistoryMemento::ChatSession);
	}
}

void QtToolBar::clearHistorySms() {
	if (_cQuteCom.getCUserProfileHandler().getCUserProfile() &&
		_cQuteCom.getCUserProfileHandler().getCUserProfile()->getCHistory()) {
		_cQuteCom.getCUserProfileHandler().getCUserProfile()->getCHistory()->clear(HistoryMemento::OutgoingSmsOk);
	}
}

void QtToolBar::clearHistoryAll() {
	if (_cQuteCom.getCUserProfileHandler().getCUserProfile() &&
		_cQuteCom.getCUserProfileHandler().getCUserProfile()->getCHistory()) {
		_cQuteCom.getCUserProfileHandler().getCUserProfile()->getCHistory()->clear(HistoryMemento::Any);
	}
}

void QtToolBar::logOff() {
	if (_userIsLogged) {
		// check for pending calls
		CUserProfile *cUserProfile = _cQuteCom.getCUserProfileHandler().getCUserProfile();
		if (cUserProfile) {
			CPhoneLine *cPhoneLine = cUserProfile->getCPhoneLine();
			if (cPhoneLine) {
				if (cPhoneLine->hasPendingCalls()) {
					
					if (QMessageBox::question(
						_qtQuteCom.getWidget(),
						tr("@product@ - Warning"),
						tr("You have unfinished call(s).") + "\n" +
							tr("Are you sure you want to log off?"),
						tr("&Log Off"),
						tr("&Cancel")
					) == 1)  {
						return;
					}
				}
			}
		}
		////
                // Close Chat
                _qtQuteCom.closeChatWindow ();
		
		// disable menubar and toolbar to avoid crashes
		//_ui->menuBar->setEnabled(false);
		_ui->actionLogOff->setEnabled(false);
		_ui->toolBar->setEnabled(false);
		////
	}

	// Calling this when the user has already been logged off is harmless and
	// will bring the login dialog.
	_cQuteCom.getCUserProfileHandler().logOff();
}

int QtToolBar::findFirstCallTab() {
	QtContactCallListWidget * widget;
	for (int i = 0; i < _ui->tabWidget->count(); i++) {
		widget = dynamic_cast<QtContactCallListWidget *>(_ui->tabWidget->widget(i));
		if (widget) {
			return i;
		}
	}
	return -1;
}

void QtToolBar::acceptCall() {
	int callIndex = findFirstCallTab();
	if (callIndex == -1) {
		return;
	}
	QtContactCallListWidget * widget = (QtContactCallListWidget *) _ui->tabWidget->widget(callIndex);
	if (!widget) {
		return;
	}

	QtPhoneCall * qtPhoneCall = widget->getFirstQtPhoneCall();
	if (qtPhoneCall) {
		qtPhoneCall->acceptCall();
	}
}

void QtToolBar::holdResumeCall() {
	int callIndex = findFirstCallTab();
	if (callIndex == -1) {
		return;
	}
	QtContactCallListWidget * widget = (QtContactCallListWidget *) _ui->tabWidget->widget(callIndex);
	if (!widget) {
		return;
	}

	QtPhoneCall * qtPhoneCall = widget->getFirstQtPhoneCall();
	if (qtPhoneCall) {
		qtPhoneCall->holdOrResume();
	}
}

void QtToolBar::hangUpCall() {
	int callIndex = findFirstCallTab();
	if (callIndex == -1) {
		return;
	}
	QtContactCallListWidget * widget;

	widget = (QtContactCallListWidget *) _ui->tabWidget->widget(callIndex);
	if (!widget) {
		return;
	}
	widget->hangup();
}

void QtToolBar::updateMenuActions() {

	bool accept = false, holdResume = false, hangUp = false;
	
	int callIndex = findFirstCallTab();
	if (callIndex > 0) {

		QtContactCallListWidget * widget = (QtContactCallListWidget *) _ui->tabWidget->widget(callIndex);
		if (widget) {

			QtPhoneCall * qtPhoneCall = widget->getFirstQtPhoneCall();

			if (qtPhoneCall) {

				switch (qtPhoneCall->getCPhoneCall().getState()) {
					
					case EnumPhoneCallState::PhoneCallStateResumed:
					case EnumPhoneCallState::PhoneCallStateTalking: {
						holdResume = true;
						hangUp = true;
						break;
					}
					
					case EnumPhoneCallState::PhoneCallStateDialing:
					case EnumPhoneCallState::PhoneCallStateRinging: {
						hangUp = true;
						break;
					}
					
					case EnumPhoneCallState::PhoneCallStateIncoming: {
						accept = true;
						hangUp = true;
						break;
					}
					
					case EnumPhoneCallState::PhoneCallStateHold: {
						holdResume= true;
						hangUp = true;
						break;
					}
					
					case EnumPhoneCallState::PhoneCallStateError: {
						hangUp = true;
						break;
					}
					
					// other cases -> set all to false
					case EnumPhoneCallState::PhoneCallStateMissed :
					case EnumPhoneCallState::PhoneCallStateRedirected:
					case EnumPhoneCallState::PhoneCallStateClosed:
					case EnumPhoneCallState::PhoneCallStateUnknown:
					default: {
						//accept = holdResume = hangUp = false;
					}
				}
			}
		}
	}

	_ui->actionAcceptCall->setEnabled(accept);
	_ui->actionHangUpCall->setEnabled(hangUp);
	_ui->actionHoldResumeCall->setEnabled(holdResume);
}

void QtToolBar::showChatWindow() {
	QWidget * chatWindow = _qtQuteCom.getChatWindow();
	if (chatWindow) {
		chatWindow->showNormal();
		chatWindow->activateWindow();
	}
}

void QtToolBar::showFileTransferWindow() {

	QtFileTransfer * fileTransfer = _qtQuteCom.getFileTransfer();
	if (fileTransfer) {
			fileTransfer->getFileTransferWidget()->show();
	}
}

void QtToolBar::retranslateUi() {
	updateShowHideContactGroupsAction();
	updateShowHideOfflineContactsAction();
	updateToggleVideoAction();
	updateActionsAccordingToUserStatus();
}

void QtToolBar::configChangedEventHandler(const std::string & key) {
	configChanged(QString::fromUtf8(key.c_str()));
}

void QtToolBar::slotConfigChanged(QString qkey) {
	std::string key(qkey.toUtf8());
	if (key == Config::VIDEO_ENABLE_KEY || key == Config::VIDEO_WEBCAM_DEVICE_KEY) {
		updateToggleVideoAction();
	}

	if (key == Config::GENERAL_TOOLBARMODE_KEY) {
		updateToolBar();
	}
}

void QtToolBar::updateToolBar() {
	
#if defined(OS_MACOSX)
	EnumToolBarMode::ToolBarMode toolBarMode = EnumToolBarMode::ToolBarModeIconsOnly;
#else	
	Config & config = ConfigManager::getInstance().getCurrentConfig();
	std::string text = config.getToolBarMode();
	EnumToolBarMode::ToolBarMode toolBarMode = EnumToolBarMode::toToolBarMode(text);	
#endif
	
	if (toolBarMode == EnumToolBarMode::ToolBarModeHidden) {
		_ui->toolBar->hide();

	} else if (toolBarMode == EnumToolBarMode::ToolBarModeIconsOnly) {
		_ui->toolBar->show();
		_ui->toolBar->setToolButtonStyle(Qt::ToolButtonIconOnly);

		// Restore icon width in case it was set by TextUnderIcon mode
		QSize size = _ui->toolBar->iconSize();
		size.setWidth(size.height());
		_ui->toolBar->setIconSize(size);

	} else if (toolBarMode == EnumToolBarMode::ToolBarModeTextUnderIcons) {
		_ui->toolBar->show();
		_ui->toolBar->setToolButtonStyle(Qt::ToolButtonTextUnderIcon);
		QSize size = _ui->toolBar->iconSize();
		size.setWidth(TOOLBAR_BUTTON_MINIMUM_WIDTH);
		_ui->toolBar->setIconSize(size);

	} else {
		LOG_WARN("Unknown toolbar mode: " + toolBarMode);
	}
}

void QtToolBar::userProfileDeleted() {
	_userIsLogged = false;
	updateActionsAccordingToUserStatus();
}

void QtToolBar::userProfileIsInitialized() {
	_userIsLogged = true;
	updateActionsAccordingToUserStatus();
}

void QtToolBar::enableMenuBar() {
	_ui->menuBar->setEnabled(true);
}

void QtToolBar::updateActionsAccordingToUserStatus() {
	// We need to use a boolean to keep trace of the user status because when
	// userProfileDeleted is called, the user profile is still alive, even if
	// it's not for a long time.
	if (_userIsLogged) {
		_ui->actionLogOff->setText(tr("Log &Off..."));
	} else {
		_ui->actionLogOff->setText(tr("Log &On..."));
	}
	//_ui->menuBar->setEnabled(_userIsLogged);
	_ui->toolBar->setEnabled(_userIsLogged);

	// This list contains all actions which needs a logged user to work
	QList<QAction*> loggedInActions;
	loggedInActions
	<< _ui->actionShowQuteComAccount
	<< _ui->actionAddContact
	<< _ui->actionSendSms
	<< _ui->actionEditMyProfile
	<< _ui->actionCreateConferenceCall
	<< _ui->actionShowHideOfflineContacts
	<< _ui->actionClearChatSessions
	<< _ui->actionClearAll
	<< _ui->actionSearchQuteComContact
	<< _ui->actionAcceptCall
	<< _ui->actionHangUpCall
	<< _ui->actionHoldResumeCall
	<< _ui->actionShowHideContactGroups
	<< _ui->actionOpenChatWindow
	<< _ui->actionClearOutgoingCalls
	<< _ui->actionClearIncomingCalls
	<< _ui->actionClearMissedCalls
	<< _ui->actionToggleVideo
	<< _ui->actionClearSms
	<< _ui->actionShowFileTransfer
	<< _ui->actionClearRejectedCalls
	;
	Q_FOREACH(QAction* action, loggedInActions) {
		action->setEnabled(_userIsLogged);
	}

}

void QtToolBar::tryingToConnect()
{
	_ui->actionLogOff->setText(tr("Log &Off..."));
	_ui->actionLogOff->setEnabled(true);
}

