/*
 *===================================================================
 *  3GPP AMR Wideband Floating-point Speech Codec
 *===================================================================
 */
#include "typedef.h"

#define L_SUBFR         64    /* Subframe size                       */
#define ORDER           16    /* Order of linear prediction filter   */
#define M               16    /* Order of LP filter                  */
#define NB_SUBFR        4     /* Number of subframe per frame        */
#define UP_SAMP         4
#define L_INTERPOL1     4
#define L_INTERPOL2     16
#define L_WINDOW        384   /* hamming cos window size             */
#define SIZE_BK1        256
#define SIZE_BK2        256
#define SIZE_BK21       64
#define SIZE_BK22       128
#define SIZE_BK23       128
#define SIZE_BK24       32
#define SIZE_BK25       32
#define SIZE_BK21_36b   128
#define SIZE_BK22_36b   128
#define SIZE_BK23_36b   64
#define SIZE_BK_NOISE1  64
#define SIZE_BK_NOISE2  64
#define SIZE_BK_NOISE3  64
#define SIZE_BK_NOISE4  32
#define SIZE_BK_NOISE5  32
#define NB_QUA_GAIN6B   64    /* Number of quantization level        */
#define NB_QUA_GAIN7B   128   /* Number of quantization level        */

/* 16-byte align start */

/*
 * isf codebooks:  two-stage VQ with split-by-5 in 2nd stage
 *
 * codebook   vector dimension    number of vectors
 * ~~~~~~~~   ~~~~~~~~~~~~~~~~    ~~~~~~~~~~~~~~~~~
 *    1_1            9                  256
 *    1_2            7                  256
 *    2_1            3                  64
 *    2_2            3                  128
 *    2_3            3                  128
 *    2_4            3                  32
 *    2_5            4                  32
 */

/*
 * 1st stage codebook; 1st split: isf0 to isf8
 */
const Float32 E_ROM_dico1_isf[SIZE_BK1 * 9] = {
   226.1477F, 422.2338F, 404.4563F, 152.3442F, 1.2553F, -102.5648F, -77.5073F, -32.1403F, 14.9134F,
   6.9890F, -26.5711F, -4.8001F, 122.2654F, 297.4533F, 158.0168F, 97.1963F, 43.4786F, -29.4927F,
   289.1461F, 493.4574F, 504.5048F, 393.1421F, 389.3840F, 397.8990F, 397.4072F, 381.0959F, 360.5083F,
   -35.6482F, 323.0566F, 370.2882F, 252.9499F, 239.6172F, 209.0563F, 203.9713F, 191.4865F, 164.6219F,
   16.1026F, -17.1264F, -109.6817F, -184.2085F, 254.5133F, 208.7087F, 75.3774F, 52.8718F, -35.2364F,
   15.8336F, -47.0775F, -139.1713F, -23.5561F, 259.0049F, 120.0214F, 23.8825F, -18.5948F, -134.3813F,
   217.4017F, 369.5988F, 409.6801F, 338.5433F, 330.4056F, 386.7012F, 434.5276F, 492.7760F, 484.6164F,
   -46.1429F, -79.6218F, 128.1106F, 199.9543F, 339.7710F, 309.8338F, 238.4430F, 156.8895F, 72.7742F,
   61.1308F, 114.6106F, 28.7939F, -131.9070F, -185.5015F, -350.3943F, -232.0556F, -62.9803F, -194.0513F,
   88.2757F, 51.2294F, -53.9555F, 120.0698F, 65.9863F, -105.8813F, -64.0929F, -151.0651F, -243.7652F,
   24.1201F, -12.6302F, -23.7808F, -98.2577F, -211.1396F, -323.5447F, -401.0499F, -204.4768F, -258.4671F,
   39.7202F, -23.9607F, 55.2442F, 43.6804F, -105.4100F, -97.8749F, -211.3002F, 9.7097F, -58.7138F,
   2.1766F, -51.7480F, -138.9340F, -268.1601F, -37.5098F, -125.8675F, -203.7929F, -12.2603F, -127.2472F,
   -14.0622F, -81.7822F, -203.4806F, -89.5766F, 119.8931F, -51.7010F, -2.1055F, -38.5536F, -149.8808F,
   23.6196F, -19.7370F, -92.7527F, -260.9933F, -380.1402F, -158.8319F, -276.5396F, -29.3644F, -67.0346F,
   10.0869F, -54.0238F, -104.0772F, 43.4993F, -118.0066F, 16.7674F, -108.4172F, -138.9907F, -140.3299F,
   222.5364F, 321.1411F, 193.8368F, -60.0340F, -121.7460F, -35.9078F, 53.6514F, 109.0639F, 145.0366F,
   -57.1173F, 143.6362F, 159.6491F, 26.6189F, 2.3305F, 30.0608F, 65.3329F, 78.8683F, 63.4622F,
   247.1436F, 350.7171F, 389.0685F, 295.2953F, 258.6614F, 266.7955F, 305.9202F, 355.1905F, 389.1698F,
   -40.0882F, 114.6876F, 237.0104F, 161.9769F, 188.5636F, 180.3751F, 187.4189F, 168.3521F, 159.5175F,
   -46.7654F, -132.1011F, -238.9261F, -204.8685F, 228.1911F, 129.1599F, 36.0960F, 169.1651F, 107.7902F,
   -69.5087F, -114.5055F, -60.0494F, -15.9078F, 105.1520F, 39.1060F, -3.6140F, 83.1345F, 62.4498F,
   324.4036F, 287.6214F, 108.7328F, 320.4871F, 489.7306F, 267.9902F, 278.2168F, 405.7287F, 184.7344F,
   -85.2571F, -118.8197F, 180.9905F, 177.4239F, 154.8864F, 106.5995F, 78.7528F, 111.5907F, 106.4840F,
   -90.5891F, 2.5823F, 2.5231F, -151.4854F, -184.4761F, -166.9757F, -147.6148F, -65.2055F, -39.1239F,
   -114.9974F, -71.3698F, 52.3087F, -18.1814F, 39.4264F, -34.2483F, -32.8691F, -45.8165F, -1.2029F,
   22.1063F, 6.7469F, -79.0146F, -247.7831F, -386.2835F, -437.2287F, -208.2931F, 68.6375F, -14.1447F,
   46.7840F, -11.0692F, 9.1612F, 43.4278F, -124.5574F, 124.1172F, -8.5368F, -30.1898F, 103.7909F,
   -105.9007F, -181.3862F, -169.7189F, -257.0920F, -250.0032F, -150.2815F, -150.4918F, -38.5650F, -26.7658F,
   -77.3138F, -101.1008F, -103.9903F, -17.1337F, -15.2577F, -54.2134F, -53.5373F, 66.9219F, 25.6872F,
   3.6265F, -56.7222F, -147.2143F, -330.5455F, -390.5518F, -43.3139F, -126.9112F, 133.5488F, 52.7399F,
   -31.5816F, -111.6706F, -148.4743F, 75.1588F, -22.0969F, 119.9764F, 29.6891F, -9.5309F, -54.7378F,
   264.5383F, 274.1302F, 96.5690F, 21.7008F, 97.2691F, 55.1255F, -40.9705F, -92.2272F, -38.6482F,
   13.8969F, -15.1538F, -27.1357F, 135.8413F, 77.4989F, -36.1739F, 125.8114F, 35.5744F, -28.2436F,
   196.2991F, 345.8166F, 588.9347F, 510.5197F, 500.8432F, 457.6244F, 436.9384F, 472.1788F, 414.4815F,
   162.4519F, 280.7038F, 386.4482F, 479.3623F, 390.8287F, 411.1292F, 372.5701F, 289.3485F, 407.8594F,
   -49.5343F, -147.0080F, -256.8338F, 54.1618F, 243.5530F, 87.1110F, 195.7264F, 119.5070F, 85.9479F,
   -43.9641F, -150.1866F, -310.9041F, 196.7455F, 171.2780F, 33.0834F, 83.3645F, -32.4579F, -75.7126F,
   228.4330F, 442.3277F, 481.6548F, 426.3000F, 487.1132F, 559.8019F, 590.4699F, 565.5949F, 513.3911F,
   -68.0544F, -164.7799F, 2.8141F, 451.2739F, 425.3554F, 461.7245F, 391.6532F, 369.1004F, 314.9413F,
   3.2743F, -49.3561F, -123.9568F, -40.1754F, -137.2377F, -271.6588F, -38.4305F, -104.5226F, -209.8071F,
   12.8092F, -40.2922F, -113.2953F, 65.4057F, -15.2443F, -159.0118F, 17.0998F, -81.3044F, -146.3446F,
   40.4305F, -8.9102F, -24.9719F, -113.7701F, -248.7160F, -332.5068F, -423.4328F, -23.8073F, -43.9266F,
   -29.4866F, -119.6504F, -169.5775F, 85.2585F, -57.7724F, -138.3062F, -265.4940F, -52.0935F, -84.4635F,
   -47.3285F, -147.4516F, -280.5169F, -38.0164F, -50.7237F, -141.1316F, -60.9737F, -147.8608F, -233.9212F,
   -22.0146F, -99.1479F, -228.8716F, 91.6618F, 61.1343F, -83.5247F, 4.2834F, -101.7035F, -58.0283F,
   -48.6094F, -104.2664F, -155.2664F, -226.4858F, -231.7223F, -205.7180F, -314.6280F, -150.3995F, 135.1819F,
   -75.2552F, -171.9896F, -276.5969F, -137.0060F, -55.2086F, -99.5797F, -195.1097F, -57.2647F, -72.3255F,
   174.9245F, 258.0028F, 192.8297F, 81.3562F, 198.8450F, 180.2081F, 132.0824F, 113.5126F, 58.1412F,
   -87.2559F, 34.2946F, 130.7081F, 62.1078F, 82.7816F, 74.4854F, 111.6818F, 120.3099F, 80.1334F,
   -11.9671F, 183.3834F, 313.7754F, 257.2813F, 241.7612F, 256.9710F, 329.4492F, 385.5374F, 434.7875F,
   -66.8195F, -94.6890F, 200.6887F, 141.3063F, 115.2635F, 204.7280F, 215.7973F, 270.9673F, 228.4647F,
   -24.9070F, -120.1441F, -175.1481F, -8.0896F, 110.9215F, 307.0280F, 174.2457F, 112.7169F, 35.9639F,
   -85.1717F, -152.2496F, -2.5803F, 66.0904F, 80.5138F, 128.8919F, 137.5303F, 159.4245F, 139.7766F,
   -14.1396F, 274.4125F, 374.6268F, 335.3957F, 336.4666F, 435.5791F, 495.5083F, 529.9642F, 509.9389F,
   -51.7794F, -133.3125F, -25.4473F, 264.8054F, 162.9175F, 171.9332F, 189.6735F, 202.1561F, 304.5228F,
   12.7820F, -17.2653F, -74.5083F, -134.5510F, -180.1133F, -294.9349F, -78.6120F, 84.8225F, -12.0772F,
   -137.7282F, -213.7303F, -17.3509F, 48.2225F, -23.9524F, -26.4293F, -30.7054F, 11.1575F, 23.4229F,
   28.3886F, -22.3550F, -158.4269F, -299.3692F, -485.7126F, -469.8884F, 93.8213F, 156.3351F, 64.4156F,
   -28.7029F, -110.1881F, -234.8032F, -83.3735F, -66.7772F, -146.3531F, 129.8178F, 13.7979F, -40.1959F,
   -11.2220F, -81.0542F, -215.8795F, -186.0052F, -249.1487F, -354.7951F, 67.0249F, -8.7287F, -52.7080F,
   -75.1895F, -93.5102F, -63.8842F, -40.2971F, -43.3423F, -18.3122F, 59.9384F, 48.9469F, 42.9198F,
   -0.3036F, -79.3921F, -222.4614F, -402.4193F, -556.4007F, -208.9512F, 60.4822F, 0.2592F, 57.2596F,
   -130.2469F, -254.9911F, -337.7263F, -76.9102F, -61.5888F, -8.3782F, -17.2822F, 37.1637F, 42.0772F,
   151.9460F, 229.6417F, 191.5279F, 12.9809F, -92.7110F, -204.7091F, -245.2630F, -53.2589F, -101.3680F,
   15.8084F, -69.0148F, -180.2926F, 176.9411F, 336.6405F, 148.3504F, 51.2260F, -50.6708F, -158.2893F,
   328.7315F, 655.2856F, 719.0299F, 605.1412F, 575.6907F, 490.5742F, 422.8277F, 353.4364F, 289.8675F,
   144.5240F, 474.8398F, 690.7612F, 638.0545F, 473.3627F, 248.6244F, 8.4810F, -128.9032F, 27.8837F,
   -29.8052F, -109.6768F, -289.4402F, -289.9344F, 350.9098F, 241.8981F, 108.0220F, 27.7568F, -86.8368F,
   -12.5998F, -103.5273F, -217.2510F, -9.8401F, 388.4400F, 266.5437F, 118.9818F, 49.1643F, -64.3603F,
   28.5445F, 288.4229F, 348.8730F, 378.0052F, 387.8595F, 690.6161F, 887.8799F, 718.7092F, 543.2699F,
   -27.0127F, -136.3472F, -228.6534F, 91.4073F, 452.4637F, 352.8853F, 244.5759F, 199.3164F, 97.9452F,
   -0.2574F, -38.8124F, -106.0784F, -81.9691F, -235.3859F, -136.9961F, -210.8024F, -316.9619F, -149.7671F,
   -6.3268F, -89.9384F, -197.0412F, 160.1820F, 58.3899F, -80.1867F, -134.0614F, -254.4108F, -249.4793F,
   40.1439F, -3.4822F, -88.7857F, -80.2522F, -219.3521F, -305.1500F, -421.5967F, -471.9935F, -60.7522F,
   55.8529F, 24.5260F, -52.9174F, -26.1113F, -123.8744F, -235.0002F, -306.0707F, -450.7246F, -249.8812F,
   -56.2297F, -152.7118F, -263.1169F, -243.0294F, -78.2260F, -99.3225F, -257.7953F, -369.9445F, -154.2971F,
   -15.4570F, -97.8124F, -244.1879F, 10.4628F, 212.1971F, 36.6411F, -51.0200F, -150.9477F, -262.8206F,
   -47.9169F, -145.0780F, -295.8412F, -176.3119F, -220.4176F, -239.8767F, -162.2583F, -277.5646F, -13.5098F,
   -45.4312F, -120.7871F, -231.7391F, -104.6837F, 93.3399F, -13.0435F, -131.8873F, -253.9684F, -52.8010F,
   36.7261F, 97.8902F, 216.2452F, 22.2667F, -121.9688F, -165.2371F, -60.1795F, -22.1111F, 91.7368F,
   -104.8126F, -27.6777F, 148.6412F, 44.3956F, -17.0786F, -34.0943F, 48.6552F, 67.5896F, 52.1341F,
   590.8627F, 669.3461F, 483.5975F, 208.4471F, 107.7190F, 123.1392F, 179.9797F, 179.2124F, 198.5541F,
   -51.3512F, -7.3050F, 448.6575F, 261.8456F, 189.7417F, 139.0593F, 120.8041F, 144.0752F, 115.5179F,
   -87.2434F, -195.6786F, -351.1762F, -282.1990F, -27.3961F, 2.1866F, 51.0065F, 120.9792F, 154.0704F,
   -38.7878F, -118.5436F, -201.9035F, 97.1421F, 24.9785F, -20.8748F, 52.5830F, -4.3230F, 177.0334F,
   -57.4864F, -155.8205F, -285.0674F, -156.5139F, 319.0307F, 288.4199F, 313.3522F, 292.7644F, 224.7193F,
   -60.1766F, -170.1086F, -288.5482F, 312.6666F, 231.6275F, 142.7810F, 206.6260F, 124.0985F, 127.5073F,
   -87.3474F, 17.7221F, -15.2086F, -151.2513F, -201.3116F, -202.2629F, -237.4979F, -150.1082F, -125.3288F,
   -123.1549F, -147.2588F, 55.9560F, -39.6018F, -44.1818F, -147.1130F, -69.1241F, -56.1722F, -4.4924F,
   45.5201F, 15.5222F, -93.3382F, -254.1893F, -410.5618F, -226.8515F, -287.7839F, -386.6172F, -128.1340F,
   10.1233F, -19.6005F, -61.5127F, -8.8349F, -177.1218F, -110.5849F, -207.5041F, -213.2586F, 74.8075F,
   -98.2741F, -195.7624F, -290.1091F, -230.1600F, -244.7515F, -194.7701F, -128.1076F, -46.1729F, -28.2903F,
   -126.3800F, -193.1177F, -95.3231F, -119.6219F, -56.1586F, -69.0881F, -102.1922F, -52.5595F, -30.4027F,
   -13.9604F, -91.5188F, -202.7529F, -375.3352F, -503.8488F, -122.5019F, -187.2047F, -144.9473F, -17.6464F,
   -37.1158F, -113.9219F, -208.9531F, -3.1283F, -117.0010F, 43.6858F, -64.0984F, -108.2942F, 77.5131F,
   -38.6401F, -49.9736F, 343.9084F, 326.3879F, 226.1680F, 136.9212F, 9.0736F, -37.1756F, -84.7974F,
   -10.5004F, -100.6890F, 48.4106F, 394.7867F, 233.0419F, 165.9509F, 56.3291F, 2.7924F, -28.5845F,
   164.3221F, 505.0226F, 640.4844F, 633.9930F, 680.5828F, 631.7908F, 585.4574F, 501.5025F, 393.1295F,
   -37.0300F, 293.6539F, 656.3401F, 612.9729F, 632.0821F, 560.9024F, 468.8401F, 382.9867F, 278.2211F,
   -26.9959F, -117.0381F, -266.9459F, -169.9414F, 442.3678F, 351.1593F, 196.7216F, 129.6217F, 42.5401F,
   -28.9160F, -126.1295F, -248.8180F, 219.9974F, 419.6462F, 237.5276F, 144.8018F, 41.0794F, -19.0438F,
   -30.6524F, 324.5342F, 466.4269F, 433.6403F, 538.3355F, 578.5025F, 582.9940F, 533.2873F, 475.3817F,
   -101.0867F, -47.3945F, 562.3863F, 521.2437F, 635.9947F, 581.9960F, 561.7294F, 477.7560F, 364.4759F,
   -32.1078F, -119.5424F, -239.6324F, -86.9073F, -147.6602F, -263.4799F, -212.9377F, -261.9229F, -330.2635F,
   20.8260F, -48.5704F, -135.4246F, 164.8506F, 20.4817F, -48.8885F, -105.4348F, -206.6621F, 3.6375F,
   30.9311F, -34.7599F, -124.8998F, -258.7565F, -390.0532F, -468.4556F, -485.3543F, -264.1984F, -115.9895F,
   -26.5921F, -106.6166F, -238.7686F, 53.4977F, -56.8913F, -155.1848F, -245.0791F, -329.9394F, -85.9503F,
   -43.8040F, -135.2615F, -311.2635F, -322.8195F, 91.5184F, -51.5741F, -73.3097F, -108.7735F, -204.0501F,
   -61.9589F, -158.0212F, -286.8211F, -163.7011F, 114.5577F, 29.0545F, -65.0919F, -65.1646F, 71.9506F,
   -59.8760F, -170.6047F, -325.2270F, -421.8182F, -131.3054F, -184.2594F, -219.1312F, -132.9352F, -98.8301F,
   -65.8626F, -165.1987F, -320.2537F, -353.1492F, -51.0331F, -7.3456F, -135.3074F, -235.8573F, 12.1485F,
   12.8460F, -11.9241F, 121.7405F, 24.2470F, -57.8971F, 19.1365F, -23.0243F, 220.3897F, 189.9156F,
   -119.6102F, -129.9914F, 75.9549F, -17.3359F, 26.2798F, 28.0181F, 57.3185F, 79.8925F, 95.0199F,
   -81.0524F, -19.0879F, 531.3251F, 384.0082F, 378.6000F, 387.1200F, 396.1896F, 433.4382F, 379.9696F,
   -82.3204F, -67.1148F, 345.0646F, 244.9884F, 277.8043F, 263.1406F, 275.4622F, 311.7924F, 291.4141F,
   -34.2889F, -127.1388F, -298.1820F, -380.4511F, 268.4245F, 354.6007F, 200.7670F, 149.3555F, 67.1469F,
   -114.0830F, -239.0252F, -314.3921F, 24.6004F, 51.2722F, 105.4223F, 101.1157F, 137.3569F, 135.9245F,
   -91.7432F, -32.7591F, 373.1560F, 319.3955F, 437.4347F, 503.4907F, 608.7997F, 578.2119F, 502.0230F,
   -70.3367F, -179.9099F, -239.7652F, 256.5794F, 269.9891F, 291.1810F, 333.6932F, 305.8695F, 278.4421F,
   -38.0671F, -120.8247F, -186.4547F, -239.7784F, -303.6527F, -286.7954F, -300.0361F, -205.3057F, -184.5087F,
   -134.2125F, -185.9339F, -13.8119F, -65.8529F, 19.0133F, -30.1823F, -58.4970F, -93.5678F, -55.1841F,
   -20.3990F, -104.7817F, -249.4716F, -358.9899F, -499.3363F, -434.6861F, -133.7299F, -130.1813F, -59.0671F,
   -26.7201F, -94.4739F, -228.6340F, -28.5857F, -81.7289F, -186.7961F, -62.0135F, -167.5409F, 51.9732F,
   -76.8652F, -194.8085F, -392.5224F, -495.1741F, -106.3757F, -87.6781F, -40.9442F, -25.9922F, 6.6919F,
   -141.6526F, -241.2805F, -161.8672F, -45.4915F, -24.1067F, 7.6906F, 3.7741F, 45.2149F, 42.3436F,
   -76.2028F, -185.3649F, -353.8127F, -492.3104F, -348.1328F, -172.4144F, -108.0814F, -55.5374F, -10.8210F,
   -88.1331F, -202.9139F, -371.2243F, -273.3005F, -107.5773F, -103.9134F, -45.1607F, -40.9613F, 31.8985F,
   157.9412F, 199.7964F, 203.2689F, 127.6304F, 6.7547F, -75.7475F, -130.0489F, -209.2208F, -228.8015F,
   -44.6527F, -50.6749F, 107.6295F, 92.4422F, 79.8121F, 133.5672F, 52.5620F, -6.2270F, -43.2877F,
   261.9115F, 472.0228F, 456.3473F, 335.7553F, 289.8814F, 234.7976F, 206.1187F, 157.4174F, 120.7698F,
   155.2229F, 242.6315F, 377.1642F, 293.6065F, 226.0318F, 155.2748F, 156.4116F, 128.6701F, 98.5061F,
   74.5256F, 70.1303F, -53.6869F, -182.3488F, 106.1648F, 41.5585F, -37.0232F, 6.5336F, -75.1722F,
   -31.2252F, -113.3890F, -244.5892F, 75.8334F, 233.5228F, 76.5742F, 8.2886F, -109.6154F, 29.9907F,
   199.0678F, 337.3158F, 432.8418F, 315.2969F, 366.8018F, 352.4917F, 361.4405F, 280.1715F, 187.7336F,
   53.4911F, 143.3279F, 208.4704F, 298.2910F, 261.7387F, 149.2119F, 115.7060F, 59.9002F, 32.6898F,
   118.3519F, 194.3090F, 56.3174F, -33.3258F, -48.8783F, -210.5418F, -188.4141F, -181.0565F, -298.4099F,
   91.1937F, 135.5274F, 26.3852F, -57.5649F, 66.0117F, -82.2194F, -94.6889F, -88.3680F, -188.1230F,
   119.8292F, 164.8051F, 60.2985F, -68.4990F, -150.8028F, -282.2156F, -282.8785F, -353.0452F, -396.4856F,
   120.5806F, 120.4464F, 62.5068F, -23.5466F, -183.4260F, -164.0083F, -233.7240F, -309.0768F, -85.4341F,
   26.4925F, 47.3529F, -53.5415F, -218.9011F, -56.8672F, -174.1118F, -201.0902F, -193.1382F, -284.6390F,
   50.9597F, 20.8274F, -88.6445F, 17.8450F, 185.2710F, 12.4814F, -62.7698F, -74.9989F, -191.3486F,
   83.3881F, 63.9878F, -27.8680F, -181.7226F, -342.0545F, -62.8876F, -178.1327F, -229.1320F, -18.6240F,
   85.1419F, 45.8095F, 15.3667F, 69.0366F, -75.9137F, -34.4539F, -88.4330F, -163.4411F, 19.3370F,
   81.8880F, 213.5459F, 222.3313F, 109.1533F, 47.1884F, -17.2578F, -19.4486F, 3.7457F, -32.8905F,
   22.8266F, 54.6145F, 70.9314F, -2.0301F, 104.3592F, 45.7102F, 41.3375F, 82.5320F, 77.2795F,
   210.7176F, 326.3462F, 356.6618F, 280.9097F, 240.9705F, 212.4392F, 230.9905F, 220.8317F, 250.9306F,
   59.6089F, 218.2601F, 340.5622F, 179.8372F, 86.7106F, 42.1021F, 73.5578F, 70.2340F, 71.3620F,
   61.9071F, 46.6476F, 111.0709F, -59.8230F, -105.9862F, 89.4424F, 34.0065F, 42.7793F, -22.3657F,
   -71.5290F, 31.9224F, 46.2106F, 8.2769F, 5.2717F, 15.6736F, 45.9446F, 74.5921F, 72.2933F,
   63.1228F, 347.0971F, 255.3817F, 42.2624F, -13.4026F, 95.4917F, 190.7911F, 219.1920F, 207.8866F,
   63.7817F, 21.7322F, 237.9311F, 133.1283F, 19.5484F, 128.6760F, 26.4302F, 103.7761F, 85.3130F,
   39.2312F, 80.2880F, 7.1932F, -118.9054F, -41.8213F, -170.3956F, -190.1681F, -25.3174F, -119.5246F,
   -33.4058F, 60.1838F, 52.1765F, -11.5418F, -17.6690F, -28.5228F, -40.5342F, -31.3702F, -37.5340F,
   95.6257F, 128.8044F, 3.7503F, -171.8902F, -331.7014F, -422.7089F, 30.8156F, 15.6229F, -103.6316F,
   76.5623F, 145.2620F, 106.4321F, -70.8798F, -192.7731F, -152.0614F, 107.2952F, 31.2426F, -23.0299F,
   0.9016F, -4.6471F, -96.0447F, -197.1141F, -39.1036F, -170.4243F, 8.3031F, -73.0702F, -168.4342F,
   -86.2673F, -18.7660F, 13.9540F, -105.8362F, -72.7468F, -57.3178F, -42.7630F, 10.3291F, 27.6536F,
   83.1109F, 54.8163F, 28.1776F, -137.0164F, -242.1605F, -32.9744F, -141.6118F, 26.8712F, 17.8517F,
   35.5325F, 65.1065F, -1.1122F, -37.2264F, -38.7638F, -41.0658F, -18.6295F, 44.3877F, 57.2718F,
   101.1713F, 97.2361F, 67.3819F, 236.9884F, 158.4350F, 20.2720F, 23.0851F, -73.7010F, -124.8067F,
   44.9800F, -33.1049F, -21.2276F, 224.2546F, 49.9567F, 88.2466F, -23.1899F, -98.6740F, 50.9732F,
   -24.3959F, 403.6523F, 511.0127F, 404.3978F, 440.2577F, 428.9172F, 401.8223F, 375.5111F, 321.4924F,
   15.1345F, 142.1195F, 295.8327F, 367.1741F, 284.5674F, 257.7681F, 257.3533F, 227.7437F, 300.6506F,
   -44.9562F, -131.9449F, -296.8129F, -184.1626F, 153.8183F, 14.5297F, 172.3343F, 69.5618F, 2.2208F,
   -22.2475F, -119.2772F, -205.2499F, 310.9423F, 177.0844F, 73.5587F, -1.5010F, -44.3403F, 96.9000F,
   27.7742F, 173.5793F, 311.4776F, 285.5815F, 427.9854F, 452.0145F, 477.2097F, 401.7775F, 316.6574F,
   52.6696F, 140.2801F, 215.3995F, 166.1714F, 292.6204F, 318.2898F, 341.5682F, 274.8850F, 196.1860F,
   51.5561F, 96.4057F, -0.0610F, -80.6567F, -175.2689F, -292.8310F, -100.7723F, -200.9591F, -247.1591F,
   96.6859F, 97.0977F, 35.4401F, 47.1563F, -76.2811F, -194.8284F, -34.9894F, -110.1535F, -169.8929F,
   30.4654F, 7.6447F, -108.2569F, -243.2014F, -383.8708F, -478.0505F, -162.0353F, -178.7682F, -249.4727F,
   135.4580F, 198.9998F, 81.4247F, -69.9475F, -181.4226F, -284.4768F, -29.8270F, -92.7044F, -189.7940F,
   -40.1188F, -133.9427F, -295.2091F, -278.5829F, -103.4496F, -237.8449F, -74.7856F, -155.3338F, -248.5054F,
   -47.4470F, -149.6037F, -292.4242F, 221.6694F, 98.4280F, -14.0849F, -138.2963F, -163.0445F, -19.4243F,
   79.5896F, 39.1792F, -58.2936F, -253.9809F, -422.3516F, -18.3412F, -2.7302F, -102.9275F, 43.5500F,
   -17.9005F, -70.1182F, -104.1536F, -126.4881F, -219.3935F, -153.8877F, -270.2488F, 155.3837F, 114.2388F,
   188.3484F, 261.8459F, 266.6991F, 243.5570F, 172.7795F, 64.3333F, 45.3547F, 14.1479F, -58.3812F,
   42.3610F, 96.6123F, 113.8369F, 96.5219F, 138.7232F, 47.5273F, 42.4855F, 87.4416F, 115.5724F,
   -5.4013F, 368.9961F, 386.7844F, 312.8328F, 294.8554F, 318.4847F, 330.7657F, 356.5357F, 348.3268F,
   113.9440F, 136.1339F, 283.1776F, 188.4109F, 151.5282F, 128.5029F, 167.5631F, 242.2409F, 260.3565F,
   -13.3113F, 76.9204F, 83.0527F, -49.6054F, 32.9182F, 193.1624F, 242.1413F, 224.6298F, 146.4784F,
   49.3300F, 80.8438F, 67.2940F, 65.3665F, 141.4567F, 78.9471F, 115.6802F, 154.3108F, 177.7814F,
   -2.1684F, 97.5184F, 210.7061F, 182.4635F, 248.3353F, 312.9833F, 448.6478F, 502.5403F, 436.6044F,
   10.4934F, 93.6577F, 144.0456F, 109.2677F, 171.7553F, 160.5641F, 247.6292F, 348.4738F, 372.3836F,
   62.0393F, 66.4595F, -22.8432F, -154.2429F, -311.1715F, -269.7167F, 30.1724F, -82.4276F, -130.4250F,
   -1.8658F, -10.8456F, -5.2486F, -28.7596F, -130.9176F, -235.6978F, 116.9998F, 34.2436F, -79.9885F,
   32.1393F, -12.8439F, -142.0784F, -272.6861F, -469.8031F, -450.3773F, 42.8701F, -56.9028F, -113.0164F,
   44.2438F, 0.5177F, -95.0709F, -229.7896F, -388.2228F, -193.8794F, 161.5550F, 62.3152F, 16.3412F,
   -21.7573F, -96.5696F, -171.8374F, -270.5201F, -389.2275F, -187.1907F, 4.2880F, -69.4009F, -139.6222F,
   -59.1336F, -137.9044F, -127.8821F, -82.4177F, -132.8951F, 55.2591F, 25.2046F, 165.8443F, 176.7896F,
   13.1713F, -65.9463F, -177.8707F, -363.8923F, -474.4700F, 53.7845F, 194.7626F, 99.8684F, 126.7437F,
   26.4565F, 54.2781F, -5.8084F, -213.5320F, -186.7453F, 6.5259F, 119.4530F, 196.2039F, 188.0246F,
   -12.3442F, -52.3945F, 174.0135F, 50.5399F, -55.6784F, -95.2558F, -196.2955F, -198.2056F, -234.0973F,
   23.6446F, -54.5462F, -134.7219F, 193.6175F, 178.7128F, -0.9450F, 7.7510F, -88.8611F, -200.7396F,
   154.0881F, 689.2718F, 650.6971F, 522.8596F, 436.5025F, 314.9000F, 250.7824F, 187.1628F, 148.5410F,
   83.9959F, 202.8234F, 359.2843F, 411.3538F, 425.6817F, 308.8017F, 206.3666F, 113.2320F, 60.3929F,
   -21.1689F, -91.1694F, -252.6742F, -235.0333F, 249.5398F, 114.6655F, -0.7796F, -65.2547F, -172.8182F,
   -30.5567F, -123.2070F, -309.0227F, -44.0884F, 320.4634F, 157.6161F, 61.6828F, -45.3025F, -139.1032F,
   206.4974F, 723.0883F, 782.3015F, 479.5419F, 242.9557F, -15.8222F, -162.4184F, 134.4517F, 319.9661F,
   -41.0232F, -148.1419F, -92.0839F, 478.0700F, 348.8458F, 292.6417F, 221.8808F, 139.1332F, 83.7320F,
   -6.8324F, -77.6477F, -56.3444F, 19.3478F, -110.6184F, -96.5414F, -225.9007F, -330.6617F, -424.5623F,
   26.9337F, -4.4548F, -148.8763F, -80.3516F, 81.8311F, -110.8947F, -150.9911F, -162.4880F, -279.5481F,
   15.4135F, -1.8583F, -56.8319F, -146.1688F, -266.5346F, -355.1809F, -419.6793F, -456.7536F, -416.2208F,
   112.1756F, 88.2059F, 26.0259F, -86.2333F, -258.4741F, -66.9660F, -164.5662F, -250.9281F, -276.0475F,
   -51.5026F, -135.9180F, -210.3467F, -175.0943F, -7.7200F, -1.5529F, -138.1960F, -292.0197F, -364.5311F,
   1.6098F, -29.3697F, -112.8375F, -233.4716F, 123.6812F, 20.4676F, -81.0632F, -116.0258F, -218.5294F,
   -34.1878F, -103.0825F, -139.6876F, -229.9859F, -246.3806F, -96.8411F, -204.4168F, -321.0166F, -418.4522F,
   27.1891F, -2.9717F, 21.2254F, -122.5167F, -201.1399F, 36.0115F, -56.9192F, -106.8464F, -192.5580F,
   77.7097F, 24.2840F, 152.6398F, 61.5673F, -54.9359F, 27.7810F, -85.7347F, -79.4392F, -81.0366F,
   59.2315F, 15.7636F, 128.6461F, 63.2966F, -11.4618F, 18.7670F, -58.2229F, 42.0642F, 49.7434F,
   247.9855F, 413.3163F, 345.0402F, 192.3750F, 145.4847F, 121.6956F, 123.6805F, 107.2007F, 94.2462F,
   104.2558F, 282.0381F, 490.4723F, 344.7097F, 244.1310F, 96.7224F, 2.9896F, -31.7180F, -23.3323F,
   -22.5118F, -53.8832F, -113.5013F, -234.2837F, -4.7630F, -0.8369F, -15.3972F, 57.4113F, 45.6486F,
   -41.6155F, -134.7804F, -200.2841F, 179.2444F, 29.8214F, 36.0840F, -106.4028F, 151.5328F, 102.1621F,
   141.5078F, 201.5805F, 79.4698F, -159.8402F, -279.6548F, -324.4731F, -129.3751F, 72.2403F, 81.4769F,
   -45.8403F, -152.7728F, -116.5580F, 262.2678F, 114.2470F, 210.0651F, 100.2266F, 64.9777F, -14.6634F,
   -39.9272F, -124.8000F, -75.7294F, -110.6360F, -223.7299F, -102.2257F, -226.1053F, -85.4944F, -173.4764F,
   -91.6640F, 30.6543F, 4.1065F, -65.4408F, -39.3890F, -89.5364F, -102.5911F, -125.5036F, -48.1974F,
   27.2660F, 19.3735F, -66.4810F, -233.9350F, -388.9383F, -229.7098F, -102.6715F, -201.6806F, -177.6036F,
   153.9685F, 141.8902F, 89.4281F, -53.1892F, -209.9941F, 8.0810F, -71.3509F, -135.8233F, -78.6157F,
   -48.4004F, -143.6756F, -250.1705F, -343.5184F, -330.9693F, -81.7413F, -159.8642F, -192.9882F, -201.0525F,
   -49.7401F, -133.3727F, -211.5154F, -166.0789F, -199.3352F, -4.0380F, -98.3591F, -184.6634F, -113.6087F,
   32.7128F, -26.7812F, -78.3951F, -264.0457F, -339.2219F, 40.1904F, -121.6728F, -51.6241F, -125.0696F,
   1.8568F, -67.7562F, -73.4413F, -115.9242F, -245.2522F, 77.1352F, -22.1839F, 2.7514F, -4.3203F,
   19.0296F, -62.4543F, 21.8088F, 217.8850F, 43.1886F, 12.7766F, -121.5772F, -171.9543F, -180.7850F,
   -0.3986F, -96.1580F, -120.0616F, 336.5914F, 176.9306F, 54.3062F, -66.4501F, -138.8101F, -90.7495F,
   109.0295F, 377.1798F, 641.5853F, 577.4881F, 571.3625F, 438.8002F, 310.6432F, 204.9863F, 132.6160F,
   -76.9013F, -15.0371F, 664.7503F, 520.0686F, 489.1051F, 371.2256F, 270.3431F, 196.9454F, 166.5973F,
   -42.2498F, -134.3667F, -336.4376F, -457.6409F, 173.4562F, 138.1508F, 34.3921F, -18.1609F, -85.9677F,
   -20.8727F, -125.4622F, -192.7956F, 434.5861F, 290.6413F, 142.0824F, 77.1623F, -13.2305F, -29.4274F,
   178.7104F, 373.0402F, 459.7581F, 474.3989F, 557.5211F, 569.2194F, 525.4768F, 358.3187F, 210.4229F,
   -26.8194F, 77.7574F, 350.3222F, 445.4183F, 524.4524F, 462.1511F, 381.7293F, 289.9692F, 204.0649F,
   47.7632F, 17.1803F, -105.1057F, 10.4223F, -60.5128F, -219.3625F, -119.8198F, -230.6158F, -301.9153F,
   60.2078F, 16.2198F, -62.5382F, 98.4891F, -50.3729F, -119.2299F, -183.9794F, -286.1903F, -145.0719F,
   52.5866F, 72.1675F, -32.1273F, -162.5446F, -282.0867F, -356.6873F, -196.9938F, -290.2792F, -343.8752F,
   58.1756F, 83.5942F, -32.7989F, -128.5429F, -265.7008F, -326.2490F, -166.5130F, -258.2744F, -31.7978F,
   -49.8436F, -148.3240F, -287.2763F, -389.7832F, -131.7967F, 6.5863F, -71.1640F, -182.3324F, -272.4139F,
   -32.6407F, -113.2022F, -199.3120F, -231.3794F, 5.2103F, 172.0263F, 60.2526F, -14.7182F, -108.9096F,
   27.2077F, -23.7367F, -95.9858F, -283.8613F, -409.0134F, -31.3871F, -148.6489F, -208.9727F, -274.8275F,
   69.4333F, -0.7143F, -57.0494F, -261.8966F, -366.3107F, 188.4734F, 53.9671F, 24.5447F, 25.3948F,
   -4.4283F, 5.6792F, 301.6043F, 172.8969F, 55.6507F, -7.7604F, -81.7609F, -49.2316F, -63.0654F,
   -12.4033F, -97.3530F, 37.2340F, 215.5548F, 48.4286F, 11.6666F, -133.9813F, 32.1272F, -33.6059F,
   57.9103F, 293.2768F, 591.9909F, 431.5314F, 338.6320F, 236.6151F, 185.2476F, 175.1540F, 156.0322F,
   -63.6185F, -100.3640F, 351.1546F, 428.5004F, 353.7403F, 293.3142F, 196.1020F, 152.1965F, 114.6518F,
   -20.0721F, -100.8514F, -174.7278F, -314.6773F, -143.5646F, 298.1905F, 181.1825F, 142.0341F, 71.3729F,
   -64.7951F, -146.2021F, -143.2170F, 33.9854F, 13.8056F, 156.0337F, 163.4005F, 334.1913F, 325.3497F,
   -79.9064F, -121.2133F, 229.6626F, 303.8755F, 306.8218F, 415.8466F, 436.8824F, 486.2830F, 451.8081F,
   -67.6130F, -122.0631F, 41.7635F, 134.7727F, 156.2778F, 308.4236F, 340.0357F, 434.5742F, 391.1108F,
   -2.6037F, -46.9405F, -151.3320F, -160.0405F, -239.8193F, -368.3138F, -88.0921F, -150.1500F, -191.7920F,
   -79.3309F, -112.6262F, -20.0215F, -129.1640F, -35.3227F, -69.4806F, -159.3188F, -223.7142F, -132.1779F,
   21.9835F, -11.3374F, -106.7910F, -244.8813F, -406.4640F, -311.7320F, -96.4711F, -182.5312F, 57.8956F,
   25.8713F, -0.7151F, -79.9440F, -80.0111F, -224.4878F, -136.2209F, -22.1062F, -137.4635F, -22.6838F,
   -17.4638F, -88.0421F, -183.7897F, -361.0791F, -194.1127F, 29.9571F, -12.3110F, 17.3506F, -52.9132F,
   -108.1431F, -191.9295F, -193.9598F, -196.0948F, -165.6809F, -79.0258F, -53.5949F, 30.1916F, 37.3619F,
   10.0420F, -70.0023F, -183.1197F, -393.7681F, -492.0758F, 102.2096F, -13.6587F, -51.6821F, -101.1752F,
   -25.8195F, -90.5090F, -174.4579F, -208.3670F, -308.3139F, -74.5783F, -38.9631F, -104.3672F, 142.1591F
};

/*
 * 1st stage codebook; 2nd split: isf9 to isf15
 */
const Float32 E_ROM_dico2_isf[SIZE_BK2 * 7] = {
   530.1115F, 512.7983F, 443.8771F, 306.1447F, 171.2327F, 70.8250F, 56.8183F,
   248.5479F, 252.9719F, 260.5429F, 221.7305F, 172.6754F, 84.8419F, 141.5120F,
   166.9014F, 171.9745F, 263.4265F, 204.8570F, 129.5061F, 45.8705F, -162.9417F,
   47.0761F, 115.3926F, 182.9788F, 181.5048F, 89.7111F, 17.3276F, -86.2999F,
   -57.3148F, -93.7950F, 58.2071F, 31.0844F, 152.3217F, 108.6931F, 41.3677F,
   -163.1767F, -217.0378F, 215.5667F, 199.7393F, 91.8320F, 56.3535F, -37.1534F,
   16.9371F, 75.2105F, 106.8429F, 58.4518F, 26.0167F, 13.3113F, -106.5544F,
   -16.6336F, -49.0904F, 66.7054F, 162.6404F, 110.2851F, 24.4686F, -138.4124F,
   -145.3240F, -33.7683F, -134.5548F, -42.3167F, -36.8413F, -70.9451F, -34.7360F,
   -234.5519F, -328.1575F, -78.2304F, 181.4482F, 100.8272F, -4.3886F, -98.7986F,
   -18.7314F, 128.6640F, 38.0685F, -113.4033F, -211.9919F, -310.5748F, -138.0866F,
   -222.6271F, -45.5429F, 72.9616F, 3.9276F, -52.1214F, -162.3586F, -29.6134F,
   -241.3416F, -50.3196F, -96.3331F, -145.0456F, 17.4087F, -29.8105F, 108.3555F,
   -399.1142F, -421.4348F, 49.0965F, 185.0085F, 99.2830F, 49.7543F, 20.2726F,
   -109.7596F, 29.7695F, -65.3607F, -141.1745F, -110.3974F, -215.1475F, -110.3518F,
   -46.4209F, -20.4624F, -0.5266F, 52.1919F, -12.3074F, -79.8395F, -162.1720F,
   415.6808F, 323.0418F, 248.6461F, 267.3440F, 181.1869F, 81.5855F, 4.8347F,
   188.1995F, 162.5893F, 175.4467F, 144.9775F, 131.0177F, 114.8023F, 75.6850F,
   280.8494F, 225.0358F, 142.4128F, 52.8503F, 44.0619F, 35.6548F, -77.7252F,
   116.2722F, 68.6422F, 192.5792F, 142.9262F, 75.7274F, 63.5846F, 13.9314F,
   -13.4854F, -92.0420F, -101.3018F, -13.9503F, -1.5377F, 38.4848F, 59.2146F,
   -38.0921F, -119.6284F, -10.4956F, 88.8682F, 35.1836F, 43.2013F, -33.7051F,
   35.4547F, 5.0305F, -82.4376F, -100.9264F, -41.5222F, 33.5725F, -24.8381F,
   28.3963F, -13.4840F, -22.4480F, -11.9734F, 63.1961F, 13.8177F, -75.0030F,
   -42.4115F, -130.9516F, -245.5204F, -25.9129F, -23.6896F, -49.8110F, 125.7094F,
   -193.2334F, -261.3991F, -284.3770F, 75.4337F, 11.9830F, -86.1044F, 47.5069F,
   126.4967F, 37.0484F, -34.7501F, -35.5687F, -159.6523F, -277.2735F, -60.1203F,
   -0.1156F, -91.4527F, 35.7808F, 12.8593F, -134.0804F, -237.8749F, -85.9030F,
   -134.0644F, -159.5199F, -185.8239F, -255.9559F, -59.8235F, 31.9001F, 86.5795F,
   -191.3466F, -290.9137F, -99.7657F, 18.9475F, -18.6283F, 52.5484F, -49.6458F,
   46.4725F, -26.2220F, -128.3100F, -152.1911F, -106.3316F, -212.7650F, -21.8051F,
   -22.1899F, -50.6587F, -4.0987F, -2.7160F, -63.9780F, -18.1671F, -8.5809F,
   384.5044F, 415.7100F, 375.3419F, 221.7875F, 82.0605F, -10.5889F, 6.1831F,
   316.8826F, 269.9225F, 294.6225F, 200.5877F, 87.3740F, -13.6950F, 64.9923F,
   258.7122F, 274.8100F, 241.2115F, 150.6316F, 22.2879F, -82.2525F, -100.3677F,
   199.2028F, 140.2225F, 163.4630F, 153.3341F, 35.3954F, -56.3718F, -6.8714F,
   -75.2059F, -12.0741F, -10.5019F, 87.1330F, 34.8811F, -55.7089F, 9.3485F,
   -43.9131F, -38.3306F, 184.0508F, 124.7862F, 72.2168F, 1.2740F, 68.2432F,
   98.5331F, 56.9808F, -18.2386F, 106.3234F, 18.7400F, -82.2337F, -91.2981F,
   56.8543F, 26.8486F, 79.2505F, 142.2529F, 26.5547F, -20.1961F, 20.0251F,
   -101.2805F, -186.8430F, -272.0712F, -136.2303F, -296.2366F, -195.7741F, 24.4320F,
   -195.6167F, -300.4726F, -112.7403F, 30.7313F, -121.6459F, -194.0080F, -41.2612F,
   98.2285F, 20.7275F, -91.7240F, -183.2970F, -349.5879F, -345.3007F, 56.7721F,
   -162.4512F, -215.0840F, 54.6896F, -51.9437F, -204.1752F, -302.6133F, 17.1211F,
   -127.3559F, -165.3954F, -278.6780F, -194.2370F, -33.4488F, -168.4021F, 38.6555F,
   -295.7049F, -301.6402F, -62.3113F, -29.8345F, -17.8150F, -12.4093F, 147.9211F,
   33.3799F, -13.6109F, -78.2303F, -156.7363F, -259.0581F, -406.3858F, -96.3685F,
   -70.3287F, -128.8603F, -35.8870F, -146.8926F, 10.7114F, -71.6699F, -42.8127F,
   499.4422F, 424.3595F, 305.0398F, 196.0019F, 126.5760F, 63.9795F, 61.1516F,
   266.3734F, 181.9720F, 175.4860F, 108.0995F, 56.9864F, 10.7695F, 159.6035F,
   247.9333F, 184.5145F, 152.3734F, 41.9318F, -90.6959F, -210.2689F, -54.1531F,
   76.7519F, 154.5846F, 129.7565F, 83.0209F, 81.7690F, -11.2691F, -31.6182F,
   58.5084F, -37.0541F, -121.7467F, 29.7202F, -30.0062F, -124.9186F, -19.6730F,
   18.1514F, 3.4725F, 18.4977F, 68.3406F, 54.2768F, 11.6546F, 150.1885F,
   85.0507F, 80.6332F, -9.5051F, -97.5608F, -37.4477F, -107.8952F, -71.4574F,
   10.1458F, 46.5077F, 14.7448F, 5.4868F, -1.5563F, -51.9743F, -20.1740F,
   -186.2883F, -239.8358F, -385.3964F, -279.2732F, -246.3584F, -317.5150F, 77.9409F,
   -290.8109F, -394.1977F, -416.1625F, -291.0747F, -246.4900F, -66.6275F, 7.0910F,
   -53.3572F, -98.2187F, -188.6130F, -239.5862F, -382.9682F, -469.8791F, 4.8130F,
   -236.2064F, -299.8000F, -219.3447F, -267.9179F, -424.8446F, -200.9900F, 22.5746F,
   -78.8698F, -167.2648F, -305.5687F, -418.8565F, -37.5231F, -91.3323F, -69.9264F,
   -187.4632F, -276.9708F, -417.8298F, -350.3085F, -51.2638F, -35.7869F, 125.5794F,
   -56.7695F, -75.4877F, -200.1077F, -284.6027F, -223.5758F, -298.6907F, -81.9158F,
   -129.2158F, -228.6000F, -204.9319F, -246.4877F, -109.8989F, -81.0975F, -118.3144F,
   455.1133F, 431.3343F, 366.9250F, 323.3162F, 279.7654F, 166.3696F, 60.4075F,
   2.2427F, -42.4551F, 320.2264F, 303.9626F, 161.9218F, 44.0682F, -10.4932F,
   148.9215F, 132.3074F, 122.6564F, 103.3307F, 47.4590F, -3.4202F, -185.0445F,
   -145.6115F, 18.4827F, 228.0817F, 172.5267F, 38.5907F, -90.1562F, -43.9888F,
   -193.6085F, -14.9922F, -111.3547F, 102.4688F, 119.0297F, 66.3613F, 1.6199F,
   -229.1052F, -217.2523F, 26.9406F, 25.7113F, 183.7947F, 138.0997F, 4.9246F,
   -54.0735F, 27.3595F, -6.9751F, 41.3897F, 26.0272F, 65.3047F, -117.9287F,
   -173.6861F, -55.0424F, 72.3582F, 74.4772F, 59.1275F, 32.2629F, -52.0966F,
   -100.2805F, -203.3997F, -281.1161F, -77.1917F, 52.4149F, -18.1558F, -70.9170F,
   -319.7560F, -456.2388F, -303.5891F, 199.8376F, 140.0785F, 37.0556F, -44.1020F,
   53.3338F, -0.8373F, -29.0162F, -53.8806F, -156.4895F, -44.6551F, -144.9031F,
   -94.4005F, -181.9505F, 79.8505F, 87.0198F, -12.1349F, -82.6727F, -75.0313F,
   -207.6459F, -248.8868F, -181.9412F, -267.7931F, 100.0395F, 108.0778F, -54.3928F,
   -445.8355F, -486.1193F, -148.8008F, -29.2654F, -20.9138F, 5.3055F, 34.4656F,
   -121.3450F, 45.0588F, -55.9190F, -194.9437F, -134.0013F, 48.6244F, -162.3284F,
   -240.5215F, -57.3889F, -52.6141F, 16.9554F, -1.6593F, 47.0915F, -144.2315F,
   326.1222F, 306.0203F, 250.4581F, 152.3629F, 138.5136F, 136.6061F, 24.8198F,
   27.9826F, 75.7094F, 172.8544F, 182.3980F, 170.1943F, 85.7041F, 145.3514F,
   181.4035F, 144.2242F, 74.8556F, 1.4272F, -61.0836F, -28.1913F, -88.2541F,
   22.2453F, 80.4679F, 118.3400F, 80.0672F, 73.2458F, 39.5447F, 103.6288F,
   -15.5237F, -80.2624F, -190.5397F, -72.0487F, 107.7333F, 24.8407F, -9.9839F,
   -84.5773F, -169.2563F, -115.8907F, 53.5535F, 128.0731F, 120.4292F, -112.8952F,
   147.5641F, 31.5040F, -120.2366F, -181.4932F, 22.0867F, -14.3306F, 88.8276F,
   -38.8943F, 9.5208F, -14.2124F, -59.1263F, 77.8884F, 3.1099F, 55.9227F,
   -166.2941F, -272.4271F, -413.5079F, -51.9034F, 151.3775F, 62.8211F, 125.4737F,
   -251.4761F, -399.5025F, -496.4470F, 15.1387F, 25.6129F, -47.9447F, 27.2588F,
   145.3730F, 69.2023F, -67.5157F, -217.2383F, -216.1160F, -118.5795F, -73.6690F,
   -45.7917F, -144.0127F, -166.1374F, -47.5218F, -180.4311F, -59.2610F, -28.7005F,
   -253.6976F, -332.1650F, -464.5507F, -299.7163F, 194.1511F, 140.7432F, 86.8261F,
   -311.8663F, -444.9106F, -568.2597F, -74.2214F, 168.0938F, 91.5796F, 69.8801F,
   16.2576F, -36.8680F, -158.1385F, -270.2551F, 14.7501F, -79.0270F, -96.1519F,
   -66.0490F, -143.0125F, -113.3699F, -34.3890F, -25.0086F, 12.5482F, -114.0537F,
   394.4650F, 360.5732F, 366.5918F, 277.4828F, 181.5850F, 89.9984F, 133.6822F,
   84.7064F, 117.2216F, 411.7273F, 263.7720F, 26.7575F, -178.8131F, -69.9364F,
   30.5200F, 177.1075F, 123.5531F, 7.2231F, -92.4543F, -193.6012F, -94.9796F,
   65.2717F, 8.1559F, 165.7135F, 84.0440F, -35.7379F, -118.5377F, -66.5232F,
   -113.0977F, -31.5885F, -27.3286F, -26.0692F, 15.5114F, 21.2791F, -23.2414F,
   -138.0497F, -166.9724F, -35.1916F, 20.7464F, 36.6236F, 3.5627F, 20.9882F,
   -11.0900F, 124.4114F, 110.4337F, 5.6888F, -93.7853F, -22.5026F, 30.8623F,
   -29.4226F, -47.2009F, 89.3512F, 13.6996F, 22.5131F, 2.3075F, -51.7825F,
   -137.2009F, -200.8273F, -290.6017F, -325.7113F, -275.5810F, -53.4554F, 64.1025F,
   -438.9569F, -542.2731F, -412.1423F, -89.9884F, -28.4069F, 15.6647F, 13.8672F,
   -63.4932F, -90.8672F, -207.9037F, -306.7191F, -456.9608F, -272.1003F, 37.6908F,
   -307.9492F, -374.7438F, -96.2506F, -168.0490F, -243.8695F, -64.5296F, -3.1349F,
   -334.4494F, -210.9558F, -246.0072F, -354.2072F, -131.6393F, -27.1645F, 29.5091F,
   -366.0860F, -407.0811F, -257.3381F, -286.1801F, -81.4441F, 77.6511F, -10.1480F,
   -204.4162F, 30.5330F, -38.3922F, -195.6373F, -339.5407F, -347.7475F, -31.6897F,
   -243.6551F, -274.7409F, -17.3959F, -136.0264F, -9.8248F, 34.1080F, -72.7180F,
   392.4115F, 321.3055F, 213.3703F, 97.3249F, 35.2497F, -8.7278F, 80.8223F,
   116.4944F, 155.2541F, 148.9464F, 124.6089F, 78.0606F, 24.1574F, 118.5191F,
   184.6981F, 148.0850F, 51.9562F, -96.3499F, -247.0329F, -172.3801F, 29.4305F,
   110.9024F, 81.4259F, 152.8654F, 45.1114F, -9.8257F, 17.3401F, 36.9414F,
   -27.9329F, 30.8783F, -37.1648F, -24.5270F, -50.2764F, -114.4973F, 79.3175F,
   -64.1897F, -136.2614F, 44.8825F, 47.6375F, 26.8591F, -0.2729F, 147.6527F,
   135.7539F, 66.4931F, 38.8074F, 22.5599F, -70.1020F, -117.9085F, 73.4314F,
   -74.1601F, -0.9165F, 58.6634F, 8.8413F, -20.0486F, -4.3857F, 84.5215F,
   -240.3710F, -336.9822F, -425.7879F, -557.5921F, -313.4519F, -18.6503F, -2.2911F,
   -375.5428F, -498.5622F, -604.7405F, -284.0899F, -22.5439F, 21.7506F, 87.1215F,
   -48.5853F, -99.7448F, -219.1610F, -385.9877F, -498.9206F, -58.0072F, -32.0391F,
   -187.4733F, -257.8106F, -348.0662F, -465.0402F, -523.0202F, -126.9367F, 7.9922F,
   -242.4066F, -358.0976F, -506.2984F, -527.3067F, 103.2345F, 112.8789F, 19.5721F,
   -329.7812F, -399.2909F, -525.3788F, -519.3241F, -114.6023F, 18.0531F, 108.4014F,
   -101.3959F, -182.8774F, -323.7689F, -459.4139F, -208.2967F, -218.5872F, -30.6618F,
   -83.9128F, -188.9391F, -321.0572F, -481.5454F, -309.1473F, 5.7541F, -53.7133F,
   508.3954F, 514.6412F, 492.8828F, 409.4624F, 279.6907F, 139.5316F, -25.0946F,
   225.7716F, 321.9763F, 361.4909F, 313.2012F, 246.2739F, 141.2217F, 40.0333F,
   183.5499F, 361.1524F, 299.5782F, 200.8792F, 127.5401F, 74.2915F, -43.7510F,
   87.8314F, 192.1731F, 193.5542F, 170.6556F, 233.6155F, 150.0142F, -17.3959F,
   16.6972F, 31.8549F, -16.4255F, 68.2118F, 202.5792F, 133.4167F, -24.8757F,
   -118.5934F, -60.2884F, 62.2425F, 224.9154F, 157.3277F, 86.2001F, 127.8335F,
   83.3985F, 95.4353F, 47.7289F, -24.4079F, 122.0239F, 35.9347F, -62.3200F,
   85.0292F, 81.2156F, 121.2625F, 104.6399F, 119.6594F, 126.2998F, -77.7217F,
   -111.4887F, -105.1822F, -30.9685F, -48.2729F, -55.9544F, -59.5831F, 92.3539F,
   -80.0937F, -149.8609F, -166.3557F, 134.5224F, 22.8807F, -72.2311F, -71.7019F,
   -106.2400F, 96.5443F, 49.1033F, -82.1511F, -202.4001F, -182.6395F, 30.4307F,
   -38.6498F, -47.0332F, 196.2347F, 62.6390F, -109.4412F, -217.5327F, 118.6330F,
   -165.3289F, -6.5073F, -110.5468F, -172.9718F, 83.9014F, 82.6244F, -54.8262F,
   -220.1304F, -267.2490F, -88.9432F, 199.2986F, 140.8517F, 50.8656F, 126.1685F,
   -167.2625F, 130.8662F, 38.2409F, -25.2887F, 14.1618F, -84.0247F, -96.2550F,
   -141.5362F, 19.9235F, 142.0336F, -6.1892F, -91.4426F, 58.7228F, -64.2743F,
   356.9441F, 345.0619F, 293.3749F, 255.2251F, 264.2440F, 181.1552F, -59.9296F,
   246.4391F, 212.9810F, 208.9891F, 281.1677F, 232.7401F, 140.5526F, -31.7890F,
   305.7318F, 278.3117F, 199.8758F, 171.6015F, 133.0515F, 98.0317F, -152.6052F,
   194.0116F, 162.7764F, 97.4355F, 145.3808F, 115.4215F, 67.7161F, -75.3193F,
   50.0116F, -43.0169F, -150.4608F, 36.4008F, 15.3428F, 67.7103F, -90.2827F,
   84.1801F, -22.9641F, -98.8488F, 180.4238F, 151.9562F, 59.9903F, 26.9906F,
   177.5510F, 105.3890F, -1.4605F, -131.7380F, -19.1906F, 90.9329F, -125.9315F,
   119.9137F, 55.8859F, 20.5790F, 85.2794F, 50.1218F, 92.1433F, -60.8729F,
   -14.4439F, -72.6866F, -93.8542F, -160.4042F, -42.8567F, 3.3977F, 156.0409F,
   -54.5578F, -142.4629F, -245.2400F, 100.6745F, 148.4848F, 83.6861F, 108.3569F,
   51.3106F, 177.4475F, 69.0944F, -111.1996F, -202.9647F, 42.3005F, -83.6978F,
   30.1802F, -55.2034F, 78.5647F, -47.9833F, -191.5273F, -51.2293F, 23.2575F,
   -5.5372F, -75.8171F, -203.4011F, -289.5156F, 106.7613F, 141.4280F, -13.0810F,
   -141.4010F, -221.2458F, -112.2780F, -89.1527F, 63.0527F, 92.4026F, 123.8935F,
   -105.1256F, 76.2826F, -29.3279F, -146.3319F, -79.8275F, 4.2937F, 30.2187F,
   -50.0448F, -103.1965F, -61.1121F, -87.1858F, -185.4814F, 103.5567F, 10.4906F,
   483.7709F, 447.9613F, 357.9065F, 269.2391F, 168.5860F, 81.9800F, -109.5565F,
   312.3829F, 259.3732F, 343.4722F, 283.4192F, 160.6555F, 62.4472F, -64.0786F,
   177.4189F, 268.0235F, 209.2622F, 107.3129F, 57.2866F, 17.7874F, 43.3733F,
   118.3413F, 189.8245F, 200.1482F, 138.5254F, 93.9936F, 70.6362F, -26.8448F,
   30.9857F, 35.8345F, 11.3604F, 57.5567F, 90.9626F, 20.1319F, 6.8180F,
   -66.7255F, 112.7395F, 51.2972F, 171.5931F, 105.7922F, 1.0289F, -4.0888F,
   161.4374F, 94.0077F, 56.1444F, 67.9007F, 60.5061F, -0.5993F, 5.5333F,
   22.6145F, 84.7579F, 96.3842F, 85.7332F, 58.1820F, 68.5529F, -7.0445F,
   89.2476F, -2.9310F, -93.5695F, -80.5243F, -200.4216F, -74.7223F, 78.7932F,
   -37.6075F, -106.1347F, -177.1898F, 12.7187F, -117.0868F, -224.7555F, 18.0710F,
   -3.9184F, -42.1867F, -96.2393F, -135.5441F, -300.7351F, -209.1140F, 3.3638F,
   -127.2731F, -168.0130F, -23.6514F, -125.4722F, -275.0623F, -116.8249F, 78.4081F,
   -0.3268F, -109.2929F, -235.6517F, -163.5220F, -72.4472F, 7.1522F, -14.1950F,
   -201.5567F, -203.8994F, -147.8831F, -113.7655F, -70.6486F, -37.9403F, 10.4648F,
   -61.9333F, -122.2785F, -205.0957F, -87.3765F, -199.0248F, -324.4658F, -76.9686F,
   -114.1747F, -179.4460F, -22.8678F, -120.9749F, -219.5858F, -55.7430F, -137.0347F,
   416.5525F, 356.2247F, 246.4192F, 151.7590F, 80.9813F, 33.7425F, -87.5668F,
   232.7198F, 199.8298F, 232.8017F, 197.1352F, 122.7492F, 47.7377F, -18.8607F,
   307.4450F, 336.4361F, 172.1882F, -36.4639F, -118.2980F, 12.7783F, -74.1220F,
   100.4413F, 183.0223F, 131.6807F, 19.7279F, 5.8399F, 116.3680F, -36.1637F,
   115.0603F, 28.3727F, -46.6535F, 9.9458F, 14.2165F, 8.8947F, 42.1125F,
   -11.0346F, -1.0867F, -12.6705F, 44.5022F, 8.2729F, 72.2938F, 41.9570F,
   188.4713F, 119.1476F, 5.8222F, -109.1441F, -124.7922F, 20.2470F, 37.4241F,
   88.4120F, 18.1026F, 44.8535F, 28.1015F, -53.1688F, 52.0384F, -48.7039F,
   7.0136F, -80.7959F, -218.2056F, -230.6367F, -196.4241F, -188.3921F, 125.5651F,
   -223.0832F, -308.3413F, -371.4560F, -67.3406F, -172.4539F, -210.1611F, 44.0330F,
   70.5425F, 5.5613F, -121.1026F, -250.2220F, -391.0280F, -78.8702F, 62.0359F,
   -53.2080F, -153.5175F, -168.9519F, -200.2387F, -356.0113F, -56.1841F, -8.4833F,
   28.1989F, -103.3445F, -275.6373F, -372.5401F, -62.2850F, 20.7805F, 129.8285F,
   -131.8626F, -230.9837F, -332.7086F, -149.5865F, -154.3909F, 21.7068F, 17.3275F,
   16.7300F, -61.8863F, -181.1369F, -350.5149F, -246.6439F, -61.1861F, -114.9767F,
   -62.9283F, -50.0990F, -128.3168F, -223.8037F, -188.6130F, -48.9473F, 4.2910F,
   397.1338F, 354.0856F, 410.7048F, 392.6840F, 265.0492F, 133.1239F, -39.9595F,
   140.3398F, 130.2926F, 612.1718F, 513.2461F, 282.4946F, 41.1820F, 3.7939F,
   -25.5253F, 283.5953F, 206.7656F, 117.4847F, 85.9801F, 16.9447F, -106.7023F,
   -199.2065F, 170.3263F, 280.7031F, 221.1866F, 139.8684F, 69.7774F, 44.7026F,
   -218.6125F, 116.3745F, 51.8822F, -46.6960F, 133.6225F, 87.8249F, 5.3826F,
   -351.2338F, -39.4894F, 84.6148F, 241.0717F, 156.3851F, 57.0277F, -22.5485F,
   -16.1558F, 137.3482F, 31.9090F, -76.7114F, 15.3505F, 47.2591F, -65.0792F,
   -82.6497F, 23.1536F, 174.7281F, 111.0391F, 165.1666F, 97.6276F, -66.0758F,
   -144.8786F, -188.8722F, -232.9240F, 11.8115F, -15.8240F, 97.1230F, 8.4134F,
   -145.3324F, -253.9705F, -310.2223F, 186.1852F, 173.7029F, 84.4147F, -31.0347F,
   -137.6416F, 107.5353F, 6.4667F, -172.8744F, -362.7197F, 36.0251F, 7.3423F,
   -273.1752F, -271.7551F, 168.5304F, 103.1478F, -19.2394F, -121.0773F, 71.0605F,
   -382.1200F, -84.7432F, -167.9304F, -156.0550F, 39.5436F, 102.1275F, 28.0496F,
   -362.7133F, -347.0801F, -139.5875F, -5.2563F, 181.0231F, 147.5667F, 92.1709F,
   -322.5983F, 22.0118F, 11.8984F, -116.7982F, -140.8088F, -50.1710F, -20.0495F,
   -342.8153F, -116.9352F, -43.2698F, 29.1059F, 25.4655F, 14.1526F, 1.1041F,
   319.3257F, 143.9371F, -9.8398F, 138.1817F, 272.2951F, 230.7682F, -67.6844F,
   120.5561F, 82.8050F, 86.9118F, 293.3890F, 188.8921F, 54.7609F, -21.7199F,
   231.6328F, 148.0905F, 27.2085F, -3.2337F, 100.6721F, 70.2593F, 42.8315F,
   64.4154F, -17.9857F, 99.5439F, 116.0344F, 85.7395F, 106.6537F, 41.0177F,
   62.6236F, -27.4056F, -139.7237F, -70.5707F, 148.2318F, 128.9077F, 124.7000F,
   -92.9236F, -144.0688F, -77.1586F, 288.9071F, 226.4347F, 124.6281F, -55.6669F,
   78.3503F, 42.4807F, -79.0042F, -178.3122F, 128.1446F, 107.8855F, -55.1806F,
   79.1238F, 66.4433F, 43.2504F, 16.2992F, 80.8281F, 140.7251F, 73.5809F,
   -134.6119F, -155.8599F, -200.3194F, -90.9865F, 253.7969F, 164.9683F, 31.5576F,
   -248.1197F, -375.5200F, -476.6351F, 180.7157F, 210.6575F, 79.8609F, 81.7351F,
   78.7261F, -9.7665F, -75.8617F, -194.4922F, -307.4282F, 75.4347F, -56.0501F,
   -175.4331F, -210.0029F, 75.9892F, -41.3638F, -129.4197F, 26.7040F, 24.1293F,
   -89.1388F, -186.3199F, -327.9780F, -225.0781F, 123.8019F, 49.8056F, 110.5614F,
   -262.0009F, -365.9398F, -315.0958F, -44.6833F, 152.6366F, 130.9091F, -24.3555F,
   96.1914F, 0.9431F, -122.7709F, -265.0437F, -118.4077F, 70.4388F, -34.3324F,
   -41.7155F, -106.2450F, 35.3080F, -77.1729F, -11.0485F, 113.3317F, -43.5801F,
   345.8234F, 448.9382F, 398.8093F, 278.1467F, 193.7783F, 109.9307F, -32.5084F,
   105.2215F, 192.0218F, 307.4095F, 251.2213F, 135.6460F, 27.1659F, 48.2501F,
   131.1255F, 248.6208F, 194.8135F, 35.9742F, -89.6413F, -70.1114F, 74.4933F,
   10.3277F, 157.1341F, 220.2146F, 132.8669F, 58.2022F, -4.2414F, 52.5998F,
   -171.9997F, 219.1700F, 183.7592F, 79.7388F, -28.2743F, -72.7875F, 54.8047F,
   -281.0813F, 5.4272F, 138.6439F, 89.6476F, 26.7423F, -52.0621F, 181.8158F,
   42.9908F, 121.1063F, 40.1449F, 4.6961F, 41.3091F, 11.3680F, 61.7374F,
   -69.6739F, 44.0933F, 62.9511F, 55.3414F, 47.1718F, 45.0520F, 10.5398F,
   -254.2462F, -161.6377F, -252.0311F, -59.3082F, -64.1228F, -5.1355F, -167.6060F,
   -249.7012F, -368.7658F, -266.1853F, -40.5172F, -31.5949F, 20.3852F, -73.7292F,
   -259.0964F, -63.9727F, -123.4555F, -266.6797F, -372.6790F, -79.9634F, -32.2913F,
   -237.8991F, -261.1591F, -67.3100F, -201.9258F, -271.1355F, 110.5556F, -31.4253F,
   -252.1870F, -59.5241F, -149.7008F, -264.8359F, -95.9333F, -15.7623F, -55.9292F,
   -291.6127F, -310.8436F, -291.0997F, -152.3425F, -38.1903F, 16.9302F, 107.5521F,
   -234.1635F, -77.8775F, -155.4929F, -169.1375F, -170.1633F, -210.0618F, 12.2871F,
   -432.5516F, -222.0327F, -146.7204F, -103.4423F, -49.3643F, -8.0516F, 0.3778F,
   330.7150F, 223.7578F, 120.2566F, 152.9689F, 119.1688F, 39.3462F, 21.3169F,
   106.4649F, 114.2936F, 78.3297F, 104.4772F, 135.1724F, 78.5670F, 48.1339F,
   284.0033F, 187.5895F, 88.2418F, 0.6583F, -25.3445F, -53.7671F, 63.9802F,
   106.5275F, 81.1295F, 67.7705F, 114.2483F, 4.6206F, 98.6728F, 68.0557F,
   132.7267F, 80.9311F, 70.2013F, 34.2614F, 45.4356F, 17.9487F, 185.5185F,
   -179.6926F, -64.9953F, -11.8397F, 5.1401F, 42.9932F, 67.5337F, 154.7699F,
   53.6060F, 34.2805F, 16.7478F, -53.4059F, -36.7348F, 13.1978F, 110.7422F,
   37.6082F, -5.5267F, 88.1530F, 15.7336F, 24.4194F, 27.1976F, 50.6641F,
   -182.5485F, -287.0959F, -395.1311F, -458.4781F, -119.8993F, 119.0388F, -26.2634F,
   -239.2084F, -359.3420F, -447.6111F, -221.6196F, -3.2312F, 35.8538F, -9.6860F,
   -70.9599F, -105.8073F, -192.0091F, -294.5047F, -334.6240F, 111.9638F, -29.3527F,
   -193.0902F, -307.5872F, -268.9553F, -266.8591F, -276.8985F, 53.4395F, -127.3159F,
   -112.3830F, -214.7503F, -352.5892F, -431.6330F, 130.6462F, 125.3650F, -24.3701F,
   -138.4310F, -254.9568F, -325.5955F, -173.9416F, 0.3136F, 147.4202F, -59.5469F,
   -63.4081F, -119.3462F, -237.6156F, -366.1194F, -116.1935F, 96.5096F, -74.8714F,
   -91.5009F, -186.5192F, -95.3213F, -190.6698F, -103.7853F, 133.5730F, -129.6124F
};

/*
 * 2nd stage codebook; 1st split: isf2_0 to isf2_2
 */
const Float32 E_ROM_dico21_isf[SIZE_BK21 * 3] = {
   128.4688F, 159.8148F, 97.2755F,
   -12.9836F, 197.1542F, 62.3669F,
   -11.1370F, -5.4038F, 227.1786F,
   -102.5247F, 49.5302F, 138.2452F,
   56.6014F, 92.4597F, 68.2462F,
   -59.4614F, 95.8021F, 47.7671F,
   10.6214F, 16.4206F, 132.7432F,
   -33.0052F, -36.4880F, 121.6274F,
   111.3621F, 86.8573F, -60.9301F,
   18.4019F, -16.9460F, -196.8595F,
   91.2159F, 47.1568F, 150.3417F,
   40.7444F, -123.8314F, 17.6656F,
   68.7560F, 76.0870F, 3.0834F,
   40.5189F, -22.9558F, -36.8239F,
   68.9463F, 20.8592F, 75.1641F,
   -13.4297F, -49.7472F, 59.4394F,
   222.6512F, 108.2263F, -13.1594F,
   -26.2678F, -128.5232F, -249.6796F,
   -61.3770F, -106.3253F, 180.3803F,
   -69.2195F, -180.5594F, 77.3840F,
   125.7118F, 69.8739F, 44.7696F,
   -150.6344F, 66.9866F, 7.4132F,
   7.5039F, -4.7150F, 76.2746F,
   -46.9159F, -98.2953F, 78.4325F,
   118.5603F, 14.1695F, -131.2373F,
   -50.1565F, -86.1739F, -148.3825F,
   66.6965F, -72.1903F, 115.4583F,
   -94.5599F, -121.9918F, 8.8523F,
   77.1679F, 15.2443F, 6.1163F,
   -1.0987F, -69.2605F, -43.3316F,
   43.3928F, -36.3404F, 29.4998F,
   -36.0829F, -87.1134F, 1.6094F,
   69.0548F, 158.6352F, -17.3680F,
   -65.7580F, 148.4039F, -58.1224F,
   -1.4207F, 106.6547F, 129.4115F,
   -163.9787F, 200.5753F, 108.3887F,
   8.0981F, 96.4962F, 18.4523F,
   -22.7513F, 51.0308F, -0.8203F,
   -1.3389F, 52.2067F, 70.5057F,
   -56.5316F, 15.5148F, 68.5082F,
   73.6822F, 28.9345F, -56.5083F,
   -10.5212F, -17.6820F, -126.8611F,
   144.6737F, -44.4661F, -8.2446F,
   -32.2712F, -161.9950F, -67.5138F,
   29.9266F, 37.2667F, -20.0451F,
   -15.7787F, -11.7016F, -26.2432F,
   27.7648F, 34.4647F, 33.4065F,
   -13.8614F, -38.4420F, 5.6107F,
   27.0098F, 76.8588F, -130.2875F,
   -76.5280F, 30.9527F, -90.1727F,
   -135.8326F, -53.3710F, 85.3346F,
   -137.4852F, -34.8750F, -33.3215F,
   18.2807F, 78.3958F, -50.7970F,
   -64.5101F, 14.5083F, -5.6925F,
   -16.6084F, 1.0141F, 33.5889F,
   -63.0082F, -42.0433F, 31.0393F,
   32.2577F, 8.3123F, -92.4112F,
   -31.6346F, -58.2893F, -92.9305F,
   58.7086F, -72.6659F, -98.0877F,
   -72.8089F, -97.1838F, -63.2264F,
   -7.5866F, 25.6299F, -54.3822F,
   -10.3321F, -19.4552F, -70.7729F,
   9.4725F, 4.3403F, 0.0721F,
   -50.7563F, -41.0679F, -38.2745F
};

/*
 * 2nd stage codebook; 2nd split: isf2_3 to isf2_5
 */
const Float32 E_ROM_dico22_isf[SIZE_BK22 * 3] = {
   -49.5271F, 121.1295F, 16.4769F,
   -94.4022F, 76.8061F, 2.1280F,
   -58.8683F, 32.7926F, -6.6703F,
   -83.7461F, 49.5749F, -58.1205F,
   -96.5332F, -51.1816F, 62.2310F,
   -104.6874F, -104.4218F, -36.9709F,
   -84.8023F, 0.4441F, -31.0082F,
   -105.8907F, -31.0867F, -72.0962F,
   -17.4561F, 170.2726F, 62.0341F,
   64.3231F, 77.6492F, 152.6545F,
   -13.0558F, 31.6496F, 73.1381F,
   -25.9540F, -16.3235F, 138.6579F,
   -116.2904F, -22.1347F, 133.9606F,
   -42.0923F, -209.9028F, 88.4509F,
   -56.0897F, -9.1734F, 75.5450F,
   68.7883F, -157.0452F, 33.8343F,
   20.5838F, 115.6837F, 9.8136F,
   -32.8948F, 98.7358F, -40.4656F,
   -22.8099F, 41.1383F, -49.3496F,
   -66.1700F, 67.9190F, -122.8227F,
   -18.8196F, 17.0630F, -114.9884F,
   -64.0458F, -162.7084F, -94.5118F,
   -54.2084F, 1.0923F, -75.9526F,
   -60.5415F, -80.9324F, -82.4130F,
   46.5760F, 125.8621F, 83.1458F,
   129.9934F, 19.7027F, 148.2803F,
   92.6546F, 96.4434F, -0.8040F,
   181.9977F, -6.0791F, 78.6212F,
   93.0392F, -99.5781F, -41.6073F,
   26.0437F, -172.0070F, -58.1634F,
   47.8475F, -34.2895F, -54.2498F,
   34.4187F, -96.6345F, -28.5027F,
   -15.8270F, 90.1190F, 65.2787F,
   -24.1274F, 60.7006F, 6.3059F,
   -25.2307F, 6.3412F, 30.0794F,
   -26.6845F, -0.7965F, -24.5917F,
   -59.0614F, -117.3599F, 62.4002F,
   -7.0821F, -130.0555F, 20.9710F,
   -21.7086F, -36.6794F, 1.9468F,
   0.6440F, -74.2564F, 5.5161F,
   35.9276F, 57.9971F, 81.4502F,
   42.2272F, 3.3575F, 106.2218F,
   42.1565F, 13.6719F, 43.0135F,
   55.6116F, -33.1310F, 56.8126F,
   18.4410F, -61.4055F, 109.0841F,
   1.0223F, -125.1248F, 96.1657F,
   16.9404F, -28.1707F, 26.5748F,
   33.5245F, -84.6827F, 52.6704F,
   14.2066F, 54.8716F, 30.7506F,
   21.9479F, 68.4119F, -19.0589F,
   10.1034F, 17.7161F, 1.1122F,
   28.3594F, 21.5207F, -39.3190F,
   42.6619F, -71.6784F, -94.4716F,
   -1.6345F, -110.6804F, -94.5938F,
   18.7091F, -26.5385F, -18.5676F,
   -2.2267F, -59.7051F, -47.6606F,
   63.0633F, 76.6895F, 37.6845F,
   90.5993F, 31.2041F, 74.2520F,
   64.4418F, 37.7231F, 4.2748F,
   100.9151F, -12.1708F, 27.5607F,
   104.1606F, -30.0584F, -35.7097F,
   121.5339F, -81.6812F, 34.1503F,
   59.3840F, -5.5457F, -8.5514F,
   58.5319F, -58.3782F, 3.4649F,
   -126.3823F, 217.4185F, 72.9956F,
   -149.8116F, 119.9919F, 17.7920F,
   -98.1555F, 10.6050F, 30.1117F,
   -142.4950F, 30.1718F, -20.4500F,
   -188.3500F, -32.7229F, 62.5055F,
   -165.4842F, -201.2336F, -25.1871F,
   -114.8346F, -46.6803F, -1.4401F,
   -185.9166F, -45.3806F, -42.6756F,
   -37.7442F, 124.2892F, 142.7720F,
   41.5423F, 244.7676F, 173.7613F,
   -74.1037F, 46.8818F, 112.0588F,
   -57.0922F, 25.5056F, 241.9654F,
   -166.9268F, 94.6864F, 141.8443F,
   -141.0368F, -145.0649F, 168.9254F,
   -135.5530F, 39.8468F, 65.4743F,
   -245.6494F, 76.1803F, -5.6309F,
   -25.5608F, 185.7902F, -18.2246F,
   -116.1471F, 125.0659F, -65.5997F,
   -21.6448F, 139.2267F, -103.0516F,
   -152.7089F, 32.1737F, -111.7466F,
   -19.9673F, -12.1771F, -217.1499F,
   -69.4452F, -156.0378F, -228.9529F,
   -79.9513F, -19.3186F, -140.5787F,
   -134.0792F, -92.8863F, -131.5222F,
   85.7696F, 178.6823F, 22.5404F,
   219.0179F, 182.4547F, 100.9819F,
   132.7898F, 105.4102F, -65.5694F,
   175.6583F, 30.2382F, -109.3055F,
   23.2833F, 65.0492F, -161.3234F,
   51.9178F, -98.2708F, -192.0347F,
   84.2797F, 61.1831F, -113.4513F,
   110.0580F, 0.1775F, -193.4730F,
   -88.3674F, 114.5595F, 71.3481F,
   -61.4713F, 52.7833F, 47.8295F,
   -61.8477F, -23.0151F, 15.1016F,
   -52.0862F, -46.0690F, -37.7005F,
   -129.6202F, -120.6827F, 44.2216F,
   -62.4412F, -165.8667F, -2.2683F,
   -58.1508F, -82.2449F, 9.3842F,
   -31.2569F, -108.3441F, -35.0348F,
   -4.4143F, 48.6610F, 132.2242F,
   50.8942F, -27.6676F, 181.6500F,
   2.0066F, -17.6839F, 71.9731F,
   92.7139F, -37.1637F, 98.9732F,
   -54.3785F, -76.9789F, 115.8870F,
   -7.4354F, -117.3479F, 199.5254F,
   -24.6757F, -59.5165F, 54.3134F,
   97.6902F, -113.0194F, 131.1002F,
   48.4163F, 132.4722F, -58.7820F,
   13.3327F, 68.6117F, -81.0715F,
   66.7414F, 64.6961F, -45.2427F,
   36.7779F, 14.7160F, -89.5910F,
   29.2181F, -25.5238F, -132.3077F,
   -30.5052F, -80.0320F, -150.5560F,
   0.1157F, -11.7322F, -63.8467F,
   -21.6912F, -43.0765F, -94.6223F,
   125.3290F, 95.1208F, 75.9759F,
   197.4010F, 92.9319F, -0.5012F,
   123.8246F, 45.4969F, 25.5317F,
   120.7415F, 34.4155F, -28.8961F,
   176.6600F, -20.0903F, -19.4091F,
   130.6288F, -84.9368F, -113.4633F,
   82.2281F, 16.0388F, -59.4614F,
   93.1589F, -21.5846F, -101.7248F
};

/*
 * 2nd stage codebook; 3rd split: isf2_6 to isf2_8
 */
const Float32 E_ROM_dico23_isf[SIZE_BK23 * 3] = {
   -3.8995F, 58.9472F, 140.2168F,
   53.1422F, 116.3810F, 87.0383F,
   99.7094F, -40.4441F, 113.1331F,
   165.1496F, 2.2135F, 71.5223F,
   -105.3934F, -104.9144F, -38.2897F,
   -20.3247F, -32.1811F, 5.1214F,
   -32.0250F, -106.9634F, -37.7452F,
   35.0256F, -95.9288F, -28.2073F,
   -116.9343F, -27.2704F, 164.5334F,
   -34.2352F, 142.6884F, 167.9453F,
   72.9835F, -124.0776F, 148.8355F,
   148.2748F, 14.5359F, 190.7047F,
   -145.5305F, -123.4847F, 30.8224F,
   -120.2600F, -39.3306F, 2.0939F,
   -52.9049F, -176.3645F, 3.0609F,
   27.9487F, -164.3167F, -60.3281F,
   70.4088F, 66.5042F, -47.2043F,
   24.2507F, 69.1997F, -15.5014F,
   127.5028F, 31.2976F, -40.8233F,
   96.9136F, 102.7271F, -1.9768F,
   -65.7818F, -70.8663F, -86.5155F,
   -0.6444F, -8.8675F, -61.8361F,
   -5.3369F, -58.3089F, -47.3390F,
   46.4765F, -35.6284F, -57.4718F,
   46.3328F, 129.6813F, -59.6592F,
   18.9988F, 118.4842F, 13.3515F,
   172.5430F, -21.3132F, -26.8827F,
   84.7525F, 177.2668F, 22.5077F,
   -140.2432F, -73.0634F, -146.3583F,
   -16.3786F, 19.6063F, -107.1305F,
   -3.0698F, -104.1769F, -97.1330F,
   33.2378F, -33.4660F, -134.9635F,
   -29.8971F, -15.7433F, 134.7990F,
   34.9346F, 52.2504F, 85.7197F,
   60.8752F, -31.2562F, 62.5970F,
   42.3279F, 15.7348F, 45.3299F,
   -61.8761F, -80.5864F, 11.3831F,
   1.9118F, -12.4255F, 68.2023F,
   -25.4049F, -61.6688F, 57.1412F,
   21.3127F, -30.3874F, 28.3554F,
   -44.5637F, -86.8534F, 137.9897F,
   -18.2977F, 31.6219F, 82.2796F,
   19.0573F, -58.8750F, 104.6892F,
   41.1433F, 1.5028F, 117.9977F,
   -102.7489F, -51.4207F, 71.3300F,
   -59.0984F, -10.9389F, 78.5554F,
   -69.1486F, -120.0273F, 65.0256F,
   39.6252F, -86.2109F, 50.9690F,
   29.0714F, 22.6919F, -38.4256F,
   12.6360F, 17.2986F, 5.2183F,
   75.9445F, 11.6720F, -55.4942F,
   66.2898F, 37.5939F, 3.0594F,
   -53.0062F, -46.4272F, -35.4140F,
   -25.4219F, 2.9542F, -21.2949F,
   1.3392F, -73.4502F, 4.6640F,
   17.5926F, -24.6153F, -19.1614F,
   58.2567F, -8.0154F, -7.3521F,
   9.4616F, 56.2212F, 37.1474F,
   99.2497F, -8.5999F, 23.5678F,
   62.8908F, 76.5199F, 37.4883F,
   -61.5740F, -23.6494F, 18.8743F,
   -27.4404F, 12.7151F, 31.8660F,
   -8.8214F, -125.3200F, 22.5806F,
   60.3707F, -57.3931F, 1.9729F,
   -142.0781F, 128.1134F, 29.9265F,
   -8.1434F, 177.0072F, 67.7176F,
   -42.2467F, 31.9985F, 246.1330F,
   143.5303F, 102.5801F, 81.3050F,
   -117.3709F, -24.3191F, -68.7836F,
   -80.1727F, 55.9021F, -61.6700F,
   -65.9167F, -159.9841F, -103.0851F,
   100.4774F, -105.1623F, -39.0393F,
   -248.4434F, 113.0562F, -0.9632F,
   -114.2012F, 244.7349F, 67.3892F,
   -149.2039F, -141.9386F, 151.3480F,
   96.8171F, 204.6967F, 174.6282F,
   -203.4799F, -43.3805F, -41.7327F,
   -154.4080F, 46.2418F, -107.0591F,
   -133.7993F, -265.7036F, -48.9596F,
   -67.3323F, -174.6485F, -259.0245F,
   29.4362F, 57.7183F, -143.2330F,
   -30.9358F, 102.7735F, -36.8946F,
   97.2809F, 57.7730F, -111.7326F,
   148.5371F, 105.7429F, -63.3760F,
   -55.6062F, -1.6898F, -72.6741F,
   -22.0825F, 43.4697F, -48.9841F,
   -13.6583F, -42.0695F, -99.1267F,
   39.0227F, 11.1441F, -94.4170F,
   -31.1674F, 118.4450F, -103.1298F,
   -30.5761F, 181.1050F, -22.4168F,
   96.8625F, -8.6867F, -192.7743F,
   258.0970F, 258.6485F, 17.2359F,
   -75.4867F, -15.7399F, -128.9883F,
   -69.4209F, 56.4875F, -131.4587F,
   -35.1219F, -77.9249F, -156.1559F,
   -15.4754F, -8.8922F, -194.7119F,
   -74.8584F, 44.6175F, 122.9601F,
   -15.9046F, 95.4389F, 74.2532F,
   34.5640F, -37.7631F, 189.4453F,
   93.9878F, 31.0970F, 82.8675F,
   -96.2834F, 15.6201F, 33.9921F,
   -60.9930F, 57.5930F, 52.2667F,
   -0.7661F, -130.2809F, 93.4385F,
   120.2708F, -79.2677F, 42.9508F,
   -179.2200F, 97.8633F, 164.7641F,
   -85.0375F, 121.2092F, 89.0752F,
   -33.4305F, -135.1092F, 255.3923F,
   71.7349F, 68.5372F, 165.8218F,
   -187.7999F, -24.7305F, 65.9037F,
   -136.4161F, 45.6197F, 73.4184F,
   -48.8887F, -218.9293F, 120.9648F,
   61.8728F, -162.4253F, 36.5784F,
   17.9416F, 66.9736F, -75.1315F,
   -24.6266F, 61.4069F, 5.3953F,
   100.0199F, -13.7281F, -105.7749F,
   125.9485F, 48.0431F, 20.6117F,
   -83.4273F, 1.6803F, -29.7397F,
   -60.9733F, 33.6154F, -6.9071F,
   49.9381F, -77.0001F, -90.5842F,
   103.7070F, -35.1051F, -38.3706F,
   -120.2797F, 129.5670F, -56.7507F,
   -51.1815F, 120.2600F, 22.5398F,
   198.7324F, 23.1592F, -132.3752F,
   219.6951F, 76.6881F, -5.4836F,
   -147.6706F, 39.1117F, -18.2240F,
   -91.4913F, 78.8627F, 0.3161F,
   40.5908F, -105.4075F, -192.5737F,
   124.6250F, -81.8382F, -127.0619F
};


/*
 * 2nd stage codebook; 4th split: isf2_9 to isf2_11
 */
const Float32 E_ROM_dico24_isf[SIZE_BK24 * 3] = {
   -30.8071F, -34.8458F, -1.7377F,
   -66.7509F, 30.0385F, -82.3340F,
   62.4748F, -75.5700F, 38.3392F,
   47.0287F, -40.0671F, 126.1063F,
   12.5178F, -8.7146F, -50.3954F,
   27.9620F, 30.3537F, -104.7235F,
   71.0260F, -29.6231F, -25.7014F,
   120.7753F, 38.6709F, -56.4990F,
   -89.5313F, -61.4613F, -32.9542F,
   -149.7260F, 38.4143F, -27.8365F,
   -35.1587F, -137.6189F, 4.6136F,
   -110.8198F, -69.6633F, 69.4652F,
   -25.4794F, -48.9702F, -64.6927F,
   -34.0250F, -68.5378F, -137.1167F,
   16.3769F, -77.3935F, -18.8382F,
   60.1539F, -54.6626F, -95.0395F,
   -29.9828F, 6.8644F, 42.3138F,
   -15.0985F, 138.6659F, 35.6532F,
   34.1357F, 3.3162F, 60.3957F,
   -1.5046F, 61.8542F, 93.5327F,
   49.8233F, 37.2486F, -21.0923F,
   2.6462F, 96.2433F, -48.4645F,
   100.8559F, 5.8835F, 34.9436F,
   80.3813F, 84.4933F, 38.3143F,
   -78.4639F, 3.3495F, 7.0416F,
   -121.9442F, 90.8917F, 79.7252F,
   -15.2997F, -67.8455F, 60.6317F,
   -56.1035F, -3.5687F, 110.7510F,
   -22.2741F, 27.4204F, -26.9833F,
   -61.1362F, 73.0974F, 7.1390F,
   21.0362F, -11.6205F, 9.0680F,
   9.3252F, 52.8599F, 21.3451F
};


/*
 * 2nd stage codebook; 5th split: isf2_12 to isf2_15
 */
const Float32 E_ROM_dico25_isf[SIZE_BK25 * 4] = {
   65.8574F, 55.5896F, -46.5520F, 44.9019F,
   80.4838F, -7.7875F, 36.7110F, 88.1867F,
   -41.5522F, 122.0924F, -8.3769F, 6.2406F,
   -24.0605F, 63.0705F, 27.8308F, 99.4548F,
   -34.8725F, 39.6270F, -72.1743F, 48.7995F,
   28.2859F, -11.6999F, -78.4246F, 134.1911F,
   -100.6717F, 12.7337F, -3.2716F, 31.4897F,
   -40.5514F, -60.1846F, 28.1000F, 115.5554F,
   56.1050F, -26.6471F, -104.7202F, -9.7237F,
   31.5196F, -30.3722F, -34.0636F, 41.4375F,
   8.5013F, 60.4295F, -72.4835F, -46.4772F,
   -18.1378F, -10.8741F, 10.5206F, 35.6699F,
   -44.7247F, -14.6405F, -68.4296F, -12.8925F,
   -36.6233F, -86.8592F, -73.8785F, 47.7951F,
   -51.6319F, -46.3776F, -74.4907F, -105.4981F,
   -67.2817F, -67.7186F, 6.9193F, -16.9668F,
   108.8168F, 52.7624F, -16.3049F, -50.0846F,
   72.9801F, -33.7268F, 89.4150F, -54.0673F,
   62.1588F, 93.8928F, 54.7414F, 17.8328F,
   26.7626F, 9.6218F, 88.5287F, 29.9594F,
   8.1440F, 44.9198F, 5.0806F, 2.9996F,
   26.4759F, -96.7328F, 49.0444F, 31.6496F,
   -58.6349F, 53.6024F, 80.8924F, -3.3883F,
   -60.0341F, -51.9482F, 112.9073F, 26.0056F,
   55.9483F, -14.6255F, -33.5049F, -127.3506F,
   70.1847F, -12.4499F, 7.3368F, -8.8635F,
   10.1728F, 65.7077F, 45.2101F, -90.9790F,
   -12.5750F, -10.0652F, 45.9762F, -30.2995F,
   1.2805F, -3.2898F, -17.5766F, -44.9828F,
   22.1316F, -84.0894F, -21.1065F, -32.5923F,
   -81.6581F, 43.6873F, -8.6894F, -65.0775F,
   -35.4129F, -58.9663F, 65.8078F, -102.2045F
};

/*
 * isf codebooks: two-stage VQ with split-by-3 in 2nd stage
 *                1st stage is kept the same as the 46 bit quantizer
 *
 *  codebook   vector dimension    number of vectors
 *  ~~~~~~~~   ~~~~~~~~~~~~~~~~    ~~~~~~~~~~~~~~~~~
 *    1_1            9                  256
 *    1_2            7                  256
 *    2_1            5                  128
 *    2_2            4                  128
 *    2_3            7                  64
 */
const Float32 E_ROM_dico21_isf_36b[SIZE_BK21_36b * 5] = {
   -20.3147F, -37.6762F, 82.8575F, 122.9161F, -28.4468F,
   31.9756F, -79.5400F, 141.6557F, 53.2923F, -77.0688F,
   -49.3360F, -129.1711F, 71.5448F, 85.1944F, 55.7117F,
   -19.1147F, -16.1797F, 217.6729F, 89.8943F, 28.1868F,
   0.9668F, -28.5091F, 63.7145F, 147.3305F, 86.3104F,
   52.0605F, 43.3415F, 108.6263F, 83.9071F, -42.8628F,
   -39.7734F, -7.6796F, 110.7755F, 44.0677F, 106.4663F,
   32.7239F, 124.4301F, 113.4192F, 7.2217F, 33.0503F,
   -9.5746F, -1.9619F, 48.6607F, 51.4268F, -79.8193F,
   -14.9729F, -2.0174F, 111.5604F, -3.5774F, -139.2041F,
   -54.5000F, -100.1092F, 35.8796F, 45.5500F, -73.7805F,
   -56.2694F, 74.4424F, 122.2707F, 20.1155F, -38.3887F,
   65.2094F, -3.9808F, 17.2638F, 96.6362F, 14.0115F,
   148.8982F, 77.1199F, 93.1280F, 28.7596F, 2.2577F,
   14.8324F, -159.4414F, 11.4260F, -1.0211F, -33.2422F,
   35.9112F, 103.8106F, 61.4044F, -9.7034F, -78.2548F,
   62.7658F, -47.4034F, 27.3267F, 32.6865F, -54.7569F,
   -6.2693F, -33.7779F, 43.5591F, -36.6678F, -73.8111F,
   -104.9376F, -105.4944F, 137.0911F, 41.7979F, -9.4558F,
   -26.6750F, -26.3288F, 192.3279F, -40.3915F, -60.3539F,
   -20.5168F, -51.3274F, 24.2191F, 47.5655F, 3.7810F,
   52.7448F, 32.7149F, 110.3630F, -21.5464F, -46.7397F,
   -4.6471F, -85.4498F, 129.3530F, -31.6973F, 65.0658F,
   85.9795F, -53.2356F, 57.2992F, -67.2997F, -16.4516F,
   54.7607F, -37.1454F, -42.4738F, -34.5633F, -75.7294F,
   0.0517F, -0.6519F, -1.3703F, -12.9917F, -148.9960F,
   -25.8564F, -84.6818F, 59.5151F, -72.5006F, -156.8837F,
   95.5045F, 42.2665F, 60.9475F, -54.5273F, -154.2295F,
   44.2810F, -53.2856F, -76.4082F, 42.9736F, -9.5286F,
   83.7851F, 46.2184F, 4.1979F, -25.1575F, -51.0520F,
   -43.1590F, -111.8518F, -2.3420F, -129.5042F, 6.3083F,
   36.5568F, 37.9981F, 30.8304F, -113.4959F, -79.9345F,
   -1.8493F, -15.3605F, -7.7093F, 98.5986F, -37.6177F,
   29.8557F, 67.8956F, 39.3578F, 63.5122F, 23.6349F,
   -27.0502F, -93.1902F, -21.6230F, 155.8476F, 2.5165F,
   -44.7515F, 124.6855F, 64.1401F, 107.4525F, 76.4296F,
   -5.6913F, 14.2025F, -18.3757F, 129.1303F, 47.2448F,
   88.4045F, 81.8179F, 106.0131F, 126.8375F, 71.9790F,
   5.1288F, -31.2676F, -85.3474F, 183.8293F, 137.8851F,
   112.3553F, 147.7620F, 6.3634F, -19.8307F, 97.9225F,
   68.1479F, 45.1480F, 20.4456F, 58.0360F, -109.0384F,
   91.7441F, 107.7617F, 15.1212F, 47.0576F, -18.7769F,
   0.0359F, -42.2360F, -42.2471F, 94.1364F, -132.5071F,
   -36.3673F, 208.5491F, 17.4150F, 13.0281F, -33.9693F,
   75.8679F, 58.1435F, -27.7181F, 158.0312F, -17.2012F,
   159.6027F, 144.6236F, 31.6370F, -72.6323F, -59.9622F,
   9.6094F, -39.8213F, -175.1189F, 48.5059F, -67.6102F,
   8.4503F, 159.5329F, -43.0075F, -121.1605F, -83.4178F,
   -10.0988F, 8.8687F, -32.4219F, 44.4166F, 5.5756F,
   -42.8299F, 63.9306F, 20.2856F, 86.9213F, -32.1874F,
   14.3835F, -9.6605F, -102.7732F, 119.7146F, -5.8143F,
   -181.8677F, 162.2914F, 114.1354F, 64.3820F, -7.2207F,
   11.2529F, -7.2901F, -66.7670F, 60.6365F, 71.2140F,
   69.9911F, 56.0744F, -10.7183F, 90.0489F, 100.9538F,
   -40.2531F, -96.6451F, -154.5512F, 92.9078F, 43.9867F,
   146.6528F, -60.2643F, -42.7063F, -1.4891F, 60.9280F,
   38.2143F, 33.0628F, -113.9071F, -1.8860F, -48.5082F,
   45.4740F, 54.2829F, -45.4768F, -38.2865F, -114.7585F,
   -5.4061F, -32.4176F, -108.6457F, -45.8426F, -147.6796F,
   41.4813F, 12.9066F, -41.5456F, -134.2382F, -189.2112F,
   46.4330F, 6.8074F, -160.9846F, 53.8969F, 64.9889F,
   149.8519F, 39.6080F, -79.5860F, 34.3934F, -61.0983F,
   -47.2568F, -110.8652F, -117.2089F, -0.3717F, -64.7306F,
   109.4393F, 12.8368F, -59.4424F, -122.1797F, -31.4694F,
   -14.4126F, 8.5189F, 89.4622F, 59.7035F, 14.3509F,
   -23.3690F, -32.3486F, 92.0827F, -3.3169F, -16.0022F,
   -66.1460F, -89.0215F, 49.1022F, -7.9295F, 141.6034F,
   -91.8393F, 6.6528F, 142.3305F, -60.9383F, 61.1140F,
   -9.9035F, -11.7998F, 27.9396F, 56.3528F, 60.7524F,
   59.7717F, -10.3224F, 100.1088F, 37.9471F, 56.4210F,
   -8.0706F, -14.3273F, 18.7969F, -25.3479F, 97.6871F,
   24.5711F, 30.0075F, 106.4596F, -49.8085F, 48.5911F,
   -50.4430F, -10.2232F, 15.4334F, 3.6211F, -45.0851F,
   -2.3100F, 32.1923F, 14.7250F, -35.3260F, -71.2667F,
   -131.2863F, -5.1029F, 11.0382F, 61.7451F, 35.4884F,
   -11.6345F, 94.0117F, 53.4418F, -66.3264F, -6.6649F,
   57.0155F, 5.4850F, -4.4053F, 12.7554F, 23.6955F,
   74.8564F, 77.0331F, 21.2315F, -32.6182F, 33.2036F,
   9.1670F, -78.1914F, -30.2862F, -11.1652F, 54.5347F,
   47.7388F, 92.4139F, 41.3020F, -133.3741F, 53.0528F,
   -22.2909F, -55.2830F, -33.2798F, -6.2896F, -28.8257F,
   -23.1116F, -35.0033F, -3.2927F, -72.9685F, -7.7727F,
   -82.3073F, -104.1028F, 84.5118F, -69.8221F, -43.1637F,
   -19.5349F, -2.7976F, 85.8117F, -104.2825F, -27.2739F,
   -22.1902F, -16.4648F, -6.6730F, -5.7920F, 27.7237F,
   12.5658F, 8.3566F, 24.7755F, -53.6096F, 12.7618F,
   -53.5741F, -68.1802F, 40.6279F, -26.4454F, 38.0614F,
   -26.1757F, -16.7503F, 52.0974F, -117.6298F, 86.4284F,
   -45.3502F, -78.0491F, -31.6007F, -35.9389F, -106.1265F,
   -24.8954F, -15.9989F, -21.2401F, -95.2454F, -86.1251F,
   -112.2460F, -94.4001F, -19.4652F, -33.9923F, -34.6433F,
   -95.7000F, 92.3440F, 39.8654F, -64.8912F, -115.0483F,
   25.8969F, 9.4794F, -63.2644F, -27.6289F, 36.9308F,
   25.6901F, 52.9741F, -35.3425F, -86.0458F, -14.2567F,
   -38.4101F, -62.7105F, -86.5748F, -73.4519F, 11.4446F,
   -7.0262F, 6.9787F, -7.4983F, -161.9427F, 3.4520F,
   19.1677F, 23.8207F, 38.9733F, 15.3764F, -21.7743F,
   -43.1688F, 31.9400F, 52.7685F, -11.9674F, 20.1452F,
   -35.0128F, -59.7929F, -36.1783F, 73.7702F, 71.2205F,
   -83.5945F, 115.1128F, 46.5742F, -28.7420F, 110.8292F,
   0.6347F, 53.5489F, 14.3686F, 18.4025F, 71.1819F,
   35.9728F, 45.7217F, 71.9726F, -20.7319F, 145.8423F,
   -8.1019F, -5.6290F, -13.8278F, 53.1928F, 152.8343F,
   57.0881F, 50.3240F, -64.1728F, -11.0774F, 129.9089F,
   36.0925F, 31.0887F, -32.6698F, 39.1693F, -52.3060F,
   -3.1405F, 84.8560F, -12.4226F, 1.0655F, -18.3673F,
   -58.9719F, 98.0129F, -83.8334F, 55.3139F, 35.7679F,
   -87.3652F, 121.2513F, -67.0447F, -107.5562F, 38.2622F,
   62.0513F, 60.4638F, -69.1208F, 43.6854F, 20.7579F,
   79.9694F, 10.7071F, 3.2196F, -93.6621F, 75.1450F,
   66.1122F, 46.9324F, -124.5950F, -78.3322F, 41.3915F,
   4.3229F, 13.8757F, -33.6575F, -92.7601F, 177.5551F,
   -42.5067F, -60.2164F, -63.5331F, 67.8862F, -21.5990F,
   -14.9811F, 12.6046F, -39.5598F, -30.2755F, -22.9980F,
   -80.2668F, -125.5028F, -37.7023F, 27.0213F, 31.0364F,
   -121.0989F, 17.0095F, 6.9604F, -72.1639F, 13.2205F,
   -45.0105F, -7.6587F, -57.9626F, -15.4159F, 79.2983F,
   -11.3128F, 60.2083F, -11.6157F, -61.5281F, 64.7932F,
   -17.6502F, -51.3289F, -123.9766F, -9.4275F, 141.7024F,
   -64.4806F, -80.0931F, -43.9064F, -86.7513F, 103.4672F,
   -12.6910F, -17.1597F, -58.5736F, 21.2452F, -75.2569F,
   -2.3195F, -14.7279F, -99.6098F, -65.8753F, -44.8940F,
   -103.7556F, 33.9613F, -73.7263F, -13.9038F, -66.1526F,
   -23.2670F, -33.9743F, -103.9167F, -170.2949F, -66.2644F,
   -26.4354F, -31.7233F, -108.7190F, 9.2282F, 14.7962F,
   -9.0340F, -7.5327F, -60.7032F, -100.1385F, 55.1481F,
   -23.7530F, -88.2969F, -220.5980F, -68.4569F, 27.6805F,
   3.3643F, -11.1758F, -92.5898F, -201.1381F, 102.8789F
};

const Float32 E_ROM_dico22_isf_36b[SIZE_BK22_36b * 4] = {
   -116.4826F, -2.3267F, 37.0276F, 11.9585F,
   -83.2536F, -34.0855F, -47.5833F, 101.8900F,
   1.6666F, -19.3089F, 81.2133F, 5.5915F,
   -50.2199F, -43.0204F, 11.8773F, 45.9389F,
   -83.6736F, 100.7533F, 42.9626F, -91.7069F,
   -15.8879F, -6.8432F, -49.2481F, 46.8567F,
   40.2748F, 25.3026F, 49.6150F, -14.6042F,
   49.1020F, -13.9172F, -9.5624F, 9.9442F,
   -53.7549F, -26.0755F, -108.5515F, -72.7670F,
   -64.1594F, -75.7696F, -78.4742F, 30.5700F,
   -82.2540F, -34.1100F, -20.0480F, -86.1655F,
   -68.0474F, -31.0203F, -36.5615F, -15.3976F,
   8.8102F, -2.3252F, -61.3614F, -93.8460F,
   8.4465F, -43.1451F, -59.6402F, -26.4485F,
   57.8873F, -1.8283F, -0.9150F, -58.1355F,
   -0.3760F, -52.6396F, -15.1757F, -69.7872F,
   26.6552F, 140.6710F, -45.5320F, -5.9904F,
   53.5205F, 18.1777F, -108.7393F, 57.1700F,
   53.1281F, 101.4898F, 52.8254F, 25.2450F,
   23.9310F, 45.2712F, -17.5932F, 37.8318F,
   90.1549F, 147.8806F, 33.8829F, -46.8215F,
   132.2255F, 69.0079F, -106.1068F, 1.3402F,
   103.9284F, 60.8167F, 10.9874F, -26.8688F,
   101.6619F, 32.7808F, -33.2101F, 33.7802F,
   -103.9522F, 60.0893F, -100.1459F, -71.1324F,
   -6.7456F, -25.3835F, -118.9063F, -2.2439F,
   -15.7859F, 68.4583F, -59.0835F, -70.2552F,
   -10.4160F, 10.4323F, -34.1781F, -24.5999F,
   47.1747F, 44.4596F, -64.8288F, -183.2536F,
   62.0332F, -25.7788F, -126.3596F, -90.0926F,
   83.7341F, 59.4975F, -55.2472F, -82.6595F,
   53.3482F, 14.1273F, -71.7379F, -20.0471F,
   -110.1113F, -92.4130F, 15.5809F, 3.7302F,
   -18.8013F, -91.7902F, -14.6105F, 98.1369F,
   -21.2096F, -126.2513F, 53.1438F, 11.3308F,
   -34.3197F, -67.9178F, 83.0592F, 77.2733F,
   -152.4262F, 38.6895F, -24.6967F, -146.6594F,
   41.7355F, -66.1454F, -64.2312F, 165.7005F,
   27.0772F, -43.2220F, 54.9943F, -65.1872F,
   28.8899F, -50.2606F, 25.5035F, 56.2652F,
   -137.7163F, -80.7691F, -80.0014F, -42.7037F,
   -62.3651F, -150.7820F, -138.8149F, 38.1900F,
   -68.7802F, -192.5785F, -7.8749F, -55.9578F,
   -98.5983F, -168.6083F, -0.7398F, 84.2329F,
   -35.1656F, -67.9774F, -65.6374F, -160.6224F,
   4.9446F, -110.8514F, -89.3759F, -62.4133F,
   -34.1288F, -108.8451F, 13.3702F, -98.1466F,
   -29.4614F, -102.8405F, -22.8325F, -16.4528F,
   163.9105F, 20.5337F, -82.4577F, -139.6505F,
   149.8660F, -13.6016F, -146.1693F, 154.5116F,
   26.6673F, -89.2330F, 126.1283F, -0.6475F,
   65.0738F, -119.9143F, 75.1351F, 75.9130F,
   179.3039F, 128.5647F, -1.9070F, -129.8349F,
   146.4228F, 30.7058F, -2.7357F, 122.1955F,
   110.1199F, -48.4108F, 78.0568F, -35.8715F,
   105.8916F, -63.1848F, -27.3019F, 70.1234F,
   -61.4940F, -116.2433F, -200.7709F, -120.8167F,
   22.5673F, -63.6047F, -213.1464F, 7.0661F,
   48.3577F, -141.9964F, 65.3628F, -93.1012F,
   32.5991F, -160.4748F, -45.5903F, 37.6377F,
   54.6730F, -43.7561F, -151.6335F, -243.7797F,
   101.1397F, -52.0073F, -123.8756F, 16.0909F,
   63.7439F, -50.8957F, -25.0308F, -130.5887F,
   88.2969F, -64.3814F, -48.5324F, -43.0123F,
   -181.9705F, -24.0096F, 2.3863F, 89.5185F,
   -59.5915F, 80.1714F, -56.6992F, 94.5562F,
   -62.1659F, 18.9433F, 76.0160F, 57.8553F,
   -22.4665F, 10.7743F, 12.0918F, 108.9830F,
   -118.1671F, 72.3971F, 109.1443F, -1.3839F,
   -23.8459F, 76.9232F, 22.9444F, 33.4435F,
   -44.5975F, 47.9554F, 65.6020F, -20.3018F,
   13.6440F, 14.2271F, 38.9794F, 49.0774F,
   -159.0490F, 40.0263F, -30.2347F, -15.6526F,
   -132.0447F, -0.4802F, -133.5965F, 61.0673F,
   -69.7882F, 40.8409F, -13.3128F, -38.0654F,
   -72.3667F, 32.7089F, -13.7936F, 42.2054F,
   -51.7645F, 41.9187F, -35.3954F, -139.3463F,
   -70.4106F, 21.2248F, -89.5260F, 9.2659F,
   -17.3511F, 18.5094F, 18.3073F, -71.0465F,
   -25.6477F, 4.9120F, 17.7504F, 1.3904F,
   -132.5788F, 97.9003F, 25.0501F, 88.4524F,
   -16.3421F, 39.3618F, -136.5657F, 107.5423F,
   -38.7664F, 155.3425F, 55.5880F, 47.1198F,
   43.2628F, 4.6910F, -39.9914F, 101.5498F,
   0.0830F, 197.2389F, 101.6377F, -36.9070F,
   62.8279F, 111.5018F, -37.6259F, 87.5036F,
   -1.4618F, 80.2742F, 122.5196F, 12.8147F,
   65.1409F, 54.2334F, 34.3093F, 79.7332F,
   -91.6271F, 123.5048F, -23.6092F, -9.5877F,
   -3.2208F, -58.6010F, -121.8443F, 78.4266F,
   -14.0112F, 114.0430F, 23.9902F, -40.4777F,
   -15.4729F, 68.1635F, -63.4068F, 16.5519F,
   -8.1780F, 157.1093F, -11.2800F, -137.0649F,
   8.0606F, 59.4657F, -140.5838F, -36.3320F,
   22.2676F, 74.5499F, 82.8755F, -76.6792F,
   29.5054F, 61.7793F, -8.0867F, -27.0956F,
   -128.1797F, -72.1649F, 129.2723F, 46.4323F,
   -20.7709F, 111.3353F, 22.0263F, 131.4695F,
   -41.9162F, -9.2182F, 158.1064F, 11.2798F,
   -7.1545F, 53.6140F, 106.2812F, 108.0610F,
   -99.5471F, 8.6443F, 67.7713F, -74.7187F,
   115.0857F, 125.6656F, 126.8208F, 118.1528F,
   8.3489F, -10.6246F, 129.8214F, -69.4937F,
   46.4587F, 5.2711F, 105.7244F, 50.4485F,
   -177.6754F, -70.1548F, 45.4547F, -74.5236F,
   -88.7724F, 24.0424F, -57.8465F, 204.7888F,
   -68.8890F, -112.1224F, 110.3182F, -61.1447F,
   -94.7905F, 5.0186F, 77.6790F, 167.9779F,
   -23.0125F, -19.1513F, 44.8958F, -142.4394F,
   28.0225F, -67.2659F, -53.6584F, 36.2990F,
   -54.0239F, -49.3550F, 55.1522F, -32.8946F,
   2.0792F, -48.4861F, 14.9542F, -7.9957F,
   -100.8337F, 121.3744F, 234.8044F, 83.3187F,
   36.5393F, 50.8425F, -23.7400F, 196.1698F,
   -0.3059F, -61.1762F, 189.5078F, 122.4017F,
   57.0232F, -28.8934F, 61.7134F, 134.6930F,
   107.9555F, 52.8736F, 109.5032F, -22.1311F,
   191.2912F, 98.4213F, 38.7853F, 16.9729F,
   104.1509F, -28.9315F, 167.6925F, 40.9011F,
   108.7356F, -8.7976F, 46.5837F, 36.6080F,
   -211.7768F, 190.5028F, 100.4753F, -45.0833F,
   -32.7791F, -95.3684F, -171.2339F, 186.7766F,
   -44.0395F, -212.9286F, 151.2602F, 39.5958F,
   -37.2152F, -119.5092F, 43.4460F, 194.5965F,
   37.2023F, 64.8673F, 8.5289F, -117.5366F,
   163.8890F, -5.9760F, -22.5312F, -30.3817F,
   105.4272F, 11.3735F, 47.6228F, -110.1422F,
   62.3479F, -93.6059F, 19.4353F, -14.9112F
};

const Float32 E_ROM_dico23_isf_36b[SIZE_BK23_36b * 7] = {
   31.6601F,  -7.0021F,  26.7324F, -10.6919F, -47.7440F, -109.3316F,   -1.6793F,
   17.6152F, -69.1639F,  81.7745F, -11.6557F, -53.0002F,  -28.8501F,   51.3155F,
   -17.3392F,  39.6296F, -29.4208F, -34.2716F, -18.7475F,  -53.4279F,  -21.0682F,
   -95.6211F, -11.0108F,  24.5639F,  -6.9345F, -43.8606F,  -40.3930F,   22.5841F,
   -30.9469F,  -2.2484F,  85.8968F, -25.2118F,  44.4649F,  -13.7549F,  -19.6283F,
   42.5807F, -25.3893F,  55.9790F, -44.6039F,  50.3451F,   29.5180F,   49.0040F,
   64.7422F,  35.0105F, -24.0120F, -94.6131F,  72.4785F,  -29.0560F,  -16.8213F,
   -17.7762F, -36.1097F,  19.1423F, -88.8131F,   9.4121F,  -60.6212F,   15.2182F,
   26.1194F,  33.0214F,  38.6243F, -16.4898F,  20.5108F,  -71.7754F, -109.7775F,
   55.6133F, -47.5797F,   0.0746F,   8.1716F, -55.6639F,   -5.8260F,   -6.7666F,
   86.9148F,  36.0665F,  -8.0422F, -18.6522F, -31.9936F,   -5.4779F,  -65.2340F,
   20.0656F, -14.5303F, -94.7449F, -11.7982F, -35.2031F,    6.9138F,  -22.0461F,
   20.9060F,  41.0458F,  29.0299F,  33.4933F,  27.0675F,    5.1646F,  -39.4529F,
   76.7175F,  28.2216F, -34.8562F,  16.7304F,  25.2291F,    7.4912F,   15.3906F,
   47.1053F,  13.3625F,  50.9902F, -31.9588F,   9.8596F,   83.1329F,  -61.0081F,
   39.4355F, -39.6936F, -53.1235F,  -8.1282F,  22.3882F,   83.7131F,    8.6440F,
   13.9302F, -48.4179F,  80.1925F,  79.6573F,  22.8321F,  -61.0888F,  -32.5848F,
   32.2601F, -45.5857F,  53.4300F,  53.4773F,  33.0895F,   45.4420F,   17.2865F,
   -36.0406F, -57.6566F, -26.6841F,   4.2674F, -39.8452F,  -76.9236F,  -85.9930F,
   -29.8696F, -72.1261F, -22.6802F,  51.6689F, -10.3407F,  -71.6716F,   33.3209F,
   -2.6190F, -11.9459F,  -0.7138F,   9.1651F,  80.0695F,  -58.7987F,    3.7833F,
   -10.6521F, -14.2910F,  -1.7854F,  -6.9002F, 114.1547F,   51.0851F,    0.4936F,
   45.5988F, -65.6262F,   3.5076F, -36.3037F,  31.3412F,  -22.8916F,  -48.6823F,
   -71.1623F, -95.4642F,  38.3287F,  -9.4939F,  52.9002F,   -8.4421F,   36.7283F,
   86.2675F,  37.9505F,  41.2317F,  16.4269F,  16.7260F,  -62.6682F,   32.6168F,
   9.9181F, -25.0994F,  -8.3770F,   2.2985F,   5.3157F,   -6.0428F,   60.1555F,
   49.0247F,   6.0545F, -54.7919F,  58.6261F,  -3.9732F,  -80.9565F,  -44.3777F,
   30.6734F, -24.5101F, -82.4513F, -27.4901F, -10.8146F,  -84.7650F,   64.6334F,
   18.0553F,  14.7350F,  -8.7675F, 109.6707F,  51.4393F,  -24.1075F,   42.7732F,
   43.6601F,  21.1278F, -43.7223F, -36.3814F,  81.3688F,   10.3561F,  115.5857F,
   44.7304F,   4.0203F, -57.4008F,  16.1462F,  84.4927F,   16.5958F, -107.7931F,
   19.3887F, -44.8683F, -99.3061F,  65.1051F,  45.6203F,   -0.6984F,   23.6828F,
   6.4828F,  56.2769F,  13.3073F, -28.0424F, -72.7066F,  -58.5212F,  106.1769F,
   -11.1674F, -25.7849F, -34.9574F, -36.9812F, -58.1843F,   50.2182F,   98.0044F,
   47.6759F,   0.1786F, -19.4605F, -91.2611F, -35.5267F,   13.8815F,   10.2127F,
   -41.1350F, -39.8329F, -34.1829F, -47.4437F, -91.9939F,   -2.6788F,   -4.2821F,
   -79.6033F,  42.6213F,   2.0653F, -74.4526F,  41.0602F,   -5.7905F,   63.6539F,
   -31.4071F,  12.4383F,  -9.2647F, -81.5461F,  16.0430F,  114.6936F,   27.3863F,
   -41.2719F, -36.6253F, -79.4997F, -46.2467F,  46.8034F,  -19.4146F,  -14.6126F,
   -31.9554F, -93.9975F,  17.8233F, -51.0152F, -11.4389F,   58.4455F,  -21.6241F,
   12.9670F,  60.7166F,  46.9564F, -34.6928F,  -3.1553F,    2.7762F,   24.2978F,
   83.1108F,  31.8670F,  23.6533F,   6.8663F, -63.0785F,   56.1031F,   59.2073F,
   11.6877F,  51.3190F,  25.3083F, -33.9761F, -99.5482F,   -6.6294F,  -41.7488F,
   -3.1041F,  33.3329F, -24.8851F,  20.0947F, -63.3399F,   87.2565F,  -20.6208F,
   -52.2549F, 102.0059F,  26.9280F, -22.0208F,  85.2782F,   28.3108F,  -43.2782F,
   0.8584F,  60.5987F, -44.2357F, -33.9700F,  19.1247F,   33.1631F,  -10.8275F,
   -63.5371F,  16.2471F,  -0.2828F, -76.6458F,   2.8322F,   15.2373F,  -95.8412F,
   5.5912F, -53.4989F, -30.8518F,   4.1365F, -62.3116F,   79.0018F, -114.5611F,
   -36.8596F,  12.7812F,  81.1609F,  39.2434F,  21.9162F,  -17.2064F,  127.1804F,
   -30.4279F, -15.8549F,  90.5057F,   4.9324F, -55.6202F,   88.7264F,   31.2457F,
   -6.4338F, -33.9992F,  78.4675F,  12.7460F, -51.8920F,    5.7002F,  -71.3454F,
   -22.6305F, -75.0886F, -18.4430F,  71.6854F, -49.9613F,   51.9112F,   38.5919F,
   -80.1990F,   4.2373F, -60.6613F,  30.4453F,  20.4006F,   28.0162F,   55.1546F,
   -96.2887F,  10.3433F,  38.5883F,  59.1444F,  23.2247F,   45.0000F,  -24.9305F,
   -30.9055F, -18.2692F,  -6.1201F,  -5.6460F,   2.1550F,   18.2502F,  -16.8166F,
   -28.0966F, -69.3654F, -10.5762F,  63.4193F,  43.8444F,   16.6643F,  -67.9380F,
   -68.3269F,  93.1501F,  72.8030F,  27.6125F, -21.1484F,  -73.4926F,  -29.7929F,
   -88.0045F,  91.0304F,  15.1885F, -15.1144F, -61.6391F,   47.5526F,   17.2017F,
   -10.0973F,  16.8435F,  32.7608F,  50.6231F, -36.3173F,  -19.8196F,    8.7833F,
   1.1920F,  35.7643F, -58.4480F,  53.1925F, -71.1458F,  -22.3426F,   37.8118F,
   -51.2779F,  69.9477F, -30.3236F,  31.2621F,  35.5419F,  -64.6025F,   35.1224F,
   -0.6572F,  57.9364F,   5.6749F,  50.6782F,  25.3456F,   68.4492F,   45.6955F,
   -53.8176F,  44.6419F, -53.4407F,  51.5645F,   1.3456F,   -4.0462F,  -72.5604F,
   54.6610F,  -1.6150F, -14.3009F,  99.1775F, -24.2467F,   36.0557F,  -42.7052F
};

/*
 * isf codebooks:  split-by-5 VQ
 *
 * codebook   vector dimension    number of vectors
 * ~~~~~~~~   ~~~~~~~~~~~~~~~~    ~~~~~~~~~~~~~~~~~
 *    1            2                  64
 *    2            3                  64
 *    3            3                  64
 *    4            4                  32
 *    5            4                  32
 */

/*
 * 1st split: isf0 to isf1
 */
const Float32 E_ROM_dico1_isf_noise[SIZE_BK_NOISE1 * 2] = {
   -105.0619F, -262.8448F,
   -86.6322F, -209.7320F,
   -91.1931F, -168.0481F,
   -53.8753F, -176.1473F,
   -82.7923F, -129.2635F,
   -75.0845F, -93.9859F,
   -33.9476F, -90.3333F,
   -74.4463F, -50.0451F,
   -27.3422F, -41.3458F,
   -64.0658F, -2.5361F,
   28.8179F, -70.0786F,
   10.5059F, -12.8463F,
   -39.9581F, 28.7359F,
   -63.4034F, 44.7503F,
   -36.8732F, 67.0368F,
   -2.3563F, 50.6058F,
   -55.9412F, 91.2822F,
   5.5577F, 85.1171F,
   -25.2586F, 105.6009F,
   34.3216F, 71.1726F,
   -48.3043F, 133.1605F,
   -17.0366F, 148.8321F,
   14.8305F, 130.6908F,
   45.5592F, 106.8464F,
   -43.7865F, 177.4889F,
   28.9675F, 168.2081F,
   -1.7812F, 190.4731F,
   68.4806F, 149.8876F,
   -32.5396F, 218.9960F,
   47.8438F, 206.6539F,
   8.1960F, 234.8430F,
   89.3668F, 188.0221F,
   90.4035F, 118.3064F,
   88.4263F, 237.5410F,
   117.3136F, 145.4498F,
   82.0645F, 73.1093F,
   119.3943F, 103.4796F,
   128.2321F, 184.8528F,
   149.1563F, 129.2199F,
   144.9058F, 51.5776F,
   54.4305F, 22.6608F,
   142.5451F, 8.2812F,
   97.4966F, -31.8767F,
   173.0963F, 84.9898F,
   188.5170F, 42.9719F,
   166.5772F, 162.2136F,
   226.1740F, 86.5496F,
   202.2992F, 129.9512F,
   223.9365F, 175.1483F,
   177.6235F, 206.6174F,
   267.4435F, 128.4469F,
   129.8694F, 226.5438F,
   232.5047F, 231.7147F,
   182.8960F, 252.1132F,
   297.7026F, 201.9668F,
   127.2108F, 277.0716F,
   189.5906F, 309.6084F,
   50.8926F, 267.2207F,
   262.1367F, 287.8791F,
   138.4480F, 342.2196F,
   34.3804F, 314.8025F,
   -25.4771F, 275.9066F,
   -13.5804F, 397.0122F,
   103.9035F, 438.8237F
};

/*
 * 2nd split: isf2 to isf4
 */
const Float32 E_ROM_dico2_isf_noise[SIZE_BK_NOISE2 * 3] = {
   -322.0252F, -345.3659F, -370.7835F,
   -314.5226F, -178.1928F, -163.4712F,
   -172.6074F, -171.2812F, -211.3224F,
   -84.6538F, -225.6710F, -309.6982F,
   -65.7865F, -173.3148F, -227.1618F,
   -112.2154F, -192.0593F, -106.9376F,
   -215.4487F, -115.9810F, -117.3159F,
   -63.6069F, -129.9179F, -139.6890F,
   -144.6460F, -90.4563F, -90.4457F,
   -68.4568F, -140.0307F, -62.0772F,
   -148.8729F, -8.2807F, -139.6018F,
   -71.8488F, -61.9658F, -63.2677F,
   -20.5518F, -74.4279F, -109.2530F,
   7.0237F, -104.2936F, -83.9866F,
   -53.7369F, 23.9918F, -110.5212F,
   27.5927F, -37.1603F, -114.8537F,
   5.1437F, -60.8316F, -213.2375F,
   -0.0302F, -32.5161F, -30.9893F,
   17.0532F, 37.8947F, -123.4512F,
   69.4656F, -20.4364F, -83.2673F,
   86.8486F, -101.9140F, -164.6497F,
   92.7595F, -46.0582F, -17.0665F,
   55.0807F, 56.5797F, -51.6089F,
   141.9268F, 31.5952F, -111.9348F,
   83.1350F, 25.3995F, 13.2634F,
   -41.6305F, 36.7624F, -1.8806F,
   35.5619F, -11.1360F, 49.0468F,
   -138.6944F, 20.1165F, -16.0470F,
   -85.7227F, -29.5173F, 56.5419F,
   -24.5306F, 39.2035F, 95.3233F,
   -280.8913F, 17.3520F, 10.4723F,
   -223.4287F, -48.5814F, 60.3781F,
   -165.0790F, 52.0898F, 122.9863F,
   -358.1462F, 27.5800F, 87.6394F,
   -104.6460F, 124.2969F, 51.2676F,
   -36.4609F, -74.1586F, 163.9110F,
   -38.0255F, 47.6372F, 191.7725F,
   -30.7896F, 123.7246F, 138.8302F,
   50.7064F, 39.0454F, 126.9589F,
   33.4334F, -114.5499F, 82.0810F,
   52.0289F, 100.6551F, 62.9260F,
   68.9398F, -28.6776F, 181.7511F,
   76.3353F, 117.0400F, 150.0834F,
   135.9951F, 8.5581F, 86.5077F,
   146.9219F, 71.3024F, 159.6375F,
   147.3144F, 111.7858F, 78.8838F,
   94.6612F, 83.2958F, 257.4856F,
   100.5727F, 220.5308F, 96.9196F,
   134.4751F, 159.3021F, -29.6067F,
   158.3509F, 171.9108F, 198.9399F,
   238.9283F, 150.3837F, 148.0492F,
   209.2346F, 237.2747F, 84.4977F,
   -21.9723F, 227.3437F, 75.1847F,
   39.1823F, 202.0239F, 221.5619F,
   -142.6753F, 174.8218F, 173.9311F,
   284.1931F, 135.5243F, 4.0420F,
   197.3545F, 139.2970F, 296.6173F,
   248.5921F, 227.1523F, 257.1579F,
   130.8344F, 201.7665F, 332.6679F,
   147.8451F, 316.0691F, 223.3704F,
   -76.1983F, 342.8470F, 323.9619F,
   206.5931F, 276.1481F, 385.6192F,
   358.4664F, 283.4258F, 153.2502F,
   488.2212F, 389.4155F, 415.1465F
};

/*
 * 3rd split: isf5 to isf7
 */
const Float32 E_ROM_dico3_isf_noise[SIZE_BK_NOISE3 * 3] = {
   -314.4522F, -327.3454F, -302.2017F,
   -203.9117F, -244.9068F, -323.6309F,
   -186.2536F, -189.6962F, -235.4297F,
   -115.3995F, -187.7246F, -247.4612F,
   -142.8871F, -150.1942F, -153.3509F,
   -72.7072F, -161.6552F, -154.8560F,
   -92.6058F, -153.8863F, -41.3953F,
   -98.5994F, -78.7129F, -107.5332F,
   -23.9335F, -69.1898F, -172.7805F,
   -32.7960F, -77.3107F, -77.6193F,
   -69.7610F, -48.7327F, -12.2728F,
   -28.2008F, -18.3660F, -63.7045F,
   -116.3960F, -85.9584F, 84.0164F,
   -24.9241F, -65.7832F, 98.1712F,
   -51.8131F, 60.9599F, -23.0835F,
   -11.5386F, -0.7831F, 49.5084F,
   20.9367F, 25.8838F, -24.0191F,
   -91.0503F, 8.0161F, 98.1430F,
   81.8089F, -19.4694F, 12.5661F,
   12.6976F, 75.6776F, 53.0320F,
   -45.8943F, -7.2062F, 185.7293F,
   78.8625F, 17.9396F, 120.6767F,
   99.9111F, 72.2135F, 20.5674F,
   13.7143F, 77.9979F, 152.4088F,
   78.0738F, 102.8484F, 94.6295F,
   -84.5040F, 118.1050F, 114.7064F,
   50.0596F, 139.6647F, -0.1577F,
   7.5966F, 168.3675F, 112.1363F,
   87.3316F, 174.4695F, 109.2434F,
   143.4298F, 64.6162F, 83.1959F,
   155.2307F, 122.7478F, 124.7551F,
   149.4719F, 147.9446F, 29.2261F,
   108.3426F, 127.0695F, 180.3353F,
   153.8867F, 197.0929F, 130.5922F,
   98.1956F, 38.2729F, -83.1688F,
   175.9048F, 59.9165F, 174.8273F,
   220.7233F, 88.4354F, 29.6527F,
   183.5505F, 149.7702F, 195.9197F,
   248.2375F, 152.2900F, 108.7161F,
   92.6034F, 52.6123F, 242.1773F,
   133.7155F, 156.6441F, 253.6524F,
   129.4868F, 215.2826F, 202.4702F,
   50.8526F, 163.2149F, 231.3041F,
   207.4225F, 119.3517F, 287.7008F,
   284.9414F, 152.0429F, 226.6699F,
   194.1092F, 217.5607F, 272.9373F,
   115.7381F, 149.7147F, 341.2299F,
   110.4872F, 243.8135F, 296.4818F,
   49.2583F, 242.9759F, 185.7527F,
   218.4281F, 232.2570F, 184.3956F,
   149.3219F, 300.8763F, 240.4531F,
   280.7402F, 239.3506F, 291.1599F,
   210.7589F, 249.5308F, 362.4219F,
   201.8519F, 322.7530F, 312.7221F,
   267.0336F, 316.7373F, 236.0985F,
   293.8933F, 307.2101F, 334.8007F,
   364.5740F, 258.0872F, 136.8430F,
   271.0491F, 175.6599F, 414.3537F,
   219.3663F, 355.8581F, 410.6574F,
   321.8781F, 317.4617F, 431.4073F,
   295.9753F, 408.9221F, 344.4007F,
   445.2552F, 358.3636F, 347.1212F,
   405.7584F, 486.9081F, 556.9660F,
   579.1823F, 650.5953F, 732.6415F
};

/*
 * 4th split: isf8 to isf11
 */
const Float32 E_ROM_dico4_isf_noise[SIZE_BK_NOISE4 * 4] = {
   -303.1361F, -333.7803F, -347.8725F, -359.4826F,
   -215.7523F, -238.3714F, -259.0660F, -289.4405F,
   -125.2913F, -144.5697F, -186.1030F, -220.5702F,
   107.0201F, -62.5400F, -178.2959F, 78.3558F,
   103.6377F, 26.1626F, -62.5396F, -119.5691F,
   -2.9518F, -82.1288F, 30.9020F, 106.1641F,
   63.5315F, 92.0127F, 120.0751F, 120.2205F,
   225.8191F, 123.6829F, 24.8449F, 116.4136F,
   -3.5121F, 76.7992F, 133.5771F, 242.3746F,
   133.9970F, 90.5385F, 122.6116F, 242.8362F,
   67.7067F, 58.1366F, 214.2467F, 205.7587F,
   139.2159F, 144.6370F, 188.0780F, 146.8699F,
   52.7259F, 173.3530F, 190.5093F, 217.0651F,
   152.5843F, 183.9790F, 190.4016F, 255.0127F,
   89.0815F, 165.5783F, 224.8295F, 325.9805F,
   164.7109F, 145.4724F, 282.2096F, 266.3416F,
   115.0926F, 262.9259F, 270.8098F, 248.1893F,
   210.4561F, 232.8534F, 230.5395F, 175.4710F,
   185.5563F, 241.3006F, 257.2568F, 319.6433F,
   287.0131F, 201.7823F, 191.6938F, 262.7153F,
   234.9706F, 135.0743F, 100.3842F, 342.4814F,
   244.1460F, 248.1535F, 331.5106F, 281.3059F,
   283.8415F, 319.6996F, 272.8347F, 232.3501F,
   255.2254F, 187.9562F, 269.5668F, 445.0594F,
   318.1439F, 297.7580F, 274.9042F, 354.5264F,
   198.0688F, 291.9682F, 350.7645F, 365.4872F,
   331.2469F, 333.9341F, 361.1275F, 306.5865F,
   252.3140F, 405.2271F, 344.4444F, 310.3618F,
   301.5996F, 329.9839F, 399.8752F, 449.5998F,
   442.5639F, 384.1796F, 319.3500F, 359.8945F,
   367.3365F, 417.2813F, 488.9795F, 508.7549F,
   620.4818F, 690.0919F, 671.0975F, 590.9387F
};

/*
 * 5th split: isf12 to isf15
 */
const Float32 E_ROM_dico5_isf_noise[SIZE_BK_NOISE5 * 4] = {
   -316.2941F, -343.3920F, -368.9643F, -99.0392F,
   97.0450F, 71.7080F, 262.1754F, 50.0767F,
   112.4468F, 274.7287F, 358.7211F, 38.6232F,
   257.1795F, 217.8952F, 258.6022F, 85.4104F,
   215.6079F, 228.7031F, 355.6118F, 81.4276F,
   218.4167F, 314.1638F, 296.3636F, 46.5526F,
   236.5851F, 302.2603F, 359.7074F, -54.3647F,
   305.5962F, 297.4161F, 292.2765F, 81.3061F,
   295.4005F, 276.4861F, 384.1555F, 21.9038F,
   212.3865F, 337.6641F, 394.4713F, 59.4888F,
   287.8841F, 272.8244F, 385.4754F, 116.6899F,
   301.1416F, 360.9641F, 343.2708F, 40.2882F,
   209.5407F, 306.5670F, 375.3650F, 158.1632F,
   260.4655F, 357.6203F, 312.8495F, 128.1323F,
   288.3843F, 275.2455F, 301.9735F, 171.2988F,
   321.4736F, 340.3829F, 387.3967F, 138.5811F,
   250.1021F, 392.3276F, 410.9932F, 144.0109F,
   282.6727F, 320.9480F, 370.8150F, 233.3823F,
   161.9622F, 255.8086F, 284.8333F, 188.2323F,
   393.9827F, 349.8717F, 309.8297F, 141.6719F,
   354.6675F, 313.7792F, 268.4562F, -9.8681F,
   396.7580F, 327.4283F, 395.0635F, 73.8789F,
   369.8889F, 434.5653F, 367.9579F, 86.8706F,
   356.9169F, 409.7761F, 383.1727F, 205.7493F,
   373.4236F, 385.5478F, 394.8253F, -47.0644F,
   305.1456F, 409.5952F, 437.7072F, 36.1164F,
   460.1152F, 411.4012F, 345.2657F, 18.4203F,
   438.6750F, 413.5906F, 461.8582F, 46.2207F,
   364.6111F, 379.8415F, 498.6693F, 139.3899F,
   433.1401F, 358.6805F, 429.9292F, 196.6537F,
   405.9484F, 502.2995F, 476.5874F, 123.6977F,
   527.5430F, 471.6736F, 394.4862F, 127.2615F
};

/*
 * Hamming_cos window for LPC analysis.
 */
const Float32 E_ROM_hamming_cos[L_WINDOW] = {
   0.0800000F, 0.0800348F, 0.0801391F, 0.0803129F, 0.0805563F,
   0.0808691F, 0.0812513F, 0.0817028F, 0.0822237F, 0.0828138F,
   0.0834730F, 0.0842012F, 0.0849983F, 0.0858642F, 0.0867987F,
   0.0878018F, 0.0888733F, 0.0900129F, 0.0912206F, 0.0924962F,
   0.0938394F, 0.0952500F, 0.0967279F, 0.0982728F, 0.0998845F,
   0.101563F, 0.103307F, 0.105118F, 0.106994F, 0.108936F,
   0.110943F, 0.113014F, 0.115151F, 0.117351F, 0.119616F,
   0.121944F, 0.124335F, 0.126789F, 0.129306F, 0.131884F,
   0.134525F, 0.137226F, 0.139989F, 0.142812F, 0.145695F,
   0.148638F, 0.151639F, 0.154700F, 0.157819F, 0.160995F,
   0.164229F, 0.167520F, 0.170867F, 0.174270F, 0.177728F,
   0.181241F, 0.184808F, 0.188429F, 0.192103F, 0.195829F,
   0.199608F, 0.203438F, 0.207319F, 0.211250F, 0.215231F,
   0.219261F, 0.223340F, 0.227466F, 0.231640F, 0.235860F,
   0.240126F, 0.244438F, 0.248794F, 0.253195F, 0.257638F,
   0.262125F, 0.266653F, 0.271223F, 0.275833F, 0.280483F,
   0.285173F, 0.289901F, 0.294667F, 0.299470F, 0.304309F,
   0.309184F, 0.314094F, 0.319038F, 0.324015F, 0.329025F,
   0.334067F, 0.339140F, 0.344244F, 0.349377F, 0.354538F,
   0.359728F, 0.364946F, 0.370189F, 0.375458F, 0.380753F,
   0.386071F, 0.391412F, 0.396776F, 0.402162F, 0.407568F,
   0.412995F, 0.418441F, 0.423905F, 0.429386F, 0.434885F,
   0.440399F, 0.445928F, 0.451472F, 0.457029F, 0.462598F,
   0.468179F, 0.473771F, 0.479373F, 0.484984F, 0.490604F,
   0.496231F, 0.501865F, 0.507504F, 0.513148F, 0.518797F,
   0.524448F, 0.530102F, 0.535757F, 0.541413F, 0.547069F,
   0.552724F, 0.558377F, 0.564027F, 0.569673F, 0.575315F,
   0.580952F, 0.586582F, 0.592205F, 0.597821F, 0.603428F,
   0.609025F, 0.614611F, 0.620187F, 0.625750F, 0.631300F,
   0.636837F, 0.642359F, 0.647865F, 0.653355F, 0.658828F,
   0.664283F, 0.669719F, 0.675136F, 0.680532F, 0.685907F,
   0.691260F, 0.696590F, 0.701896F, 0.707178F, 0.712435F,
   0.717665F, 0.722869F, 0.728045F, 0.733192F, 0.738311F,
   0.743399F, 0.748457F, 0.753483F, 0.758477F, 0.763438F,
   0.768365F, 0.773257F, 0.778115F, 0.782936F, 0.787720F,
   0.792467F, 0.797176F, 0.801846F, 0.806477F, 0.811067F,
   0.815616F, 0.820124F, 0.824589F, 0.829011F, 0.833389F,
   0.837723F, 0.842012F, 0.846256F, 0.850453F, 0.854603F,
   0.858706F, 0.862760F, 0.866766F, 0.870722F, 0.874628F,
   0.878484F, 0.882288F, 0.886041F, 0.889741F, 0.893389F,
   0.896983F, 0.900523F, 0.904009F, 0.907439F, 0.910814F,
   0.914133F, 0.917395F, 0.920601F, 0.923748F, 0.926838F,
   0.929869F, 0.932842F, 0.935755F, 0.938608F, 0.941401F,
   0.944133F, 0.946804F, 0.949413F, 0.951961F, 0.954446F,
   0.956869F, 0.959229F, 0.961525F, 0.963758F, 0.965926F,
   0.968030F, 0.970070F, 0.972044F, 0.973953F, 0.975796F,
   0.977574F, 0.979285F, 0.980930F, 0.982509F, 0.984020F,
   0.985464F, 0.986841F, 0.988151F, 0.989392F, 0.990566F,
   0.991671F, 0.992709F, 0.993678F, 0.994578F, 0.995409F,
   0.996172F, 0.996866F, 0.997490F, 0.998046F, 0.998532F,
   0.998949F, 0.999296F, 0.999574F, 0.999783F, 0.999922F,
   0.999991F, 1.00000F, 0.999924F, 0.999698F, 0.999320F,
   0.998791F, 0.998111F, 0.997280F, 0.996298F, 0.995166F,
   0.993883F, 0.992450F, 0.990867F, 0.989134F, 0.987252F,
   0.985220F, 0.983039F, 0.980710F, 0.978233F, 0.975607F,
   0.972834F, 0.969914F, 0.966848F, 0.963635F, 0.960277F,
   0.956773F, 0.953125F, 0.949332F, 0.945396F, 0.941317F,
   0.937096F, 0.932733F, 0.928229F, 0.923585F, 0.918801F,
   0.913879F, 0.908818F, 0.903619F, 0.898284F, 0.892814F,
   0.887208F, 0.881468F, 0.875595F, 0.869589F, 0.863452F,
   0.857185F, 0.850788F, 0.844262F, 0.837609F, 0.830829F,
   0.823923F, 0.816893F, 0.809739F, 0.802463F, 0.795066F,
   0.787548F, 0.779911F, 0.772157F, 0.764285F, 0.756298F,
   0.748197F, 0.739983F, 0.731657F, 0.723220F, 0.714674F,
   0.706019F, 0.697258F, 0.688392F, 0.679421F, 0.670348F,
   0.661174F, 0.651899F, 0.642526F, 0.633056F, 0.623490F,
   0.613830F, 0.604077F, 0.594233F, 0.584299F, 0.574276F,
   0.564167F, 0.553972F, 0.543694F, 0.533333F, 0.522892F,
   0.512372F, 0.501774F, 0.491101F, 0.480353F, 0.469533F,
   0.458641F, 0.447680F, 0.436652F, 0.425558F, 0.414399F,
   0.403177F, 0.391895F, 0.380553F, 0.369154F, 0.357699F,
   0.346190F, 0.334629F, 0.323017F, 0.311356F, 0.299648F,
   0.287895F, 0.276098F, 0.264260F, 0.252381F, 0.240465F,
   0.228512F, 0.216524F, 0.204504F, 0.192453F, 0.180373F,
   0.168265F, 0.156132F, 0.143976F, 0.131797F, 0.119599F,
   0.107383F, 0.0951502F, 0.0829032F, 0.0706437F, 0.0583736F,
   0.0460946F, 0.0338087F, 0.0215176F, 0.00922329F
};


/*
 * Quantization table for VQ of pitch and innovation gains.
 */
const Float32 E_ROM_qua_gain6b[NB_QUA_GAIN6B * 2] = {
   0.095577F, 0.650507F,
   0.096232F, 1.737050F,
   0.187418F, 3.168846F,
   0.255896F, 4.962348F,
   0.274392F, 1.237365F,
   0.306340F, 2.191559F,
   0.340913F, 7.624112F,
   0.349443F, 0.694540F,
   0.393850F, 0.283414F,
   0.410411F, 3.335519F,
   0.467326F, 1.722400F,
   0.492699F, 1.024880F,
   0.502447F, 2.596945F,
   0.538748F, 4.284797F,
   0.594507F, 1.400231F,
   0.601562F, 0.715119F,
   0.615667F, 6.097708F,
   0.625075F, 2.174328F,
   0.662782F, 3.231284F,
   0.690967F, 1.751661F,
   0.696845F, 0.878992F,
   0.710639F, 1.185743F,
   0.715220F, 0.482280F,
   0.751490F, 2.486668F,
   0.764345F, 4.108078F,
   0.767472F, 12.8F,
   0.769099F, 1.658386F,
   0.803928F, 0.792686F,
   0.810832F, 1.183175F,
   0.818967F, 2.972155F,
   0.821480F, 6.255056F,
   0.833469F, 1.761157F,
   0.861276F, 2.207579F,
   0.863275F, 0.599961F,
   0.880429F, 0.913681F,
   0.880816F, 3.532393F,
   0.881643F, 1.383632F,
   0.892825F, 4.900598F,
   0.906974F, 8.542764F,
   0.925368F, 1.772085F,
   0.928281F, 2.841080F,
   0.933752F, 1.101603F,
   0.947124F, 2.305683F,
   0.956012F, 0.517399F,
   0.961870F, 3.892433F,
   0.972823F, 5.841643F,
   0.977796F, 1.463091F,
   0.981803F, 0.833284F,
   1.008362F, 2.077002F,
   1.011696F, 3.039251F,
   1.031157F, 1.162086F,
   1.031888F, 0.332692F,
   1.050599F, 4.104371F,
   1.074819F, 1.668828F,
   1.092217F, 2.572842F,
   1.108887F, 5.736302F,
   1.112667F, 0.769072F,
   1.135654F, 15.9999F,
   1.143096F, 10.587959F,
   1.168006F, 1.241697F,
   1.185414F, 3.211269F,
   1.186919F, 2.000422F,
   1.187154F, 4.421983F,
   1.243327F, 7.268204F
};

const Float32 E_ROM_qua_gain7b[NB_QUA_GAIN7B * 2] = {
   0.012445F, 0.215546F,
   0.028326F, 0.965442F,
   0.053042F, 0.525819F,
   0.065409F, 1.495322F,
   0.078212F, 2.323725F,
   0.100504F, 0.751276F,
   0.112617F, 3.427530F,
   0.113124F, 0.309583F,
   0.121763F, 1.140685F,
   0.143515F, 7.519609F,
   0.162430F, 0.568752F,
   0.164940F, 1.904113F,
   0.165429F, 4.947562F,
   0.194985F, 0.855463F,
   0.213527F, 1.281019F,
   0.223544F, 0.414672F,
   0.243135F, 2.781766F,
   0.257180F, 1.659565F,
   0.269488F, 0.636749F,
   0.286539F, 1.003938F,
   0.328124F, 2.225436F,
   0.328761F, 0.330278F,
   0.336807F, 11.500983F,
   0.339794F, 3.805726F,
   0.344454F, 1.494626F,
   0.346165F, 0.738748F,
   0.363605F, 1.141454F,
   0.398729F, 0.517614F,
   0.415276F, 2.928666F,
   0.416282F, 0.862935F,
   0.423421F, 1.873310F,
   0.444151F, 0.202244F,
   0.445842F, 1.301113F,
   0.455671F, 5.519512F,
   0.484764F, 0.387607F,
   0.488696F, 0.967884F,
   0.488730F, 0.666771F,
   0.508189F, 1.516224F,
   0.508792F, 2.348662F,
   0.531504F, 3.883870F,
   0.548649F, 1.112861F,
   0.551182F, 0.514986F,
   0.564397F, 1.742030F,
   0.566598F, 0.796454F,
   0.589255F, 3.081743F,
   0.598816F, 1.271936F,
   0.617654F, 0.333501F,
   0.619073F, 2.040522F,
   0.625282F, 0.950244F,
   0.630798F, 0.594883F,
   0.638918F, 4.863197F,
   0.650102F, 1.464846F,
   0.668412F, 0.747138F,
   0.669490F, 2.583027F,
   0.683757F, 1.125479F,
   0.691216F, 1.739274F,
   0.718441F, 3.297789F,
   0.722608F, 0.902743F,
   0.728827F, 2.194941F,
   0.729586F, 0.633849F,
   0.730907F, 7.432957F,
   0.731017F, 0.431076F,
   0.731543F, 1.387847F,
   0.759183F, 1.045210F,
   0.768606F, 1.789648F,
   0.771245F, 4.085637F,
   0.772613F, 0.778145F,
   0.786483F, 1.283204F,
   0.792467F, 2.412891F,
   0.802393F, 0.544588F,
   0.807156F, 0.255978F,
   0.814280F, 1.544409F,
   0.817839F, 0.938798F,
   0.826959F, 2.910633F,
   0.830453F, 0.684066F,
   0.833431F, 1.171532F,
   0.841208F, 1.908628F,
   0.846440F, 5.333522F,
   0.868280F, 0.841519F,
   0.868662F, 1.435230F,
   0.871449F, 3.675784F,
   0.881317F, 2.245058F,
   0.882020F, 0.480249F,
   0.882476F, 1.105804F,
   0.902856F, 0.684850F,
   0.904419F, 1.682113F,
   0.909384F, 2.787801F,
   0.916558F, 7.500981F,
   0.918444F, 0.950341F,
   0.919721F, 1.296319F,
   0.940272F, 4.682978F,
   0.940273F, 1.991736F,
   0.950291F, 3.507281F,
   0.957455F, 1.116284F,
   0.957723F, 0.793034F,
   0.958217F, 1.497824F,
   0.962628F, 2.514156F,
   0.968507F, 0.588605F,
   0.974739F, 0.339933F,
   0.991738F, 1.750201F,
   0.997210F, 0.936131F,
   1.002422F, 1.250008F,
   1.006040F, 2.167232F,
   1.008848F, 3.129940F,
   1.014404F, 5.842819F,
   1.027798F, 4.287319F,
   1.039404F, 1.489295F,
   1.039628F, 8.947958F,
   1.043214F, 0.765733F,
   1.045089F, 2.537806F,
   1.058994F, 1.031496F,
   1.060415F, 0.478612F,
   1.072132F, 12.8F,
   1.074778F, 1.910049F,
   1.076570F, 15.9999F,
   1.107853F, 3.843067F,
   1.110673F, 1.228576F,
   1.110969F, 2.758471F,
   1.140058F, 1.603077F,
   1.155384F, 0.668935F,
   1.176229F, 6.717108F,
   1.179008F, 2.011940F,
   1.187735F, 0.963552F,
   1.199569F, 4.891432F,
   1.206311F, 3.316329F,
   1.215323F, 2.507536F,
   1.223150F, 1.387102F,
   1.296012F, 9.684225F
};

/*
 * 1/4 resolution interpolation filter (-3 dB at 0.791*fs/2)
 */
const Float32 E_ROM_inter4_1[UP_SAMP * L_INTERPOL1 + 1] = {
   /* cut-off frequency at 0.9*fs/2 */
   0.900000F,
   0.818959F, 0.604850F, 0.331379F, 0.083958F,
   -0.075795F, -0.130717F, -0.105685F, -0.046774F,
   0.004467F, 0.027789F, 0.025642F, 0.012571F,
   0.001927F, -0.001571F, -0.000753F, 0.000000f
};

const Word16 E_ROM_inter4_2[UP_SAMP * 2 * L_INTERPOL2] = {
   0, 1, 2, 1,
   -2, -7, -10, -7,
   4, 19, 28, 22,
   -2, -33, -55, -49,
   -10, 47, 91, 92,
   38, -52, -133, -153,
   -88, 43, 175, 231,
   165, -9, -209, -325,
   -275, -60, 226, 431,
   424, 175, -213, -544,
   -619, -355, 153, 656,
   871, 626, -16, -762,
   -1207, -1044, -249, 853,
   1699, 1749, 780, -923,
   -2598, -3267, -2147, 968,
   5531, 10359, 14031, 15401,
   14031, 10359, 5531, 968,
   -2147, -3267, -2598, -923,
   780, 1749, 1699, 853,
   -249, -1044, -1207, -762,
   -16, 626, 871, 656,
   153, -355, -619, -544,
   -213, 175, 424, 431,
   226, -60, -275, -325,
   -209, -9, 165, 231,
   175, 43, -88, -153,
   -133, -52, 38, 92,
   91, 47, -10, -49,
   -55, -33, -2, 22,
   28, 19, 4, -7,
   -10, -7, -2, 1,
   2, 1, 0, 0
};

/*
 * High band gain table for 23.85 kbit/s mode
 * For synthesis of signal at 16kHz with HF extension.
 */
const Float32 E_ROM_hp_gain[16] = {
   0.2212F, 0.2852F, 0.3416F, 0.3954F,
   0.4532F, 0.5114F, 0.5691F, 0.6265F,
   0.6842F, 0.7450F, 0.8173F, 0.9060F,
   1.0236F, 1.1996F, 1.4825F, 1.9976F
};


/*
 * isp and isf tables for initialization
 */
const Word16 E_ROM_isp[M] = {
   32138, 30274, 27246, 23170, 18205, 12540, 6393, 0,
   -6393, -12540, -18205, -23170, -27246, -30274, -32138, 1475
};

const Float32 E_ROM_isf[M] = {
   400.0, 800.0, 1200.0, 1600.0, 2000.0, 2400.0, 2800.0, 3200.0,
   3600.0, 4000.0, 4400.0, 4800.0, 5200.0, 5600.0, 6000.0, 1500.0
};

/*
 * Table of lag_window for autocorrelation.
 * noise floor = 1.0001   = (0.9999  on r[1] ..r[16])
 * Bandwidth expansion = 60 Hz
 * Sampling frequency  = 12800 Hz
 */
const Float32 E_ROM_lag_window[M] = { /*1.00000000    (not stored)*/
   0.99946642F,
   0.99816680F,
   0.99600452F,
   0.99298513F,
   0.98911655F,
   0.98440880F,
   0.97887397F,
   0.97252619F,
   0.96538186F,
   0.95745903F,
   0.94877797F,
   0.93936038F,
   0.92922986F,
   0.91841155F,
   0.90693212F,
   0.89481968F
};

/*
 * means of ISFs
 */
const Word16 E_ROM_mean_isf[ORDER] = {

   738, 1326, 2336, 3578, 4596, 5662, 6711, 7730,
   8750, 9753, 10705, 11728, 12833, 13971, 15043, 4037
};

const Float32 E_ROM_f_mean_isf[ORDER] = {
   288.411774F,
   518.149414F,
   912.352051F,
   1397.743652F,
   1795.418823F,
   2211.536133F,
   2621.461182F,
   3019.680176F,
   3417.989746F,
   3809.700928F,
   4181.547363F,
   4581.064941F,
   5012.819824F,
   5457.521484F,
   5876.145020F,
   1576.906494F
};

/* ISF mean for comfort noise */
const Float32 E_ROM_mean_isf_noise[16] = {
   186.625259F,
   429.613403F,
   864.525085F,
   1276.010254F,
   1648.228394F,
   2039.998535F,
   2420.943115F,
   2828.022461F,
   3214.584961F,
   3575.200195F,
   3944.685303F,
   4338.971191F,
   4743.657227F,
   5149.993164F,
   5533.081055F,
   1485.457031F
};

/*
 * Upsample FIR filter coefficients.
 * 1/5 resolution interpolation filter (in Q14)
 * -1.5dB @ 6kHz, -6dB @ 6.4kHz, -10dB @ 6.6kHz, -20dB @ 6.9kHz, -25dB @ 7kHz, -55dB @ 8kHz
 */
const Word16 E_ROM_fir_up[120] = {
   -1, -4, -7, -6, 0,
   12, 24, 30, 23, 0,
   -33, -62, -73, -52, 0,
   68, 124, 139, 96, 0,
   -119, -213, -235, -160, 0,
   191, 338, 368, 247, 0,
   -291, -510, -552, -369, 0,
   430, 752, 812, 542, 0,
   -634, -1111, -1204, -809, 0,
   963, 1708, 1881, 1288, 0,
   -1616, -2974, -3432, -2496, 0,
   3792, 8219, 12368, 15317, 16384,
   15317, 12368, 8219, 3792, 0,
   -2496, -3432, -2974, -1616, 0,
   1288, 1881, 1708, 963, 0,
   -809, -1204, -1111, -634, 0,
   542, 812, 752, 430, 0,
   -369, -552, -510, -291, 0,
   247, 368, 338, 191, 0,
   -160, -235, -213, -119, 0,
   96, 139, 124, 68, 0,
   -52, -73, -62, -33, 0,
   23, 30, 24, 12, 0,
   -6, -7, -4, -1, 0
};

/*
 * Downsample FIR filter coefficients (x4/5)
 */
const Word16 E_ROM_fir_down[120] = {
   -1, -3, -6, -5,
   0, 9, 19, 24,
   18, 0, -26, -50,
   -58, -41, 0, 54,
   99, 111, 77, 0,
   -95, -170, -188, -128,
   0, 153, 270, 294,
   198, 0, -233, -408,
   -441, -295, 0, 344,
   601, 649, 434, 0,
   -507, -888, -964, -647,
   0, 770, 1366, 1505,
   1030, 0, -1293, -2379,
   -2746, -1997, 0, 3034,
   6575, 9894, 12254, 13107,
   12254, 9894, 6575, 3034,
   0, -1997, -2746, -2379,
   -1293, 0, 1030, 1505,
   1366, 770, 0, -647,
   -964, -888, -507, 0,
   434, 649, 601, 344,
   0, -295, -441, -408,
   -233, 0, 198, 294,
   270, 153, 0, -128,
   -188, -170, -95, 0,
   77, 111, 99, 54,
   0, -41, -58, -50,
   -26, 0, 18, 24,
   19, 9, 0, -5,
   -6, -3, -1, 0
};

/* 16-byte align end */

/* weighting of the correlation function in open loop LTP search */
const Float32 E_ROM_corrweight[199]= {
   0.32872895427062F, 0.32940279686838F, 0.33008397607000F, 0.33077263774882F,
   0.33146893201434F, 0.33217301337469F, 0.33288504090694F, 0.33360517843561F,
   0.33433359471985F, 0.33507046364991F, 0.33581596445331F, 0.33657028191132F,
   0.33733360658650F, 0.33810613506181F, 0.33888807019203F, 0.33967962136837F,
   0.34048100479698F, 0.34129244379225F, 0.34211416908597F, 0.34294641915312F,
   0.34378944055574F, 0.34464348830569F, 0.34550882624786F, 0.34638572746505F,
   0.34727447470606F, 0.34817536083854F, 0.34908868932841F, 0.35001477474762F,
   0.35095394331240F, 0.35190653345404F, 0.35287289642476F, 0.35385339694105F,
   0.35484841386753F, 0.35585834094415F, 0.35688358756025F, 0.35792457957894F,
   0.35898176021598F, 0.36005559097716F, 0.36114655265930F, 0.36225514641972F,
   0.36338189491998F, 0.36452734355021F, 0.36569206174066F, 0.36687664436818F,
   0.36808171326584F, 0.36930791884477F, 0.37055594183846F, 0.37182649518049F,
   0.37312032602824F, 0.37443821794611F, 0.37578099326358F, 0.37714951562500F,
   0.37854469274995F, 0.37996747942529F, 0.38141888075233F, 0.38289995567554F,
   0.38441182082240F, 0.38595565468744F, 0.38753270219810F, 0.38914427970450F,
   0.39079178044081F, 0.39247668051248F, 0.39420054547049F, 0.39596503754272F,
   0.39777192360213F, 0.39962308396279F, 0.40152052210853F, 0.40346637547426F,
   0.40546292741845F, 0.40751262054712F, 0.40961807157476F, 0.41178208793838F,
   0.41400768641652F, 0.41629811404829F, 0.41865687169900F, 0.42108774068111F,
   0.42359481291430F, 0.42618252519967F, 0.42885569829436F, 0.43161958160963F,
   0.43447990452326F, 0.43744293550645F, 0.44051555052483F, 0.44370531250000F,
   0.44702056402976F, 0.45047053608887F, 0.45406547610287F, 0.45781679965235F,
   0.46173727119116F, 0.46584122063850F, 0.47014480466210F, 0.47466632408736F,
   0.47942661240838F, 0.48444951522162F, 0.49000000000000F, 0.50000000000000F,
   0.50000000000000F, 0.50000000000000F, 0.50000000000000F, 0.50000000000000F,
   0.50000000000000F, 0.50000000000000F, 0.49000000000000F, 0.48444951522162F,
   0.47942661240838F, 0.47466632408736F, 0.47014480466210F, 0.46584122063850F,
   0.46173727119116F, 0.45781679965235F, 0.45406547610287F, 0.45047053608887F,
   0.44702056402976F, 0.44370531250000F, 0.44051555052483F, 0.43744293550645F,
   0.43447990452326F, 0.43161958160963F, 0.42885569829436F, 0.42618252519967F,
   0.42359481291430F, 0.42108774068111F, 0.41865687169900F, 0.41629811404829F,
   0.41400768641652F, 0.41178208793838F, 0.40961807157476F, 0.40751262054712F,
   0.40546292741845F, 0.40346637547426F, 0.40152052210853F, 0.39962308396279F,
   0.39777192360213F, 0.39596503754272F, 0.39420054547049F, 0.39247668051248F,
   0.39079178044081F, 0.38914427970450F, 0.38753270219810F, 0.38595565468744F,
   0.38441182082240F, 0.38289995567554F, 0.38141888075233F, 0.37996747942529F,
   0.37854469274995F, 0.37714951562500F, 0.37578099326358F, 0.37443821794611F,
   0.37312032602824F, 0.37182649518049F, 0.37055594183846F, 0.36930791884477F,
   0.36808171326584F, 0.36687664436818F, 0.36569206174066F, 0.36452734355021F,
   0.36338189491998F, 0.36225514641972F, 0.36114655265930F, 0.36005559097716F,
   0.35898176021598F, 0.35792457957894F, 0.35688358756025F, 0.35585834094415F,
   0.35484841386753F, 0.35385339694105F, 0.35287289642476F, 0.35190653345404F,
   0.35095394331240F, 0.35001477474762F, 0.34908868932841F, 0.34817536083854F,
   0.34727447470606F, 0.34638572746505F, 0.34550882624786F, 0.34464348830569F,
   0.34378944055574F, 0.34294641915312F, 0.34211416908597F, 0.34129244379225F,
   0.34048100479698F, 0.33967962136837F, 0.33888807019203F, 0.33810613506181F,
   0.33733360658650F, 0.33657028191132F, 0.33581596445331F, 0.33507046364991F,
   0.33433359471985F, 0.33360517843561F, 0.33288504090694F, 0.33217301337469F,
   0.33146893201434F, 0.33077263774882F, 0.33008397607000F, 0.32940279686838F,
   0.32872895427062F, 0.32806230648432F, 0.32740271564992F
};

const UWord8 E_ROM_tipos[36] = {
   0, 1, 2, 3, /* starting point &ipos[0], 1st iter */
   1, 2, 3, 0, /* starting point &ipos[4], 2nd iter */
   2, 3, 0, 1, /* starting point &ipos[8], 3rd iter */
   3, 0, 1, 2, /* starting point &ipos[12], 4th iter */
   0, 1, 2, 3,
   1, 2, 3, 0,
   2, 3, 0, 1,
   3, 0, 1, 2,
   0, 1, 2, 3 /* end point for 24 pulses &ipos[35], 4th iter */
};

/*
 * Chebyshev polynomial grid points
 *
 * grid[0] = 1.0;
 * grid[grid_points + 1] = -1.0;
 * for (i = 1; i < grid_points; i++)
 *   grid[i] = cos((6.283185307*i)/(2.0*grid_points));
 *
 */
const Float32 E_ROM_grid[101] = {
   1.00000F, 0.999507F, 0.998027F, 0.995562F, 0.992115F,
   0.987688F, 0.982287F, 0.975917F, 0.968583F, 0.960294F,
   0.951057F, 0.940881F, 0.929776F, 0.917755F, 0.904827F,
   0.891007F, 0.876307F, 0.860742F, 0.844328F, 0.827081F,
   0.809017F, 0.790155F, 0.770513F, 0.750111F, 0.728969F,
   0.707107F, 0.684547F, 0.661312F, 0.637424F, 0.612907F,
   0.587785F, 0.562083F, 0.535827F, 0.509041F, 0.481754F,
   0.453990F, 0.425779F, 0.397148F, 0.368124F, 0.338738F,
   0.309017F, 0.278991F, 0.248690F, 0.218143F, 0.187381F,
   0.156434F, 0.125333F, 0.0941082F, 0.0627904F, 0.0314107F,
   -8.09643e-008F,
   -0.0314108F, -0.0627906F, -0.0941084F, -0.125333F, -0.156435F,
   -0.187381F, -0.218143F, -0.248690F, -0.278991F, -0.309017F,
   -0.338738F, -0.368125F, -0.397148F, -0.425779F, -0.453991F,
   -0.481754F, -0.509041F, -0.535827F, -0.562083F, -0.587785F,
   -0.612907F, -0.637424F, -0.661312F, -0.684547F, -0.707107F,
   -0.728969F, -0.750111F, -0.770513F, -0.790155F, -0.809017F,
   -0.827081F, -0.844328F, -0.860742F, -0.876307F, -0.891007F,
   -0.904827F, -0.917755F, -0.929777F, -0.940881F, -0.951057F,
   -0.960294F, -0.968583F, -0.975917F, -0.982287F, -0.987688F,
   -0.992115F, -0.995562F, -0.998027F, -0.999507F, -1.00000F
};

/*
 * Bandpass FIR filter coefficients for higher band generation (gain=4.0)
 */
const Float32 E_ROM_fir_6k_7k[31] = {
   -0.001005F, 0.001428F, 0.000965F, -0.000863F, -0.011296F,
   0.034231F, -0.043397F, -0.000000F, 0.115906F, -0.271028F,
   0.376868F, -0.335243F, 0.108276F, 0.237003F, -0.549363F,
   0.675000F, -0.549363F, 0.237003F, 0.108276F, -0.335243F,
   0.376868F, -0.271028F, 0.115906F, -0.000000F, -0.043397F,
   0.034231F, -0.011296F, -0.000863F, 0.000965F, 0.001428F,
   -0.001005F
};

/*
 * LTP gain & fixed codebook gain consealement
 */

/* attenuation factors for codebook gain in lost frames */
const Word16 E_ROM_cdown_unusable[7] = {32767, 16384, 8192, 8192, 8192, 4915, 3277};

const Word16 E_ROM_cdown_usable[7] = {32767, 32113, 32113, 32113, 32113, 32113, 22938};

/* attenuation factors for adaptive codebook gain in lost frames */
const Word16 E_ROM_pdown_unusable[7] = {32767, 31130, 29491, 24576, 7537, 1638, 328};
/* attenuation factors for adaptive codebook gain in bad frames */
const Word16 E_ROM_pdown_usable[7] = {32767, 32113, 31457, 24576, 7537, 1638, 328};


/* FIR filter coefficients (interpol) */
const Float32 E_ROM_fir_ipol[61] = {
   0.999980F,
   0.934870F, 0.754870F, 0.501632F, 0.231474F, -0.000000F,
   -0.152337F, -0.209502F, -0.181536F, -0.098630F, 0.000000F,
   0.078607F, 0.114831F, 0.104252F, 0.058760F, -0.000000F,
   -0.049374F, -0.073516F, -0.067781F, -0.038681F, 0.000000F,
   0.033082F, 0.049550F, 0.045881F, 0.026258F, -0.000000F,
   -0.022499F, -0.033672F, -0.031122F, -0.017761F, 0.000000F,
   0.015088F, 0.022452F, 0.020614F, 0.011674F, -0.000000F,
   -0.009736F, -0.014331F, -0.012999F, -0.007264F, 0.000000F,
   0.005872F, 0.008488F, 0.007546F, 0.004123F, -0.000000F,
   -0.003163F, -0.004431F, -0.003804F, -0.001997F, 0.000000F,
   0.001388F, 0.001829F, 0.001459F, 0.000702F, -0.000000F,
   -0.000383F, -0.000424F, -0.000267F, -0.000091F, 0.000000F
};


/*
 * LPC interpolation coef
 */
const Float32 E_ROM_f_interpol_frac[NB_SUBFR] = {0.45F, 0.8F, 0.96F, 1.0F};

/*
 * LPC interpolation coef in Q15
 */
const Word16 E_ROM_interpol_frac[NB_SUBFR] = {14746, 26214, 31457, 32767};

/*
 * Math tables
 */

/* table used in power of two computation */
const Word16 E_ROM_pow2[33] = {
   16384, 16743, 17109, 17484, 17867, 18258, 18658, 19066, 19484, 19911,
   20347, 20792, 21247, 21713, 22188, 22674, 23170, 23678, 24196, 24726,
   25268, 25821, 26386, 26964, 27554, 28158, 28774, 29405, 30048, 30706,
   31379, 32066, 32767
};

/* table used in inverse square root computation */
const Word16 E_ROM_isqrt[49] = {
   32767, 31790, 30894, 30070, 29309, 28602, 27945, 27330, 26755, 26214,
   25705, 25225, 24770, 24339, 23930, 23541, 23170, 22817, 22479, 22155,
   21845, 21548, 21263, 20988, 20724, 20470, 20225, 19988, 19760, 19539,
   19326, 19119, 18919, 18725, 18536, 18354, 18176, 18004, 17837, 17674,
   17515, 17361, 17211, 17064, 16921, 16782, 16646, 16514, 16384
};

/* table used in logarithm computation */
const Word16 E_ROM_log2[33] = {
   0, 1455, 2866, 4236, 5568, 6863, 8124, 9352, 10549, 11716,
   12855, 13967, 15054, 16117, 17156, 18172, 19167, 20142, 21097, 22033,
   22951, 23852, 24735, 25603, 26455, 27291, 28113, 28922, 29716, 30497,
   31266, 32023, 32767
};

/* table of cos(x) in Q15 */
const Word16 E_ROM_cos[129] = {
   32767,
   32758, 32729, 32679, 32610, 32522, 32413, 32286, 32138,
   31972, 31786, 31581, 31357, 31114, 30853, 30572, 30274,
   29957, 29622, 29269, 28899, 28511, 28106, 27684, 27246,
   26791, 26320, 25833, 25330, 24812, 24279, 23732, 23170,
   22595, 22006, 21403, 20788, 20160, 19520, 18868, 18205,
   17531, 16846, 16151, 15447, 14733, 14010, 13279, 12540,
   11793, 11039, 10279, 9512, 8740, 7962, 7180, 6393,
   5602, 4808, 4011, 3212, 2411, 1608, 804, 0,
   -804, -1608, -2411, -3212, -4011, -4808, -5602, -6393,
   -7180, -7962, -8740, -9512, -10279, -11039, -11793, -12540,
   -13279, -14010, -14733, -15447, -16151, -16846, -17531, -18205,
   -18868, -19520, -20160, -20788, -21403, -22006, -22595, -23170,
   -23732, -24279, -24812, -25330, -25833, -26320, -26791, -27246,
   -27684, -28106, -28511, -28899, -29269, -29622, -29957, -30274,
   -30572, -30853, -31114, -31357, -31581, -31786, -31972, -32138,
   -32286, -32413, -32522, -32610, -32679, -32729, -32758, -32768
};


/*
 * excitation energy adjustment depending on speech coder mode used
 */
const Float32 E_ROM_en_adjust[9] = {
   -1.7969F, /* mode0 = 7k  :  -5.4dB  */
   -1.3984F, /* mode1 = 9k  :  -4.2dB  */
   -1.1016F, /* mode2 = 12k :  -3.3dB  */
   -1.0000F, /* mode3 = 14k :  -3.0dB  */
   -0.9531F, /* mode4 = 16k :  -2.85dB */
   -0.8984F, /* mode5 = 18k :  -2.7dB  */
   -0.8984F, /* mode6 = 20k :  -2.7dB  */
   -0.8984F, /* mode7 = 23k :  -2.7dB  */
   -0.8984F  /* mode8 = 24k :  -2.7dB  */
};









