/*
    qutim

    Copyright (c) 2008 by Rustam Chakin <qutim.develop@gmail.com>

 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************
*/


#include "qutim.h"

eventEater::eventEater(QWidget *parent) : QObject(parent)
{
	changed = false;
}

eventEater::~eventEater()
{
	
}

bool eventEater::eventFilter(QObject *obj, QEvent *event)
{
	if ( event->type() == QEvent::MouseButtonDblClick ||
			event->type() == QEvent::MouseButtonPress ||
			event->type() == QEvent::MouseButtonRelease ||
			event->type() == QEvent::MouseMove ||
			event->type() == QEvent::MouseTrackingChange ||
			event->type() == QEvent::KeyPress ||
			event->type() == QEvent::KeyRelease ||
			event->type() == QEvent::KeyboardLayoutChange)
	{
		changed = true;
	}
	
	return QObject::eventFilter(obj, event);
}



qutIM::qutIM(QWidget *parent)
    : QWidget(parent)
{
	timer = new QTimer(this);
	connect(timer, SIGNAL(timeout()), this, SLOT(hide()));
	
	ui.setupUi(this);
	eventObject = new eventEater(this);
	qApp->installEventFilter(eventObject);
	letMeQuit = false;
	onlineList = NULL;
	offlineList = NULL;
	setAttribute(Qt::WA_AlwaysShowToolTips, true);
	setFocus(Qt::ActiveWindowFocusReason);
	if ( QSystemTrayIcon::isSystemTrayAvailable() )
	{
		createActions();
		createTrayIcon();
		trayIcon->show();
		connect(trayIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)),
	             this, SLOT(trayActivated(QSystemTrayIcon::ActivationReason)));
	}
	currentStatusMenu = NULL;
	unreadMessages = false;
	loadMainSettings();
//	settingsDialog = new qutimSettings(this);
//	connect ( settingsDialog, SIGNAL(generalSettingsChanged()),
//			this, SLOT(reloadGeneralSettings()) );
	
	ui.contactListTree->header()->hide(); // hide contact list header

	ui.contactListTree->setAlternatingRowColors(true); 
	ui.contactListTree->setAllColumnsShowFocus(true);
	ui.contactListTree->header()->setResizeMode(QHeaderView::Stretch);
	ui.contactListTree->header()->setResizeMode(0, QHeaderView::ResizeToContents);
	ui.contactListTree->header()->setResizeMode(2, QHeaderView::ResizeToContents);
//	ui.contactListTree->header()->resizeSection(0,16);
	ui.contactListTree->header()->setStretchLastSection(false);
	ui.contactListTree->hideColumn(3);
	createLoginDialog();
	createMainMenu();
	createAccountLine();
	
	
	if ( QSystemTrayIcon::isSystemTrayAvailable() )
		updateTrayStatus();
	
	connect( ui.contactListTree, SIGNAL(itemClicked(QTreeWidgetItem *, int)),
				this, SLOT(itemClicked(QTreeWidgetItem *, int)));
	connect( ui.contactListTree, SIGNAL(itemDoubleClicked(QTreeWidgetItem *, int)),
					this, SLOT(itemDoubleClicked(QTreeWidgetItem *)));
	
//	ui.contactListTree->setSortingEnabled(true);
//	ui.contactListTree->sortByColumn( 1, Qt::AscendingOrder );
//	ui.contactListTree->resizeColumnToContents(0);
//	show();
	
	connect(timer, SIGNAL(timeout()), this, SLOT(hide()));
	QTimer::singleShot(60000, this, SLOT(checkEventChanging()));
	createContactList();
	updateTrayToolTip();
	aboutWindowOpen = false;
}

qutIM::~qutIM()
{
	
}

void qutIM::createTrayIcon()
{
	trayMenu = new QMenu(this);
	trayMenu->addAction(settingsAction);
	trayMenu->addAction(switchUserAction);
	trayMenu->addSeparator();
	trayMenu->addAction(quitAction);
	trayIcon = new QSystemTrayIcon(this);
	trayIcon->setIcon(QIcon(":/icons/qutim.png"));
	trayIcon->setContextMenu(trayMenu);
}

void qutIM::closeEvent(QCloseEvent *event)
{
	hide();
	if ( !letMeQuit ) 
		event->ignore();
}

void qutIM::trayActivated(QSystemTrayIcon::ActivationReason reason)
{
	switch(reason)
	{
	case QSystemTrayIcon::Trigger:
		if (! unreadMessages )
			setVisible(!isVisible());
		else
			readMessages();
		break;
	default:
		;
	}
	
}

void qutIM::createActions()
{
	quitAction = new QAction(QIcon(":/icons/crystal_project/exit.png"), 
			tr("&Quit"), this);
	settingsAction = new QAction(QIcon(":/icons/crystal_project/settings.png"), 
			tr("&Settings..."), this);
	switchUserAction = new QAction(QIcon(":/icons/crystal_project/switch_user.png"),
			tr("Switch user"), this);
	connect(quitAction, SIGNAL(triggered()), this, SLOT(appQuit()));
	connect(settingsAction, SIGNAL(triggered()), this, SLOT(qutimSettingsMenu()));
	connect(switchUserAction, SIGNAL(triggered()), this, SLOT(switchUser()));
}

void qutIM::appQuit()
{
	
	saveMainSettings(); // save all main settings on exit
	letMeQuit = true;
	foreach(icqAccount *account, icqList)
		account->getProtocol()->getContactListClass()->appExiting();
	QCoreApplication::exit(0);
//	close();

}

void qutIM::saveMainSettings()
{
	QSettings settings(QSettings::IniFormat, QSettings::UserScope, "qutim", "mainsettings");
	
	//window size and position saving
	//save if "save size and position" feature is enabled
	if ( settings.value("mainwindow/saveSizPos", false).toBool())
	{
	settings.beginGroup("mainwindow");
	settings.setValue("size", size());
	settings.setValue("position", pos());
	settings.endGroup();
	}
}

void qutIM::loadMainSettings()
{
	QSettings settings(QSettings::IniFormat, QSettings::UserScope, "qutim", "mainsettings");
	
	currentStyle = settings.value("style/path", ":/qss/default.qss").toString();
	loadStyle(currentStyle);
	QDesktopWidget desktop;
	
	//load size and position if "save size and position" feature is enabled
	
	if ( settings.value("mainwindow/saveSizPos", false).toBool()) 
	{
	QSize size = settings.value("mainwindow/size", QSize(0,0)).toSize();
	QPoint moveTo = settings.value("mainwindow/position", QPoint(-1, -1)).toPoint();
	
	if ( size.width() < 40 || size.height() < 40 )
		resize(150,desktop.height() - 64);
	else
		resize(size);
	
	if ( moveTo.x() < 0 && moveTo.y() < 0 )
		move(desktop.width() - 150, 0);
	else
		move(moveTo);
	} else {
		resize(150,desktop.height() - 64);
		move(desktop.width() - 150, 0);
	}
	
	//hide main window if "hide on startup" feature is enabled
	
//	if( settings.value("mainwindow/hideStart", false).toBool() )
//		hide();
//	else show();
	
	createMenuAccounts = settings.value("mainwindow/accountMenu", true).toBool();
	addToStatusTrayMenu  = false;
	if ( settings.value("systray/inmenu", false).toBool())
		updateCurrentTrayStatusmenu(true);
	
	hideSeparators = settings.value("contactlist/hidesep", false).toBool();
	showOffline = settings.value("contactlist/offline", false).toBool();
	hideEmptyGroups = settings.value("contactlist/hideempty", false).toBool();
	showGroups = settings.value("contactlist/groups", false).toBool();
	 
	if (mergeAccounts = settings.value("contactlist/merge", false).toBool())
//		 createGroups();
		reupdateList();	
	
	sorting = settings.value("contactlist/sort", 3).toInt();
	ui.contactListTree->model()->setHeaderData(0,Qt::Horizontal,QVariant(sorting));
	if ( !settings.value("contactlist/clicons",false).toBool())
		ui.contactListTree->hideColumn(2);
	if ( settings.value("contactlist/ontop", false).toBool())
		setWindowFlags(windowFlags() | Qt::WindowStaysOnTopHint);
	else
		setWindowFlags(windowFlags() & Qt::WindowStaysOnTopHint);
	autoHide = settings.value("contactlist/ahide",false).toBool();
	hideSec = settings.value("contactlist/hidesec",60).toInt();
	ui.contactListTree->setAlternatingRowColors(settings.value("contactlist/colorrow",true).toBool()); 
	
	
}

void qutIM::qutimSettingsMenu()
{
	settingsAction->setDisabled(true);
	switchUserAction->setDisabled(true);
	if (currentStatusMenu)
		currentStatusMenu->setEnabled(false);
	settingsDialog = new qutimSettings(this);
	settingsDialog->setIcqList(&icqList);
	connect ( settingsDialog, SIGNAL(generalSettingsChanged()),
			this, SLOT(reloadGeneralSettings()) );
	connect ( settingsDialog, SIGNAL(destroyed()),
				this, SLOT(destroySettings()) );
	
	settingsDialog->loadAllSettings(); //load all settings to "Settings" window
	settingsDialog->applyDisable(); //disable "Settings" window's Apply button 
	settingsDialog->show();
	settingsDialog->setAttribute(Qt::WA_QuitOnClose, false); //don't close app on "Settings" exit
	settingsDialog->setAttribute(Qt::WA_DeleteOnClose, true);
}

void qutIM::destroySettings()
{
	settingsAction->setDisabled(false);
	switchUserAction->setDisabled(false);
	if ( currentStatusMenu )
		currentStatusMenu->setDisabled(false);
}

void qutIM::createContactList()
{
	foreach(icqAccount *account, icqList)		
	if ( account->getAutoConnect() )
				account->autoconnecting();	

}

void qutIM::createLoginDialog()
{
	QSettings settings(QSettings::IniFormat, QSettings::UserScope, "qutim", "mainsettings");
	if(settings.value("logindialog/showstart", true).toBool())
	{
	loginDialog loginD(this);
	loginD.loadSettings(); //load all login settings
	if( loginD.exec() )
		loginD.saveSettings(); //save all on "sign in"
	}
	
	if( settings.value("mainwindow/hideStart", false).toBool() )
		hide();
	else show();
//	loadMainSettings();
}

void qutIM::createAccountLine()
{
	QSettings settings(QSettings::IniFormat, QSettings::UserScope, "qutim", "accounts");
	QStringList accountList = settings.value("accounts/list").toStringList();
	int index = 2, accountLineWidth = 0;
	foreach( QString accountFromList, accountList)
	{
		icqAccount *account = new icqAccount(trayIcon, ui.contactListTree, accountFromList, this);
		connect(account, SIGNAL(statusChanged(QIcon *)),
				this, SLOT(updateTrayIcon(QIcon *)));
		
		
		
		account->createAccountButton(ui.gridLayout1, ui.accountLine, index);
		icqList.append(account);
		
		if( createMenuAccounts )
			account->createMenuAccount(trayMenu, settingsAction); //add account status to tray menu
		
		if ( addToStatusTrayMenu )
		{
				account->createTrayMenuStatus(currentStatusMenu);
				connect(account, SIGNAL(changeStatusInTrayMenu(const QString &)),
						this, SLOT(setStatusFrom(const QString &)));
				
		}
		
		
		connect(account, SIGNAL(getNewMessage()),
				this, SLOT(getNewMessage()));
		
		if ( mergeAccounts )
					account->getProtocol()->getContactListClass()->mergeAccounts(true);
		
//		if ( account->getAutoConnect() )
//			account->autoconnecting();
		
		
		accountLineWidth += 23;
		++index;
		
		account->getProtocol()->getContactListClass()->initializaMenus(addFindUsers, serviceMessages ,sendMultiple,
				privacyList, changeMyDetails, changeMyPassword);
	}
	
	if( createMenuAccounts )
		trayMenu->insertSeparator(settingsAction);
	
	if( accountLineWidth > 100)setMinimumSize(QSize(accountLineWidth, 100));
	
		
}

void qutIM::reloadGeneralSettings()
{
	QSettings settings(QSettings::IniFormat, QSettings::UserScope, "qutim", "mainsettings");
	
	QString cStyle = settings.value("style/path", ":/qss/default.qss").toString();
	
	if ( cStyle != currentStyle)
	{
		currentStyle = cStyle;
		loadStyle(currentStyle);
	}
	
	bool tmpAccountMenu = settings.value("mainwindow/accountMenu", true).toBool();
	if ( createMenuAccounts != tmpAccountMenu)
	{
		foreach( icqAccount *account, icqList )
		{
			if ( tmpAccountMenu)
				
				account->createMenuAccount(trayMenu, settingsAction);
			else
				account->removeMenuAccount(trayMenu);
		}
		if ( tmpAccountMenu )
			trayMenu->insertSeparator(settingsAction);
		createMenuAccounts = tmpAccountMenu;
	}
	updateTrayStatus();	
	updateCurrentTrayStatusmenu(settings.value("systray/inmenu", false).toBool());
	if (currentStatusMenu)
		currentStatusMenu->setEnabled(false);
	
	bool hideSep = settings.value("contactlist/hidesep", false).toBool();
	bool hdEmp = settings.value("contactlist/hideempty", false).toBool(); 
	bool mrgAcc = settings.value("contactlist/merge", false).toBool();	
	bool shGrp = settings.value("contactlist/groups", false).toBool();
	bool shOff = settings.value("contactlist/offline", false).toBool();
	if ( mergeAccounts != mrgAcc)
	{
		if ( mrgAcc )
		{
			hideEmptyGroups = hdEmp;
			showGroups = shGrp;
			showOffline = shOff;
			hideSeparators = hideSep;
			mergeAccountsToList(mrgAcc);
		} else 
		{
			mergeAccountsToList(mrgAcc);
			hideEmptyGroups = hdEmp;
			showGroups = shGrp;
			showOffline = shOff;
			hideSeparators = hideSep;
		}
	}
	else 
	{
		
		if ( hideSeparators != hideSep )
		{
					hideSeparators = hideSep;
					setHideSeparators(hideSeparators);
		}
		
		if ( showGroups != shGrp )
		{
			showHideGroups(shGrp);
				
				
		}
		if ( hideEmptyGroups != hdEmp)
			hideEmptyGr(hdEmp);
		
		if ( showOffline != shOff)
			hideOffline(shOff);
		
		
	}
	
	quint8 tmpSort = settings.value("contactlist/sort", 3).toInt();
	if ( tmpSort != sorting)
	{
		sorting = tmpSort;
		ui.contactListTree->model()->setHeaderData(0,Qt::Horizontal,QVariant(sorting));
		updateSorting();
	}
	ui.contactListTree->setColumnHidden(2, !settings.value("contactlist/clicons", false).toBool());
	if ( settings.value("contactlist/ontop", false).toBool())
		
		setWindowFlags(windowFlags() | Qt::WindowStaysOnTopHint);
	else
		setWindowFlags(windowFlags() & ~0x00040000);
	show();
	
	
	autoHide = settings.value("contactlist/ahide",false).toBool();
	hideSec = settings.value("contactlist/hidesec",60).toInt();
	if ( !autoHide )
		timer->stop();
	ui.contactListTree->setAlternatingRowColors(settings.value("contactlist/colorrow",true).toBool()); 
}

void qutIM::addAccount(const QString &account)
{
	icqAccount *newaccount = new icqAccount(trayIcon, ui.contactListTree, account, this);
	connect(newaccount, SIGNAL(statusChanged(QIcon*)),
			this, SLOT(updateTrayIcon(QIcon*)));
	icqList.append(newaccount);
	 newaccount->createAccountButton(ui.gridLayout1, ui.accountLine, icqList.size() + 2);
	if( icqList.size() * 23 > 100 )
		setMinimumSize(QSize(icqList.size()*23 + 23, 100));
	if ( createMenuAccounts )
		if( icqList.count() - 1 )
			newaccount->createMenuAccount(trayMenu, icqList.at(0)->getAction());
		else
		{
			newaccount->createMenuAccount(trayMenu, settingsAction);
			trayMenu->insertSeparator(settingsAction);
		}
	if ( addToStatusTrayMenu )
	{
				newaccount->createTrayMenuStatus(currentStatusMenu);
				connect(newaccount, SIGNAL(changeStatusInTrayMenu(const QString &)),
										this, SLOT(setStatusFrom(const QString &)));
				
	}
	
	connect(newaccount, SIGNAL(getNewMessage()),
				this, SLOT(getNewMessage()));
	
	
	if ( mergeAccounts )
				newaccount->getProtocol()->getContactListClass()->mergeAccounts(true);
	if ( newaccount->getAutoConnect() )
				newaccount->autoconnecting();
	
	newaccount->getProtocol()->getContactListClass()->initializaMenus(addFindUsers, serviceMessages, 
			sendMultiple, privacyList, changeMyDetails, changeMyPassword);
	
	
}

void qutIM::removeAccount(const QString &account)
{
	foreach( icqAccount *tmpAccount, icqList)
	{
		if ( account == tmpAccount->getIcquin() )
		{
			
			if ( createMenuAccounts )
				tmpAccount->removeMenuAccount(trayMenu);
			
			if ( addToStatusTrayMenu )
			{
						tmpAccount->removeTrayMenuStatus(currentStatusMenu);
						disconnect(tmpAccount, 0,
											this, 0);
			}
			
			tmpAccount->removeAccountButton();
			tmpAccount->removeContactList();
			tmpAccount->deleteingAccount = true;
			icqList.removeAt(icqList.indexOf(tmpAccount));
			disconnect(tmpAccount, 0,
					this, 0);
			if ( mergeAccounts )	
			foreach( treeBuddyItem *buddy,
					tmpAccount->getProtocol()->getContactListClass()->getBuddyList())
				{
					if ( buddy->isOffline )
						offlineList->removeChild(buddy);
					else
						onlineList->removeChild(buddy);
						
						delete buddy;
				}
			
			
			delete tmpAccount;
			updateTrayStatus();
			break;
		}
	}
//	if ( mergeAccounts )
//		reupdateList();
}

void qutIM::switchUser()
{
	loginDialog loginD(this);
	loginD.setRunType(true);
	connect(&loginD, SIGNAL(addingAccount(const QString &)),
			this, SLOT(addAccount(const QString &)));
	connect(&loginD, SIGNAL(removingAccount(const QString &)),
				this, SLOT(removeAccount(const QString &)));
	loginD.loadSettings(); //load all login settings
	if( loginD.exec() )
		loginD.saveSettings(); //save all on "sign in"
}

void qutIM::updateTrayStatus()
{
	QSettings settings(QSettings::IniFormat, QSettings::UserScope, "qutim", "mainsettings");
	switch ( settings.value("systray/show", 1).toInt() )
		{
		case 0:
			trayIcon->setIcon(QIcon(":/icons/qutim.png"));
			tempIcon = QIcon(":/icons/qutim.png");
//			addToStatusTrayMenu = false;
			updateCurrentTrayStatusmenu(false);
			break;
		case 1:
			setStatusFrom(settings.value("systray/current", "").toString());
			break;
		default:
			break;
		}
}

void qutIM::setStatusFrom(const QString &icqUin)
{
	
	bool accountInList = false;
	if ( icqUin != "")
	{		
		foreach( icqAccount *account, icqList )
		{
			account->setTrayCurrent(false);
			account->setChooseStatusCheck(false);
			if ( account->getIcquin() == icqUin)
			{
				account->setTrayCurrent(true);
				account->setChooseStatusCheck(true);
				trayIcon->setIcon(account->getCurrentIcon());
				tempIcon = account->getCurrentIcon();
				accountInList = true;
			}
		}
	}
	else if (!icqList.empty())
	{
		icqList.at(0)->setTrayCurrent(true);
		trayIcon->setIcon(icqList.at(0)->getCurrentIcon());
		tempIcon = icqList.at(0)->getCurrentIcon();
	}
	
	if ( accountInList )
	{
		icqList.at(0)->setTrayCurrent(true);
		trayIcon->setIcon(icqList.at(0)->getCurrentIcon());	
		tempIcon = icqList.at(0)->getCurrentIcon();
	}
}

void qutIM::updateTrayIcon(QIcon *statusIcon)
{
	trayIcon->setIcon(*statusIcon);
	tempIcon = *statusIcon;
}

void qutIM::updateCurrentTrayStatusmenu(bool show)
{
	
	if ( show )
	{
		if ( !addToStatusTrayMenu)
		{
			currentStatusMenu = new QMenu(this);
			currentStatusMenu->setTitle(tr("Tray status from:"));
			currentStatusMenu->setIcon(QIcon(":/icons/crystal_project/status.png"));
			trayMenu->insertMenu(switchUserAction, currentStatusMenu);
			addToStatusTrayMenu = true;
			foreach( icqAccount *account, icqList)
			{
				account->createTrayMenuStatus(currentStatusMenu);
				connect(account, SIGNAL(changeStatusInTrayMenu(const QString &)),
										this, SLOT(setStatusFrom(const QString &)));
			}
		}
	} else {
		
		if ( addToStatusTrayMenu ) 
		{
			foreach( icqAccount *account, icqList)
			{
				account->removeTrayMenuStatus(currentStatusMenu);
				disconnect(account, SIGNAL(changeStatusInTrayMenu(const QString &)),
										this, SLOT(setStatusFrom(const QString &)));
			}
			delete currentStatusMenu;
			currentStatusMenu = NULL;
		}
		addToStatusTrayMenu = false;
	}
	
}

void qutIM::itemClicked(QTreeWidgetItem *item, int column)
{
	if ( item->type() == 1  && column == 0)
	{
		treeGroupItem *group = dynamic_cast<treeGroupItem *>(item);
			group->groupClicked();
	} else if ( item->type() == 0 )
	{
		if (item->childCount())
		{
				item->setExpanded(!item->isExpanded());
				item->sortChildren(1, Qt::AscendingOrder);
		}
	}
	
}

void qutIM::itemDoubleClicked(QTreeWidgetItem *item)
{
	if ( item->type() == 1)
	{
		treeGroupItem *group = dynamic_cast<treeGroupItem *>(item);
		group->groupClicked();
	}
	
	if ( item->type() == 2 )
	{
			treeBuddyItem *buddy = dynamic_cast<treeBuddyItem *>(item);
			buddy->par->doubleClickedBuddy(buddy);
	} //else if (item->type() == 3)
	//{
		
	//}
}

void qutIM::getNewMessage()
{
	if (QSystemTrayIcon::isSystemTrayAvailable())
	{
	if ( !unreadMessages )
	{
		unreadMessages = true;
		msgIcon = true;
		updateMessageIcon();
	}
	icqAccount *tmpAccount = qobject_cast<icqAccount *>(sender());
	if ( !messageStack.contains(tmpAccount) )
		messageStack.append(tmpAccount);
	}
}

void qutIM::updateMessageIcon()
{
	
	if ( unreadMessages )
	{
		if ( msgIcon )
			trayIcon->setIcon(QIcon(":/icons/crystal_project/message.png"));
		else 
			trayIcon->setIcon(tempIcon);
		msgIcon = !msgIcon;
		QTimer::singleShot(1000,this, SLOT(updateMessageIcon()));
	} else
		trayIcon->setIcon(tempIcon);
}

void qutIM::readMessages()
{
	unreadMessages = false;
	updateMessageIcon();
	foreach( icqAccount *tmpAccount, messageStack)
		tmpAccount->readMessageStack();
	messageStack.clear();
}

void qutIM::accountReadAllMessages()
{
	messageStack.removeAll((icqAccount *)sender());
	unreadMessages = false;
}

void qutIM::mergeAccountsToList(bool flag)
{
	mergeAccounts = flag;
	if ( flag )
	{
//		ui.contactListTree->clear();
		createGroups();	
		foreach( icqAccount *account, icqList )
		{
			getGroupList(account->getProtocol()->getContactListClass()->mergeAccounts(flag));
			
		}
		setHideSeparators(hideSeparators);
		updateSorting();
		
	} else {
		removeGroups();
	}
}

void qutIM::createGroups()
{
	if ( !showGroups )
	{
		onlineList = new contactSeparator(ui.contactListTree);
		onlineList->setText(1,tr("Online"));
		onlineList->setTextAlignment(1, Qt::AlignCenter);
		offlineList = new contactSeparator(ui.contactListTree);
		offlineList->setText(1,tr("Offline"));
		offlineList->setTextAlignment(1, Qt::AlignCenter);
		offlineList->setFont(1, QFont(offlineList->font(1).family(), 7));
		QLinearGradient linearGrad(0,0,0,16);
		linearGrad.setColorAt(0, QColor(255,255,255));
		linearGrad.setColorAt(0.5, QColor(248,248,248));
		linearGrad.setColorAt(1, QColor(255,255,255));
		QBrush brush(linearGrad);
		offlineList->setBackground(1, brush);
		offlineList->setBackground(0, brush);
		offlineList->setBackground(2, brush);
		onlineList->setBackground(0, brush);
		onlineList->setBackground(1, brush);
		onlineList->setBackground(2, brush);
		onlineList->setFont(1, QFont(offlineList->font(1).family(), 7));
		onlineList->setHidden(true);
		offlineList->setHidden(true);
	} 
	
}

void qutIM::removeGroups()
{
	if (!showGroups)
	{
		if ( offlineList )
		{
			offlineList->takeChildren();
			delete offlineList;
			
		}
		
		if ( onlineList )
		{
			onlineList->takeChildren();
			
			delete onlineList;
		}
	} else {
		foreach( treeGroupItem *group, mergeGroupList)
		{
			group->onlineList->takeChildren();
			group->offlineList->takeChildren();
			delete group;
		}
		mergeGroupList.clear();
	}
	foreach( icqAccount *account, icqList )
	{
		account->getProtocol()->getContactListClass()->mergeAccounts(false);
	}
}

void qutIM::getGroupList(QHash<quint16, treeGroupItem *> groupList)
{
	if ( !showGroups )
	{
		foreach(treeGroupItem *group, groupList)
		{
			
			offlineList->addChildren(group->offlineList->takeChildren());
			onlineList->addChildren(group->onlineList->takeChildren());
		}
		if ( onlineList->childCount() )
		{
			onlineList->setHidden(false);
			onlineList->setExpanded(true);
		}
		if ( offlineList->childCount() )
		{
			if ( !showOffline )
				offlineList->setHidden(false);	
				offlineList->setExpanded(true);
		}
		onlineList->sortChildren(1,Qt::AscendingOrder);
		if ( !showOffline )
			offlineList->sortChildren(1,Qt::AscendingOrder);
	} else {
		foreach(treeGroupItem *group, groupList)
		{
			if ( !mergeGroupList.contains(group->name) )
			{
				treeGroupItem *newGroup = new treeGroupItem(ui.contactListTree);
				newGroup->setGroupText(group->name);
				newGroup->setOnOffLists();
				newGroup->userCount = 1;
				newGroup->setExpanded(true);
//				newGroup->userCount = 0;
				newGroup->onlineList->setHidden(true);
				newGroup->offlineList->setHidden(true);
				mergeGroupList.insert(newGroup->name, newGroup);
			} else {
				mergeGroupList.value(group->name)->userCount += group->userCount;
			}
			
			treeGroupItem *newGroup2 = mergeGroupList.value(group->name);
			newGroup2->onlineList->addChildren(group->onlineList->takeChildren());
			newGroup2->offlineList->addChildren(group->offlineList->takeChildren());
			newGroup2->userCount = newGroup2->onlineList->childCount() + newGroup2->offlineList->childCount();
//			newGroup2->userOnline += group->userOnline;
			newGroup2->userOnline = newGroup2->onlineList->childCount();
			newGroup2->updateText();
			if (newGroup2->userCount > 0)
			{
				if ( newGroup2->onlineList->childCount() )
				{
					newGroup2->onlineList->setHidden(false);
					newGroup2->onlineList->setExpanded(true);
				}
				if ( newGroup2->offlineList->childCount() )
				{
					if ( !showOffline )
						newGroup2->offlineList->setHidden(false);	
						newGroup2->offlineList->setExpanded(true);
				}
//				newGroup2->setExpanded(true);
				newGroup2->setHidden(false);
			}
			newGroup2->onlineList->sortChildren(1,Qt::AscendingOrder);
			if ( !showOffline )
				newGroup2->offlineList->sortChildren(1,Qt::AscendingOrder);
		}
		hideEmptyGr(hideEmptyGroups);
			
	}
	updateSorting();
	
}

void qutIM::buddyChangeStatus(treeBuddyItem *buddy, bool isOffline)
{
	if ( !showGroups )
	{
		if ( isOffline )
		{
			onlineList->removeChild(buddy);
			offlineList->addChild(buddy);
			offlineList->sortChildren(1,Qt::AscendingOrder);
		} else {
			offlineList->removeChild(buddy);
			onlineList->addChild(buddy);
			
		}
		if ( !onlineList->childCount() )
				onlineList->setHidden(true);
		else
		{
			onlineList->setHidden(false);
			onlineList->setExpanded(true);
		}
		
		if ( !offlineList->childCount() )
				offlineList->setHidden(true);	
		else
		{
			if ( !showOffline )
				offlineList->setHidden(false);
			offlineList->setExpanded(true);
		}
	} else {
		treeGroupItem *group = mergeGroupList.value(buddy->groupName);
		if ( isOffline )
		{
			group->onlineList->removeChild(buddy);
			group->offlineList->addChild(buddy);
			group->buddyOffline();
		} else {
			group->offlineList->removeChild(buddy);
			group->onlineList->addChild(buddy);
			group->buddyOnline();
			
		}
		if ( !group->onlineList->childCount() )
			group->onlineList->setHidden(true);
		else
		{
			group->onlineList->setHidden(false);
			group->onlineList->setExpanded(true);
		}
		
		if ( !group->offlineList->childCount() )
			group->offlineList->setHidden(true);	
		else
		{
			if ( !showOffline )
				group->offlineList->setHidden(false);
			group->offlineList->setExpanded(true);
		}
		
		
	}
}

void qutIM::showHideGroups(bool flag)
{
	showGroups = flag;
	if ( flag )
	{
		if (offlineList)
		{
			offlineList->takeChildren();
			delete offlineList;
		}
		if (onlineList)
		{
				onlineList->takeChildren();
				delete onlineList;
		}

	} else {
	foreach( treeGroupItem *group, mergeGroupList)
	{
		group->onlineList->takeChildren();
		group->offlineList->takeChildren();
		delete group;
	}
		mergeGroupList.clear();
		createGroups();
	}
	
	foreach( icqAccount *account, icqList )
	{
		account->getProtocol()->getContactListClass()->mergeAccounts(false);
		account->getProtocol()->getContactListClass()->mergeAccounts(true);
		getGroupList(account->getProtocol()->getContactListClass()->groupList);
				
	}
	setHideSeparators(hideSeparators);
	updateSorting();
}

void qutIM::hideEmptyGr(bool flag)
{
		hideEmptyGroups = flag;
		foreach(treeGroupItem *group, mergeGroupList)
		{
			if ( !group->userCount ) 
			{
				group->setHidden(flag);
			}
				
		}
	
}

void qutIM::updateSorting()
{
	if ( mergeAccounts )
	{
		setUpdatesEnabled(false);
		if ( showGroups)
		{
			foreach( treeGroupItem *group, mergeGroupList)
			{
				group->onlineList->sortChildren(1,Qt::AscendingOrder);
				group->offlineList->sortChildren(1,Qt::AscendingOrder);
			}
		} else {
			onlineList->sortChildren(1,Qt::AscendingOrder);
			offlineList->sortChildren(1,Qt::AscendingOrder);
		}
		setUpdatesEnabled(true);
	} else {
	
	foreach(icqAccount *account, icqList)
	{
		account->getProtocol()->getContactListClass()->updateSorting();
	}
	}
}

void qutIM::hideOffline(bool flag)
{
	showOffline = flag;
	if ( showGroups )
	{
		foreach( treeGroupItem *group, mergeGroupList)
		{
			group->offlineList->sortChildren(1, Qt::AscendingOrder);
			if ( group->offlineList->childCount() )
			{
				group->offlineList->setHidden(flag);
			}
		}
	} else {
		
		offlineList->sortChildren(1, Qt::AscendingOrder);
		if ( offlineList->childCount() )
			offlineList->setHidden(flag);
	}
}


void qutIM::focusOutEvent ( QFocusEvent */*event*/ )
{
	if ( autoHide )
		timer->start(hideSec * 1000);
}

void qutIM::focusInEvent(QFocusEvent */*event*/)
{
	timer->stop();
}

void qutIM::reupdateList()
{
	removeGroups();
	createGroups();	
	foreach( icqAccount *account, icqList )
	{
		getGroupList(account->getProtocol()->getContactListClass()->mergeAccounts(true));
		
	}
	setHideSeparators(hideSeparators);
			updateSorting();
	
}

void qutIM::checkEventChanging()
{
	if ( eventObject->changed )
	{
		eventObject->changed = false;
		foreach(icqAccount *account, updateEventList)
			account->getProtocol()->updateAutoAway();
	} 
	QTimer::singleShot(60000, this, SLOT(checkEventChanging()));
}

void qutIM::addToEventList(bool add)
{
	icqAccount *account = (icqAccount*)sender();
	if ( add )
		updateEventList.append(account);
	else
		updateEventList.removeAll(account);
}

void qutIM::updateTrayToolTip()
{
	QString toolTip;
	foreach(icqAccount *account, icqList)
	{
		toolTip.append(tr("<img src='%1'>  %2<br>").arg(account->currentIconPath).arg(account->getIcquin()));
	}
	toolTip.chop(4);
	trayIcon->setToolTip(toolTip);
}

void qutIM::setHideSeparators(bool hide)
{
	ui.contactListTree->setUpdatesEnabled(false);
	if ( !showGroups )
	{
		onlineList->hideSeparator(hide);
		
		offlineList->hideSeparator(hide);
		onlineList->sortChildren(1, Qt::AscendingOrder);
		offlineList->sortChildren(1, Qt::AscendingOrder);
		
	} else {
		foreach(treeGroupItem *group, mergeGroupList)
		{
			
			group->hideSeparators(hide);	
		}
	}

	ui.contactListTree->setUpdatesEnabled(true);
}

void qutIM::createMainMenu()
{
	addFindUsers  = new QMenu(tr("&Add/find users"));
	addFindUsers->setIcon(QIcon(":/icons/crystal_project/search.png"));
	
	sendMultiple = new QMenu(tr("Send multiple"));
	sendMultiple->setIcon(QIcon(":/icons/crystal_project/multiple.png"));
	
	privacyList = new QMenu(tr("Privacy lists"));
	privacyList->setIcon(QIcon(":/icons/crystal_project/privacylist.png"));
		
	serviceMessages = new QMenu(tr("Service messages"));
	serviceMessages->setIcon(QIcon(":/icons/crystal_project/servicemessage.png"));
	
	changeMyDetails = new QMenu(tr("View/change my details"));
	changeMyDetails->setIcon(QIcon(":/icons/crystal_project/changedetails.png"));
		
	changeMyPassword = new QMenu(tr("Change my password"));
	changeMyPassword->setIcon(QIcon(":/icons/crystal_project/password.png"));
	
	mainMenu = new QMenu(this);
	mainMenu->addMenu(addFindUsers);
	mainMenu->addMenu(sendMultiple);
	mainMenu->addMenu(privacyList);
	mainMenu->addMenu(serviceMessages);
	mainMenu->addMenu(changeMyDetails);
	mainMenu->addMenu(changeMyPassword);
	mainMenu->addAction(settingsAction);
	mainMenu->addAction(switchUserAction);
	mainMenu->addSeparator();
	mainMenu->addAction(quitAction);
	ui.menuButton->setMenu(mainMenu);
}


void qutIM::on_contactListTree_customContextMenuRequested(const QPoint &point)
{
	QTreeWidgetItem *clickedItemForContext = 0;
		
	clickedItemForContext = ui.contactListTree->itemAt(point);
		
	if ( clickedItemForContext )
	{
			if ( clickedItemForContext->type() == 1 )
			{
				treeGroupItem *group = dynamic_cast<treeGroupItem *>(clickedItemForContext);
				
				if ( group->fromItem)
					group->par->showGroupMenu(group);
			}
			if ( clickedItemForContext->type() == 2)
			{
				treeBuddyItem *buddy = dynamic_cast<treeBuddyItem *>(clickedItemForContext);
				buddy->par->showBuddyMenu(buddy);
			}
	}
}

void qutIM::loadStyle(const QString &path)
{
	QFile file(path);
    if(file.open(QFile::ReadOnly))
    {
    	qApp->setStyleSheet("");
	    QString styleSheet = QLatin1String(file.readAll());
	    qApp->setStyleSheet(styleSheet);
	    file.close();
    }
}

void qutIM::on_infoButton_clicked()
{
	if ( !aboutWindowOpen )
	{
		aboutWindowOpen = true;
		infoWindow = new aboutInfo;
		connect(infoWindow, SIGNAL(destroyed ( QObject * )),
				this, SLOT(infoWindowDestroyed(QObject *)));
		infoWindow->show();
	}
}

void qutIM::infoWindowDestroyed(QObject *)
{
	aboutWindowOpen = false;

}
