% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/survival-survreg-tidiers.R
\name{augment.survreg}
\alias{augment.survreg}
\title{Augment data with information from a(n) survreg object}
\usage{
\method{augment}{survreg}(
  x,
  data = model.frame(x),
  newdata = NULL,
  type.predict = "response",
  type.residuals = "response",
  ...
)
}
\arguments{
\item{x}{An \code{survreg} object returned from \code{\link[survival:survreg]{survival::survreg()}}.}

\item{data}{A \link[base:data.frame]{base::data.frame} or \code{\link[tibble:tibble]{tibble::tibble()}} containing the original
data that was used to produce the object \code{x}. Defaults to
\code{stats::model.frame(x)} so that \code{augment(my_fit)} returns the augmented
original data. \strong{Do not} pass new data to the \code{data} argument.
Augment will report information such as influence and cooks distance for
data passed to the \code{data} argument. These measures are only defined for
the original training data.}

\item{newdata}{A \code{\link[base:data.frame]{base::data.frame()}} or \code{\link[tibble:tibble]{tibble::tibble()}} containing all
the original predictors used to create \code{x}. Defaults to \code{NULL}, indicating
that nothing has been passed to \code{newdata}. If \code{newdata} is specified,
the \code{data} argument will be ignored.}

\item{type.predict}{Character indicating type of prediction to use. Passed
to the \code{type} argument of the \code{\link[stats:predict]{stats::predict()}} generic. Allowed arguments
vary with model class, so be sure to read the \code{predict.my_class}
documentation.}

\item{type.residuals}{Character indicating type of residuals to use. Passed
to the \code{type} argument of \code{\link[stats:residuals]{stats::residuals()}} generic. Allowed arguments
vary with model class, so be sure to read the \code{residuals.my_class}
documentation.}

\item{...}{Additional arguments. Not used. Needed to match generic
signature only. \strong{Cautionary note:} Misspelled arguments will be
absorbed in \code{...}, where they will be ignored. If the misspelled
argument has a default value, the default value will be used.
For example, if you pass \code{conf.lvel = 0.9}, all computation will
proceed using \code{conf.level = 0.95}. Two exceptions here are:
\itemize{
\item \code{tidy()} methods will warn when supplied an \code{exponentiate} argument if
it will be ignored.
\item \code{augment()} methods will warn when supplied a \code{newdata} argument if it
will be ignored.
}}
}
\description{
Augment accepts a model object and a dataset and adds
information about each observation in the dataset. Most commonly, this
includes predicted values in the \code{.fitted} column, residuals in the
\code{.resid} column, and standard errors for the fitted values in a \code{.se.fit}
column. New columns always begin with a \code{.} prefix to avoid overwriting
columns in the original dataset.

Users may pass data to augment via either the \code{data} argument or the
\code{newdata} argument. If the user passes data to the \code{data} argument,
it \strong{must} be exactly the data that was used to fit the model
object. Pass datasets to \code{newdata} to augment data that was not used
during model fitting. This still requires that at least all predictor
variable columns used to fit the model are present. If the original outcome
variable used to fit the model is not included in \code{newdata}, then no
\code{.resid} column will be included in the output.

Augment will often behave differently depending on whether \code{data} or
\code{newdata} is given. This is because there is often information
associated with training observations (such as influences or related)
measures that is not meaningfully defined for new observations.

For convenience, many augment methods provide default \code{data} arguments,
so that \code{augment(fit)} will return the augmented training data. In these
cases, augment tries to reconstruct the original data based on the model
object with varying degrees of success.

The augmented dataset is always returned as a \link[tibble:tibble]{tibble::tibble} with the
\strong{same number of rows} as the passed dataset. This means that the passed
data must be coercible to a tibble. If a predictor enters the model as part
of a matrix of covariates, such as when the model formula uses
\code{\link[splines:ns]{splines::ns()}}, \code{\link[stats:poly]{stats::poly()}}, or \code{\link[survival:Surv]{survival::Surv()}}, it is represented
as a matrix column.

We are in the process of defining behaviors for models fit with various
\code{na.action} arguments, but make no guarantees about behavior when data is
missing at this time.
}
\examples{
\dontshow{if (rlang::is_installed(c("survival", "ggplot2"))) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}

# load libraries for models and data
library(survival)

# fit model
sr <- survreg(
  Surv(futime, fustat) ~ ecog.ps + rx,
  ovarian,
  dist = "exponential"
)

# summarize model fit with tidiers + visualization
tidy(sr)
augment(sr, ovarian)
glance(sr)

# coefficient plot
td <- tidy(sr, conf.int = TRUE)

library(ggplot2)

ggplot(td, aes(estimate, term)) +
  geom_point() +
  geom_errorbarh(aes(xmin = conf.low, xmax = conf.high), height = 0) +
  geom_vline(xintercept = 0)
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[=augment]{augment()}}, \code{\link[survival:survreg]{survival::survreg()}}

Other survreg tidiers: 
\code{\link{glance.survreg}()},
\code{\link{tidy.survreg}()}

Other survival tidiers: 
\code{\link{augment.coxph}()},
\code{\link{glance.aareg}()},
\code{\link{glance.cch}()},
\code{\link{glance.coxph}()},
\code{\link{glance.pyears}()},
\code{\link{glance.survdiff}()},
\code{\link{glance.survexp}()},
\code{\link{glance.survfit}()},
\code{\link{glance.survreg}()},
\code{\link{tidy.aareg}()},
\code{\link{tidy.cch}()},
\code{\link{tidy.coxph}()},
\code{\link{tidy.pyears}()},
\code{\link{tidy.survdiff}()},
\code{\link{tidy.survexp}()},
\code{\link{tidy.survfit}()},
\code{\link{tidy.survreg}()}
}
\concept{survival tidiers}
\concept{survreg tidiers}
\value{
A \code{\link[tibble:tibble]{tibble::tibble()}} with columns:
  \item{.fitted}{Fitted or predicted value.}
  \item{.resid}{The difference between observed and fitted values.}
  \item{.se.fit}{Standard errors of fitted values.}

}
