/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: m_message.c,v 1.90.2.3 2005/01/15 23:53:31 amcwilliam Exp $
 */

#include "config.h"
#include "struct.h"
#include "common.h"
#include "sys.h"
#include "numeric.h"
#include "msg.h"
#include "channel.h"
#include "h.h"
#include "memory.h"
#include "hook.h"
#include "modules.h"
#include "xmode.h"
#include <time.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>

Hook *h_chanmsg = NULL;
Hook *h_usermsg = NULL;
Hook *h_dccsend = NULL;

#define TARG_CHANNEL	0x01
#define TARG_CHANADMIN	0x02
#define TARG_CHANOP	0x04
#define TARG_HALFOP	0x08
#define TARG_VOICE	0x10
#define TARG_CLIENT	0x20

typedef struct msg_target {
	void *ptr;
	unsigned int flags;
} MessageTarget;

static MessageTarget targets[HARD_TARGET_LIMIT + 1];
static int tcount = 0;

Module MOD_HEADER(m_message) = {
	"m_message",
	"/PRIVMSG and /NOTICE commands",
	6, "$Revision: 1.90.2.3 $"
};

int MOD_LOAD(m_message)()
{
	memset(targets, 0, sizeof(targets));

	if ((h_chanmsg = register_hook(&MOD_HEADER(m_message), "h_chanmsg")) == NULL) {
		return MOD_FAILURE;
	}
	if ((h_usermsg = register_hook(&MOD_HEADER(m_message), "h_usermsg")) == NULL) {
		return MOD_FAILURE;
	}
	if ((h_dccsend = register_hook(&MOD_HEADER(m_message), "h_dccsend")) == NULL) {
		return MOD_FAILURE;
	}

	/* Directed messages must be added first! */
	if (register_command(&MOD_HEADER(m_message), &CMD_PRIVMSG_D, m_privmsg_direct) == NULL) {
		return MOD_FAILURE;
	}
	if (register_command(&MOD_HEADER(m_message), &CMD_NOTICE_D, m_notice_direct) == NULL) {
		return MOD_FAILURE;
	}
	if (register_command(&MOD_HEADER(m_message), &CMD_PRIVMSG, m_privmsg) == NULL) {
		return MOD_FAILURE;
	}
	if (register_command(&MOD_HEADER(m_message), &CMD_NOTICE, m_notice) == NULL) {
		return MOD_FAILURE;
	}

	MOD_SET_FLAG(&MOD_HEADER(m_message), MOD_FLAG_PERM);
	return MOD_SUCCESS;
}

int MOD_UNLOAD(m_message)()
{
	return MOD_SUCCESS;
}

enum {
	CTCP_NONE = 0,
	CTCP_YES,
	CTCP_ACTION,
	CTCP_DCC,
	CTCP_DCCSEND
};

static int check_for_ctcp(char *str, char **dcc_ptr)
{
	char *p = str;

	while ((p = strchr(p, 1)) != NULL) {
		p++; /* skip \001 */

		if (!myncmp(p, "DCC", 3)) {
			if (dcc_ptr != NULL) {
				*dcc_ptr = p;
			}
			if (!myncmp(p + 3, " SEND", 5)) {
				return CTCP_DCCSEND;
			}
			return CTCP_DCC;
		}

		if (!myncmp(p, "ACTION", 6)) {
			return CTCP_ACTION;
		}

		if (*(p + 1) != '\0') {
			continue;
		}

		return CTCP_YES;
	}

	return CTCP_NONE;
}

static void msg_channel_flags(int notice, aClient *cptr, aClient *sptr, int ti, char *text, int ismine)
{
	aChannel *chptr = (aChannel *)targets[ti].ptr;
	unsigned int f = 0;
	char c;

	ASSERT(chptr != NULL);

	/* Verify we are allowed to talk in the channel */

	if (ismine && !IsULine(sptr)) {
		int cant_send = can_send(sptr, chptr, text);

		if (cant_send) {
			if (!notice) {
				msg_error(sptr, cant_send, chptr->chname);
			}
			return;
		}
	}

	if (targets[ti].flags & TARG_VOICE) {
		f = (CMODE_VOICE|CMODE_HALFOP|CMODE_CHANOP|CMODE_CHANADMIN);
		c = '+';
	}
	else if (targets[ti].flags & TARG_HALFOP) {
		f = (CMODE_HALFOP|CMODE_CHANOP|CMODE_CHANADMIN);
		c = '%';
	}
	else if (targets[ti].flags & TARG_CHANOP) {
		f = (CMODE_CHANOP|CMODE_CHANADMIN);
		c = '@';
	}
	else {
		f = CMODE_CHANADMIN;
		c = '*';
	}

	sendto_channel_msg_butone(cptr, sptr, chptr, f, (notice) ? &CMD_NOTICE : &CMD_PRIVMSG,
		"%c%s :%s", c, chptr->chname, text);
}

static void msg_channel(int notice, aClient *cptr, aClient *sptr, int ti, char *text, int ismine)
{
	aChannel *chptr = (aChannel *)targets[ti].ptr;

	ASSERT(chptr != NULL);

	if (!notice && ismine) {
		switch (check_for_ctcp(text, NULL)) {
			case CTCP_NONE:
				break;
			case CTCP_DCCSEND:
			case CTCP_DCC:
				msg_error(sptr, CANT_SEND_DCC, chptr->chname);
				return;
				break;
			case CTCP_YES:
				if (chptr->mode.mode & CMODE_NOCTCP) {
					msg_error(sptr, CANT_SEND_CTCP, chptr->chname);
					return;
				}
				/* fall through */
			case CTCP_ACTION:
#ifdef FLUD
				if (!ServerInfo->hub && check_for_flud(sptr, NULL, chptr)) {
					return;
				}
#endif
				break;
			default:
				break;
		}
	}

	/* Verify we are allowed to talk in the channel */

	if (ismine && !IsULine(sptr)) {
		int cant_send = can_send(sptr, chptr, text);
		HookData chanmsg_data = HOOKDATA_INIT;

		if (cant_send) {
			if (!notice) {
				msg_error(sptr, cant_send, chptr->chname);
			}
			return;
		}

		/* Run channel message hooks until we receive FLUSH_BUFFER */

		chanmsg_data.sptr = sptr;
		chanmsg_data.chptr = chptr;
		chanmsg_data.i = notice;
		chanmsg_data.c = text;

		if (hook_run_until(h_chanmsg, &chanmsg_data, FLUSH_BUFFER) == FLUSH_BUFFER) {
			return;
		}
	}
	
	sendto_channel_msg_butone(cptr, sptr, chptr, ALL_MEMBERS,
		(notice) ? &CMD_NOTICE : &CMD_PRIVMSG, "%s :%s", chptr->chname, text);
}

static void msg_client(int notice, aClient *cptr, aClient *sptr, int ti, char *text, int ismine, int directed)
{
	aClient *acptr = (aClient *)targets[ti].ptr;

	ASSERT(acptr != NULL);

	if (!notice && MyConnect(acptr)) {
		char *dcc_send = NULL;

		switch (check_for_ctcp(text, &dcc_send)) {
			case CTCP_NONE: /* So we dont check flud */
				break;
			case CTCP_DCCSEND:
			{
				HookData dccsend_data = HOOKDATA_INIT;

				dccsend_data.cptr = cptr;
				dccsend_data.sptr = sptr;
				dccsend_data.acptr = acptr;
				dccsend_data.c = dcc_send;

				if (hook_run_until(h_dccsend, &dccsend_data, FLUSH_BUFFER) == FLUSH_BUFFER) {
					return;
				}

				/* fall through */
			}
			default:
#ifdef FLUD
				if (!ServerInfo->hub && check_for_flud(sptr, acptr, NULL)) {
					return;
				}
#endif
				break;
		}
	}

	if (!is_silenced(sptr, acptr)) {
		if (ismine) {
			HookData usermsg_data = HOOKDATA_INIT;

			usermsg_data.sptr = sptr;
			usermsg_data.acptr = acptr;
			usermsg_data.i = notice;
			usermsg_data.c = text;

			if (hook_run_until(h_usermsg, &usermsg_data, FLUSH_BUFFER) == FLUSH_BUFFER) {
				return;
			}
		}

		if (!notice && MyConnect(acptr) && IsPerson(acptr) && !BadPtr(acptr->user->away)) {
			send_me_numeric(sptr, RPL_AWAY, acptr->name, acptr->user->away);
		}

		/* If this is directed and the target is not ours, bounce it on */

		if (directed && !MyClient(acptr)) {
			sendto_one_client_onserver(acptr, sptr,
				(notice) ? &CMD_NOTICE_D : &CMD_PRIVMSG_D, ":%s", text);
		}
		else {
			sendto_one_client_prefixed(acptr, sptr,
				(notice) ? &CMD_NOTICE : &CMD_PRIVMSG, ":%s", text);
		}
	}
}

static void msg_directed(int notice, aClient *cptr, aClient *sptr, char *nick, char *text, int ismine)
{
	aClient *server = NULL, *target = NULL;
	char *s;

	if ((s = strchr(nick, '@')) != NULL) {
		server = find_server((s + 1), NULL);
	}
	if (server == NULL) {
		if (!notice) {
			target_left(sptr, nick, MSG_PRIVMSG, text);
		}
		return;
	}

	/* Is this message destined for me, or another server? */

	if (!IsMe(server)) {
		/* It's going somewhere else, and it's *from* a local client */
		if (ismine) {
			*s = '\0';
			target = find_person(nick, NULL);
			*s = '@'; /* Preserve nick@server.name */

			if ((target != NULL) && (target->uplink == server)) {
				sendto_one_client_onserver(target, sptr,
					(notice) ? &CMD_NOTICE_D : &CMD_PRIVMSG_D, ":%s", text);
				return;
			}
		}

		/* So it's from another remote client, destined for some other
		 * remote client. Just bounce it along...
		 */

		sendto_one_client_nopostfix(target, sptr, (notice) ? &CMD_NOTICE_D : &CMD_PRIVMSG_D,
			"%s :%s", nick, text);
	}
	else {
		/* It's destined for one of my clients, extract the nick */

		*s = '\0';
		target = find_person(nick, NULL);
		*s = '@'; /* Preserve nick@server.name */

		/* MyConnect(target) should be true (in theory) */

		if ((target != NULL) && MyConnect(target)) {
			sendto_one_client_prefixed(target, sptr,
				(notice) ? &CMD_NOTICE : &CMD_PRIVMSG, ":%s", text);
		}
		else if (!notice) {
			target_left(sptr, nick, MSG_PRIVMSG, text);
		}
	}
}

static void msg_special(int notice, aClient *cptr, aClient *sptr, char *nick, char *text, int ismine)
{
	char *s = NULL;
	
	if ((*(nick + 1) == '$') || (*(nick + 1) == '#')) {
		nick++;
	}
	else if (MyClient(sptr)) {
		send_me_notice(sptr, ":The command %s %s is no longer supported, please use $%s",
				(notice) ? MSG_NOTICE : MSG_PRIVMSG, nick, nick);
		return;
	}

	if ((s = strrchr(nick, '.')) == NULL) {
		send_me_numeric(sptr, ERR_NOTOPLEVEL, nick);
		return;
	}
	while (*++s != '\0') {
		if ((*s == '.') || IsWildChar(*s)) {
			break;
		}
	}
	if (IsWildChar(*s)) {
		send_me_numeric(sptr, ERR_WILDTOPLEVEL, nick);
		return;
	}
	
	sendto_match_msg_butone(IsServer(cptr) ? cptr : NULL, sptr, (nick + 1),
		(*nick == '#') ? MATCH_HOST : MATCH_SERVER,
		(notice) ? &CMD_NOTICE : &CMD_PRIVMSG, "$%s :%s", nick, text);
}

static inline int add_target(void *ptr, unsigned int flags)
{
	int i;

	for (i = 0; i < tcount; i++) {
		if (targets[i].ptr == ptr) {
			return 0; /* Duplicate target, bounce it */
		}
	}

	if (tcount < GeneralConfig.max_targets) {
		targets[tcount].ptr = ptr;
		targets[tcount++].flags = flags;

		return 1;
	}

	return -1;
}

static void build_target_list(int notice, aClient *cptr, aClient *sptr, char *target_list, char *text, int ismine)
{
	char *nick, *p = NULL, *pnick;
	aChannel *chptr;
	aClient *acptr;
	unsigned short prefix;

	/* We parse HARD_TARGET_LIMIT targets from the list, and implement a soft
	 * target limit in add_target(), checking the return value for error
	 */

	tcount = 0;
	for (nick = strtoken(&p, target_list, ",");
	  !BadPtr(nick) && (tcount < HARD_TARGET_LIMIT + 1); nick = strtoken(&p, NULL, ",")) {
		/* Channel messages are the most common */

		if (IsChanPrefix(*nick)) {
			if ((chptr = find_channel(nick, NULL)) != NULL) {
				if (add_target(chptr, TARG_CHANNEL) == -1) {
					send_me_numeric(sptr, ERR_TOOMANYTARGETS, nick, tcount);
					return;
				}
			}
			else if (!notice) {
				target_left(sptr, nick, MSG_PRIVMSG, text);
			}
			continue; /* If its a valid channel name, just continue */
		}

		/* Now check for a client message */

		acptr = (IsServer(cptr)) ? find_person_target(nick) : find_person(nick, NULL);
		if (acptr != NULL) {
			if (add_target(acptr, TARG_CLIENT) == -1) {
				send_me_numeric(sptr, ERR_TOOMANYTARGETS, nick, tcount);
				return;
			}
			continue;
		}

		/* Now see if we have a channel with flag prefixes */

		pnick = nick; /* Save original pointer */
		for (prefix = 0; *nick != '\0'; nick++) {
			if (!IsChanPrefix(*nick)) {
				break;
			}
			switch (*nick) {
				/* Each of these falls through */
				case '+':
					prefix |= TARG_VOICE;
				case '%':
					prefix |= TARG_HALFOP;
				case '@':
					prefix |= TARG_CHANOP;
				case '*':
					prefix |= TARG_CHANADMIN;
				default:
					break;
			}
		}

		if (prefix) {
			if (BadPtr(nick)) {
				if (!notice) {
					send_me_numeric(sptr, ERR_NORECIPIENT, MSG_PRIVMSG);
				}
				continue;
			}

			if ((chptr = find_channel(nick, NULL)) != NULL) {
				/* only +o and +h can send messages to @#channel etc. */
				if (ismine && !has_mode(sptr, chptr, CMODE_CHANOP|CMODE_HALFOP)) {
					if (!notice) {
						send_me_numeric(sptr, ERR_CHANOPRIVSNEEDED, chptr->chname);
					}
					continue;
				}
				if (add_target(chptr, TARG_CHANNEL|prefix) == -1) {
					send_me_numeric(sptr, ERR_TOOMANYTARGETS, nick, tcount);
					return;
				}
				continue;
			}
		}

		/* Check for a directed nick@server.name message */
		if (in_str(nick, '@')) {
			msg_directed(notice, cptr, sptr, nick, text, ismine);
			continue;
		}

		/* Check for an addressed $$server.name or $#host.mask */
		if ((*nick == '$') && (HasMode(sptr, UMODE_OPER) || IsULine(sptr))) {
			msg_special(notice, cptr, sptr, nick, text, ismine);
			continue;
		}

		if (!notice) {
			target_left(sptr, nick, MSG_PRIVMSG, text);
		}
	}
}

static int parse_message(int notice, int directed, aClient *cptr, aClient *sptr, int parc, char *parv[])
{
	int ismine = 0, ti;
	unsigned short type;

	if (parc < 3 || BadPtr(parv[2])) {
		if (!notice) {
			send_me_numeric(sptr, (parc < 2) ? ERR_NORECIPIENT : ERR_NOTEXTTOSEND, MSG_PRIVMSG);
		}
		return 0;
	}

	if (MyClient(sptr)) {
		/* Drop messages from squelched spambots */
		if (FloodConfig.spambot_squelch_time) {
			if ((sptr->localUser->join_part_count >= FloodConfig.max_join_part_count)
			  && mycmp(parv[0], parv[1])) {
				return 0;
			}
		}

		ismine = 1;
	}

	build_target_list(notice, cptr, sptr, parv[1], parv[2], ismine);

	for (ti = 0; ti < tcount; ti++) {
		type = (targets[ti].flags & (TARG_CLIENT|TARG_CHANNEL));
		ASSERT(type > 0);

		if (type == TARG_CHANNEL) {
			if (targets[ti].flags & (TARG_CHANADMIN|TARG_CHANOP|TARG_HALFOP|TARG_VOICE)) {
				msg_channel_flags(notice, cptr, sptr, ti, parv[2], ismine);
			}
			else {
				msg_channel(notice, cptr, sptr, ti, parv[2], ismine);
			}
		}
		else {
			msg_client(notice, cptr, sptr, ti, parv[2], ismine, directed);
		}
	}

	return 0;
}

/*
 * m_privmsg, m_notice
 *	parv[0] = sender prefix
 *	parv[1] = targets
 *	parv[2] = message
 */
int m_privmsg(aClient *cptr, aClient *sptr, int parc, char *parv[])
{
	return parse_message(0, 0, cptr, sptr, parc, parv);
}

int m_privmsg_direct(aClient *cptr, aClient *sptr, int parc, char *parv[])
{
	return parse_message(0, 1, cptr, sptr, parc, parv);
}

int m_notice(aClient *cptr, aClient *sptr, int parc, char *parv[])
{
	return parse_message(1, 0, cptr, sptr, parc, parv);
}

int m_notice_direct(aClient *cptr, aClient *sptr, int parc, char *parv[])
{
	return parse_message(1, 1, cptr, sptr, parc, parv);
}
