/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: m_rehash.c,v 1.42.2.2 2005/01/15 23:53:34 amcwilliam Exp $
 */

#include "config.h"
#include "struct.h"
#include "common.h"
#include "sys.h"
#include "numeric.h"
#include "msg.h"
#include "channel.h"
#include "h.h"
#include "memory.h"
#include "modules.h"
#include "res.h"
#include "blalloc.h"
#include "user_ban.h"
#include <time.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>

Module MOD_HEADER(m_rehash) = {
	"m_rehash",
	"/REHASH command",
	6, "$Revision: 1.42.2.2 $"
};

int MOD_LOAD(m_rehash)()
{
	if (register_command(&MOD_HEADER(m_rehash), &CMD_REHASH, m_rehash) == NULL) {
		return MOD_FAILURE;
	}
	return MOD_SUCCESS;
}

int MOD_UNLOAD(m_rehash)()
{
	return MOD_SUCCESS;
}

static void rehash_motd();
static void rehash_opermotd();
static void rehash_rules();
static void rehash_dns();
#ifdef USE_THROTTLE
static void rehash_throttle();
#endif
#ifdef USE_BLALLOC
static void rehash_blockheap();
#endif
static void rehash_akills();
static void rehash_tklines();
static void rehash_tsqlines();
static void rehash_tsglines();

typedef struct rehash_opt {
	char *opt;
	void (*handle)();
} RehashOption;

static RehashOption rehash_option_table[] = {
	{ "MOTD",	rehash_motd },
	{ "OPERMOTD",	rehash_opermotd },
	{ "RULES",	rehash_rules },
	{ "DNS",	rehash_dns },
#ifdef USE_THROTTLE
	{ "THROTTLE",	rehash_throttle },
#endif
#ifdef USE_BLALLOC
	{ "BLOCKHEAP",	rehash_blockheap },
#endif
	{ "AKILLS",	rehash_akills },
	{ "TKLINES",	rehash_tklines },
	{ "TSQLINES",	rehash_tsqlines },
	{ "TSGLINES",	rehash_tsglines },
	{ NULL,		NULL }
};

static void rehash_motd()
{
	if (motd != NULL) {
		destroy_message_file(motd);
	}
	motd = load_message_file(MOTD_FILE, MFTYPE_MOTD);
}

static void rehash_opermotd()
{
	if (opermotd != NULL) {
		destroy_message_file(opermotd);
	}
	opermotd = load_message_file(OPERMOTD_FILE, MFTYPE_OPERMOTD);
}

static void rehash_rules()
{
	if (rules != NULL) {
		destroy_message_file(rules);
	}
	rules = load_message_file(RULES_FILE, MFTYPE_RULES);
}

static void rehash_dns()
{
	restart_resolver();
}

#ifdef USE_THROTTLE
static void rehash_throttle()
{
	throttle_rehash();
}
#endif

#ifdef USE_BLALLOC
static void rehash_blockheap()
{
	blockheap_cleanup();
}
#endif

static void rehash_akills()
{
	remove_userbans_match_flags(BAN_NETWORK, 0);
}

static void rehash_tklines()
{
	remove_userbans_match_flags(BAN_LOCAL|BAN_TEMPORARY, 0);
}

static void rehash_tsqlines()
{
	remove_simbans_match_flags(SBAN_NICK|BAN_LOCAL, 0);
	remove_simbans_match_flags(SBAN_CHAN|BAN_LOCAL, 0);
}

static void rehash_tsglines()
{
	remove_simbans_match_flags(SBAN_GCOS|BAN_LOCAL, 0);
}

/*
 * m_rehash
 *	parv[0] = sender prefix
 *	parv[1] = options
 */
int m_rehash(aClient *cptr, aClient *sptr, int parc, char *parv[])
{
	RehashOption *ropt;

	if (!OPHasFlag(sptr, OFLAG_REHASH)) {
		send_me_numericNA(sptr, ERR_NOPRIVILEGES);
		return 0;
	}

	if (parc < 2 || BadPtr(parv[1])) {
		ircdlog(LOG_DEFAULT, "%s rehashing ircd.conf", get_client_name(sptr, FALSE));

		sendto_realops("%s has issued a config file reload", sptr->name);
		send_me_numeric(sptr, RPL_REHASHING, "ircd.conf");

		Internal.do_rehash = 1;
		return 0;
	}

	if ((*parv[1] == '?') && (*(parv[1] + 1) == '\0')) {
		send_me_noticeNA(sptr, ":*** Available rehash options:");

		for (ropt = rehash_option_table; (ropt->opt != NULL); ropt++) {
			send_me_notice(sptr, ":%s", ropt->opt);
		}
#ifndef STATIC_MODULES
		send_me_noticeNA(sptr, ":MODULES");
#endif

		return 0;
	}

#ifndef STATIC_MODULES
	if (!mycmp("MODULES", parv[1])) {
		ircdlog(LOG_DEFAULT, "%s reloading modules", get_client_name(sptr, FALSE));

		sendto_realops("%s has issued a modules reload", sptr->name);
		send_me_numeric(sptr, RPL_REHASHING, "MODULES");

		Internal.do_module_rehash = 1;
		return 0;
	}
#endif

	for (ropt = rehash_option_table; (ropt->opt != NULL); ropt++) {
		if (!mycmp(ropt->opt, parv[1])) {
			ircdlog(LOG_DEFAULT, "%s rehashing %s", get_client_name(sptr, FALSE), ropt->opt);

			sendto_realops("%s is rehashing %s", sptr->name, ropt->opt);
			send_me_numeric(sptr, RPL_REHASHING, ropt->opt);

			(*ropt->handle)();
			return 0;
		}
	}

	send_me_notice(sptr, ":Unknown rehash option %s - see /REHASH ?", parv[1]);
	return 0;
}
