/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: xmode.c,v 1.10.2.1 2004/12/07 03:05:42 pneumatus Exp $
 */

#include "struct.h"
#include "common.h"
#include "sys.h"
#include "numeric.h"
#include "channel.h"
#include "h.h"
#include "memory.h"
#include "xmode.h"

const char *xmode_errtab[] = {
	"Unknown/No error",
	"Mode table is full",
	"Mode flag already exists",
	"Mode flag not found"
};

const char *xmode_errstr(int error)
{
	ASSERT(error < 0);
	return xmode_errtab[-error];
}

static int find_table_slot(xModeTable *x)
{
	int i;

	ASSERT(x != NULL);

	for (i = 0; i <= XMODE_TABLE_SIZE; i++) {
		if (x->table[i].flag == '\0') {
			return i;
		}
	}

	return -1;
}

xModeTable *init_xmode_table()
{
	xModeTable *x;
	long modeval = 1;
	int i = 0;

	x = (xModeTable *)MyMalloc(sizeof(xModeTable));
	x->highest = -1;

	for (i = 0; i <= XMODE_TABLE_SIZE; i++) {
		x->table[i].flag = '\0';
		x->table[i].mode = modeval;
		modeval *= 2;
	}

	for (i = 0; i < XMODE_MAP_SIZE; i++) {
		x->map[i] = -1;
	}

	return x;
}

void destroy_xmode_table(xModeTable *x)
{
	ASSERT(x != NULL);
	MyFree(x);
}

int add_xmode(xModeTable *x, unsigned char flag, long *mode)
{
	int i;

	ASSERT(x != NULL);

	if (x->map[flag] >= 0) {
		return XMERR_FLAG_COLLISION;
	}

	if ((i = find_table_slot(x)) == -1) {
		return XMERR_TABLE_FULL;
	}

	if (i > x->highest) {
		x->highest = i;
	}
	x->table[i].flag = flag;
	x->map[flag] = i;

	if (mode != NULL) {
		*mode = x->table[i].mode;
	}

	return i;
}

static void clear_xmode(xModeTable *x, int mindex)
{
	ASSERT(x != NULL);
	ASSERT(mindex >= 0);
	ASSERT(x->map[x->table[mindex].flag] == mindex);

	x->map[x->table[mindex].flag] = -1;
	x->table[mindex].flag = '\0';

	while ((x->highest >= 0) && (x->table[x->highest].flag == '\0')) {
		x->highest--;
	}
}

int del_xmode(xModeTable *x, unsigned char flag)
{
	if (x->map[flag] == -1) {
		return XMERR_FLAG_NOTFOUND;
	}
	clear_xmode(x, x->map[flag]);
	return 0;
}
