/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: adns.c,v 1.27.2.1 2004/12/07 03:05:06 pneumatus Exp $
 */

#include "struct.h"
#include "common.h"
#include "sys.h"
#include "res.h"
#include "numeric.h"
#include "h.h"
#include "memory.h"
#include "fd.h"
#include "internal.h"
#include "adns.h"
#include "internal.h"
#include <stdlib.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/socket.h>

adns_state ircd_dns;

static void dns_io()
{
	struct adns_pollfd pollfds[MAXFD_POLL], *pfd;
	int pfd_cnt, i;

	adns__consistency(ircd_dns, 0, cc_entex);
	pfd_cnt = adns__pollfds(ircd_dns, pollfds);

	for (i = 0; i < pfd_cnt; i++) {
		pfd = &pollfds[i];
		if (pfd->events & ADNS_POLLIN) {
			engine_set_call(pfd->fd, FDEV_READ, dns_read, NULL, 0);
		}
		if (pfd->events & ADNS_POLLOUT) {
			engine_set_call(pfd->fd, FDEV_WRITE, dns_write, NULL, 0);
		}
	}
}

static void adns_timeout()
{
	adns_processtimeouts(ircd_dns, &Internal.ircd_time);
}

int init_resolver()
{
	int i;

	i = adns_init(&ircd_dns, adns_if_noautosys, 0);

	if (ircd_dns == NULL) {
		return 0;
	}

	add_event_approx("adns_timeout", adns_timeout, NULL, 2, 1);
	dns_io();

	return 1;
}

void restart_resolver()
{
	adns__rereadconfig(ircd_dns);
}

void dns_delete_queries(DNSQuery *q)
{
	if (q != NULL && (q->query != NULL)) {
		adns_cancel(q->query);
		q->callback = NULL;
	}
	dns_io();
}

static void dns_process_callbacks()
{
	adns_query query, r;
	adns_answer *answer;
	DNSQuery *q;
	int i, fail = 0;
	void *xq = &q, *xr = &r;

	adns_forallqueries_begin(ircd_dns);

	while ((query = adns_forallqueries_next(ircd_dns, xr)) != NULL) {
		i = adns_check(ircd_dns, &query, &answer, xq);

		if (i == EAGAIN) {
			continue;
		}

		ASSERT(q->callback != NULL);
		q->query = NULL;

		if (!i) {
			q->callback(q->ptr, answer);
			continue;
		}

		q->callback(q->ptr, NULL);

		if ((answer != NULL) && (answer->status == adns_s_systemfail)) {
			fail = 1;
		}
	}

	if (fail) {
		sendto_realops_lev(DEBUG_LEV, "adns got global system failure, restarting resolver...");
		init_resolver();
	}
}

void dns_read(int fd, void *unused, int engine_status_unused)
{
	adns_processreadable(ircd_dns, fd, &Internal.ircd_time);
	dns_process_callbacks();
	dns_io();
}

void dns_write(int fd, void *unused, int engine_status_unused)
{
	adns_processwriteable(ircd_dns, fd, &Internal.ircd_time);
	dns_process_callbacks();
	dns_io();
}

int dns_gethost(const char *name, DNSQuery *q)
{
	int i;

	ASSERT(ircd_dns->nservers > 0);

	i = adns_submit(ircd_dns, name, adns_r_addr, adns_qf_owner, q, &q->query);
	dns_io();

	return i;
}

int dns_getaddr(struct in_addr *ip, DNSQuery *q)
{
	struct sockaddr_in addr;
	int i;

	ASSERT(ircd_dns->nservers > 0);

	memset(&addr, '\0', sizeof(struct sockaddr_in));
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = ip->s_addr;
	addr.sin_port = 0;

	i = adns_submit_reverse(ircd_dns, (struct sockaddr *)&addr, adns_r_ptr,
		adns_qf_owner|adns_qf_cname_loose|adns_qf_quoteok_anshost, q, &q->query);
	dns_io();

	return i;
}

void report_dns_nameservers(aClient *sptr)
{
	int i;

	for (i = 0; i < ircd_dns->nservers; i++) {
		send_me_debug(sptr, "a :%s", inetntoa((const char *)&ircd_dns->servers[i].addr));
	}
}
