/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: user_mode.c,v 1.31.2.3 2005/05/02 01:16:14 amcwilliam Exp $
 */

#include "struct.h"
#include "patchlevel.h"
#include "common.h"
#include "sys.h"
#include "numeric.h"
#include "msg.h"
#include "h.h"
#include "memory.h"
#include "xmode.h"
#include <sys/stat.h>
#include <fcntl.h>

static char buf[BUFSIZE];

xModeTable *usermodes = NULL;
char usermode_str[XMODE_TABLE_SIZE + 1];

XMODE(UMODE_OPER);
XMODE(UMODE_INVISIBLE);
XMODE(UMODE_SNOTICE);
XMODE(UMODE_KNOTICE);
XMODE(UMODE_FNOTICE);
XMODE(UMODE_DNOTICE);
XMODE(UMODE_SPAMBOT);
XMODE(UMODE_DCCNOTICE);
XMODE(UMODE_LCLICONN);
XMODE(UMODE_GCLICONN);
XMODE(UMODE_WALLOPS);
XMODE(UMODE_GLOBOPS);
XMODE(UMODE_CHATOPS);
XMODE(UMODE_REGNICK);
XMODE(UMODE_SPYING);
XMODE(UMODE_SADMIN);
XMODE(UMODE_NETADMIN);
XMODE(UMODE_ADMIN);
XMODE(UMODE_MASKED);
XMODE(UMODE_PRIVACY);
XMODE(UMODE_DEAF);
XMODE(UMODE_RSTAFF);
XMODE(UMODE_REJNOTICES);
XMODE(UMODE_NORECVQTHROTTLE);
XMODE(UMODE_SECURE);

long ALL_UMODES;	/* All user modes */
long SEND_UMODES;	/* Global user modes */
long NEEDOPER_UMODES;	/* Oper-only user modes */
long AUTOOPER_UMODES;	/* Auto-oper user modes */
long SERVONLY_UMODES;	/* Modes only servers can set */
long ONCONNECT_UMODES;	/* Modes set on local-user connect */

static void update_usermode_str()
{
	char *p = usermode_str;
	int i;

	for (i = 0; i <= usermodes->highest; i++) {
		if (usermodes->table[i].flag != '\0') {
			*p++ = usermodes->table[i].flag;
		}
	}

	*p = '\0';
}

xMode *add_user_mode(char flag, long *mode, unsigned int options)
{
	int i;
	xMode *t;

	if ((i = add_xmode(usermodes, (unsigned char)flag, mode)) < 0) {
		ircdlog(LOG_ERROR, "Failed to create user mode '%c' (%s)", flag, xmode_errstr(i));
		sendto_realops("Failed to create user mode '%c' (%s)", flag, xmode_errstr(i));
		return NULL;
	}

	t = &usermodes->table[i];

	ALL_UMODES |= t->mode;
	if (options & UOPT_GLOBAL) {
		SEND_UMODES |= t->mode;
	}
	if (options & UOPT_NEEDOPER) {
		NEEDOPER_UMODES |= t->mode;
	}
	if (options & UOPT_AUTOOPER) {
		AUTOOPER_UMODES |= t->mode;
	}
	if (options & UOPT_SERVONLY) {
		SERVONLY_UMODES |= t->mode;
	}
	if (options & UOPT_ONCONNECT) {
		ONCONNECT_UMODES |= t->mode;
	}

	update_usermode_str();

	return t;
}

int del_user_mode(char flag, long *mode)
{
	int error;
	aClient *cptr;
	long old, umode;

	ASSERT(mode != NULL);
	umode = *mode;

	for (cptr = client; cptr != NULL; cptr = cptr->next) {
		if (!IsPerson(cptr)) {
			continue;
		}

		old = cptr->umode;
		cptr->umode &= ~umode;

		if (MyClient(cptr)) {
			send_umode_out(cptr, cptr, old);
		}
	}

	if ((error = del_xmode(usermodes, (unsigned char)flag))) {
		ircdlog(LOG_ERROR, "Unable to remove user mode '%c' (%s)", flag, xmode_errstr(error));
		sendto_realops("Unable to remove user mode '%c' (%s)", flag, xmode_errstr(error));
		return 0;
	}

	ALL_UMODES &= ~umode;
	if (umode & SEND_UMODES) {
		SEND_UMODES &= ~umode;
	}
	if (umode & NEEDOPER_UMODES) {
		NEEDOPER_UMODES &= ~umode;
	}
	if (umode & AUTOOPER_UMODES) {
		AUTOOPER_UMODES &= ~umode;
	}
	if (umode & SERVONLY_UMODES) {
		SERVONLY_UMODES &= ~umode;
	}
	if (umode & ONCONNECT_UMODES) {
		ONCONNECT_UMODES &= ~umode;
	}

	update_usermode_str();
	return 1;
}

void init_user_mode()
{
	usermodes = init_xmode_table();

	add_user_mode('o', &UMODE_OPER, UOPT_GLOBAL|UOPT_AUTOOPER);
	add_user_mode('i', &UMODE_INVISIBLE, UOPT_GLOBAL);
	add_user_mode('w', &UMODE_WALLOPS, UOPT_GLOBAL|UOPT_NEEDOPER|UOPT_AUTOOPER);
	add_user_mode('s', &UMODE_SNOTICE, UOPT_AUTOOPER);
	add_user_mode('c', &UMODE_LCLICONN, UOPT_NEEDOPER|UOPT_AUTOOPER);
	add_user_mode('r', &UMODE_REGNICK, UOPT_GLOBAL|UOPT_SERVONLY);
	add_user_mode('R', &UMODE_RSTAFF, UOPT_NEEDOPER);
	add_user_mode('k', &UMODE_KNOTICE, UOPT_NEEDOPER|UOPT_AUTOOPER);
	add_user_mode('y', &UMODE_SPYING, UOPT_NEEDOPER);
	add_user_mode('d', &UMODE_DEAF, UOPT_GLOBAL|UOPT_SERVONLY);
	add_user_mode('e', &UMODE_DCCNOTICE, UOPT_NEEDOPER|UOPT_AUTOOPER);
	add_user_mode('g', &UMODE_GLOBOPS, UOPT_GLOBAL|UOPT_NEEDOPER|UOPT_AUTOOPER);
	add_user_mode('b', &UMODE_CHATOPS, UOPT_GLOBAL|UOPT_NEEDOPER);
	add_user_mode('a', &UMODE_SADMIN, UOPT_GLOBAL|UOPT_NEEDOPER);
	add_user_mode('A', &UMODE_ADMIN, UOPT_GLOBAL|UOPT_NEEDOPER);
	add_user_mode('f', &UMODE_FNOTICE, UOPT_NEEDOPER|UOPT_AUTOOPER);
	add_user_mode('D', &UMODE_DNOTICE, UOPT_NEEDOPER);
	add_user_mode('m', &UMODE_SPAMBOT, UOPT_NEEDOPER|UOPT_AUTOOPER);
	add_user_mode('x', &UMODE_MASKED, UOPT_GLOBAL);
	add_user_mode('N', &UMODE_NETADMIN, UOPT_GLOBAL|UOPT_NEEDOPER);
	add_user_mode('G', &UMODE_GCLICONN, UOPT_LOCAL|UOPT_NEEDOPER);
	add_user_mode('p', &UMODE_PRIVACY, UOPT_GLOBAL);
	add_user_mode('j', &UMODE_REJNOTICES, UOPT_NEEDOPER);
	add_user_mode('F', &UMODE_NORECVQTHROTTLE, UOPT_LOCAL|UOPT_NEEDOPER);
	add_user_mode('z', &UMODE_SECURE, UOPT_GLOBAL|UOPT_SERVONLY);
}

char *get_user_modes(aClient *cptr)
{
	static char flagbuf[XMODE_TABLE_SIZE + 1];
	char *p = flagbuf;
	int i;

	flagbuf[0] = '\0';

	*p++ = '+';
	for (i = 0; i <= usermodes->highest; i++) {
		if (cptr->umode & usermodes->table[i].mode) {
			*p++ = usermodes->table[i].flag;
		}
	}
	*p = '\0';

	return flagbuf;
}

/*
 * do_user_mode
 *	parv[0] = sender prefix
 *	parv[1] = client to change mode for
 *	parv[2] = modes to change
 */
int do_user_mode(aClient *cptr, aClient *sptr, int parc, char *parv[])
{
	aClient *acptr;
	xMode *t;
	short change = XMODE_ADD;
	unsigned long old_umodes;
	int i, badflag = 0;
	char *p;

	if (parc < 2 || *parv[1] == '\0') {
		send_me_numeric(sptr, ERR_NEEDMOREPARAMS, "MODE");
		return 0;
	}

	acptr = MyClient(sptr) ? find_person(parv[1], NULL) : find_person_target(parv[1]);
	if (acptr == NULL) {
		if (MyConnect(sptr)) {
			send_me_numeric(sptr, ERR_NOSUCHNICK, parv[1]);
		}
		return 0;
	}
	if (IsServer(sptr) || sptr != acptr || acptr->from != sptr->from) {
		if (!IsServer(cptr)) {
			send_me_numericNA(sptr, ERR_USERSDONTMATCH);
		}
		return 0;
	}
	if (parc < 3) {
		send_me_numeric(sptr, RPL_UMODEIS, get_user_modes(sptr));
		return 0;
	}

	old_umodes = sptr->umode;

	for (p = parv[2]; p != NULL && *p != '\0'; p++) {
		if (*p == '+') {
			change = XMODE_ADD;
			continue;
		}
		if (*p == '-') {
			change = XMODE_DEL;
			continue;
		}

		if (IsSpace(*p) || IsEol(*p)) {
			continue;
		}

		if ((i = usermodes->map[(unsigned char)*p]) == -1) {
			badflag = 1;
			continue;
		}

		t = &usermodes->table[i];

		if (!IsServer(cptr) && !IsULine(sptr) && (SERVONLY_UMODES & t->mode)) {
			continue;
		}

		if (change == XMODE_ADD) {
			sptr->umode |= t->mode;
		}
		else {
			sptr->umode &= ~t->mode;
		}
	}
	if (badflag && MyConnect(sptr)) {
		send_me_numericNA(sptr, ERR_UMODEUNKNOWNFLAG);
	}

	if (!(old_umodes & UMODE_OPER) && HasMode(sptr, UMODE_OPER) && !IsServer(cptr)) {
		DelMode(sptr, UMODE_OPER);
	}
	if (!(old_umodes & UMODE_OPER) && HasMode(sptr, UMODE_OPER)) {
		Count.oper++;
	}
	if ((old_umodes & UMODE_OPER) && !HasMode(sptr, UMODE_OPER)) {
		Count.oper--;
		if (MyConnect(sptr)) {
			dlink_del(&oper_list, sptr, NULL);

			detach_oper(sptr);
			attach_class(sptr);

			sptr->localUser->oflags = 0;
		}
	}
	if (!(old_umodes & UMODE_INVISIBLE) && HasMode(sptr, UMODE_INVISIBLE)) {
		Count.invisi++;
	}
	if ((old_umodes & UMODE_INVISIBLE) && !HasMode(sptr, UMODE_INVISIBLE)) {
		Count.invisi--;
	}
	if ((old_umodes & UMODE_MASKED) && !HasMode(sptr, UMODE_MASKED)) {
		if (HasVhost(sptr)) {
			ClearVhost(sptr);
		}
	}
	if (!HasMode(sptr, UMODE_OPER) && !IsServer(cptr)) {
		DelMode(sptr, NEEDOPER_UMODES);
	}
	if (MyClient(sptr) && HasMode(sptr, UMODE_OPER)) {
		if (HasMode(sptr, UMODE_SADMIN) && !OPHasFlag(sptr, OFLAG_SADMIN)) {
			DelMode(sptr, UMODE_SADMIN);
		}
		if (HasMode(sptr, UMODE_NETADMIN) && (!OPHasFlag(sptr, OFLAG_NETADMIN)
		  || !GeneralConfig.enable_netadmins)) {
			DelMode(sptr, UMODE_NETADMIN);
		}
		if (HasMode(sptr, UMODE_ADMIN) && !OPHasFlag(sptr, OFLAG_ADMIN)) {
			DelMode(sptr, UMODE_ADMIN);
		}
		if (HasMode(sptr, UMODE_FNOTICE) && !OPHasFlag(sptr, OFLAG_FNOTICE)) {
			DelMode(sptr, UMODE_FNOTICE);
		}
		if (HasMode(sptr, UMODE_RSTAFF) && !OPHasFlag(sptr, OFLAG_RSTAFF)) {
			DelMode(sptr, UMODE_RSTAFF);
		}
		if (HasMode(sptr, UMODE_LCLICONN) && !OPHasFlag(sptr, OFLAG_LCLICONN)) {
			DelMode(sptr, UMODE_LCLICONN);
		}
		if (HasMode(sptr, UMODE_GCLICONN) && !OPHasFlag(sptr, OFLAG_GCLICONN)) {
			DelMode(sptr, UMODE_GCLICONN);
		}
		if (HasMode(sptr, UMODE_NORECVQTHROTTLE) && !OPHasFlag(sptr, OFLAG_NORECVQTHROTTLE)) {
			DelMode(sptr, UMODE_NORECVQTHROTTLE);
		}
	}

	if (!HasVhost(sptr)) {
		strncpyzt(sptr->user->maskedhost, maskme(sptr), HOSTLEN + 1);
	}

	send_umode_out(cptr, sptr, old_umodes);
	return 0;
}

void send_umode(aClient *cptr, aClient *sptr, int old, int sendmask, char *umode_buf)
{
	short change = XMODE_NONE;
	char *p = umode_buf;
	xMode *t;
	int i;

	*p = '\0';
	for (i = 0; i <= usermodes->highest; i++) {
		t = &usermodes->table[i];
		if (t->flag == '\0' || (MyClient(sptr) && !(t->mode & sendmask))) {
			continue;
		}

		if ((t->mode & old) && !(sptr->umode & t->mode)) {
			if (change != XMODE_DEL) {
				change = XMODE_DEL;
				*p++ = '-';
			}
			*p++ = t->flag;
		}
		else if (!(t->mode & old) && (sptr->umode & t->mode)) {
			if (change != XMODE_ADD) {
				change = XMODE_ADD;
				*p++ = '+';
			}
			*p++ = t->flag;
		}
	}
	*p = '\0';

	if (*umode_buf != '\0' && (cptr != NULL)) {
		sendto_one_client_real(cptr, sptr, sptr, &CMD_MODE, ":%s", umode_buf);
	}
}

void send_umode_out(aClient *cptr, aClient *sptr, int old)
{
	send_umode(NULL, sptr, old, SEND_UMODES, buf);

	if (*buf != '\0') {
		aClient *acptr;
		dlink_node *node;

		DLINK_FOREACH_DATA(lserver_list.head, node, acptr, aClient) {
			if (acptr != cptr && (acptr != sptr)) {
				sendto_one_client_real(acptr, sptr, sptr, &CMD_MODE, ":%s", buf);
			}
		}
	}

	if (cptr != NULL && MyClient(cptr)) {
		send_umode(cptr, sptr, old, ALL_UMODES, buf);
	}
}
