/*
 *  Copyright 2001-2004 Adrian Thurston <adriant@ragel.ca>
 */

/*  This file is part of Ragel.
 *
 *  Ragel is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Ragel is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Ragel; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#include "ragel.h"
#include "ftabcodegen.h"
#include "redfsm.h"
#include "parsetree.h"

/* Integer array line length. */
#define IALL 8

/* Transition array line length. */
#define TALL 4

/* Init base data. */
FTabCodeGen::FTabCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out )
:
	TabCodeGen(fsmName, parseData, redFsm, out)
{ }

/* Write out the out-func for a state. */
std::ostream &FTabCodeGen::STATE_OUT_ACTION( RedStateAp *state )
{
	/* This function is only called if there are any out actions, so need
	 * not guard against there being none. */
	out << state->eofAction->id+1;
	return out;
}

/* Write out the function for a transition. */
std::ostream &FTabCodeGen::TRANS_ACTION( RedTransAp *trans )
{
	if ( trans->action != 0 )
		out << trans->action->id+1;
	else
		out << "0";
	return out;
}

/* Write out the function switch. This switch is keyed on the values
 * of the func index. */
std::ostream &FTabCodeGen::ACTION_SWITCH()
{
	/* Loop the actions. */
	for ( ActionTableMap::Iter act = redFsm->actionMap; act.lte(); act++ ) {
		/* Write the entry label. */
		out << "\tcase " << act->id+1 << ":\n";

		/* Loop the items in the list of action items. */
		for ( ActionTable::Iter item = act->key; item.lte(); item++ ) {
			/* Get the action data. */
			Action *action = parseData->actionIndex[item->value];

			/* Write the preprocessor line directive for going into the source
			 * file. */
			out << "#line " << action->loc.line << " \""; LDIR_PATH(inputFile) << "\"\n";
			
			/* Wrap the block in brakets. */
			out << "\t{"; ACTION(action, 0, false) << "}\n";
		}

		out << "\tbreak;\n";
	}

	/* Write the directive for going back into the output file. The line
	 * number is for the next line, so add one. */
	out << "#line " << outFilter->line + 1 << " \""; LDIR_PATH(outputFile) << "\"\n";
	return out;
}

std::ostream &FTabCodeGen::TRANSITIONS()
{
	/* Transitions must be written ordered by their id. */
	RedTransAp **transPtrs = new RedTransAp*[redFsm->transSet.length()];
	for ( TransApSet::Iter trans = redFsm->transSet; trans.lte(); trans++ )
		transPtrs[trans->id] = trans;

	/* Keep a count of the num of items in the array written. */
	out << '\t';
	int totalTransData = 0;
	for ( int t = 0; t < redFsm->transSet.length(); t++ ) {
		/* Write out the target state. */
		RedTransAp *trans = transPtrs[t];
		out << trans->targ->position << ", ";
		if ( ++totalTransData % IALL == 0 )
			out << "\n\t";

		/* Write the function for the transition. */
		TRANS_ACTION( trans ) << ", ";
		if ( ++totalTransData % IALL == 0 )
			out << "\n\t";
	}

	out << "0" << "\n";
	delete[] transPtrs;
	return out;
}

std::ostream &FTabCodeGen::LOCATE_TRANS()
{
	out <<
		"	_specs = *_pcs++;\n"
		"	_keys = "; FSM_NAME() << "_k + *_pcs++;\n"
		"	_inds = "; FSM_NAME() << "_i + *_pcs++;\n"
		"\n"
		"	if ( _specs & SPEC_ANY_SINGLE ) {\n"
		"		int indsLen = *_pcs++;\n"
		"		"; ALPH_TYPE() << " *match = "; FSM_NAME() << 
					"_bsearch( "; GET_KEY() << ", _keys, indsLen );\n"
		"		if ( match != 0 ) {\n"
		"			_trans = "; FSM_NAME() << "_t + (_inds[match - _keys]<<1);\n"
		"			goto match;\n"
		"		}\n"
		"		_keys += indsLen;\n"
		"		_inds += indsLen;\n"
		"	}\n"
		"\n"
		"	if ( _specs & SPEC_ANY_RANGE ) {\n"
		"		int indsLen = *_pcs++;\n"
		"		"; ALPH_TYPE() << " *match = "; FSM_NAME() << 
					"_range_bsearch( "; GET_KEY() << ", _keys, (indsLen<<1) );\n"
		"		if ( match != 0 ) {\n"
		"			_trans = "; FSM_NAME() << "_t + (_inds[(match - _keys)>>1]<<1);\n"
		"			goto match;\n"
		"		}\n"
		"		_keys += (indsLen<<1);\n"
		"		_inds += indsLen;\n"
		"	}\n"
		"\n"
		"	_trans = "; FSM_NAME() << "_t + ((*_inds)<<1);\n";

	return out;
}



/* Init base class. */
CFTabCodeGen::CFTabCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out )
:
	FTabCodeGen(fsmName, parseData, redFsm, out)
{ }

std::ostream &CFTabCodeGen::CALL( NameInst *name, int targState, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "{fsm->_st[fsm->_top++] = _cs; _cs = " << 
			targ->value->position << "; goto again;}";
	return out;
}

std::ostream &CFTabCodeGen::RET( bool inFinish )
{
	out << "{_cs = fsm->_st[--fsm->_top]; goto again;}";
	return out;
}

std::ostream &CFTabCodeGen::CALLE( char *stateExpr, int targState, bool inFinish )
{
	out << "{fsm->_st[fsm->_top++] = _cs; _cs = (" << stateExpr << "); goto again;}";
	return out;
}

void CFTabCodeGen::writeOutHeader()
{
	out <<
		"/* Only non-static data: current state. */\n"
		"struct "; FSM_NAME() << "\n"
		"{\n"
		"	int _cs;\n";
		STRUCT_DATA() <<
		"};\n"
		"\n"
		"/* Init the fsm. */\n"
		"int "; FSM_NAME() << "_init( struct "; FSM_NAME() << " *fsm );\n"
		"\n"
		"/* Execute some chunk of data. */\n"
		"int "; FSM_NAME() << "_execute( struct "; FSM_NAME() << " *fsm, "; EL_TYPE() << 
				" *data, int len );\n"
		"\n"
		"/* Indicate to the fsm tha there is no more data. */\n"
		"int "; FSM_NAME() << "_finish( struct "; FSM_NAME() << " *fsm );\n"
		"\n";
}

void CFTabCodeGen::writeOutCode()
{
	/* Need the state positions. */
	calcStatePositions();
	
	/* State machine data. */
	out << 
		"#define SPEC_ANY_SINGLE  0x01\n"
		"#define SPEC_ANY_RANGE   0x02\n"
		"#define SPEC_EOF_ACTION  0x04\n"
		"\n";

	/* Write the array of keys. */
	out <<
		"static "; ALPH_TYPE() << " "; FSM_NAME() << "_k[] = {\n";
		KEYS() <<
		"};\n"
		"\n";

	/* Write the array of indicies. */
	out << 
		"static "; INDEX_TYPE() << " "; FSM_NAME() << "_i[] = {\n";
		INDICIES() <<
		"};\n"
		"\n";

	/* Write the array of states. */
	out <<
		"static int "; FSM_NAME() << "_s[] = {\n";
		STATES() <<
		"};\n"
		"\n"
		"static int "; FSM_NAME() << "_t[] = {\n";
		TRANSITIONS() <<
		"};\n"
		"\n"
		"static int "; FSM_NAME() << "_start = " << 
				redFsm->startState->position << ";\n"
		"\n";
	
	/* Init routine. */
	out << 
		"int "; FSM_NAME() << "_init( struct "; FSM_NAME() << " *fsm )\n"
		"{\n"
		"	fsm->_cs = "; FSM_NAME() << "_start;\n";

	/* If there are any calls, then the stack top needs initialization. */
	if ( anyActionCalls() || anyActionRets() )
		out << "	fsm->_top = 0;\n";

	INIT_CODE() <<
		"	if ( fsm->_cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	/* The binary search. */
	BSEARCH() << "\n";

	/* The binary search for a range. */
	RANGE_BSEARCH() << "\n";

	/* Execution function. */
	out << 
		"int "; FSM_NAME() << "_execute( struct "; FSM_NAME() << " *fsm, "; EL_TYPE() <<
				" *_data, int _len )\n"
		"{\n"
		"	"; EL_TYPE() << " *_p = _data-1;\n"
		"	"; EL_TYPE() << " *_pe = _data+_len;\n"
		"	int _cs = fsm->_cs, *_pcs";

	if ( anyRegCurStateRef() )
		out << ", _ps";
	out << ";\n\tint _specs";

	if ( anyActions() ) 
		out << ", _acts";

	out <<
		";\n"
		"	"; ALPH_TYPE() << " *_keys;\n"
		"	"; INDEX_TYPE() << " *_inds;\n"
		"	int *_trans;\n"
		"\n";

	out <<
		"again:\n"
		"	if ( ++_p == _pe )\n"
		"		goto out;\n"
		"	_pcs = "; FSM_NAME() << "_s + _cs;\n";

	if ( anyEofActions() ) {
		out <<
			"	if ( _p == 0 )\n"
			"		goto eofActions;\n"
			"\n";
	}

	LOCATE_TRANS() <<
		"\n"
		"match:\n";

	if ( anyRegCurStateRef() )
		out << "	_ps = _cs;\n";

	out <<
		"	_cs = *_trans++;\n"
		"\n";

	if ( anyActions() ) {
		out << 
			"	if ( *_trans == 0 )\n"
			"		goto again;\n"
			"	_acts = *_trans;\n"
			"\n";

		/* Only used if there are out actions. */
		if ( anyEofActions() )
			out << "execFuncs:\n";

		out <<
			"	switch ( _acts ) {\n";
			ACTION_SWITCH() <<
			"	}\n"
			"\n";
	}

	out <<
		"	goto again;\n"
		"\n";

	/* If there are any actions, then jump to the func execution. */
	if ( anyEofActions() ) {
		out <<
			"eofActions:\n"
			"	if ( *_pcs & SPEC_EOF_ACTION ) {\n"
			"		_acts = *(_pcs + (*_pcs>>8)-1);\n"
			"		_p = 0; _pe = (("; EL_TYPE() << "*)0)+1;\n"
			"		goto execFuncs;\n"
			"	}\n"
			"\n";
	}

	out << 
		"out:\n"
		"	fsm->_cs = _cs;\n";

	if ( redFsm->errState != 0 ) {
		out <<
			"	if ( _cs == "; ERROR_STATE() << " )\n"
			"		return -1;\n";
	}

	out << 
		"	if ( _cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	/* Finish routine. */
	out <<
		"int "; FSM_NAME() << "_finish( struct "; FSM_NAME() << " *fsm )\n"
		"{\n";

	if ( anyEofActions() ) {
		/* May need to execute code in the action loop. Must use special
		 * params to execute func. */
		out << "	return "; FSM_NAME() << "_execute( fsm, 0, 1 );\n";
	}
	else {
		if ( redFsm->errState != 0 ) {
			out << 
				"	if ( fsm->_cs == "; ERROR_STATE() << " )\n"
				"		return -1;\n";
		}

		out <<
			"	if ( fsm->_cs >= "; FIRST_FINAL_STATE() << " )\n"
			"		return 1;\n"
			"	return 0;\n";
	}

	out <<
		"}\n"
		"\n";

	/* Cleanup after ourselves. */
	out <<
		"#undef SPEC_ANY_SINGLE\n"
		"#undef SPEC_ANY_RANGE\n"
		"#undef SPEC_EOF_ACTION\n"
		"\n";
}


/* Init base data. */
CCFTabCodeGen::CCFTabCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out )
:
	FTabCodeGen(fsmName, parseData, redFsm, out) 
{ }

std::ostream &CCFTabCodeGen::CALL( NameInst *name, int targState, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "{this->_st[this->_top++] = _cs; _cs = " << 
			targ->value->position << "; goto again;}";
	return out;
}

std::ostream &CCFTabCodeGen::RET( bool inFinish )
{
	out << "{_cs = this->_st[--this->_top]; goto again;}";
	return out;
}

std::ostream &CCFTabCodeGen::CALLE( char *stateExpr, int targState, bool inFinish )
{
	out << "{this->_st[this->_top++] = _cs; _cs = (" << stateExpr << "); goto again;}";
	return out;
}

void CCFTabCodeGen::writeOutHeader()
{
	out <<
		"class "; FSM_NAME() << "\n"
		"{\n"
		"public:\n"
		"	int _cs;\n"
		"\n"
		"	/* Initialize the machine for execution. */\n"
		"	int init( );\n"
		"\n"
		"	/* Execute some chunk of data. */\n"
		"	int execute( "; EL_TYPE() << " *data, int len );\n"
		"\n"
		"	/* Indicate to the fsm tha there is no more data. */\n"
		"	int finish( );\n";
		STRUCT_DATA() <<
		"};\n"
		"\n";
}

void CCFTabCodeGen::writeOutCode()
{
	/* Need the state positions. */
	calcStatePositions();
	
	/* State machine data. */
	out << 
		"#define SPEC_ANY_SINGLE  0x01\n"
		"#define SPEC_ANY_RANGE   0x02\n"
		"#define SPEC_EOF_ACTION  0x04\n"
		"\n";

	/* Write the array of keys. */
	out <<
		"static "; ALPH_TYPE() << " "; FSM_NAME() << "_k[] = {\n";
		KEYS() <<
		"};\n"
		"\n";

	/* Write the array of indicies. */
	out << 
		"static "; INDEX_TYPE() << " "; FSM_NAME() << "_i[] = {\n";
		INDICIES() <<
		"};\n"
		"\n";

	/* Write the array of states. */
	out <<
		"static int "; FSM_NAME() << "_s[] = {\n";
		STATES() <<
		"};\n"
		"\n"
		"static int "; FSM_NAME() << "_t[] = {\n";
		TRANSITIONS() <<
		"};\n"
		"\n"
		"static int "; FSM_NAME() << "_start = " << 
				redFsm->startState->position << ";\n"
		"\n";

	/* Init routine. */
	out << 
		"int "; FSM_NAME() << "::init( )\n"
		"{\n"
		"	this->_cs = "; FSM_NAME() << "_start;\n";

	/* If there are any calls, then the stack top needs initialization. */
	if ( anyActionCalls() || anyActionRets() )
		out << "	this->_top = 0;\n";

	INIT_CODE() <<
		"	if ( this->_cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	/* The binary search. */
	BSEARCH() << "\n";

	/* The binary search for a range. */
	RANGE_BSEARCH() << "\n";

	/* Execution function. */
	out << 
		"int "; FSM_NAME() << "::execute( "; EL_TYPE() << " *_data, int _len )\n"
		"{\n"
		"	"; EL_TYPE() << " *_p = _data-1;\n"
		"	"; EL_TYPE() << " *_pe = _data+_len;\n"
		"	int _cs = this->_cs, *_pcs";

	if ( anyRegCurStateRef() )
		out << ", _ps";
	out << ";\n\tint _specs";
	
	if ( anyActions() ) 
		out << ", _acts";

	out <<
		";\n"
		"	"; ALPH_TYPE() << " *_keys;\n"
		"	"; INDEX_TYPE() << " *_inds;\n"
		"	int *_trans;\n"
		"\n";

	out <<
		"again:\n"
		"	if ( ++_p == _pe )\n"
		"		goto out;\n"
		"	_pcs = "; FSM_NAME() << "_s + _cs;\n";

	if ( anyEofActions() ) {
		out <<
			"	if ( _p == 0 )\n"
			"		goto eofActions;\n"
			"\n";
	}

	LOCATE_TRANS() <<
		"\n"
		"match:\n";
	
	if ( anyRegCurStateRef() )
		out << "	_ps = _cs;\n";

	out <<
		"	_cs = *_trans++;\n"
		"\n";

	if ( anyActions() ) {
		out << 
			"	if ( *_trans == 0 )\n"
			"		goto again;\n"
			"	_acts = *_trans;\n"
			"\n";

		if ( anyEofActions() )
			out << "execFuncs:\n";

		out << 
			"	switch ( _acts ) {\n";
			ACTION_SWITCH() <<
			"	}\n"
			"\n";
	}

	out <<
		"	goto again;\n"
		"\n";

	/* If there are any functions, then jump to the func execution. */
	if ( anyEofActions() ) {
		out <<
			"eofActions:\n"
			"	if ( *_pcs & SPEC_EOF_ACTION ) {\n"
			"		_acts = *(_pcs + (*_pcs>>8)-1);\n"
			"		_p = 0; _pe = (("; EL_TYPE() << "*)0)+1;\n"
			"		goto execFuncs;\n"
			"	}\n"
			"\n";
	}

	out <<
		"out:\n"
		"	this->_cs = _cs;\n";

	if ( redFsm->errState != 0 ) {
		out <<
			"	if ( _cs == "; ERROR_STATE() << " )\n"
			"		return -1;\n";
	}
	
	out <<
		"	if ( _cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	/* Finish routine. */
	out <<
		"int "; FSM_NAME() << "::finish( )\n"
		"{\n";

	if ( anyEofActions() ) {
		/* May need to execute code in the action loop. Must use special
		 * params to execute func. */
		out << "	return execute( 0, 1 );\n";
	}
	else {
		if ( redFsm->errState != 0 ) {
			out << 
				"	if ( this->_cs == "; ERROR_STATE() << " )\n"
				"		return -1;\n";
		}

		out <<
			"	if ( this->_cs >= "; FIRST_FINAL_STATE() << " )\n"
			"		return 1;\n"
			"	return 0;\n";
	}

	out <<
		"}\n"
		"\n";

	/* Cleanup after ourselves. */
	out <<
		"#undef SPEC_ANY_SINGLE\n"
		"#undef SPEC_ANY_RANGE\n"
		"#undef SPEC_EOF_ACTION\n"
		"\n";
}
