/*
 *  Copyright 2001-2003 Adrian Thurston <thurston@cs.queensu.ca>
 */

/*  This file is part of Ragel.
 *
 *  Ragel is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Ragel is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Ragel; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */


#include "fsmgraph.h"
#include "fsmdump.h"

using std::ostream;
using std::endl;

FsmDump::FsmDump( char *fsmName, ParseData *parseData, FsmAp *graph, std::ostream &out )
:
	fsmName(fsmName),
	parseData(parseData),
	graph(graph),
	out(out)
{
}

/* Write out a key of a transition. Considers the signedness of the key. */
void FsmDump::dumpFsmKey( int onChar )
{
	if ( graph->keyOps->isAlphSigned )
		out << onChar;
	else
		out << (unsigned int) onChar;
}

/* Dump the target state, actions and priorities of a transition. */
void FsmDump::dumpTransData( TransAp *trans )
{
	if ( trans->toState != 0 ) 
		out << trans->toState->alg.stateNum;
	else
		out << "xx";
	out << " [";
	
	/* Print the priorities, adding in commas in between. */
	PriorTable::Iter prior = trans->priorTable;
	for ( ; prior.lte(); prior++ ) {
		out << prior->desc->priority;
		if ( !prior.last() )
			out << ",";
	}

	out << "](";

	ActionTable *tables[3];
	tables[0] = &trans->fromState->fromStateActionTable;
	tables[1] = &trans->actionTable;
	tables[2] = &trans->toState->toStateActionTable;

	for ( int a = 0; a < 3; a++ ) {
		/* Print the actions, adding in commas in between. */
		for ( ActionTable::Iter action = *tables[a]; action.lte(); action++ ) {
			out << action->value;
			if ( !action.last() )
				out << ",";
		}
	}
		
	out << ")";
}

/* Dump a transition. */
void FsmDump::dumpTrans( TransAp *trans )
{
	out << " ";
	dumpFsmKey( trans->lowKey );
	if ( !graph->keyOps->eq( trans->lowKey, trans->highKey ) ) {
		out << "-";
		dumpFsmKey( trans->highKey );
	}
	out << " ";
	dumpTransData( trans );
	out << " |";
}

/* Dump a default transition. */
void FsmDump::dumpDefTrans( TransAp *trans )
{
	out << " df ";
	dumpTransData( trans );
	out << " |";
}

/* Walk a list of range transitions and dump the individual transitions. */
void FsmDump::dumpTransList( const TransList &list )
{
	for ( TransList::Iter trans = list; trans.lte(); trans++ )
		dumpTrans( trans );
}

/* Write out a state. */
void FsmDump::dumpState( StateAp *state )
{
	out << state->alg.stateNum << " [";

	PriorTable::Iter prior = state->outPriorTable;
	for ( ; prior.lte(); prior++ ) {
		out << prior->desc->priority;
		if ( !prior.last() )
			out << ",";
	}

	out << "](";

	/* Print the out actions, adding in commas in between. */
	ActionTable::Iter action = state->outActionTable;
	for ( ; action.lte(); action ++ ) {
		out << action->value;
		if ( !action.last() )
			out << ",";
	}

	out << ")\n->";

	/* If there is a default trans, dump it. */
	if ( state->outDefault != 0 )
		dumpDefTrans( state->outDefault );

	/* Dump the range transitions. */
	dumpTransList( state->outList );

	out << endl;
}

/* Write out a graph. */
void FsmDump::dumpGraph( )
{
	/* Set the state numbers so we print something nice! */
	graph->setStateNumbers();

	/* Some header info. */
	out << "DUMP of " << fsmName << endl;
	out << "StartState: " << graph->startState->alg.stateNum << endl;
	for ( EntryMap::Iter en = graph->entryPoints; en.lte(); en++ ) {
		out << "Entry Point: " << en->key << " -> " << 
				en->value->alg.stateNum << endl;
	}

	out << "Final States:";

	/* Dump Final states. */
	StateAp **st = graph->finStateSet.data;
	int nst = graph->finStateSet.length();
	for ( int i = 0; i < nst; i++, st++ )
		out << " " << (*st)->alg.stateNum << " |";
	out << endl;

	/* Walk the list of states, printing. */
	for ( StateList::Iter st = graph->stateList; st.lte(); st++ ) 
		dumpState( st );
}
