#!/usr/bin/env ruby

require 'test/unit'
require 'rake'

class TestFileList < Test::Unit::TestCase
  FileList = Rake::FileList

  def setup
    create_test_data
  end

  def teardown
    FileList.select_default_ignore_patterns
    FileUtils.rm_rf("testdata")
  end

  def test_create
    fl = FileList.new
    assert_equal 0, fl.size
  end

  def test_create_with_args
    fl = FileList.new("testdata/*.c", "x")
    assert_equal ["testdata/abc.c", "testdata/x.c", "testdata/xyz.c", "x"].sort,
      fl.sort
  end

  def test_create_with_block
    fl = FileList.new { |f| f.include("x") }
    assert_equal ["x"], fl.resolve
  end

  def test_create_with_brackets
    fl = FileList["testdata/*.c", "x"]
    assert_equal ["testdata/abc.c", "testdata/x.c", "testdata/xyz.c", "x"].sort,
      fl.sort
  end

  def test_create_with_brackets_and_filelist
    fl = FileList[FileList["testdata/*.c", "x"]]
    assert_equal ["testdata/abc.c", "testdata/x.c", "testdata/xyz.c", "x"].sort,
      fl.sort
  end

  def test_include_with_another_array
    fl = FileList.new.include(["x", "y", "z"])
    assert_equal ["x", "y", "z"].sort, fl.sort
  end

  def test_include_with_another_filelist
    fl = FileList.new.include(FileList["testdata/*.c", "x"])
    assert_equal ["testdata/abc.c", "testdata/x.c", "testdata/xyz.c", "x"].sort,
      fl.sort
  end

  def test_append
    fl = FileList.new
    fl << "a.rb" << "b.rb"
    assert_equal ['a.rb', 'b.rb'], fl
  end

  def test_add_many
    fl = FileList.new
    fl.include %w(a d c)
    fl.include('x', 'y')
    assert_equal ['a', 'd', 'c', 'x', 'y'], fl
    assert_equal ['a', 'd', 'c', 'x', 'y'], fl.resolve
  end

  def test_add_return
    f = FileList.new
    g = f << "x"
    assert_equal f.object_id, g.object_id
    h = f.include("y")
    assert_equal f.object_id, h.object_id
  end
  
  def test_match
    fl = FileList.new
    fl.include('test/test*.rb')
    assert fl.include?("test/test_filelist.rb")
    assert fl.size > 3
    fl.each { |fn| assert_match(/\.rb$/, fn) }
  end

  def test_add_matching
    fl = FileList.new
    fl << "a.java"
    fl.include("test/*.rb")
    assert_equal "a.java", fl[0]
    assert fl.size > 2
    assert fl.include?("test/test_filelist.rb")
  end

  def test_multiple_patterns
    create_test_data
    fl = FileList.new
    fl.include('*.c', '*xist*')
    assert_equal [], fl
    fl.include('testdata/*.c', 'testdata/*xist*')
    assert_equal [
      'testdata/x.c', 'testdata/xyz.c', 'testdata/abc.c', 'testdata/existing'
    ].sort, fl.sort
  end

  def test_reject
    fl = FileList.new
    fl.include %w(testdata/x.c testdata/abc.c testdata/xyz.c testdata/existing)
    fl.reject! { |fn| fn =~ %r{/x} }
    assert_equal [
      'testdata/abc.c', 'testdata/existing'
    ], fl
  end

  def test_exclude
    fl = FileList['testdata/x.c', 'testdata/abc.c', 'testdata/xyz.c', 'testdata/existing']
    fl.each { |fn| touch fn, :verbose => false }
    x = fl.exclude(%r{/x.+\.})
    assert_equal FileList, x.class
    assert_equal [
      'testdata/x.c', 'testdata/abc.c', 'testdata/existing'
    ], fl
    assert_equal fl.object_id, x.object_id
    fl.exclude('testdata/*.c')
    assert_equal ['testdata/existing'], fl
    fl.exclude('testdata/existing')
    assert_equal [], fl
  end

  def test_exclude_return_on_create
    fl = FileList['testdata/*'].exclude(/.*\.c$/)
    assert_equal FileList, fl.class
  end

  def test_default_exclude
    fl = FileList.new
    fl.clear_exclude
    fl.include("**/*~", "**/*.bak", "**/core")
    assert fl.member?("testdata/core"), "Should include core"
    assert fl.member?("testdata/x.bak"), "Should include .bak files"
  end

  def test_unique
    fl = FileList.new
    fl << "x.c" << "a.c" << "b.rb" << "a.c"
    assert_equal ['x.c', 'a.c', 'b.rb', 'a.c'], fl
    fl.uniq!
    assert_equal ['x.c', 'a.c', 'b.rb'], fl
  end

  def test_to_string
    fl = FileList.new
    fl << "a.java" << "b.java"
    assert_equal  "a.java b.java", fl.to_s
    assert_equal  "a.java b.java", "#{fl}"
  end

  def test_to_array
    fl = FileList['a.java', 'b.java']
    assert_equal  ['a.java', 'b.java'], fl.to_a
    assert_equal  Array, fl.to_a.class
    assert_equal  ['a.java', 'b.java'], fl.to_ary
    assert_equal  Array, fl.to_ary.class
  end

  def test_to_s_pending
    fl = FileList['testdata/abc.*']
    assert_equal  %{testdata/abc.c}, fl.to_s
  end

  def test_inspect_pending
    fl = FileList['testdata/abc.*']
    assert_equal  %{["testdata/abc.c"]}, fl.inspect
  end

  def test_sub
    fl = FileList["testdata/*.c"]
    f2 = fl.sub(/\.c$/, ".o")
    assert_equal FileList, f2.class
    assert_equal ["testdata/abc.o", "testdata/x.o", "testdata/xyz.o"].sort,
      f2.sort
    f3 = fl.gsub(/\.c$/, ".o")
    assert_equal FileList, f3.class
    assert_equal ["testdata/abc.o", "testdata/x.o", "testdata/xyz.o"].sort,
      f3.sort
  end

  def test_sub!
    f = "x/a.c"
    fl = FileList[f, "x/b.c"]
    res = fl.sub!(/\.c$/, ".o")
    assert_equal ["x/a.o", "x/b.o"].sort, fl.sort
    assert_equal "x/a.c", f
    assert_equal fl.object_id, res.object_id
  end

  def test_sub_with_block
    fl = FileList["src/org/onestepback/a.java", "src/org/onestepback/b.java"]
# The block version doesn't work the way I want it to ...
#    f2 = fl.sub(%r{^src/(.*)\.java$}) { |x|  "classes/" + $1 + ".class" }
    f2 = fl.sub(%r{^src/(.*)\.java$}, "classes/\\1.class")
    assert_equal [
      "classes/org/onestepback/a.class",
      "classes/org/onestepback/b.class"
    ].sort,
      f2.sort
  end

  def test_string_ext
    assert_equal "one.net", "one.two".ext("net")
    assert_equal "one.net", "one.two".ext(".net")
    assert_equal "one.net", "one".ext("net")
    assert_equal "one.net", "one".ext(".net")
    assert_equal "one.two.net", "one.two.c".ext(".net")
    assert_equal "one/two.net", "one/two.c".ext(".net")
    assert_equal "one.x/two.net", "one.x/two.c".ext(".net")
    assert_equal "one.x\\two.net", "one.x\\two.c".ext(".net")
    assert_equal "one.x/two.net", "one.x/two".ext(".net")
    assert_equal "one.x\\two.net", "one.x\\two".ext(".net")
    assert_equal ".onerc.net", ".onerc.dot".ext("net")
    assert_equal ".onerc.net", ".onerc".ext("net")
    assert_equal ".a/.onerc.net", ".a/.onerc".ext("net")
    assert_equal "one", "one.two".ext('')
    assert_equal "one", "one.two".ext
    assert_equal ".one", ".one.two".ext
    assert_equal ".one", ".one".ext
    assert_equal ".", ".".ext("c")
    assert_equal "..", "..".ext("c")
  end

  def test_filelist_ext
    assert_equal FileList['one.c', '.one.c'],
      FileList['one.net', '.one'].ext('c')
  end

  def test_gsub
    create_test_data
    fl = FileList["testdata/*.c"]
    f2 = fl.gsub(/a/, "A")
    assert_equal ["testdAtA/Abc.c", "testdAtA/x.c", "testdAtA/xyz.c"].sort,
      f2.sort
  end

  def test_egrep
    files = FileList['test/test*.rb']
    found = false
    the_line_number = __LINE__ + 1
    files.egrep(/XYZZY/) do |fn, ln, line |
      assert_equal 'test/test_filelist.rb', fn
      assert_equal the_line_number, ln
      assert_match(/files\.egrep/, line)
      found = true
    end
    assert found, "should have foudn a matching line"
  end


  def test_ignore_special
    f = FileList['testdata/*']
    assert ! f.include?("testdata/CVS"), "Should not contain CVS"
    assert ! f.include?("testdata/.svn"), "Should not contain .svn"
    assert ! f.include?("testdata/.dummy"), "Should not contain dot files"
    assert ! f.include?("testdata/x.bak"), "Should not contain .bak files"
    assert ! f.include?("testdata/x~"), "Should not contain ~ files"
    assert ! f.include?("testdata/core"), "Should not contain core files"
  end

  def test_clear_ignore_patterns
    f = FileList['testdata/*', 'testdata/.svn']
    f.clear_exclude
    assert f.include?("testdata/abc.c")
    assert f.include?("testdata/xyz.c")
    assert f.include?("testdata/CVS")
    assert f.include?("testdata/.svn")
    assert f.include?("testdata/x.bak")
    assert f.include?("testdata/x~")
  end

  def test_exclude_with_alternate_file_seps
    fl = FileList.new
    assert fl.exclude?("x/CVS/y")
    assert fl.exclude?("x\\CVS\\y")
    assert fl.exclude?("x/.svn/y")
    assert fl.exclude?("x\\.svn\\y")
    assert fl.exclude?("x/core")
    assert fl.exclude?("x\\core")
  end

  def test_add_default_exclude_list
    fl = FileList.new
    fl.exclude(/~\d+$/)
    assert fl.exclude?("x/CVS/y")
    assert fl.exclude?("x\\CVS\\y")
    assert fl.exclude?("x/.svn/y")
    assert fl.exclude?("x\\.svn\\y")
    assert fl.exclude?("x/core")
    assert fl.exclude?("x\\core")
    assert fl.exclude?("x/abc~1")
  end

  def test_basic_array_functions
    f = FileList['b', 'c', 'a']
    assert_equal 'b', f.first
    assert_equal 'b', f[0]
    assert_equal 'a', f.last
    assert_equal 'a', f[2]
    assert_equal 'a', f[-1]
    assert_equal ['a', 'b', 'c'], f.sort
    f.sort!
    assert_equal ['a', 'b', 'c'], f
  end

  def test_flatten
    assert_equal ['a', 'testdata/x.c', 'testdata/xyz.c', 'testdata/abc.c'].sort,
      ['a', FileList['testdata/*.c']].flatten.sort
  end

  def test_clone
    a = FileList['a', 'b', 'c']
    b = a.clone
    a << 'd'
    assert_equal ['a', 'b', 'c', 'd'], a
    assert_equal ['a', 'b', 'c'], b
  end

  def test_array_comparisons
    fl = FileList['b', 'b']
    a = ['b', 'a']
    b = ['b', 'b']
    c = ['b', 'c']
    assert_equal( 1,  fl <=> a )
    assert_equal( 0,  fl <=> b )
    assert_equal( -1, fl <=> c )
    assert_equal( -1, a <=> fl )
    assert_equal( 0,  b <=> fl )
    assert_equal( 1,  c <=> fl )
  end

  def test_array_equality
    a = FileList['a', 'b']
    b = ['a', 'b']
    assert a == b
    assert b == a
#    assert a.eql?(b)
#    assert b.eql?(a)
    assert ! a.equal?(b)
    assert ! b.equal?(a)
  end

  def test_enumeration_methods
    a = FileList['a', 'b']
    b = a.collect { |it| it.upcase }
    assert_equal ['A', 'B'], b
    assert_equal FileList,  b.class

    b = a.map { |it| it.upcase }
    assert_equal ['A', 'B'], b
    assert_equal FileList,  b.class

    b = a.sort
    assert_equal ['a', 'b'], b
    assert_equal FileList,  b.class

    b = a.sort_by { |it| it }
    assert_equal ['a', 'b'], b
    assert_equal FileList,  b.class

    b = a.find_all { |it| it == 'b'}
    assert_equal ['b'], b
    assert_equal FileList,  b.class

    b = a.select { |it| it.size == 1 }
    assert_equal ['a', 'b'], b
    assert_equal FileList,  b.class

    b = a.reject { |it| it == 'b' }
    assert_equal ['a'], b
    assert_equal FileList,  b.class

    b = a.grep(/./)
    assert_equal ['a', 'b'], b
    assert_equal FileList,  b.class

    b = a.partition { |it| it == 'b' }
    assert_equal [['b'], ['a']], b
    assert_equal Array, b.class
    assert_equal FileList,  b[0].class
    assert_equal FileList,  b[1].class

    b = a.zip(['x', 'y'])
    assert_equal [['a', 'x'], ['b', 'y']], b
    assert_equal Array, b.class
    assert_equal Array, b[0].class
    assert_equal Array, b[1].class
  end

  def test_array_operators
    a = ['a', 'b']
    b = ['c', 'd']
    f = FileList['x', 'y']
    g = FileList['w', 'z']

    r = f + g
    assert_equal ['x', 'y', 'w', 'z'], r
    assert_equal FileList, r.class

    r = a + g
    assert_equal ['a', 'b', 'w', 'z'], r
    assert_equal Array, r.class

    r = f + b
    assert_equal ['x', 'y', 'c', 'd'], r
    assert_equal FileList, r.class

    r = FileList['w', 'x', 'y', 'z'] - f
    assert_equal ['w', 'z'], r
    assert_equal FileList, r.class

    r = FileList['w', 'x', 'y', 'z'] & f
    assert_equal ['x', 'y'], r
    assert_equal FileList, r.class

    r = f * 2
    assert_equal ['x', 'y', 'x', 'y'], r
    assert_equal FileList, r.class

    r = f * ','
    assert_equal 'x,y', r
    assert_equal String, r.class

    r = f | ['a', 'x']
    assert_equal ['a', 'x', 'y'].sort, r.sort
    assert_equal FileList, r.class
  end

  def test_other_array_returning_methods
    f = FileList['a', nil, 'b']
    r = f.compact
    assert_equal ['a', 'b'], r
    assert_equal FileList, r.class

    f = FileList['a', 'b']
    r = f.concat(['x', 'y'])
    assert_equal ['a', 'b', 'x', 'y'], r
    assert_equal FileList, r.class

    f = FileList['a', ['b', 'c'], FileList['d', 'e']]
    r = f.flatten
    assert_equal ['a', 'b', 'c', 'd', 'e'], r
    assert_equal FileList, r.class

    f = FileList['a', 'b', 'a']
    r = f.uniq
    assert_equal ['a', 'b'], r
    assert_equal FileList, r.class

    f = FileList['a', 'b', 'c', 'd']
    r = f.values_at(1,3)
    assert_equal ['b', 'd'], r
    assert_equal FileList, r.class
  end

  def create_test_data
    verbose(false) do
      mkdir "testdata" unless File.exist? "testdata"
      mkdir "testdata/CVS" rescue nil
      mkdir "testdata/.svn" rescue nil
      touch "testdata/.dummy"
      touch "testdata/x.bak"
      touch "testdata/x~"
      touch "testdata/core"
      touch "testdata/x.c"
      touch "testdata/xyz.c"
      touch "testdata/abc.c"
      touch "testdata/existing"
    end
  end
  
end
