/*
   Name: $RCSfile: mod_init.c,v $
   Author: Alan Moran
   $Date: 2005/11/26 15:04:17 $
   $Revision: 1.5 $
   $Id: mod_init.c,v 1.5 2005/11/26 15:04:17 a_j_moran Exp $

   Legal Notice:

   This program is free software; you can redistribute it and/or
   modify it under the terms of the license contained in the
   COPYING file that comes with this distribution.

 */

/**
   @file

   @brief Functions to support the init module.

   The init modules registers assets in the registry.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include "globals.h"
#include "mime.h"
#include "mod_init.h"

static rpl_str_t rpl_init_base_dir;
static rpl_str_t terr_dir, tout_dir, parse_dir, xslt_dir;

/**
   Configure the tidy module.

   @param fns pointer to module interface to be configured.
 */
void
rpl_mod_init_configure(rpl_mod_fns *fns)
{
	fns->init = rpl_mod_init_init;
	fns->process = rpl_mod_init_process;
	fns->cleanup = rpl_mod_init_cleanup;
	rpl_init_base_dir = rpl_wk_get_srcdir();
	fns->basedir = rpl_init_base_dir;
}

/**
   Perform registry initialization.
 */
rpl_wk_status 
rpl_mod_init_init()
{
	rpl_wk_status status = RPL_WK_OK;

	rpl_reg_init();
	/* define error and output directories */
	terr_dir = rpl_str_concat(rpl_cfg_get_ds_basedir(), "/", RPL_DS_TERR_DIR, RPL_STR_EOC);
	rpl_fs_mkdir(terr_dir);
	tout_dir = rpl_str_concat(rpl_cfg_get_ds_basedir(), "/", RPL_DS_TOUT_DIR, RPL_STR_EOC);
	rpl_fs_mkdir(tout_dir);
	parse_dir = rpl_str_concat(rpl_cfg_get_ds_basedir(), "/", RPL_DS_PARSE_DIR, RPL_STR_EOC);
	rpl_fs_mkdir(parse_dir);
	xslt_dir = rpl_wk_get_webdir();

	return status; 
}

/**
   @param filename name of file to be registered.
   @param st_buf stat of file.
 */
rpl_wk_status 
rpl_mod_init_process(rpl_c_str_t filename, struct stat statbuf)
{
	rpl_wk_status status = RPL_WK_OK;
	rpl_web_asset *wa;
	rpl_str_t msg, mime_type, rdp, fp, key, terr_fp, tout_fp, parse_fp, xslt_fp;

	assert(filename != NULL);

	msg = rpl_str_concat(rpl_message_get("WK_PROCESSING", RPL_EOM), "init ", filename, RPL_STR_EOC);
	rpl_log_info(msg);
	rpl_me_free(msg);

	/* extract key information */
	if(rpl_fs_resolve_paths(filename, rpl_init_base_dir, &rdp, &fp))
		return RPL_WK_ERR;
	key = rpl_reg_create_key(rdp, fp);

	if(S_ISREG(statbuf.st_mode))
	{
		/* TODO: decide how best to set the contributor detail */
		wa = (rpl_web_asset *)rpl_me_malloc(sizeof(rpl_web_asset));
		rpl_wa_init(wa);

		/* set mime type based on filename extension */
		mime_type = rpl_mime_get_type(filename);
		rpl_wa_set_mime_type(mime_type, wa);
		rpl_me_free(mime_type);

		/* populate web asset fields */
		rpl_wa_set_rel_dir(rdp, wa);
		rpl_wa_set_filename(fp, wa);
		rpl_wa_set_key(wa);

		rpl_reg_insert(wa);	


	} else if(S_ISDIR(statbuf.st_mode)) {

		if(strlen(rdp) + strlen(fp) > 0)
		{
			/* use relative path to resolve directories for errors and output */
			terr_fp = (rpl_str_t)rpl_me_malloc(strlen(terr_dir) + strlen(key) + 2);
			sprintf(terr_fp, "%s/%s", terr_dir, key);
			tout_fp = (rpl_str_t)rpl_me_malloc(strlen(tout_dir) + strlen(key) + 2);
			sprintf(tout_fp, "%s/%s", tout_dir, key);
			parse_fp = (rpl_str_t)rpl_me_malloc(strlen(parse_dir) + strlen(key) + 2);
			sprintf(parse_fp, "%s/%s", parse_dir, key);
			xslt_fp = (rpl_str_t)rpl_me_malloc(strlen(xslt_dir) + strlen(key) + 2);
			sprintf(xslt_fp, "%s/%s", xslt_dir, key);

			rpl_fs_mkdir(terr_fp);
			rpl_fs_mkdir(tout_fp);
			rpl_fs_mkdir(parse_fp);
			rpl_fs_mkdir(xslt_fp);

			rpl_me_free(terr_fp);
			rpl_me_free(tout_fp);
			rpl_me_free(parse_fp);
			rpl_me_free(xslt_fp);

			rpl_me_free(rdp);
			rpl_me_free(fp);
			rpl_me_free(key);
		}
	}


	return status; 
}

/**
   Release resources held during processing. 
 */
rpl_wk_status 
rpl_mod_init_cleanup()
{
	rpl_wk_status status = RPL_WK_OK;

	rpl_me_free(terr_dir);
	rpl_me_free(tout_dir);
	rpl_me_free(parse_dir);
	/* do not free these
	rpl_me_free(xslt_dir);
	rpl_me_free(rpl_init_base_dir);
	*/
	return status;
}

