/*
   Name: $RCSfile: mod_parse.c,v $
   Author: Alan Moran
   $Date: 2005/11/26 15:04:17 $
   $Revision: 1.5 $
   $Id: mod_parse.c,v 1.5 2005/11/26 15:04:17 a_j_moran Exp $

   Legal Notice:

   This program is free software; you can redistribute it and/or
   modify it under the terms of the license contained in the
   COPYING file that comes with this distribution.

 */

/**
   @file

   @brief Functions to support the parse module.

   This module passes files over to the rapple parser that performs
   a variety of simple parsing tasks (incl. handling entities etc.)
   that are not done by the other tools (e.g., tidy.)

 */

#include <sys/types.h>
#include <sys/stat.h>
#include "globals.h"
#include "parser.h"
#include "mod_parse.h"

static rpl_str_t rpl_parse_base_dir;
static rpl_str_t out_dir;

/**
   Configure the tidy module.

   @param fns pointer to module interface to be configured.
 */
void
rpl_mod_parse_configure(rpl_mod_fns *fns)
{
	fns->init = rpl_mod_parse_init;
	fns->process = rpl_mod_parse_process;
	fns->cleanup = rpl_mod_parse_cleanup;
	rpl_parse_base_dir = rpl_str_concat(rpl_cfg_get_ds_basedir(), "/", RPL_DS_TOUT_DIR, RPL_STR_EOC);
	fns->basedir = rpl_parse_base_dir;
}

/**
   Perform registry initialization.
 */
rpl_wk_status 
rpl_mod_parse_init()
{
	rpl_wk_status status = RPL_WK_OK;

	/* define error and output directories */
	out_dir = rpl_str_concat(rpl_cfg_get_ds_basedir(), "/", RPL_DS_PARSE_DIR, RPL_STR_EOC);

	return status; 
}

/**
   @param filename name of file to be registered.
   @param st_buf stat of file.
 */
rpl_wk_status 
rpl_mod_parse_process(rpl_c_str_t filename, struct stat statbuf)
{
	rpl_wk_status status = RPL_WK_OK;
	rpl_reg_item item;
	rpl_str_t msg, rdp, fp, key, outfp, ctnt, pctnt;

    assert(filename != NULL);

	msg = rpl_str_concat(rpl_message_get("WK_PROCESSING", RPL_EOM), "parse ", filename, RPL_STR_EOC);
	rpl_log_info(msg);
	rpl_me_free(msg);

	if(S_ISREG(statbuf.st_mode))
	{
		if(rpl_fs_resolve_paths(filename, rpl_parse_base_dir, &rdp, &fp))
			return RPL_WK_ERR;
		key = rpl_reg_create_key(rdp, fp);

		/* write the parsed file to the appropriate location in the datastore */
		outfp = (rpl_str_t)rpl_me_malloc(strlen(out_dir) + strlen(rdp) + strlen(fp) + 3);
		sprintf(outfp, "%s/%s/%s", out_dir, rdp, fp);

		/* retrieve asset from registry */
		item = rpl_reg_search(key);
		if(item == &RPL_REG_WA_NULL)
		{
			rpl_log_error(rpl_message_get("REG_ASSET_NOT_FOUND", key, RPL_EOM));
			return RPL_WK_ERR;
		}
		
    	if(rpl_wa_is_transformable(*item))
		{
			ctnt = rpl_fs_f2str(filename);
			pctnt = rpl_parse(ctnt);
			rpl_fs_str2f(pctnt, outfp);
			/* without the check we can get free junk pointer errors */
			if(strlen(pctnt) > 0)
				rpl_me_free(pctnt);
			rpl_me_free(ctnt);

		} else {
			rpl_fs_cp(filename, (rpl_c_str_t)outfp);
		}

		if(strlen(rdp) > 0)
			rpl_me_free(rdp);
		if(strlen(fp) > 0)
			rpl_me_free(fp);
		rpl_me_free(outfp);
	} 

	return status; 
}

/**
   Release resources held during processing. 
 */
rpl_wk_status 
rpl_mod_parse_cleanup()
{
	rpl_wk_status status = RPL_WK_OK;

	rpl_me_free(out_dir);
	rpl_me_free(rpl_parse_base_dir);

	return status;
}

