/*
   Name: $RCSfile: workflow.c,v $
   Author: Alan Moran
   $Date: 2005/10/25 13:53:03 $
   $Revision: 1.6 $
   $Id: workflow.c,v 1.6 2005/10/25 13:53:03 a_j_moran Exp $

   Legal Notice:

   This program is free software; you can redistribute it and/or
   modify it under the terms of the license contained in the
   COPYING file that comes with this distribution.

 */

/**
   @file

   @brief Functions to support workflow functionality.

   The workflow is based on modules that conform to a specific interface
   and which can be generically called by the workflow manager.  This
   module implements the specifics of the workflow.
*/

#include "globals.h"
#include "mod_init.h"
#include "mod_tidy.h"
#include "mod_index.h"
#include "mod_parse.h"
#include "mod_xslt.h"
#include "mod_link.h"
#include "mod_db.h"

static rpl_list 	wk_modules;
static rpl_str_t 	rpl_wk_domain, rpl_wk_srcdir, rpl_wk_webdir;

static void
rpl_wk_load(rpl_mod_configure_fp configure)
{
	rpl_mod_fns *fns;
	
	/* configure the function family */	
	if((fns = (rpl_mod_fns *)rpl_me_malloc(sizeof(rpl_mod_fns))) == NULL)
		rpl_log_fatal(rpl_message_get("OUT_OF_MEMORY", RPL_EOM));
	configure(fns);

	rpl_list_append(&wk_modules, &fns->node, fns);
}

rpl_str_t
rpl_wk_get_domain()
{
	return rpl_wk_domain;
}

rpl_str_t
rpl_wk_get_srcdir()
{
	return rpl_wk_srcdir;
}

rpl_str_t
rpl_wk_get_webdir()
{
	return rpl_wk_webdir;
}

void
rpl_wk_process(rpl_str_t domain, rpl_str_t srcdir, rpl_str_t webdir)
{
	rpl_mod_fns *mod;

	assert((domain != NULL) && (srcdir != NULL) && (webdir != NULL));

	rpl_wk_domain = domain;
	rpl_wk_srcdir = srcdir;
	rpl_wk_webdir = webdir;
	
	/* configure module list */
	rpl_list_init(&wk_modules);

	/* add workflow modules */
	rpl_wk_load(rpl_mod_init_configure);
	rpl_wk_load(rpl_mod_tidy_configure);
	rpl_wk_load(rpl_mod_index_configure);
	rpl_wk_load(rpl_mod_parse_configure);
	rpl_wk_load(rpl_mod_xslt_configure);
	rpl_wk_load(rpl_mod_link_configure);
#ifdef HAVE_MYSQL
	rpl_wk_load(rpl_mod_db_configure);
#endif

	/* process assets */
	mod = rpl_list_first(&wk_modules);
	while(mod != NULL) 
	{
		if(mod->init != NULL)
		{
			if(mod->init() == RPL_WK_ERR)
				rpl_log_fatal(rpl_message_get("WK_INIT_ERROR", RPL_EOM));
		}
		if((mod->basedir != NULL) && (mod->process != NULL))
		{
			rpl_fs_recurse(mod->basedir, mod->process);
		}
		if(mod->cleanup != NULL)
			mod->cleanup();
		mod = rpl_list_next(&mod->node);
	}	
}


