/*

  rawtran -  RAW to FITS converter
  Copyright (C) 2007-x  Filip Hroch, Masaryk University, Brno, CZ

  Rawtran is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.
  
  Rawtran is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with Rawtran.  If not, see <http://www.gnu.org/licenses/>.

*/

/*

  ToDo:
   - implement darkframe conversion ppm to pgm
   - implement conversion of FITS darkframe to pgm (using of mean darks)

 */

#include "rawtran.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <dirent.h>
#include <fnmatch.h>

#include <assert.h>

void help();
void version();
int dcraw_presented();
int in_filters(char *, char *[]);


int main(int argc,char *argv[])
{
  //  char *fits = "!rawtran.fits.gz[compress GZIP]";
  char *fits = "!rawtran.fits";
  char *opts = NULL, *execs = "-q 3 -w", *adds = NULL, *dark = NULL;
  int type = 0;
  int bitpix = 16;
  int status = 1;
  int i;
  char *filter = NULL;
  char *raw = NULL;
  char *standard_filters[] = {"X","Y","Z","R","V","B","u","s", NULL};
  char *instr_filters[] = {"P","Ri","Gi","Gi1","Gi2","Bi","all", NULL};

  if( argc == 1 ) {
    help();
    return(!dcraw_presented());
  }    
  
  for( i = 1; i < argc; i++) {

    /* help */
    if( strcmp(argv[i],"--help") == 0 || strcmp(argv[i],"-h") == 0 ){
      help();
      return(!dcraw_presented());
    }

    /* version and license */
    if( strcmp(argv[i],"--version") == 0 ){
      version();
      return(0);
    }
    
    /* output name */
    else if( strcmp(argv[i],"-o") == 0 && i++ < argc )
      fits = argv[i];

    /* select band */
    else if( strcmp(argv[i],"-c") == 0 && i++ < argc )
      filter = argv[i];
    
    /* select bits per pixel  */
    else if( strcmp(argv[i],"-B") == 0 && i++ < argc ) {
      if( sscanf(argv[i],"%d",&bitpix) != 1 )
	bitpix = 16;
    }

    /* conversion switches */
    else if( strcmp(argv[i],"-C") == 0 && i++ < argc )
      opts = argv[i];

    /* default switches */
    else if( strcmp(argv[i],"-X") == 0 && i++ < argc )
      execs = argv[i];

    /* additional switches */
    else if( strcmp(argv[i],"-A") == 0 && i++ < argc )
      adds = argv[i];

    else if( (strcmp(argv[i],"-D") == 0 || strcmp(argv[i],"-E") == 0) && i+1 < argc ) {
      if( dark ) {
	fprintf(stderr,"Options -D and -E should not to be specified together.\n");
	status = 1;
	goto finish;
      }
      /* RAW dark frame */
      if( strcmp(argv[i],"-D") == 0 ) 
	dark = darksh(argv[++i]);

      /* FITS dark frame */
      else if( strcmp(argv[i],"-E") == 0 ) 
	dark = darkfits(argv[++i]);	
    }

    else {

      if( *argv[i] == '-' )
	fprintf(stderr,"Warning: an unknown parameter: `%s'.\n",argv[i]);
      
      raw = argv[i];
    }
  }

  if( ! raw ) { 
    fprintf(stderr,"Unspecified raw photo.\n");
    return(1);
  }

  /* determine type of conversion */
  if( filter == NULL || in_filters(filter,standard_filters) ) {
    type = 0;
    if( ! opts ) 
      opts = "-4 -o 5";
  }
  else if( in_filters(filter,instr_filters) ) {
    type = 1;
    if( ! opts ) 
      opts = "-4 -D";
  }
  else {
    fprintf(stderr,"An unknown filter specified.\n");
    return(1);
  }

  status = rawtran(raw,fits,type,filter,bitpix,opts,execs,adds,dark);


 finish:
  if( dark ) {
    unlink(dark);
    free(dark);
  }
  return status;
}

void help()
{
  fprintf(stdout,
	 "RAWTRAN    RAW to FITS converter\n"
	 "Usage: rawtran [options] file\n"
	 "Options:\n"
	 " -c  select a band (X,Y,Z,R,V,B,u,s,P,Ri,Gi,Gi1,Gi2,Bi,all)\n"
	 "   * standard bands:\n"
	 "              XYZ stand for color bands of CIE 1931 colorspace\n"
	 "              RVB stand for color bands of Johnson UBVRI color system\n"
	 "              u stand for unfiltered (sum of X+Y+Z color bands)\n"
	 "              s stand for scotopic vision\n"
	 "   * instrumental bands:\n"
	 "              P stand for one plain linear fits with all the instrumental band\n"
	 "              Ri,Gi1,Gi2,Bi stand for Bayer's mask elements\n"
	 "              Gi stand for arithmetical mean of Gi-s, Gi=(Gi1 + Gi2)/2\n"
	 "              all stand for 4-dim image with R,Gi1,Gi2,B layers\n"
         " -o  output filename, '-' for std. output (default: !rawtran.fits)\n"
	 " -A  additional options passed to dcraw (default: '')\n"
	 " -B  bits per pixel of output image: 8,16,32,-32 (default:16)\n"
	 " -C  conversions for dcraw (default: '-4 -o 5' standard,'-4 -D' instrumental)\n"
	 " -D  use the RAW frame as a dark frame\n"
	 " -E  use the FITS frame as a dark frame\n"
	 " -X  options passed to dcraw (default: '-q 3 -w')\n"
         " -h,--help\t give this help\n"
         "    --version\t display software version and license\n");
}

void version()
{
  fprintf(stdout,"RAWTRAN, %s, (C) 2007-12,15 F. Hroch, Masaryk University in Brno, CZ\n\n",VERSION);
  fprintf(stdout,"RAWTRAN comes with ABSOLUTELY NO WARRANTY.\n"
	  "You may redistribute copies of RAWTRAN\n"
	  "under the terms of the GNU General Public License.\n"
	  "For more information about these matters, see the file named COPYING.\n");
}

int dcraw_presented()
{
  /*

    This part is looking for dcraw by an UNPORTABLE way:
      * closing of std. output by >&- works under sh only
      * the WEXITSTATUS may work under GNU only
      * WEXITSTATUS is used for right identification of status dcraw

   */

  int status;
  status = system("dcraw >&-");

  if( WEXITSTATUS(status) == 1 ) {
    fprintf(stderr,"Info: dcraw is available.\n");
    return(1);
  }
  else {
    fprintf(stderr,"\nWarning: dcraw not found.\n\n");
    return(0);
  }
}


int in_filters(char *filter, char *filters[])
{
  /* determine type of conversion */
  int i;

  for(i = 0; filters[i] != NULL; i++)
    if( strcmp(filters[i],filter) == 0 )
      return 1;

  return 0;
}
