/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "dhcpeditwidget.h"
#include <RdsEntityManager>
#include <QMessageBox>
#include <QDebug>
#include <RdsUtils>
#include <RdsDaemonManager>
#include <RdsFileManager>
#include <RdsClient>
#include <RdsDhcpValues>

#define IP_VALIDATOR QRegExp("^([0-9]{0,3}\\.){0,3}[0-9]{0,3}$")
#define IP_VALIDATOR_STRONG QRegExp("^(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))$")


DhcpEditWidget::DhcpEditWidget(RdsDhcpManager *manager, QWidget* parent)
		: RdsEntityWidget(manager, parent), Ui::DhcpEditWidget()
{
	setupUi(this);
	setApplyButton(ApplyButton);
	setDiscardButton(DiscardButton);

	_manager = manager;
	_configured = true;

	addWidget("defaultleasetime", DefaultLeaseTime);
	addWidget("maxleasetime", MaxLeaseTime);
	addWidget("domainname", DomainName, None, "^[A-Za-z][A-Za-z0-9.\\-]*$", "The domain name is not valid.");
	addWidget("gateway", Gateway, None, IP_VALIDATOR.pattern(), "The gateway must be a valid IP address.");
	addWidget("dnsservers", DnsServers, "listString", SIGNAL(changed()));
	DnsServers->setSeperator(QRegExp(",[ ]*"), ", ");
	DnsServers->setErrorText("You must specify a valid IP address.");
	DnsServers->setWeakValidator(IP_VALIDATOR);
	DnsServers->setStrongValidator(IP_VALIDATOR_STRONG);

	ReturnValue ret = rdsClient()->listInterfaces();
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get a list of network interfaces: " + ret.errString());
		setEnabled(false);
		return;
	}
	addWidget("interfaces", Interfaces, "checked", SIGNAL(changed()));
	Interfaces->setList(ret.toStringList());
	addWidget("enabled", Enabled);
	addWidget("primarydhcp", PrimaryDhcp);

	QObject::connect(manager, SIGNAL(entityRenamed(QString, QString)), this, SLOT(entityRenamed(QString, QString)));
}

DhcpEditWidget::~DhcpEditWidget()
{
}

void DhcpEditWidget::entityRenamed(QString oldid, QString newid)
{
	if (oldid != input()) return;

	QStringList outlist = outputs();
	for (int i = 0; i < outlist.size(); i++)
	{
		if (outlist[i] == input()) outlist[i] = newid;
	}

	setInput(newid);
	setOutputs(outlist);

	//setField("name", newid);
}

ReturnValue DhcpEditWidget::getData()
{
	QVariantMap fields;

	if (input() == "") return(ReturnValue(1, "Invalid Folder"));

	ReturnValue err = true;

	ReturnValue ret = _manager->interfaces();
	if (ret.isError()) err = ret;
	fields["interfaces"] = ret;

	ret = _manager->enabled();
	if (ret.isError()) err = ret;
	bool enabled = ret.toBool();
	ret = _manager->configured();
	if (ret.isError()) err = ret;
	_configured = ret.toBool();
	fields["enabled"] = (enabled && _configured);

	ret = _manager->values();
	if (ret.isError()) err = ret;
	RdsDhcpValues values = ret;

	ret = values.valueExists("authoratative");
	if (ret.isError()) err = ret;
	fields["primarydhcp"] = ret;

	ret = values.value("default-lease-time");
	if (ret.isError()) err = ret;
	fields["defaultleasetime"] = ret;

	ret = values.value("max-lease-time");
	if (ret.isError()) err = ret;
	fields["maxleasetime"] = ret;

	ret = values.option("domain-name");
	if (ret.isError()) err = ret;
	fields["domainname"] = ret.toString().replace("\"", "");

	ret = values.option("domain-name-servers");
	if (ret.isError()) err = ret;
	fields["dnsservers"] = ret;

	ret = values.option("routers");
	if (ret.isError()) err = ret;
	fields["gateway"] = ret;


	if (err.isError())
	{
		QMessageBox msg(QMessageBox::Warning, "Error", "There was an error reading settings: " + ret.errString(), QMessageBox::Ok);
		QAbstractButton *editbutton = msg.addButton("Edit Anyway", QMessageBox::RejectRole);
		msg.exec();
		if (msg.clickedButton() == editbutton)
		{
			if (QMessageBox::warning(this, "Warning", "Editing in this state may be dangerous. Continue?", QMessageBox::Yes | QMessageBox::No, QMessageBox::No) == QMessageBox::Yes)
			{
				return(fields);
			}
		}
		return(err);
	}

	return(fields);
}

#define SET_FIELD(field,function) if (name == field) \
	{ \
		ret = function; \
		if (ret.isError()) err = ret; \
	}

#define SET_VALUE_STRING(field,key) \
	SET_FIELD(field,values.setValue(key, value.toString()));
#define SET_VALUE_STRING_QUOTED(key,function) \
	SET_FIELD(field,values.setValue(key, "\"" + value.toString() + "\""));
#define SET_VALUE_BOOL(key,function) \
	SET_FIELD(field,values.setValue(key, value.toBool()));
#define SET_OPTION_STRING(field,key) \
	SET_FIELD(field,values.setOption(key, value.toString()));
#define SET_OPTION_STRING_QUOTED(key,function) \
	SET_FIELD(field,values.setOption(key, "\"" + value.toString() + "\""));
#define SET_OPTION_BOOL(key,function) \
	SET_FIELD(field,values.setOption(key, value.toBool()));

bool DhcpEditWidget::setData(QString id, QVariantMap fields)
{
	ReturnValue err = true;
	ReturnValue ret;

	ret = _manager->values();
	if (ret.isError()) err = ret;
	RdsDhcpValues values = ret;

	foreach(QString name, fields.keys())
	{
		QVariant value = fields[name];
		
		if(name == "enabled")
		{
			ret = _manager->setEnabled(value.toBool());
			if(ret.isError()) err = ret;
		}
		else if(name == "primarydhcp")
		{
			if(value.toBool())
			{
				ret = values.setValue("authoratative","");
			}
			else
			{
				ret = values.removeValue("authoratative");
			}
			if(ret.isError()) err = ret;
		}
		else if(name == "interfaces")
		{
			ret = _manager->setInterfaces(value.toStringList());
			if(ret.isError()) err = ret;
		}
		
		SET_VALUE_STRING("defaultleasetime", "default-lease-time");
		SET_VALUE_STRING("maxleasetime", "max-lease-time");
		SET_OPTION_STRING("domainname", "domain-name");
		SET_OPTION_STRING("gateway", "routers");
		SET_OPTION_STRING("dnsservers", "domain-name-servers");
	}

	if (err.isError())
	{
		qWarning() << "Failed to save:" << id << err.errString();
		return(false);
	}
	else return(true);
}

void DhcpEditWidget::entityUpdated(QString id)
{
	if (id != input()) return;

	if (unsavedChanges())
	{
		if (QMessageBox::question(this, "DHCP Settings Changed",
		                          "Another user has made changes to the item you are editing. Would you like to overwrite your changes?",
		                          QMessageBox::Yes | QMessageBox::No, QMessageBox::No) == QMessageBox::Yes)
		{
			RdsEntityWidget::entityUpdated(id);
		}
	}
	else
	{
		RdsEntityWidget::entityUpdated(id);
	}
}

bool DhcpEditWidget::endSetData()
{
	ReturnValue ret = _manager->save();
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to save changes: " + ret.errString());
		return(false);
	}
	
	RdsDaemonManager mgr;
	ret = mgr.init();
	
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get RdsDaemonManager service: " + ret.errString());
		return(false);
	}
	
	ret = mgr.restartService("Dhcp");
	
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to restart the DHCP server: " + ret.errString());
		return(false);
	}
	
	return(true);
}

bool DhcpEditWidget::validate()
{
	if(!QRegExp(IP_VALIDATOR_STRONG).exactMatch(Gateway->text()))
	{
		QMessageBox::critical(this, "Error", "The gateway must be a valid IP address");
		return(false);
	}

	return(true);
}

