/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "dhcpmanagerwidget.h"
#include <QMessageBox>
#include <RdsClient>
#include <QDebug>
#include <QInputDialog>
#include <QMenu>
#include <QSettings>
#include <QShortcut>
#include <RdsUtils>
#include <RdsDhcpValues>
#include <RdsDaemonManager>
#include "addsubnetdialog.h"
#include "dhcpeditwidget.h"
#include "dhcpgroupeditwidget.h"
#include "dhcphosteditwidget.h"
#include "dhcpsubneteditwidget.h"
#include "dhcpconfigurewidget.h"
#include "config.h"



DhcpManagerWidget::DhcpManagerWidget(QWidget* parent, Qt::WFlags fl)
		: QWidget(parent, fl), Ui::DhcpManagerWidget()
{
	setupUi(this);
	pushButton_4->setIcon(QPixmap(findRdsIcon("icons/16x16/clear_left.png")));
	AddButton->setIcon(QPixmap(findRdsIcon("icons/32x32/add.png")));
	DeleteButton->setIcon(QPixmap(findRdsIcon("icons/32x32/remove.png")));
	ReloadButton->setIcon(QPixmap(findRdsIcon("icons/32x32/reload.png")));

	ReturnValue ret = rdsClient()->getService("DhcpManager");
	if (ret.isError())
	{
		QMessageBox::critical(this, "Failed to get dhcp manager", ret.errString());
		return;
	}

	_manager = new RdsDhcpManager(this);
	*_manager = ret;

	_model = new RdsDhcpModel(_manager, this);
	QObject::connect(_model, SIGNAL(loaded()), this, SLOT(onModelLoaded()));
	_sort = new RdsEntitySortModel(_model);
	_sort->setSourceModel(_model);
	View->setModel(_sort);
	View->sortByColumn(0, Qt::AscendingOrder);
	_selection = View->selectionModel();
	_model->setupSelection(_sort, View);
	View->header()->resizeSection(0, 200);


	QObject::connect(_model, SIGNAL(inputChanged(QString, QString)), this, SLOT(inputChanged(QString, QString)));
	QObject::connect(_model, SIGNAL(outputsChanged(QStringList)), this, SLOT(outputsChanged(QStringList)));
	QObject::connect(FilterEntry, SIGNAL(textChanged(QString)), _sort, SLOT(setFilterRegExp(QString)));

	//Actions and Menu stuff
	View->setContextMenuPolicy(Qt::CustomContextMenu);
	QObject::connect(View, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(contextMenu(const QPoint&)));

	_newgroupaction = new QAction("New Group", this);
	QObject::connect(_newgroupaction, SIGNAL(triggered()), this, SLOT(addGroup()));
	_newsharednetworkaction = new QAction("New Shared Network", this);
	QObject::connect(_newsharednetworkaction, SIGNAL(triggered()), this, SLOT(addSharedNetwork()));
	_newsubnetaction = new QAction("New Subnet", this);
	QObject::connect(_newsubnetaction, SIGNAL(triggered()), this, SLOT(addSubnet()));
	_newhostaction = new QAction("New Host", this);
	QObject::connect(_newhostaction, SIGNAL(triggered()), this, SLOT(addHost()));
	_removeaction = new QAction("Delete", this);
	QObject::connect(_removeaction, SIGNAL(triggered()), this, SLOT(on_DeleteButton_clicked()));

	QMenu *addmenu = new QMenu(this);
	addmenu->addAction(_newgroupaction);
	addmenu->addAction(_newsharednetworkaction);
	addmenu->addAction(_newsubnetaction);
	addmenu->addAction(_newhostaction);
	AddButton->setMenu(addmenu);

	ret = _manager->configured();
	if (ret.isError())
	{
		qWarning() << "Failed to get configured state:" << ret;
		_configured = true;
	}
	else
	{
		_configured = ret.toBool();
	}
	
	if(_configured)
	{
		DhcpEditWidget *w = new DhcpEditWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput("");
	}
	else
	{
		DhcpConfigureWidget *w = new DhcpConfigureWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput("");
		QObject::connect(w, SIGNAL(reload()), this, SLOT(onConfigured()));
		AddButton->setEnabled(false);
		DeleteButton->setEnabled(false);
		_newgroupaction->setEnabled(false);
		_newhostaction->setEnabled(false);
		_newsharednetworkaction->setEnabled(false);
		_newsubnetaction->setEnabled(false);
		_removeaction->setEnabled(false);
	}
	
	_removeaction->setShortcut(QKeySequence("Del"));
	new QShortcut(QKeySequence("Del"), this, SLOT(on_DeleteButton_clicked()));
}

DhcpManagerWidget::~DhcpManagerWidget()
{
}

void DhcpManagerWidget::onModelLoaded()
{
	RdsEntityModel::Cache *managercache = _model->getCache("root/m");
	if(managercache != NULL)
	{
		View->expand(_sort->mapFromSource(_model->indexFromCache(managercache)));
	}
}


void DhcpManagerWidget::inputChanged(QString id, QString type)
{
	_input = id;
	//qDebug() << "Input Changed:" << id << type;
	if (_edit != NULL)
	{
		if (_edit->unsavedChanges() && _edit->isEnabled())
		{
			if (QMessageBox::question(this, "Unsaved Changes", "The user you were editing has unsaved changes. Would you like to apply them?", QMessageBox::Apply | QMessageBox::Discard, QMessageBox::Discard) == QMessageBox::Apply)
			{
				_edit->apply();
			}
		}

		delete _edit;
		_edit = NULL;
	}

	if (!_configured)
	{
		DhcpConfigureWidget *w = new DhcpConfigureWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput(id);
		QObject::connect(w, SIGNAL(reload()), this, SLOT(onConfigured()));
	}
	else if (type == "manager")
	{
		DhcpEditWidget *w = new DhcpEditWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput(id);
	}
	else if (type == "group")
	{
		DhcpGroupEditWidget *w = new DhcpGroupEditWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput(id);
	}
	else if (type == "sharednetwork")
	{
		DhcpGroupEditWidget *w = new DhcpGroupEditWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput(id);
	}
	else if (type == "host")
	{
		DhcpHostEditWidget *w = new DhcpHostEditWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput(id);
	}
	else if (type == "subnet")
	{
		DhcpSubnetEditWidget *w = new DhcpSubnetEditWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput(id);
	}
	else
	{
		DhcpEditWidget *w = new DhcpEditWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput("");
	}
}

void DhcpManagerWidget::outputsChanged(QStringList outputs)
{
	//qDebug() << "Output Changed" << outputs;
	if (_edit != NULL) _edit->setOutputs(outputs);
}

void DhcpManagerWidget::addGroup()
{
	QString input = _model->input() == "" ? "root/m" : _model->input();
	ReturnValue ret = _manager->values(input);
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get values object: " + ret.errString());
		return;
	}
	RdsDhcpValues values = ret;

	RdsEntityModel::Cache *cache = _model->getCache(input);
	if (cache == NULL)
	{
		QMessageBox::critical(this, "Error", "Invalid parent.");
		return;
	}

	if (cache->type() == "host")
	{
		QMessageBox::critical(this, "Error", "You can not add a group to a host.");
		return;
	}

	bool ok = false;
	QString name = QInputDialog::getText(this, "New Group", "Enter the name for the new group", QLineEdit::Normal, "Group", &ok);

	if (!ok)
	{
		return;
	}

	if (name == "")
	{
		QMessageBox::warning(this, "Error", "You must give the new group a name.");
		return;
	}

	if (name.contains('/'))
	{
		QMessageBox::critical(this, "Error", "The / character is not allowed.");
		return;
	}

	ret = values.addGroup(name);
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to create the group: " + ret.errString());
		return;
	}

	_manager->addEntity(input + "/g" + name);

	save();
}

void DhcpManagerWidget::addSharedNetwork()
{
	QString input = _model->input() == "" ? "root/m" : _model->input();
	ReturnValue ret = _manager->values(input);
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get values object: " + ret.errString());
		return;
	}
	RdsDhcpValues values = ret;

	RdsEntityModel::Cache *cache = _model->getCache(input);
	if (cache == NULL)
	{
		QMessageBox::critical(this, "Error", "Invalid parent.");
		return;
	}

	if (cache->type() == "host")
	{
		QMessageBox::critical(this, "Error", "You can not add a shared network to a host.");
		return;
	}

	bool ok = false;
	QString name = QInputDialog::getText(this, "New Shared Network", "Enter the name for the new shared network", QLineEdit::Normal, "Shared Network", &ok);

	if (!ok)
	{
		return;
	}

	if (name == "")
	{
		QMessageBox::warning(this, "Error", "You must give the new shared network a name.");
		return;
	}

	if (!QRegExp("^[A-Za-z][A-Za-z0-9 \\-_]*$").exactMatch(name))
	{
		QMessageBox::warning(this, "Error", "The name must start with a letter and contain only letters, numbers, spaces, dashes ( - ), and underscores ( _ ).");
		return;
	}

	ret = values.addSharedNetwork(name);
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to crate the shared network: " + ret.errString());
		return;
	}

	_manager->addEntity(input + "/n" + name);

	save();
}

void DhcpManagerWidget::addSubnet()
{
	QString input = _model->input() == "" ? "root/m" : _model->input();
	ReturnValue ret = _manager->values(input);
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get values object: " + ret.errString());
		return;
	}
	RdsDhcpValues values = ret;

	RdsEntityModel::Cache *cache = _model->getCache(input);
	if (cache == NULL)
	{
		QMessageBox::critical(this, "Error", "Invalid parent.");
		return;
	}

	if (cache->type() == "host")
	{
		QMessageBox::critical(this, "Error", "You can not add a subnet to a host.");
		return;
	}
	if(cache->type() == "subnet")
	{
		QMessageBox::critical(this, "Error", "You may not add a subnet to another subnet.");
		return;
	}

	AddSubnetDialog dialog(cache->id(), _manager, this);
	if (!dialog.exec()) return;

	save();
}

void DhcpManagerWidget::addHost()
{
	QString input = _model->input() == "" ? "root/m" : _model->input();
	ReturnValue ret = _manager->values(input);
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get values object: " + ret.errString());
		return;
	}
	RdsDhcpValues values = ret;

	RdsEntityModel::Cache *cache = _model->getCache(input);
	if (cache == NULL)
	{
		QMessageBox::critical(this, "Error", "Invalid parent.");
		return;
	}

	if (cache->type() == "host")
	{
		QMessageBox::critical(this, "Error", "You can not add a host to another host.");
		return;
	}

	bool ok = false;
	QString name = QInputDialog::getText(this, "New Host", "Enter the name for the new host", QLineEdit::Normal, "hostname", &ok);

	if (!ok)
	{
		return;
	}

	if (name == "")
	{
		QMessageBox::warning(this, "Error", "You must give the new host a name.");
		return;
	}

	if (!QRegExp("^[A-Za-z][A-Za-z0-9\\-]*$").exactMatch(name))
	{
		QMessageBox::warning(this, "Error", "The host name must start with a letter and contain only letters, numbers, and the - character.");
		return;
	}

	ret = values.addHost(name);
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to create the host: " + ret.errString());
		return;
	}

	_manager->addEntity(input + "/h" + name);

	save();
}


void DhcpManagerWidget::on_DeleteButton_clicked()
{
	QStringList outputs = _model->outputs();

	if (outputs.size() == 0)
	{
		QMessageBox::information(this, "Error", "You must select something to delete.");
		return;
	}
	else if (outputs.size() == 1)
	{
		RdsEntityModel::Cache *cache = _model->getCache(outputs[0]);
		if (cache == NULL) return;
		
		int result;
		
		if (cache->type() == "group")
		{
			result = QMessageBox::question(this, "Delete", QString("Are you sure you want to delete the group %1?").arg(cache->name()),
						       QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
		}
		else if (cache->type() == "sharednetwork")
		{
			result = QMessageBox::question(this, "Delete", QString("Do you want to delete the shared network %1?").arg(cache->name()),
						       QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
		}
		else if (cache->type() == "subnet")
		{
			result = QMessageBox::question(this, "Delete", QString("Do you want to delete the subnet %1?").arg(cache->name()),
						       QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
		}
		else if (cache->type() == "host")
		{
			result = QMessageBox::question(this, "Delete", QString("Do you want to delete the host %1?").arg(cache->name()),
						       QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
		}
		else return;
		
		if (result != QMessageBox::Yes) return;
	}
	else
	{
		int result = QMessageBox::question(this, "Delete Items", QString("Do you want to delete %1 items?").arg(outputs.count()),
						   QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
						   
						   if (result != QMessageBox::Yes)	return;
	}

	bool dosave = false;

	foreach(QString id, outputs)
	{
		RdsEntityModel::Cache *cache = _model->getCache(id);
		if (cache == NULL) continue;
		if (cache->parent() == NULL) continue;
		if (cache->type() == "manager") continue;

		RdsEntityModel::Cache *parent = cache->parent();
		
		ReturnValue ret = _manager->values(parent->id());
		if (ret.isError())
		{
			QMessageBox::critical(this, "Error", "Failed to get parent for \"" + cache->name() + "\": " + ret.errString());
			continue;
		}

		RdsDhcpValues values = ret;
		if (cache->type() == "group") ret = values.removeGroup(cache->name());
		if (cache->type() == "sharednetwork") ret = values.removeSharedNetwork(cache->name());
		if (cache->type() == "subnet") ret = values.removeSubnet(cache->name());
		if (cache->type() == "host") ret = values.removeHost(cache->name());

		if (ret.isError())
		{
			QMessageBox::critical(this, "Error", "Failed to remove \"" + cache->name() + "\": " + ret.errString());
			continue;
		}

		_manager->entityRemoved(id);

		dosave = true;
	}

	if (dosave) save();
}

void DhcpManagerWidget::contextMenu(const QPoint & pos)
{
	Q_UNUSED(pos);
	QMenu menu;
	menu.addAction(_newgroupaction);
	menu.addAction(_newsharednetworkaction);
	menu.addAction(_newsubnetaction);
	menu.addAction(_newhostaction);
	menu.addAction(_removeaction);

	menu.exec(QCursor::pos());
}

void DhcpManagerWidget::on_ReloadButton_clicked()
{
	_model->reload();
}

bool DhcpManagerWidget::save()
{
	ReturnValue ret = _manager->save();
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to save changes: " + ret.errString());
		return(false);
	}

	RdsDaemonManager mgr;
	ret = mgr.init();

	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get RdsDaemonManager service: " + ret.errString());
		return(false);
	}

	ret = mgr.restartService("Dhcp");

	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to restart the DHCP server: " + ret.errString());
		return(false);
	}

	return(true);
}

void DhcpManagerWidget::onConfigured()
{
	_configured = true;
	_model->reload();
	
	_edit->hide();
	
	AddButton->setEnabled(true);
	DeleteButton->setEnabled(true);
	_newgroupaction->setEnabled(true);
	_newhostaction->setEnabled(true);
	_newsharednetworkaction->setEnabled(true);
	_newsubnetaction->setEnabled(true);
	_removeaction->setEnabled(true);
	
	DhcpEditWidget *w = new DhcpEditWidget(_manager, EditFrame);
	EditLayout->addWidget(w);
	_edit = w;
	w->setInput("");
}

