/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
***************************************************************************/


#include "networksavedialog.h"
#include <RdsClient>
#include <QDebug>
#include <QMessageBox>
#include <RdsLdapSession>
#include <RdsUtils>
#include <QHostAddress>
#include "sleeper.h"
#include "main.h"

NetworkSaveDialog::NetworkSaveDialog(QWidget* parent, Qt::WFlags fl)
		: QDialog(parent, fl), Ui::NetworkSaveDialog(),
		_client(0),
		_provisionator(0),
		_loadingframe(0),
		_timer(0),
		_port(0)
{
	setupUi(this);

	LogView->setVisible(false);
	resize(600, 0);
	_user = ConnectionSettings::user;
	_pass = ConnectionSettings::pass;
	_oldip = ConnectionSettings::host;
	_port = ConnectionSettings::port;
	_service = ConnectionSettings::service;
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-0.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-1.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-2.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-3.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-4.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-5.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-6.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-7.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-8.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-9.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-10.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-11.png"));
	QTimer *animtimer = new QTimer(this);
	QObject::connect(animtimer, SIGNAL(timeout()), this, SLOT(onTimerUpdate()));
	animtimer->start(75);

	_timer = new QTimer(this);
	QObject::connect(_timer, SIGNAL(timeout()), this, SLOT(timeout()));
}

NetworkSaveDialog::~NetworkSaveDialog()
{
	onCompleted(ReturnValue(1, "An unknown error occured"));
}

ReturnValue NetworkSaveDialog::setup(RdsProvisionator* p, QVariantMap settings)
{
	_provisionator = p;
	_settings = settings;

	QObject::connect(_provisionator, SIGNAL(completed(ReturnValue)), this, SLOT(onCompleted(ReturnValue)));
	QObject::connect(_provisionator, SIGNAL(minorProgress(QString)), this, SLOT(onMinorProgress(QString)));
	QObject::connect(_provisionator, SIGNAL(majorProgress(QString)), this, SLOT(onMajorProgress(QString)));

	onMajorProgress("Configuring Network");

	ConnectionSettings::ignoredisconnect = true;

	_provisionator->setNetworkSettings(NULL , "", _settings, true);
	QtRpc::Sleeper::sleep(1);
	rdsClient()->disconnect();
	_client = new RdsClient(this);

	_ip = _settings["ip"].toString();
	_reverted = false;
	_time.start();
	doConnect();

	return(true);
}

void NetworkSaveDialog::onTimerUpdate()
{
	if (_loadingframe == 11) _loadingframe = 0;
	else _loadingframe++;

	AnimLabel->setPixmap(_loadingframes[_loadingframe]);
}

void NetworkSaveDialog::timeout()
{
	onMinorProgress("Connection timed out.");
	doConnect();
}

void NetworkSaveDialog::connected(uint id, ReturnValue ret)
{
	_timer->stop();

	if (ret.isError())
	{
		onMinorProgress("Failed to connect: " + ret.errString());
		doConnect();
	}
	else
	{
		onMinorProgress("Connected to: " + _ip);
		onMajorProgress("Connected To: " + _ip);
		if (_reverted)
		{
			onMinorProgress("Failed to connect to the server!");
			onMinorProgress("The server has reverted to its old network settings.");
			onCompleted(ReturnValue(1, "Failed to connect to the server! The server has reverted to its old network settings."));
			return;
		}

		ret = _client->provisionator();
		if (ret.isError())
		{
			onMinorProgress("Failed to get provision service: " + ret.errString());
			onCompleted(ReturnValue(1, QString("Failed to get provision service: %1, %2").arg(ret.errNumber()).arg(ret.errString())));
			return;
		}

		*_provisionator = ret;

		ret = _provisionator->networkWorked();
		if (ret.isError())
		{
			onMinorProgress("Failed to confirm network settings: " + ret.errString());
			onCompleted(ReturnValue(1, QString("Failed to confirm network settings: %1, %2").arg(ret.errNumber()).arg(ret.errString())));
			return;
		}

		ConnectionSettings::ignoredisconnect = false;
		ret = _provisionator->applyNetworkSettings(_settings);
		if (ret.isError()) onCompleted(ret);
	}
}

void NetworkSaveDialog::doConnect()
{
	if (_reverted)
	{
		if (_time.elapsed() >= 30000)
		{
			onMinorProgress("Failed to connect to the server!");
			onMinorProgress("The server tried to revert to its old network settings, but has failed.");
			onCompleted(ReturnValue(1, "The server tried to revert to its old network settings, but has failed."));
			return;
		}
	}
	else
	{
		if (_time.elapsed() >= 70000)
		{
			onMajorProgress("Reverting Network Settings");
			onMinorProgress("Failed to connect to the server within 60 seconds.");
			onMinorProgress("Trying the old IP address.");
			_ip = _oldip;
			_reverted = true;
			_time.restart();
		}
	}

	onMajorProgress("Connecting To: " + _ip);
	_client->connect(QString("tcp://%1:%2/%3").arg(_ip).arg(_port).arg(_service),
	this, SLOT(connected(uint, ReturnValue)),
	QtRpc::AuthToken(_user, _pass));
	_timer->start(10000);
}

void NetworkSaveDialog::onMajorProgress(const QString& msg)
{
	LogView->append(" - " + msg);
	StatusLabel->setText(msg);
}

void NetworkSaveDialog::onMinorProgress(const QString& msg)
{
	LogView->append("  * " + msg);
}

void NetworkSaveDialog::onCompleted(const QtRpc::ReturnValue& ret)
{
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to set network settings:" + ret.errString() + "\nThis application will now close.");
		exit(1);
		reject(); // REJECTED!
	}
	else
	{
		QMessageBox::information(this, "Success", "The network settings have been changed. You must restart this program to make any further changes.");
		exit(0);
		accept();
	}
}

void NetworkSaveDialog::on_ShowLogCheck_toggled(bool state)
{
	if (!state)
	{
		resize(600, 0);
	}
}

