/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "passwordpolicywidget.h"
#include <RdsClient>
#include <QDebug>
#include <QMessageBox>
#include <RdsLdapSession>
#include <RdsUtils>

inline ReturnValue getValue(QString attr, const LdapResult &result)
{
	if (!result.contains(attr.toLower()) || (result[attr.toLower()].count() == 0))
	{
		return(ReturnValue(1, attr + " attribute does not exist"));
	}

	return(QVariant::fromValue(result[attr.toLower()][0]));
}

PasswordPolicyWidget::PasswordPolicyWidget(QWidget* parent, Qt::WFlags fl)
		: QWidget(parent, fl), Ui::PasswordPolicyWidget()
{
	setupUi(this);

	load();
}

PasswordPolicyWidget::~PasswordPolicyWidget()
{
}

void PasswordPolicyWidget::load()
{
	setEnabled(false);
	ReturnValue ret = rdsLdapSession()->read(this, SLOT(loadReturned(uint, ReturnValue)), RdsUtils::baseDn(),
	                  QStringList() << "maxPwdAge" << "minPwdAge" << "minPwdLength" << "pwdProperties"
	                  << "pwdHistoryLength");
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to load the password policy: " + ret.errString());
	}
}

void PasswordPolicyWidget::loadReturned(uint id, ReturnValue ret)
{
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to load the password policy: " + ret.errString());
		return;
	}

	LdapResult result = ret.value<LdapResult>();

	ret = getValue("minPwdLength", result);
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to load minimum password age: " + ret.errString());
		return;
	}
	PasswordLengthBox->setValue(ret.toInt());

	ret = getValue("pwdHistoryLength", result);
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to load password history length: " + ret.errString());
		return;
	}
	HistoryLengthBox->setValue(ret.toInt());

	ret = getValue("pwdProperties", result);
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to load password complexity value: " + ret.errString());
		return;
	}
	ComplexCheck->setChecked((ret.toInt() != 0));

	ret = getValue("maxPwdAge", result);
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to load maximum password age: " + ret.errString());
		return;
	}
	MaxAgeBox->setValue(abs(ret.toString().toDouble() / (24 * 60 * 60 * 1e7)));

	ret = getValue("minPwdAge", result);
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to load minimum password age: " + ret.errString());
		return;
	}
	MinAgeBox->setValue(abs(ret.toString().toDouble() / (24 * 60 * 60 * 1e7)));

	_maxdays = MaxAgeBox->value();
	_mindays = MinAgeBox->value();
	_minlength = PasswordLengthBox->value();
	_historylength = HistoryLengthBox->value();
	_complex = ComplexCheck->isChecked();

	ApplyButton->setEnabled(false);
	DiscardButton->setEnabled(false);

	setEnabled(true);
}


void PasswordPolicyWidget::save()
{
	setEnabled(false);
	RdsLdapActions actions;
	QString tmp = QString("%1").arg((int)MaxAgeBox->value() * -24 * 60 * 60 * 1e7, 0, 'f');
	tmp = tmp.left(tmp.indexOf('.'));
	actions.add(RdsLdapActions::Replace, "maxPwdAge", tmp);
	tmp = QString("%1").arg((int)MinAgeBox->value() * -24 * 60 * 60 * 1e7, 0, 'f');
	tmp = tmp.left(tmp.indexOf('.'));
	actions.add(RdsLdapActions::Replace, "minPwdAge", tmp);
	actions.add(RdsLdapActions::Replace, "minPwdLength", QString("%1")
	            .arg(PasswordLengthBox->value()));
	actions.add(RdsLdapActions::Replace, "pwdHistoryLength", QString("%1")
	            .arg(HistoryLengthBox->value()));
	actions.add(RdsLdapActions::Replace, "pwdProperties",
	            ComplexCheck->isChecked() ? "1" : "0");

	ReturnValue ret = rdsLdapSession()->modify(this, SLOT(saveReturned(uint, ReturnValue)), RdsUtils::baseDn(),
	                  actions);
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to save the password policy: " + ret.errString());
		setEnabled(true);
	}
}

void PasswordPolicyWidget::saveReturned(uint id, ReturnValue ret)
{
	setEnabled(true);
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to load the password policy: " + ret.errString());
		return;
	}

	load();
}

void PasswordPolicyWidget::on_ApplyButton_clicked()
{
	save();
}

void PasswordPolicyWidget::on_DiscardButton_clicked()
{
	load();
}

void PasswordPolicyWidget::dataChanged()
{
	if ((_maxdays == MaxAgeBox->value()) &&
	        (_mindays == MinAgeBox->value()) &&
	        (_minlength == PasswordLengthBox->value()) &&
	        (_historylength == HistoryLengthBox->value()) &&
	        (_complex == ComplexCheck->isChecked()))
	{
		ApplyButton->setEnabled(false);
		DiscardButton->setEnabled(false);
	}
	else
	{
		ApplyButton->setEnabled(true);
		DiscardButton->setEnabled(true);
	}
}


