/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsdaemonmanager.h"
#include "rdsdaemonmanager_p.h"

#include <QStringList>
#include <QDebug>

#include <RdsEntity>

QTRPC_SERVICEPROXY_PIMPL_IMPLEMENT(RdsDaemonManager);

RdsDaemonContainer::RdsDaemonContainer(const QString &name)
{
	RdsDaemonManagerPrivate::registerDaemon(name.toLower(), this);
}

QHash<QString, RdsDaemonContainer*>& RdsDaemonManagerPrivate::instances()
{
	static QHash<QString, RdsDaemonContainer*> instances;
	return instances;
}

QMutex& RdsDaemonManagerPrivate::mutex()
{
	static QMutex mutex;
	return mutex;
}

RdsDaemonManager::RdsDaemonManager(QObject *parent)
		: RdsEntityManager(parent)
{
}

RdsDaemonManager::RdsDaemonManager(const RdsDaemonManager &other)
		: RdsEntityManager(other)
{
	Q_UNUSED(other);
}

RdsDaemonManager::~RdsDaemonManager()
{
}

RdsDaemonManager& RdsDaemonManager::operator=(const RdsDaemonManager & other)
{
	Q_UNUSED(other);
	return *this;
}

ReturnValue RdsDaemonManager::auth(QtRpc::AuthToken token)
{
	createInternalObject();
	if (token.serverData().contains("authenticated") && (token.serverData().value("authenticated").toBool() == true))
		return(true);
	else
		return(ReturnValue(1, "Not Authenticated"));
}

ReturnValue RdsDaemonManager::listEntities(const QString &dn, bool loadmore) const
{
	QMutexLocker locker(&RdsDaemonManagerPrivate::mutex());
	ReturnValue ret;
	if ((dn == "") || (dn == "root"))
	{
		RdsEntity entity;
		entity.setId("root");
		entity.setType("root");
		entity.setVisible(false);
		entity.setName("");
		entity.setParent("");

		foreach(QString instance, RdsDaemonManagerPrivate::instances().keys())
		{
			ret = listEntities(instance, loadmore);
			if (!ret.isError()) entity.children() << ret.value<RdsEntity>();
		}

		return ReturnValue::fromValue<RdsEntity>(entity);
	}
	else if (RdsDaemonManagerPrivate::instances().contains(dn))
	{
		RdsEntity entity;
		entity.setId(dn);
		entity.setType("daemon");
		entity.setVisible(true);
		entity.setName(dn);
		entity.setParent("root");

		return ReturnValue::fromValue<RdsEntity>(entity);
	}
	return ReturnValue(1, "An entity by that name was not found");
}

void RdsDaemonManagerPrivate::registerDaemon(const QString &name, RdsDaemonContainer* instance)
{
	RdsDaemonManagerPrivate::mutex().lock();
	RdsDaemonManagerPrivate::instances()[name.toLower()] = instance;
	RdsDaemonManagerPrivate::mutex().unlock();
}

ReturnValue RdsDaemonManager::list() const
{
	QMutexLocker locker(&RdsDaemonManagerPrivate::mutex());
	return QStringList(RdsDaemonManagerPrivate::instances().keys());
}

ReturnValue RdsDaemonManager::stopService(const QString &name)
{
	QMutexLocker locker(&RdsDaemonManagerPrivate::mutex());
	if (RdsDaemonManagerPrivate::instances().contains(name.toLower()))
		return RdsDaemonManagerPrivate::instances().value(name.toLower())->getInstance()->stopService();
	else
		return ReturnValue(1, "Failed to find a daemon instance by that name");
}

ReturnValue RdsDaemonManager::startService(const QString &name)
{
	QMutexLocker locker(&RdsDaemonManagerPrivate::mutex());
	if (RdsDaemonManagerPrivate::instances().contains(name.toLower()))
		return RdsDaemonManagerPrivate::instances().value(name.toLower())->getInstance()->startService();
	else
		return ReturnValue(1, "Failed to find a daemon instance by that name");
}

ReturnValue RdsDaemonManager::restartService(const QString &name)
{
	QMutexLocker locker(&RdsDaemonManagerPrivate::mutex());
	if (RdsDaemonManagerPrivate::instances().contains(name.toLower()))
		return RdsDaemonManagerPrivate::instances().value(name.toLower())->getInstance()->restartService();
	else
		return ReturnValue(1, "Failed to find a daemon instance by that name");
}

ReturnValue RdsDaemonManager::reloadService(const QString &name)
{
	QMutexLocker locker(&RdsDaemonManagerPrivate::mutex());
	if (RdsDaemonManagerPrivate::instances().contains(name.toLower()))
		return RdsDaemonManagerPrivate::instances().value(name.toLower())->getInstance()->reloadService();
	else
		return ReturnValue(1, "Failed to find a daemon instance by that name");
}

ReturnValue RdsDaemonManager::reloadConfig(const QString &name)
{
	QMutexLocker locker(&RdsDaemonManagerPrivate::mutex());
	if (RdsDaemonManagerPrivate::instances().contains(name.toLower()))
		return RdsDaemonManagerPrivate::instances().value(name.toLower())->getInstance()->reloadConfig();
	else
		return ReturnValue(1, "Failed to find a daemon instance by that name");
}

