/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "config.h"
#include "rdsdhcpmanager.h"
#include "rdsdhcpmanager_p.h"
#include "rdsdhcpvalues_p.h"
#include <RdsDhcpSubnet>
#include <RdsDhcpSharedNetwork>
#include <RdsDhcpGroup>
#include <RdsDhcpHost>
#include <QStringList>
#include <QDebug>
#include <QMutexLocker>
#include <QFile>
#include <QTextStream>
#include <QTemporaryFile>
#include <QProcess>

#include <RdsEntity>
#include <RdsDhcpValues>
#include <RdsUtils>
#include <RdsDaemonManager>

#define RDS_LOCK QMutexLocker __locker(&RdsDhcpManagerPrivate::mutex());

QTRPC_SERVICEPROXY_PIMPL_IMPLEMENT(RdsDhcpManager);
RDS_REGISTER_DAEMON(RdsDhcpManager, "Dhcp");


QMutex &RdsDhcpManagerPrivate::mutex()
{
	static QMutex mutex(QMutex::Recursive);;
	return mutex;
}

RdsDhcpManagerPrivate::DHCPValues*& RdsDhcpManagerPrivate::values()
{
	static DHCPValues* values(0);
	return values;
}

QList<QHostAddress> &RdsDhcpManagerPrivate::excluded()
{
	static QList<QHostAddress> excluded;
	return excluded;
}

RdsDhcpManager::RdsDhcpManager()
		: RdsEntityManager(),
		RdsDaemonInstance()
{
	if (!qxt_d().values())
		reload();
}

RdsDhcpManager::RdsDhcpManager(QObject *parent)
		: RdsEntityManager(parent),
		RdsDaemonInstance()
{
	if (!qxt_d().values())
		reload();
}

RdsDhcpManager::RdsDhcpManager(const RdsDhcpManager& other)
		: RdsEntityManager(other),
		RdsDaemonInstance()
{
}

RdsDhcpManager::~RdsDhcpManager()
{
}

ReturnValue RdsDhcpManager::stopService()
{
	return RdsUtils::runCommand(RDS_INITD_DHCP, QStringList() << "stop");
}

ReturnValue RdsDhcpManager::startService()
{
	return RdsUtils::runCommand(RDS_INITD_DHCP, QStringList() << "start");
}

ReturnValue RdsDhcpManager::restartService()
{
	return RdsUtils::runCommand(RDS_INITD_DHCP, QStringList() << "restart");
}

ReturnValue RdsDhcpManager::reloadService()
{
	return RdsUtils::runCommand(RDS_INITD_DHCP, QStringList() << "restart");
}

ReturnValue RdsDhcpManager::reloadConfig()
{
	return reload();
}

QString RdsDhcpManager::toString() const
{
	return qxt_d().values()->getOutput().toString();
}

ReturnValue RdsDhcpManager::save() const
{
	RDS_LOCK;
	ReturnValue ret = qxt_d().values()->getOutput();
	if (ret.isError())
		return ret;

	//We put temporary files in /var/log/ because DHCPDs apparmor profile can read from there
	QTemporaryFile file("/var/log/rds-dns-tmp");
	file.open();
	file.write(ret.toString().toLocal8Bit());
	file.setPermissions(QFile::ReadOwner | QFile::WriteOwner | QFile::ReadGroup | QFile::ReadOther);
	file.flush();
	QProcess proc;
	proc.setProcessChannelMode(QProcess::MergedChannels);
	proc.start("dhcpd3", QStringList() << "-t" << "-cf" << file.fileName());
	proc.waitForFinished();
	if (proc.exitCode() != 0)
	{
		qWarning() << "Failed to validate dhcp.conf file:" << proc.readAll();
		qWarning() << "See /tmp/baddhcpd.conf for more details";
		QProcess::execute("cp", QStringList() << file.fileName() << "/tmp/baddhcpd.conf");
		return ReturnValue(1, "Wrote out an invalid configuration file. Please check for any abnormalities in the data.");
	}

	QProcess::execute("cp", QStringList() << file.fileName() << RDS_DHCP_CONFIGFILE);

	QFile(RDS_DHCP_CONFIGFILE).setPermissions(QFile::ReadOwner | QFile::WriteOwner | QFile::ReadGroup | QFile::ReadOther);
	return true;
}

ReturnValue RdsDhcpManager::reload()
{
	RDS_LOCK;

	qRegisterMetaType<QList<QPair<QHostAddress, QHostAddress> > >("QList<QPair<QHostAddress, QHostAddress> >");
	qRegisterMetaTypeStreamOperators<QList<QPair<QHostAddress, QHostAddress> > >("QList<QPair<QHostAddress, QHostAddress> >");

	QFile(RDS_DHCP_CONFIGFILE).setPermissions(QFile::ReadOwner | QFile::WriteOwner | QFile::ReadGroup | QFile::ReadOther);
	QProcess proc;
	proc.setProcessChannelMode(QProcess::ForwardedChannels);

	if (!qxt_d().values())
		qxt_d().values() = new RdsDhcpManagerPrivate::DHCPValues();

	proc.start("dhcpd3", QStringList() << "-t" << "-cf" << RDS_DHCP_CONFIGFILE);
	proc.waitForFinished();
	if (proc.exitCode() != 0)
	{
		return ReturnValue(1, "Invalid configuration file, not reloading.");
	}
	QFile file(RDS_DHCP_CONFIGFILE);
	if (!file.open(QFile::ReadOnly))
		return ReturnValue(1, "Failed to open DHCP file");
	if (!parse(file.readAll()))
		return ReturnValue(1, "Failed to parse the dhcpd.conf file");
	return true;
}

ReturnValue RdsDhcpManager::values() const
{
	RdsDhcpValues* values = new RdsDhcpValues();
	values->setData(qxt_d().values());
	return values;
}

bool RdsDhcpManagerPrivate::insertValue(const QString &line, const QStringList &breadcrumb, const QString &key, const QString &value, DHCPValues* data)
{
	RDS_LOCK;
	if (!data)
		data = values();
	if (key == "empty-shared-network")
	{
		if (!line.isEmpty())
			return true;
		QString snName = value;
		if (snName.startsWith('"') && snName.endsWith('"'))
			snName = snName.mid(1, snName.count() - 2);

		SharedNetwork* nw = data->sharedNetworks.value(snName, 0);
		if (!nw)
		{
			nw = new SharedNetwork();
			nw->parent = data;
			nw->name = snName;
			data->sharedNetworks[snName] = nw;
		}
		return true;
	}
	if (breadcrumb.count() == 0)
	{
		if (!key.isEmpty() && !line.isEmpty())
		{
			if (key == "option")
			{
				QString optionKey = value.left(value.indexOf(" "));
				QString optionValue = value.mid(value.indexOf(" ") + 1);
				data->options.insertMulti(optionKey, optionValue);
			}
			else
			{
				data->values.insertMulti(key, value);
			}
		}
		return true;
	}
	if (breadcrumb.at(0) == "subnet")
	{
		if (breadcrumb.count() < 2)
			return false;
		Subnet* subnet;
		if (data->subnetList.count() < breadcrumb.at(1).toInt())
		{
			qCritical() << "Attempted to fetch a Subnet out of range";
			return false;
		}
		if (data->subnetList.count() < (breadcrumb.at(1).toInt() + 1))
		{
			subnet = new Subnet();
			subnet->parent = data;
			data->subnetList << subnet;
//			qDebug() << "__SUBNET_INIT Created" << subnet;
		}
		else
			subnet = data->subnetList.at(breadcrumb.at(1).toInt());

		if (breadcrumb.count() > 2)
		{
			QStringList nbc = breadcrumb;
			nbc.removeFirst();
			nbc.removeFirst();
			return insertValue(line, nbc, key, value, subnet);
		}
		if (key == "name")
		{
			if (!line.isEmpty())
				return true;
			subnet->name = value;
			return true;
		}
		else if (key == "__SUBNET_INIT")
		{
//			qDebug() << "__SUBNET_INIT for" << subnet;
			QStringList snvalues = value.split(" ", QString::SkipEmptyParts);
			if (snvalues.count() != 3)
				return false;
			QHostAddress address(snvalues.at(0).simplified());
			QHostAddress netmask(snvalues.at(2).simplified());
//			qDebug() << "__SUBNET_INIT__SUBNET_INIT__SUBNET_INIT" << snvalues << address.toString() << netmask.toString();
			subnet->address = address;
			subnet->netmask = netmask;
			return true;
		}
		else if (key == "range")
		{
			if (!line.isEmpty())
				return true;
			//qDebug() << value;
			subnet->addRange(QHostAddress(value.split(" ", QString::SkipEmptyParts).first()), QHostAddress(value.split(" ", QString::SkipEmptyParts).last()));
			return true;
		}
		else if (key == "hidden")
		{
// 			qDebug() << "Attempting to delete this subnet:" << breadcrumb;
			subnet->name = "__RDS_DELETE_ME";
			return true;
		}
		else
		{
			QStringList nbc = breadcrumb;
			nbc.removeFirst();
			nbc.removeFirst();
			return insertValue(line, nbc, key, value, subnet);
		}
	}
	else if (breadcrumb.at(0) == "shared-network")
	{
		if (breadcrumb.count() < 2)
			return false;
		QString snName = breadcrumb.at(1);
		if (snName.startsWith('"') && snName.endsWith('"'))
			snName = snName.mid(1, snName.count() - 2);

		SharedNetwork* nw = data->sharedNetworks.value(snName, 0);
		if (!nw)
		{
			nw = new SharedNetwork();
			nw->parent = data;
			nw->name = snName;
			data->sharedNetworks[snName] = nw;
		}
		QStringList nbc = breadcrumb;
		nbc.removeFirst();
		nbc.removeFirst();
		return insertValue(line, nbc, key, value, nw);
	}
	else if (breadcrumb.at(0) == "group")
	{
		if (breadcrumb.count() < 2)
			return false;
		Group *group;
		QString snName = breadcrumb.at(1);
		if (snName.startsWith('"') && snName.endsWith('"'))
			snName = snName.mid(1, snName.count() - 2);

		if (data->groupList.count() < snName.toInt())
		{
			qCritical() << "Attempted to fetch a Group out of range";
			return false;
		}
		if (data->groupList.count() < (snName.toInt() + 1))
		{
			group = new Group();
			group->parent = data;
			data->groupList << group;
		}
		else
			group = data->groupList.at(snName.toInt());
		if (breadcrumb.count() > 2)
		{
			QStringList nbc = breadcrumb;
			nbc.removeFirst();
			nbc.removeFirst();
			return insertValue(line, nbc, key, value, group);
		}
		if (key == "name")
		{
			if (!line.isEmpty())
				return true;
			group->name = value;
			return true;
		}
		QStringList nbc = breadcrumb;
		nbc.removeFirst();
		nbc.removeFirst();
		return insertValue(line, nbc, key, value, group);
	}
	else if (breadcrumb.at(0) == "host")
	{
		if (breadcrumb.count() < 2)
			return false;
		if (key == "fixed-address")
			RdsDhcpManagerPrivate::excluded() << QHostAddress(value);
		QString snName = breadcrumb.at(1);
		if (snName.startsWith('"') && snName.endsWith('"'))
			snName = snName.mid(1, snName.count() - 2);

		Host* host = data->hosts.value(snName, 0);
		if (!host)
		{
			host = new Host();
			host->name = snName;
			host->parent = data;
			data->hosts[snName] = host;
		}
		QStringList nbc = breadcrumb;
		nbc.removeFirst();
		nbc.removeFirst();
		return insertValue(line, nbc, key, value, host);
	}
//	qDebug() << "Request to insert the following unhandled... KEY:" << key << "VALUE:" << value << "BREADCRUMB" << breadcrumb;
	if (key.isEmpty())
		data->otherValues = (data->otherValues + value + line).trimmed();
	else
		data->otherValues = (data->otherValues + "\n" + QString(breadcrumb.count() - 1, '\t') + line + ";").trimmed();
	return false;
}

ReturnValue RdsDhcpManager::auth(QtRpc::AuthToken token)
{
	createInternalObject();
	if (token.serverData().contains("authenticated") && (token.serverData().value("authenticated").toBool() == true))
		return(true);
	else
		return(ReturnValue(1, "Not Authenticated"));
}

bool RdsDhcpManager::parse(const QString &data)
{
	RDS_LOCK;
	if (!qxt_d().values())
		qxt_d().values() = new RdsDhcpManagerPrivate::DHCPValues();
	RdsDhcpManagerPrivate::DHCPValues tmpValues;
	QStringList comments;
	QStringList breadcrumb;
	bool endOfLine = false;
	quint32 wordNum = 1;
	quint32 lineNumber = 1;
	quint32 brackets = 0;
	QHash<QString, quint32> groupnum;
	QHash<QString, quint32> subnetnum;
	QString key;
	QString value;
	QString word;
	QString line;

	for (QString::const_iterator i = data.begin(); i != data.end(); ++i)
	{
		if (*i == '\n')
			lineNumber++;
		if (endOfLine)
		{
			if (*i == '\n')
			{
				if (comments.at(comments.count() - 1).startsWith("(RDS)"))
				{
					QString comment = comments.at(comments.count() - 1);
					comment = comment.mid(5).simplified();
					if (!comment.isEmpty())
					{
						QStringList parts = comment.split(" ", QString::SkipEmptyParts);
						QString rdscommentkey = parts.takeFirst();
						QString rdscommentvalue = parts.join(" ");
						qxt_d().insertValue("", breadcrumb, rdscommentkey, rdscommentvalue, &tmpValues);
						if (rdscommentkey == "hidden")
						{
							QStringList breadcrumbCopy = breadcrumb;
							breadcrumbCopy.removeLast();
							subnetnum[breadcrumbCopy.join("__RDSDELIM__")]--;
						}
					}
					comments.removeLast();
				}
				endOfLine = false;
			}
			else
				comments[comments.count() - 1] += *i;
			continue;
		}
		if (*i == ';' || *i == ' ' || *i == '\t' || *i == '\n')
		{
			if (!word.isEmpty())
			{
				line += word + " ";
			}
			if (!word.isEmpty())
			{
				switch (wordNum)
				{
					case 1:
						key = word;
						breadcrumb << word;
						break;
					default:
					{
						value = (value + " " + word).simplified();
						break;
					}
				}
				// parse the words
				word = QString::Null();
				wordNum++;
			}
			if (!line.isEmpty())
			{
				switch (i->toAscii())
				{
					case ';':
					{
						breadcrumb.removeLast();
						qxt_d().insertValue(line, breadcrumb, key, value, &tmpValues);
						line = QString::Null();
						value = QString::Null();
						comments = QStringList();
						wordNum = 1;
					}
					break;
					case '\n':
					{
					}
					break;
				}
			}
			continue;
		}
		if (*i == '#')
		{
			comments << QString::Null();
			endOfLine = true;
			continue;
		}
		if (*i == '{')
		{
			brackets++;
			if (key == "group")
			{
				if (!groupnum.contains(breadcrumb.join("__RDSDELIM__")))
					groupnum[breadcrumb.join("__RDSDELIM__")] = 0;
				breadcrumb << QString::number(groupnum[breadcrumb.join("__RDSDELIM__")]++);
			}
			else if (key == "subnet")
			{
				if (!subnetnum.contains(breadcrumb.join("__RDSDELIM__")))
					subnetnum[breadcrumb.join("__RDSDELIM__")] = 0;
				breadcrumb << QString::number(subnetnum[breadcrumb.join("__RDSDELIM__")]++);
			}
			else
				breadcrumb << value;
			if (key == "subnet")
				qxt_d().insertValue(line + word + "{\n", breadcrumb, "__SUBNET_INIT", value, &tmpValues); //special case...
			else
				qxt_d().insertValue(line + word + "{\n", breadcrumb, QString(), " ", &tmpValues);
			line = QString::Null();
			word = QString::Null();
			value = QString::Null();
			comments = QStringList();
			wordNum = 1;
			continue;
		}
		if (*i == '}')
		{
			brackets--;
			qxt_d().insertValue(line + "} ", breadcrumb, QString(), "\n", &tmpValues);
			breadcrumb.removeLast();
			breadcrumb.removeLast();
			continue;
		}
		if (*i == ';' || *i == '\n')
		{
			continue;
		}
		if (*i == '\\')
		{
			i++;
		}
		word += *i;
	}
	tmpValues.initializeValues();
	*(qxt_d().values()) = tmpValues;
	return true;
}

ReturnValue RdsDhcpManager::listEntities(const QString &dn, bool loadmore) const
{
	RDS_LOCK;

	ReturnValue ret;
	if ((dn == "") || (dn == "root"))
	{
		RdsEntity entity;
		entity.setId("root");
		entity.setType("root");
		entity.setVisible(false);
		entity.setName("");
		entity.setParent("");

		ret = listEntities("root/m", loadmore);
		if (!ret.isError()) entity.children() << ret.value<RdsEntity>();
		return ReturnValue::fromValue<RdsEntity>(entity);
	}
	else if (dn == "root/m")
	{
		return ReturnValue::fromValue<RdsEntity>(qxt_d().values()->listEntities(dn));
	}
	else if (dn.startsWith("root/m/"))
	{
		ReturnValue ret = values(dn);
		if (ret.isError())
			return ret;
		RdsDhcpValues v = ret;
		if (v.qxt_d().values)
			return ReturnValue::fromValue<RdsEntity>(v.qxt_d().values->listEntities(dn));
		else
			return ReturnValue(1, "Invalid DHCP values object");
	}
	return ReturnValue(1, "Entity not found");
}

ReturnValue RdsDhcpManager::values(const QString& dn) const
{
	RDS_LOCK;

	if (dn == "root/m")
		return values();
	if (!dn.startsWith("root/m/"))
		return ReturnValue(1, "Invalid values object, you must only use ID's fetched from listEntities");

	QStringList path = dn.split("/", QString::SkipEmptyParts);
	path.removeFirst();
	path.removeFirst();
	ReturnValue ret;
	ret = values();
	if (ret.isError())
		return ret;
	RdsDhcpValues values = ret;
	foreach(QString pathSegment, path)
	{
		switch (pathSegment.at(0).toAscii())
		{
			case 's': //Subnet
			{
				QString index = pathSegment.mid(1);
				ret = values.subnet(index);
				if (ret.isError())
					return ret;
				values = ret;
				break;
			}
			case 'g': //Group
			{
				QString index = pathSegment.mid(1);
				ret = values.group(index);
				if (ret.isError())
					return ret;
				values = ret;
				break;
			}
			case 'n': //sharedNetwork
			{
				QString index = pathSegment.mid(1);
				ret = values.sharedNetwork(index);
				if (ret.isError())
					return ret;
				values = ret;
				break;
			}
			case 'h': //host
			{
				QString index = pathSegment.mid(1);
				ret = values.host(index);
				if (ret.isError())
					return ret;
				values = ret;
				break;
			}
		}
	}
	return ret;
}

ReturnValue RdsDhcpManager::move(const QString& path, const QString& newpath)
{
	RDS_LOCK;
	ReturnValue ret;

	if (!this->values(newpath).isError())
		return ReturnValue(1, "Destination object already exists");

	QString name = newpath.mid(newpath.lastIndexOf("/") + 2);
	if (name.isEmpty())
		return ReturnValue(1, "DHCP object must have a name");

	if (path.at(path.lastIndexOf("/") + 1) != newpath.at(newpath.lastIndexOf("/") + 1))
		return ReturnValue(1, "You cannot change the type of the DHCP values object");

	if (newpath.left(newpath.lastIndexOf("/") + 1).startsWith(path))
		return ReturnValue(1, "You cannot move an object into itself");

	ret = this->values(path);
	if (ret.isError())
		return ret;
	RdsDhcpValues values = ret;

	ret = this->values(path.left(path.lastIndexOf("/")));
	if (ret.isError())
		return ret;
	RdsDhcpValues parent = ret;

	ret = this->values(newpath.left(newpath.lastIndexOf("/")));
	if (ret.isError())
		return ret;
	RdsDhcpValues newParent = ret;

	if (values.qxt_d().values == 0 ||
	        parent.qxt_d().values == 0 ||
	        newParent.qxt_d().values == 0)
		return ReturnValue(1, "Received an invalid DHCP values object");

	QString type = values.qxt_d().values->type();
	if (type == "root")
	{
		return ReturnValue(1, "Cannot rename the root values object");
	}
	else if (type == "subnet")
	{
		RdsDhcpManagerPrivate::Subnet* v = static_cast<RdsDhcpManagerPrivate::Subnet*>(values.qxt_d().values);
		if (v->parent)
			v->parent->subnets.remove(v->name);
		v->parent = newParent.qxt_d().values;
		newParent.qxt_d().values->subnets.insert(name, v);
		v->name = name;
		return this->values(newpath);
	}
	else if (type == "sharednetwork")
	{
		RdsDhcpManagerPrivate::SharedNetwork* v = static_cast<RdsDhcpManagerPrivate::SharedNetwork*>(values.qxt_d().values);
		if (v->parent)
			v->parent->sharedNetworks.remove(v->name);
		v->parent = newParent.qxt_d().values;
		newParent.qxt_d().values->sharedNetworks.insert(name, v);
		v->name = name;
		return this->values(newpath);
	}
	else if (type == "group")
	{
		RdsDhcpManagerPrivate::Group* v = static_cast<RdsDhcpManagerPrivate::Group*>(values.qxt_d().values);
		if (v->parent)
			v->parent->groups.remove(v->name);
		v->parent = newParent.qxt_d().values;
		newParent.qxt_d().values->groups.insert(name, v);
		v->name = name;
		return this->values(newpath);
	}
	else if (type == "host")
	{
		RdsDhcpManagerPrivate::Host* v = static_cast<RdsDhcpManagerPrivate::Host*>(values.qxt_d().values);
		if (v->parent)
			v->parent->hosts.remove(v->name);
		v->parent = newParent.qxt_d().values;
		newParent.qxt_d().values->hosts.insert(name, v);
		v->name = name;
		return this->values(newpath);
	}

	return ReturnValue(1, "Received a DHCP values object of an unknown type: " + type);
}

RdsEntity RdsDhcpManagerPrivate::DHCPValues::listEntities(const QString &base) const
{
	RdsEntity entity;
	entity.setId(base);
	entity.setType("manager");
	entity.setVisible(true);
	entity.setName("DHCP Server");
	entity.setParent(base.left(base.lastIndexOf("/")));

	ReturnValue ret;
	for (QHash<QString, Subnet*>::const_iterator i = subnets.begin(); i != subnets.end(); ++i)
	{
		entity.children() << i.value()->listEntities(base + "/s" + i.key());
	}
	for (QHash<QString, Group*>::const_iterator i = groups.begin(); i != groups.end(); ++i)
	{
		entity.children() << i.value()->listEntities(base + "/g" + i.key());
	}
	for (QHash<QString, SharedNetwork*>::const_iterator i = sharedNetworks.begin(); i != sharedNetworks.end(); ++i)
	{
		entity.children() << i.value()->listEntities(base + "/n" + i.key());
	}
	for (QHash<QString, Host*>::const_iterator i = hosts.begin(); i != hosts.end(); ++i)
	{
		entity.children() << i.value()->listEntities(base + "/h" + i.key());
	}
	return entity;
}

RdsEntity RdsDhcpManagerPrivate::Subnet::listEntities(const QString &base) const
{
	RdsEntity entity = DHCPValues::listEntities(base);
	entity.setType("subnet");
	entity.setName(name);
	return entity;
}

RdsEntity RdsDhcpManagerPrivate::SharedNetwork::listEntities(const QString &base) const
{
	RdsEntity entity = DHCPValues::listEntities(base);
	entity.setType("sharednetwork");
	entity.setName(name);
	return entity;
}

RdsEntity RdsDhcpManagerPrivate::Group::listEntities(const QString &base) const
{
	RdsEntity entity = DHCPValues::listEntities(base);
	entity.setType("group");
	entity.setName(name);
	return entity;
}

RdsEntity RdsDhcpManagerPrivate::Host::listEntities(const QString &base) const
{
	RdsEntity entity = DHCPValues::listEntities(base);
	entity.setType("host");
	entity.setName(name);
	return entity;
}

QString RdsDhcpManagerPrivate::DHCPValues::type() const
{
	return "root";
}

QString RdsDhcpManagerPrivate::Subnet::type() const
{
	return "subnet";
}

QString RdsDhcpManagerPrivate::SharedNetwork::type() const
{
	return "sharednetwork";
}

QString RdsDhcpManagerPrivate::Group::type() const
{
	return "group";
}

QString RdsDhcpManagerPrivate::Host::type() const
{
	return "host";
}

RdsDhcpManager& RdsDhcpManager::operator=(const RdsDhcpManager & other)
{
	Q_UNUSED(other);
	return *this;
}

RdsDhcpManagerPrivate::DHCPValues& RdsDhcpManagerPrivate::DHCPValues::operator=(const DHCPValues & other)
{
	values = other.values;
	options = other.options;
	otherValues = other.otherValues;
	for (QHash<QString, Subnet*>::const_iterator i = other.subnets.begin(); i != other.subnets.end(); ++i)
	{
		if (!subnets.contains(i.key()))
		{
			subnets[i.key()] = new Subnet();
			subnets[i.key()]->parent = this;
		}
		*(subnets[i.key()]) = *(i.value());
	}
	for (QHash<QString, Group*>::const_iterator i = other.groups.begin(); i != other.groups.end(); ++i)
	{
		if (!groups.contains(i.key()))
		{
			groups[i.key()] = new Group();
			groups[i.key()]->parent = this;
		}
		*(groups[i.key()]) = *(i.value());
	}
	for (QHash<QString, SharedNetwork*>::const_iterator i = other.sharedNetworks.begin(); i != other.sharedNetworks.end(); ++i)
	{
		if (!sharedNetworks.contains(i.key()))
		{
			sharedNetworks[i.key()] = new SharedNetwork();
			sharedNetworks[i.key()]->parent = this;
		}
		*(sharedNetworks[i.key()]) = *(i.value());
	}
	for (QHash<QString, Host*>::const_iterator i = other.hosts.begin(); i != other.hosts.end(); ++i)
	{
		if (!hosts.contains(i.key()))
		{
			hosts[i.key()] = new Host();
			hosts[i.key()]->parent = this;
		}
		*(hosts[i.key()]) = *(i.value());
	}
	// remove ones that no longer exist!

	for (QHash<QString, Subnet*>::iterator i = subnets.begin(); i != subnets.end();)
	{
		if (!other.subnets.contains(i.key()))
		{
			delete i.value();
			i.value() = 0;
			i = subnets.erase(i);
		}
		else
		{
			++i;
		}
	}
	for (QHash<QString, Group*>::iterator i = groups.begin(); i != groups.end();)
	{
		if (!other.groups.contains(i.key()))
		{
			delete i.value();
			i.value() = 0;
			i = groups.erase(i);
		}
		else
		{
			++i;
		}
	}
	for (QHash<QString, SharedNetwork*>::iterator i = sharedNetworks.begin(); i != sharedNetworks.end();)
	{
		if (!other.sharedNetworks.contains(i.key()))
		{
			delete i.value();
			i.value() = 0;
			i = sharedNetworks.erase(i);
		}
		else
		{
			++i;
		}
	}
	for (QHash<QString, Host*>::iterator i = hosts.begin(); i != hosts.end();)
	{
		if (!other.hosts.contains(i.key()))
		{
			delete i.value();
			i.value() = 0;
			i = hosts.erase(i);
		}
		else
		{
			++i;
		}
	}
	subnetList = subnets.values();
	groupList = groups.values();
	return *this;
}

RdsDhcpManagerPrivate::Subnet& RdsDhcpManagerPrivate::Subnet::operator=(const Subnet & other)
{
	DHCPValues::operator=(other);
	address = other.address;
	netmask = other.netmask;
	addresses = other.addresses;
	name = other.name;
	return *this;
}

RdsDhcpManagerPrivate::Group& RdsDhcpManagerPrivate::Group::operator=(const Group & other)
{
	DHCPValues::operator=(other);
	name = other.name;
	return *this;
}

RdsDhcpManagerPrivate::SharedNetwork& RdsDhcpManagerPrivate::SharedNetwork::operator=(const SharedNetwork & other)
{
	DHCPValues::operator=(other);
	name = other.name;
	return *this;
}

RdsDhcpManagerPrivate::Host& RdsDhcpManagerPrivate::Host::operator=(const Host & other)
{
	DHCPValues::operator=(other);
	name = other.name;
	return *this;
}

ReturnValue RdsDhcpManagerPrivate::DHCPValues::getOutput(int indentation) const
{
	RDS_LOCK;
	ReturnValue ret;
	QString output;
	for (QVariantMap::const_iterator i = values.begin(); i != values.end(); ++i)
	{
		output += QString(indentation, '\t') + i.key() + ' ' + i.value().toString() + ";\n";
	}
	for (QVariantMap::const_iterator i = options.begin(); i != options.end(); ++i)
	{
		output += QString(indentation, '\t') + "option " + i.key() + ' ' + i.value().toString() + ";\n";
	}
	if (otherValues.count() > 0)
	{
		output += "\n" + QString(indentation, '\t') + "# Literal output taken from the config file. These are the values that RDS doesn't understand\n" + otherValues;
	}

	if (subnets.count() > 0)
	{
		output += "\n" + QString(indentation, '\t') + "# Subnets\n";
		for (QHash<QString, Subnet*>::const_iterator i = subnets.begin(); i != subnets.end(); ++i)
		{
			ret = i.value()->getOutput(indentation + 1);
			if (ret.isError())
				return ret;
			output += ret.toString();
		}
	}
	if (groups.count() > 0)
	{
		output += "\n" + QString(indentation, '\t') + "# Groups\n";
		for (QHash<QString, Group*>::const_iterator i = groups.begin(); i != groups.end(); ++i)
		{
			ret = i.value()->getOutput(indentation + 1);
			if (ret.isError())
				return ret;
			output += ret.toString();
		}
	}
	if (sharedNetworks.count() > 0)
	{
		output += "\n" + QString(indentation, '\t') + "# SharedNetwork\n";
		for (QHash<QString, SharedNetwork*>::const_iterator i = sharedNetworks.begin(); i != sharedNetworks.end(); ++i)
		{
			ret = i.value()->getOutput(indentation + 1);
			if (ret.isError())
				return ret;
			output += ret.toString();
		}
	}
	if (hosts.count() > 0)
	{
		output += "\n" + QString(indentation, '\t') + "# Host\n";
		for (QHash<QString, Host*>::const_iterator i = hosts.begin(); i != hosts.end(); ++i)
		{
			ret = i.value()->getOutput(indentation + 1);
			if (ret.isError())
				return ret;
			output += ret.toString();
		}
	}
	output += "\n";
	return output;
}

ReturnValue RdsDhcpManagerPrivate::Subnet::getOutput(int indentation) const
{
	RDS_LOCK;
	ReturnValue ret;
	QString output;
	output += QString(indentation - 1, '\t') + "subnet " + address.toString() + " netmask " + netmask.toString() + " {\n";
	output += QString(indentation, '\t') + "#(RDS)name " + name + "\n";
	ret = DHCPValues::getOutput(indentation);
	if (ret.isError())
		return ret;
	output += ret.toString();
	// Put the shit here!
	typedef QPair<QHostAddress, QHostAddress> RangePair;
	QList<RangePair> ranges = getFullRanges();
	foreach(RangePair range, ranges)
	{
		output += QString(indentation, '\t') + "#(RDS)range " + range.first.toString() + " " + range.second.toString() + "\n";
	}

	ranges = getRanges();
	foreach(RangePair range, ranges)
	{
		output += QString(indentation, '\t') + "range " + range.first.toString() + " " + range.second.toString() + ";\n";
	}

	output += QString(indentation - 1, '\t') + "}\n";
	return output;
}

ReturnValue RdsDhcpManagerPrivate::SharedNetwork::getOutput(int indentation) const
{
	RDS_LOCK;
	ReturnValue ret;
	QString output, contents;
	output += QString(indentation - 1, '\t') + "shared-network \"" + name + "\" {\n";
	ret = DHCPValues::getOutput(indentation);
	if (ret.isError())
		return ret;
	contents = ret.toString();
	if (contents.count() < 2) // It's always at least 1
	{
		return QString(indentation - 1, '\t') + "#There is an empty shared network named " + name + ". The following line puts it into RDS\n" + "#(RDS)empty-shared-network " + name + "\n";
	}
	else
	{
		if (subnets.count() < 1)
		{
			output += QString(indentation, '\t') + "# We have to add this bad subnet so that the shared network can exists, since it does not contain any valid subnets.\n";
			output += QString(indentation, '\t') + "subnet 0.0.0.0 netmask 0.0.0.0 {\n";
			output += QString(indentation + 1, '\t') + "#(RDS)hidden hidden\n";
			output += QString(indentation, '\t') + "}\n";
		}
	}
	output += contents;
	output += QString(indentation - 1, '\t') + "}\n";
	return output;
}

ReturnValue RdsDhcpManagerPrivate::Group::getOutput(int indentation) const
{
	RDS_LOCK;
	ReturnValue ret;
	QString output;
	output += QString(indentation - 1, '\t') + "group {\n";
	output += QString(indentation, '\t') + "#(RDS)name " + name + "\n";
	ret = DHCPValues::getOutput(indentation);
	if (ret.isError())
		return ret;
	output += ret.toString();
	// Put the shit here!

	output += QString(indentation - 1, '\t') + "}\n";
	return output;
}

ReturnValue RdsDhcpManagerPrivate::Host::getOutput(int indentation) const
{
	RDS_LOCK;
	ReturnValue ret;
	QString output;
	output += QString(indentation - 1, '\t') + "host " + name + " {\n";
	ret = DHCPValues::getOutput(indentation);
	if (ret.isError())
		return ret;
	output += ret.toString();
	// Put the shit here!

	output += QString(indentation - 1, '\t') + "}\n";
	return output;
}

RdsDhcpManagerPrivate::DHCPValues::~DHCPValues()
{
	foreach(RdsDhcpValues* value, refs)
	{
		///@todo Do something with the services using this object.
		value->setData(0);
	}
	for (QHash<QString, Subnet*>::const_iterator i = subnets.begin(); i != subnets.end(); ++i)
	{
		delete i.value();
	}
	for (QHash<QString, Group*>::const_iterator i = groups.begin(); i != groups.end(); ++i)
	{
		delete i.value();
	}
	for (QHash<QString, SharedNetwork*>::const_iterator i = sharedNetworks.begin(); i != sharedNetworks.end(); ++i)
	{
		delete i.value();
	}
	for (QHash<QString, Host*>::const_iterator i = hosts.begin(); i != hosts.end(); ++i)
	{
		delete i.value();
	}
}

void RdsDhcpManagerPrivate::DHCPValues::initializeValues()
{
	foreach(RdsDhcpManagerPrivate::Subnet* sn, subnetList)
	{
		if (sn->name.isEmpty())
		{
			sn->name = QObject::tr("Subnet");
		}
		else if (sn->name == "__RDS_DELETE_ME")
		{
			continue;
		}
		if (subnets.contains(sn->name))
		{
			int i = 1;
			while (subnets.contains(QString("%1 %2").arg(sn->name).arg(i)))
				i++;
			sn->name = QString("%1 %2").arg(sn->name).arg(i);
		}
		subnets.insert(sn->name, sn);
	}
	subnetList.clear();

	foreach(RdsDhcpManagerPrivate::Group* g, groupList)
	{
		if (g->name.isEmpty())
		{
			g->name = QObject::tr("Group");
		}
		if (groups.contains(g->name))
		{
			int i = 1;
			while (groups.contains(QString("%1 %2").arg(g->name).arg(i)))
				i++;
			g->name = QString("%1 %2").arg(g->name).arg(i);
		}
		groups.insert(g->name, g);
	}
	groupList.clear();

	for (QHash<QString, Subnet*>::const_iterator i = subnets.begin(); i != subnets.end(); ++i)
	{
		i.value()->name.remove('/');
		i.value()->initializeValues();
	}
	for (QHash<QString, Group*>::const_iterator i = groups.begin(); i != groups.end(); ++i)
	{
		i.value()->name.remove('/');
		i.value()->initializeValues();
	}
	for (QHash<QString, SharedNetwork*>::const_iterator i = sharedNetworks.begin(); i != sharedNetworks.end(); ++i)
	{
		i.value()->name.remove('/');
		i.value()->initializeValues();
	}
	for (QHash<QString, Host*>::const_iterator i = hosts.begin(); i != hosts.end(); ++i)
	{
		i.value()->name.remove('/');
		i.value()->initializeValues();
	}
}

QList<QPair<QHostAddress, QHostAddress> > RdsDhcpManagerPrivate::Subnet::getRanges() const
{
	QList<QPair<QHostAddress, QHostAddress> > ranges;
	QList<QHostAddress> addressPool = addresses;
	foreach(QHostAddress addr, RdsDhcpManagerPrivate::excluded())
	{
		addressPool.removeAll(addr);
	}
	while (addressPool.count() > 0)
	{
		QHostAddress start = addressPool.first();
		QHostAddress end = start;
		do
		{
			addressPool.removeAll(end);
			end = QHostAddress(end.toIPv4Address() + 1);
		}
		while (addressPool.contains(end));
		end = QHostAddress(end.toIPv4Address() - 1);

		ranges << QPair<QHostAddress, QHostAddress>(start, end);
	}

	return ranges;
}

QList<QPair<QHostAddress, QHostAddress> > RdsDhcpManagerPrivate::Subnet::getFullRanges() const
{
	QList<QPair<QHostAddress, QHostAddress> > ranges;
	QList<QHostAddress> addressPool = addresses;
	while (addressPool.count() > 0)
	{
		QHostAddress start = addressPool.first();
		QHostAddress end = start;
		do
		{
			addressPool.removeAll(end);
			end = QHostAddress(end.toIPv4Address() + 1);
		}
		while (addressPool.contains(end));
		end = QHostAddress(end.toIPv4Address() - 1);

		ranges << QPair<QHostAddress, QHostAddress>(start, end);
	}

	return ranges;
}

void RdsDhcpManagerPrivate::Subnet::addRange(const QHostAddress &start, const QHostAddress &end)
{
	for (quint32 i = start.toIPv4Address(); i <= end.toIPv4Address(); ++i)
	{
		addresses << QHostAddress(i);
	}
}

ReturnValue RdsDhcpManager::interfaces()
{
	QFile defaults("/etc/default/dhcp3-server");
	if (!defaults.open(QFile::ReadOnly))
	{
		return(ReturnValue(1, "Failed to open /etc/default/dhcp3-server for reading"));
	}

	QTextStream stream(&defaults);
	QString line;

	while ((line = stream.readLine()) != QString::Null())
	{
		line = line.trimmed();
		QStringList parts = line.split("=", QString::SkipEmptyParts);
		if (parts.size() != 2) continue;
		if (parts[0] != "INTERFACES") continue;
		parts[1] = parts[1].replace("\"", "");

		QStringList interfaces = parts[1].split(" ", QString::SkipEmptyParts);

		return(interfaces);
	}

	return(QStringList());
}

ReturnValue RdsDhcpManager::setInterfaces(QStringList interfaces)
{
	QStringList lines;
	QFile defaults("/etc/default/dhcp3-server");
	if (!defaults.open(QFile::ReadOnly))
	{
		return(ReturnValue(1, "Failed to open /etc/default/dhcp3-server for reading"));
	}

	QTextStream stream(&defaults);
	QString line;
	bool found = false;

	while ((line = stream.readLine()) != QString::Null())
	{
		QString newline = line.trimmed();
		if (newline.startsWith("INTERFACES"))
		{
			lines << "INTERFACES=\"" + interfaces.join(" ") + "\"\n";
			found = true;
		}
		else
		{
			lines << line;
		}
	}

	if (!found)
	{
		lines << "INTERFACES=\"" + interfaces.join(" ") + "\"\n";
	}

	defaults.close();

	if (!defaults.open(QFile::WriteOnly | QFile::Truncate))
	{
		return(ReturnValue(1, "Failed to open /etc/default/dhcp3-server for writing"));
	}

	foreach(line, lines)
	{
		defaults.write(line.toAscii() + "\n");
	}

	defaults.close();

	return(true);
}

ReturnValue RdsDhcpManager::enabled()
{
	QFile defaults("/etc/default/dhcp3-server");
	if (!defaults.open(QFile::ReadOnly))
	{
		return(ReturnValue(1, "Failed to open /etc/default/dhcp3-server for reading"));
	}

	QTextStream stream(&defaults);
	QString line;

	while ((line = stream.readLine()) != QString::Null())
	{
		line = line.trimmed();
		if (line.startsWith("exit 0")) return(false);
	}

	return(true);
}

ReturnValue RdsDhcpManager::setEnabled(bool enabled)
{
	if (!enabled)
	{
		stopService();
	}

	QStringList lines;
	QFile defaults("/etc/default/dhcp3-server");
	if (!defaults.open(QFile::ReadOnly))
	{
		return(ReturnValue(1, "Failed to open /etc/default/dhcp3-server for reading"));
	}

	QTextStream stream(&defaults);
	QString line;
	bool found = false;

	while ((line = stream.readLine()) != QString::Null())
	{
		QString newline = line.trimmed();
		if (newline.startsWith("exit"))
		{
			if (!enabled) lines << "exit 0";
			found = true;
		}
		else
		{
			lines << line;
		}
	}

	if (!found)
	{
		if (!enabled) lines << "exit 0";
	}

	defaults.close();

	if (!defaults.open(QFile::WriteOnly | QFile::Truncate))
	{
		return(ReturnValue(1, "Failed to open /etc/default/dhcp3-server for writing"));
	}

	foreach(line, lines)
	{
		defaults.write(line.toAscii() + "\n");
	}

	defaults.close();

	return(true);
}

ReturnValue RdsDhcpManager::configured()
{
	ReturnValue ret = values();
	RdsDhcpValues v = ret;

	if ((qxt_d().values()->sharedNetworks.size() == 0) &&
	        (qxt_d().values()->subnets.size() == 0) &&
	        (qxt_d().values()->groups.size() == 0) &&
	        (qxt_d().values()->hosts.size() == 0)) return(false);
	else
		return(true);
}

ReturnValue RdsDhcpManager::checkRange(const QString& id, const QHostAddress& start, const QHostAddress& end)
{
	ReturnValue ret = listAllSubnets();
	if (ret.isError()) return(ret);

	QStringList list = ret.toStringList();
	quint32 startaddr = start.toIPv4Address();
	quint32 endaddr = end.toIPv4Address();

	foreach(QString childid, list)
	{
		if (id == childid) continue;

		ret = values(childid);
		if (ret.isError())
		{
			qWarning() << "Failed to get values:" << childid << ret;
			continue;
		}

		RdsDhcpSubnet subnet = ret;
		ret = subnet.ranges();
		if (ret.isError())
		{
			qWarning() << "Failed to get range list" << childid << ret;
			continue;
		}

		foreach(RdsDhcpSubnet::Range range, ret.value<RdsDhcpSubnet::RangeList>())
		{
			if (!(((range.first.toIPv4Address() < startaddr) && (range.second.toIPv4Address() < startaddr)) ||
			        ((range.first.toIPv4Address() > endaddr) && (range.second.toIPv4Address() > endaddr))))
			{
				return(false);
			}
		}
	}

	return(true);
}

ReturnValue RdsDhcpManager::listAllGroups(QString baseid)
{
	ReturnValue ret = listEntities(baseid, true);
	if (ret.isError()) return(ret);

	RdsEntity entity = ret.value<RdsEntity>();

	QStringList list;
	foreach(RdsEntity child, entity.children())
	{
		if (child.type() == "group") list << child.id();
		ret = listAllGroups(child.id());
		if (ret.isError()) continue;

		list << ret.toStringList();
	}

	return(list);
}

ReturnValue RdsDhcpManager::listAllSharedNetworks(QString baseid)
{
	ReturnValue ret = listEntities(baseid, true);
	if (ret.isError()) return(ret);

	RdsEntity entity = ret.value<RdsEntity>();

	QStringList list;
	foreach(RdsEntity child, entity.children())
	{
		if (child.type() == "sharednetwork") list << child.id();
		ret = listAllSharedNetworks(child.id());
		if (ret.isError()) continue;

		list << ret.toStringList();
	}

	return(list);
}

ReturnValue RdsDhcpManager::listAllSubnets(QString baseid)
{
	ReturnValue ret = listEntities(baseid, true);
	if (ret.isError()) return(ret);

	RdsEntity entity = ret.value<RdsEntity>();

	QStringList list;
	foreach(RdsEntity child, entity.children())
	{
		if (child.type() == "subnet") list << child.id();
		ret = listAllSubnets(child.id());
		if (ret.isError()) continue;

		list << ret.toStringList();
	}

	return(list);
}

ReturnValue RdsDhcpManager::listAllHosts(QString baseid)
{
	ReturnValue ret = listEntities(baseid, true);
	if (ret.isError()) return(ret);

	RdsEntity entity = ret.value<RdsEntity>();

	QStringList list;
	foreach(RdsEntity child, entity.children())
	{
		if (child.type() == "host") list << child.id();
		ret = listAllHosts(child.id());
		if (ret.isError()) continue;

		list << ret.toStringList();
	}

	return(list);
}
