/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#ifndef SAMBACONFIGPARSER_H
#define SAMBACONFIGPARSER_H

#include <QxtPimpl>
#include <QtGlobal>
#include <ReturnValue>
#include <ServiceProxy>
#include <QHash>
#include <QMap>

#include "RdsDaemonInstance"

class RdsSambaConfigParserPrivate;
class QString;
class QStringList;

/**
	@author Chris Vickery <chris@resara.com>
	The const-ness of each function represents weather or not it actually edits the file or simply reads values from it
*/
class RdsSambaConfigParser : public QtRpc::ServiceProxy, public RdsDaemonInstance
{
	Q_OBJECT;
	QXT_DECLARE_PRIVATE(RdsSambaConfigParser);
	QTRPC_SERVICEPROXY_PIMPL(RdsSambaConfigParser);
public:
	RdsSambaConfigParser();
	RdsSambaConfigParser(const RdsSambaConfigParser &other);
	RdsSambaConfigParser(const QString &fileName);
	RdsSambaConfigParser(const char* fileName);
	RdsSambaConfigParser& operator=(const RdsSambaConfigParser &other);
	~RdsSambaConfigParser();

	bool parse(const QString &data);
	QString toString() const;

	virtual ReturnValue stopService();
	virtual ReturnValue startService();
	virtual ReturnValue restartService();
	virtual ReturnValue reloadService();
	virtual ReturnValue reloadConfig();

public slots:
	ReturnValue save() const;
	ReturnValue reload();
	/**
	 *        Gets the name/path of the configuration file
	 * @return returns a QString with the file name/path
	 */

	ReturnValue fileName() const;
	/**
	 *	Sets the file name of the configuration file
	 * @param fileName The name of the config file
	 * @return true on success, error on failure
	 */
	ReturnValue setFileName(const QString &fileName);

	/**
	 * Gets the value for a key
	 * @param section the section the key is in
	 * @param key the key to get the value of
	 * @return returns a QString with the value
	 */
	ReturnValue getValue(const QString &section, const QString &key) const;
	/**
	 *        sets a value for a key
	 * @param section the section the key is in
	 * @param key the key to set the value of
	 * @param value the value to give the key
	 * @return true
	 */
	ReturnValue setValue(const QString &section, const QString &key, const QString &value);

	/**
	 *        Enables a key
	 * @param section the section the key is in
	 * @param key the name of the key to enable
	 * @return true
	 */
	ReturnValue enableKey(const QString &section, const QString &key);
	/**
	 *        Disables a key
	 * @param section The section the key is in
	 * @param key The key to disable
	 * @return true
	 */
	ReturnValue disableKey(const QString &section, const QString &key);
	/**
	 *        Removes a key from the config file
	 * @param section The section the key is in
	 * @param key The key to remove
	 * @return true
	 */
	ReturnValue removeKey(const QString &section, const QString &key);
	/**
	 *        Enables a section
	 * @param section The section to Enable
	 * @return true
	 */
	ReturnValue enableSection(const QString &section);
	/**
	 *        Disables a section
	 * @param section the section to disable
	 * @return true
	 */
	ReturnValue disableSection(const QString &section);
	/**
	 *        Removes a section and all keys and rdsKeys
	 * @param section The section to remove
	 * @return true
	 */
	ReturnValue removeSection(const QString &section);
	/**
	 *        Re-names a section
	 * @param section The section to re-name
	 * @param newname The new name for the section
	 * @return true
	 */
	ReturnValue renameSection(const QString &section, const QString &newname);

	/**
	 *        Lists the sections in the configuration file
	 * @param listCommented
	 * @return QStringList of sections
	 */
	ReturnValue listSections(bool listCommented = false) const;
	/**
	 *        Lists the keys in a section
	 * @param section The section to get a list of keys from
	 * @param listCommented include keys that are disabled
	 * @return QStringList of keys in section
	 */
	ReturnValue listKeys(const QString &section, bool listCommented = false) const;
	/**
	 *        Determines whether a section exists
	 * @param section The section to check if it exists
	 * @param listCommented
	 * @return true if the section exists, false if it doesn't
	 */
	ReturnValue sectionExists(const QString &section, bool listCommented = false) const;
	/**
	 *        Determines whether a key exists in a section
	 * @param section The section to look for the key in
	 * @param key The key to look for
	 * @param listCommented
	 * @return true if the key exists, false if it doesn't
	 */
	ReturnValue keyExists(const QString &section, const QString &key, bool listCommented = false) const;

	/**
	 *        Gets the rds value for the key in the given section
	 * @param section The section the key is in
	 * @param key The key to get the value of
	 * @return QString containing the rds value
	 */
	ReturnValue getRdsValue(const QString &section, const QString &key) const;
	/**
	 *        Sets an rds value
	 * @param section The section to add the rds value to
	 * @param key The rds key to give the rds value to
	 * @param value The rds value
	 * @return true
	 */
	ReturnValue setRdsValue(const QString &section, const QString &key, const QString &value);

	/**
	 * 	  Enables a rds key
	 * @param section The section the rds key is in
	 * @param key The rds key to enable
	 * @return true
	 */
	ReturnValue enableRdsKey(const QString &section, const QString &key);
	/**
	 * 	  Disable a rds key
	 * @param section The section the rds key is in
	 * @param key The rds key to disable
	 * @return true
	 */
	ReturnValue disableRdsKey(const QString &section, const QString &key);
	/**
	 * 	  Removes an rds key
	 * @param section The section the rds key is in
	 * @param key The rds key to remove
	 * @return true
	 */
	ReturnValue removeRdsKey(const QString &section, const QString &key);

	/**
	 *        Gets a list of the rds keys in a section
	 * @param section The section to get a list of rds keys from
	 * @param listCommented include keys that are disabled
	 * @return QStringList of rds keys in the section
	 */
	ReturnValue listRdsKeys(const QString &section, bool listCommented = false) const;
	/**
	 *        Checks if an rds key exists
	 * @param section The section to look for the rds key in
	 * @param key the rds key to look for
	 * @param listCommented
	 * @return true if the rds key exists, false if it doesn't
	 */
	ReturnValue rdsKeyExists(const QString &section, const QString &key, bool listCommented = false) const;
	/**
	 * 	  Parses the configuration file into a QMap<section, QMap<key, value> >
	 * @param includeRdsValues
	 * @return QMap<section, QMap<key, value> >
	 */
	QMap<QString, QMap<QString, QString> > toMap(bool includeRdsValues = false) const;

};

#endif
