/***************************************************************************
 *   Copyright (C) 2009 by Brendan Powers   *
 *   brendan@resara.com   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "rdsthreadedjob.h"
#include "rdsthreadedjob_p.h"

QTRPC_SERVICEPROXY_PIMPL_IMPLEMENT(RdsThreadedJob);

void RdsThreadedJob::setSpawner(JobSpawnerBase* spawner)
{
	QMutexLocker locker(&qxt_d().mutex);
	qxt_d().spawner = spawner;
	qxt_d().thread.start();
	qxt_d().waiter.wait(&qxt_d().mutex);
}

RdsThreadedJob::RdsThreadedJob()
{
	qxt_d().thread._parentJob = this;
	qxt_d().thread._parentD = &qxt_d();
}

RdsThreadedJob::RdsThreadedJob(const RdsThreadedJob&other)
{
	operator=(other);
}

RdsThreadedJob::~RdsThreadedJob()
{
}

RdsThreadedJob& RdsThreadedJob::operator=(const RdsThreadedJob & other)
{
	RdsJob::operator=(other);
}

RdsJob* RdsThreadedJob::getJob()
{
	if (!qxt_d().thread._job)
		return 0;
	return qxt_d().thread._job;
}

ReturnValue RdsThreadedJob::pauseInternal()
{
	if (!qxt_d().thread._helper)
		return ReturnValue(1, "Internal helper object failed to initialize properly.");

	return qxt_d().thread._helper->invokeMethod("pause");
}

ReturnValue RdsThreadedJob::resumeInternal()
{
	if (!qxt_d().thread._helper)
		return ReturnValue(1, "Internal helper object failed to initialize properly.");

	return qxt_d().thread._helper->invokeMethod("resume");
}

ReturnValue RdsThreadedJob::startInternal()
{
	if (!qxt_d().thread._helper)
		return ReturnValue(1, "Internal helper object failed to initialize properly.");

	return qxt_d().thread._helper->invokeMethod("start");
}

ReturnValue RdsThreadedJob::stopInternal()
{
	if (!qxt_d().thread._helper)
		return ReturnValue(1, "Internal helper object failed to initialize properly.");

	return qxt_d().thread._helper->invokeMethod("stop");
}

void RdsThreadedJobThread::run()
{
	// This should never happen
	if (!_parentJob || !_parentD)
	{
		qCritical() << "Internal parent pointers in the RdsThreadedJobThread object were not set before starting the thread!";
		return;
	}

	if (!_parentD->spawner)
	{
		qCritical() << "Failed to find the job spawner object, internal RdsThreadedJobThread cannot continue";
		_parentD->mutex.lock();
		_parentD->waiter.wakeAll();
		_parentD->mutex.unlock();
	}

	_parentD->mutex.lock();

	_job = _parentD->spawner->getNewJobInstance();
	_helper = new RdsThreadedJobHelper(_job);

	connect(_job, SIGNAL(progressChanged(int, const QString&, const QString&)), _parentJob, SLOT(progressChangedSlot(int, const QString&, const QString&)));
	connect(_job, SIGNAL(statusChanged(RdsJob::Status)), _parentJob, SLOT(statusChangedSlot(RdsJob::Status)));
	connect(_job, SIGNAL(newLogMessages(const QList<RdsJob::LogEntry>&)), _parentJob, SLOT(newLogMessagesSlot(const QList<RdsJob::LogEntry>&)));
	connect(_job, SIGNAL(finished(const ReturnValue&)), _parentJob, SLOT(finishedSlot(const ReturnValue&)));

	_parentD->waiter.wakeAll();
	_parentD->mutex.unlock();

	exec();
}

void RdsThreadedJob::progressChangedSlot(int percent, const QString& progressText, const QString& progressDetails)
{
	setProgress(percent);
	setProgressText(progressText);
	setProgressDetails(progressDetails);
}

void RdsThreadedJob::statusChangedSlot(RdsJob::Status status)
{
	setStatus(status);
}

void RdsThreadedJob::newLogMessagesSlot(const QList<RdsJob::LogEntry> &logs)
{
	foreach(RdsJob::LogEntry l, logs)
	{
		log(l.log, l.level);
	}
}

void RdsThreadedJob::finishedSlot(const ReturnValue& ret)
{
	finish(ret);
}


RdsThreadedJobHelper::RdsThreadedJobHelper(RdsJob* job)
		: _job(job)
{
}

ReturnValue RdsThreadedJobHelper::invokeMethod(const QString &slot)
{
	QMutexLocker locker(&_mutex);
	if (!QMetaObject::invokeMethod(this, "doInvokeMethod", Qt::BlockingQueuedConnection, Q_ARG(const QString&, slot)))
		return ReturnValue(1, "Failed to call invoke method on my internal thread helper object");
	return _ret;
}

void RdsThreadedJobHelper::doInvokeMethod(const QString &slot)
{
	ReturnValue ret;
	QMetaObject::invokeMethod(_job, qPrintable(slot), Qt::DirectConnection, Q_RETURN_ARG(ReturnValue, ret));
	_ret = ret;
}




