/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsorganizationalunit.h"
#include "rdsorganizationalunit_p.h"

#include <RdsUtils>
#include <RdsLdapSession>
#include <QStringList>
#include <QDebug>
#include <RdsUser>
#include <RdsGroup>
#include <RdsComputer>

RdsOrganizationalUnit::RdsOrganizationalUnit()
		: RdsAdObject()
{
	QXT_INIT_PRIVATE(RdsOrganizationalUnit);
}

RdsOrganizationalUnit::RdsOrganizationalUnit(QString path)
		: RdsAdObject(path)
{
	QXT_INIT_PRIVATE(RdsOrganizationalUnit);
	qxt_d().path = path;
}

RdsOrganizationalUnit::RdsOrganizationalUnit(const RdsOrganizationalUnit& other)
		: RdsAdObject(other)
{
	QXT_INIT_PRIVATE(RdsOrganizationalUnit);
	qxt_d().path = other.qxt_d().path;
}

RdsOrganizationalUnit::~RdsOrganizationalUnit()
{
}

RdsOrganizationalUnit& RdsOrganizationalUnit::operator=(const RdsOrganizationalUnit & other)
{
	RdsAdObject::operator=(other);
	qxt_d().path = other.qxt_d().path;
	return *this;
}

QString RdsOrganizationalUnit::dn() const
{
	if (qxt_d().path != "") return(qxt_d().path);
	else return(RdsUtils::baseDn());
}

QString RdsOrganizationalUnit::userDn() const
{
	if (qxt_d().path != "") return(qxt_d().path);
	else return("CN=Users," + RdsUtils::baseDn());
}

QString RdsOrganizationalUnit::groupDn() const
{
	if (qxt_d().path != "") return(qxt_d().path);
	else return("CN=Users," + RdsUtils::baseDn());
}

QString RdsOrganizationalUnit::computerDn() const
{
	if (qxt_d().path != "") return(qxt_d().path);
	else return("CN=Computers," + RdsUtils::baseDn());
}

ReturnValue RdsOrganizationalUnit::listOus() const
{
	ReturnValue ret = rdsLdapSession()->list(dn(), "objectClass=OrganizationalUnit", false);
	if (ret.isError()) return(ret);

	QStringList list = ret.toStringList();
	if (qxt_d().path == "")
	{
		list.removeAll("OU=Domain Controllers," + dn());
	}

	return(list);
}

ReturnValue RdsOrganizationalUnit::listUsers() const
{
	ReturnValue ret = rdsLdapSession()->list(userDn(), "(&(objectClass=user)(!(objectClass=computer)))", false);
	return(ret);
}

ReturnValue RdsOrganizationalUnit::listGroups() const
{
	ReturnValue ret = rdsLdapSession()->list(groupDn(), "objectClass=group", false);
	return(ret);
}


ReturnValue RdsOrganizationalUnit::listComputers() const
{
	ReturnValue ret = rdsLdapSession()->list(computerDn(), "objectClass=computer", false);
	return(ret);
}

ReturnValue RdsOrganizationalUnit::createOu(QString name)
{
	RdsLdapActions actions;
	actions.add(RdsLdapActions::Add, "objectClass", QStringList() << "top" << "OrganizationalUnit");
	ReturnValue ret = rdsLdapSession()->add("OU=" + name + "," + dn(), actions);
	if (ret.isError()) return(ret);
	else return("OU=" + name + "," + dn());
}

ReturnValue RdsOrganizationalUnit::deleteOu(QString name)
{
	return(rdsLdapSession()->remove("OU=" + name + "," + dn()));
}

ReturnValue RdsOrganizationalUnit::ou() const
{
	ReturnValue ret = readAttribute("ou");
	if (ret.isError()) return(ret);
	LdapValues values = ret.value<LdapValues>();
	return(QString(values[0]));
}

ReturnValue RdsOrganizationalUnit::setOu(QString cn)
{
	return(rename("OU=" + cn));
}

ReturnValue RdsOrganizationalUnit::description() const
{
	ReturnValue ret = read();
	if (ret.isError()) return(ret);
	LdapResult entry = ret.value<LdapResult>();

	if (!entry.contains("description") || entry["description"].count() == 0) return("");
	else return(entry["description"][0]);
}

ReturnValue RdsOrganizationalUnit::setDescription(QString desc)
{
	RdsLdapActions actions;

	if (desc == "")
		actions.add(RdsLdapActions::Remove, "description", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "description", desc);

	return(rdsLdapSession()->modify(dn(), actions));
}

ReturnValue RdsOrganizationalUnit::remove()
{
	if (qxt_d().path == "") return(ReturnValue(1, "You cannot remove the base OU!"));
	return(RdsLdapObject::remove());
}

ReturnValue RdsOrganizationalUnit::createUser(QString name, QString account)
{
	//qDebug() << "Creating a user:" << ("CN=" + name + "," + userDn()) << account;
	return(RdsUser::createUser("CN=" + name + "," + userDn(), account));
}

ReturnValue RdsOrganizationalUnit::createGroup(QString name)
{
	return(RdsGroup::createGroup("CN=" + name + "," + groupDn()));
}

ReturnValue RdsOrganizationalUnit::getOu(const QString &ou)
{
	RdsOrganizationalUnit obj("OU=" + ou + "," + dn());
	if(obj.ou().isError())
		return ReturnValue(1, "Failed to find an OU by that name");
	return obj.dn();
}

ReturnValue RdsOrganizationalUnit::getUser(const QString &fullname)
{
	RdsUser obj("CN=" + fullname + "," + userDn());
	if(obj.cn().isError())
		return ReturnValue(1, "Failed to find a user by that name");
	return obj.cn();
}

ReturnValue RdsOrganizationalUnit::getGroup(const QString &group)
{
	RdsGroup obj("CN=" + group + "," + groupDn());
	if(obj.cn().isError())
		return ReturnValue(1, "Failed to find a group by that name");
	return obj.cn();
}

ReturnValue RdsOrganizationalUnit::getComputer(const QString &computer)
{
	RdsComputer obj("CN=" + computer + "," + computerDn());
	if(obj.cn().isError())
		return ReturnValue(1, "Failed to find a computer by that name");
	return obj.cn();
}

QDataStream& operator<<(QDataStream& d, const RdsOrganizationalUnit& object)
{
	d << object.dn();
	return(d);
}

QDataStream& operator>>(QDataStream& d, RdsOrganizationalUnit& object)
{
	QString dn;
	d >> dn;
	object.qxt_d().path = dn;
	return(d);
}
