/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdssid.h"
#include "rdssid_p.h"

#include <QString>
#include <QByteArray>
#include <QStringList>
#include <QDebug>
#include <RdsAdObject>
#include <RdsUtils>
#include "security.h"

RdsSid RdsSidPrivate::domainsid;

RdsSid::RdsSid()
{
	QXT_INIT_PRIVATE(RdsSid);
	qxt_d().data = new RdsSidData();
	qxt_d().data->version = 1;
	qxt_d().data->authority = 0;
	qxt_d().data->rid = 0;
}

RdsSid::RdsSid(const QString &sid)
{
	QXT_INIT_PRIVATE(RdsSid);
	qxt_d().data = new RdsSidData();
	qxt_d().parseString(sid);
}

RdsSid::RdsSid(const char *sid)
{
	QXT_INIT_PRIVATE(RdsSid);
	qxt_d().data = new RdsSidData();
	qxt_d().parseString(sid);
}

RdsSid::RdsSid(const QByteArray &data)
{
	QXT_INIT_PRIVATE(RdsSid);
	qxt_d().data = new RdsSidData();
	qxt_d().data->version = data[0];
	int length = data[1];
	char *ptr = (char *) & qxt_d().data->authority;
	ptr[0] = data[7];
	ptr[1] = data[6];
	ptr[2] = data[5];
	ptr[3] = data[4];
	ptr[4] = data[3];
	ptr[5] = data[2];
	const char *buff = data.constData();
	uint *num = (uint *)(buff + 8);

	for (int i = 0; i < (length - 1); i++)
	{
		qxt_d().data->domain << num[i];
	}

	qxt_d().data->rid = num[length -1];
}

RdsSid::RdsSid(const RdsSid &other)
{
	QXT_INIT_PRIVATE(RdsSid);
	qxt_d().data = other.qxt_d().data;
}

RdsSid::~RdsSid()
{
}

RdsSid& RdsSid::operator=(const RdsSid & other)
{
	qxt_d().data = other.qxt_d().data;
	return *this;
}

RdsSid& RdsSid::operator=(const QString & str)
{
	qxt_d().parseString(str);
	return *this;
}

bool RdsSid::operator==(const RdsSid &other) const
{
	return (
	           (qxt_d().data->version == other.qxt_d().data->version) &&
	           (qxt_d().data->authority == other.qxt_d().data->authority) &&
	           (qxt_d().data->domain == other.qxt_d().data->domain) &&
	           (qxt_d().data->rid == other.qxt_d().data->rid)
	       );
}

bool RdsSid::operator!=(const RdsSid &other) const
{
	return !operator==(other);
}

bool RdsSid::operator==(const QString &str) const
{
	return operator==(RdsSid(str));
}

bool RdsSid::operator!=(const QString &str) const
{
	return !operator==(str);
}

QString RdsSid::toString() const
{
	QString str = QString("S-%1-%2").arg(qxt_d().data->version).arg(qxt_d().data->authority);

	foreach(uint d, qxt_d().data->domain)
	{
		str += QString("-%1").arg(d);
	}

	str += QString("-%1").arg(qxt_d().data->rid);

	return str;
}

#define CONVERT_TO_STRING(a,b) if(toString() == b) return a;
#define CONVERT_TO_STRING_RID(a,b) case b: return a;

QString RdsSid::toShortString() const
{

	if (qxt_d().data->domain == getDomainSid().domain())
	{
		switch (qxt_d().data->rid)
		{
				CONVERT_TO_STRING_RID("CA", DOMAIN_RID_CERT_ADMINS);
				CONVERT_TO_STRING_RID("CD", 574);
				CONVERT_TO_STRING_RID("DA", DOMAIN_RID_ADMINS);
				CONVERT_TO_STRING_RID("DC", DOMAIN_RID_DOMAIN_MEMBERS);
				CONVERT_TO_STRING_RID("DD", DOMAIN_RID_DCS);
				CONVERT_TO_STRING_RID("DG", DOMAIN_RID_GUESTS);
				CONVERT_TO_STRING_RID("DU", DOMAIN_RID_USERS);
				CONVERT_TO_STRING_RID("EA", DOMAIN_RID_ENTERPRISE_ADMINS);
				CONVERT_TO_STRING_RID("HI", 12288);
				CONVERT_TO_STRING_RID("LA", DOMAIN_RID_ADMINISTRATOR);
				CONVERT_TO_STRING_RID("LG", DOMAIN_RID_GUEST);
				CONVERT_TO_STRING_RID("LK", DOMAIN_RID_KRBTGT);
				CONVERT_TO_STRING_RID("LW", 4096);
				CONVERT_TO_STRING_RID("ME", 8192);
				CONVERT_TO_STRING_RID("PA", DOMAIN_RID_POLICY_ADMINS);
				CONVERT_TO_STRING_RID("RO", DOMAIN_RID_READONLY_DCS);
				CONVERT_TO_STRING_RID("RS", DOMAIN_RID_RAS_SERVERS);
				CONVERT_TO_STRING_RID("SA", DOMAIN_RID_SCHEMA_ADMINS);
				CONVERT_TO_STRING_RID("SI", 16384);
			default:
				return(toString());
		}
	}
	else
	{
		CONVERT_TO_STRING("AN",SID_NT_ANONYMOUS);
		CONVERT_TO_STRING("AO", SID_BUILTIN_ACCOUNT_OPERATORS);
		CONVERT_TO_STRING("AU", SID_NT_AUTHENTICATED_USERS);
		CONVERT_TO_STRING("BA", SID_BUILTIN_ADMINISTRATORS);
		CONVERT_TO_STRING("BO", SID_BUILTIN_BACKUP_OPERATORS);
		CONVERT_TO_STRING("BU", SID_BUILTIN_USERS);
		CONVERT_TO_STRING("CG", SID_CREATOR_GROUP);
		CONVERT_TO_STRING("CO", SID_CREATOR_OWNER);
		CONVERT_TO_STRING("ED", SID_NT_ENTERPRISE_DCS);
		CONVERT_TO_STRING("IU", SID_NT_INTERACTIVE);
		CONVERT_TO_STRING("LS", SID_NT_LOCAL_SERVICE);
		CONVERT_TO_STRING("NO", SID_BUILTIN_NETWORK_CONF_OPERATORS);
		CONVERT_TO_STRING("NS", SID_NT_NETWORK_SERVICE);
		CONVERT_TO_STRING("NU", SID_NT_NETWORK);
		CONVERT_TO_STRING("PO", SID_BUILTIN_PRINT_OPERATORS);
		CONVERT_TO_STRING("PS", SID_NT_SELF);
		CONVERT_TO_STRING("PU", SID_BUILTIN_POWER_USERS);
		CONVERT_TO_STRING("RC", SID_NT_RESTRICTED);
		CONVERT_TO_STRING("RD", SID_BUILTIN_REMOTE_DESKTOP_USERS);
		CONVERT_TO_STRING("RE", SID_BUILTIN_REPLICATOR);
		CONVERT_TO_STRING("RU", SID_BUILTIN_PREW2K);
		CONVERT_TO_STRING("SO", SID_BUILTIN_SERVER_OPERATORS);
		CONVERT_TO_STRING("SU", SID_NT_SERVICE);
		CONVERT_TO_STRING("SY", SID_NT_SYSTEM);
		CONVERT_TO_STRING("WD", SID_WORLD);
		return(toString());
	}
}

#define CHECK_FOR_ABREVIATION(a,b) if(str == a) str = b;
#define CHECK_FOR_ABREVIATION_RID(a,b) if(str == a) { domain.setRid(b); str = domain.toString(); }

void RdsSidPrivate::parseString(QString str)
{
	if (str.size() == 2)
	{
		RdsSid domain = RdsSid::getDomainSid();

		str = str.toUpper();
		if (str == "AN") str = SID_NT_ANONYMOUS;
		CHECK_FOR_ABREVIATION("AO", SID_BUILTIN_ACCOUNT_OPERATORS);
		CHECK_FOR_ABREVIATION("AU", SID_NT_AUTHENTICATED_USERS);
		CHECK_FOR_ABREVIATION("BA", SID_BUILTIN_ADMINISTRATORS);
		CHECK_FOR_ABREVIATION("BO", SID_BUILTIN_BACKUP_OPERATORS);
		CHECK_FOR_ABREVIATION("BU", SID_BUILTIN_USERS);
		CHECK_FOR_ABREVIATION_RID("CA", DOMAIN_RID_CERT_ADMINS);
		CHECK_FOR_ABREVIATION_RID("CD", 574);
		CHECK_FOR_ABREVIATION("CG", SID_CREATOR_GROUP);
		CHECK_FOR_ABREVIATION("CO", SID_CREATOR_OWNER);
		CHECK_FOR_ABREVIATION_RID("DA", DOMAIN_RID_ADMINS);
		CHECK_FOR_ABREVIATION_RID("DC", DOMAIN_RID_DOMAIN_MEMBERS);
		CHECK_FOR_ABREVIATION_RID("DD", DOMAIN_RID_DCS);
		CHECK_FOR_ABREVIATION_RID("DG", DOMAIN_RID_GUESTS);
		CHECK_FOR_ABREVIATION_RID("DU", DOMAIN_RID_USERS);
		CHECK_FOR_ABREVIATION_RID("EA", DOMAIN_RID_ENTERPRISE_ADMINS);
		CHECK_FOR_ABREVIATION("ED", SID_NT_ENTERPRISE_DCS);
		CHECK_FOR_ABREVIATION_RID("HI", 12288);
		CHECK_FOR_ABREVIATION("IU", SID_NT_INTERACTIVE);
		CHECK_FOR_ABREVIATION_RID("LA", DOMAIN_RID_ADMINISTRATOR);
		CHECK_FOR_ABREVIATION_RID("LG", DOMAIN_RID_GUEST);
		CHECK_FOR_ABREVIATION_RID("LK", DOMAIN_RID_KRBTGT);
		CHECK_FOR_ABREVIATION("LS", SID_NT_LOCAL_SERVICE);
		CHECK_FOR_ABREVIATION_RID("LW", 4096);
		CHECK_FOR_ABREVIATION_RID("ME", 8192);
		CHECK_FOR_ABREVIATION("NO", SID_BUILTIN_NETWORK_CONF_OPERATORS);
		CHECK_FOR_ABREVIATION("NS", SID_NT_NETWORK_SERVICE);
		CHECK_FOR_ABREVIATION("NU", SID_NT_NETWORK);
		CHECK_FOR_ABREVIATION_RID("PA", DOMAIN_RID_POLICY_ADMINS);
		CHECK_FOR_ABREVIATION("PO", SID_BUILTIN_PRINT_OPERATORS);
		CHECK_FOR_ABREVIATION("PS", SID_NT_SELF);
		CHECK_FOR_ABREVIATION("PU", SID_BUILTIN_POWER_USERS);
		CHECK_FOR_ABREVIATION("RC", SID_NT_RESTRICTED);
		CHECK_FOR_ABREVIATION("RD", SID_BUILTIN_REMOTE_DESKTOP_USERS);
		CHECK_FOR_ABREVIATION("RE", SID_BUILTIN_REPLICATOR);
		CHECK_FOR_ABREVIATION_RID("RO", DOMAIN_RID_READONLY_DCS);
		CHECK_FOR_ABREVIATION_RID("RS", DOMAIN_RID_RAS_SERVERS);
		CHECK_FOR_ABREVIATION("RU", SID_BUILTIN_PREW2K);
		CHECK_FOR_ABREVIATION_RID("SA", DOMAIN_RID_SCHEMA_ADMINS);
		CHECK_FOR_ABREVIATION_RID("SI", 16384);
		CHECK_FOR_ABREVIATION("SO", SID_BUILTIN_SERVER_OPERATORS);
		CHECK_FOR_ABREVIATION("SU", SID_NT_SERVICE);
		CHECK_FOR_ABREVIATION("SY", SID_NT_SYSTEM);
		CHECK_FOR_ABREVIATION("WD", SID_WORLD);
	}

	data->domain.clear();
	QStringList parts = str.split("-", QString::SkipEmptyParts);

	//If its too small, create a NULL sid
	if (parts.size() < 4)
	{
		data->version = 1;
		data->authority = 0;
		data->rid = 0;
		return;
	}

	uint domainsize = parts.count()  - 4;

	data->version = parts[1].toInt();
	data->authority = parts[2].toLongLong();

	for (uint i = 0; i < domainsize; i++)
	{
		data->domain << parts[i+3].toLongLong();
	}

	data->rid = parts.last().toLongLong();

	return;
}

QByteArray RdsSid::toBinary() const
{
	uint length = qxt_d().data->domain.size() + 1;
	int size = 2 + 6 + (length * sizeof(uint));
	QByteArray data(size, '\0');
	char *buff = data.data();

	buff[0] = qxt_d().data->version;
	buff[1] = (uchar)length;
	char *tmp = (char *) & qxt_d().data->authority;
	buff[2] = tmp[5];
	buff[3] = tmp[4];
	buff[4] = tmp[3];
	buff[5] = tmp[2];
	buff[6] = tmp[1];
	buff[7] = tmp[0];

	uint *num = (uint *) & buff[8];
	uint i = 0;
	for (i = 0; i < length - 1; i++)
	{
		num[i] = qxt_d().data->domain[i];
	}

	num[i] = qxt_d().data->rid;

	return(data);
}

uchar RdsSid::version() const
{
	return(qxt_d().data->version);
}

void RdsSid::setVersion(uchar ver)
{
	qxt_d().data->version = ver;
}

unsigned long long RdsSid::authority() const
{
	return(qxt_d().data->authority);
}

void RdsSid::setAuthority(unsigned long long auth)
{
	qxt_d().data->authority = auth;
}

QList<uint> RdsSid::domain() const
{
	return(qxt_d().data->domain);
}

void RdsSid::setDomain(QList<uint> dom)
{
	qxt_d().data->domain = dom;
}

uint RdsSid::rid() const
{
	return(qxt_d().data->rid);
}

void RdsSid::setRid(uint r)
{
	qxt_d().data->rid = r;
}

bool RdsSid::isNull() const
{
	return((qxt_d().data->version == 1) &&
			(qxt_d().data->authority == 0) &&
			(qxt_d().data->domain.size() == 0) &&
			(qxt_d().data->rid == 0));
}

RdsSid RdsSid::getDomainSid()
{
	if(RdsSidPrivate::domainsid.isNull())
	{
		RdsAdObject domain(RdsUtils::baseDn());
		ReturnValue ret = domain.sid();
		if(ret.isError())
		{
			qWarning() << "Failed to look up domain SID:" << ret;
		}
		else
		{
			RdsSidPrivate::domainsid = ret.value<RdsSid>();
			QList<uint> d = RdsSidPrivate::domainsid.domain();
			d << RdsSidPrivate::domainsid.rid();
			RdsSidPrivate::domainsid.setDomain(d);
			RdsSidPrivate::domainsid.setRid(0);
		}
	}
	return(RdsSidPrivate::domainsid);
}

QDebug operator<<(QDebug dbg, const RdsSid& sid)
{
	dbg.nospace() << "RdsSid(" << sid.toString() << ")";
	return dbg.space();
}

QDataStream& operator<<(QDataStream& d, const RdsSid& sid)
{
	d << sid.qxt_d().data->version;
	d << sid.qxt_d().data->authority;
	d << sid.qxt_d().data->domain;
	d << sid.qxt_d().data->rid;
	return d;
}

QDataStream& operator>>(QDataStream& d, RdsSid& sid)
{
	d >> sid.qxt_d().data->version;
	d >> sid.qxt_d().data->authority;
	d >> sid.qxt_d().data->domain;
	d >> sid.qxt_d().data->rid;
	return d;
}

