/***************************************************************************
 *   Copyright (C) 2009 by Resara LLC   *
 *   brendan@resara.com   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "rdsldapobjecttester.h"
/** @NOTE
	this test creates a structure on the ldap server that looks like this:

	dc=local
		dc=resara
			.
			.
			.
			ou=testou
			ou=testarea
				ou=depthou
					cn=depthdude
				cn=normal
				cn=freak

*/
REGISTER_TEST(RdsLdapObjectTester);


RdsLdapObjectTester::RdsLdapObjectTester(QObject *parent)
		: QObject(parent)
{
}


RdsLdapObjectTester::~RdsLdapObjectTester()
{
}


void listAttrs(RdsLdapObject *object)
{
	ReturnValue ret = object->read();
	QVERIFY(!ret.isError());
	//QHash<QString, QList<QByteArray>
	LdapResult hash = ret.value<LdapResult>();
	QList<QString> keys = hash.keys();
	for (int i = 0; i < keys.size(); ++i)
	{
		LdapValues list = hash[keys[i]];
		for (int j = 0; j < list.size(); ++j)
			qDebug() << keys[i] << "=" << list[j];
	}
}

bool objContains(RdsLdapObject *object, const QString &name, const QString &value)
{
	ReturnValue ret = object->readAttribute(name);
	if (ret.isError()) return false;
	LdapValues list = ret.value<QList<QByteArray> >();
	for (int i = 0; i < list.size(); ++i)
	{
		if (list[i] == value) return true;
	}
	return false;
}

void RdsLdapObjectTester::initTestCase()
{
	RdsLdapObject obj1("CN=depthdude,ou=depthou,ou=testarea,dc=resara,dc=local");
	obj1.remove();
	
	obj1.setDn("ou=depthou,ou=testarea,dc=resara,dc=local");
	obj1.remove();

	obj1.setDn("cn=normal,ou=testarea,dc=resara,dc=local");
	obj1.remove();

	obj1.setDn("cn=freak,ou=testarea,dc=resara,dc=local");
	obj1.remove();

	obj1.setDn("ou=newname,dc=resara,dc=local");// sometimes sneaks in if a verify fails
	obj1.remove();

	obj1.setDn("ou=testarea,dc=resara,dc=local");
	obj1.remove();

	RdsLdapActions action;
	action.add(RdsLdapActions::Add, "objectclass", "organizationalUnit");
	action.add(RdsLdapActions::Add, "ou", "testarea");
	action.add(RdsLdapActions::Add, "description", "test area");
	TEST_FUNCTION(obj1.add(action));

	action.clear();
	action.add(RdsLdapActions::Add, "objectclass", "person");
	action.add(RdsLdapActions::Add, "cn", "freak");
	action.add(RdsLdapActions::Add, "sn", "freak");
	obj1.setDn("CN=freak,OU=testarea,DC=resara,DC=local");
	TEST_FUNCTION(obj1.add(action));

	action.clear();
	action.add(RdsLdapActions::Add, "objectclass", "person");
	action.add(RdsLdapActions::Add, "cn", "normal");
	action.add(RdsLdapActions::Add, "sn", "normal");
	obj1.setDn("CN=normal,OU=testarea,DC=resara,DC=local");
	TEST_FUNCTION(obj1.add(action));

	action.clear();
	action.add(RdsLdapActions::Add, "objectclass", "organizationalUnit");
	action.add(RdsLdapActions::Add, "ou", "depthou");
	obj1.setDn("OU=depthou,OU=testarea,DC=resara,DC=local");
	TEST_FUNCTION(obj1.add(action));

	action.clear();
	action.add(RdsLdapActions::Add, "objectclass", "person");
	action.add(RdsLdapActions::Add, "cn", "depthdude");
	action.add(RdsLdapActions::Add, "sn", "depthdude");
	obj1.setDn("CN=depthdude,OU=depthou,OU=testarea,DC=resara,DC=local");
	TEST_FUNCTION(obj1.add(action));
}

void RdsLdapObjectTester::init()
{
	RdsLdapActions action;
	action.add(RdsLdapActions::Add, "objectclass", "organizationalUnit");
	action.add(RdsLdapActions::Add, "ou", "newou");
	action.add(RdsLdapActions::Add, "l", "testlocation");
	action.add(RdsLdapActions::Add, "st", "state");
	action.add(RdsLdapActions::Add, "description", "test unit");
	obj.setDn("ou=newou,dc=resara,dc=local");
	TEST_FUNCTION(obj.add(action));
	sesh = obj.session();
}

void RdsLdapObjectTester::cleanup()
{
	QVERIFY(obj.remove().toBool());
}

void RdsLdapObjectTester::cleanupTestCase()
{
	RdsLdapObject obj1("CN=depthdude,ou=depthou,ou=testarea,dc=resara,dc=local");
	obj1.remove();
	
	obj1.setDn("ou=depthou,ou=testarea,dc=resara,dc=local");
	obj1.remove();

	obj1.setDn("cn=normal,ou=testarea,dc=resara,dc=local");
	obj1.remove();

	obj1.setDn("cn=freak,ou=testarea,dc=resara,dc=local");
	obj1.remove();

	obj1.setDn("ou=testarea,dc=resara,dc=local");
	obj1.remove();
}

void RdsLdapObjectTester::constructorTest()
{
	RdsLdapObject obj1;
	obj1.setDn(QString("ou=testarea,dc=resara,dc=local"));
	QCOMPARE(obj1.dn(), QString("ou=testarea,dc=resara,dc=local"));
	RdsLdapObject obj2(obj1), obj3(QString("ou=testarea,dc=resara,dc=local"));
	QCOMPARE(obj1.dn(), obj2.dn());
	QCOMPARE(obj1.session(), obj2.session());
	QCOMPARE(obj3.dn(), QString("ou=testarea,dc=resara,dc=local"));
	RdsLdapObject obj4(QString("ou=testarea,dc=resara,dc=local"), sesh);
	QCOMPARE(obj4.session(), sesh);
}

void RdsLdapObjectTester::addModifyTest()
{
	//add things that should work
	RdsLdapActions actions;
	RdsLdapObject obj1("cn=testdude,ou=testarea,dc=resara,dc=local");
	obj1.remove();
	actions.add(RdsLdapActions::Add, "cn", "testdude");
	actions.add(RdsLdapActions::Add, "sn", "testdude");
	actions.add(RdsLdapActions::Add, "objectclass", "person");
	actions.add(RdsLdapActions::Add, "description", "this is a test person");
	ReturnValue ret = obj1.add(actions);
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QVERIFY(objContains(&obj1, "cn", "testdude"));
	QVERIFY(objContains(&obj1, "sn", "testdude"));
	QVERIFY(objContains(&obj1, "objectclass", "person"));
	QVERIFY(objContains(&obj1, "description", "this is a test person"));
	QVERIFY2(!(ret = obj1.remove()).isError(), qPrintable(ret.errString()));

	QList<QString> list;
	list << "description1" << "description2" << "description3";
	actions.clear();
	actions.add(RdsLdapActions::Add, "objectclass", "person");
	actions.add(RdsLdapActions::Add, "sn", "dude");
	actions.add(RdsLdapActions::Add, "cn", "testdude");
	actions.add(RdsLdapActions::Add, "description", "this is a test person");
	actions.add(RdsLdapActions::Add, "description", "still a test person");
	actions.add(RdsLdapActions::Add, "description", list);
	ret = obj1.add(actions);
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QVERIFY(objContains(&obj1, "cn", "testdude"));
	QVERIFY(objContains(&obj1, "sn", "dude"));
	QVERIFY(objContains(&obj1, "objectclass", "person"));
	QVERIFY(objContains(&obj1, "description", "this is a test person"));
	QVERIFY(objContains(&obj1, "description", "still a test person"));
	QVERIFY(objContains(&obj1, "description", "description1"));
	QVERIFY(objContains(&obj1, "description", "description2"));
	QVERIFY(objContains(&obj1, "description", "description3"));

	actions.clear();
	actions.add(RdsLdapActions::Add, "objectclass", "person");
	QVERIFY(obj1.add(actions).isError()); //can't add() to an object that already exists
	QVERIFY2(!(ret = obj1.remove()).isError(), qPrintable(ret.errString()));

	//add things that should break
	actions.clear();
	actions.add(RdsLdapActions::Add, "objectclass", "person");
	actions.add(RdsLdapActions::Add, "", "a description");
	ret = obj1.add(actions);
	QVERIFY(ret.isError());
	actions.clear();
	actions.add(RdsLdapActions::Add, "objectclass", "person");
	actions.add(RdsLdapActions::Add, "description", "");
	ret = obj1.add(actions);
	QVERIFY(ret.isError());
	actions.clear();
	actions.add(RdsLdapActions::Add, "objectclass", "person");
	actions.add(RdsLdapActions::Add, "blorg", "broked");
	ret = obj1.add(actions);
	QVERIFY(ret.isError());
	actions.clear();
	actions.add(RdsLdapActions::Add, "objectclass", "person");
	actions.add(RdsLdapActions::Add, "cn", "newou");
	ret = obj1.add(actions);
	QVERIFY(ret.isError());

	RdsLdapObject obj2("cn:ziltoid,ou=testarea,dc=resara,dc=local");
	actions.clear();
	actions.add(RdsLdapActions::Add, "description", "omniscient");
	QVERIFY(obj2.add(actions).isError());

	//modify things that should work
	obj1.remove();
	actions.clear();
	actions.add(RdsLdapActions::Add, "objectclass", "person");
	actions.add(RdsLdapActions::Add, "description", list);
	actions.add(RdsLdapActions::Add, "sn", "dude");
	actions.add(RdsLdapActions::Add, "cn", "testdude");
	ret = obj1.add(actions);//modify from nothing (aka add)
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QVERIFY(objContains(&obj1, "cn", "testdude"));
	QVERIFY(objContains(&obj1, "sn", "dude"));
	QVERIFY(objContains(&obj1, "objectclass", "person"));
	QVERIFY(objContains(&obj1, "description", "description1"));
	QVERIFY(objContains(&obj1, "description", "description2"));
	QVERIFY(objContains(&obj1, "description", "description3"));
	QVERIFY2(!(ret = obj1.remove()).isError(), qPrintable(ret.errString()));

	//back to member obj, done with obj1
	actions.clear();
	actions.add(RdsLdapActions::Add, "description", list);
	actions.add(RdsLdapActions::Remove, "description", "test unit");
	actions.add(RdsLdapActions::Remove, "st", "");
	actions.add(RdsLdapActions::Replace, "l", "dude");
	ret = obj.modify(actions);
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QVERIFY(objContains(&obj, "l", "dude"));
	QVERIFY(!objContains(&obj, "l", "testlocation"));
	QVERIFY(!objContains(&obj, "st", "state"));
	QVERIFY(objContains(&obj, "ou", "newou"));
	QVERIFY(objContains(&obj, "objectclass", "organizationalUnit"));
	QVERIFY(!objContains(&obj, "description", "test unit"));
	QVERIFY(objContains(&obj, "description", "description1"));
	QVERIFY(objContains(&obj, "description", "description2"));
	QVERIFY(objContains(&obj, "description", "description3"));

	actions.clear();
	actions.add(RdsLdapActions::Replace, "description", "test unit");
	ret = obj.modify(actions);
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QVERIFY(objContains(&obj, "l", "dude"));
	QVERIFY(objContains(&obj, "ou", "newou"));
	QVERIFY(objContains(&obj, "objectclass", "organizationalUnit"));
	QVERIFY(objContains(&obj, "description", "test unit"));
	QVERIFY(!objContains(&obj, "description", "description1"));
	QVERIFY(!objContains(&obj, "description", "description2"));
	QVERIFY(!objContains(&obj, "description", "description3"));

	actions.clear();
	actions.add(RdsLdapActions::Add, "description", list);
	actions.add(RdsLdapActions::Remove, "description", "");
	ret = obj.modify(actions);
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QVERIFY(objContains(&obj, "l", "dude"));
	QVERIFY(objContains(&obj, "ou", "newou"));
	QVERIFY(objContains(&obj, "objectclass", "organizationalUnit"));
	QVERIFY(!objContains(&obj, "description", "test unit"));
	QVERIFY(!objContains(&obj, "description", "description1"));
	QVERIFY(!objContains(&obj, "description", "description2"));
	QVERIFY(!objContains(&obj, "description", "description3"));

	//modify things that should break
	actions.clear();
	actions.add(RdsLdapActions::Remove, "description", "");
	ret = obj.modify(actions);
	QVERIFY(ret.isError());
	actions.clear();
	actions.add(RdsLdapActions::Remove, "", "dude");
	ret = obj.modify(actions);
	QVERIFY(ret.isError());
	actions.clear();
	actions.add(RdsLdapActions::Add, "l", "new");
	ret = obj.modify(actions);
	QVERIFY(ret.isError());
	actions.clear();
	actions.add(RdsLdapActions::Replace, "ou", "testou");
	ret = obj.modify(actions);
	QVERIFY(ret.isError());
	actions.clear();
	actions.add(RdsLdapActions::Remove, "blorg", "broked");
	ret = obj.modify(actions);
	QVERIFY(ret.isError());
}

void RdsLdapObjectTester::RenameTest()
{
	//valid re-names
	ReturnValue ret = obj.rename("ou=newname");
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QCOMPARE(ret.toString(), QString("ou=newname,dc=resara,dc=local"));

	//invalid re-names
	TEST_FUNCTION_ERROR(obj.rename("ou:orgunit"));
	TEST_FUNCTION_ERROR(obj.rename("dc=funky"));
	TEST_FUNCTION_ERROR(obj.rename("cn=COMMONNAME"));
	TEST_FUNCTION_ERROR(obj.rename("IWANTTHISNAME"));
	TEST_FUNCTION_ERROR(obj.rename(""));
	TEST_FUNCTION_ERROR(obj.rename("tard=ugly"));
	TEST_FUNCTION_ERROR(obj.rename("ou=newname,ou=testarea"));
}

void RdsLdapObjectTester::searchTest()
{
	RdsLdapObject obj1("OU=testarea,dc=resara,dc=local");
	//qDebug() << "SEARCHING FOR depthdude OR normal";
	ReturnValue ret = obj1.search("(|(cn=depthdude)(cn=normal))");
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	//QHash<QString, QHash<QString, QList<QByteArray> > >
	LdapResults ldapresultshash = ret.value<LdapResults>();
	QList<QString> ldapresultskeys = ldapresultshash.keys();
// 	qDebug() << ldapresultskeys.size();
// 	for (int k = 0; k < ldapresultskeys.size(); ++k)
// 		qDebug() << ldapresultskeys[k];
	QVERIFY2(ldapresultskeys.size() == 2, "wrong number of search results found");

	ret = obj1.search("(|(cn=depthdude)(cn=normal))", false);
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	//QHash<QString, QHash<QString, QList<QByteArray> > >
	ldapresultshash = ret.value<LdapResults>();
	ldapresultskeys = ldapresultshash.keys();
// 	qDebug() << ldapresultskeys.size();
// 	for (int k = 0; k < ldapresultskeys.size(); ++k)
// 		qDebug() << ldapresultskeys[k];
	QVERIFY2(ldapresultskeys.size() == 1, "wrong number of search results found");
}

void RdsLdapObjectTester::readTest()
{
	QStringList attrs;
	attrs << "l" << "objectclass";
	QString attr = "l";

	//qDebug() << "read()";
	ReturnValue ret = obj.read();
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	//QHash<QString, QList<QByteArray>
	LdapResult hash = ret.value<LdapResult>();
	QList<QString> keys = hash.keys();
	for (int i = 0; i < keys.size(); ++i)
	{
		QList<QByteArray> list = hash[keys[i]];
		for (int j = 0; j < list.size(); ++j)
			QVERIFY(objContains(&obj, keys[i], list[j]));
	}

	//qDebug() << "read(attrs)";
	ret = obj.read(attrs);
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	//QHash<QString, QList<QByteArray>
	hash = ret.value<LdapResult>();
	for (int i = 0; i < keys.size(); ++i)
	{
		LdapValues list = hash[keys[i]];
		for (int j = 0; j < list.size(); ++j)
			QVERIFY(objContains(&obj, keys[i], list[j]));
	}

	//qDebug() << "readAttribute(\"l\")";
	ret = obj.readAttribute("l");
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	// QList<QByteArray> (LdapValues)
	LdapValues list = ret.value<LdapValues>();
	for (int i = 0; i < list.size(); ++i)
		QCOMPARE(QByteArray("testlocation"), list[i]);

	RdsLdapActions action;
	action.add(RdsLdapActions::Add, "description", "another description");
	ret =  obj.modify(action);
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	//qDebug() << "readAttribute(\"description\")";
	ret = obj.readAttribute("description");
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	list = ret.value<LdapValues>();
	QVERIFY(list.size() == 2);
	if (list[0] != "test unit" && list[0] != "another description") QFAIL("read attribute (description) failed");
	if (list[1] != "test unit" && list[1] != "another description") QFAIL("read attribute (description) failed");
	QVERIFY(list[0] != list[1]);
}

//virtual ReturnValue(stringlist) list(QString filter = "(objectClass=*)", bool recursive = true) const;
void RdsLdapObjectTester::listTest()
{
	ReturnValue ret;
	RdsLdapObject obj1("ou=testarea,dc=resara,dc=local");

	QVERIFY2(!(ret = obj1.list()).isError(), qPrintable(ret.errString()));
	QStringList list = ret.toStringList();
// 	for (int i = 0; i < list.size() && i < 100; ++i)
// 		qDebug() << i << list[i];
	QCOMPARE(list.size(), 5);
	QVERIFY(list.contains("CN=depthdude,OU=depthou,OU=testarea,DC=resara,DC=local"));
	QVERIFY(list.contains("OU=depthou,OU=testarea,DC=resara,DC=local"));
	QVERIFY(list.contains("CN=normal,OU=testarea,DC=resara,DC=local"));
	QVERIFY(list.contains("OU=testarea,DC=resara,DC=local"));
	QVERIFY(list.contains("CN=freak,OU=testarea,DC=resara,DC=local"));

	QVERIFY2(!(ret =  obj1.list("(cn=*)")).isError(), qPrintable(ret.errString()));
	list = ret.toStringList();
// 	for (int i = 0; i < list.size() && i < 100; ++i)
// 		qDebug() << i << list[i];
	QCOMPARE(list.size(), 3);
	QVERIFY(list.contains("CN=depthdude,OU=depthou,OU=testarea,DC=resara,DC=local"));
	QVERIFY(list.contains("CN=normal,OU=testarea,DC=resara,DC=local"));
	QVERIFY(list.contains("CN=freak,OU=testarea,DC=resara,DC=local"));

	QVERIFY2(!(ret =  obj1.list("(cn=*)", false)).isError(), qPrintable(ret.errString()));
	list = ret.toStringList();
	QCOMPARE(list.size(), 2);
	/*for (int i = 0; i < list.size() && i < 100; ++i)
		qDebug() << i << list[i];*/
	QVERIFY(list.contains("CN=freak,OU=testarea,DC=resara,DC=local"));
	QVERIFY(list.contains("CN=normal,OU=testarea,DC=resara,DC=local"));
}

//ReturnValue move(QString newparent);
void RdsLdapObjectTester::moveTest()
{
	TEST_FUNCTION_ERROR(obj.move("ou:testarea,dc=resara,dc=local"));
	ReturnValue ret = TEST_FUNCTION(obj.move("dc=resara,dc=local"));
	QCOMPARE(ret.toString(), QString("ou=newou,dc=resara,dc=local"));
	ret = TEST_FUNCTION(obj.move("ou=testarea,dc=resara,dc=local"));
	QCOMPARE(ret.toString(), QString("ou=newou,ou=testarea,dc=resara,dc=local"));
}

void RdsLdapObjectTester::streamTest()
{
	RdsLdapObject obj2;
	ReturnValue ret, ret2;
	QByteArray ar;
	{
		QDataStream stream(&ar, QIODevice::WriteOnly);
		stream << obj;
	}
	{
		QDataStream stream(ar);
		stream >> obj2;
	}

	QCOMPARE(obj.dn(), obj2.dn());
	QCOMPARE(obj.session(), obj2.session());
	QVERIFY2(!(ret = obj.list()).isError(), qPrintable(ret.errString()));
	QVERIFY2(!(ret2 = obj2.list()).isError(), qPrintable(ret.errString()));
	QCOMPARE(ret.toStringList(), ret2.toStringList());
}

void RdsLdapObjectTester::RemoveTest()
{
	RdsLdapObject obj2("ou=nonexistant,dc=resara,dc=local");
	RdsLdapObject obj3("ou:nonexistant,dc=resara,dc=local");
	RdsLdapObject obj4, obj5("ou=existant,dc=resara,dc=local");
	RdsLdapActions act;
	act.add(RdsLdapActions::Add, "ou", "existant");
	act.add(RdsLdapActions::Add, "objectclass", "organizationalUnit");
	QVERIFY(!obj5.add(act).isError());
	QVERIFY(obj2.remove().isError());
	QVERIFY(obj3.remove().isError());
	QVERIFY(obj4.remove().isError());
	QVERIFY(!obj5.remove().isError());
	QVERIFY(obj5.remove().isError());
	QCOMPARE(obj5.dn(), QString("ou=existant,dc=resara,dc=local"));
}
