/* writeris.c: functions to write RIS datasets and to create DocBook bibliography entries */
/* markus@mhoenicka.de 4-28-00 */

/*
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <dbi/dbi.h>

#include "refdb.h"
#include "linklist.h"
#include "refdbd.h" /* depends on dbi.h */
#include "backend.h"
#include "writeris.h" 
#include "strfncs.h"
#include "xmlhelper.h"
#include "backend-scrn.h"
#include "backend-ris.h"
#include "backend-risx.h"
#include "backend-db31.h"
#include "backend-teix.h"
#include "backend-bibtex.h" /* depends on refdbd.h */
#include "backend-html.h"
#include "backend-dbib.h"
#include "backend-citationlistx.h"
#include "backend-mods.h"

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  render_ref(): formats the result of a query

  int render_ref returns 0 if successful, > 0 if failed

  struct renderinfo* ptr_rendinfo ptr to a structure with the info
                             how the reference should be rendered

 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int render_ref(struct renderinfo* ptr_rendinfo) {
  int retval;

  if (ptr_rendinfo->ref_format == REFRIS || ptr_rendinfo->ref_format == REFCGIRIS) { /* RIS or cgi-ris*/
    retval = render_ris(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == RISX) { /* RISX */
    retval = render_risx(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == REFMODS) { /* MODS */
    retval = render_mods(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == REFDOCBK
	   || ptr_rendinfo->ref_format == REFDOCBKX) { /* DocBook < 5 */
    retval = render_db31(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == REFDOCBKX5) { /* DocBook >= 5 */
    retval = render_db50(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == REFTEIX) { /* TEI P4 XML */
    retval = render_teix(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == REFTEIX5) { /* TEI P5 XML */
    retval = render_teix5(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == REFBIBTEX) { /* bibtex */
    retval = render_bibtex(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == REFHTML
	   || ptr_rendinfo->ref_format == REFXHTML
	   || ptr_rendinfo->ref_format == REFCGIHTML) { /* (x)html or cgi*/
    retval = render_html(ptr_rendinfo, 0 /* permanent tables */);
  }
  else if (ptr_rendinfo->ref_format == REFCITATIONLISTX) { /* citationlistx */
    retval = render_citationlistx(ptr_rendinfo);
  }
  else { /* screen output; make this default, just in case */
    retval = render_scrn(ptr_rendinfo);
  }

  return retval;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  prepare_render(): writes a header for the output of a query

  int prepare_render returns 0 if successful, > 0 if failed

  struct renderinfo* ptr_rendinfo ptr to a structure with the info
                             how the reference should be rendered

 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int prepare_render(struct renderinfo* ptr_rendinfo) {
  int retval;
  if (ptr_rendinfo->ref_format == REFRIS || ptr_rendinfo->ref_format == REFCGIRIS) {
    retval = prepare_render_ris(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == REFDOCBK
	   || ptr_rendinfo->ref_format == REFDOCBKX) {
    retval = prepare_render_db31(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == REFDOCBKX5) { /* DocBook >= 5 */
    retval = prepare_render_db50(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == RISX) { /* RISX */
    retval = prepare_render_risx(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == REFMODS) { /* MODS */
    retval = prepare_render_mods(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == REFTEIX) { /* TEI P 4XML */
    retval = prepare_render_teix(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == REFTEIX5) { /* TEI P5 XML */
    retval = prepare_render_teix5(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == REFBIBTEX) {
    retval = prepare_render_bibtex(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == REFHTML       /* plain html/xhtml */
	   || ptr_rendinfo->ref_format == REFXHTML   /* or html via cgi */
	   || ptr_rendinfo->ref_format == REFCGIHTML) { 
    retval = prepare_render_html(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == REFCITATIONLISTX) { /* citationlistx */
    retval = prepare_render_citationlistx(ptr_rendinfo);
  }
  else { /* default */
    retval = prepare_render_scrn(ptr_rendinfo);
  }

  return retval;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  finish_render(): writes a footer for the output of a query

  int finish_render returns 0 if successful, > 0 if failed

  struct renderinfo* ptr_rendinfo ptr to a structure with the info
                             how the reference should be rendered

 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int finish_render(struct renderinfo* ptr_rendinfo) {
  int retval = 0;

  if(ptr_rendinfo->ref_format == REFRIS || ptr_rendinfo->ref_format == REFCGIRIS) {
    retval = finish_render_ris(ptr_rendinfo);
  }
  else if(ptr_rendinfo->ref_format == REFDOCBK
	  || ptr_rendinfo->ref_format == REFDOCBKX) {
    retval = finish_render_db31(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == REFDOCBKX5) { /* DocBook >= 5 */
    retval = finish_render_db50(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == REFTEIX) {
    retval = finish_render_teix(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == REFTEIX5) { /* TEI P5 XML */
    retval = finish_render_teix5(ptr_rendinfo);
  }
  else if(ptr_rendinfo->ref_format == RISX) {
    retval = finish_render_risx(ptr_rendinfo);
  }
  else if(ptr_rendinfo->ref_format == REFMODS) {
    retval = finish_render_mods(ptr_rendinfo);
  }
  else if(ptr_rendinfo->ref_format == REFBIBTEX) {
    retval = finish_render_bibtex(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == REFHTML
	   || ptr_rendinfo->ref_format == REFXHTML
	   || ptr_rendinfo->ref_format == REFCGIHTML) { /* html or cgi */
    retval = finish_render_html(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format == REFCITATIONLISTX) { /* citationlistx */
    retval = finish_render_citationlistx(ptr_rendinfo);
  }
  else if (ptr_rendinfo->ref_format != 5) { /* default */
    retval = finish_render_scrn(ptr_rendinfo);
  }

  return retval;
}



