/***************************************************************************
    file	         : kb_querylog.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>

#include	<qwidget.h>
#include	<qdatetime.h>


#include	"kb_classes.h"
#include	"kb_location.h"
#include	"kb_gui.h"
#include	"kb_node.h"
#include	"kb_notifier.h"
#include	"kb_viewer.h"
#include	"kb_options.h"
#include	"kb_select.h"
#include	"kbase.h"



#ifndef 	_WIN32
#include	"kb_querylog.moc"
#else
#include	"kb_querylog.h"
#endif




static	QString	timeNow ()
{
	QTime	now	= QTime::currentTime() ;
	return	QString().sprintf("%02d:%02d:%02d", now.hour(), now.minute(), now.second()) ;
}


static	GUIElement	queryLogGUI[] =
{
{	GTAction,	KB::GRNone,	true,	"Clear log",		"editclear",	ACCEL(NoAccel),	SLOT(clearLog ()),"KB_clear",	0,	"Clear log"		},
{	GTNone,		KB::GRNone,	false,	0,			0,		ACCEL(NoAccel),	0,		0,		0,	0			}
}	;


/*  ------------------------------------------------------------------  */

/*  KBQueryLogEntry							*/
/*  KBQueryLogEntry							*/
/*		: Constructor for query log entry			*/
/*  after	: QListViewItem * : Insert after			*/
/*  parent	: QListView *	  : Parent list view			*/
/*  count	: uint		  : Query count				*/
/*  server	: const QString & : Name of server			*/
/*  text	: const QString & : Query text				*/
/*  rc		: bool		  : Query execution succeeded/failed	*/
/*  argc	: uint		  : Number of placeholder values	*/
/*  argv	: const KBValue * : Vector of placeholder values	*/
/*  (returns)	: KBQueryLogEntry :					*/

KBQueryLogEntry::KBQueryLogEntry
	(	QListView	*parent,
		QListViewItem	*after,
		uint		count,
		const QString	&server,
		const QString	&text,
		bool		rc,
		uint		argc,
		const KBValue	*argv
	)
	:
	QListViewItem
	(	parent,
		after,
		QString("%1").arg(count),
		timeNow(),
		server,
//		QString(text).replace("\r", " ").replace("\n", " "),
		text,
		rc ? "1" : "0"
	),
	m_extra	(false)
{
	if ((argc > 0) && (argv != 0))
	{
		/* Limit the number of arguments saved to the limit	*/
		/* but set the extra flag to note when there were more.	*/
		if (argc > KBOptions::getLogMaxArgs())
		{	argc	= KBOptions::getLogMaxArgs() ;
			m_extra	= true	  ;
		}

		/* For each argument, store the raw text (truncated if	*/
		/* they are too long) and the associated type.		*/
		for (uint idx = 0 ; idx < argc ; idx += 1)
		{
			QString	s  = argv[idx].getQueryText() ;
			if (s.length() > KBOptions::getLogMaxArgLen())
			{	s.truncate(KBOptions::getLogMaxArgLen()) ;
				s.append  (" ...."  ) ;
			}
			m_args .append (s) ;
			m_types.append (argv[idx].getType()->getDescrip()) ;
		}
	}
}

/*  KBQueryLogEntry							*/
/*  showArgs	: Show arguments					*/
/*  lv		: QListView *	: List view into which to display	*/
/*  (returns)	: void		:					*/

void	KBQueryLogEntry::showArgs
	(	QListView	*lv
	)
{
	lv->clear	()   ;
	lv->setSorting	(-1) ;

	QListViewItem *last  = 0 ;

	for (uint arg = 0 ; arg < m_args.count() ; arg += 1)
		last = new QListViewItem
				(	lv,
					last,
					QString("%1").arg(arg),
					m_types[arg],
					m_args [arg]
				)	;

	if (m_extra)
		last = new QListViewItem (lv, last, "", "....", "....") ;
}



/*  ------------------------------------------------------------------  */

/*  KBQueryLog								*/
/*  KBQueryLog	: Constructor for text logging windows			*/
/*  parent	: QWidget *	  : Parent widget			*/
/*  caption	: const QString & : Caption				*/
/*  action	: KToggleAction * : Show/hide action for this window	*/
/*  (returns)	: KBQueryLog	  :					*/

KBQueryLog::KBQueryLog
	(	QWidget		*parent,
		const QString	&caption,
		TKToggleAction	*action
	)
	:
	KBasePart (0, parent, WDestructiveClose|WStyle_NormalBorder),
	m_split	  (new QSplitter (Qt::Vertical, m_partWidget)),
	m_query	  (new QTextView (m_split)),
	m_args	  (new QListView (m_split)),
	m_log	  (new QListView (m_split)),
	m_action  (action)
{
	m_topWidget = m_split ;

	m_query->setTextFormat  (Qt::RichText) ;

	m_args ->addColumn	("",	      60) ;
	m_args ->addColumn      ("Type",      60) ;
	m_args ->addColumn	("Argument", 400) ; 
	m_args ->setSorting	(-1) ;

	m_log  ->addColumn      ("Count",     60) ;
	m_log  ->addColumn      ("Time",      60) ;
	m_log  ->addColumn      ("Server",    60) ;
	m_log  ->addColumn      ("Query",    400) ;
	m_log  ->setSorting	(-1) ;

	m_split->show		() ;

	TKConfig *config = TKConfig::getConfig() ;
	config->setGroup ("Query Log") ;

	QSize		size	= config->readSizeEntry   ("geometry" ) ;
	QValueList<int> depths	= config->readIntListEntry("depths"   ) ;
	QValueList<int>	aw	= config->readIntListEntry("argWidths") ;
	QValueList<int>	lw	= config->readIntListEntry("logWidths") ;

	if (size.isEmpty()) size = QSize(500, 500) ;
	while (depths.count() < 3) depths.append(0) ;
	if (depths[0] == 0) depths[0] = 300 ;
	if (depths[1] == 0) depths[1] = 100 ;
	if (depths[2] == 0) depths[2] = 100 ;

	setLocalGUISpec (&queryLogGUI[0])   ;
	setGUI (m_gui = new KBaseGUI (this, this, "rekallui.text.log")) ;

#if	__KB_EMBEDDED
	m_partWidget->showMaximized () ;
#else
	m_partWidget->resize	(size.width(), size.height(), true, false) ;
#endif
	m_partWidget->show   	()	   ;
	m_partWidget->setCaption(caption ) ;

	if (aw.count() >= 3)
	{	m_args->setColumnWidth (0, aw[0]) ;
		m_args->setColumnWidth (1, aw[1]) ;
		m_args->setColumnWidth (2, aw[2]) ;
	}
	if (lw.count() >= 4)
	{	m_log ->setColumnWidth (0, lw[0]) ;
		m_log ->setColumnWidth (1, lw[1]) ;
		m_log ->setColumnWidth (2, lw[2]) ;
		m_log ->setColumnWidth (3, lw[3]) ;
	}

	extern	KBaseApp *kbaseApp	 ;
	m_split	  ->setSizes	(depths) ;
	kbaseApp  ->addViewer	(this  ) ;

	connect
	(	m_log,
		SIGNAL(doubleClicked(QListViewItem *)),
		SLOT  (clickQuery   (QListViewItem *))
	)	;

	m_count	= 0  ;
	m_last	= 0  ;
}

/*  KBQueryLog								*/
/*  ~KBQueryLog	: Destructor for text logging window			*/
/*  (returns)	:		:					*/

KBQueryLog::~KBQueryLog ()
{
}

/*  KBQueryLog								*/
/*  logQuery	: Add query to log					*/
/*  server	: const QString & : Server name				*/
/*  text	: const QString & : Text				*/
/*  rc		: bool		  : Success/failure flag		*/ 
/*  argc	: uint		  : Number of arguments			*/
/*  argv	: const KBValue * : Vector of arguments			*/
/*  (returns)	: void		  :					*/

void	KBQueryLog::logQuery
	(	const QString	&server,
		const QString	&text,
		bool		rc,
		uint		argc,
		const KBValue	*argv
	)
{
	while (m_log->childCount() >= (int)KBOptions::getLogMaxQueries())
		delete m_log->firstChild() ;

	m_count += 1 ;
	m_last	 = new KBQueryLogEntry
		       (	m_log,
				m_last,
				m_count,
				server,
				text,
				rc,
				argc, argv
		       ) ;
}

/*  KBQueryLog								*/
/*  queryClose	: Close handler						*/
/*  (returns)	: void			:				*/

bool	KBQueryLog::queryClose ()
{
	TKConfig	*config = TKConfig::getConfig () ;
	QValueList<int>	aw	;
	QValueList<int>	lw	;

	aw.append (m_args->columnWidth(0)) ;
	aw.append (m_args->columnWidth(1)) ;
	aw.append (m_args->columnWidth(2)) ;

	lw.append (m_log ->columnWidth(0)) ;
	lw.append (m_log ->columnWidth(1)) ;
	lw.append (m_log ->columnWidth(2)) ;
	lw.append (m_log ->columnWidth(3)) ;

	config->setGroup   ("Query Log") ;
	config->writeEntry ("geometry",  m_partWidget->size ()) ;
	config->writeEntry ("depths",    m_split     ->sizes()) ;
	config->writeEntry ("argWidths", aw) ;
	config->writeEntry ("logWidths", lw) ;
	config->sync	   () ;


	m_action->setChecked (false) ;
	return	true	;
}

/*  KBQueryLog								*/
/*  clickQuery	: User clicks query					*/
/*  item	: QListViewItem *	: Item				*/
/*  (returns)	: void			:				*/

void	KBQueryLog::clickQuery
	(	QListViewItem	*item
	)
{
	if (item == 0) return ;

	KBQueryLogEntry		*log	= (KBQueryLogEntry *)item ;
	QString			query	= log->getQuery() ;

	KBSelect		select	;

	if (select.parseQuery (query))
		m_query->setText (select.getPrettyText(false, 0)) ;
	else	m_query->setText (query + "<br/><i>(" + select.lastError().getMessage() + ")</i>") ;

//	KBQryParser		parser	(select) ;
//
//	if (parser.parseQuery (query))
//		m_query->setText (parser.getPrettyText()) ;
//	else	m_query->setText (query + "<br/><i>(" + parser.lastError() + ")</i>") ;

	log->showArgs (m_args) ;
}

void	KBQueryLog::clearLog ()
{
	while (m_log ->childCount() > 0) delete m_log ->firstChild () ;
	while (m_args->childCount() > 0) delete m_args->firstChild () ;
	m_query->setText(QString::null) ;
	m_last	= 0 ;
}

