/***************************************************************************
    file	         : kb_datetime.cpp
    copyright            : (C) 1999,2000,2001,2002,2003,2004 by Mike Richardson
			   (C) 2000,2001,2002,2003,2004 by theKompany.com
			   (C) 2001,2002,2003,2004 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/


#include	<stdio.h>
#include	<stdlib.h>
#include	<ctype.h>

#include	<qintdict.h>
#include	<qlist.h>
#include 	<qregexp.h>

#ifdef _WIN32

int 	strcasecmp
	(	const char	*s1,
		const char	*s2
	)
{
	while (*s1 && *s2)
	{
		char ch1 = *s1++ ;
		char ch2 = *s2++ ;
		if (isupper(ch1)) ch1 = tolower(ch1) ;
		if (isupper(ch2)) ch2 = tolower(ch2) ;
		if (ch1 != ch2) return ch1 - ch2 ;
	}

	return *s1 - *s2 ;
}

int	strncasecmp
	(	const char	*s1,
		const char	*s2,
		size_t		len
	)
{
	if(!len)
		return(0);

	while(--len && *s1 && *s2)
	{
		char ch1 = *s1++;
		char ch2 = *s2++;

		if(isupper(ch1))
			ch1 = tolower(ch1);

		if(isupper(ch2))
			ch2 = tolower(ch2);

		if(ch1 != ch2)
			return ch1 - ch2;
	}

	return *s1 - *s2;
}

#endif

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"


#if		__KB_KDE

#include	<klocale.h>
#include	<kglobal.h>

static	KLocale	*locale	;

#endif		// __KB_KDE


struct	LIBCOMMON_API	DTItem
{	int	fchar	;
	int	offset	;
	int	rem	;
	cchar	*fmt	;
}	;

#define	DT_YEAR		0
#define	DT_MONTH	1
#define	DT_DAY		2
#define	DT_HOUR		3
#define	DT_MIN		4
#define	DT_SEC		5
#define	DT_YDAY		6
#define	DT_WDAY		7
#define	DT_SIZE		8

#define	DT_FORMAT	99

static	DTItem	dtItems[] =
{
	{	'C',	DT_YEAR,	-100,	"%02d"			},
	{	'd',	DT_DAY,		0,	"%02d"			},
	{	'e',	DT_DAY,		0,	"%d"			},
	{	'H',	DT_HOUR,	0,	"%02d"			},
	{	'I',	DT_HOUR,	12,	"%02d"			},
	{	'j',	DT_YDAY,	0,	"%03d"			},
	{	'k',	DT_HOUR,	0,	"%2d"			},
	{	'l',	DT_HOUR,	12,	"%2d"			},
	{	'm',	DT_MONTH,	0,	"%02d"			},
	{	'M',	DT_MIN,		0,	"%02d"			},
	{	'S',	DT_SEC,		0,	"%02d"			},
	{	'u',	DT_WDAY,	0,	"%d"			},
	{	'y',	DT_YEAR,	100,	"%02d"			},
	{	'Y',	DT_YEAR,	0,	"%04d"			},

	{	'c',	DT_FORMAT,	0,	"%x %X",		},
	{	'D',	DT_FORMAT,	0,	"%m/%d/%y"		},
	{	'r',	DT_FORMAT,	0,	"%I:%M:%S %p"		},
	{	'R',	DT_FORMAT,	0,	"%H:%M"			},
	{	'T',	DT_FORMAT,	0,	"%H:%M:%S"		},
	{	'+',	DT_FORMAT,	0,	"%a %b %e %T %Z %Y"	}
}	;

static	QIntDict<DTItem>	dtMap	;

/*  The following table lists the weekdays, the months, plus some other	*/
/*  things than can appear in dates. Note that we should do something	*/
/*  about making this language dependant.				*/

static	cchar	*names[] =
{	"sunday",
	"monday",
	"tuesday",
	"wednesday",
	"thursday",
	"friday",
	"saturday",
	"january",
	"february",
	"march",
	"april",
	"may",
	"june",
	"july",
	"august",
	"september",
	"october",
	"november",
	"december",
	"am",
	"pm",
	"gmt",
	"pst",
	"bst",
	"edt",
	"st",
	"nd",
	"rd",
	"th",
	0
}	;
#define	NUMDAYS	7
#define	NUMMONS	12
#define	NUMISAM	(NUMDAYS+NUMMONS+0)
#define	NUMISPM	(NUMDAYS+NUMMONS+1)

static	int	mDays [] =
{	31,    28,    31,    30,    31,    30,
	31,    31,    30,    31,    30,    31
}	;


static	KBDateTime::DateOrder	dateOrder = KBDateTime::DayMonthYear ;



class 	LIBCOMMON_API	DTToken
{
public	:
	int	sep	;
	QString	token	;
	bool	digit	;
	int	nidx	;
}	;


/*  getName	: Locate name entry for specified string		*/
/*  str		: cchar *	: String				*/
/*  slen	: uint		: Number of relevant characters		*/
/*  (returns)	: int		: Index in name list or negative	*/

static	int	getName
	(	cchar	*str,
		uint	slen
	)
{
	cchar	**nptr	;

	for (nptr = &names[0] ; *nptr != 0 ; nptr += 1)
	{
		cchar	*nam	= *nptr	      ;
		int	nlen	= strlen(nam) ;

		/* If the name is less than three charactes then do a	*/
		/* length-dependant match. This distinguishes between,	*/
		/* say, "thursday" and "th".				*/
		if (nlen < 3)
		{	if (strcasecmp (str, nam) == 0) break ;
			continue ;
		}

		if ((slen >= 3) && (strncasecmp (str, nam, slen) == 0))
			break	;
	}

	return	(*nptr == 0) ? -1 : nptr - &names[0] ;
}

/*  fixYear	: Decode and fix year string				*/
/*  tokp	: DTToken	: Token					*/
/*  (returns)	: int		: Year as a number			*/

static	int	fixYear
	(	DTToken	*tokp
	)
{
	int	yno	= tokp->token.toInt() ;
	if (tokp->token.length() == 2)
		if	(yno >= 70) yno += 1900 ;
		else if (yno <  70) yno += 2000 ;
	return	yno	;
}


#if	__KB_KDE

/*  weekDayName	: Get name of day of week				*/
/*  idx		: int		: Day number				*/
/*  abbrev	: bool		: Return abbreviated form		*/
/*  (returns)	: QString	: Name					*/

static	QString	weekDayName
	(	int		idx,
		bool		abbrev
	)
{
	return	locale->weekDayName (idx, abbrev) ;
}

/*  monthName	: Get name of month					*/
/*  idx		: int		: Month number				*/
/*  abbrev	: bool		: Return abbreviated form		*/
/*  (returns)	: QString	: Name					*/

static	QString	monthName
	(	int		idx,
		bool		abbrev
	)
{
	return	locale->monthName (idx, abbrev) ;
}

/*  formatDate	: Get specified date in default format			*/
/*  date	: const QDate &	: Date					*/
/*  (returns)	: QString	: Formatted version			*/

static	QString	formatDate
	(	const QDate	&date
	)
{
	return	locale->formatDate (date) ;
}

/*  formatTime	: Get specified time in default format			*/
/*  time	: const QTime &	: Time					*/
/*  (returns)	: QString	: Formatted version			*/

static	QString	formatTime
	(	const QTime	&time
	)
{
	return	locale->formatTime (time) ;
}

#endif	// __KB_KDE

#if	__KB_TKC

/*  weekDayName	: Get name of day of week				*/
/*  idx		: int		: Day number				*/
/*  abbrev	: bool		: Return abbreviated form		*/
/*  (returns)	: QString	: Name					*/

static	QString	weekDayName
	(	int		idx,
		bool		abbrev
	)
{
	static	const char	*dayFull[] =
	{	0,
		"Monday",
		"Tuesday",
		"Wednesday",
		"Thursday",
		"Friday",
		"Saturday",
		"Sunday"
	}	;
	static	const char	*dayAbbrev[] =
	{	0,
		"Mon",
		"Tue",
		"Wed",
		"Thu",
		"Fri",
		"Sat",
		"Sun"
	}	;

	if (idx < 1) return "???" ;
	if (idx > 7) return "???" ;

	return	abbrev ? dayAbbrev[idx] : dayFull[idx] ;
}

/*  monthName	: Get name of month					*/
/*  idx		: int		: Month number				*/
/*  abbrev	: bool		: Return abbreviated form		*/
/*  (returns)	: QString	: Name					*/

static	QString	monthName
	(	int		idx,
		bool		abbrev
	)
{
	static	const char	*monFull[] =
	{	0,
		"January",
		"February",
		"March",
		"April",
		"May",
		"June",
		"July",
		"August",
		"September",
		"October",
		"November",
		"December"
	}	;
	static	const char	*monAbbrev[] =
	{	0,
		"Jan",
		"Feb",
		"Mar",
		"Apr",
		"May",
		"Jun",
		"Jul",
		"Aug",
		"Sep",
		"Oct",
		"Nov",
		"Dec",
	}	;

	if (idx <  1) return "???" ;
	if (idx > 12) return "???" ;

	return	abbrev ? monAbbrev[idx] : monFull[idx] ;
}

/*  formatDate	: Get specified date in default format			*/
/*  date	: const QDate &	: Date					*/
/*  (returns)	: QString	: Formatted version			*/

static	QString	formatDate
	(	const QDate	&date
	)
{
	return	QString().sprintf
		(	"%04d-%02d-%02d",
			date.year  (),
			date.month (),
			date.day   ()
		)	;
}

/*  formatTime	: Get specified time in default format			*/
/*  time	: const QTime &	: Time					*/
/*  (returns)	: QString	: Formatted version			*/

static	QString	formatTime
	(	const QTime	&time
	)
{
	return	QString().sprintf
		(	"%02d:%02d:%02d",
			time.hour  (),
			time.minute(),
			time.second()
		)	;
}

#endif	// __KB_TKC

/*  KBDateTime								*/
/*  KBDateTime	: Constructor for date-time object			*/
/*  _dtstr	: const QString & : Date/Time string			*/
/*  (returns)	: KBDateTime	  :					*/

KBDateTime::KBDateTime
	(	const QString	&_dtstr
	)
{
#define	SF(f,v)	\
	{	if (f != -1) return ;	\
		f = v ;			\
	}

	int		year	= -1	;
	int		mon	= -1	;
	int		day	= -1	;
	int		hour	= -1	;
	int		min	= -1	;
	int		sec	= -1	;
	int		ampm	= -1	;
	int		lch	= ' '	;
	int		left	;
	QList<DTToken>	tokens	;
	DTToken		*tokp	;

//	fprintf
//	(	stderr,
//		"KBDateTime::KBDateTime: [%s]\n",
//		(cchar *)_dtstr
//	)	;

	/* Duplicate the date/time string and store it as the raw data.	*/
	/* This copy is also used in the parser below, but is left	*/
	/* unchanged.							*/
	const char 	*copy  = strdup (_dtstr)   ;
	const char	*dtstr = copy	    	   ;
	m_raw.assign	(copy, strlen (copy ))	;

	m_valid		= false	    ;
	m_hasDate	= false	    ;
	m_hasTime	= false	    ;
	tokens.setAutoDelete (true) ;

	while (*dtstr)
	{
		cchar	*tok	;

		if (!isalnum(*dtstr))
		{	lch	= *dtstr;
			dtstr    += 1	;
			continue	;
		}

		tokens.append (tokp = new DTToken) ;

		tokp->sep	= lch	;
		tokp->nidx	= 0	;
		tokp->digit	= isdigit(*dtstr)  ;
		tok		= dtstr	;

		while (tokp->digit ? isdigit(*dtstr) : isalpha(*dtstr))
			dtstr += 1 ;

		tokp->token	= QString(tok).left(dtstr - tok) ;

		if (!tokp->digit)
		{
			if ((tokp->nidx = getName(tok, dtstr - tok)) < 0)
				return	;
		}
		else	tokp->nidx = atol (tok) ;
	}

//	fprintf
//	(	stderr,
//		"KBDateTime::KBDateTime: [%s] -> %d tokens\n",
//		(cchar *)_dtstr,
//		tokens.count()
//	)	;
//	LITER
//	(	DTToken,
//		tokens,
//		tokp,
//		fprintf (stderr, " .... %c %s %d\n", tokp->sep, (cchar *)tokp->token, tokp->nidx) ;
//	)

	/* Special case: this is really a hack for XBase files, where	*/
	/* dates are stored as YYYYMMDD, and there is no time component	*/
	if (tokens.count() == 1)
	{
		static	QRegExp	xbdate	("^[0-9][0-9][0-9][0-9][0-9][0-9]$") ;

		const QString	&dstr	= tokens.at(0)->token ;

		if (xbdate.match (dstr))
		{
			year	= dstr.mid(0, 4).toInt() ;
			mon	= dstr.mid(4, 2).toInt() ;
			day	= dstr.mid(6, 2).toInt() ;

			m_dt.setDate (QDate(year, mon, day)) ;
			m_dt.setTime (QTime()) ;

			m_hasDate = true  ;
			m_hasTime = false ;
			m_valid	  = true  ;
			return	;
		}

		return	;
	}

	while ((left = tokens.count()) > 0)
	{
		DTToken	*tok0	= tokens.at(0) ;
		DTToken	*tok1	= left > 1 ? tokens.at(1) : 0 ;
		DTToken	*tok2	= left > 2 ? tokens.at(2) : 0 ;
		DTToken	*tok3	= left > 3 ? tokens.at(3) : 0 ;
		uint	drop	= 0 ;

//		fprintf
//		(	stderr,
//			" .... token [%s] with %d remaining\n",
//			(cchar *)tok0->token,
//			left
//		)	;

		/* First set of cases are where there are three tokens	*/
		/* available. Look for dates, and times where the	*/
		/* number of seconds is set.				*/
		if (tok2 != 0)
		{
			/* NN:NN:NN					*/
			/* Decode this as a time with the value order	*/
			/* of (hour, minute, second).			*/
			if	(tok0->digit &&  tok1->digit && tok2->digit
					&& (tok1->sep == ':') && (tok2->sep == ':'))
			{
				SF(hour, tok0->nidx) ;
				SF(min,  tok1->nidx) ;
				SF(sec,  tok2->nidx) ;

				if ((tok3 != 0) && (tok3->sep == '.'))
					drop	= 4  ;
				else	drop	= 3  ;
				goto	advance	;
			}
			/* NN-CCC-NN					*/
			/* This is a date in order (day,month,year)	*/
			/* where the month has been named.		*/
			else if (tok0->digit && !tok1->digit && tok2->digit
					&& (tok1->sep == '-') && (tok2->sep == '-'))
			{
				
				SF(day,  tok0->nidx   ) ;
				SF(mon,  tok1->nidx - NUMDAYS + 1) ;
				SF(year, fixYear(tok2)) ;
				drop	= 3	;
				goto	advance	;
			}
			/* NN-NN-NN					*/
			/* NN/NN/NN					*/
			/* NN.NN.NN					*/
			/* Decode thiese as a date. If the first token	*/
			/* is more then two characters long then treat	*/
			/* it as (year,month, day). Otherwise, treat as	*/
			/* (day,month,year) or (month,day,year)		*/
			/* depending on the default date ordering.	*/
			else if (tok0->digit &&  tok1->digit && tok2->digit
					&& (tok1->sep == tok2->sep)
					&& ( (tok1->sep == '-') ||
					     (tok1->sep == '/') ||
					     (tok1->sep == '.') ) )
			{
				if	(tok0->token.length() > 2)
				{
					SF(year, fixYear(tok0)) ;
					SF(mon,  tok1->nidx   ) ;
					SF(day,  tok2->nidx   ) ;
				}
				else if	(dateOrder == KBDateTime::DayMonthYear)
				{
					SF(year, fixYear(tok2)) ;
					SF(mon,  tok1->nidx   ) ;
					SF(day,  tok0->nidx   ) ;
				}
				else
				{
					SF(year, fixYear(tok2)) ;
					SF(mon,  tok0->nidx   ) ;
					SF(day,  tok1->nidx   ) ;
				}

				drop	= 3	;
				goto	advance	;
			}
		}

		/* Not three tokens or nothing matched, next check for	*/
		/* a two-token case, which can only be a time with the	*/
		/* seconds unspecified (and interpreted as zero).	*/
		if (tok1 != 0)
			if	(tok0->digit && tok1->digit && (tok1->sep == ':'))
			{
				SF(hour, tok0->nidx) ;
				SF(min,  tok1->nidx) ;
				SF(sec,  0) ;
				drop	= 2	;
				goto	advance ;
			}

		/* Allow three and four digit tokens which have a	*/
		/* leading + or -; these are GMT offsets. For the	*/
		/* moment, just skip them.				*/
		if ((tok0->token.length() >= 3) && (tok0->token.length() <= 4))
			if ((tok0->sep == '+') || (tok0->sep == '-'))
			{
				drop	= 1	;
				goto	advance	;
			}

		/* Only one token or again, no matches. Days are of	*/
		/* no use, but check for months or for AM/PM.		*/
		if (!tok0->digit)
		{
			if	(tok0->nidx <  NUMDAYS)
				;
			else if	(tok0->nidx <  NUMDAYS+NUMMONS)
				SF(mon,  tok0->nidx - NUMDAYS + 1)
			else if (tok0->nidx == NUMISAM)
				SF(ampm, NUMISAM)
			else if (tok0->nidx == NUMISPM)
				SF(ampm, NUMISPM)

			drop	= 1	;
			goto	advance	;
		}

		/* Two digit numbers are interpreted as day numbers, so	*/
		/* we can handle things like '11 July'.			*/
		if (tok0->token.length() <= 2)
		{
			if ((tok0->sep != '+') && (tok0->sep != '-'))
				SF(day,  tok0->nidx) ;
			drop	= 1	;
			goto	advance	;
		}

		/* Four digit tokens can only be years ...		*/
		if ((tok0->token.length() <= 4) && tok0->digit)
		{
			SF(year, tok0->nidx) ;
			drop	= 1	;
			goto	advance	;
		}

		/* ... and anything else ends the scan.			*/
//		fprintf
//		(	stderr,
//			"KBDateTime::KBDateTime: fail on [%s]\n",
//			(cchar *)tok0->token
//		)	;
		return	;

		advance	:
			while (drop > 0)
			{	tokens.remove ((uint)0) ;
				drop	-= 1 ;
			}
	}

//	fprintf
//	(	stderr,
//		" .... decoded: hour=%d ampm=%d\n",
//		hour,
//		ampm
//	)	;

	if (hour != -1)
	{	if (ampm != 	  -1) if (hour >= 12) return ;
		if (ampm ==  NUMISPM) hour += 12 ;
	}


//	fprintf
//	(	stderr,
//		" .... decoded: %d-%d-%d %d:%d:%d\n",
//		year, mon, day,
//		hour, min, sec
//	)	;

	if ((year != -1) && (mon != -1) && (day != -1))
	{
		if ((year < 1752) || (year > 8000)) return ;
		if ((mon  <    1) || (mon  >   12)) return ;
		if ( day  <    1		  ) return ;

		mDays[1] = year % 4   != 0 ? 28 :
			   year % 400 == 0 ? 29 :
			   year % 100 == 0 ? 28 : 29 ;


		if ( day  > mDays[mon-1]) return ;

		m_dt.setDate (QDate(year, mon, day)) ;
		m_hasDate = true ;
	}
	else	m_dt.setDate (QDate()) ;

	if ((hour != -1) && (min != -1) && (sec != -1))
	{
		if (hour >= 24) return ;
		if (min  >= 60) return ;
		if (sec  >= 60) return ;

		m_dt.setTime (QTime(hour, min, sec)) ;
		m_hasTime = true ;
	}
	else	m_dt.setTime (QTime()) ;	

	m_valid	= m_hasDate || m_hasTime ;
}

KBDateTime::KBDateTime
	(	const QDateTime	&dt
	)
	:
	m_dt	(dt),
	m_valid	(dt.isValid())
{
	m_raw	= (QCString)defFormat(KB::ITDateTime) ;
}

/*  KBDateTime								*/
/*  format	: Format into a text string				*/
/*  _fmt	: const QString & : Format specification		*/
/*  (returns)	: QString	  : Text string				*/
	
QString	KBDateTime::format
	(	const QString	&_fmt
	)
	const
{
	if (dtMap.count() == 0)
	{
		for (uint idx = 0 ; idx < sizeof(dtItems)/sizeof(DTItem) ; idx += 1)
			dtMap.insert (dtItems[idx].fchar, &dtItems[idx]) ;

#if	__KB_KDE
		locale	= KGlobal::locale () ;

		/* Try to figure the default date order. Format a date	*/
		/* which can then be interpreted. If that does not	*/
		/* help then the US is MDY and everywhere else is DMY.	*/
		QDate	dTest (2000, 10, 1)  ;
		QString	dText = locale->formatDate (dTest, true) ;
		int	dayno = dText.left(2).toInt() ;

		if	(dayno == 1 )
		{
			dateOrder = DayMonthYear ;
		}
		else if (dayno == 10)
		{
			dateOrder = MonthDayYear ;
		}
		else
		{
			dateOrder = locale->country().lower() == "us" ? MonthDayYear : DayMonthYear ;
		}

		fprintf
		(	stderr,
			"Date Format: [%s] [%s]\n",
			(cchar *)dText,
			dateOrder == DayMonthYear ? "DMY" : "MDY"
		)	;
#endif
	}

	/* If the decoded date/time is marked as invalid then make do	*/
	/* with the original raw text.					*/
	if (!m_valid) return m_raw ;

	QString	res	;
	QString	tmp	;
	DTItem	*di	;
	int	idx	;
	char	fc	;

	int	dv[DT_SIZE] ;

	dv[DT_YEAR ]	= m_dt.date().year      () ;
	dv[DT_MONTH]	= m_dt.date().month     () ;
	dv[DT_DAY  ]	= m_dt.date().day       () ;	
	dv[DT_HOUR ]	= m_dt.time().hour      () ;	
	dv[DT_MIN  ]	= m_dt.time().minute    () ;	
	dv[DT_SEC  ]	= m_dt.time().second    () ;	
	dv[DT_YDAY ]	= m_dt.date().dayOfYear () ;		
	dv[DT_WDAY ]	= m_dt.date().dayOfWeek () ;

	for (cchar *fmt = _fmt ; *fmt != 0 ; fmt += 1)
	{
		if (*fmt != '%')
		{
			res.append (*fmt) ;
			continue ;
		}

		fmt	+= 1 ;
		if (*fmt == 0) break ;

		if ((di = dtMap.find (*fmt)) != 0)
		{
			if (di->offset == DT_FORMAT)
			{
				res	+= format (di->fmt) ;
				continue ;
			}

			int	v = dv[di->offset] ;
			if (di->rem > 0) v = v %  di->rem ;
			if (di->rem < 0) v = v / -di->rem ;
			tmp.sprintf (di->fmt, v) ;
			res += tmp ;
			continue   ;
		}

		switch (fc = *fmt)
		{
			case 0   :
				return	res	;

			case 'a' :
			case 'A' :
				idx	= m_dt.date().dayOfWeek () ;
				tmp	= weekDayName (idx, fc == 'a') ;
				break	;

			case 'b' :
			case 'h' :
			case 'B' :
				idx	= m_dt.date().month     () ;
				tmp    	= monthName   (idx, fc != 'B') ;
				break	;

			case 'D' :
				tmp	= format ("%m/%d/%y") ;
				break	;

			case 'n' :
				tmp	= "\n"	;
				break	;

			case 'p' :
				tmp	= m_dt.time().hour() >= 12 ? "PM" : "AM" ;
				break	;
				
			case 'P' :
				tmp	= m_dt.time().hour() >= 12 ? "pm" : "am" ;
				break	;

			case 't' :
				tmp	= "\t"	;
				break	;

			case 'x' :
				tmp	= formatDate (m_dt.date()) ;
				break	;

			case 'X' :
				tmp    += formatTime (m_dt.time()) ;
				break	;

			case '%' :
				tmp	= "%"	;
				break	;

			default	 :
				tmp	= ""	;
				break	;
		}

		res	+= tmp ;
	}

	return	res	;
}


/*  KBDateTime								*/
/*  defFormat	: Get date and time in default format 			*/
/*  iType	: KB::IType	: Required internal type		*/
/*  (returns)	: QString	: Formatted date and time		*/

QString	KBDateTime::defFormat
	(	KB::IType	iType
	)
	const
{
	/* Default format is ISO-8601, or parts thereof depending on	*/
	/* what is wanted.						*/

	/* If we do not have a valid decoded date/time then make do	*/
	/* with the raw string. Maybe the server can make more sense	*/
	/* of it.							*/
	if (!m_valid) return m_raw ;

	if (iType == KB::ITDate)
		return	QString().sprintf ("%04d-%02d-%02d",
					   m_dt.date().year  (),
					   m_dt.date().month (),
					   m_dt.date().day   ()) ;

	if (iType == KB::ITTime)
		return	QString().sprintf ("%02d:%02d:%02d",
					   m_dt.time().hour  (),
					   m_dt.time().minute(),
					   m_dt.time().second()) ;

	if (iType == KB::ITDateTime)
		return	QString().sprintf ("%04d-%02d-%02d %02d:%02d:%02d",
					   m_dt.date().year  (),
					   m_dt.date().month (),
					   m_dt.date().day   (),
					   m_dt.time().hour  (),
					   m_dt.time().minute(),
					   m_dt.time().second()) ;

	/* Gak! Should not get here!					*/
	KBError::EError
	(	TR("Unexpected request to KBDateTime::defFormat"),
		QString(TR("KBDateTime::defFormat(%1)")).arg((int)iType),
		__ERRLOCN
	)	;
	return	m_raw	;
}


void	KBDateTime::setDateOrder
	(	DateOrder	_dateOrder
	)
{
	dateOrder = _dateOrder	;
}
