/***************************************************************************
    file	         : kb_db.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>

#include	<qstring.h>
#include	<qstringlist.h>
#include	<qdict.h>
#include	<qasciidict.h>
#include	<qdir.h>
#include	<qregexp.h>


#include	"kb_classes.h"
#include	"kb_database.h"
#include	"kb_dbinfo.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_databuffer.h"
#include	"kb_callback.h"
#include	"kb_appptr.h"
#include	"kb_desktop.h"

#ifndef		_WIN32
#include	"kb_sshtunnel.h"
#endif

#include	"kb_locator.h"


#ifndef 	_WIN32
#define		LIBLOCN		"libkbase_common"
#define		LIBBASE		"libkbase"
#define		LIBEXTN		".so"
#else
#define		LIBLOCN		"kbase_common"
#define		LIBBASE		"kbase"
#define		LIBEXTN		".dll"
#endif


#ifndef 	_WIN32
#include	<sys/types.h>
#include	<sys/wait.h>
#include	<unistd.h>
#include	<signal.h>
#include	<errno.h>
#endif

/*  KBServer								*/
/*  KBServer	: Constructor for base database connection class	*/
/*  (returns)	: KBServer	 :					*/

KBServer::KBServer ()
{
	m_tableCache.setAutoDelete (true) ;

	m__conn		= false	;

	m_showAllTables	= false	;
	m_cacheTables	= false	;
	m_printQueries	= false	;
	m_pkReadOnly	= false	;
	m_fakeKeys	= false	;
	m_readOnly	= false	;
	m_dataCodec	= 0	;
	m_objCodec	= 0	;

#ifndef	_WIN32
	m_sshPID	= 0	;
	m_sshPort	= -1	;
#endif
}



/*  KBServer								*/
/*  KBServer	: Destructor for base database connection class		*/
/*  (returns)	:		:					*/

KBServer::~KBServer ()
{
#ifndef	_WIN32
	if (m_sshPID != 0)
	{
		fprintf
		(	stderr,
			"KBServer::~KBServer: killing tunnel %d/%d\n",
			m_sshPID,
			m_sshPort
		)	;

		kill	(m_sshPID, SIGKILL) ;
		sleep	(2) ;
		waitpid	(m_sshPID, 0, WNOHANG) ;

		m_sshPID  = 0  ;
		m_sshPort = -1 ;

		fprintf
		(	stderr,
			"KBServer::~KBServer: done\n"
		)	;
	}
#endif
}

/*  KBServer								*/
/*  connect	: Connect to RDBMS					*/
/*  svInfo	: KBServerInfo * : Server information			*/
/*  (returns)	: bool		 : Success				*/

bool	KBServer::connect
	(	KBServerInfo	*svInfo
	)
{
	m_serverName		= svInfo->serverName	() ;
	m_host	  		= svInfo->hostName	() ;
	m_user			= svInfo->userName	() ;
	m_password		= svInfo->password	() ;
	m_database		= svInfo->database	() ;
	m_port			= svInfo->portNumber	() ;
	m_showAllTables		= svInfo->showAllTables	() ;
	m_cacheTables		= svInfo->cacheTables	() ;
	m_printQueries		= svInfo->printQueries	() ;
	m_pkReadOnly		= svInfo->pkReadOnly	() ;
	m_fakeKeys		= svInfo->fakeKeys	() ;
	m_readOnly		= svInfo->readOnly	() ;
#ifndef	_WIN32
	m_sshTarget		= svInfo->sshTarget	() ;
#endif
	QString	dataCodec	= svInfo->dataEncoding	() ;
	QString	objCodec	= svInfo->objEncoding	() ;

	if (!dataCodec.isEmpty() && (dataCodec != "UTF8"))
	{
		m_dataCodec	= QTextCodec::codecForName(dataCodec) ;

		fprintf
		(	stderr,
			"KBServer::connect: dataCodec [%s]->[%p]\n",
			(cchar *)dataCodec,
			(void  *)m_dataCodec
		)	;

		if (m_dataCodec == 0)
		{
			m_lError = KBError
				   (	KBError::Error,
					QString	(TR("Cannot find data codec for encoding '%1'"))
						.arg(dataCodec),
					QString::null,
					__ERRLOCN
				   )	;
			return	false ;
		}
	}

	if (!objCodec.isEmpty() && (objCodec != "UTF8"))
	{
		m_objCodec	= QTextCodec::codecForName(objCodec) ;

		fprintf
		(	stderr,
			"KBServer::connect: objCodec [%s]->[%p]\n",
			(cchar *)objCodec,
			(void  *)m_objCodec
		)	;

		if (m_objCodec == 0)
		{
			m_lError = KBError
				   (	KBError::Error,
					QString	(TR("Cannot find object codec for encoding '%1'"))
						.arg(objCodec),
					QString::null,
					__ERRLOCN
				   )	;
			return	false ;
		}
	}

	return	doConnect (svInfo) ;
}

/*  KBServer								*/
/*  placeHolder	: Get default placeholder value for slot		*/
/*  slot	: uint		: Slot number				*/
/*  (returns)	: QString	: Placeholder text			*/

QString	KBServer::placeHolder
	(	uint
	)
{
	return	QString ("?") ;
}

/*  KBServer								*/
/*  keepsCase	: Check if server preserves name case			*/
/*  (returns)	: bool		: True if so				*/

bool	KBServer::keepsCase ()
{
	return	true	;
}

/*  KBServer								*/
/*  subPlaceList: Substitute placeholders				*/
/*  sql		: const QString & : SQL statement with placeholders	*/
/*  nvals	: uint		  : Number of substitution values	*/
/*  values	: KBValue *	  : Value list				*/
/*  buffer	: KBDataBuffer  & : Result buffer			*/
/*  codec	: QTextCodec *	  : Non-default codec			*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBServer::subPlaceList
	(	const QString	&sql,
		uint		nvals,
		const KBValue	*values,
		KBDataBuffer	&buffer,
		QTextCodec	*codec,
		KBError		&pError
	)
{
	/* This version of KBServer::subPlaceList is the real thing,	*/
	/* and is used to do real substitution to generate the query to	*/
	/* present to the server. Here, data is put into the query	*/
	/* string in encoded (utf8), and possibly escaped, format.	*/
	uint	idx1	= 0	;
	bool	quote	= false	;
	uint	_nvals	= nvals	;

	QRegExp	seps	("['?]");

	while (idx1 < sql.length())
	{
		int	idx2 = sql.find (seps, idx1) ;

		if (idx2 < 0)
		{
			buffer.append (sql.mid (idx1)) ;
			break	;
		}

		buffer.append (sql.mid (idx1, idx2 - idx1)) ;

		QChar	ch = sql.at(idx2) ;
		idx1   	   = idx2 + 1	  ;

		if (ch == '\'')
		{	quote	= !quote     ;
			buffer.append ('\'') ;
			continue  ;
		}

		if ((ch == '?') && quote)
		{
			buffer.append ('?' ) ;
			continue  ;
		}

		if (nvals == 0)
		{
			pError	= KBError
				  (	KBError::Error,
				  	QString(TR("Insufficient (%1) values for placeholders")).arg(_nvals),
				  	sql,
				  	__ERRLOCN
				  )	;
			return	false	;
		}

		values->getQueryText (buffer, codec)	;

		nvals	-= 1 ;
		values	+= 1 ;
	}

	if (nvals > 0)
	{
		pError	= KBError
			  (	KBError::Error,
			  	QString(TR("Excess (%1) values for placeholders")).arg(_nvals),
			  	sql,
			  	__ERRLOCN
			  )	;
		return	false	;
	}

//	fprintf
//	(	stderr,
//		"subPlaceList [%s]\n",
//		buffer.data()
//	)	;
	return	true	;
}

/*  KBServer								*/
/*  subPlaceList: Substitute placeholders				*/
/*  sql		: const QString &: SQL statement with placeholders	*/
/*  nvals	: uint		 : Number of substitution values	*/
/*  values	: KBValue *	 : Value list				*/
/*  pError	: KBError &	 : Error return				*/
/*  (returns)	: QString	 : Substituted SQL			*/

QString	KBServer::subPlaceList
	(	const QString	&sql,
		uint		nvals,
		const KBValue	*values,
		KBError		&pError
	)
{
	/* This version of KBServer::subPlaceList is used to generate	*/
	/* a version of the query which can be presented to the user.	*/
	KBDataBuffer	buffer	;
	uint		idx1	= 0	;
	bool		quote	= false	;
	uint		_nvals	= nvals	;

	QRegExp	seps	("['?]");

	while (idx1 < sql.length())
	{
		int	idx2 = sql.find (seps, idx1) ;

		if (idx2 < 0)
		{
			buffer.append (sql.mid (idx1)) ;
			break	;
		}

		buffer.append (sql.mid (idx1, idx2 - idx1)) ;

		QChar	ch = sql.at(idx2) ;
		idx1   	   = idx2 + 1	  ;

		if (ch == '\'')
		{	quote	= !quote     ;
			buffer.append ('\'') ;
			continue  ;
		}

		if ((ch == '?') && quote)
		{
			buffer.append ('?' ) ;
			continue  ;
		}

		if (nvals == 0)
		{
			pError	= KBError
				  (	KBError::Error,
				  	QString(TR("Insufficient (%1) values for placeholders")).arg(_nvals),
				  	sql,
				  	__ERRLOCN
				  )	;
			return	QString::null ;
		}

		if (!values->isNull())
			switch (values->getType()->getIType())
			{
				case KB::ITString :
					{	QString	t = values->getRawText() ;
						if (t.length() > 80)
						{	t.truncate  (80) ;
							t += "..." ;
						}
						buffer.append ("'");
						buffer.append (t  );
						buffer.append ("'");
					}
					break	;

				case KB::ITBinary :
					buffer.append ("[binary data]") ;
					break	;

				default	:
					values->getQueryText (buffer, 0) ;
					break	;
			}
		else	buffer.append ("null")	;

		nvals	-= 1 ;
		values	+= 1 ;
	}

	if (nvals > 0)
	{
		pError	= KBError
			  (	KBError::Error,
			  	QString(TR("Excess (%1) values for placeholders")).arg(_nvals),
			  	sql,
			  	__ERRLOCN
			  )	;
		return	QString::null ;
	}

//	fprintf
//	(	stderr,
//		"KBServer::subPlaceList [%s]\n",
//		buffer.data()
//	)	;
	return	QString::fromUtf8(buffer.data()) ;

#if	0
	/* This version of KBServer::subPlaceList is used to generate	*/
	/* a version of the query which can be presented to the user.	*/
	uint	idx1	= 0	;
	bool	quote	= false	;
	QString	res	= ""	;

	QRegExp	seps	("['?]");

	while (idx1 < sql.length())
	{
		int	idx2 = sql.find (seps, idx1) ;

		if (idx2 < 0)
		{
			res += sql.mid (idx1) ;
			break	;
		}

		res	+= sql.mid (idx1, idx2 - idx1) ;

		QChar	ch = sql.at(idx2) ;
		idx1   	   = idx2 + 1	  ;

		if (ch == '\'')
		{	quote	= !quote  ;
			res    += '\''	  ;
			continue  ;
		}

		if ((ch == '?') && quote)
		{
			res	+= '?'	  ;
			continue  ;
		}

		if (nvals == 0)
		{
			pError	= KBError
				  (	KBError::Error,
				  	TR("Insufficient values for placeholders"),
				  	QString::null,
				  	__ERRLOCN
				  )	;
			return	QString::null	;
		}

		if (!values->isNull())
			switch (values->getType()->getIType())
			{
				case KB::ITString :
					{	QString	t = values->getRawText() ;
						if (t.length() > 80)
						{	t.truncate  (80) ;
							t += "..." ;
						}
						res    	+= QString("'%1'").arg(t) ;
					}
					break	;

				case KB::ITBinary :
					res	+= "[binary data]" ;
					break	;

				default	:
					res	+= values->getRawText () ;
					break	;
			}
		else	res	+= "null"	;

		nvals	-= 1 ;
		values	+= 1 ;
	}

	if (nvals > 0)
	{
		pError	= KBError
			  (	KBError::Error,
			  	TR("Excess values for placeholders"),
			  	QString::null,
			  	__ERRLOCN
			  )	;
		return	QString::null	;
	}

	return	res	;
#endif
}


/*  KBServer								*/
/*  listDrivers	: Get a list of available drivers			*/
/*  drvList	: KBDriverDetailsList	: Driver list			*/
/*  pError	: KBError &	  	: Error return			*/
/*  (returns)	: bool		  	: Success			*/

bool	KBServer::listDrivers
	(	KBDriverDetailsList	&drvList,
		KBError			&
	)
{
	/* Scan the local desktop files which describe the rekall	*/
	/* "services". These are the same files as used for the KTrader	*/
	/* version, but the decoding in KBDesktop is much simplified.	*/
	QString		 dtDir	= locateDir
				  (	"appdata",
					"services/rekall_dummy.desktop"
				  )	;
	uint idx;

	QList<KBDesktop> dtDefs	;
	KBDesktop::scan (dtDir  + "/services", "rekall_", dtDefs) ;

	for (idx = 0 ; idx < dtDefs.count() ; idx += 1)
	{
		KBDesktop *desktop = dtDefs.at(idx) ;

		//desktop->print () ;

		if (desktop->property ("ServiceTypes") != "Rekall/Driver")
			continue	;

		QString		tag	= desktop->property("X-KDE-Driver-Tag") ;
		QString		comment	= desktop->property("Comment") ;
		QString		info	= desktop->property("Info"   ) ;
		QString		ftext	= desktop->property("Flags"  ) ;
	
		QStringList	flist	= QStringList::split ('|', ftext) ;
		uint		flags	= 0 ;

		for (uint idx = 0 ; idx < flist.count() ; idx += 1)
		{	const QString	&f = flist[idx] ;
			if	(f == "AF_HOST"		) flags |= 0x0001 ;
			else if	(f == "AF_PORTNUMBER"	) flags |= 0x0002 ;
			else if	(f == "AF_SOCKETNAME"	) flags |= 0x0004 ;
			else if	(f == "AF_FLAGS"	) flags |= 0x0008 ;
			else if	(f == "AF_USERPASSWORD"	) flags |= 0x0010 ;
			else if	(f == "AF_SSHTUNNEL"	) flags |= 0x0020 ;
		}

		if (info.isEmpty()) info = comment ;

		fprintf
		(	stderr,
			"Found db driver tag=[%s] comment=[%s] flags=[%s/%d]\n",
			(cchar *)tag,
			(cchar *)comment,
			(cchar *)ftext,
			flags
		)	;

		drvList.append (KBDriverDetails(tag, comment, info, flags)) ;
	}

	return	true ;
}

/*  KBServer								*/
/*  lastError	: Get last error					*/
/*  (returns)	: const KBError&: Last error if any			*/

const	KBError	&KBServer::lastError ()
{
	return	m_lError	;
}

bool	KBServer::listDatabases
	(	QStringList	&
	)
{
	m_lError = KBError
		   (	KBError::Error,
			TR("Server does not support database listing"),
			QString::null,
			__ERRLOCN
		   )	;
	return	false	;
}

/*  KBServer								*/
/*  listTypes	: Return a list of all column types			*/
/*  (returns)	: QString	: Types					*/

QString	KBServer::listTypes ()
{
	static	QString	_types
		(	"Bool|SmallInt|Integer|Float|Double|Decimal|"
			"Date|Time|Date/Time|Text|Binary|Primary Key"
		)	;
	return	_types	;
}

/*  KBServer								*/
/*  listTables	: List tables and other objects				*/
/*  tableList	: KBTableDetailsList &	: Return list			*/
/*  type	: uint			: Type mask			*/
/*  (returns)	: bool			: Success			*/

bool	KBServer::listTables
	(	KBTableDetailsList	&tableList,
		uint			type
	)
{
	if (m_cacheTables)
	{
		if (m_tableList.count() == 0)
			if (!doListTables (m_tableList, KB::IsAny))
				return	false	;

		for (uint idx = 0 ; idx < m_tableList.count() ; idx += 1)
			if ((m_tableList[idx].m_type & type) != 0)
				tableList.append (m_tableList[idx]) ;

		return	true	;
	}

	return	doListTables	(tableList, type) ;
}

/*  KBServer								*/
/*  listFields	: Get table fields and related stuff			*/
/*  tabSpec	: KBTableSpec &	: Table specification object		*/
/*  (returns)	: bool		: Success				*/

bool	KBServer::listFields
	(	KBTableSpec	&tabSpec
	)
{
	/* If table caching is enabled then check in the table cache	*/
	/* and return details from there if found.			*/
	if (m_cacheTables)
	{
		KBTableSpec *spec = m_tableCache.find (tabSpec.m_name) ;
		if (spec != 0)
		{
			fprintf
			(	stderr,
				"Got table from cache [%s]\n",
				(cchar *)tabSpec.m_name
			)	;
			tabSpec = *spec ;
			return	true	;
		}
	}

	/* If not, or if there is no cache entry, then do the real	*/
	/* thing to get the details.					*/
	if (!doListFields (tabSpec))
		return	false	;

	fprintf
	(	stderr,
		"KBServer::listFields: post: pkro=%d fake=%d cache=%d\n",
		m_pkReadOnly,
		m_fakeKeys,
		m_cacheTables
	)	;

	/* If the primary-key-read-only flag is set then update any	*/
	/* such fields.							*/
	if (m_pkReadOnly)
		LITER
		(	KBFieldSpec,
			tabSpec.m_fldList,
			fSpec,

			if ((fSpec->m_flags & KBFieldSpec::Primary) != 0)
				fSpec->m_flags |= KBFieldSpec::ReadOnly ;
		)

	/* If requested to fake an insert key then scan to see if there	*/
	/* is (a) an column which is marked as insert-available, in	*/
	/* which case we don't need to fake anything, or (b) a unique	*/
	/* column which can be used for faking.				*/
	if (m_fakeKeys)
	{
		KBFieldSpec *insField = 0 ;

		LITER
		(	KBFieldSpec,
			tabSpec.m_fldList,
			fSpec,

			if ((fSpec->m_flags & KBFieldSpec::InsAvail) != 0)
			{
				insField = fSpec ;
				break	 ;
			}

			if ((fSpec->m_flags & KBFieldSpec::Unique  ) != 0)
				if (insField == 0)
					insField = fSpec ;
		)

		if (insField != 0)
			if ((insField->m_flags & KBFieldSpec::InsAvail) == 0)
				tabSpec.m_fakeKey = insField ;
	}

	/* If table chaching is enabled then copy the table details	*/
	/* into a new cache entry.					*/
	if (m_cacheTables)
	{
		fprintf
		(	stderr,
			"Added table to cache [%s]\n",
			(cchar *)tabSpec.m_name
			)	;
		m_tableCache.insert (tabSpec.m_name, new KBTableSpec (tabSpec)) ;
	}

	return	true	;
}

/*  KBServer								*/
/*  flushTableCache							*/
/*		: Flush all cached table information			*/
/*  (returns)	: void		:					*/

void	KBServer::flushTableCache ()
{
	m_tableList .clear () ;
	m_tableCache.clear () ;
}

/*  KBServer								*/
/*  createTable	: Create a new table					*/
/*  tabSpec	: KBTableSpec &	: Table specification			*/
/*  assoc	: bool		: Create associated stuff		*/
/*  best	: bool		: Use best column match			*/
/*  (returns)	: bool		: Success				*/

bool	KBServer::createTable
	(	KBTableSpec	&tabSpec,
		bool		assoc,
		bool		best
	)
{
	m_tableCache.remove   (tabSpec.m_name) ;
	return	doCreateTable (tabSpec, assoc, best) ;
}

/*  KBServer								*/
/*  renameTable	: Rename a  table					*/
/*  oldName	: cchar *	: Current table name			*/
/*  newName	: cchar *	: New table name			*/
/*  assoc	: bool		: Rename associated stuff		*/
/*  (returns)	: bool		: Success				*/

bool	KBServer::renameTable
	(	cchar	*oldName,
		cchar	*newName,
		bool	assoc
	)
{
	m_tableCache.remove   (oldName) ;
	m_tableCache.remove   (newName) ;
	return	doRenameTable (oldName, newName, assoc) ;
}

/*  KBServer								*/
/*  dropTable	: Drop a table						*/
/*  table	: cchar *	: Table name				*/
/*  assoc	: bool		: Drop associated stuff			*/
/*  (returns)	: bool		: Success				*/

bool	KBServer::dropTable
	(	cchar	*table,
		bool	assoc
	)
{
	m_tableCache.remove (table) ;
	return	doDropTable (table, assoc) ;
}

/*  KBServer								*/
/*  noViews	: Set views unsupported error				*/
/*  (returns)	: void		:					*/

void	KBServer::noViews ()
{
	m_lError = KBError
		   (	KBError::Error,
			TR("Database does not support views"),
			QString::null,
			__ERRLOCN
		   )	;
}

/*  KBServer								*/
/*  viewExists	: See if view exists					*/
/*  view	: const QString &	: View name			*/
/*  exists	: bool &		: Actual result			*/
/*  (returns)	: bool			: Success			*/

bool	KBServer::viewExists
	(	const QString	&,
		bool		&
	)
{
	noViews ()	;
	return	false	;
}

/*  KBServer								*/
/*  createView	: Create a new view					*/
/*  viewSpec	: KBTableSpec &		: View specification		*/
/*  (returns)	: bool			: Success			*/

bool	KBServer::createView
	(	KBTableSpec	&
	)
{
	noViews ()	;
	return	false	;
}

/*  KBServer								*/
/*  renameView	: Rename a view						*/
/*  oldName	: cchar *	: Current view name			*/
/*  newName	: cchar *	: New view name				*/
/*  (returns)	: bool		: Success				*/

bool	KBServer::renameView
	(	cchar	*,
		cchar	*
	)
{
	noViews ()	;
	return	false	;
}

/*  KBServer								*/
/*  dropView	: Drop a view						*/
/*  name	: cchar *	: View name				*/
/*  (returns)	: bool		: Success				*/

bool	KBServer::dropView
	(	cchar	*
	)
{
	noViews ()	;
	return	false	;
}


/*  KBServer								*/
/*  noSequences	: Set sequences unsupported error			*/
/*  (returns)	: void		:					*/

void	KBServer::noSequences ()
{
	m_lError = KBError
		   (	KBError::Error,
			TR("Database does not support sequences"),
			QString::null,
			__ERRLOCN
		   )	;
}

/*  KBServer								*/
/*  sequenceExists							*/
/*		: See if sequence exists				*/
/*  sequence	: const QString &	: Sequence name			*/
/*  exists	: bool &		: Actual result			*/
/*  (returns)	: bool			: Success			*/

bool	KBServer::sequenceExists
	(	const QString	&,
		bool		&
	)
{
	noSequences ()	;
	return	false	;
}

/*  KBServer								*/
/*  descSequence: Describe a sequence					*/
/*  seqSpec	: KBSequenceSpec &	: Sequence specification	*/
/*  (returns)	: bool			: Success			*/

bool	KBServer::descSequence
	(	KBSequenceSpec	&
	)
{
	noSequences ()	;
	return	false	;
}

/*  KBServer								*/
/*  createSequence							*/
/*		: Create a new sequence					*/
/*  tabSpec	: KBSequenceSpec &	: Sequence specification	*/
/*  (returns)	: bool			: Success			*/

bool	KBServer::createSequence
	(	KBSequenceSpec	&
	)
{
	noSequences ()	;
	return	false	;
}

/*  KBServer								*/
/*  renameSequence							*/
/*		: Rename a sequence					*/
/*  oldName	: cchar *	: Current sequence name			*/
/*  newName	: cchar *	: New sequence name			*/
/*  (returns)	: bool		: Success				*/

bool	KBServer::renameSequence
	(	cchar	*,
		cchar	*
	)
{
	noSequences ()	;
	return	false	;
}

/*  KBServer								*/
/*  dropSequence: Drop a sequence					*/
/*  name	: cchar *	: Sequence name				*/
/*  (returns)	: bool		: Success				*/

bool	KBServer::dropSequence
	(	cchar	*
	)
{
	noSequences ()	;
	return	false	;
}


/*  KBServer								*/
/*  optionFlags	: Get server option flags				*/
/*  (returns)	: uint		: Flags					*/

uint	KBServer::optionFlags ()
{
	/* Default returns the whole lot with certain exceptions.	*/
	return	0xffffffff & ~AF_SSHTUNNEL ;
}


/*  KBServer								*/
/*  defOperatorMap							*/
/*		: Get the default operator map				*/
/*  map		: cchar *&	: Vector of textual operators		*/
/*  (returns)	: uint		: Number of operators in map		*/

uint	KBServer::defOperatorMap
	(	cchar		**&map
	)
{
	static	cchar	*opMap[] =
	{
		"=",		// EQ
		"!=",		// NEQ
		"<=",		// LE
		">=",		// GE
		"<",		// LT
		">",		// GT
		"like",		// Like
	}	;

	map	= opMap	;
	return	sizeof(opMap)/sizeof(cchar *) ;
}

/*  KBServer								*/
/*  operatorMap	: Get the operator map for the driver			*/
/*  map		: cchar *&	: Vector of textual operators		*/
/*  (returns)	: uint		: Number of operators in map		*/

uint	KBServer::operatorMap
	(	cchar		**&map
	)
{
	return	defOperatorMap (map) ;
}

/*  KBServer								*/
/*  printQuery	: Print query and arguments				*/
/*  rawQuery	: const QString & : Raw query text			*/
/*  nvals	: uint		  : Number of arguments			*/
/*  values	: const KBValue * : Vector of arguments			*/
/*  success	: bool		  : Execution success			*/
/*  (returns)	: void		  :					*/

void	KBServer::printQuery
	(	const QString	&rawQuery,
		uint		nvals,
		const KBValue	*values,
		bool		success
	)
{
	if (m_printQueries)
	{
		fprintf	(stderr, "Rekall query: [%d][%s]\n", success, (cchar *)rawQuery) ;
		for (uint idx = 0 ; idx < nvals ; idx += 1)
		{	QString	val = values[idx].getQueryText() ;
			if (val.length() > 64)
				val = val.left (64) + " ....." ;
			fprintf	(stderr, "      %5d: [%s]\n", idx, (cchar *)val) ;
		}
		fprintf	(stderr, "Rekall query: ----\n") ;
	}

	if (KBAppPtr::getCallback() != 0)
		KBAppPtr::getCallback()->logQuery
		(	m_serverName,
			rawQuery,
			success,
			nvals,
			values
		)	;
}

/*  KBServer								*/
/*  rekallPrefix: Modify name with RDBMS-specific prefix		*/
/*  name	: const QString & : Name to modify			*/
/*  (returns)	: QString	  : Modified name			*/

QString	KBServer::rekallPrefix
	(	const QString	&name
	)
{
	return	"__" + name	;
}

/*  KBServer								*/
/*  doMapExpression							*/
/*		: Map SQL expression for driver				*/
/*  expr	: const QString & : Expression to map			*/
/*  bra		: cchar *	  : Opening brackey			*/
/*  ket		: cchar *	  : Closing brackey			*/
/*  spec	: const QString & : Special name characters		*/
/*  (returns)	: QString	  : Mapped expression			*/

QString	KBServer::doMapExpression
	(	const QString	&expr,
		cchar		*bra,
		cchar		*ket,
		const QString	&spec
	)
{
	static	QAsciiDict<void> *keywords ;
	static	cchar	*__keywords[] =
	{	"and",
		"or",
		"between",
		"like",
		"in",
		0
	}	;

	if (keywords == 0)
	{	keywords = new QAsciiDict<void> (17, false, false) ;
		for (cchar **kp = __keywords ; *kp != 0 ; kp += 1)
			keywords->insert (*kp, (void *)keywords) ;
	}

	bool	spaceOK	= true	;	/* To be added as argument	*/

	/* This method provides a common routine to map expressions,	*/
	/* where simple words are wrapped in specified brackets, and	*/
	/* words start with a letter or special character, and then	*/
	/* contain alphanumerices or the special character. Only handle	*/
	/* single-character quoted strings, and backslash is escape.	*/
	QString	res	;
	bool	quoted	= false	;
	uint	offset	= 0	;


	while (offset < expr.length())
	{
		QChar ch = expr.at(offset) ;

		/* Backslash is treated as an escape both inside and	*/
		/* outside quoted text.					*/
		if (ch == '\\')
		{	res    += ch	;
			res    += expr.at(offset+1) ;
			offset += 2	;
			continue	;
		}

		/* Unless escaped, a quote character toggles the quote	*/
		/* state.						*/
		if (ch == '\'')
		{	quoted	= !quoted ;
			res    += ch	  ;
			offset += 1	  ;
			continue  ;
		}

		/* If quoted then the character passes though unchanged	*/
		/* as also if it cannot start a new word.		*/
		if (quoted)
		{	res    += ch	;
			offset += 1	;
			continue  	;
		}

		if (!ch.isLetterOrNumber() && (spec.find(ch) < 0))
		{	res    += ch	;
			offset += 1	;
			continue	;
		}

		/* OK, found a word or a number, separate it out. The	*/
		/* token is stored in "token", and "nonDigit" is set as	*/
		/* soon as any non-digit/non-space character is found.	*/
		bool	nonDigit = false;
		QString	token	 ;

		while (ch.isLetterOrNumber() || (spec.find(ch) >= 0) || (spaceOK && (ch == ' ')))
		{
			/* If the character is a space and we have not	*/
			/* hit a non-digit character then we have a	*/
			/* number, so drp out of the loop.		*/
			if ((ch == ' ') && !nonDigit)
				break	;

			/* If the character is not a digit and not a	*/
			/* space then set the "nonDigit" flag. We use	*/
			/* this later to decide if we have a number or	*/
			/* or not.					*/
			if (!ch.isNumber() && (ch != ' '))
				nonDigit = true ;

			/* Next bit of nasty code handles the case	*/
			/* where spaces are allowed in words. Scan over	*/
			/* to the next non-space; if it is a word	*/
			/* character then include the space in the word	*/
			if (spaceOK && (ch == ' '))
			{
				uint 	over	= 1 ;
				QChar	ch2	;

				while (offset + over < expr.length())
					if (expr.at(offset + over) != ' ')
					{	ch2	= expr.at(offset + over) ;
						break	;
					}

				/* Found the next character; if allowed	*/
				/* in a word then put the spaces back	*/
				/* and continue.			*/
				if (ch2.isLetterOrNumber() || (spec.find(ch2) >= 0))
				{
					while (over > 0)
					{	token	+= ' '	;
						offset	+= 1	;
						over	-= 1	;
					}

					ch = ch2 ;
					continue ;
				}

				/* Not allowed in a word. Skip over and	*/
				/* finish the word.			*/
				offset	+= over	 ;
				break	;
			}

			token  += ch	;
			offset += 1	;
			ch	= expr.at(offset) ;

		}

		/* If the token contains any non-digits then it gets	*/
		/* quoted, otherwise it is a number and is left as-is.	*/
		if (nonDigit && (keywords->find(token.latin1()) == 0))
		{
			res	+= bra	 ;
			res	+= token ;
			res	+= ket	 ;
		}
		else	res	+= token ;
	}

//	fprintf
//	(	stderr,
//		"KBServer::doMapExpression: [%s]->[%s]\n",
//		(cchar *)expr,
//		(cchar *)res
//	)	;
	return	res	;
}

/*  KBServer								*/
/*  mapExpression: Map SQL expression for driver			*/
/*  expr	 : const QString & : Expression to map			*/
/*  (returns)	 : QString	   : Mapped expression			*/

QString	KBServer::mapExpression
	(	const QString	&expr
	)
{
	return	expr	;
}

/*  KBServer								*/
/*  transaction	: Server transaction request				*/
/*  op		: Transaction	: Specific operation			*/
/*  activeCookie: void **	: Pass/return active cookie		*/
/*  (returns)	: bool		: Success				*/

bool	KBServer::transaction
	(	Transaction	,
		void		**activeCookie
	)
{
	if (activeCookie != 0) *activeCookie	= 0 ;

	m_lError = KBError
		   (	KBError::Error,
			TR("Transactions not supported"),
			QString::null,
			__ERRLOCN
		   )	;
	return	false	;
}

/*  KBServer								*/
/*  qryCursor	: Create cursor						*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Query text				*/
/*  cursor	: const QString & : Cursor name				*/
/*  (returns)	: bool		  : Success				*/

KBSQLCursor
	*KBServer::qryCursor
	(	bool		,
		const QString	&,
		const QString	&
	)
{
	m_lError = KBError
		  (	KBError::Error,
			TR("Cursors not supported"),
			QString::null,
			__ERRLOCN
		  )	;
	return	0 ;
}

/*  KBServer								*/
/*  setting	: Get server setting by name				*/
/*  setting	: const QString & : Required setting			*/
/*  (returns)	: QString	  : Value				*/

QString	KBServer::setting
	(	const QString	&setting
	)
{
	if (setting == "servername") return m_serverName ;
	if (setting == "host"	   ) return m_host	 ;
	if (setting == "user"	   ) return m_user	 ;
	if (setting == "password"  ) return m_password	 ;
	if (setting == "database"  ) return m_database	 ;

	return	QString::null ;
}

/*  KBServer								*/
/*  syntaxToText: Get text message for syntax element			*/
/*  syntax	: Syntax	: Element				*/
/*  (returns)	: QString	: Message				*/

QString	KBServer::syntaxToText
	(	KBServer::Syntax	syntax
	)
{
	switch (syntax)
	{
		case Limit	: return TR("select limit/offset") ;
		default		: break	 ;
	}

	return	TR("unknown syntax element") ;
}

/*  KBServer								*/
/*  getSyntax	: Get text for syntax element				*/
/*  syntax	: Syntax	: Element				*/
/*  ...		: ...		: Arguments				*/
/*  (returns)	: QString	: Text					*/

bool	KBServer::getSyntax
	(	QString			&,
		KBServer::Syntax	syntax,
		...
	)
{
	m_lError = KBError
		   (	KBError::Error,
			QString(TR("Driver does not support %1")).arg(syntaxToText(syntax)),
			QString::null,
			__ERRLOCN
		   )	;
	return	false	;
}


#ifndef	_WIN32

/*  KBServer								*/
/*  openSSHTunnel: Open SSH tunnel to target system			*/
/*  dbport	 : int		: Default server port			*/
/*  (returns)	 : int		: Local port or negative on error	*/

int	KBServer::openSSHTunnel
	(	int		dbport
	)
{
	fprintf
	(	stderr,
		"KBServer::openSSHTunnel: pid=%d port=%d h=[%s] p=[%s/%d]\n",
		m_sshPID,
		m_sshPort,
		(cchar *)m_host,
		(cchar *)m_port,
		dbport
	)	;

	if (m_sshPort >= 0)
		return	m_sshPort ;

	QStringList	bits	= QStringList::split (':', m_sshTarget) ;

	if (m_host.isEmpty() || ((m_port.toInt() <= 0) && (dbport < 0)))
	{
		m_lError = KBError
			   (	KBError::Error,
				QString	(TR("Must set host and port for SSH tunneling")),
				QString::null,
				__ERRLOCN
			   )	;
		return	-1 ;
	}
	if (bits.count() != 2)
	{
		m_lError = KBError
			   (	KBError::Error,
				QString	(TR("SSH target should have format name@host:port")),
				QString::null,
				__ERRLOCN
			   )	;
		return	-1 ;
	}

	if (dbport < 0) dbport	= m_port .toInt () ;
	int		port	= bits[1].toInt () ;
	QString		link	= QString("%1:%2:%3").arg(port).arg(m_host).arg(dbport) ;

	fprintf
	(	stderr,
		"KBServer::openSSHTunnel: [%s]->[%s]\n",
		(cchar *)m_sshTarget,
		(cchar *)link
	)	;

	/* The ssh tunnel command runs as a child process, so fork for	*/
	/* the tunnel. SHould not fail but one never knows ....		*/
	if ((m_sshPID = fork()) < 0)
	{
		m_lError = KBError
			   (	KBError::Error,
				QString	(TR("Failed to form SSH tunnel")),
				strerror(errno),
				__ERRLOCN
			   )	;
	}

	/* In the child, run the ssh command. Note that we close all	*/
	/* streams execpt stdout and stderr, and reopen stdin from the	*/
	/* null device.							*/
	if (m_sshPID == 0)
	{
		for (uint idx = 3 ; idx < 128 ; idx += 1)
			close (idx) ;

		freopen	("/dev/null", "r", stdin) ;

		execlp
		(	"ssh",
			"ssh",
			"-N",
			"-C",
			"-L",
			(cchar *)link,
			(cchar *)bits[0],
			0
		)	;

		fprintf
		(	stderr,
			"KBServer::openSSHTunnel: execlp returned: %s\n",
			strerror(errno)
		)	;

		exit	(1) ;
	}

	/* Back in the parent, we need to wait for the tunnel to appear	*/
	/* so run a dialog which waits for the specified port to become	*/
	/* available, or for the child to disappear.			*/
	KBSSHTunnel ssht(m_sshTarget, m_sshPID, port, m_lError) ;

	if (!ssht.exec())
	{
		kill	(m_sshPID, SIGKILL) ;
		sleep	(2) ;
		waitpid	(m_sshPID, 0, WNOHANG) ;

		m_sshPID = 0	;
		return	 -1	;
	}

	return	m_sshPort = port ;
}
#endif

bool	KBServer::execInitSQL
	(	const QString	&initSQL
	)
{
	uint	ptr	= 0 ;
	int	quoted	= 0 ;

	while (ptr < initSQL.length())
	{
		uint	ptr2	= ptr ;

		while (ptr2 < initSQL.length())
		{
			if (quoted != 0)
				if (initSQL[ptr2] == QChar(quoted))
				{	ptr2	+= 1 ;
					continue ;
				}

			if (initSQL[ptr2] == '"' )
			{	quoted	 = '"'	;
				ptr2	+= 1	;
				continue ;
			}
			if (initSQL[ptr2] == '\'')
			{	quoted	 = '\''	;
				ptr2	+= 1	;
				continue ;
			}

			if (initSQL[ptr2] == ';' )
				break	;

			ptr2	+= 1	;
		}

		QString	cmd	= initSQL.mid(ptr, ptr2 - ptr).stripWhiteSpace() ;

		fprintf
		(	stderr,
			"KBServer::execInitSQL: [%d,%d][%s]\n",
			ptr,
			ptr2,
			(cchar *)cmd
		)	;

		ptr = ptr2 + 1	;

		if (cmd.length() > 0)
			if (!command (true, cmd, 0, 0, 0))
				return false ;
	}

	return	true	;
}

/*  ------------------------------------------------------------------  */

/*  KBSQLSelect								*/
/*  KBSQLSelect	: Constructor for select query base class		*/
/*  server	: KBServer *	: Server connection object		*/
/*  data	: bool		: Query for data			*/
/*  codec	: QTextCodec *	: Non-default codec			*/
/*  (returns)	: KBSQLSelect	:					*/

KBSQLSelect::KBSQLSelect
	(	KBServer	*server,
		bool		data,
		const QString	&query
	)
	:
	KBSQLQuery (server, data, query)
{
	m_types	  = 0 ;
	m_nFields = 0 ;
}

/*  KBSQLSelect								*/
/*  ~KBSQLSelect: Destructor for select query base class		*/

KBSQLSelect::~KBSQLSelect ()
{
	if (m_types != 0)
	{
		for (uint idx = 0 ; idx < m_nFields ; idx += 1)
			m_types[idx]->deref() ;
		delete [] m_types ;
	}

	dumpAllRows () ;
}

/*  KBSQLSelect								*/
/*  getNumFields: Get number of fields returned by query		*/
/*  (returns)	: uint		: Number of fields			*/

uint	KBSQLSelect::getNumFields ()
{
	return	m_nFields	;
}

/*  KBSQLSelect								*/
/*  putInCache	: Put an entire row of data into the cache		*/
/*  row		: uint		: Row number				*/
/*  values	: KBValue *	: Values				*/
/*  (returns)	: void		:					*/

void	KBSQLSelect::putInCache
	(	uint		row,
		KBValue		*values
	)
{
	dumpRow	(row) ;
	m_valueCache.insert (row, values) ;
}

/*  KBSQLSelect								*/
/*  putInCache	: Put a value into the cache				*/
/*  row		: uint		: Row number				*/
/*  col		: uint		: Column number				*/
/*  value	: KBValue &	: Value to store			*/
/*  (returns)	: void		:					*/

void	KBSQLSelect::putInCache
	(	uint		row,
		uint		col,
		KBValue		&value
	)
{
	KBValue	*extant	= m_valueCache.find(row) ;

	if (extant == 0)
	{	extant	= new KBValue [getNumFields()] ;
		m_valueCache.insert (row, extant) ;
	}

	extant[col] = value ;
}

/*  KBSQLSelect								*/
/*  row		: uint		: Row number				*/
/*  col		: uint		: Column number				*/
/*  value	: KBValue &	: Location for retrieved value		*/
/*  (returns)	: bool		: True if row is in the cache		*/

bool	KBSQLSelect::getFromCache
	(	uint		row,
		uint		col,
		KBValue		&value
	)
{
	KBValue	*extant	= m_valueCache.find(row) ;

	if (extant != 0)
	{	value	= extant[col] ;
		return	true ;
	}

	return	false	;
}

/*  KBSQLSelect								*/
/*  dumRow	: Dump a specified row from the cache			*/
/*  row		: uint		: Row number				*/
/*  (returns)	: void		:					*/

void	KBSQLSelect::dumpRow
	(	uint		row
	)
{
	KBValue	*extant	= m_valueCache.find(row) ;

	if (extant != 0)
	{	delete	[] extant ;
		m_valueCache.remove (row) ;
	}

//	fprintf
//	(	stderr,
//		"KBSQLSelect::dumpRow: [%d]->[%p]\n",
//		row,
//		(void *)extant
//	)	;
}

/*  KBSQLSelect								*/
/*  dumRow	: Dump rows up to a specified row number		*/
/*  row		: uint		: Row number				*/
/*  (returns)	: void		:					*/

void	KBSQLSelect::dumpRowsTo
	(	uint		row
	)
{
	QIntDictIterator<KBValue> cacheIter (m_valueCache) ;

	while (cacheIter.current() != 0)
	{
		if (cacheIter.currentKey() < (int)row)
		{
//			fprintf
//			(	stderr,
//				"KBSQLSelect::dumpRowsTo: [%d]: dump [%ld]\n",
//				row,
//				cacheIter.currentKey()
//			)	;

			delete	[] cacheIter.current() ;
			m_valueCache.remove (cacheIter.currentKey()) ;
		}
//		else
//			fprintf
//			(	stderr,
//				"KBSQLSelect::dumpRowsTo: [%d]: miss [%ld]\n",
//				row,
//				cacheIter.currentKey()
//			)	;

		cacheIter += 1 ;
	}
}

/*  KBSQLSelect								*/
/*  dumpAllRows	: Clear the cache					*/
/*  (returns)	: void		:					*/

void	KBSQLSelect::dumpAllRows ()
{
	QIntDictIterator<KBValue> cacheIter (m_valueCache) ;

	while (cacheIter.current() != 0)
	{
		delete	[] cacheIter.current() ;
		cacheIter += 1 ;
	}

	m_valueCache.clear() ;
}

/*  KBSQLSelect								*/
/*  rowExists	: See if a specified row exists				*/
/*  row		: uint		: Row number				*/
/*  dump	: bool		: Dump other rows			*/
/*  (returns)	: bool		: Row exists				*/

bool	KBSQLSelect::rowExists
	(	uint		row,
		bool		dump
	)
{
	/* The default is to compare against the number of rows, but	*/
	/* this can be overridden by drivers where the number of rows	*/
	/* is not known until they are fetched.				*/
	if (dump) dumpRowsTo (row) ;
	return	(int)row < m_nRows ;
}


/*  ------------------------------------------------------------------  */

/*  KBSQLUpdate								*/
/*  KBSQLUpdate	: Constructor for update query base class		*/
/*  server	: KBServer *	: Server connection object		*/
/*  data	: bool		: Query for data			*/
/*  query	: const QString&: Query text				*/
/*  tabName	: const QString&: Table name				*/
/*  (returns)	: KBSQLUpdate	:					*/

KBSQLUpdate::KBSQLUpdate
	(	KBServer	*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)
	:
	KBSQLQuery (server, data, query),
	m_tabName  (tabName)
{
}

/*  KBSQLUpdate								*/
/*  ~KBSQLUpdate: Destructor for update query base class		*/

KBSQLUpdate::~KBSQLUpdate ()
{
}



/*  ------------------------------------------------------------------  */

/*  KBSQLInsert								*/
/*  KBSQLInsert	: Constructor for insert query base class		*/
/*  server	: KBServer *	: Server connection object		*/
/*  data	: bool		: Query for data			*/
/*  query	: const QString&: Query text				*/
/*  tabName	: const QString&: Table name				*/
/*  (returns)	: KBSQLInsert	:					*/

KBSQLInsert::KBSQLInsert
	(	KBServer	*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)
	:
	KBSQLQuery (server, data, query),
	m_tabName  (tabName)
{
}

/*  KBSQLInsert								*/
/*  ~KBSQLInsert: Destructor for insert query base class		*/

KBSQLInsert::~KBSQLInsert ()
{
}


/*  ------------------------------------------------------------------  */

/*  KBSQLDelete								*/
/*  KBSQLDelete	: Constructor for delete query base class		*/
/*  server	: KBServer *	: Server connection object		*/
/*  data	: bool		: Query for data			*/
/*  query	: const QString&: Query text				*/
/*  tabName	: const QString&: Table name				*/
/*  (returns)	: KBSQLDelete	:					*/

KBSQLDelete::KBSQLDelete
	(	KBServer	*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)
	:
	KBSQLQuery (server, data, query),
	m_tabName  (tabName)
{
}

/*  KBSQLDelete								*/
/*  ~KBSQLDelete: Destructor for delete query base class		*/

KBSQLDelete::~KBSQLDelete ()
{
}

/*  ------------------------------------------------------------------  */

/*  KBSQLCursor								*/
/*  KBSQLCursor	: Constructor for cursor base class			*/
/*  server	: KBServer *	: Server connection object		*/
/*  data	: bool		: Query for data			*/
/*  query	: const QString&: Cursor text				*/
/*  cursor	: const QString&: Cursor name				*/
/*  (returns)	: KBSQLCursor	:					*/

KBSQLCursor::KBSQLCursor
	(	KBServer	*server,
		bool		data,
		const QString	&query,
		const QString	&cursor
	)
	:
	KBSQLQuery (server, data, query),
	m_cursor   (cursor)
{
	m_types	  = 0 ;
	m_nFields = 0 ;
}

/*  KBSQLCursor								*/
/*  ~KBSQLCursor: Destructor for cursor base class			*/

KBSQLCursor::~KBSQLCursor ()
{
	if (m_types != 0)
	{
		for (uint idx = 0 ; idx < m_nFields ; idx += 1)
			m_types[idx]->deref() ;
		delete [] m_types ;
	}
}

