/***************************************************************************
    file	         : kb_value.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>

#include	<qglobal.h>
#include	<qobject.h>
#include	<qstringlist.h>
#include	<qregexp.h>

#ifdef		Q_OS_LINUX
#include	<locale.h>
#include	<monetary.h>
#endif

#if		__KB_KDE
#include	<kglobal.h>
#include	<klocale.h>
#include	<kstddirs.h>
#include	<ksimpleconfig.h>
#endif

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_databuffer.h"


#define	CN_PARENS	0
#define	CN_BEFOREALL	1
#define	CN_AFTERALL	2
#define	CN_BEFORESYM	3
#define	CN_AFTERSYM	4


struct	KBFormatInfo
{
	QString	decimalPoint	;
	QString	thousandSep	;
	QString	currencySymbol	;
	QString	currencyPoint	;
	QString	currencySep	;
	uint	currencyPos	;
	uint	currencyNeg	;
}	;

static	QDict<KBFormatInfo>	formatInfo	;
static	KBFormatInfo		*formatDef	;


/*  getFormatInfo: Get formatting information for language		*/
/*  lang	 : QString	 : Language				*/
/*  (returns)	: KBFormatInfo * : Format information			*/

static	KBFormatInfo	*getFormatInfo
	(	QString	lang	= QString::null
	)
{
//	fprintf	(stderr, "getFormatInfo: reg lang = [%s]\n", (cchar *)lang) ;

	/* First time in create a default format information structure	*/
	/* which is initialised to defualt values. This is stored in	*/
	/* the format dictionary under the name "default".		*/
	if (formatDef == 0)
	{
		formatDef = new KBFormatInfo ;

		formatDef->decimalPoint		= "."		;
		formatDef->thousandSep		= ","		;
		formatDef->currencySymbol	= "$"		;
		formatDef->currencyPoint	= "."		;
		formatDef->currencySep		= ","		;
		formatDef->currencyNeg		= CN_PARENS	;

		formatInfo.insert ("default", formatDef)	;
	}

	KBFormatInfo	*fi = 0 ;

#if	__KB_KDE
	/* KDE. First, if the language is not explicitely given then	*/
	/* get the currently set country.				*/
	if (lang.isEmpty())
		lang	= KGlobal::locale()->country() ;

	/* Look up the language. If we have been here before then it	*/
	/* will be in the dictionary, otherwise we will have to create	*/
	/* a new entry ....						*/
	if ((fi = formatInfo.find (lang)) == 0)
	{
		/* ... which we decode ourselves, since KDE does not	*/
		/* seem to want to set locale languages unless said	*/
		/* language is properly installed, even though the	*/
		/* "entry.desktop" files are present.			*/
		QString	spec = locate("locale", QString::fromLatin1("l10n/%1/entry.desktop").arg(lang)) ;

		fprintf
		(	stderr,
			"getFormatInfo [%s]\n",
			(cchar *)spec
		)	;

		if (!spec.isEmpty())
		{
			KSimpleConfig	le (spec, true) ;
			le.setGroup ("KCM Locale") ;

			fi = new KBFormatInfo ;
			fi->decimalPoint	= le.readEntry   ("DecimalSymbol", 		  ".") ;
			fi->thousandSep		= le.readEntry   ("ThousandsSeparator",		  ",") ;
			fi->currencySymbol	= le.readEntry   ("CurrencySymbol",		  "$") ;
			fi->currencyPoint	= le.readEntry   ("MonetaryDecimalSymbol", 	  ".") ;
			fi->currencySep		= le.readEntry   ("MonetaryThousandsSeparator",	  ",") ;
			fi->currencyPos		= le.readNumEntry("PositiveMonetarySignPosition", CN_BEFOREALL) ;
			fi->currencyNeg		= le.readNumEntry("NegativeMonetarySignPosition", CN_PARENS   ) ;

			fi->thousandSep.replace	(QRegExp("\\$0"), QString::null) ;
			fi->currencySep.replace	(QRegExp("\\$0"), QString::null) ;

//			fprintf	(stderr, "DecimalSymbol                 : %s\n", (cchar *)fi->decimalPoint  ) ;
//			fprintf	(stderr, "ThousandsSeparator            : %s\n", (cchar *)fi->thousandSep   ) ;
//			fprintf	(stderr, "CurrencySymbol                : %s\n", (cchar *)fi->currencySymbol) ;
//			fprintf	(stderr, "MonetaryDecimalSymbol         : %s\n", (cchar *)fi->currencyPoint ) ;
//			fprintf	(stderr, "MonetaryThousandsSeparator    : %s\n", (cchar *)fi->currencySep   ) ;
//			fprintf	(stderr, "PositiveMonetarySignPosition  : %d\n", fi->currencyPos   ) ;
//			fprintf	(stderr, "NegativeMonetarySignPosition  : %d\n", fi->currencyNeg   ) ;
		}
		else
		{
			fi	= formatDef	;

//			TKMessageBox::sorry
//			(	0,
//				QString(TR("Cannot locate locale '%1'")).arg(lang),
//				"Locale not found"
//			)	;
		}

		formatInfo.insert (lang, fi) ;
		setlocale (LC_ALL, "") ;
	}

#endif
#if	defined(Q_OS_LINUX) && __KB_TKC
	/* Under QT on linux, use the locale routines to get the	*/
	/* information. As fore KDE above, this is stored locally for	*/
	/* future reuse.						*/
	if ((fi = formatInfo.find (lang)) == 0)
	{
		char		*le = setlocale (LC_ALL, lang) ;
		struct	lconv	*lc = localeconv () ;

		// fprintf (stderr, "getFormatInfo: setlocale -> %d\n", le) ;

		if (le != 0)
		{
			fi = new KBFormatInfo ;
			fi->decimalPoint	= lc->decimal_point	;
			fi->thousandSep		= lc->thousands_sep	;
			fi->currencySymbol	= lc->currency_symbol	;
			fi->currencyPoint	= lc->mon_decimal_point	;
			fi->currencySep		= lc->mon_thousands_sep	;
			fi->currencyPos		= lc->p_sign_posn	;
			fi->currencyNeg		= lc->n_sign_posn	;
		}
		else
		{
			fi	= formatDef	;

//			TKMessageBox::sorry
//			(	0,
//				QString(TR("Cannot locate locale '%1'")).arg(lang),
//				"Locale not found"
//			)	;
		}

		formatInfo.insert (lang, fi) ;
		setlocale (LC_ALL, "") ;
	}
#endif

	/* Return the format definition if we have one or the default	*/
	/* if not.							*/
	return	fi != 0 ? fi : formatDef ;
}


/*  ------------------------------------------------------------------  */

/*  KBValue	:							*/
/*  deFormat	: Attempt to undo formatting				*/
/*  text	: const QString & : Text to deformat			*/
/*  type	: KBType *	  : Database type			*/
/*  fmt		: const QString & : Format string			*/
/*  (returns)	: const QString & : Deformatted text			*/

const QString
	&KBValue::deFormat
	(	const QString	&text,
		KBType		*type,
		const QString	&fmt
	)
{
	static		QString	t ;
	KBFormatInfo	*fi	= getFormatInfo () ;

	fprintf
	(	stderr,
		"deFormat: [%s][%s][%s] ....\n",
		(cchar *)text,
		(cchar *)type->getDescrip(),
		(cchar *)fmt
	)	;

	if (fmt.isEmpty() || text.isEmpty())
		return	text	;

	switch (type->getIType())
	{
		case KB::ITString	:
			/* Value is a string, not a lot we can do here	*/
			/* since the format could be arbitrarily	*/
			/* complicated, and its not the sort of thing	*/
			/* to be formatted.				*/
			return	text	;

		case KB::ITDate		:
		case KB::ITTime		:
		case KB::ITDateTime	:
			/* Assume that any date/time formatting leaves	*/
			/* the text in a recognisable format.		*/
			return	text	;

		case KB::ITBinary	:
		case KB::ITDriver	:
			/* Not much we can do here, these are very	*/
			/* unlikely to be formatted, and in any case	*/
			/* the canonical form could be anything.	*/
			return	text	;

		case KB::ITBool		:
			/* Have to assume the text is recognisable	*/
			/* still.					*/
			return	text	;

		case KB::ITFixed	:
			/* Skip anything that is not a digit. We assume	*/
			/* that the formatting will be like inserting	*/
			/* thousands separators.			*/
			t = "" ;
			{
			for (uint cs1 = 0 ; cs1 < text.length() ; cs1 += 1)
			{
				QChar ch = text.at(cs1) ;

				if (ch.isDigit() || (ch == '-') || (ch == '+'))
					t.append (ch) ;
			}
			}
			return	t ;

		case KB::ITFloat	:
			/* Similar to fixed point above but allow the	*/
			/* decimal point character and 'E' (for		*/
			/* exponent) through.				*/
			t = "" ;
			{
			for (uint cs2 = 0 ; cs2 < text.length() ; cs2 += 1)
			{
				QChar ch = text.at(cs2) ;

				if (ch.isDigit() || (ch == '-') || (ch == '+'))
				{	t.append (ch) ;
					continue ;
				}
				if (ch == fi->decimalPoint)
				{	t.append ('.') ;
					continue ;
				}
				if ((ch == 'E') || (ch == 'e'))
				{	t.append (ch) ;
					continue ;
				}
			}
			}
			fprintf(stderr, "deFormat: .... [%s]\n", (cchar *)t);
			return	t ;

		default	:
			break	;
	}

	/* This is likely an error since all types should have been	*/
	/* caught above, but silently pass the text back. Errors will	*/
	/* likely be caught elsewhere.					*/
	return	text	;
}

/*  ------------------------------------------------------------------  */

/*  The following routines handle the KBDataArray structure used to	*/
/*  store underlying data.						*/

static	int	numDataArrays	;

/*  allocData	: Allocate a data array and load data			*/
/*  data	: cchar *	: Data buffer				*/
/*  len		: uint		: Data buffer length			*/
/*  (returns)	: KBDataArray *	: Data array				*/

static	KBDataArray *allocData
	(	cchar		*data,
		uint		len
	)
{
	KBDataArray *array = (KBDataArray *)malloc(KBDASIZE(len)) ;

	array->m_refCount  = 1	 ;
	array->m_length	   = len ;
	array->m_data[len] = 0	 ;

	memcpy	(&array->m_data[0], data, len) ;

	numDataArrays	  += 1	 ;

	return	array ;
}

/*  allocData	: Allocate a data array and load data			*/
/*  data	: QCString	: Data buffer				*/
/*  (returns)	: KBDataArray *	: Data array				*/

static	KBDataArray *allocData
	(	QCString	data
	)
{
	uint	    len	   = data.length() ;
	KBDataArray *array = (KBDataArray *)malloc(KBDASIZE(len)) ;

	array->m_refCount  = 1	 ;
	array->m_length	   = len ;
	array->m_data[len] = 0	 ;

	memcpy	(&array->m_data[0], data.data(), len) ;

	numDataArrays	  += 1	 ;

	return	array ;
}

/*  allocData	: Allocate a empty data array data			*/
/*  len		: uint		: Data buffer length			*/
/*  (returns)	: KBDataArray *	: Data array				*/

static	KBDataArray *allocData
	(	uint		len
	)
{
	KBDataArray *array = (KBDataArray *)malloc(KBDASIZE(len)) ;

	array->m_refCount  = 1	 ;
	array->m_length	   = len ;
	array->m_data[len] = 0	 ;

	memset	(&array->m_data[0], 0, len) ;

	numDataArrays	  += 1	 ;

	return	array ;
}

/*  derefData	: Dereference a data array				*/
/*  array	: KBDataArray *	: The data array			*/
/*  (returns)	: void		:					*/

inline	void	derefData
	(	KBDataArray	*array
	)
{
	if ((array->m_refCount -= 1) == 0)
	{
		free ((void *)array) ;
		numDataArrays -= 1   ;
	}
}

/*  refData	: Add reference to a data array				*/
/*  array	: KBDataArray *	: The data array			*/
/*  (returns)	: void		:					*/

inline	void	refData
	(	KBDataArray	*array
	)
{
	array->m_refCount += 1 ;
}


/*  ------------------------------------------------------------------  */

/*  KBValue								*/
/*  KBValue	: Constructor for KBValue class				*/
/*  (returns)	: KBValue						*/

KBValue::KBValue ()
{
	type	= &_kbUnknown	;
	data	= 0		;
	d	= 0		;
}

/*  KBValue								*/
/*  KBValue	: Constructor for KBValue class				*/
/*  v		: const KBValue & : Extant value			*/
/*  (returns)	: KBValue	  :					*/

KBValue::KBValue
	(	const KBValue	&v
	)
	:
	type	(v.type)
{
	if ((data = v.data) != 0)
		refData (data) ;

	if ((d = v.d) != 0)
		d->ref () ;

	type->ref  () ;
}

/*  KBValue								*/
/*  KBValue	: Constructor for KBValue class				*/
/*  v		: const KBValue & : Extant value			*/
/*  type	: KBType *	  : Type information			*/
/*  (returns)	: KBValue	  :					*/

KBValue::KBValue
	(	const KBValue	&v,
		KBType		*type
	)
	:
	type	(type)
{
	if ((data = v.data) != 0)
		refData (data) ;

	if (data != 0)
		switch (type->getIType())
		{
			case KB::ITDate     :
			case KB::ITTime     :
			case KB::ITDateTime :
				setDateTime ()	;
				break	;

			default	:
				d	= 0	;
				break	;
		}
	else	d	= 0	;

	type->ref  () ;
}

/*  KBValue								*/
/*  KBValue	: Constructor for KBValue class				*/
/*  type	: KBType *	  : Type information			*/
/*  (returns)	: KBValue	  :					*/

KBValue::KBValue
	(	KBType	*type
	)
	:
	type	(type)
{
	d	= 0  ;
	data	= 0  ;
	type->ref () ;
}

/*  KBValue								*/
/*  KBValue	: Constructor for KBValue class				*/
/*  text	: const QString & : Text				*/
/*  type	: KBType *	  : Type information			*/
/*  (returns)	: KBValue						*/

KBValue::KBValue
	(	const QString 	&text,
		KBType		*type
	)
	:
	type	(type )
{
	store	(text.utf8  ()) ;

	if (data != 0)
		switch (type->getIType())
		{
			case KB::ITDate     :
			case KB::ITTime     :
			case KB::ITDateTime :
				setDateTime ()	;
				break	;

			default	:
				d	= 0	;
				break	;
		}
	else	d	= 0	;

	type->ref () ;
}

/*  KBValue								*/
/*  KBValue	: Constructor for KBValue class				*/
/*  text	: const QString & : Text				*/
/*  type	: KBType *	  : Type information			*/
/*  (returns)	: KBValue						*/

KBValue::KBValue
	(	const QString 	&text,
		KBType		*type,
		const QString	&fmt
	)
	:
	type	(type )
{
	store	(deFormat(text, type, fmt).utf8()) ;

	if (data != 0)
		switch (type->getIType())
		{
			case KB::ITDate     :
			case KB::ITTime     :
			case KB::ITDateTime :
				setDateTime ()	;
				break	;

			default	:
				d	= 0	;
				break	;
		}
	else	d	= 0	;

	type->ref () ;
}

/*  KBValue								*/
/*  KBValue	: Constructor for KBValue class				*/
/*  bytes	: const QByteArray & : Data				*/
/*  type	: KBType *	     : Type information			*/
/*  codec	: QTextCodec *	     : Non-default codec		*/
/*  (returns)	: KBValue						*/

KBValue::KBValue
	(	const QByteArray &bytes,
		KBType		 *type,
		QTextCodec	 *codec
	)
	:
	type	(type)
{
	if (!bytes.isNull())
	{
		if ((codec != 0) && (type->getIType() != KB::ITBinary))
		{
			QString	unicode = codec->toUnicode(bytes.data(), bytes.size())  ;
			cchar	*text	= (cchar *)unicode ;
			data	= allocData (text, strlen(text)) ;

//			fprintf
//			(	stderr,
//				"KBValue::KBValue: [%.*s]->[%s]\n",
//				bytes.size(),
//				bytes.data(),
//				text
//			)	;
		}
		else	
			data	= allocData (bytes.data(), bytes.size()) ;
	}
	else	data	= 0 ;

	if (data != 0)
		switch (type->getIType())
		{
			case KB::ITDate     :
			case KB::ITTime     :
			case KB::ITDateTime :
				setDateTime ()	;
				break	;

			default	:
				d	= 0	;
				break	;
		}
	else	d	= 0	;

	type->ref () ;
}

/*  KBValue								*/
/*  KBValue	: Constructor for KBValue class				*/
/*  bytes	: cchar *	: Data buffer				*/
/*  len		: uint		: Buffer size				*/
/*  type	: KBType *	: Type information			*/
/*  codec	: QTextCodec *	: Non-default codec			*/
/*  (returns)	: KBValue						*/

KBValue::KBValue
	(	cchar		*bytes,
		KBType		*type,
		QTextCodec	*codec
	)
	:
	type	(type)
{
	if (bytes != 0)
	{
		if ((codec != 0) && (type->getIType() != KB::ITBinary))
		{
			data	= allocData (codec->toUnicode(bytes).utf8()) ;
//
//			fprintf
//			(	stderr,
//				"KBValue::KBValue: [%s]->[%s]\n",
//				bytes,
//				data->m_data
//			)	;
		}
		else	data	= allocData (bytes, strlen(bytes)) ;

		switch (type->getIType())
		{
			case KB::ITDate     :
			case KB::ITTime     :
			case KB::ITDateTime :
				setDateTime ()	;
				break	;

			default	:
				d	= 0	;
				break	;
		}
	}
	else
	{	data	= 0	;
		d	= 0	;
	}

	type->ref () ;
}

/*  KBValue								*/
/*  KBValue	: Constructor for KBValue class				*/
/*  bytes	: cchar *	: Data buffer				*/
/*  blen	: uint		: Buffer length				*/
/*  type	: KBType *	: Type information			*/
/*  codec	: QTextCodec *	: Non-default codec			*/
/*  (returns)	: KBValue						*/

KBValue::KBValue
	(	cchar		*bytes,
		uint		blen,
		KBType		*type,
		QTextCodec	*codec
	)
	:
	type	(type)
{
	if (bytes != 0)
	{
		if ((codec != 0) && (type->getIType() != KB::ITBinary))
			data	= allocData (codec->toUnicode(bytes, blen).utf8()) ;
		else	data	= allocData (bytes, blen) ;

		switch (type->getIType())
		{
			case KB::ITDate     :
			case KB::ITTime     :
			case KB::ITDateTime :
				setDateTime ()	;
				break	;

			default	:
				d	= 0	;
				break	;
		}
	}
	else
	{	data	= 0	;
		d	= 0	;
	}

	type->ref () ;
}

/*  KBValue								*/
/*  KBValue	: Constructor for KBValue class				*/
/*  num		: int		  : Integer value			*/
/*  type	: KBType *	  : Type information			*/
/*  (returns)	: KBValue						*/

KBValue::KBValue
	(	int		num,
		KBType		*type
	)
	:
	type	(type)
{
	store	(QCString().setNum(num)) ;

	d	= 0  ;
	type->ref () ;
}

/*  KBValue								*/
/*  KBValue	: Constructor for KBValue class				*/
/*  dbl		: double	  : Double value			*/
/*  type	: KBType *	  : Type information			*/
/*  (returns)	: KBValue						*/

KBValue::KBValue
	(	double		dbl,
		KBType		*type
	)
	:
	type	(type)
{
	store	(QCString().setNum(dbl, 'g', 14)) ;

	d	= 0  ;
	type->ref () ;
}

KBValue::KBValue
	(	const QDateTime	&dt,
		KBType		*type
	)
	:
	type	(type)
{
	store	    (dt.toString("yyyy-MM-hh hh:mm:ss").utf8()) ;
	setDateTime (dt) ;
}

/*  KBValue								*/
/*  KBValue								*/
/*  ~KBValue	: Destructor for KBValue class				*/
/*  (returns)	:		:					*/

KBValue::~KBValue ()
{
	if (data != 0) derefData (data) ;
	if (d    != 0) d   ->deref () ;
	if (type != 0) type->deref () ;
}


void	KBValue::store
	(	const QCString	&cstr
	)
{
	if (cstr.isNull())
		data	= 0 ;
	else	data	= allocData (cstr.data(), cstr.length()) ;
}

void	KBValue::setDateTime ()
{
	d	= new KBDateTime (QString::fromUtf8(data->m_data, data->m_length)) ;
}

void	KBValue::setDateTime
	(	const QDateTime	&dt
	)
{
	d	= new KBDateTime (dt) ;
}


/*  KBValue								*/
/*  operator =	: Assignment operator					*/
/*  v		: const KBValue & : Value to be assigned		*/
/*  (returns)	: const KBValue & : New value				*/

const	KBValue	&KBValue::operator =
	(	const KBValue	&v
	)
{
	type->deref ()	 ;
	if (d    != 0) d->deref  () ;
	if (data != 0) derefData (data) ;

	type	= v.type ;
	data	= v.data ;
	d	= v.d	 ;

	type->ref   ()	 ;
	if (d    != 0) d->ref  () ;
	if (data != 0) refData (data) ;

	return	*this	;
}

/*  KBValue								*/
/*  operator =	: Assignment operator					*/
/*  v		: const QString & : Value to be assigned		*/
/*  (returns)	: const KBValue & : New value				*/

const	KBValue	&KBValue::operator =
	(	const QString	&v
	)
{
	if (d    != 0) d->deref  () ;
	if (data != 0) derefData (data) ;

	type->deref() ;

	type	= &_kbString ;
	d	= 0	     ;
	store	(v.utf8())   ;

	return	*this ;
}

/*  KBValue								*/
/*  operator =	: Assignment operator					*/
/*  v		: cchar * 	  : Value to be assigned		*/
/*  (returns)	: const KBValue & : New value				*/

const	KBValue	&KBValue::operator =
	(	cchar		*v
	)
{
	if (d    != 0) d->deref  () ;
	if (data != 0) derefData (data) ;

	d	= 0 ;

	if (v == 0)
		data	= 0	;
	else	data	= allocData (v, strlen(v)) ;

	if (type == &_kbUnknown)
		type = &_kbString ;

	return	*this	 ;
}

/*  KBValue								*/
/*  operator ==	: Equality operator					*/
/*  v		: const KBValue & : Other operand			*/
/*  (returns)	: bool		  : Equal				*/

bool	KBValue::operator ==
	(	const KBValue	&v
	)	const
{
	if ((data == 0) && (v.data == 0))
		return	true  ;

	if ((data == 0) || (v.data == 0))
		return	false ;

	if (data->m_length != v.data->m_length)
		return	false ;

	return	memcmp (data->m_data, v.data->m_data, data->m_length) == 0 ;
}

/*  KBValue								*/
/*  operator !=	: Inequality operator					*/
/*  v		: const KBValue & : Other operand			*/
/*  (returns)	: bool		  : Equal				*/

bool	KBValue::operator !=
	(	const KBValue	&v
	)	const
{
	return	! (*this == v) ;
}

/*  KBValue								*/
/*  getRawText	: Get raw value text					*/
/*  (returns)	: QString	  : Text				*/

QString	KBValue::getRawText  () const
{
	return	data == 0 ?
			QString::null :
			QString::fromUtf8(data->m_data, data->m_length) ;
}


static	QString	zeroPad
	(	QString		string,
		uint		len
	)
{
	QString	pfx	;

	if ((string.at(0) == '+') || (string.at(0) == '-'))
	{
		pfx	= string.at (0) ;
		string	= string.mid(1) ;
		len	= len == 0 ? 0 : len - 1 ;
	}

	while (string.length() < len) string = "0" + string ;

	return	pfx + string	;
}

/*  KBValue								*/
/*  formatNumber: Apply general number format				*/
/*  ftext	: const QString & : Format				*/
/*  ok		: bool *	  : Return validity			*/
/*  (returns)	: QString	  : Formatted text			*/

QString	KBValue::formatNumber
	(	const QString		&ftext,
		bool			*ok
	)
	const
{
	KBFormatInfo *fi    = getFormatInfo () ;

	QStringList  fbits  = QStringList::split (';', ftext) ;
	double	     value  = data == 0 ? 0.0 : strtod (data->m_data, 0) ;
	QString	     format ;

	/* Various cases:						*/
	/* Value is null then use 4th string if present		else	*/
	/* Value zero then use 3rd string if present		else	*/
	/* Value negative then use 2nd string if present	else	*/
	/* Use first string						*/
	if	((data == 0) && (fbits.count() >= 4))
	{
		format	= fbits[3] ;
	}
	else if	((value == 0.0) && (fbits.count() >= 3))
	{
		format	= fbits[2] ;
	}
	else if ((value <  0.0) && (fbits.count() >= 2))
	{
		format	= fbits[1] ;
		value	= - value  ;
	}
	else
	{
		format	= fbits[0] ;
	}

	uint	lWhole	= 0	;
	uint	lFract	= 0	;
	uint	lExp	= 0	;
	bool	dp	= false	;
	bool	sci	= false	;
	bool	perc	= false	;
	bool	quot1	= false	;
	uint	idx1	= 0	;
	QChar	ch	;

	/* First loop scans the format string looking for characters	*/
	/* which have special significance. Note that silly format	*/
	/* strings will give unpredictable results!			*/
	while (idx1 < format.length())
	{
		QChar	qch	= format[idx1]	;
		char	ch	= qch		;

		if (quot1)
		{
			if (ch == '"') quot1 = false ;
			idx1 += 1 ;
			continue  ;
		}

		switch (ch)
		{
			case '.'  :
				/* Decimal point. This should acutally	*/
				/* map to the locale-specific character	*/
				/* Note that we have hit this case.	*/
				dp = true	;
				break		;

			case '0'  :
			case '#'  :
				/* Zero and zero-space characters. The	*/
				/* length count increments depending on	*/
				/* which part we are in.		*/
				if	(sci) lExp   += 1 ;
				else if (dp ) lFract += 1 ;
				else	      lWhole += 1 ;
				break	;

			case '\\' :
				/* Next character is escaped so skip	*/
				/* over that as well.			*/
				idx1	+= 1	;
				break	;

			case 'E'  :
			case 'e'  :
				/* Number is to be output in scientific	*/
				/* (exponential) format.		*/
				sci	= true	;
				break	;

			case '%'  :
				/* Number will be treated as a		*/
				/* percentage.				*/
				perc	= true	;
				break	;

			case '"'  :
				/* Start quoting to include characters	*/
				/* literaly to the closing quote.	*/
				quot1	= true	;
				break	;

			default	  :
				/* Any other character will stand for	*/
				/* itself.				*/
				break	 ;
		}

		idx1	+= 1 ;
	}

	/* If the number is to be treated as a percentage then the	*/
	/* value is multipled by 100 ...				*/
	if (perc)
		value *= 100.0	;

	QStringList	vBits	;

	if (sci)
	{
		/* Scientific output. Convert the number to a canonical	*/
		/* exponential format, split it into its components,	*/
		/* and then adjust to get the right number of digits.	*/
		QString	 vText	= QString("%1").arg(value, 0, 'e', lWhole + lFract - 1) ;
		vBits		= QStringList::split (QRegExp("[.Ee]"), vText) ;
		int	 exp	= vBits[2].toInt()  ;

		while ((vBits[0].length() < lWhole) && (vBits[1].length() > 0))
		{	vBits[0] += vBits[1].at (0) ;
			vBits[1]  = vBits[1].mid(1) ;
			exp	 -= 1 ;
		}

		vBits[2] = QString("%1").arg(exp) ;
	}
	else
	{
		/* Normal decimal format, convert value to text and	*/
		/* then split into parts.				*/
		QString	 vText  = QString("%1").arg(value, 0, 'f', lFract) ;
		vBits		= QStringList::split ('.', vText) ;
	}

	uint	part	= 0 	;
	uint	idx2	= 0 	;
	uint	sptr	= 0 	;
	bool	quot2	= false	;
	bool	ongo	= false	;
	QString	stext	= zeroPad (vBits[0], lWhole) ;

	QString	res	;

	while (idx2 < format.length())
	{
		if (quot2)
		{
			if (ch == '"')
				quot1	 = false ;
			else	res	+= ch	 ;

			idx1 += 1 ;
			continue  ;
		}

		switch ((char)(ch = format.at(idx2)))
		{
			case '.'  :
				/* Add the nominal decimal point, then	*/
				/* move on to the fractional part of	*/
				/* the number.				*/
				res	+= fi->decimalPoint ;
				if (part == 0)
				{	part	= 1	;
					sptr	= 0	;
					ongo	= false	;
					stext	= zeroPad (vBits[1], lFract) ;
				}
				break		;

			case ','  :
				/* This is a thousand separator, and	*/
				/* only appears once the number has	*/
				/* started.				*/
				if (ongo)
					res	+= fi->thousandSep	;
				else	res	+= " " ;
				break	;

			case '0'  :
				/* Zero spbstitutes the next digit in	*/
				/* the number. Flag that we are now	*/
				/* outputing digits.			*/
				res	+= stext[sptr] ;
				sptr	+= 1	;
				ongo	 = true	;
				break	;

			case '#'  :
				/* Has outputs a digit if the digit is	*/
				/* non-zero, or the number has already	*/
				/* been started.			*/
				if ((stext[sptr] != '0') || ongo)
				{	res	+= stext[sptr] ;
					ongo	 = true	;
				}
				else	res	+= ' '	;

				sptr	+= 1	;
				break	;

			case '$'  :
				/* Substitute the currency symbol ...	*/
				res	+= fi->currencySymbol ;
				break	;

			case '\\' :
				/* Escape the following character.	*/
				idx1	+= 1	;
				res	+= format[idx1] ;
				break	;

			case 'E'  :
			case 'e'  :
				/* Output the exponentiation character	*/
				/* and move to that part of the number.	*/
				res	+= ch	;
				if (part == 1)
				{	part	= 2	;
					sptr	= 0	;
					ongo	= false	;
					stext	= zeroPad(vBits[2], lExp) ;
				}
				break	;

			case '"'  :
				/* Start quoting to include characters	*/
				/* literaly to the closing quote.	*/
				quot2	= true	;
				break	;

			default	  :
				/* Anything else outputs as itself.	*/
				res	+= format.at(idx2) ;
				break	 ;
		}

		idx2	+= 1 ;
	}

	if (ok != 0) *ok = true	;
	return	res ;
}

/*  KBValue								*/
/*  formatCurrency							*/
/*		: Apply currency format					*/
/*  ftext	: const QString & : Format				*/
/*  ok		: bool *	  : Return validity			*/
/*  (returns)	: QString	  : Formatted text			*/

QString	KBValue::formatCurrency
	(	const QString		&ftext,
		bool			*ok
	)
	const
{
	KBFormatInfo	*fi	= getFormatInfo (ftext) ;
	double		value 	= data == 0 ? 0.0 : strtod (data->m_data, 0) ;
	bool		neg	= false	 ;

	if (value < 0)
	{	neg	= true	 ;
		value	= -value ;
	}

	QString	vtext	= QString("%1").arg(value, 0, 'f', 2) ;
	QString	res	;

	int	dotpos	= vtext.find('.') ;
	if (dotpos == -1)
	{	dotpos	= vtext.length()  ;
		vtext.append  ('.') 	  ;
	}
	else	vtext.replace (dotpos, 1, fi->currencyPoint) ;

	while ((dotpos -= 3) > 0)
		vtext.insert (dotpos, fi->currencySep) ;

	if (neg)
		switch (fi->currencyNeg)
		{
			case CN_BEFOREALL	:
				res	= QString("%1-%2" ).arg(fi->currencySymbol).arg(vtext) ;
				break	;

			case CN_AFTERALL	:
				res	= QString("%1%2-" ).arg(fi->currencySymbol).arg(vtext) ;
				break	;

			case CN_BEFORESYM	:
				res	= QString("-%1%2" ).arg(fi->currencySymbol).arg(vtext) ;
				break	;

			case CN_AFTERSYM	:
				res	= QString("%1-%2" ).arg(fi->currencySymbol).arg(vtext) ;
				break	;

			default	:
				res	= QString("(%1%2)").arg(fi->currencySymbol).arg(vtext) ;
				break	;
		}
	else
		res	= QString("%1%2").arg(fi->currencySymbol).arg(vtext) ;

	if (ok != 0) *ok = true	;
	return	res	;
}

/*  KBValue								*/
/*  getText	: Get textual value					*/
/*  fmt		: const QString & : Format specification		*/
/*  ok		: bool *	  : Return validity			*/
/*  (returns)	: QString	  : Text				*/

QString	KBValue::getText
	(	const QString	&fmt,
		bool		*ok
	)
	const
{
	/* Assume valid until shown otherwise				*/
	if (ok != 0) *ok = true	;

	/* Null is handled irrespective of any format whatsoever. Also,	*/
	/* if the type is Unknown or Raw, or the format is empty then	*/
	/* just use the raw text string.				*/
	if ( (data == 0) 
		 || (type->getIType() == KB::ITUnknown)
		 || (type->getIType() == KB::ITRaw    ) || fmt.isEmpty() )
		return	getRawText () ;

	/* Split the format specification into (type:format). If it	*/
	/* does not split then return an error text unless the format	*/
	/* specification is empty.					*/
	QString	fType	;
	QString	fForm	;
	QString	r	;
	int	cs	;

	if ((cs = fmt.find (':')) >= 0)
	{	fType	= fmt.left (cs) ;
		fForm	= fmt.mid  (cs + 1) ;
	}
	else if (!fmt.isEmpty())
	{	if (ok != 0) *ok = false ;
		return	"format?" + fmt  ;
	}

	/* Now decide what to do based on the internal type and the	*/
	/* format specification, if any.				*/
	switch (type->getIType())
	{
		case KB::ITString	:
			/* Check that the format type is empty or set	*/
			/* to "string". If OK then format accordingly.	*/
			/* Note that this is only useful for read-only	*/
			/* fields, since we don't reverse the format.	*/
			if (!fType.isEmpty() && (fType != "String"))
			{	if (ok != 0) *ok = false ;
				return	QString("%1?%2").arg(fType).arg(data->m_data) ;
			}
			if (fForm.isEmpty()) return getRawText() ;

			/* QString::sprintf expects %s arguments to be	*/
			/* in UTF-8 encoding, so this works correctly.	*/
			r.sprintf (fForm, data->m_data) ;
			return	r ;
			

		case KB::ITFixed	:
			/* As for String but the format type must be	*/
			/* set to "Number" or "Fixed".			*/
			if (fType == "Number")
				return	formatNumber   (fForm, ok) ;
			if (fType == "Currency")
				return	formatCurrency (fForm, ok) ;
			if (!fType.isEmpty() && (fType != "Fixed"   ))
			{	if (ok != 0) *ok = false ;
				return	QString("%1?%2").arg(fType).arg(data->m_data) ;
			}
			if (fForm.isEmpty()) return getRawText () ;

			r.sprintf (fForm, strtol(data->m_data, 0, 0)) ;
			return	r ;

		case KB::ITFloat	:
			/* Much as for fixed ...			*/
			if (fType == "Number")
				return	formatNumber   (fForm, ok) ;
			if (fType == "Currency")
				return	formatCurrency (fForm, ok) ;
			if (!fType.isEmpty() && (fType != "Float"   ))
			{	if (ok != 0) *ok = false ;
				return	QString("%1?%2").arg(fType).arg(data->m_data) ;
			}
			if (fForm.isEmpty()) return getRawText () ;

			r.sprintf (fForm, strtod(data->m_data, 0)) ;
			return	r ;

		case KB::ITDate		:
			/* In the case of a date, if there is a format	*/
			/* then apply it to the decoded data, which is	*/
			/* actually a "KBDateTime".			*/
			if (!fType.isEmpty() && (fType != "Date"    ))
			{	if (ok != 0) *ok = false ;
				return	QString("%1?%2").arg(fType).arg(data->m_data) ;
			}
			if (fForm.isEmpty() || (d == 0)) return getRawText () ;
			if (!((KBDateTime *)d)->isValid ())
			{	if (ok != 0) *ok = false ;
				return	"" ;
			}
			return	((KBDateTime *)d)->format (fForm) ;

		case KB::ITTime		:
			/* Similarly for time ...			*/
			if (!fType.isEmpty() && (fType != "Time"    ))
			{	if (ok != 0) *ok = false ;
				return	QString("%1?%2").arg(fType).arg(data->m_data) ;
			}
			if (fForm.isEmpty() || (d == 0)) return getRawText () ;
			return	((KBDateTime *)d)->format (fForm) ;

		case KB::ITDateTime	:
			/* ... and date/time				*/
			if (!fType.isEmpty() && (fType != "DateTime"))
			{	if (ok != 0) *ok = false ;
				return	QString("%1?%2").arg(fType).arg(data->m_data) ;
			}
			if (fForm.isEmpty() || (d == 0)) return getRawText () ;
			return	((KBDateTime *)d)->format (fForm) ;

		case KB::ITBinary	:
			/* Binary data is treated as a vector of bytes	*/
			/* of specified length. In this case we ignore	*/
			/* the format completely; also, getting the	*/
			/* contents of the data as a string may be	*/
			/* meaningless, but that is the user's problem.	*/
			return	getRawText () ;

		case KB::ITDriver	:
			/* For driver-specific types assume that the	*/
			/* raw value is correctly serialised.		*/
			return	getRawText () ;

		case KB::ITBool		:
			/* Boolean. Use whatever the server database	*/
			/* passed back as text.				*/
			return	getRawText () ;

		default	:
			KBError::EFault
			(
				QString (TR("KBValue::getText: Unknown type %1")).arg((int)type->getIType()),
				QString::null,
				__ERRLOCN
			)	;
			break	;
	}

	return	getRawText ()	;
}


/*  KBValue								*/
/*  getQueryText: Get value for use in a query				*/
/*  buffer	: KBDataBuffer & : Result buffer			*/
/*  codec	: QTextCodec *	 : Non-default codec			*/
/*  (returns)	: void		 :					*/

void	KBValue::getQueryText
	(	KBDataBuffer	&buffer,
		QTextCodec	*codec
	)
	const
{
	/* The query text is generated by the server-specific type,	*/
	/* except that we assume that null values are always expressed	*/
	/* as "null". This version of the method adds the text to the	*/
	/* supplied buffer.						*/
	if (data == 0)
		buffer.append 	   ("null") ;
	else	type->getQueryText (data, d, buffer, codec) ;
}

/*  KBValue								*/
/*  getQueryText: Get value for use in a query				*/
/*  (returns)	: QString	:					*/

QString	KBValue::getQueryText () const
{
	/* The query text is generated by the server-specific type,	*/
	/* except that we assume that null values are always expressed	*/
	/* as "null". This method just returns the text, for use in the	*/
	/* query logger.						*/
	return	data == 0 ? QString("null") : type->getQueryText (data, d) ;
}

/*  KBValue								*/
/*  getType	: Get value type object					*/
/*  (returns)	: KBType *	: Type object				*/

KBType	*KBValue::getType () const
{
	return	type	;
}

/*  KBValue								*/
/*  isNull	: Test if value is null					*/
/*  (returns)	: bool		: True if null				*/

bool	KBValue::isNull () const
{
	return	data == 0 ;
}

/*  KBValue								*/
/*  isEmpty	: Test if value is empty				*/
/*  (returns)	: bool		: True if empty				*/

bool	KBValue::isEmpty () const
{
	return	data == 0 ? true : data->m_length == 0 ;
}

/*  KBValue								*/
/*  setFalse	: Set value to notional false				*/
/*  (returns)	: void		:					*/

void	KBValue::setFalse ()
{
	*this	= KBValue () ;
}

/*  KBValue								*/
/*  setTrue	: Set value to notional true				*/
/*  (returns)	: void		:					*/

void	KBValue::setTrue ()
{
	*this	= 1 ;
}

/*  KBValue								*/
/*  isTrue	: Deterine if value is true				*/
/*  (returns)	: bool		: True if true				*/

bool	KBValue::isTrue () const
{
	if (data == 0) return false ;


	switch (type->getIType())
	{
		case KB::ITString	:
		case KB::ITFixed	:
			/* Any non-zero value is true.			*/
			return	getRawText().toInt   () != 0 ;

		case KB::ITFloat	:
			/* Just as for fixed ...			*/
			return	getRawText().toDouble() != 0 ;

		case KB::ITDate		:
		case KB::ITTime		:
		case KB::ITDateTime	:
			/* Always treated as true.			*/
			return	true	;

		case KB::ITBinary	:
			/* Any non-empty data is considered true.	*/
			return	data->m_length > 0 ;

		case KB::ITBool		:
			/* Default is to generate zero for false and	*/
			/* one for true. May be overridden by the	*/
			/* database drivers.				*/
			{
				QString	cs  = getRawText().lower() ;
				bool	 ok ;
				int	 tv ;

				if (cs == "yes"  ) return true	;
				if (cs == "true" ) return true	;
				if (cs == "t"    ) return true	;

				if (cs == "no"   ) return false	;
				if (cs == "false") return false	;
				if (cs == "f"    ) return false	;

				tv = cs.toInt(&ok) ;
				if (!ok) tv = cs.length() > 0 ;

				return tv	;
			}

		default	:
			KBError::EFault
			(
				QString (TR("KBValue::isTrue: Unknown type %1")).arg((int)type->getIType()),
				QString::null,
				__ERRLOCN
			)	;
			break	;
	}

	return	false	;
}

/*  KBValue								*/
/*  getDateTime	: Get date and time object				*/
/*  (returns)	: KBDateTime *	: Date and time or null if not such	*/

const KBDateTime *KBValue::getDateTime() const
{
	switch (type->getIType())
	{
		case KB::ITDate	    :
		case KB::ITTime	    :
		case KB::ITDateTime :
			return	(KBDateTime *)d ;

		default	:
			break	;
	}

	return	0 ;
}

void	KBValue::setRawData
	(	QByteArray	&ba
	)
	const
{
	if (data == 0)
		ba.setRawData   (0, 0) ;
	else	ba.setRawData   (data->m_data, data->m_length) ;
}

void	KBValue::resetRawData
	(	QByteArray	&ba
	)
	const
{
	if (data == 0)
		ba.resetRawData (0, 0) ;
	else	ba.resetRawData (data->m_data, data->m_length) ;
}

/*  KBValue								*/
/*  preallocate	: Preallocate value to empty data array			*/
/*  length	: uint		: Required length			*/
/*  (returns)	: char *	: Data buffer				*/

char	*KBValue::preallocate
	(	uint		length
	)
{
	/* This method is used in drivers and elsewhere where a large	*/
	/* to be read. By preallocating, we can read direct into the	*/
	/* value and save an allocate-copy-deallocate cycle.		*/
	if (d    != 0) d->deref  () ;
	if (data != 0) derefData (data) ;

	d	= 0 ;
	data	= allocData (length) ;

	return	data->m_data ;
}

/*  ------------------------------------------------------------------  */

/*  getNumDataArrays							*/
/*		: Return the number of data arrays			*/
/*  (returns)	: int		: Number of data arrays			*/

int	LIBCOMMON_API	getNumDataArrays ()
{
	/* This is used for debugging, to check that we don't forget to	*/
	/* remove orphaned data arrays.					*/
	return	numDataArrays ;
}
