/***************************************************************************
    file	         : kb_table.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<time.h>
#include	<sys/types.h>

#ifndef 	_WIN32
#include	<unistd.h>
#else
#include	<process.h>
#define 	getpid _getpid
#endif

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_location.h"

#include	"kb_database.h"
#include	"kb_dblink.h"
#include	"kb_select.h"
#include	"kb_node.h"
#include	"kb_table.h"
#include	"kb_tableinfo.h"
#include	"kb_nodereg.h"


/*  KBTable								*/
/*  KBTable	: Constructor for table object				*/
/*  parent	: KBNode *	   : Parent node			*/
/*  aList	: QDict<QString> & : Attribute dictionary		*/
/*		: bool *	:					*/
/*  (returns)	: KBTable	   :					*/

KBTable::KBTable
	(	KBNode			*parent,
		const QDict<QString>	&aList,
		bool			*
	)
	:
	KBNode		(parent, "KBTable",	aList),
	m_ident		(this,	"ident",	aList),
	m_table		(this,	"table",	aList),
	m_alias		(this,	"alias",	aList),
	m_primary	(this,	"primary",	aList),
	m_ptype		(this,	"ptype",	aList),
	m_pexpr		(this,	"pexpr",	aList),
	m_parent	(this,	"parent",	aList),
	m_field		(this,	"field",	aList),
	m_field2	(this,	"field2",	aList),
	m_where		(this,	"where",	aList),
	m_order		(this,	"order",	aList),
	m_jtype		(this,	"jtype",	aList),
	m_x		(this,	"x",		aList),
	m_y		(this,	"y",		aList),
	m_w		(this,	"w",		aList),
	m_h		(this,	"h",		aList)
{
	m_grouped	= false	;
	m_uflags	= 0	;

	fprintf
	(	stderr,
		"KBTable::KBTable: table=[%s] name=[%s]\n",
		(cchar *)getTable(),
		(cchar *)getName ()
	)	;

	if (getTable().isEmpty())
	{	QString	*name	= aList.find("name") ;
		if (name != 0) m_table.setValue (*name) ;
	}
}

/*  KBTable								*/
/*  KBTable	: Constructor for table object				*/
/*  table	: const QString & : Table name				*/
/*  alias	: const QString & : Possible alias			*/
/*  primary	: const QString & : Primary key column			*/
/*  ptype	: const QString & : Primary key type (as string)	*/
/*  field	: const QString & : Field linking to parent		*/
/*  field2	: const QString & : Link field in parent		*/
/*  where	: const QString & : Where criteria			*/
/*  order	: const QString & : Order criteria			*/
/*  x		: uint		  : Design layout X-position		*/
/*  y		: uint		  : Design layout Y-position		*/
/*  w		: uint		  : Design layout width			*/
/*  h		: uint		  : Design layout height		*/
/*  (returns)	: KBTable	  :					*/

KBTable::KBTable
	(	KBNode		*parent,
		const QString	&table,
		const QString	&alias,
		const QString	&primary,
		const QString	&ptype,
		const QString	&pexpr,
		const QString	&field,
		const QString	&field2,
		const QString	&where,
		const QString	&order,
		uint		x,
		uint		y,
		uint		w,
		uint		h
	)
	:
	KBNode		(parent, "KBTable"),
	m_ident		(this,	"ident",	"" 	),
	m_table		(this,	"table",	table	),
	m_alias		(this,	"alias",	alias	),
	m_primary	(this,	"primary",	primary	),
	m_ptype		(this,	"ptype",	ptype	),
	m_pexpr		(this,	"pexpr",	pexpr	),
	m_parent	(this,	"parent",	""	),
	m_field		(this,	"field",	field	),
	m_field2	(this,	"field2",	field2	),
	m_where		(this,	"where",	where	),
	m_order		(this,	"order",	order	),
	m_jtype		(this,	"jtype",	""	),
	m_x		(this,	"x",		x	),
	m_y		(this,	"y",		y	),
	m_w		(this,	"w",		w	),
	m_h		(this,	"h",		h	)
{
	static	long	idbase	;
	static	int	idseq	;

	if (idbase == 0) idbase = time (0) ;

	m_ident.setValue(QString("%1.%2.%3").arg(getpid()).arg(idbase).arg(idseq)) ;
	idseq += 1 ;

	m_grouped	= false	;
	m_uflags	= 0	;
}

/*  KBTable								*/
/*  KBTable	: Constructor for table object				*/
/*  parent	: KBNode *	: Parent node				*/
/*  table	: KBTable *	: Table node to be replicated		*/
/*  (returns)	: KBTable	:					*/

KBTable::KBTable
	(	KBNode		*parent,
		KBTable		*table
	)
	:
	KBNode		(parent, "KBTable"),
	m_ident		(this,	"ident",	table),
	m_table		(this,	"table",	table),
	m_alias		(this,	"alias",	table),
	m_primary	(this,	"primary",	table),
	m_ptype		(this,	"ptype",	table),
	m_pexpr		(this,	"pexpr",	table),
	m_parent	(this,	"parent",	table),
	m_field		(this,	"field",	table),
	m_field2	(this,	"field2",	table),
	m_where		(this,	"where",	table),
	m_order		(this,	"order",	table),
	m_jtype		(this,	"jtype",	table),
	m_x		(this,	"x",		table),
	m_y		(this,	"y",		table),
	m_w		(this,	"w",		table),
	m_h		(this,	"h",		table)
{
	m_grouped	= false	;
	m_uflags	= 0	;
}

/*  KBTable								*/
/*  ~KBTable	: Destructor for table object				*/
/*  (returns)	:		:					*/

KBTable::~KBTable ()
{
}

/*  KBTable								*/
/*  getQueryInfo: Get information for query				*/
/*  tabList	: QList<KBTable *> & : Return list of sub-tables	*/
/*  (returns)	: void		     :					*/

void	KBTable::getQueryInfo
	(	QList<KBTable>	&tabList
	)
{
	CITER
	(	Table,
		t,
		tabList.append (t)
	)
}


/*  KBTable								*/
/*  getTableText: Get text for list of tables				*/
/*  pretty	: bool		: Pretty RTF format 			*/
/*  (returns)	: QString	: Join text				*/

QString	KBTable::getTableText
	(	bool	pretty
	)
{
	QString	text	 ;

	text = getTable() ;
	if (getTable() != getName())
		text += QString(pretty ? " <i>%1</i>" : " %1").arg(getAlias()) ;

	CITER
	(	Table,
		t,
		text += ", " + t->getTableText (pretty) ;
	)

	return	text	;
}

/*  KBTable								*/
/*  addToSelect	: Add to select object					*/
/*  select	: KBSelect &	   : Select object in question		*/
/*  descend	: bool		   : True if descending from above	*/
/*  (returns)	: void		   :					*/

void	KBTable::addToSelect
	(	KBSelect	&select,
		bool		descend
	)
{
	QString	jtype	;
	QString	jexpr	;

	fprintf
	(	stderr,
		"KBTable::addToSelect: [%p] [%s]\n",
		(void  *)this,
		(cchar *)getTable()
	)	;

	/* "Descend" is set if we this table is to be linked to a	*/
	/* parent, such as when descending though multiple query levels	*/
	if (descend)
		select.appendTable (getTable(), getAlias(), getJType(),    getJExpr()   ) ;
	else	select.appendTable (getTable(), getAlias(), QString::null, QString::null) ;


	if (!m_where.getValue().isEmpty())
		select.appendWhere (m_where.getValue()) ;

	if (!m_order.getValue().isEmpty())
		select.appendOrder (m_order.getValue()) ;

	CITER
	(	Table,
		t,

		fprintf
		(	stderr,
			" ...... [%s]\n",
			(cchar *)t->getTable()
		)	;
		t->addToSelect (select, true) ;
	)
}

/*  KBTable								*/
/*  getFieldList: Gte list of fields					*/
/*  fldList	: QList<KBFieldList> &	: Result list			*/
/*  dbLink	: KBDBLink &		: Link to server		*/
/*  addName	: bool			: Add table/alias name		*/
/*  (returns)	: bool			: Success			*/

bool	KBTable::getFieldList
	(	QList<KBFieldSpec>	&fldList,
		KBDBLink		&dbLink,
		bool			addName
	)
{
	KBTableSpec tabSpec (getTable()) ;

	if (!dbLink.listFields (tabSpec))
	{	setError (dbLink.lastError()) ;
		return	  false	  ;
	}

//	fprintf
//	(	stderr,
//		"KBTable::getFieldList: [%s] has %d fields\n",
//		(cchar *)getTable(),
//		tabSpec.fldList.count()
//	)	;

	for (uint idx = 0 ; idx < tabSpec.m_fldList.count() ; idx += 1)
	{
		KBFieldSpec *ns = new KBFieldSpec (*tabSpec.m_fldList.at(idx)) ;
		if (addName) ns->m_name = getName() + "." + ns->m_name  ;
		ns->m_table = this  ;
		fldList.append (ns) ;
	}

	CITER
	(	Table,
		table,
		if (!table->getFieldList (fldList, dbLink, addName))
		{	setError (table->lastError ()) ;
			return	 false ;
		}
	)

	return	true ;
}

/*  KBTable								*/
/*  setGeometry	: Set design geometry					*/
/*  g		: QRect		: Geometry				*/
/*  (returns)	: void		:					*/

void	KBTable::setGeometry
	(	QRect	g
	)
{
	m_x.setValue (g.x     ()) ;
	m_y.setValue (g.y     ()) ;
	m_w.setValue (g.width ()) ;
	m_h.setValue (g.height()) ;
}


/*  KBTable								*/
/*  setPrimary	: Set column as primary					*/
/*  colnum	: const QString & : Primary column name			*/
/*  utype	: UniqueType	  : Actual uniqueness type		*/
/*  (returns)	: void		  :					*/

void	KBTable::setPrimary
	(	const QString	&column,
		UniqueType	utype
	)
{
	if (column.isEmpty())
	{
		m_primary.setValue ("") ;
		m_ptype  .setValue (Auto) ;
		return	;
	}

	m_primary.setValue(column) ;
	m_ptype  .setValue(utype ) ;
}

/*  KBTable								*/
/*  getPrimary	: Get primary column					*/
/*  (returns)	: QString	: Column name or null if not set	*/

QString	KBTable::getPrimary ()
{
	if (m_ptype.getIntValue() == PrimaryKey)
		return	m_primary.getValue() ;

	return	QString::null	 ;
}



NEWNODE	(Table, (cchar *)0, KF_QUERY)


/*  STATIC  ----------------------------------------------------------  */

/*  KBTable								*/
/*  findParent	: Find parent of table in list				*/
/*  tabList	: QList<KBTable> & : Table list				*/
/*  child	: KBTable *	   : Putative child			*/
/*  (returns)	: KBTable *	   : Parent or null if none		*/

KBTable	*KBTable::findParent
	(	QList<KBTable>	&tabList,
		KBTable		*child
	)
{
	KBTable	*parent	= 0 ;

	LITER
	(	KBTable,
		tabList,
		table,

		/* We enforce the restriction that a table can have at	*/
		/* most one parent, so report an error if not. This	*/
		/* should never actually happen.			*/
		if (table->getIdent() == child->getParent())
			if (parent != 0)
			{
				KBError::EError
				(	TR("Table in query has multiple parents"),
					QString ("%1: %2 and %3")
						.arg(child ->getTable())
						.arg(table ->getTable())
						.arg(parent->getTable()),
					__ERRLOCN
				)	;
				return	0	;
			}
			else	parent	= table ;
	)

	return	parent	;
}

/*  addChildren	: Add other chilren from list				*/
/*  tabList	: QList<KBTable> & : All tables list			*/
/*  addList	: QList<KBTable> & : Add tables list			*/
/*  table	: KBTable *	   : Parent table			*/
/*  cident	: QString	   : Table to ignore			*/
/*  (returns)	: void		   :					*/
 
static	void	addChildren
	(	QList<KBTable>	&tabList,
		QList<KBTable>	&addList,
		KBTable		*table,
		QString		cident
	)
{
	/* Iterate through the list of tables, skipping any that are	*/
	/* (a) not children of the table and (b) that have the given	*/
	/* identifier							*/
	LITER
	(	KBTable,
		tabList,
		child,

		if (child->getParent() != table->getIdent())
			continue ;
		if (child->getIdent () == cident)
			continue ;

		addList.removeRef (child) ;

		KBTable	*copy = new KBTable (table, child) ;

		/* If the join fields are set the construct the join	*/
		/* expression to the parent.				*/
		if (!copy->getField ().isEmpty() && !copy->getField2().isEmpty())
			copy->setJExpr
			(	QString ("%1.%2 = %3.%4")
					.arg (copy ->getName  ())
					.arg (copy ->getField ())
					.arg (table->getName  ())
					.arg (copy ->getField2())
			)	;

		addChildren (tabList, addList, copy, QString ("") ) ;
	)
	
}

/*  KBTable								*/
/*  blockUp	: Build blocked structure for tables			*/
/*  tabList	: QList<KBTable> & : Table list				*/
/*  table	: KBTable *	   : Top-level table			*/
/*  block	: QList<KBTable> & : Block list				*/
/*  pError	: KBError &	   : Error return			*/
/*  (returns)	: bool		   : Success				*/

bool	KBTable::blockUp
	(	QList<KBTable>	&tabList,
		KBTable		*table,
		QList<KBTable>	&block,
		KBError		&
	)
{
	QList<KBTable> addList (tabList) ;

	/* First work though the nesting levels. We start with the	*/
	/* specified top-level table and work up through successive	*/
	/* parents.							*/
	while (table != 0)
	{
		KBTable *next = new KBTable (0, table) ;

		addList.removeRef (table) ;
		block  .append    (next ) ;
		table	= KBTable::findParent (tabList, table) ;

		/* If we find a parent and the link fields are set then	*/
		/* construct the corresponding join expression ...	*/
		if ( (table != 0) && !next->getField ().isEmpty() &&
				     !next->getField2().isEmpty())
			next->setJExpr
			(	QString ("%1.%2 = %3.%4")
					.arg (next ->getName  ())
					.arg (next ->getField ())
					.arg (table->getName  ())
					.arg (next ->getField2())
			)	;
	}

	/* Then, for each table added to the block level list, add the	*/
	/* children of each (except we skip the prior child that is	*/
	/* already in the list).					*/	
	for (uint idx = 0 ; idx < block.count() ; idx += 1)
		addChildren
		(	tabList,
			addList,
			block.at(idx),
			idx == 0 ? QString("") : block.at(idx-1)->getIdent()
		)	;

	/* The "addList" will now contain all the tables that were not	*/
	/* added to the blocking structure. We add all such tables to	*/
	/* the first one in the block structure. This is arbitrary but	*/
	/* does mean that queries with disconnected tables will	"do the	*/
	/* right thing".						*/
	LITER
	(	KBTable,
		addList,
		table,

//		fprintf	(stderr, "blockUp: not added: %s\n", (cchar *)table->getName()) ;
		new KBTable (block.at(0), table) ;
	)

#if	0
	QString	text	;
	LITER
	(	KBTable,
		block,
		table,
		table->printNode (text, 0, false)
	)
	fprintf	(stderr, "----\n") ;
	fprintf	(stderr, "%s", (cchar *)text) ;
	fprintf	(stderr, "----\n") ;
#endif

	return	true	;
}

/*  KBTable								*/
/*  blockUp	: Build blocked structure for tables			*/
/*  tabList	: QList<KBTable> & : Table list				*/
/*  ident	: QString	   : Top-level table identifier		*/
/*  block	: QList<KBTable> & : Block list				*/
/*  pError	: KBError &	   : Error return			*/
/*  (returns)	: bool		   : Success				*/

bool	KBTable::blockUp
	(	QList<KBTable>	&tabList,
		QString		ident,
		QList<KBTable>	&block,
		KBError		&pError
	)
{
	KBTable	*top	= 0 ;

	if (ident.isEmpty())
		for (uint idx = 0 ; idx < tabList.count() ; idx += 1)
			if (tabList.at(idx)->getParent().isEmpty())
			{
				fprintf	(stderr, "blockUp: flatten to [%s]\n",
						 (cchar *)tabList.at(idx)->getTable()) ;
				ident	= tabList.at(idx)->getIdent() ;
				break	;
			}
	LITER
	(	KBTable,
		tabList,
		table,

		if (table->getIdent() == ident)
		{	top	= table	;
			break	;
		}
	)

	if (top == 0)
	{
		pError	= KBError
			  (	KBError::Error,
				TR("Cannot find top-level table in query"),
				QString(TR("Required ident %1")).arg(ident),
				__ERRLOCN
			  )	;
		return	false	;
	}

	return	blockUp	(tabList, top, block, pError) ;
}

KBNode	*KBTable::replicate
	(	KBNode		*
	)
{
	return	0 ;
}

#if	0
/*  KBTable								*/
/*  decodeUnique: Decode unique columns string				*/
/*  spec	: const QString & : Specification string		*/
/*  unique	: QStringList &	  : Return list of columns		*/
/*  (return)	: UniqueType	  : Uniqueness type			*/

KBTable::UniqueType
	KBTable::decodeUnique
	(	const QString	&spec,
		QStringList	&unique
	)
{
	int	offset	;

	unique.clear ()	;

	if ((offset = spec.find(':')) >= 0)
	{
		QStringList u = QStringList::split (',', spec.mid(offset+1)) ;
		UniqueType  t = (UniqueType)(int)spec[0] ;

		switch (t)
		{
			case Auto	:
				return	Auto	;

			case PrimaryKey :
			case AnyUnique	:
			case AnySingle	:
				unique.append (u[0])	;
				return	t		;

#ifdef	__NOTYET
			case Multiple	:
				unique	= u		;
				return	Multiple	;

			case TableSetup :
				return	TableSetup	;
#endif
			default		:
				break	;

		}

		unique.append ("") ;
		return	AnySingle  ;
		
	}

	/* Backward compatability, if there is no prefix then pass back	*/
	/* the nominal legacy uniqueness type.				*/
	unique.append (spec)	   ;
	return	KBTable::Legacy100 ;
}
#endif
