/***************************************************************************
    file	         : tkc_pyvaluelist.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	"kb_python.h"

#include	<compile.h>
#include	<frameobject.h>

#include	<qapp.h>

#include	"tkc_pydebugbase.h"

#ifndef		_WIN32
#include	"tkc_pyvaluelist.moc"
#else
#include 	"tkc_pyvaluelist.h"
#endif

#if		__KB_EMBEDDED
#include	<qpe/qpeapplication.h>
#endif

#define	ADDSUBVAL(_l,_o) \
	if (showObject((PyObject *)_o)) \
		entries.insert (_l, TKCPyValue::allocValue ((PyObject *)_o)) ;

/*  TKCPyValueKill							*/
/*  TKCPyValueKill: Constructor for the value killer			*/
/*  theItem	  : TKCPyValueItem * : Associated item			*/
/*  (returns)	  : TKCPyValueKill   :					*/

TKCPyValueKill::TKCPyValueKill
	(	TKCPyValueItem	*theItem
	)
	:
	theItem	(theItem)
{
	/* Connect the value-destroyed signal so that we are notified	*/
	/* when the value vanishes.					*/
	connect	(theItem->value(), SIGNAL(destroyed()), this, SLOT(valueGone())) ;
}

/*  TKCPyValueKill							*/
/* ~TKCPyValueKill: Destructor for the value killer			*/
/*  (returns)	  : TKCPyValueKill   :					*/

TKCPyValueKill::~TKCPyValueKill ()
{
}

/*  TKCPyValueKill							*/
/*  valueGone	  : Handle death of the value				*/
/*  (returns)	  : void	:					*/

void	TKCPyValueKill::valueGone ()
{
	/* Clear the value pointer in the item, so that the latter does	*/
	/* not try to delete it, then delete the item.			*/
	if (theItem != 0)
	{
		theItem->m_value = 0	;
		delete	theItem		;
	}
}

void	TKCPyValueKill::detach ()
{
	theItem	= 0 ;
}

/*  ------------------------------------------------------------------  */

/*  TKCPyValueList							*/
/*  TKCPyValueList							*/
/*		: Constructor for debugger value list view		*/
/*  parent	: QWidget *	     : Parent widget			*/
/*  debug	: TKCPyDebugWidget * :					*/
/*  (returns)	: TKCPyValueList     :					*/

TKCPyValueList::TKCPyValueList
	(	QWidget		 *parent,
		TKCPyDebugWidget *debug

	)
	:
	QListView (parent),
	debug	  (debug)
{
#if	__KB_EMBEDDED
	QPEApplication::setStylusOperation
	(	this->viewport(),
		QPEApplication::RightOnHold
	)	;
#endif

	setRootIsDecorated (true) ;
}

/*  TKCPyValueList							*/
/*  showObject	: Check whether to display object			*/
/*  pyObj	: PyObject *	: Object				*/
/*  (returns)	:		:					*/

bool	TKCPyValueList::showObject
	(	PyObject	*pyObj
	)
{
	return	pyObj != 0 ;
}

/*  TKCPyValueList							*/
/*  expandClass	: Expand where value is a class				*/
/*  item	: TKCPyValueItem *	: Item to expand		*/
/*  entries	: QDIct<TKCPyValue> &	: Entries list			*/
/*  (returns)	: void			:				*/

void	TKCPyValueList::expandClass
	(	TKCPyValueItem		*item,
		QDict<TKCPyValue>	&entries
	)
{
	PyClassObject	*clObj	= (PyClassObject *)item->value()->value() ;

	ADDSUBVAL ("Name",  clObj->cl_name ) ;
	ADDSUBVAL ("Bases", clObj->cl_bases) ;

	TKCPyDebugBase::loadDictionary (clObj->cl_dict, entries) ;
}

/*  TKCPyValueList							*/
/*  expandDict	: Expand a dictionary					*/
/*  item	: TKCPyValueItem *	: Item to expan			*/
/*  entries	: QDIct<TKCPyValue> &	: Entries list			*/
/*  (returns)	: void			:				*/

void	TKCPyValueList::expandDict
	(	TKCPyValueItem		*item,
		QDict<TKCPyValue>	&entries
	)
{
	TKCPyDebugBase::loadDictionary (item->value()->value(), entries) ;
}

/*  TKCPyValueList							*/
/*  expandCode	: Expand where value is code 				*/
/*  item	: TKCPyValueItem *	: Item to expan			*/
/*  entries	: QDIct<TKCPyValue> &	: Entries list			*/
/*  (returns)	: void			:				*/

void	TKCPyValueList::expandCode
	(	TKCPyValueItem		*item,
		QDict<TKCPyValue>	&entries
	)
{
	PyCodeObject	*coObj	= (PyCodeObject *)item->value()->value() ;

	ADDSUBVAL ("Name",   coObj->co_name    ) ;
	ADDSUBVAL ("Source", coObj->co_filename) ;
}

/*  TKCPyValueList							*/
/*  expandFrame	: Expand where value is a frame 			*/
/*  item	: TKCPyValueItem *	: Item to expan			*/
/*  entries	: QDIct<TKCPyValue> &	: Entries list			*/
/*  fast	: QDIct<TKCPyValue> &	: Python2.2 fast locals		*/
/*  (returns)	: void			:				*/

void	TKCPyValueList::expandFrame
	(	TKCPyValueItem		*item,
		QDict<TKCPyValue>	&entries,
		QDict<TKCPyValue>	&fast
	)
{
	PyFrameObject	*frame	= (PyFrameObject *)item->value()->value() ;

	ADDSUBVAL ("Builtins", frame->f_builtins) ;
	ADDSUBVAL ("Globals",  frame->f_globals ) ;
	ADDSUBVAL ("Locals",   frame->f_locals  ) ;
	ADDSUBVAL ("Code",     frame->f_code    ) ;

#if	__PY22PLUS
	PyObject	*floc	;
	PyObject	*map	= frame->f_code->co_varnames ;
 
	for (int idx = 0 ; idx < frame->f_nlocals ; idx += 1)
		if ((floc = frame->f_localsplus[idx]) != 0)
			fast.insert
			(	PyString_AsString(PyTuple_GET_ITEM(map, idx)),
				TKCPyValue::allocValue (floc)
			)	;
#endif
}

/*  TKCPyValueList							*/
/*  expandFunction : Expand where value is a function			*/
/*  item	: TKCPyValueItem *	: Item to expan			*/
/*  entries	: QDict<TKCPyValue> &	: Entries list			*/
/*  (returns)	: void			:				*/

void	TKCPyValueList::expandFunction
	(	TKCPyValueItem		*item,
		QDict<TKCPyValue>	&entries
	)
{
	PyFunctionObject *fnObj	= (PyFunctionObject *)item->value()->value() ;

	ADDSUBVAL ("Name",	 fnObj->func_name ) ;
	ADDSUBVAL ("Code",	 fnObj->func_code ) ;
	ADDSUBVAL ("Document",   fnObj->func_doc  ) ;
//	ADDSUBVAL ("Dictionary", fnObj->func_dict ) ;
}

/*  TKCPyValueList							*/
/*  expandInstance : Expand where value is an instance			*/
/*  item	: TKCPyValueItem *	: Item to expan			*/
/*  entries	: QDict<TKCPyValue> &	: Entries list			*/
/*  (returns)	: void			:				*/

void	TKCPyValueList::expandInstance
	(	TKCPyValueItem		*item,
		QDict<TKCPyValue>	&entries
	)
{
	PyInstanceObject *inObj	= (PyInstanceObject *)item->value()->value() ;

	ADDSUBVAL ("Class", inObj->in_class) ;
	TKCPyDebugBase::loadDictionary (inObj->in_dict, entries)   ;
}

/*  TKCPyValueList							*/
/*  expandList	: Expand where value is a list				*/
/*  item	: TKCPyValueItem *	: Item to expan			*/
/*  entries	: QDict<TKCPyValue> &	: Entries list			*/
/*  (returns)	: void			:				*/

void	TKCPyValueList::expandList
	(	TKCPyValueItem		*item,
		QDict<TKCPyValue>	&entries
	)
{
	PyObject *liObj	= item->value()->value() ;

	for (int idx = 0 ; idx < PyList_Size(liObj) ; idx += 1)
		ADDSUBVAL
		(	QString("%1").arg(idx),
			PyList_GetItem (liObj, idx)
		)
}

/*  TKCPyValueList							*/
/*  expandModule: Expand a module					*/
/*  item	: TKCPyValueItem *	: Item to expan			*/
/*  entries	: QDict<TKCPyValue> &	: Entries list			*/
/*  (returns)	: void			:				*/
/*  (returns)	: void		:					*/

void	TKCPyValueList::expandModule
	(	TKCPyValueItem		*item,
		QDict<TKCPyValue>	&entries
	)
{
	TKCPyDebugBase::getModuleDict (item->value()->value(), entries) ;
}

/*  TKCPyValueList							*/
/*  expandTuple	: Expand where value is a tuple				*/
/*  item	: TKCPyValueItem *	: Item to expan			*/
/*  entries	: QDict<TKCPyValue> &	: Entries list			*/
/*  (returns)	: void			:				*/

void	TKCPyValueList::expandTuple
	(	TKCPyValueItem		*item,
		QDict<TKCPyValue>	&entries
	)
{
	PyObject *tpObj	= item->value()->value() ;

	for (int idx = 0 ; idx < PyTuple_Size(tpObj) ; idx += 1)
		ADDSUBVAL
		(	QString("%1").arg(idx),
			PyTuple_GetItem (tpObj, idx)
		)
}

/*  TKCPyValueList							*/
/*  insertEntries: Insert entries into tree				*/
/*  item	 : TKCPyValueItem *	: Item into which to insert	*/
/*  after	 : TKCPyValueItem *	: Item after which to insert	*/
/*  dict	 : QDict<TKCPyValue> &	: Entries list			*/
/*  (returns)	 : TKCPyValueItem *	: Last item inserted		*/

TKCPyValueItem
	*TKCPyValueList::insertEntries
	(	TKCPyValueItem		*item,
		TKCPyValueItem		*after,
		QDict<TKCPyValue>	&dict
	)
{
	QDictIterator<TKCPyValue> iter (dict) ;
	
	while (iter.current())
	{
		TKCPyValue 	*value	= iter.current() ;
		TKCPyValueItem	*entry	= item->scanForObject (value->value(), false) ;

		if 	(entry != 0)
			entry->setValid () ;
		else if (showObject (value->value()))
			after	= new TKCPyValueItem (item, after, iter.currentKey(), value) ;

		if (value->deref()) delete value ;

		iter	+= 1 ;
	}

	return	after	;
}

/*  TKCPyValueList							*/
/*  expand	: Expand list item					*/
/*  item	: TKCPyValueItem * : Item to expand			*/
/*  (returns)	: bool		   : Expanded				*/

bool	TKCPyValueList::expand
	(	TKCPyValueItem	*item
	)
{	
	/* Open depends on the type of python object. In all cases, a	*/
	/* values dictionary (possibly empty) is assembled and		*/
	/* displayed.							*/
	QDict<TKCPyValue> values ;
	QDict<TKCPyValue> fast	 ;

	TKCPyDebugBase::inDebugger (true) ;

	switch (item->value()->type()->typeCode)
	{
		case TKCPyType::Class	:
			expandClass	(item, values) ;
			break	;

		case TKCPyType::Code	:
			expandCode	(item, values) ;
			break	;

		case TKCPyType::Dict	:
			expandDict	(item, values) ;
			break	;

		case TKCPyType::Frame	:
			expandFrame	(item, values, fast) ;
			break	;

		case TKCPyType::Function:
			expandFunction	(item, values) ;
			break	;

		case TKCPyType::Instance:
			expandInstance	(item, values) ;
			break	;

		case TKCPyType::List	:
			expandList	(item, values) ;
			break	;

		case TKCPyType::Module	:
			expandModule	(item, values) ;
			break	;

		case TKCPyType::Tuple	:
			expandTuple	(item, values) ;
			break	;

		default	:
			/* Unknown. Must be marked as expandable but	*/
			/* expansion not implemented.			*/
			TKCPyDebugBase::inDebugger (false) ;
			return	false	;
	}

	item->invalidate (false)  ;

	TKCPyValueItem	*after	= 0 ;
	after	= insertEntries	(item, after, values) ;
	after	= insertEntries	(item, after, fast  ) ;

	TKCPyDebugBase::inDebugger (false) ;
	return	true	;
}





/*  ------------------------------------------------------------------  */

/*  TKCPyValueItem							*/
/*  TKCPyValueItem							*/
/*		: Constructor for python value display item		*/
/*  parent	: QListView *	 : Parent list view			*/
/*  key		: const QSTring &: Key text				*/
/*  value	: TKCPyValue *	 : The value				*/
/*  (returns)	: TKCPyValueItem :					*/

TKCPyValueItem::TKCPyValueItem
	(	QListView	*parent,
		const QString	&key,
		TKCPyValue	*value
	)
	:
	QListViewItem	(parent, key),
	m_value		(value),
	killer		(this)
{
	fillIn	() ;
}

/*  TKCPyValueItem							*/
/*  TKCPyValueItem							*/
/*		: Constructor for python value display item		*/
/*  parent	: QListViewItem *: Parent list view item		*/
/*  key		: const QSTring &: Key text				*/
/*  value	: TKCPyValue *	 : The value				*/
/*  (returns)	: TKCPyValueItem :					*/

TKCPyValueItem::TKCPyValueItem
	(	QListViewItem	*parent,
		const QString	&key,
		TKCPyValue	*value
	)
	:
	QListViewItem	(parent, key),
	m_value		(value),
	killer		(this)
{
	fillIn	() ;
}

/*  TKCPyValueItem							*/
/*  TKCPyValueItem							*/
/*		: Constructor for python value display item		*/
/*  parent	: QListView *	 : Parent list view			*/
/*  after	: QListViewItem *: Item after which to insert		*/
/*  key		: const QSTring &: Key text				*/
/*  value	: TKCPyValue *	 : The value				*/
/*  (returns)	: TKCPyValueItem :					*/

TKCPyValueItem::TKCPyValueItem
	(	QListView	*parent,
		QListViewItem	*after,
		const QString	&key,
		TKCPyValue	*value
	)
	:
	QListViewItem	(parent, after, key),
	m_value		(value),
	killer		(this)
{
	fillIn	() ;
}

/*  TKCPyValueItem							*/
/*  TKCPyValueItem							*/
/*		: Constructor for python value display item		*/
/*  parent	: QListViewItem *: Parent list view item		*/
/*  after	: QListViewItem *: Item after which to insert		*/
/*  key		: const QSTring &: Key text				*/
/*  value	: TKCPyValue *	 : The value				*/
/*  (returns)	: TKCPyValueItem :					*/

TKCPyValueItem::TKCPyValueItem
	(	QListViewItem	*parent,
		QListViewItem	*after,
		const QString	&key,
		TKCPyValue	*value
	)
	:
	QListViewItem	(parent, after, key),
	m_value		(value),
	killer		(this)
{
	fillIn	() ;
}

/*  TKCPyValueItem							*/
/*  ~TKCPyValueItem							*/
/*		: Destructor for python value display item		*/
/*  (returns)	:		:					*/

TKCPyValueItem::~TKCPyValueItem ()
{
	/* If the value is still extant, then dereference it and delete	*/
	/* it if it becomes free. Any other value-items which point to	*/
	/* it will be notified.						*/
	if ((m_value != 0) && m_value->deref())
	{	killer.detach () ;
		delete m_value	 ;
	}
}


/*  TKCPyValueItem							*/
/*  fillIn	: Fill in display values				*/
/*  (returns)	: void		:					*/

void	TKCPyValueItem::fillIn ()
{
	const TKCPyType *type = m_value->type() ;

	setText (1, type->typeName) ;

	/* If the type is marked as simple then show its text string as	*/
	/* its value. Compilcated stuff it not shown this way ....	*/
	if (type->simpleVal)
		setText (2, TKCPyDebugBase::getPythonString (m_value->value())) ;

	/* ... and is probably something that can be expanded.		*/
	setExpandable (type->expandable) ;

	valid	= true	;
	m_value->ref () ;
}

/*  TKCPyValueItem							*/
/*  invalidate	: Mark all items as invalid				*/
/*  self	: bool		: True for self and siblings		*/
/*  (returns)	: void		:					*/

void	TKCPyValueItem::invalidate
	(	bool	self
	)
{
	TKCPyValueItem	*item1 = (TKCPyValueItem *)firstChild  () ;
	TKCPyValueItem	*item2 = (TKCPyValueItem *)nextSibling () ;

	if (item1 != 0) item1->invalidate(true) ;

	if (self)
	{
		if (item2 != 0) item2->invalidate(true) ;
		valid = false ;
	}
}

/*  TKCPyValueItem							*/
/*  clean	: Clean out invalid items				*/
/*  (returns)	: void		   :					*/

void	TKCPyValueItem::clean ()
{
	TKCPyValueItem	*item1 = (TKCPyValueItem *)firstChild  () ;
	TKCPyValueItem	*item2 = (TKCPyValueItem *)nextSibling () ;

	if (item2 != 0) item2->clean () ;

	if (!valid)
	{	delete	this	;
		return	;
	}

	if (item1 != 0) item1->clean () ;
}

/*  TKCPyValueItem							*/
/*  scanForObject: Scan for specified object				*/
/*  object	 : PyObject *	    : Object required			*/
/*  recurse	 : bool		    : Recurse down			*/
/*  (returns)	 : TKCPyValueItem * : Item containing object or null	*/

TKCPyValueItem
	*TKCPyValueItem::scanForObject
	(	PyObject	*object,
		bool		recurse
	)
{
	TKCPyValueItem	*item1 = (TKCPyValueItem *)firstChild  () ;
	TKCPyValueItem	*item2 = (TKCPyValueItem *)nextSibling () ;

	if (m_value->value() == object) return this ;

	if (recurse && (item1 != 0))
	{
		TKCPyValueItem *item = item1->scanForObject (object, true) ;
		if (item != 0) return item ;
	}

	return	item2 == 0 ? 0 : item2->scanForObject (object, recurse) ;
}

void	TKCPyValueItem::setValid ()
{
	valid	= true	;
	if (m_value->type()->simpleVal)
		setText (2, TKCPyDebugBase::getPythonString (m_value->value())) ;

	if (!isOpen()) return ;

	((TKCPyValueList *)listView())->expand (this) ;
}

/*  TKCPyValueItem							*/
/*  setOpen	: Open or close the item				*/
/*  open	: bool		: Open, else close			*/
/*  (returns)	: void		:					*/

void	TKCPyValueItem::setOpen
	(	bool	open
	)
{
	/* If closing then delete all children. We do this so that an	*/
	/* open will get a fresh snapshot.				*/
	if (!open)
	{
		QListViewItem *child ;
		while ((child = firstChild()) != 0) delete child ;
		QListViewItem::setOpen (false) ;
		return	;
	}

	if (!((TKCPyValueList *)listView())->expand (this))
	{
		QListViewItem::setExpandable (false) ;
		QListViewItem::setOpen	     (false) ;
		return	;
	}

	QListViewItem::setOpen (true) ;
}

/*  TKCPyValueList							*/
/*  scanForObject: Scan for specified object				*/
/*  object	 : PyObject *	    : Object required			*/
/*  recurse	 : bool		    : Recurse down			*/
/*  (returns)	 : TKCPyValueItem * : Item containing object or null	*/

TKCPyValueItem
	*TKCPyValueList::scanForObject
	(	PyObject	*object,
		bool		recurse
	)
{
	TKCPyValueItem	*item	= (TKCPyValueItem *)firstChild() ;

	return	item == 0 ? 0 : item->scanForObject(object, recurse) ;
}

/*  TKCPyValueList							*/
/*  invalidate	: Mark all items as invalid				*/
/*  (returns)	: void		:					*/

void	TKCPyValueList::invalidate ()
{
	TKCPyValueItem	*item	= (TKCPyValueItem *)firstChild() ;
	if (item != 0) item->invalidate(true) ;
}

/*  TKCPyValueList							*/
/*  clean	: Remove all invalid entries				*/
/*  (returns)	: void		:					*/

void	TKCPyValueList::clean ()
{
	TKCPyValueItem	*item	= (TKCPyValueItem *)firstChild() ;
	if (item != 0) item->clean () ;		
}

