/*
 * Copyright (c) 1998, 1999, 2000 Phil Thompson (phil@river-bank.demon.co.uk)
 *
 * The SIP parser.
 */

%{
#include <stdlib.h>
#include <string.h>

#include "sip.h"


#define	MAX_NESTED_IF	10

#define	inMainModule()	(currentSpec -> module == currentModule)


/* The different global names types. */

typedef enum {
	func_name,
	var_name
} nameType;


/* Table of Python methods that can go in the PyMethods section. */

static specialPyMethod pyMethods[] = {
	{"__repr__",		SPM_METH_REPR,0,	"PyObject *",	1},
	{"__str__",		SPM_METH_STR,0,		"PyObject *",	1},
	{"__cmp__",		SPM_METH_CMP,0,		"int",		2},
	{"__hash__",		SPM_METH_HASH,0,	"long",		1},
	{"__call__",		SPM_METH_CALL,0,	"PyObject *",	3},
	{"__richcompare__",	SPM_METH_RICH,0,	"PyObject *",	-1, 0x02010000},
	{NULL}
};


/* Table of Python methods that can go in the PyNumberMethods section. */

static specialPyMethod pyNumMethods[] = {
	{"__add__",		SPM_NUM_ADD,0,		"PyObject *",	2},
	{"__sub__",		SPM_NUM_SUB,0,		"PyObject *",	2},
	{"__mul__",		SPM_NUM_MUL,0,		"PyObject *",	2},
	{"__div__",		SPM_NUM_DIV,0,		"PyObject *",	2},
	{"__mod__",		SPM_NUM_MOD,0,		"PyObject *",	2},
	{"__divmod__",		SPM_NUM_DIVMOD,0,	"PyObject *",	2},
	{"__pow__",		SPM_NUM_POW,0,		"PyObject *",	3},
	{"__neg__",		SPM_NUM_NEG,0,		"PyObject *",	1},
	{"__pos__",		SPM_NUM_POS,0,		"PyObject *",	1},
	{"__abs__",		SPM_NUM_ABS,0,		"PyObject *",	1},
	{"__nonzero__",		SPM_NUM_NONZERO,0,	"int",		1},
	{"__invert__",		SPM_NUM_INVERT,0,	"PyObject *",	1},
	{"__lshift__",		SPM_NUM_LSHIFT,0,	"PyObject *",	2},
	{"__rshift__",		SPM_NUM_RSHIFT,0,	"PyObject *",	2},
	{"__and__",		SPM_NUM_AND,0,		"PyObject *",	2},
	{"__xor__",		SPM_NUM_XOR,0,		"PyObject *",	2},
	{"__or__",		SPM_NUM_OR,0,		"PyObject *",	2},
	{"__coerce__",		SPM_NUM_COERCE,0,	"int",		-1},
	{"__int__",		SPM_NUM_INT,0,		"PyObject *",	1},
	{"__long__",		SPM_NUM_LONG,0,		"PyObject *",	1},
	{"__float__",		SPM_NUM_FLOAT,0,	"PyObject *",	1},
	{"__oct__",		SPM_NUM_OCT,0,		"PyObject *",	1},
	{"__hex__",		SPM_NUM_HEX,0,		"PyObject *",	1},
	{"__iadd__",		SPM_NUM_IADD,0,		"PyObject *",	2, 0x02000000},
	{"__isub__",		SPM_NUM_ISUB,0,		"PyObject *",	2, 0x02000000},
	{"__imul__",		SPM_NUM_IMUL,0,		"PyObject *",	2, 0x02000000},
	{"__idiv__",		SPM_NUM_IDIV,0,		"PyObject *",	2, 0x02000000},
	{"__imod__",		SPM_NUM_IMOD,0,		"PyObject *",	2, 0x02000000},
	{"__ipow__",		SPM_NUM_IPOW,0,		"PyObject *",	3, 0x02000000},
	{"__ilshift__",		SPM_NUM_ILSHIFT,0,	"PyObject *",	2, 0x02000000},
	{"__irshift__",		SPM_NUM_IRSHIFT,0,	"PyObject *",	2, 0x02000000},
	{"__iand__",		SPM_NUM_IAND,0,		"PyObject *",	2, 0x02000000},
	{"__ixor__",		0,SPM_NUM_IXOR,		"PyObject *",	2, 0x02000000},
	{"__ior__",		0,SPM_NUM_IOR,		"PyObject *",	2, 0x02000000},
	{NULL}
};


/* Table of Python methods that can go in the PySequenceMethods section. */

static specialPyMethod pySeqMethods[] = {
	{"__len__",		SPM_SEQ_LEN,0,		"int",		1},
	{"__add__",		SPM_SEQ_CONCAT,0,	"PyObject *",	2},
	{"__mul__",		SPM_SEQ_REPEAT,0,	"PyObject *",	-1},
	{"__getitem__",		SPM_SEQ_GETITEM,0,	"PyObject *",	-1},
	{"__setitem__",		SPM_SEQ_SETITEM,0,	"int",		-1},
	{"__getslice__",	SPM_SEQ_GETSLICE,0,	"PyObject *",	-1},
	{"__setslice__",	SPM_SEQ_SETSLICE,0,	"int",		-1},
	{"__contains__",	SPM_SEQ_CONTAINS,0,	"int",		2, 0x01060000},
	{"__iadd__",		SPM_SEQ_ICONCAT,0,	"PyObject *",	2, 0x02000000},
	{"__imul__",		SPM_SEQ_IREPEAT,0,	"PyObject *",	-1, 0x02000000},
	{NULL}
};


/* Table of Python methods that can go in the PyMappingMethods section. */

static specialPyMethod pyMapMethods[] = {
	{"__len__",		SPM_MAP_LEN,0,		"int",		1},
	{"__getitem__",		SPM_MAP_GETITEM,0,	"PyObject *",	2},
	{"__setitem__",		SPM_MAP_SETITEM,0,	"int",		3},
	{NULL}
};


static sipSpec *currentSpec;		/* The current spec being parsed. */
static moduleDef *currentModule;	/* The current module being parsed. */
static classVersDef *currentClassVers;	/* The version of the current class. */
static int sectionFlags;		/* The current section flags. */
static int currentOverIsVirt;		/* Set if the overload is virtual. */
static int currentIsStatic;		/* Set if the current is static. */
static int importDepth;			/* The nr. of nested imports. */
static char *previousFile;		/* The file just parsed. */
static parserContext newContext;	/* The new pending context. */
static int currentVersion;		/* The index of the version. */
static versionAnded versScope[MAX_NESTED_IF];	/* Current version nest. */


static nameDef *cacheName(sipSpec *,moduleDef *,char *);
static void getVersion(versionAnded *);
static classDef *findClass(sipSpec *,char *);
static classVersDef *newClass(sipSpec *,char *);
static optFlag *findOptFlag(optFlags *,char *,flagType);
static memberDef *findFunction(sipSpec *,moduleDef *,classVersDef *,char *);
static void checkGlobalNames(sipSpec *,char *,nameType);
static void checkClassNames(classVersDef *,char *,nameType);
static void newModule(FILE *,char *);
static void appendCodeBlock(codeBlock **,codeBlock *);
static void appendToClassList(classList **,classDef *);
static void parseFile(FILE *,char *,moduleDef *);
static void handleEOF(void);
static void handleEOM(void);
static long evaluateEnumValue(enumDef *,enumValueDef *,valueDef *);
static versionQual *findQualifier(sipSpec *,char *);
%}

%union {
	char		qchar;
	char		*text;
	long		number;
	double		real;
	argDef		memArg;
	funcArgs	arglist;
	codeBlock	*codeb;
	valueDef	value;
	valueDef	*valp;
	optFlags	optflags;
	optFlag		flag;
	overDef		*over;
	scopedNameDef	*scpvalp;
	enumValueDef	*evalp;
	fcallDef	fcall;
	versionAnded	versanded;
}

%token			TK_DOC
%token			TK_EXPORTEDDOC
%token			TK_MAKEFILE
%token			TK_VARCODE
%token			TK_CPPCODE
%token			TK_PREPYCODE
%token			TK_PYCODE
%token 			TK_COPYING
%token <codeb>		TK_CODEBLOCK
%token			TK_IF
%token			TK_END
%token <text>		TK_NAME
%token <text>		TK_PYMETHODNAME
%token <text>		TK_FILENAME
%token <text>		TK_STRING
%token			TK_VIRTUALCODE
%token			TK_MEMBERCODE
%token			TK_FROMCLASS
%token			TK_CANTOCLASS
%token			TK_TOCLASS
%token			TK_TOSUBCLASS
%token			TK_INCLUDE
%token			TK_IMPORT
%token			TK_VERSIONCODE
%token			TK_HEADERCODE
%token			TK_EXPHEADERCODE
%token			TK_MODULE
%token			TK_CLASS
%token			TK_STRUCT
%token			TK_PUBLIC
%token			TK_PROTECTED
%token			TK_PRIVATE
%token			TK_SIGNALS
%token			TK_SLOTS
%token			TK_PYMETHODS
%token			TK_PYNUMMETHODS
%token			TK_PYSEQMETHODS
%token			TK_PYMAPMETHODS
%token			TK_BOOL
%token			TK_SHORT
%token			TK_INT
%token			TK_LONG
%token			TK_FLOAT
%token			TK_DOUBLE
%token			TK_CHAR
%token			TK_VOID
%token			TK_VIRTUAL
%token			TK_ENUM
%token			TK_UNSIGNED
%token			TK_SCOPE
%token			TK_CONST
%token			TK_STATIC
%token			TK_SIPRXCON
%token			TK_SIPRXDIS
%token			TK_SIPSLOTCON
%token			TK_SIPSLOTDIS
%token			TK_SIPARRAY
%token			TK_SIPUARRAY
%token			TK_SIPARRAYSIZE
%token			TK_SIPARRAYUSIZE
%token <number>		TK_NUMBER
%token <real>		TK_REAL
%token			TK_TYPEDEF
%token			TK_DEFVERSION
%token			TK_PRIMARYVERS
%token			TK_VERSION
%token			TK_EXPOSE
%token <qchar>		TK_QCHAR

%type <memArg>		argvalue
%type <memArg>		argtype
%type <memArg>		cpptype
%type <memArg>		basetype
%type <arglist>		arglist
%type <arglist>		rawarglist
%type <number>		optslot
%type <number>		optref
%type <number>		optconst
%type <number>		optabstract
%type <number>		deref
%type <value>		simplevalue
%type <valp>		value
%type <valp>		expr
%type <valp>		optassign
%type <codeb>		optvarcode
%type <codeb>		hdrcode
%type <codeb>		cppcode
%type <codeb>		codeblock
%type <codeb>		virtualcode
%type <codeb>		membercode
%type <text>		optname
%type <optflags>	optflags
%type <optflags>	flaglist
%type <flag>		flag
%type <flag>		flagvalue
%type <over>		function
%type <qchar>		optunop
%type <qchar>		binop
%type <scpvalp>		scopepart
%type <scpvalp>		scopedname
%type <evalp>		enumvalue
%type <evalp>		enumvalues
%type <evalp>		optenumvalues
%type <fcall>		valuelist
%type <versanded>	versionrange

%%

specification: 	statement
	|	specification statement
	;

statement:	{
			/*
			 * We don't do these in parserEOF() because the parser
			 * is reading ahead and that would be too early.
			 */

			if (previousFile != NULL)
			{
				handleEOF();

				if (newContext.prevmod != NULL)
					handleEOM();

				free(previousFile);
				previousFile = NULL;
			}
	} modstatement
	;

modstatement:	module
	|	copying
	|	ifstart
	|	ifend
	|	include
	|	import
	|	version
	|	primaryvers
	|	expose
	|	variable
	|	globalfunc
	|	versioncode
	|	hdrcode {
			if (inMainModule())
				appendCodeBlock(&currentSpec -> hdrcode,$1);
		}
	|	exphdrcode
	|	cppcode {
			if (inMainModule())
				appendCodeBlock(&currentSpec -> cppcode,$1);
		}
	|	prepycode
	|	pycode
	|	doc
	|	exporteddoc
	|	makefile
	|	opaqueclass
	|	class
	|	enum
	|	typedef
	;

version:	TK_DEFVERSION TK_NAME TK_STRING TK_STRING {
			versionQual *vq;

			/* See if the version has already been defined. */

			for (vq = currentSpec -> versions; vq != NULL; vq = vq -> next)
				if (strcmp(vq -> name,$2) == 0)
					yyerror("Version has already been defined");

			vq = sipMalloc(sizeof (versionQual));

			vq -> name = $2;
			vq -> order = -1;
			vq -> vqcpp = $3;
			vq -> vqpy = $4;
			vq -> next = currentSpec -> versions;

			currentSpec -> versions = vq;

			/* Assume it is a secondary. */

			currentSpec -> nrsecs++;
		}
	;

primaryvers:	TK_PRIMARYVERS {
			if (currentSpec -> latestversion > 1)
				yyerror("%%PrimaryVersions has already been defined");
		}
		'{' versionlist '}'
	;

versionlist:	versionname
	|	versionlist versionname
	;

versionname:	TK_NAME {
			versionQual *vq;

			vq = findQualifier(currentSpec,$1);

			if (vq -> order >= 0)
				yyerror("Version appears multiple times in primary list");

			vq -> order = currentSpec -> latestversion++;

			currentSpec -> nrsecs--;
		}
	;

ifstart:	TK_IF TK_VERSION '(' versionrange ')' {
			if (currentVersion >= MAX_NESTED_IF)
				yyerror("Too many nested %If directives");

			versScope[currentVersion++] = $4;
		}
	;

versionrange:	TK_NAME {
			int i;

			$$.secondary = findQualifier(currentSpec,$1);

			/* Primaries can only be used in ranges. */

			if ($$.secondary -> order >= 0)
				yyerror("A primary version must be specified as part of a range");

			/*
			 * Check there isn't already a secondary in scope and
			 * set the primary.
			 */

			$$.lowidx = 0;
			$$.uppidx = currentSpec -> latestversion;

			for (i = 0; i < currentVersion; ++i)
				if (versScope[i].secondary != NULL)
					yyerror("Cannot nest secondary versions");
				else
				{
					/*
					 * Keep overwriting with more and more
					 * specific primaries.
					 */

					$$.lowidx = versScope[i].lowidx;
					$$.uppidx = versScope[i].uppidx;
				}
		}
	|	optname '-' optname {
			if ($1 == NULL)
				$$.lowidx = (currentVersion > 0 ?
					versScope[currentVersion - 1].lowidx :
					0);
			else if (($$.lowidx = findQualifier(currentSpec,$1) -> order) < 0)
				yyerror("Lower bound is a secondary version");

			if ($3 == NULL)
				$$.uppidx = (currentVersion > 0 ?
					versScope[currentVersion - 1].uppidx :
					currentSpec -> latestversion);
			else if (($$.uppidx = findQualifier(currentSpec,$3) -> order) < 0)
				yyerror("Upper bound is a secondary version");

			/* Sanity checks on the bounds. */

			if ($$.lowidx == $$.uppidx)
				yyerror("Lower and upper bounds must be different");

			if ($$.lowidx > $$.uppidx)
				yyerror("Later version specified as lower bound");

			/* Set the secondary. */

			$$.secondary = (currentVersion > 0 ?
				versScope[currentVersion - 1].secondary :
				NULL);
		}
	;

ifend:		TK_END {
			if (currentVersion-- <= 0)
				yyerror("Too many %End directives");
		}
	;

expose:		TK_EXPOSE TK_NAME {
			/* Only bother if this is the main module. */

			if (inMainModule())
			{
				expFuncDef *ef;

				/* Check it isn't already there. */

				for (ef = currentSpec -> exposed; ef != NULL; ef = ef -> next)
					if (strcmp(ef -> name,$2) == 0)
						yyerror("Function has already been exposed in this module");

				ef = sipMalloc(sizeof (expFuncDef));

				ef -> name = $2;
				ef -> next = currentSpec -> exposed;

				currentSpec -> exposed = ef;
			}
		}
	;

module:		TK_MODULE TK_NAME {
			/* Check the module hasn't already been defined. */

			moduleDef *mod;

			for (mod = currentSpec -> moduleList; mod != NULL; mod = mod -> next)
				if (mod -> name != NULL && strcmp(mod -> name -> text,$2) == 0)
					yyerror("Module is already defined");

			currentModule -> name = cacheName(currentSpec,currentModule,$2);
		}
	;

include:	TK_INCLUDE TK_FILENAME {
			parseFile(NULL,$2,NULL);
		}
	;

import:		TK_IMPORT TK_FILENAME {
			/* Ignore if the file has already been imported. */

			moduleDef *mod;

			for (mod = currentSpec -> moduleList; mod != NULL; mod = mod -> next)
				if (strcmp(mod -> file,$2) == 0)
					break;

			if (mod == NULL)
				newModule(NULL,$2);
		}
	;

optvarcode:	{
			$$ = NULL;
		}
	|	TK_VARCODE codeblock {
			$$ = $2;
		}
	;

versioncode:	TK_VERSIONCODE codeblock {
			appendCodeBlock(&currentSpec -> versioncode,$2);
		}
	;

copying:	TK_COPYING codeblock {
			if (inMainModule())
				appendCodeBlock(&currentSpec -> copying,$2);
		}
	;

hdrcode:	TK_HEADERCODE codeblock {
			$$ = $2;
		}
	;

exphdrcode:	TK_EXPHEADERCODE codeblock {
			appendCodeBlock(&currentSpec -> exphdrcode,$2);
		}
	;

cppcode:	TK_CPPCODE codeblock {
			$$ = $2;
		}
	;

prepycode:	TK_PREPYCODE codeblock {
			if (inMainModule())
				appendCodeBlock(&currentSpec -> prepycode,$2);
		}
	;

pycode:		TK_PYCODE codeblock {
			if (inMainModule())
				appendCodeBlock(&currentSpec -> pycode,$2);
		}
	;

doc:		TK_DOC codeblock {
			if (inMainModule())
				appendCodeBlock(&currentSpec -> docs,$2);
		}
	;

exporteddoc:	TK_EXPORTEDDOC codeblock {
			appendCodeBlock(&currentSpec -> docs,$2);
		}
	;

makefile:	TK_MAKEFILE TK_FILENAME TK_FILENAME codeblock {
			if (inMainModule())
			{
				mkTemplateDef *mtd;

				mtd = sipMalloc(sizeof (mkTemplateDef));

				mtd -> name = $2;
				mtd -> objext = $3;
				mtd -> templ = NULL;

				appendCodeBlock(&mtd -> templ,$4);

				mtd -> next = currentSpec -> mktemplates;
				currentSpec -> mktemplates = mtd;
			}
		}
	;

codeblock:	TK_CODEBLOCK TK_END {
			$$ = $1;
			getVersion(&$$ -> version);
		}
	;

enum:		TK_ENUM optname '{' optenumvalues '}' ';' {
			int sflags;
			enumDef *ed, **headp;
			enumValueDef *evd;

			if (currentClassVers != NULL)
			{
				headp = &currentClassVers -> enums;

				if ((sflags = sectionFlags) & ~(SECT_IS_PUBLIC | SECT_IS_PROT))
					yyerror("Class enums must be in the public or protected sections");
			}
			else
			{
				headp = &currentSpec -> enums;
				sflags = 0;
			}

			ed = sipMalloc(sizeof (enumDef));

			ed -> enumflags = sflags;
			ed -> name = $2;

			ed -> cvd = currentClassVers;
			ed -> module = currentModule;
			ed -> values = $4;
			getVersion(&ed -> version);
			ed -> next = *headp;

			*headp = ed;

			/* Check a version of a value doesn't already exist. */

			for (evd = ed -> values; evd != NULL; evd = evd -> next)
			{
				enumDef *ed2;

				for (ed2 = ed -> next; ed2 != NULL; ed2 = ed2 -> next)
				{
					enumValueDef *evd2;

					if (!versionsOverlap(&ed -> version,&ed2 -> version))
						continue;

					for (evd2 = ed2 -> values; evd2 != NULL; evd2 = evd2 -> next)
						if (evd -> name == evd2 -> name)
							yyerror("Different versions of enum value overlap");
				}
			}
		}
	;

optname:	{
			$$ = NULL;
		}
	|	TK_NAME {
			$$ = $1;
		}
	;

optenumvalues:	{
			$$ = NULL;
		}
	|	enumvalues {
			$$ = $1;
		}
	;

enumvalues:	enumvalue
	|	enumvalues ',' enumvalue {
			enumValueDef *evd;

			/* Append the new value to the list. */

			for (evd = $1; evd -> next != NULL; evd = evd -> next)
				;

			evd -> next = $3;
		}
	;

enumvalue:	TK_NAME optassign {
			/*
			 * Note that we don't use the assigned value.  This is
			 * a hangover from when enums where generated in
			 * Python.  We can remove it when we have got around to
			 * updating all the .sip files.
			 */

			$$ = sipMalloc(sizeof (enumValueDef));
			$$ -> name = cacheName(currentSpec,currentModule,$1);
			$$ -> next = NULL;
		}
	;

optassign:	{
			$$ = NULL;
		}
	|	'=' expr {
			$$ = $2;
		}
	;

expr:		value
	|	expr binop value {
			valueDef *vd;
 
			if ($1 -> vtype == string_value || $3 -> vtype == string_value)
				yyerror("Invalid binary operator for string");
 
			/* Find the last value in the existing expression. */
 
			for (vd = $1; vd -> next != NULL; vd = vd -> next)
				;
 
			vd -> vbinop = $2;
			vd -> next = $3;

			$$ = $1;
		}
	;

binop:		'-' {
			$$ = '-';
		}
	|	'+' {
			$$ = '+';
		}
	|	'|' {
			$$ = '|';
		}
	;

optunop:	{
			$$ = '\0';
		}
	|	'!' {
			$$ = '!';
		}
	|	'~' {
			$$ = '~';
		}
	|	'-' {
			$$ = '-';
		}
	|	'+' {
			$$ = '+';
		}
	;

value:		optunop simplevalue {
			if ($1 != '\0' && $2.vtype == string_value)
				yyerror("Invalid unary operator for string");
 
			/*
			 * Convert the value to a simple expression on the
			 * heap.
			 */
 
			$$ = sipMalloc(sizeof (valueDef));
 
			*$$ = $2;
			$$ -> vunop = $1;
			$$ -> vbinop = '\0';
			$$ -> next = NULL;
		}
	;

scopedname:	scopepart
	|	scopedname TK_SCOPE scopepart {
			scopedNameDef *snd;

			/* Append to the list. */

			for (snd = $1; snd -> next != NULL; snd = snd -> next)
				;

			snd -> next = $3;
		}
	;

scopepart: 	TK_NAME {
			$$ = sipMalloc(sizeof (scopedNameDef));
			$$ -> name = $1;
			$$ -> next = NULL;
		}
	;

simplevalue:	scopedname {
			/*
			 * We let the C++ compiler decide if the value is a
			 * valid one - no point in building a full C++ parser
			 * here.
			 */

			$$.vtype = scoped_value;
			$$.u.vscp = $1;
		}
	|	scopedname '(' valuelist ')' {
			fcallDef *fcd;

			fcd = sipMalloc(sizeof (fcallDef));
			*fcd = $3;
			fcd -> name = $1;

			$$.vtype = fcall_value;
			$$.u.fcd = fcd;
		}
	|	TK_REAL {
			$$.vtype = real_value;
			$$.u.vreal = $1;
		}
	|	TK_NUMBER {
			$$.vtype = numeric_value;
			$$.u.vnum = $1;
		}
	|	TK_STRING {
			$$.vtype = string_value;
			$$.u.vstr = cacheName(currentSpec,currentModule,$1);
		}
	|	TK_QCHAR {
			$$.vtype = qchar_value;
			$$.u.vqchar = $1;
		}
	;

valuelist:	{
			/* No values. */

			$$.nrArgs = 0;
		}
	|	value {
			/* The single or first value. */

			$$.args[0] = $1;
			$$.nrArgs = 1;
		}
	|	valuelist ',' value {
			/* Check that it wasn't ...(,value...). */

			if ($$.nrArgs == 0)
				yyerror("First argument to function call is missing");

			/* Check there is room. */

			if ($1.nrArgs == MAX_NR_ARGS)
				yyerror("Too many arguments to function call");

			$$ = $1;

			$$.args[$$.nrArgs] = $3;
			$$.nrArgs++;
		}
	;

typedef:	TK_TYPEDEF cpptype TK_NAME ';' {
			typedefDef *td, **headp;

			if (currentClassVers != NULL)
				headp = &currentClassVers -> typedefs;
			else
				headp = &currentSpec -> typedefs;

			td = sipMalloc(sizeof (typedefDef));

			td -> name = $3;
			td -> type = $2;
			getVersion(&td -> version);
			td -> next = *headp;

			*headp = td;

			/* Check a version doesn't already exist. */

			for (td = (*headp) -> next; td != NULL; td = td -> next)
				if (strcmp(td -> name,$3) == 0 && versionsOverlap(&td -> version,&(*headp) -> version))
					yyerror("Different versions of typedef overlap");
		}
	;

opaqueclass:	TK_CLASS TK_NAME ';' {
			classVersDef *cvd;

			cvd = newClass(currentSpec,$2);
			setCannotCreate(cvd);
			setIsOpaque(cvd);
		}
	;

class:		TK_CLASS TK_NAME {
			if (currentClassVers != NULL)
				yyerror("Cannot nest class definitions");

			currentClassVers = newClass(currentSpec,$2);

			/* Assume it has no non-private ctors. */

			setCannotCreate(currentClassVers);

			sectionFlags = SECT_IS_PRIVATE;
		} superclasses '{' classbody '}' ';' {
			overDef *od;

			if (isMapped(currentClassVers))
			{
				if (currentClassVers -> supers != NULL ||
				    currentClassVers -> ctors != NULL ||
				    currentClassVers -> members != NULL)
					yyerror("Mapped class cannot have super classes, member functions or constructors");

				if (currentClassVers -> convtosubcode != NULL)
					yyerror("Mapped class cannot have a %ConvertToSubClassCode directive");

				if (currentClassVers -> convfromcode == NULL)
					yyerror("Mapped class must have a %ConvertFromClassCode directive");

				if (currentClassVers -> convtocode == NULL)
					yyerror("Mapped class must have a %ConvertToClassCode directive");

				if (currentClassVers -> canconvtocode == NULL)
					yyerror("Mapped class must have a %CanConvertToClassCode directive");
			}
			else if (currentClassVers -> ctors == NULL)
			{
				/* Provide a default ctor. */

				currentClassVers -> ctors = sipMalloc(sizeof (ctorDef));
 
				currentClassVers -> ctors -> args.nrArgs = 0;
				currentClassVers -> ctors -> cppcode = NULL;
				getVersion(&currentClassVers -> ctors -> version);
				currentClassVers -> ctors -> next = NULL;

				resetCannotCreate(currentClassVers);
			}

			/*
			 * We can't create this class if it has private
			 * abstract functions.
			 */

			for (od = currentClassVers -> overs; od != NULL; od = od -> next)
				if (isAbstract(od) && isPrivate(od))
				{
					setCannotCreate(currentClassVers);
					break;
				}

			currentClassVers = NULL;
		}
	;

superclasses:
	|	':' superlist
	;

superlist:	superclass
	|	superlist ',' superclass
	;

superclass:	TK_NAME {
			classDef *cd = findClass(currentSpec,$1);

			appendToClassList(&currentClassVers -> supers,cd);
			addToUsedList(&currentClassVers -> cd -> used,cd);
		}
	;

classbody:	classline
	|	classbody classline
	;

classline:	ifstart
	|	ifend
	|	enum
	|	cppcode {
			appendCodeBlock(&currentClassVers -> cppcode,$1);
		}
	|	hdrcode {
			appendCodeBlock(&currentClassVers -> hdrcode,$1);
		}
	|	typedef
	|	ctor
	|	dtor
	|	varmember
	|	TK_FROMCLASS codeblock {
			if (currentClassVers -> convfromcode != NULL)
				yyerror("Class has more than one %ConvertFromClassCode directive");

			currentClassVers -> convfromcode = $2;
			setIsMapped(currentClassVers);
		}
	|	TK_CANTOCLASS codeblock {
			if (currentClassVers -> canconvtocode != NULL)
				yyerror("Class has more than one %CanConvertToClassCode directive");

			currentClassVers -> canconvtocode = $2;
		}
	|	TK_TOCLASS codeblock {
			if (currentClassVers -> convtocode != NULL)
				yyerror("Class has more than one %ConvertToClassCode directive");

			currentClassVers -> convtocode = $2;
		}
	|	TK_TOSUBCLASS codeblock {
			if (currentClassVers -> convtosubcode != NULL)
				yyerror("Class has more than one %ConvertToSubClassCode directive");

			currentClassVers -> convtosubcode = $2;
		}
	|	TK_PYMETHODNAME membercode {
			int *comb, *combx;
			specialPyMethod *spm;
			spmDef *sd;

			if ($2 == NULL)
				yyerror("Code to implement the special Python method must be given");

			/* Find the method according to the section. */

			if (sectionFlags & SECT_IS_PYMETHOD)
			{
				spm = pyMethods;
				comb = &currentClassVers -> combmeth;
				combx = &currentClassVers -> combmethx;
			}
			else if (sectionFlags & SECT_IS_PYNUMMETHOD)
			{
				spm = pyNumMethods;
				comb = &currentClassVers -> combnummeth;
				combx = &currentClassVers -> combnummethx;
			}
			else if (sectionFlags & SECT_IS_PYSEQMETHOD)
			{
				spm = pySeqMethods;
				comb = &currentClassVers -> combseqmeth;
				combx = &currentClassVers -> combseqmethx;
			}
			else if (sectionFlags & SECT_IS_PYMAPMETHOD)
			{
				spm = pyMapMethods;
				comb = &currentClassVers -> combmapmeth;
				combx = &currentClassVers -> combmapmethx;
			}
			else
				yyerror("Special Python methods must be in the PyMethods, PyNumberMethods, PySequenceMethods or PyMappingMethods sections");

			while (spm -> name != NULL)
			{
				if (strcmp(spm -> name,$1) == 0)
					break;

				++spm;
			}

			if (spm -> name == NULL)
				yyerror("Invalid special Python method for this section");

			/*
			 * Check one of the same name (not ID) hasn't already
			 * been given.
			 */

			for (sd = currentClassVers -> spms; sd != NULL; sd = sd -> next)
				if (strcmp(sd -> spm -> name,$1) == 0)
					yyerror("Different versions of special Python method overlap");

			sd = sipMalloc(sizeof (spmDef));

			sd -> spm = spm;
			sd -> code = $2;
			sd -> next = currentClassVers -> spms;

			currentClassVers -> spms = sd;

			*comb |= spm -> id;
			*combx |= spm -> idxtra;
		}
	|	TK_PUBLIC optslot ':' {
			sectionFlags = SECT_IS_PUBLIC | $2;
		}
	|	TK_PROTECTED optslot ':' {
			sectionFlags = SECT_IS_PROT | $2;
		}
	|	TK_PRIVATE optslot ':' {
			sectionFlags = SECT_IS_PRIVATE | $2;
		}
	|	TK_SIGNALS ':' {
			sectionFlags = SECT_IS_SIGNAL;
		}
	|	TK_PYMETHODS ':' {
			sectionFlags = SECT_IS_PYMETHOD;
		}
	|	TK_PYNUMMETHODS ':' {
			sectionFlags = SECT_IS_PYNUMMETHOD;
		}
	|	TK_PYSEQMETHODS ':' {
			sectionFlags = SECT_IS_PYSEQMETHOD;
		}
	|	TK_PYMAPMETHODS ':' {
			sectionFlags = SECT_IS_PYMAPMETHOD;
		}
	;

optslot:	{
			$$ = 0;
		}
	|	TK_SLOTS {
			$$ = SECT_IS_SLOT;
		}
	;

dtor:		'~' TK_NAME '(' ')' ';' membercode {
			if (strcmp(classVersName(currentClassVers),$2) != 0)
				yyerror("Destructor doesn't have the same name as its class");

			if (isDtor(currentClassVers))
				yyerror("Destructor has already been defined");

			currentClassVers -> dtorcode = $6;
			setIsDtor(currentClassVers);

			if (sectionFlags & ~SECT_IS_PUBLIC)
				setNoPublicDtor(currentClassVers);
		}
	;

ctor:		TK_NAME '(' arglist ')' ';' membercode {
			ctorDef *ct, **ctp;

			/* Check the name of the constructor. */

			if (strcmp(classVersName(currentClassVers),$1) != 0)
				yyerror("Constructor doesn't have the same name as its class");

			free($1);

			if (sectionFlags & ~(SECT_IS_PUBLIC | SECT_IS_PROT | SECT_IS_PRIVATE))
				yyerror("Constructor must be in the public, private or protected sections");

			/* Add to the list of constructors. */

			ct = sipMalloc(sizeof (ctorDef));

			ct -> ctorflags = sectionFlags;
			ct -> args = $3;
			ct -> cppcode = $6;
			getVersion(&ct -> version);
			ct -> next = NULL;

			if (isProtectedCtor(ct))
				setIsComplex(currentClassVers);

			if (!isPrivateCtor(ct))
				resetCannotCreate(currentClassVers);

			/*
			 * Find the end of the list and, at the same time,
			 * check for any existing one with the same signature
			 * and an overlapping version.
			 */

			for (ctp = &currentClassVers -> ctors; *ctp != NULL; ctp = &(*ctp) -> next)
				if (sameFuncArgs(&(*ctp) -> args,&ct -> args) && versionsOverlap(&(*ctp) -> version,&ct -> version))
					yyerror("Different versions of ctor with the same signature overlap");

			*ctp = ct;
		}
	;

function:	cpptype TK_NAME '(' arglist ')' optconst optabstract optflags ';' membercode {
			char *pname;
			overDef *od, **odp, **headp;
			optFlag *of;

			headp = (currentClassVers != NULL ?
					&currentClassVers -> overs :
					&currentSpec -> overs);

			/* Use the C++ name if a Python name wasn't given. */

			if ((of = findOptFlag(&$8,"PyName",name_flag)) != NULL)
				pname = of -> fvalue.sval;
			else
				pname = $2;

			/* Create a new overload definition. */

			od = sipMalloc(sizeof (overDef));

			od -> overflags = 0;

			if ($6)
				setIsConst(od);

			if ($7)
				setIsAbstract(od);

			if (findOptFlag(&$8,"ReleaseLock",bool_flag) != NULL)
				setIsBlocking(od);

			od -> cppname = $2;
			od -> args = $4;
			od -> common = findFunction(currentSpec,currentModule,currentClassVers,pname);
			od -> cppcode = $10;
			od -> virtcode = NULL;
			getVersion(&od -> version);

			if ((of = findOptFlag(&$8,"PreHook",name_flag)) != NULL)
				od -> prehook = of -> fvalue.sval;
			else
				od -> prehook = NULL;

			if ((of = findOptFlag(&$8,"PostHook",name_flag)) != NULL)
				od -> posthook = of -> fvalue.sval;
			else
				od -> posthook = NULL;

			od -> next = NULL;

			if ($1.atype == voidptr_type && $1.nrderefs == 0)
				od -> result = NULL;
			else
			{
				od -> result = sipMalloc(sizeof (argDef));
				*od -> result = $1;
			}

			/*
			 * Find the end of the list and, at the same time,
			 * check for any existing one with the same signature
			 * and an overlapping version.
			 */

			for (odp = headp; *odp != NULL; odp = &(*odp) -> next)
				if ((*odp) -> common == od -> common &&
				    sameOverload(*odp,od) &&
				    versionsOverlap(&(*odp) -> version,&od -> version))
					yyerror("Different versions of function with the same signature overlap");

			*odp = od;

			$$ = od;
		}
	;

globalfunc:	function {
			if (isAbstract($1))
				yyerror("Function specified as abstract");
		}
	;

memberfunction:	function virtualcode {
			overDef *od;

			if (sectionFlags & ~(SECT_IS_PUBLIC | SECT_IS_PROT | SECT_IS_PRIVATE | SECT_IS_SLOT | SECT_IS_SIGNAL))
				yyerror("Function must be in the public, private, protected, slot or signal sections");

			/* Set the overload flags. */

			$1 -> overflags |= sectionFlags;

			if (isProtected($1))
				setIsComplex(currentClassVers);

			if ((isSlot($1) || isSignal($1)) && !isPrivate($1))
			{
				if (isSignal($1))
					setIsComplex(currentClassVers);

				currentSpec -> sigslots = TRUE;
			}

			if (currentIsStatic)
			{
				if (!isPublic($1) || isSlot($1) || isSignal($1))
					yyerror("Static functions must be public");

				if (currentOverIsVirt)
					yyerror("Static functions cannot be virtual");

				setIsStatic($1);
				currentIsStatic = FALSE;
			}

			/* Check that all are static or none of them are. */

			for (od = currentClassVers -> overs; od != NULL; od = od -> next)
			{
				if (od == $1 || od -> common != $1 -> common)
					continue;

				if (versionsOverlap(&od -> version,&$1 -> version) && isStatic(od) != isStatic($1))
					yyerror("Cannot mix static and non-static member functions with the same Python name");
			}

			if (isAbstract($1) && $1 -> cppcode != NULL)
				yyerror("Cannot provide member code for abstract functions");

			if (currentOverIsVirt)
			{
				if (isSignal($1))
					yyerror("Virtual signals aren't supported");

				setIsVirtual($1);
				setIsComplex(currentClassVers);

				$1 -> virtcode = $2;

				currentOverIsVirt = FALSE;
			}
			else if ($2 != NULL)
				yyerror("Virtual code provided for non-virtual function");
		}
	;

optconst:	{
			$$ = FALSE;
		}
	|	TK_CONST {
			$$ = TRUE;
		}
	;

optabstract:	{
			$$ = 0;
		}
	|	'=' TK_NUMBER {
			if ($2 != 0)
				yyerror("Abstract virtual function '= 0' expected");

			$$ = TRUE;
		}
	;

optflags:	{
			$$.nrFlags = 0;
		}
	|	'/' flaglist '/' {
			$$ = $2;
		}
	;


flaglist:	flag {
			$$.flags[0] = $1;
			$$.nrFlags = 1;
		}
	|	flaglist ',' flag {
			/* Check there is room. */

			if ($1.nrFlags == MAX_NR_FLAGS)
				yyerror("Too many optional flags");

			$$ = $1;

			$$.flags[$$.nrFlags++] = $3;
		}
	;

flag:		TK_NAME {
			$$.ftype = bool_flag;
			$$.fname = $1;
		}
	|	TK_NAME '=' flagvalue {
			$$ = $3;
			$$.fname = $1;
		}
	;

flagvalue:	TK_NAME {
			$$.ftype = name_flag;
			$$.fvalue.sval = $1;
		}
	|	TK_STRING {
			$$.ftype = string_flag;
			$$.fvalue.sval = $1;
		}
	;

membercode:	{
			$$ = NULL;
		}
	|	TK_MEMBERCODE codeblock {
			$$ = $2;
		}
	;

virtualcode:	{
			$$ = NULL;
		}
	|	TK_VIRTUALCODE codeblock {
			$$ = $2;
		}
	;

arglist:	rawarglist {
			int a, nrrxcon, nrrxdis, nrslotcon, nrslotdis, nrarray, nrarraysize;

			nrrxcon = nrrxdis = nrslotcon = nrslotdis = nrarray = nrarraysize = 0;

			for (a = 0; a < $1.nrArgs; ++a)
				switch ($1.args[a].atype)
				{
				case rxcon_type:
					++nrrxcon;
					break;

				case rxdis_type:
					++nrrxdis;
					break;

				case slotcon_type:
					++nrslotcon;
					break;

				case slotdis_type:
					++nrslotdis;
					break;

				case array_type:
				case uarray_type:
					++nrarray;
					break;

				case arraysize_type:
				case arrayusize_type:
					++nrarraysize;
					break;
				}

			if (nrrxcon != nrslotcon || nrrxcon > 1)
				yyerror("SIP_RXOBJ_CON and SIP_SLOT_CON must both be given and at most once");

			if (nrrxdis != nrslotdis || nrrxdis > 1)
				yyerror("SIP_RXOBJ_DIS and SIP_SLOT_DIS must both be given and at most once");

			if (nrarray != nrarraysize || nrarray > 1)
				yyerror("SIP_ARRAY/SIP_UARRAY and SIP_ARRAY_SIZE/SIP_ARRAY_USIZE must both be given and at most once");

			$$ = $1;
		}
	;

rawarglist:	{
			/* No arguments. */

			$$.nrArgs = 0;
		}
	|	argvalue {
			/* The single or first argument. */

			$$.args[0] = $1;
			$$.nrArgs = 1;
		}
	|	rawarglist ',' argvalue {
			/* Check that it wasn't ...(,arg...). */

			if ($$.nrArgs == 0)
				yyerror("First argument to member function is missing");

			/*
			 * If this argument has no default value, then all
			 * previous ones mustn't either.
			 */

			if ($3.defval == NULL)
			{
				int a;

				for (a = 0; a < $1.nrArgs; ++a)
					if ($1.args[a].defval != NULL)
						yyerror("Compulsory argument given after optional argument");
			}

			/* Check there is room. */

			if ($1.nrArgs == MAX_NR_ARGS)
				yyerror("Too many arguments to member function");

			$$ = $1;

			$$.args[$$.nrArgs] = $3;
			$$.nrArgs++;
		}
	;

argvalue:	TK_SIPRXCON {
			$$.atype = rxcon_type;
			$$.argflags = 0;
			$$.nrderefs = 0;

			currentSpec -> sigslots = TRUE;
		}
	|	TK_SIPRXDIS {
			$$.atype = rxdis_type;
			$$.argflags = 0;
			$$.nrderefs = 0;

			currentSpec -> sigslots = TRUE;
		}
	|	TK_SIPSLOTCON '(' arglist ')' {
			$$.atype = slotcon_type;
			$$.argflags = 0;
			$$.nrderefs = 0;

			$$.u.sa = sipMalloc(sizeof (funcArgs));
			*$$.u.sa = $3;

			currentSpec -> sigslots = TRUE;
		}
	|	TK_SIPSLOTDIS '(' arglist ')' {
			$$.atype = slotdis_type;
			$$.argflags = 0;
			$$.nrderefs = 0;

			$$.u.sa = sipMalloc(sizeof (funcArgs));
			*$$.u.sa = $3;

			currentSpec -> sigslots = TRUE;
		}
	|	TK_SIPARRAY {
			$$.atype = array_type;
			$$.argflags = 0;
			$$.nrderefs = 0;
		}
	|	TK_SIPUARRAY {
			$$.atype = uarray_type;
			$$.argflags = 0;
			$$.nrderefs = 0;
		}
	|	TK_SIPARRAYSIZE {
			$$.atype = arraysize_type;
			$$.argflags = 0;
			$$.nrderefs = 0;
		}
	|	TK_SIPARRAYUSIZE {
			$$.atype = arrayusize_type;
			$$.argflags = 0;
			$$.nrderefs = 0;
		}
	|	argtype optassign {
			$$ = $1;
			$$.defval = $2;
		}
	;

varmember:	TK_STATIC {currentIsStatic = TRUE;} varmem
	|	varmem
	;

varmem:		member
	|	variable
	;

member:		TK_VIRTUAL {currentOverIsVirt = TRUE;} memberfunction
	|	memberfunction
	;

variable:	cpptype TK_NAME ';' optvarcode {
			varDef *vd, **headp;
			moduleDef *vmod;

			/* Check the name and find the right variable list. */

			if (currentClassVers != NULL)
			{
				checkClassNames(currentClassVers,$2,var_name);

				if (sectionFlags & ~SECT_IS_PUBLIC)
					yyerror("Class variables must be in the public section");

				headp = &currentClassVers -> vars;
			}
			else
			{
				checkGlobalNames(currentSpec,$2,var_name);

				headp = &currentSpec -> globals;
			}

			vd = sipMalloc(sizeof (varDef));

			vd -> name = cacheName(currentSpec,currentModule,$2);
			vd -> module = currentModule;
			vd -> varflags = 0;
			vd -> type = $1;
			getVersion(&vd -> version);
			vd -> accessfunc = $4;
			vd -> next = *headp;

			*headp = vd;

			/* Check the versions don't conflict. */

			for (vd = (*headp) -> next; vd != NULL; vd = vd -> next)
				if (strcmp(vd -> name -> text,$2) == 0 && versionsOverlap(&vd -> version,&(*headp) -> version))
					yyerror("Different versions of variable overlap");

			if (currentIsStatic)
			{
				setIsStaticVar(*headp);
				currentIsStatic = FALSE;
			}

			if (currentClassVers != NULL && !isStaticVar(*headp) && (*headp) -> accessfunc != NULL)
				yyerror("%%VariableCode cannot be specified for non-static class variables");
		}
	;

cpptype:	TK_CONST basetype deref optref {
			$$ = $2;
			$$.nrderefs = $3;
			$$.argflags = ARG_IS_CONST | $4;
		}
	|	basetype deref optref {
			$$ = $1;
			$$.nrderefs = $2;
			$$.argflags = $3;
		}
	;

argtype:	cpptype optflags {
			$$ = $1;

			if (findOptFlag(&$2,"Transfer",bool_flag) != NULL)
				$$.argflags |= ARG_XFERRED;

			if (findOptFlag(&$2,"TransferThis",bool_flag) != NULL)
				$$.argflags |= ARG_THIS_XFERRED;

			if (findOptFlag(&$2,"TransferBack",bool_flag) != NULL)
				$$.argflags |= ARG_XFERRED_BACK;
		}
	;

optref:		{
			$$ = 0;
		}
	|	'&' {
			$$ = ARG_IS_REF;
		}
	;

deref:		{
			$$ = 0;
		}
	|	deref '*' {
			$$ = $1 + 1;
		}
	;

basetype:	scopedname {
			$$.atype = defined_type;
			$$.u.snd = $1;
		}
	|	TK_NAME '<' cpptype '>' {
			templateDef *td;

			td = sipMalloc(sizeof(templateDef));
			td -> name = $1;
			td -> type = $3;

			$$.atype = template_type;
			$$.u.td = td;
		}
	|	TK_STRUCT TK_NAME {
			$$.atype = struct_type;
			$$.u.sname = $2;
		}
	|	TK_UNSIGNED TK_SHORT {
			$$.atype = ushort_type;
		}
	|	TK_SHORT {
			$$.atype = short_type;
		}
	|	TK_UNSIGNED {
			$$.atype = uint_type;
		}
	|	TK_INT {
			$$.atype = int_type;
		}
	|	TK_LONG {
			$$.atype = long_type;
		}
	|	TK_UNSIGNED TK_LONG {
			$$.atype = ulong_type;
		}
	|	TK_FLOAT {
			$$.atype = float_type;
		}
	|	TK_DOUBLE {
			$$.atype = double_type;
		}
	|	TK_BOOL {
			$$.atype = bool_type;
		}
	|	TK_UNSIGNED TK_CHAR {
			$$.atype = ustring_type;
		}
	|	TK_CHAR {
			$$.atype = string_type;
		}
	|	TK_VOID {
			$$.atype = voidptr_type;
		}
	;

%%


/*
 * Parse the specification.
 */

void parse(sipSpec *spec,FILE *fp,char *filename,char *cppMName)
{
        /* Initialise the spec. */
 
	spec -> cppmname = cppMName;
	spec -> namecache = NULL;
	spec -> moduleList = NULL;
	spec -> versions = NULL;
	spec -> latestversion = 1;
	spec -> nrsecs = 0;
	spec -> classes = NULL;
	spec -> used = NULL;
	spec -> qobjclass = -1;
	spec -> exposed = NULL;
	spec -> enums = NULL;
	spec -> globals = NULL;
	spec -> othfuncs = NULL;
	spec -> overs = NULL;
	spec -> typedefs = NULL;
	spec -> versioncode = NULL;
	spec -> copying = NULL;
	spec -> hdrcode = NULL;
	spec -> exphdrcode = NULL;
	spec -> cppcode = NULL;
	spec -> prepycode = NULL;
	spec -> pycode = NULL;
	spec -> docs = NULL;
	spec -> mktemplates = NULL;
	spec -> sigargs = NULL;
	spec -> sigslots = FALSE;

	currentSpec = spec;
	currentModule = NULL;
	currentClassVers = NULL;
	currentOverIsVirt = FALSE;
	currentIsStatic = FALSE;
	importDepth = -1;
	previousFile = NULL;
	currentVersion = 0;

	newModule(fp,filename);
	spec -> module = currentModule;

	yyparse();

	handleEOF();
	handleEOM();
}


/*
 * Tell the parser that a complete file has now been read.
 */

void parserEOF(char *name,parserContext *pc)
{
	previousFile = sipStrdup(name);
	newContext = *pc;
}


/*
 * Append a class definition to a class list.  Append is needed specifically
 * for the list of super-classes because the order is important to Python.
 */

static void appendToClassList(classList **clp,classDef *cd)
{
	classList *new;

	/* Find the end of the list. */

	while (*clp != NULL)
		clp = &(*clp) -> next;

	new = sipMalloc(sizeof (classList));

	new -> c = cd;
	new -> next = NULL;

	*clp = new;
}


/*
 * Return a class list (but not the classes) to the heap.
 */

void freeClassList(classList *cl)
{
	while (cl != NULL)
	{
		classList *this;

		this = cl;
		cl = this -> next;

		free(this);
	}
}


/*
 * Create a new module for the current specification and make it current.
 */

static void newModule(FILE *fp,char *filename)
{
	parseFile(fp,filename,currentModule);

	currentModule = sipMalloc(sizeof (moduleDef));

	currentModule -> name = NULL;
	currentModule -> file = filename;
	currentModule -> depth = ++importDepth;
	currentModule -> next = currentSpec -> moduleList;

	currentSpec -> moduleList = currentModule;
}


/*
 * Switch to parsing a new file.
 */

static void parseFile(FILE *fp,char *name,moduleDef *prevmod)
{
	parserContext pc;

	pc.ifdepth = currentVersion;
	pc.prevmod = prevmod;

	setInputFile(fp,name,&pc);
}


/*
 * Find a class definition in a parse tree.
 */

static classDef *findClass(sipSpec *pt,char *cname)
{
	classDef *cd;

	/* See if the name is already used. */

	for (cd = pt -> classes; cd != NULL; cd = cd -> next)
		if (strcmp(cd -> name -> text,cname) == 0)
			return cd;

	/*
	 * Create a new one.  For the moment we assume it will be defined in
	 * the same module.
	 */

	cd = sipMalloc(sizeof (classDef));

	cd -> name = cacheName(currentSpec,currentModule,cname);
	cd -> module = NULL;
	cd -> classflags = 0;
	cd -> classnr = -1;
	cd -> used = NULL;
	cd -> cvds = NULL;
	cd -> next = pt -> classes;

	pt -> classes = cd;

	return cd;
}


/*
 * Add a class to a class list if it isn't already there.
 */

void addToUsedList(classList **clp,classDef *cd)
{
	classList *cl;

	while ((cl = *clp) != NULL)
	{
		/* Don't bother if it is already there. */

		if (cl -> c == cd)
			return;

		clp = &cl -> next;
	}

	cl = sipMalloc(sizeof (classList));

	cl -> c = cd;
	cl -> next = NULL;

	*clp = cl;
}


/*
 * Find an undefined (or create a new) class definition in a parse tree.
 */

static classVersDef *newClass(sipSpec *pt,char *cname)
{
	classDef *cd;
	classVersDef *cvd;

	cd = findClass(pt,cname);

	if (cd -> module == NULL)
		cd -> module = currentModule;
	else if (cd -> module != currentModule)
		yyerror("Class has already been defined in another module");

	if (inMainModule())
		addToUsedList(&pt -> used,cd);

	/* Create the particular version. */

	cvd = sipMalloc(sizeof (classVersDef));

	cvd -> classversflags = 0;
	cvd -> cd = cd;
	getVersion(&cvd -> version);
	cvd -> supers = NULL;
	cvd -> hierachy = NULL;
	cvd -> enums = NULL;
	cvd -> typedefs = NULL;
	cvd -> ctors = NULL;
	cvd -> dtorcode = NULL;
	cvd -> members = NULL;
	cvd -> overs = NULL;
	cvd -> vmembers = NULL;
	cvd -> visible = NULL;
	cvd -> vars = NULL;
	cvd -> cppcode = NULL;
	cvd -> hdrcode = NULL;
	cvd -> convfromcode = NULL;
	cvd -> convtocode = NULL;
	cvd -> convtosubcode = NULL;
	cvd -> canconvtocode = NULL;
	cvd -> spms = NULL;
	cvd -> combmeth = 0;
	cvd -> combmethx = 0;
	cvd -> combnummeth = 0;
	cvd -> combnummethx = 0;
	cvd -> combseqmeth = 0;
	cvd -> combseqmethx = 0;
	cvd -> combmapmeth = 0;
	cvd -> combmapmethx = 0;
	cvd -> combcmpmeth = 0;
	cvd -> combcmpmethx = 0;
	cvd -> next = cd -> cvds;

	cd -> cvds = cvd;

	/* Check the versions don't overlap. */

	for (cvd = cd -> cvds -> next; cvd != NULL; cvd = cvd -> next)
		if (versionsOverlap(&cvd -> version,&cd -> cvds -> version))
			yyerror("Different versions of class overlap");

	return cd -> cvds;
}


/*
 * Cache a name in a module.
 */

static nameDef *cacheName(sipSpec *pt,moduleDef *mod,char *name)
{
	nameDef *nd;

	/* See if it already exists. */

	for (nd = pt -> namecache; nd != NULL; nd = nd -> next)
		if (strcmp(nd -> text,name) == 0)
		{
			/*
			 * Use the module that is deepest in the import
			 * hierachy.  This will guarantee it really exists.
			 */

			if (nd -> module -> depth < mod -> depth)
				nd -> module = mod;

			return nd;
		}

	/* Create a new one. */

	nd = sipMalloc(sizeof (nameDef));

	nd -> text = name;
	nd -> module = mod;
	nd -> next = pt -> namecache;

	pt -> namecache = nd;

	return nd;
}


/*
 * Find (or create) an overloaded function name.
 */

static memberDef *findFunction(sipSpec *pt,moduleDef *mod,classVersDef *cvd,char *pname)
{
	memberDef *md, **flist;
	nameDef *pnd;

	/* Check there is no name clash. */

	if (cvd != NULL)
	{
		checkClassNames(cvd,pname,func_name);
		flist = &cvd -> members;
	}
	else
	{
		checkGlobalNames(pt,pname,func_name);
		flist = &pt -> othfuncs;
	}

	/* Cache the name. */

	pnd = cacheName(pt,mod,pname);

	/* See if it already exists. */

	for (md = *flist; md != NULL; md = md -> next)
		if (md -> pyname == pnd)
			return md;

	/* Create a new one. */

	md = sipMalloc(sizeof (memberDef));

	md -> pyname = pnd;
	md -> module = mod;
	md -> next = *flist;

	*flist = md;

	return md;
}


/*
 * Get the version from the current scope.
 */

static void getVersion(versionAnded *va)
{
	if (currentVersion > 0)
		*va = versScope[currentVersion - 1];
	else
	{
		va -> lowidx = 0;
		va -> uppidx = currentSpec -> latestversion;
		va -> secondary = NULL;
	}
}


/*
 * Search a set of flags for a particular one and check its type.
 */

static optFlag *findOptFlag(optFlags *flgs,char *name,flagType ft)
{
	int f;

	for (f = 0; f < flgs -> nrFlags; ++f)
		if (strcmp(flgs -> flags[f].fname,name) == 0)
		{
			if (flgs -> flags[f].ftype != ft)
				yyerror("Optional flag has a value of the wrong type");

			return &flgs -> flags[f];
		}

	return NULL;
}


/*
 * Check that there isn't a class variable with a particular Python name.
 */

static void checkClassNames(classVersDef *cvd,char *nm,nameType nt)
{
	varDef *vd;
	memberDef *md;

	/* Check the class variables. */

	for (vd = cvd -> vars; vd != NULL; vd = vd -> next)
		if (strcmp(vd -> name -> text,nm) == 0)
		{
			if (nt != var_name)
				yyerror("There is already a class variable with the same name");

			return;
		}

	/* Check the member functions. */

	for (md = cvd -> members; md != NULL; md = md -> next)
		if (strcmp(md -> pyname -> text,nm) == 0)
		{
			if (nt != func_name)
				yyerror("There is already a member function with the same name");

			return;
		}
}


/*
 * Append a code block to a list of them.  Append is needed to give the
 * specifier easy control over the order of the documentation.
 */

static void appendCodeBlock(codeBlock **headp,codeBlock *new)
{
	while (*headp != NULL)
		headp = &(*headp) -> next;

	*headp = new;
}


/*
 * Check that the name of a global doesn't clash with existing names and that
 * it was first defined in the same module.
 */

static void checkGlobalNames(sipSpec *pt,char *nm,nameType nt)
{
	varDef *vd;
	memberDef *md;
	classDef *cd;
	enumDef *ed;

	/* Check the variables. */

	for (vd = pt -> globals; vd != NULL; vd = vd -> next)
		if (strcmp(vd -> name -> text,nm) == 0)
		{
			if (nt != var_name)
				yyerror("There is already a global variable with the same name");

			if (vd -> module != currentModule)
				yyerror("A global variable with the same name has been defined in another module");

			return;
		}

	/* Check the functions. */

	for (md = pt -> othfuncs; md != NULL; md = md -> next)
		if (strcmp(md -> pyname -> text,nm) == 0)
		{
			if (nt != func_name)
				yyerror("There is already a global function with the same name");

			if (md -> module != currentModule)
				yyerror("A global function with the same name has been defined in another module");

			return;
		}
}


/*
 * Returns TRUE if the first version is a subset of the second.
 */

int versionIsSubset(versionAnded *va1,versionAnded *va2)
{
	/* If the second secondary is conditional the first one must match. */

	if (va2 -> secondary != NULL && va1 -> secondary != va2 -> secondary)
		return FALSE;

	/* Check the primaries. */

	return (va1 -> lowidx >= va2 -> lowidx && va1 -> uppidx <= va2 -> uppidx);
}


/*
 * Check if two version qualifiers overlap, ie both might be true at the same
 * time.
 */

int versionsOverlap(versionAnded *va1,versionAnded *va2)
{
	versionAnded *fst, *sec;

	/* Can't overlap if both have secondaries and they are different. */

	if (va1 -> secondary != NULL && va2 -> secondary != NULL && va1 -> secondary != va2 -> secondary)
		return FALSE;

	/* Order them. */

	if (va1 -> lowidx < va2 -> lowidx)
	{
		fst = va1;
		sec = va2;
	}
	else
	{
		fst = va2;
		sec = va1;
	}

	/*
	 * They overlap if the end of the first is after the start of the
	 * second.
	 */

	return (fst -> uppidx > sec -> lowidx);
}


/*
 * Handle the end of a fully parsed a file.
 */

static void handleEOF()
{
	/*
	 * Check that the number of nested if's is the same as when we started
	 * the file.
	 */

	if (currentVersion > newContext.ifdepth)
		fatal("Too many %%If statements in %s\n",previousFile);

	if (currentVersion < newContext.ifdepth)
		fatal("Too many %%End statements in %s\n",previousFile);
}


/*
 * Handle the end of a fully parsed a module.
 */

static void handleEOM()
{
	/* Check the current module has been named. */

	if (currentModule -> name == NULL)
		fatal("No %%Module has been specified for module defined int %s\n",previousFile);

	/* The previous module is now current. */

	currentModule = newContext.prevmod;

	--importDepth;
}


/*
 * Find an existing version qualifier.
 */

static versionQual *findQualifier(sipSpec *pt,char *name)
{
	versionQual *vq;

	for (vq = pt -> versions; vq != NULL; vq = vq -> next)
		if (strcmp(vq -> name,name) == 0)
			break;

	if (vq == NULL)
		yyerror("Version hasn't been defined");

	return vq;
}
