/***************************************************************************
    file	         : kb_ora10i.cpp
    copyright            : (C) 1999,2000,2001,2002,2003,2004 by Mike Richardson
			   (C) 2001,2002,2003,2004 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/



#include 	"kb_ora10i.h"
#include 	"kb_ora10iadvanced.h"
#include 	"kb_serverinfo.h"
#include	"kb_build.h"

#include 	<qregexp.h>
#include	<qasciidict.h>


static	Ora10iTypeMap typeMap[] =
{
{	"LONG",		"%1\t%2 LONG",			KB::ITFixed,	"Long",		0				},
{	"INTEGER",	"%1\t%2 INTEGER",		KB::ITFixed,	"Integer",	0				},
{	"BINARY_FLOAT",	"%1\t%2 BINARY_FLOAT",		KB::ITFloat,	"Float",	FF_PREC				},
{	"BINARY_DOUBLE","%1\t%2 BINARY_DOUBLE",		KB::ITFloat,	"Double",	FF_PREC				},
{	"NUMBER",	"%1\t%2 NUMBER(%3,%4)",		KB::ITFloat,	"Numeric",	FF_LENGTH|FF_PREC		},
{	"CHAR",		"%1\t%2 CHAR(%3)",		KB::ITString,	"Char",		FF_LENGTH			},
{	"VARCHAR2",	"%1\t%2 VARCHAR2(%3)",		KB::ITString,	"VarChar",	FF_LENGTH			},
{	"RAW",		"%1\t%2 RAW(%3)",		KB::ITBinary,	"Raw",		FF_LENGTH			},
{	"LONG RAW",	"%1\t%2 LONG RAW",		KB::ITBinary,	"LongRaw",	0				},
{	"DATE",		"%1\t%2 DATE",			KB::ITDate,	"Date",	 	0				},
{	"TIMESTAMP",	"%1\t%2 TIMESTAMP",		KB::ITDateTime,	"DateTime",	0				},
{	"BLOB",		"%1\t%2 BLOB",			KB::ITBinary,	"Blob",		0				},
{	"CLOB",		"%1\t%2 CLOB",			KB::ITString,	"Clob",		0				},


{	"VARRAW",	"%1\t%2 VARRAW",		KB::ITBinary,	"VarRaw",	0		 |FF_NOCREATE	},
{	"LONG VARRAW",	"%1\t%2 LONG VARRAW",		KB::ITBinary,	"LongVarRaw",	0		 |FF_NOCREATE	},
{	"LONG VARCHAR",	"%1\t%2 LONG VARCHAR(%3)",	KB::ITString,	"LongVarChar",	FF_LENGTH	 |FF_NOCREATE	},
{ 	"VARCHAR",	"%1\t%2 VARCHAR(%3)",		KB::ITString,	"VarChar",	FF_LENGTH	 |FF_NOCREATE	},
{	"UNSIGNED",	"%1\t%2 UNSIGNED",		KB::ITFixed,	"Unsigned",	0		 |FF_NOCREATE	},
{	"CHARZ",	"%1\t%2 CHARZ(%3)",		KB::ITString,	"Charz",	FF_LENGTH	 |FF_NOCREATE 	},

}	;

static	QAsciiDict<Ora10iTypeMap> dIdentToType	;

struct	KBOCIDefine
{
	int	m_ind			;
	union
	{	int	m_int		;
		char	m_str[80]	;
	}	m_d			;
}	;


static	QString	fetchOCIError
	(	dvoid		*handle,
		ub4		type
	)
{
	char	errb[512]	;
	sb4	errcode		;

	if (OCIErrorGet
		(	handle,
			1,
			(text *)0,
			&errcode,
			(OraText *)errb,
			sizeof(errb),
			type
		) != 0)
	{
		return	"Unable to retrieve OCI error" ;
	}

	return	errb	;
			
}

//void	convertToHex
//	(	const unsigned char	*data,
//		uint			length,
//		uchar			*hex
//	)
//{
//	static	uchar	hexchars[] =
//	{
//		'0',	'1',	'2',	'3',
//		'4',	'5',	'6',	'7',
//		'8',	'9',	'A',	'B',
//		'C',	'D',	'E',	'F',
//	}	;
//
//	while (length > 0)
//	{
//		hex[0] = hexchars[(data[0] & 0xf0)>>4] ;
//		hex[1] = hexchars[(data[0] & 0x0f)   ] ;
//		hex    += 2 ;
//		data   += 1 ;
//		length -= 1 ;
//	}
//
//	hex[0] = 0 ;
//}

/*  ------------------------------------------------------------------	*/

/*  KBOra10iValue							*/
/*  KBODBCValue	: Constructor for KBOra10i binding value object		*/
/*  value	: const KBValue & : Underlying value			*/
/*  (returns)	: KBOra10iValue	  :	  				*/

KBOra10iValue::KBOra10iValue
	(	const KBValue	&value
	)
{
	/* NOTE:							*/
	/* This code assumes that the argument object exists until	*/
	/* after this object is destroyed, since in the case of some	*/
	/* types it will contain a pointer into the value.		*/
	m_alloc		= 0	;
	m_pIndPtr	= value.isNull() ? -1 : 0 ;

	switch (value.getType()->getIType())
	{
		case KB::ITFixed:
			m_d.m_fixed 	= value.getRawText().toInt();
			m_valueSize	= sizeof(int)	;
			m_valuePtr	= &m_d.m_fixed	;
			m_SymbolicType	= SQLT_INT	;			
			break	;

		case KB::ITFloat:
			m_d.m_dbl 	= value.getRawText().toDouble();
			m_valueSize	= sizeof(double);
			m_valuePtr	= &m_d.m_dbl	;
			m_SymbolicType	= SQLT_FLT	;
			break	;

		case KB::ITDate:
		case KB::ITTime:
		case KB::ITDateTime:
			{
			const KBDateTime *dt = value.getDateTime() ;

			if (dt != 0)
			{
				QDate date	 (dt->getDate()) ;
				QTime time	 (dt->getTime()) ;

				m_d.m_stamp[0]	= date.year  () / 100 + 100 ;
				m_d.m_stamp[1]	= date.year  () % 100 + 100 ;
				m_d.m_stamp[2]	= date.month () ;
				m_d.m_stamp[3]	= date.day   () ;
				m_d.m_stamp[4]	= time.hour  () + 1 ;
				m_d.m_stamp[5]	= time.minute() + 1 ;
				m_d.m_stamp[6]	= time.second() + 1 ;
			}
			else	memset (m_d.m_stamp, 0, sizeof(m_d.m_stamp)) ;

			m_valuePtr	= m_d.m_stamp		;
			m_valueSize	= sizeof(m_d.m_stamp)	;
			m_SymbolicType	= SQLT_DAT		;
			}
			break	;

		case KB::ITBinary	:
			m_valuePtr	= value.dataPtr	  ()	;
			m_valueSize	= value.dataLength()	;
			m_SymbolicType	= SQLT_BIN 		;
			break	;

		case KB::ITString	:
			/* Oracle expects the string to be null		*/
			/* terminated, Conveniently KBValue does this	*/
			/* so we can just bump the length up by one.	*/
			m_valuePtr	= value.dataPtr	  ()	;
			m_valueSize	= value.dataLength() + 1;
			m_SymbolicType	= SQLT_STR 		;
			break	;

		case KB::ITUnknown	:
		default			:
			m_valuePtr	= value.dataPtr	  ()	;
			m_valueSize	= value.dataLength() + 1;
			m_SymbolicType	= SQLT_STR 		;
			break	;
	}
}

/*  KBOra10iValue							*/
/*  ~KBOra10iValue: Destructor for OCI binding value object		*/
/*  (returns)	:		:					*/

KBOra10iValue::~KBOra10iValue()
{
	DELOBJ	(m_alloc) ;
}



#if 0
static QString sqlCodeToCode(int code)
{
	switch (code)
	{
		case SQLT_LNG			: return "LONG"		;
		case SQLT_UIN			: return "UNSIGNED INT"	;
		case SQLT_INT			: return "INTEGER"	;
		case SQLT_FLT			: return "FLOAT"	;
		case SQLT_FLT			: return "DOUBLE"	;
		case SQLT_NUM			: return "DECIMAL"	;
		case SQLT_NUM			: return "NUMBER"	;
		case SQLT_AFC			: return "CHAR"		;
		case SQLT_AVC			: return "CHARZ"	;
		case SQLT_VCS			: return "VARCHAR"	;
		case SQLT_CHR			: return "VARCHAR2"	;
		case SQLT_LVC			: return "LONG VARCHAR"	;
		case SQLT_BIN			: return "RAW"		;
		case SQLT_VBI			: return "VARRAW"	;
		case SQLT_LBI			: return "LONG RAW"	;
		case SQLT_LVB			: return "LONG VARRAW"	;
		case SQLT_DATE			: return "DATE"		;
		case SQLT_TIMESTAMP		: return "TIMESTAMP"	;
		default				: break			;
	}

	return	QString("SQL_%1").arg(code);
}
#endif

/*  ------------------------------------------------------------------  */

/*  KBOra10iType							*/
/*  KBOra10iType: Constructor for DB2 type object			*/
/*  oraType	: ub2		: Oracle type				*/
/*  iType	: KB::IType	: KBase internal type			*/
/*  length	: uint		: Database field length			*/
/*  nullOK	: bool		: True if null is OK			*/
/*  (returns)	: KBOra10iType	:					*/

KBOra10iType::KBOra10iType
	(	ub2		oraType,
		KB::IType	iType,
		ub4		length,
		bool		nullOK
	)
	:
	KBType		("ora10i", iType, length, 0, nullOK),
	m_oraType	(oraType)
{
}

/*  KBOra10iType							*/
/*  isValid	: Test if value is valid for type			*/
/*  value	: const QByteArray &: Value to check			*/
/*  null	: bool		    : Value is null			*/
/*  pError	: KBError &	    : Error return			*/
/*  (returns)	: bool		    : Value				*/

bool	KBOra10iType::isValid
	(	const QString 	&value,
		KBError		&pError,
		const QString	&where
	)
{
	return KBType::isValid	(value, pError, where) ;
}

/*  KBOra10iType							*/
/*  getQueryText: Get text for insertion into a query			*/
/*  value	: const QByteArray &: Raw text of value to convert	*/
/*  d		: KBShared *	    : Decoded representation		*/
/*  (returns)	: QString	    : Text				*/

void 	KBOra10iType::getQueryText
	(	KBDataArray	*value,
		KBShared	*d,
		KBDataBuffer	&buffer,
		QTextCodec	*codec
	)
{
	return	KBType::getQueryText	(value, d, buffer, codec) ;
}

/*  ------------------------------------------------------------------  */

/*  ------------------------------------------------------------------  */

/*  KBOra10i								*/
/*  KBOra10i	: Constructor for Oracle10i database connection class	*/
/*  (returns)	: KBServer	:					*/

KBOra10i::KBOra10i()
	:
	KBServer()
{
	m_pEnvHandle 		= (OCIEnv	*)0 ;
	m_pServerContextHandle	= (OCISvcCtx	*)0 ;
	m_pErrorHandle		= (OCIError	*)0 ;
}

/*  KBOra10i								*/
/*  ~KBOra10i	: Destructor for DB2 database connection class		*/
/*  (returns)	:							*/

KBOra10i::~KBOra10i()
{
	if (m__conn)
	{
		OCILogoff (m_pServerContextHandle, m_pErrorHandle);
		cleanUp () ;
	}
}

void	KBOra10i::cleanUp()
{
	if (m_pServerContextHandle)
		OCIHandleFree ((dvoid*)m_pServerContextHandle,	OCI_HTYPE_SVCCTX) ;

	if (m_pErrorHandle)
		OCIHandleFree ((dvoid*)m_pErrorHandle,		OCI_HTYPE_ERROR	) ;

	if (m_pEnvHandle)
		OCIHandleFree ((dvoid*)m_pEnvHandle,		OCI_HTYPE_ENV	) ;

	m__conn = false	;
}

/*  KBOra10i								*/
/*  rekallPrefix: Modify name with rekall table prefix			*/
/*  name	: const QString & : Base name				*/
/*  (returns)	: QSTring	  : Modified name			*/

QString KBOra10i::rekallPrefix
	(	const QString	&name
	)
{
	return "RKL$" + name.upper() ;
}

/*  KBOra10i								*/
/*  checkOra10iOK: Check whether SQL statement executed correctly	*/
/*  pErrorHandle: OCIError	: Statement handle			*/
/*  ora10iRC	: sword		: ODBC return code			*/
/*  where	: cchar *	: First text string for possible error	*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		: True if statement executed correctly	*/

bool	KBOra10i::checkOra10iOK
	(	sword		status,
		const QString	&where,
		KBError		&pError
	)
{
	char 	errorBuffer[512];
	sb4	errorCode	;
	
	OCIErrorGet
	(	(dvoid*)m_pErrorHandle,
		(ub4)1,
		(text*)NULL,
		&errorCode,
		(text*)errorBuffer,
		(ub4)sizeof(errorBuffer),
		OCI_HTYPE_ERROR
	)	;

	QString	errMsg	;
	QString	errDet	;

	switch(status)
	{
		case OCI_SUCCESS	  :
		case OCI_SUCCESS_WITH_INFO:
			return	true	;

		case OCI_NEED_DATA:

			fprintf(stderr, "Warning - OCI_NEED_DATA\n");
			break;

		case OCI_NO_DATA:
			errMsg	= QString(TR("%1: OCI Error: No data")).arg(where) ;
			break;

		case OCI_INVALID_HANDLE	:
			errMsg	= QString(TR("%1: OCI Error: Invalid handle")).arg(where) ;
			break	;

		case OCI_STILL_EXECUTING:
			errMsg	= QString(TR("%1: OCI Error: Still executing")).arg(where) ;
			break	;

		case OCI_CONTINUE:
			errMsg	= QString(TR("%1: OCI Error: Returned continue")).arg(where) ;
			break	;

		case OCI_ERROR:
			errMsg	= where		;
			errDet	= errorBuffer	;
			break	;

		default	:
			errMsg	= QString(TR("%1: Unknown OCI error code")).arg(where) ;
			errDet	= errorBuffer	;
			break	;
	}

	fprintf
	(	stderr,
		"KBOra10i::checkOra10iOK : %s\n"
		"                      : %s\n",
		(cchar *)errMsg,
		(cchar *)errDet
	)	;

	pError	= KBError (KBError::Error, errMsg, errDet, __ERRLOCN) ;
	return	false	;
}

/*  KBOra10i								*/
/*  connect	: Open connection to database				*/
/*  svInfo	: KBServerInfo *: Server information			*/
/*  (returns)	: bool		: Success				*/

bool	KBOra10i::doConnect
	(	KBServerInfo 	*svInfo
	)
{
	sword	ora10iRC	;

	/* Check that we are not already connected. Note that the	*/
	/* notConnected routine sets an error message.			*/
	if (m__conn)
	{	
		m_lError = KBError
			   (	KBError::Error,
				"Already connected to a Oracle database",
				QString::null,
				__ERRLOCN
			   )	;
		return	false	;
	}

	if (svInfo->advanced() != 0)
	{
		if (svInfo->advanced()->isType("ora10i"))
		{
			KBOra10iAdvanced *a = (KBOra10iAdvanced *)svInfo->advanced() ;

			m_showRowID		= a->m_showRowID	;
			m_showOra10iObjects	= a->m_showOra10iObjects;
			m_useTimeouts		= a->m_useTimeouts	;
#ifdef	__NOTYET
			m_stmtTimeout		= a->m_stmtTimeout	;
#endif
			m_lockTimeout		= a->m_lockTimeout	;
			m_connectStr		= a->m_connectStr	;
		}
		else	/* If the type is wrong then show a warning.	*/
			/* This should never happen unless someone	*/
			/* hand edits the XML database file.		*/
			KBError::EError
			(	TR("Driver error"),
				TR("Invalid advanced options, ignoring"),
				__ERRLOCN
			)	;
	}
	else
	{
		m_showRowID		= false	;
		m_showOra10iObjects	= false	;
		m_useTimeouts		= false	;
#ifdef	__NOTYET
		m_stmtTimeout		= -1	;
#endif
		m_lockTimeout		= -1	;
		m_connectStr		= ""	;
	}


	m_activeCookie		= 0	;
	m_inTransaction		= false	;
	m_pEnvHandle		= 0	;
	m_pErrorHandle		= 0	;
	m_pServerContextHandle	= 0	;

	/* Create Environment, Error handle, Server Contex and register	*/
	/* our version with the Ora10iOCI subsystem.			*/
	ora10iRC = OCIEnvCreate
		   (	&m_pEnvHandle,
			OCI_DEFAULT|OCI_OBJECT,
			NULL, NULL,
			NULL, NULL,
			0,
			NULL
		   )	;

	if (ora10iRC != OCI_SUCCESS)
	{
		m_lError = KBError
			   (	KBError::Error,
				"Failed to allocate OCI environment handle",
				QString::null,
				__ERRLOCN
			   )	;
		return	false	;
	}

	ora10iRC = OCIHandleAlloc
		  (	(dvoid  *)m_pEnvHandle,
			(dvoid **)&m_pErrorHandle, 
			OCI_HTYPE_ERROR,
			(size_t  )0,
			(dvoid **)0
	  	)	; 

	if (ora10iRC != OCI_SUCCESS)
	{
		m_lError = KBError
			   (	KBError::Error,
				"Failed to allocate OCI error handle",
				fetchOCIError(m_pEnvHandle, OCI_HTYPE_ENV),
				__ERRLOCN
			   )	;
		return	false	;
	}

	ora10iRC = OCIHandleAlloc
		  (	(dvoid  *)m_pEnvHandle,
			(dvoid **)&m_pServerContextHandle,
			OCI_HTYPE_SVCCTX,
			(size_t  )0,
			(dvoid **)0
		  )	;

	if (ora10iRC != OCI_SUCCESS)
	{
		m_lError = KBError
			   (	KBError::Error,
				"Failed to allocate OCI context handle",
				fetchOCIError(m_pEnvHandle, OCI_HTYPE_ENV),
				__ERRLOCN
			   )	;
		return	false	;
	}


	QCString user	   = m_user    .local8Bit() ;
	QCString password  = m_password.local8Bit() ;
	QCString connect   ;

	if (m_connectStr.isEmpty() || (m_connectStr[0] == '#'))
	{
		QString	 host	   = m_host	  	;
		QString	 port	   = m_port	  	;
		if (host.isEmpty()) host = "localhost"	;
		if (port.isEmpty()) port = "1521" 	;

		cchar	 *cstr	   = "(DESCRIPTION=(ADDRESS=(PROTOCOL=TCP)(HOST=%1)(PORT=%2))(CONNECT_DATA=(SID=%3)))" ;
		connect   = QString(cstr)
				.arg(host)
				.arg(port)
				.arg(m_database).local8Bit() ;
	}
	else	connect	  = m_connectStr.local8Bit()	;

	ora10iRC = OCILogon
		  (	m_pEnvHandle,
			m_pErrorHandle,
			&m_pServerContextHandle, 
			(const OraText *)user    .data  (),
			user    .length(), 
			(const OraText *)password.data  (),
			password.length(),
			(const OraText *)connect .data  (),
			connect .length()
		  )	;


	if (!checkOra10iOK (ora10iRC, "Unable to log in to database", m_lError))
	{
		cleanUp	()	;
		return	false	;
	}

	/* Now that we have made a connection we must check that we	*/
	/* have logged on to the correct version. In our case Oracle	*/
	/* 10i								*/
	int	serverVersion	= 10 ;
	char	versionInfo[512];

	ora10iRC = OCIServerVersion
		  (	m_pServerContextHandle,
			m_pErrorHandle,
			(text *)versionInfo,
			sizeof(versionInfo),
			OCI_HTYPE_SVCCTX
		  )	;
	versionInfo[sizeof(versionInfo)-1] = 0 ;

	QRegExp	vers	("([0-9]+)\\.[0-9\\.]+[0-9]") ;
	if (vers.search(QString::fromUtf8(versionInfo)) >= 0)
		serverVersion = vers.cap(1).toInt();

	if (serverVersion < 10)
	{
		m_lError = KBError
		(	KBError::Error,
			"Not connecting to an Oracle 10i database",
			QString("Oracle version %1").arg(QCString(versionInfo, 512)),
			__ERRLOCN
		)		;
		cleanUp	()	;
		return	false	;
	}

	m__conn = true	;
	return	true	;
}

/*  KBOra10i								*/
/*  qrySelect	: Open a select query					*/
/*  data	: bool		 : Querying for data			*/
/*  select	: const QString& : Select query				*/
/*  update	: bool		 : Select for update query		*/
/*  (returns)	: KBSQLSelect *  : Select query class or NULL on error	*/

KBSQLSelect
	*KBOra10i::qrySelect
	(	bool		data,
		const QString	&selectStr,
		bool		update
	)
{
	return	new KBOra10iQrySelect (this, data, selectStr, update) ;
}


/*  KBOra10i								*/
/*  qryUpdate	: Open an update query					*/
/*  data	: bool		  : Querying for data			*/
/*  update	: const QString & : Update query			*/
/*  tabName	: const QString & : Table name				*/
/*  (returns)	: KNQryUpdate *	  : Update query class or NULL on error	*/

KBSQLUpdate
	*KBOra10i::qryUpdate
	(	bool		data,
		const QString	&updateStr,
		const QString	&tabName
	)
{
	return	new KBOra10iQryUpdate (this, data, updateStr, tabName) ;
}

/*  KBOra10i								*/
/*  qryInsert	: Open an insert query					*/
/*  data	: bool		  : Querying for data			*/
/*  insert	: const QString & : Insert query			*/
/*  tabName	: const QString & : Table name				*/
/*  (returns)	: KNQryInsert *	  : Insert query class or NULL on error	*/

KBSQLInsert
	*KBOra10i::qryInsert
	(	bool		data,
		const QString	&insertStr,
		const QString	&tabName
	)
{
	return	new KBOra10iQryInsert (this, data, insertStr, tabName) ;
}

/*  KBOra10i								*/
/*  qryDelete	: Open a delete query					*/
/*  data	: bool		  : Querying for data			*/
/*  _delete	: const QString & : Delete query			*/
/*  tabName	: const QString & : Table name				*/
/*  (returns)	: KNQryDelete *	  : Delete query class or NULL on error	*/

KBSQLDelete
	*KBOra10i::qryDelete
	(	bool		data,
		const QString	&deleteStr,
		const QString	&tabName
	)
{
	return	new KBOra10iQryDelete (this, data, deleteStr, tabName) ;
}

/*  KBOra10i								*/
/*  transaction	: Server transaction request				*/
/*  op		: Transaction	: Specific operation			*/
/*  activeCookie: void **	: Pass/return active cookie		*/
/*  (returns)	: bool		: Success				*/

bool	KBOra10i::transaction
	(	Transaction	op,
		void		**activeCookie
	)
{
	sword	ora10iRC ;

	switch (op)
	{
		case BeginTransaction	:
			/* If we are being passed an active transaction	*/
			/* cookie *and* there is one in progress, then	*/
			/* pass back the in-progess cookie and return	*/
			/* an error. The caller can use this to figure	*/
			/* out what to abort or to warn about.		*/
			if ((activeCookie != 0) && (m_activeCookie != 0))
			{
				*activeCookie	= m_activeCookie ;
				m_lError	= KBError
						  (	KBError::Warning,
						  	TR("Transaction already in progress"),
						  	QString::null,
						  	__ERRLOCN
						  )	;
				return	false	;
			}
			
			/* If passed an active transaction cookie then	*/
			/* this is noted against future transactions.	*/
			if (activeCookie != 0)
				m_activeCookie	= *activeCookie ;

			m_inTransaction = true	;
			return	true	;


		case CommitTransaction	:
			/* If passed an active transaction cookie then	*/
			/* clear it. Probably not used but behave in a	*/
			/* sensible way in case.			*/
			if (activeCookie != 0) *activeCookie = 0 ;
			m_activeCookie  = 0	;
			m_inTransaction = false	;

			ora10iRC = OCITransCommit
				   (	m_pServerContextHandle,
					m_pErrorHandle,
					OCI_DEFAULT
				   )	;
			return	checkOra10iOK
				   (	ora10iRC,
					"Error committing work",
					m_lError
				   )	;

		case RollbackTransaction:
			/* As above ....				*/
			if (activeCookie != 0) *activeCookie = 0 ;
			m_activeCookie	= 0	;
			m_inTransaction = false	;

			ora10iRC = OCITransRollback
				   (	m_pServerContextHandle,
					m_pErrorHandle,
					OCI_DEFAULT
				   )	;
			return	checkOra10iOK
				   (	ora10iRC,
					"Error rolling back work",
					m_lError
				   )	;

		default	:
			break	;
	}

	m_lError = KBError
		   (	KBError::Fault,
			TR("Unknown driver transaction operation"),
			QString(TR("Code: %1")).arg(op),
			__ERRLOCN
		   )	;
	return	false	;
}

/*  KBOra10i								*/
/*  objectExists: See if named object exists				*/
/*  name	: const QString & : Object name				*/
/*  type	: KB::TableType	  : Object type				*/
/*  exists	: bool &	  : True if object exists		*/
/*  (returns)	: bool		  : Success				*/

bool	KBOra10i::objectExists
	(	const QString	&name,
		KB::TableType	type,
		bool		&exists
	)
{
	cchar	*query	;

	switch (type)
	{
		case KB::IsTable :
			query	= "select 0 from user_tables    where table_name    = '%1'" ;
			break	;

		case KB::IsView :
			query	= "select 0 from user_views     where view_name     = '%1'" ;
			break	;

		case KB::IsSequence :
			query	= "select 0 from user_sequences where sequence_name = '%1'" ;
			break	;

		default	:
			m_lError= KBError
				  (	KBError::Fault,
					TR("Existance test for unrecognised object"),
					QString::null,
					__ERRLOCN
				  )	;
			exists	= false	;
			return	  false ;
	}

	sword	ora10iRC;
	OCIStmt	*stmt	= executeQuery
			 (	QString(query).arg(name.upper()),
				0, 0,
				"Testing for object existance"
			 )	;

	OCIDefine	*pDef	= 0 ;
	sb2		ind	= 0 ;
	ub2		rlen	= 0 ;
	char		b[32]	;
	ora10iRC = OCIDefineByPos
		   (	stmt,
			&pDef,
			m_pErrorHandle,
			1,
			b,
			32,
			SQLT_CHR,
			(dvoid *)&ind,
			&rlen,
			(ub2   *)0,
			OCI_DEFAULT
		   )	;

	if (ora10iRC == OCI_SUCCESS)
		ora10iRC = OCIStmtFetch2
			   (	stmt,
				m_pErrorHandle,
				1, 
				OCI_FETCH_NEXT,
				(sb4)0,
				OCI_DEFAULT
		   	   )	;

	OCIHandleFree ((dvoid *)stmt, OCI_HTYPE_STMT) ;

	fprintf
	(	stderr,
		"KBOra10i::objectExists: %s: %d->%d\n",
		(cchar *)name,
		ora10iRC,
		ora10iRC == OCI_SUCCESS
	)	;

	if (ora10iRC == OCI_NO_DATA)
	{
		exists	= false	;
		return	true	;
	}

	if (!checkOra10iOK (ora10iRC, "Testing for object existance", m_lError))
	{
		exists	= false	;
		return	false	;
	}

	exists	= true	;		
	return	true	;
}

/*  KBOra10i								*/
/*  tableExists	: See if named table exists				*/
/*  table	: const QString & : Table name				*/
/*  exists	: bool &	  : True if table exists		*/
/*  (returns)	: bool		  : Success				*/

bool	KBOra10i::tableExists
	(	const QString	&table,
		bool		&exists
	)
{
//	KBTableDetailsList tabList ;
//
//	if (!doListTables (tabList, KB::IsTable))
//	{
//		exists	= false	;
//		return	false	;
//	}
//
//	for (uint idx = 0 ; idx < tabList.count() ; idx += 1)
//		if (tabList[idx].m_name.lower() == table.lower())
//		{	exists	= true	;
//			return	true	;
//		}
//
//	exists	= false	;
//	return	true	;
//
	return	objectExists (table, KB::IsTable, exists) ;
}

/*  KBOra10i								*/
/*  doListObjects							*/
/*		: List objects in database				*/
/*  objList	: KBTableDetailsList &					*/
/*				: Result list				*/
/*  query	: cchar *	: Query used to retrieve objects	*/
/*  what	: cchar *	: Object type text			*/
/*  type	: KB::TableType	: Type flag				*/
/*  permissions	: uint		: Associated permissions		*/
/*  (returns)	: bool		: Success				*/

bool	KBOra10i::doListObjects
	(	KBTableDetailsList	&objList,
		cchar			*query,
		cchar			*what,
		KB::TableType		type,
		uint			perm
	)
{
	/* Prepare the SQL query for execution and then execute the	*/
	/* OCIStmtExecute routine whicn retrieves the list of tables.	*/
	OCIStmt	*stmt	= executeQuery
			  (	query,
				0,
				0,
				QString("Failed to retrieve OCI %1 list").arg(what)
			  )	;
	if (stmt == 0)
	{
		fprintf
		(	stderr,
			"KBOra10i::doListTables: Failed to retrieve OCI %s list\n",
			what
		)	;
		return	false	;
	}
	
	char		objName[40]	;
	OCIDefine	*pDef		;
	sword		ora10iRC	;
	int		ind		= 0 ;

	objName[0] = 0 ;

	ora10iRC = OCIDefineByPos
		  (	stmt,
			&pDef,
			m_pErrorHandle,
			1,
			(ub1  *)objName,
			sizeof(objName),
			SQLT_STR,
			(void *)&ind,
			(ub2  *)0,
			(ub2  *)0,
			OCI_DEFAULT
		  )	;

	ora10iRC = OCIStmtFetch2
		  (	stmt,
			m_pErrorHandle,
			1,
			OCI_FETCH_NEXT,
			(sb4)0,
			OCI_DEFAULT
		  )	;

	while (ora10iRC != OCI_NO_DATA)
	{
		QString aObjName;

		if (ind == -1)
			aObjName = "UnknownName"	;
		else	aObjName = objName		;

		if (!m_showAllTables)
			if (aObjName.left(8).lower() == "__rekall")
				aObjName = QString::null;

		if (!aObjName.isNull())
			objList.append
			(	KBTableDetails
				(	aObjName,
					type,
					perm
			)	)	;

		objName[0] = 0;
		ora10iRC = OCIStmtFetch2
			  (	stmt,
				m_pErrorHandle,
				1,
				OCI_FETCH_NEXT,
				(sb4)0,
				OCI_DEFAULT
			  )	;
	}

	OCIHandleFree ((dvoid *)stmt, OCI_HTYPE_STMT) ;
	return	true	;
}

/*  KBOra10i								*/
/*  doListTables: List tables in database				*/
/*  tabList	: QList<QString & : Result list				*/
/*  types	: uint		  : Required types			*/
/*  (returns)	: bool		  : Success				*/

bool	KBOra10i::doListTables
	(	KBTableDetailsList	&tabList,
		uint			type
	)
{
	if ((type & KB::IsTable) != 0)
		if (!doListObjects
			(	tabList, 
				"SELECT table_name FROM user_tables",
				"table",
				KB::IsTable,
				QP_SELECT|QP_INSERT|QP_UPDATE|QP_DELETE
			))
			return	false	;

	if ((type & KB::IsView) != 0)
		if (!doListObjects
			(	tabList, 
				"SELECT view_name FROM user_views",
				"view",
				KB::IsView,
				QP_SELECT
			))
			return	false	;

	if ((type & KB::IsSequence) != 0)
		if (!doListObjects
			(	tabList, 
				"SELECT sequence_name FROM user_sequences",
				"sequence",
				KB::IsSequence,
				QP_SELECT
			))
			return	false	;

	return	true	;
}


/*  KBOra10i								*/
/*  listFields	: List fields in table					*/
/*  tabSpec	: KBTableSpec &	: Table specification			*/
/*  (returns)	: bool		: Success				*/

bool	KBOra10i::doListFields
	(	KBTableSpec	&tabSpec
	)
{
	KBOCIDefine	colName		;
	KBOCIDefine	colType		;
	KBOCIDefine	colLength	;
	KBOCIDefine	colNullable	;
	KBOCIDefine	colStatus	;
	KBOCIDefine	colConstraint	;
	KBOCIDefine	indKeyType	;
	OCIStmt		*stmt		;
//	KBOCIDefine	seqName		;
//	KBOCIDefine	seqOwner	;

	tabSpec.m_fldList.clear() ;
	tabSpec.m_prefKey = -1	  ;

	/* The query below is used to gather as much information in one	*/
	/* go with a single query. Note that because there may be	*/
	/* multiple constraints per row, multiple rows my be returned	*/
	/* for each column. See the main loop for the associated checks	*/
	static	char	fieldsQuery[]	=
		"SELECT	all_tab_columns.column_name,	"
		"	all_tab_columns.data_type,	"
		"	all_tab_columns.data_length,	"
		"	all_tab_columns.nullable,	"
		"	all_constraints.status,		"
		"	all_constraints.constraint_type	"
		"FROM	all_tab_columns			"
		"LEFT	OUTER JOIN			"
		"	all_cons_columns		"
		"	ON	all_cons_columns.table_name     = all_tab_columns .table_name		"
		"	AND	all_cons_columns.column_name    = all_tab_columns .column_name		"
		"LEFT	OUTER JOIN			"
		"	all_constraints			"
		"	ON	all_constraints.table_name	= all_tab_columns .table_name		"
		"	AND 	all_constraints.constraint_name = all_cons_columns.constraint_name	"
		"	AND	all_constraints.owner		= all_cons_columns.owner		"
		"WHERE	all_tab_columns .table_name 	= '%1'						"
		"ORDER	BY all_tab_columns.column_id	"
		;

	OCIDefine	*pDef		;
	sword 		ora10iRC	;
	QString 	fieldListSQL	;

	fieldListSQL	= QString(fieldsQuery).arg(tabSpec.m_name.upper()) ;

	if ((stmt = executeQuery (fieldListSQL, 0, 0, "Failed to retrieve OCI column list")) == 0)
	{
		fprintf
		(	stderr,
			"KBOra10i::doListFields: Failed to retrieve OCI column list\n"
		)	;
		return	false ;
	}

	ora10iRC = OCIDefineByPos
		   (	stmt,
			&pDef,
			m_pErrorHandle,
			1,
			(ub1  *)colName.m_d.m_str,
			sizeof (colName.m_d.m_str),
			SQLT_STR,
			(void *)&colName.m_ind,
			(ub2  *)0,
			(ub2  *)0,
			OCI_DEFAULT
		   )	;

	ora10iRC = OCIDefineByPos
		   (	stmt,
			&pDef,
			m_pErrorHandle,
			2,
			(ub1  *)colType.m_d.m_str,
			sizeof (colType.m_d.m_str),
			SQLT_STR,
			(void *)&colType.m_ind,
			(ub2  *)0,
			(ub2  *)0,
			OCI_DEFAULT
		   )	;

	ora10iRC = OCIDefineByPos
		   (	stmt,
			&pDef,
			m_pErrorHandle,
			3,
			(ub2  *)&colLength.m_d.m_int,
			sizeof  (colLength.m_d.m_int),
			SQLT_INT,
			(void *)&colLength.m_ind,
			(ub2  *)0,
			(ub2  *)0,
			OCI_DEFAULT
		   )	;

	ora10iRC = OCIDefineByPos
		   (	stmt,
			&pDef,
			m_pErrorHandle,
			4,
			(ub1  *)colNullable.m_d.m_str,
			sizeof (colNullable.m_d.m_str),
			SQLT_STR,
			(void *)&colNullable.m_ind,
			(ub2  *)0,
			(ub2  *)0,
			OCI_DEFAULT
		   )	;

	ora10iRC = OCIDefineByPos
		   (	stmt,
			&pDef,
			m_pErrorHandle,
			5,
			(ub1  *)colStatus.m_d.m_str,
			sizeof (colStatus.m_d.m_str),
			SQLT_STR,
			(void *)&colStatus.m_ind,
			(ub2  *)0,
			(ub2  *)0,
			OCI_DEFAULT
		   )	;

	ora10iRC = OCIDefineByPos
		   (	stmt,
			&pDef,
			m_pErrorHandle,
			6,
			(ub1  *)colConstraint.m_d.m_str,
			sizeof (colConstraint.m_d.m_str),
			SQLT_STR,
			(void *)&colConstraint.m_ind,
			(ub2  *)0,
			(ub2  *)0,
			OCI_DEFAULT
		   )	;

	ora10iRC = OCIStmtFetch2
		   (	stmt,
			m_pErrorHandle,
			1,
			OCI_FETCH_NEXT,
			(sb4   )0,
			OCI_DEFAULT
		   )	;

	/* Last is set to the last column details set up, and is used	*/
	/* to check for repeated rows returned for the same column.	*/
	KBFieldSpec	*last	= 0 	;
	bool		gotPKey	= false	;

	while (ora10iRC != OCI_NO_DATA)
	{
		fprintf
		(	stderr,
			"KBOra10i::doListFields: %12s: %12s\n",
			colName.m_d.m_str,
			colType.m_d.m_str
		)	;

		if ((last == 0) || (last->m_name != colName.m_d.m_str))
		{
			/* If the column contains a timestamp then the	*/
			/* type seems to come back like TIMESTAMP(7) so	*/
			/* check for this here.				*/
			QString		type	= colType.m_d.m_str ;
			if (type.left(10).upper() == "TIMESTAMP(") type = "TIMESTAMP" ;

			Ora10iTypeMap	*ptr	= dIdentToType.find(type) ;
			QString		ftype	= ptr == 0 ?
							QString("<%1>").arg(colType.m_d.m_str) :
							QString(ptr->m_kbName) ;

			KB::IType	itype	= ptr == 0 ?
							KB::ITUnknown	:
							ptr->m_kbType	;
			uint		flags	= 0 ;

			if (colNullable.m_d.m_str[0] == 'N') 
				flags	|= KBFieldSpec::NotNull ;

			last	= new KBFieldSpec
					 (	tabSpec.m_fldList.count(),
						colName.m_d.m_str,
						ftype,
						itype,
						flags,
						colLength.m_d.m_int,
						0
					 )	;

			tabSpec.m_fldList.append (last) ;
		}

		/* Check for constraints. This is where we may have	*/
		/* multiple rows for the same column.			*/
		switch (colConstraint.m_d.m_str[0])
		{
			case 'P' :
				/* Primary key column. This is always	*/
				/* the preferred key.			*/
				last->m_flags |= KBFieldSpec::Primary|KBFieldSpec::Unique ;
				tabSpec.m_prefKey = last->m_colno	;
				gotPKey		  = true		;
				break	;

			case 'U' :
				/* Unique column. Use this as the	*/
				/* preferred key if we do not already	*/
				/* have one.				*/
				last->m_flags |= KBFieldSpec::Unique    ;
				if (tabSpec.m_prefKey < 0)
					tabSpec.m_prefKey = last->m_colno ;
				break	;

			case 'O' :
				/* Apparently read-only for views...	*/
				last->m_flags |= KBFieldSpec::ReadOnly  ;
				break	;

			default	:
				break	;
		}

		ora10iRC = OCIStmtFetch2
			   (	stmt,
				m_pErrorHandle,
				1,
				OCI_FETCH_NEXT,
				(sb4)0,
				OCI_DEFAULT
			   )	;
	}

	OCIHandleFree ((dvoid *)stmt, OCI_HTYPE_STMT) ;

	/* Optionally, ROWID's may be show as (the last) table column,	*/
	/* in which case add it in. These are marked as read-only and	*/
	/* unique (clearly:) and insert-avail since they are always	*/
	/* retrieved as part of an insert.				*/
	if (m_showRowID)
	{
		last	= new KBFieldSpec
				 (	tabSpec.m_fldList.count(),
					"ROWID",
					"Char",
					KB::ITString,
					KBFieldSpec::ReadOnly|
						KBFieldSpec::Unique  |
						KBFieldSpec::InsAvail|
						KBFieldSpec::Serial,
					18,
					0
				 )	;
		tabSpec.m_fldList.append (last) ;
	}

	bool	exists	;
	if (tabSpec.m_prefKey >= 0)
		if (objectExists (tabSpec.m_name + "_seq", KB::IsSequence, exists))
			if (exists)
			{
				KBFieldSpec *pSpec = tabSpec.m_fldList.at(tabSpec.m_prefKey) ;
				pSpec->m_typeName  = "Primary Key"	;
				pSpec->m_flags    |= KBFieldSpec::Serial;
			}


	static	char	indexQuery[]	=
		"SELECT	ind_cols.column_name,		"
		"	cons.constraint_type		"
		"FROM	all_constraints cons,		"
		"	all_ind_columns ind_cols	"
		"WHERE	cons.table_name = '%1'		"
		"AND	ind_cols.index_name = cons.constraint_name"
		;

	QString 	indexListSQL	;

	indexListSQL	= QString(indexQuery).arg(tabSpec.m_name.upper()) ;

	if ((stmt = executeQuery (indexListSQL, 0, 0, "Failed to retrieve OCI index list")) == 0)
	{
		fprintf
		(	stderr,
			"KBOra10i::doListFields: Failed to retrieve OCI index list\n"
		)	;
		return	false ;
	}

	ora10iRC = OCIDefineByPos
		   (	stmt,
			&pDef,
			m_pErrorHandle,
			1,
			(ub1  *)colName.m_d.m_str,
			sizeof (colName.m_d.m_str),
			SQLT_STR,
			(void *)&colName.m_ind,
			(ub2  *)0,
			(ub2  *)0,
			OCI_DEFAULT
		   )	;

	ora10iRC = OCIDefineByPos
		   (	stmt,
			&pDef,
			m_pErrorHandle,
			2,
			(ub1   *)indKeyType.m_d.m_str,
			sizeof  (indKeyType.m_d.m_str),
			SQLT_STR,
			(void  *)&indKeyType.m_ind,
			(ub2   *)0,
			(ub2   *)0,
			OCI_DEFAULT
		   )	;


	ora10iRC = OCIStmtFetch2
		   (	stmt,
			m_pErrorHandle,
			1,
			OCI_FETCH_NEXT,
			(sb4   )0,
			OCI_DEFAULT
		   )	;

	while (ora10iRC != OCI_NO_DATA)
	{
		fprintf
		(	stderr,
			"KBOra10i::doListFields:  index col [%s]\n",
			colName.m_d.m_str
		)	;

		KBFieldSpec *fspec = tabSpec.findField(QString(colName.m_d.m_str).upper()) ;
		if (fspec != 0)
		{
			if (indKeyType.m_d.m_str[0] == 'P')
			{
				fspec->m_flags |= KBFieldSpec::Primary|KBFieldSpec::NotNull ;
				tabSpec.m_prefKey = fspec->m_colno ;
			}
			
			if (indKeyType.m_d.m_str[0] == 'U')
			{
				fspec->m_flags |= KBFieldSpec::Unique ;
				if (tabSpec.m_prefKey < 0)
					tabSpec.m_prefKey = fspec->m_colno ;
			}
		}

		ora10iRC = OCIStmtFetch2
			   (	stmt,
				m_pErrorHandle,
				1,
				OCI_FETCH_NEXT,
				(sb4   )0,
				OCI_DEFAULT
			   )	;
	}

	return	true	;
}

/*  KBODBC								*/
/*  prepare	: Prepare statement for execution			*/
/*  queryStr	: const QString & : SQL statement			*/
/*  (returns)	: OCIStmt *	  : Handle or null on error		*/

OCIStmt *KBOra10i::prepare
	(	const QString	&queryStr
	)
{
	sword 	ora10iRC	;
	OCIStmt	*stmt	;

	fprintf
	(	stderr,
		"KBOra10i::prepare: %s\n",
		queryStr.latin1()
	)	;

	if (queryStr.isEmpty())
	{
		m_lError = KBError
			   (	KBError::Error,
				TR("Invalid empty query"),
				QString::null,
				__ERRLOCN
			   )	;
		return	0  ;
	}

	ora10iRC = OCIHandleAlloc
		  (	(dvoid  *)m_pEnvHandle,
			(dvoid **)&stmt,
			OCI_HTYPE_STMT,
			0, 0
		  )	;
	if (!checkOra10iOK (ora10iRC, "Preparing query", m_lError))
		return	0 ;


	cchar	*ascii	= queryStr ;

	ora10iRC = OCIStmtPrepare
		  (	stmt,
			m_pErrorHandle,
			(const OraText *)ascii,
			qstrlen(ascii),
			OCI_NTV_SYNTAX,
			OCI_DEFAULT
		  )	;
	if (!checkOra10iOK (ora10iRC, "Preparing query", m_lError))
	{
		OCIHandleFree ((dvoid *)stmt, OCI_HTYPE_STMT) ;
		return	0 ;
	}

	return	stmt ;
}

/*  KBODBC								*/
/*  executeQuery: Simple SQL execution 					*/
/*  sql		: const QString & : SQL statement			*/
/*  nvals	: uint		  : Number of paramaters to bind	*/
/*  values	: const KBValue * : Paramaters to bind			*/
/*  nRows	: int &		  : Return rows if not a SELECT		*/
/*  rawQuery	: const QString & : Raw query text			*/
/*  what	: const QString &   : Caller for error messages		*/
/*  (returns)	: bool		  : Success				*/

bool	KBOra10i::executeQuery
	(	OCIStmt		*stmt,
		uint		nvals,
		const KBValue	*values,
		int		&nRows,
		const QString	&rawQuery,
		const QString	&what
	)
{
	QList<KBOra10iValue> vList ;
	vList.setAutoDelete (true) ;

	nRows	= -1 ;

	/* Bind the paramaters first, if there are any, and return on	*/
	/* failure. Note that the caller may have have already done the	*/
	/* bindings, in which case "nvals" should be zero.		*/
	if (nvals > 0)
		if (!bindParameters (stmt, nvals, values, vList))
			return	false	;


	/* Find out if this is a select or not, and execute. If it is	*/
	/* a select then do not process any rows at this stage; for	*/
	/* others do thr row processing, and find out how may rows	*/
	/* where processed.						*/
	sword	ora10iRC 	;
	ub2	stmtType = 0	;

	ora10iRC = OCIAttrGet
		  (	stmt,
			OCI_HTYPE_STMT,
			(dvoid*)&stmtType,
			NULL,
			OCI_ATTR_STMT_TYPE,
			m_pErrorHandle
		  )	;

	if (ora10iRC == OCI_SUCCESS)
	{
		ub4 iters = stmtType == OCI_STMT_SELECT ? 0 : 1 ;
		ub4 mode  = stmtType == OCI_STMT_SELECT ? OCI_DEFAULT : ocimode() ;

		ora10iRC = OCIStmtExecute
			  (	m_pServerContextHandle,
				stmt,
				m_pErrorHandle,
				iters,
				0,
				(CONST OCISnapshot*)NULL,
				(OCISnapshot*)NULL,
				mode
			  )	;
	}

	if ((ora10iRC == OCI_SUCCESS) && (stmtType != OCI_STMT_SELECT))
	{
		ub4	rows	= 0 ;
		ora10iRC = OCIAttrGet
			   (	(dvoid *)stmt,
				(ub4	)OCI_HTYPE_STMT,
				(dvoid *)&rows,
				(ub4   *)0,
				(ub4	)OCI_ATTR_ROW_COUNT,
				m_pErrorHandle
			   )	;
		nRows	= rows	;
	}

	if (!checkOra10iOK (ora10iRC, what, m_lError))
	{
		fprintf
		(	stderr,
			"KBOra10i::executeQuery: Failed: [%s]\n",
			(cchar *)rawQuery
		)	;
		printQuery (rawQuery, nvals, values, false) ;
		return	false	;
	}

	printQuery (rawQuery, nvals, values, true) ;

	return	true	;
}


/*  KBODBC								*/
/*  executeQuery: Simple SQL execution 					*/
/*  sql		: const QString & : SQL statement			*/
/*  nvals	: uint		  : Number of paramaters to bind	*/
/*  values	: const KBValue * : Paramaters to bind			*/
/*  where	: const QString & : First text string for error		*/
/*  (returns)	: OCIStmt *	  : Handle or null on error		*/

OCIStmt	*KBOra10i::executeQuery
	(	const QString	&sql,
		uint 		nvals,
		const KBValue	*values,
		const QString	&where
	)
{
	int	rows	  ;

	OCIStmt	*stmt	= prepare (sql) ;
	if (stmt == 0)
		return 0  ;

	if (!executeQuery (stmt, nvals, values, rows, sql, where))
	{
		OCIHandleFree ((dvoid *)stmt, OCI_HTYPE_STMT) ;
		return	0 ;
	}

	return	stmt	;
}

/*  KBOra10i								*/
/*  createTable	: Create a new table					*/
/*  tabSpec	: KBTableSpec &	: Table specification			*/
/*  assoc	: bool		: Create associated stuff		*/
/*  best	: bool		: Create best match			*/
/*  (returns)	: bool		: Success				*/

bool	KBOra10i::doCreateTable
	(	KBTableSpec	&tabSpec,
		bool 		assoc,
		bool		best
	)
{
	cchar 	*sep = "" ;

	QString tabName	  = QString("RKL$_%1_%2").arg(tabSpec.m_name.left(4)).arg(QString("%1").arg(time(0)).right(5));
	QString	createSQL = QString("CREATE TABLE %1\n(").arg(tabName) ;

	/* Main loop generates SQL create for the columns and column	*/
	/* types, plus options like auto-increment and primary key.	*/
	KBFieldSpec		   *pkSpec  = 0 ;

	for
	(	QListIterator<KBFieldSpec> listIter1(tabSpec.m_fldList) ;
		listIter1.current() != 0 ;
		listIter1 += 1
	)
	{ 
		KBFieldSpec *pSpec1 = listIter1.current() ;

		/* Special cases. If the field is "Primary Key" then we	*/
		/* create an "Int4" column marked as primary key, not	*/
		/* null and auto increment ...				*/
		QString	ftype = pSpec1->m_typeName ;
		
		if (ftype == "Primary Key")
		{
			createSQL += QString("%1\t%2 INTEGER NOT NULL PRIMARY KEY")
					.arg(sep)
					.arg(pSpec1->m_name) ;
			sep	   = ",\n" ;
			pkSpec	   = pSpec1;
			continue   ;
		}

		/* ... while a foreign key is also "Int4" not null.	*/
		if (ftype == "Foreign Key")
		{
			createSQL += QString("%1\t%2 INTEGER NOT NULL")
					.arg(sep)
					.arg(pSpec1->m_name) ;
			sep	   = ",\n" ;
			continue   ;
		}

		/* Map the types used when creating the object and	*/
		/* design dictionary tables.				*/
		if	(ftype == "_Text"   ) ftype = "VarChar"	;
		else if	(ftype == "_Integer") ftype = "Integer"	;
		else if	(ftype == "_Binary" ) ftype = "Blob"	;

		/* Scan though the RDBMS specific mapping table	looking	*/
		/* for the type,					*/
		Ora10iTypeMap *mapp = 0 ;
		for (uint typ = 0 ; typ < sizeof(typeMap)/sizeof(Ora10iTypeMap) ; typ += 1)
			if (typeMap[typ].m_kbName == ftype)
				if ((typeMap[typ].m_flags & FF_NOCREATE) == 0)
				{
					mapp	= &typeMap[typ] ;
					break	;
				}

		/* If there is no mapping but the use-best-match flag	*/
		/* is set, then look for a mapping based on the		*/
		/* internal types.					*/
		if ((mapp == 0) && best)
		{
			KB::IType itype	= pSpec1->m_typeIntl ;

			if (itype == KB::ITBool) itype	= KB::ITFixed	 ;
			if (itype == KB::ITDate) itype	= KB::ITDateTime ;
			if (itype == KB::ITTime) itype	= KB::ITDateTime ;

			for (uint typ = 0 ; typ < sizeof(typeMap)/sizeof(Ora10iTypeMap) ; typ += 1)
				if (typeMap[typ].m_kbType == itype)
					if ((typeMap[typ].m_flags & FF_NOCREATE) == 0)
					{
						mapp	= &typeMap[typ] ;
						break	;
					}
		}


		if (mapp == 0)
		{
			m_lError = KBError
				   (	KBError::Fault,
					TR("Error mapping column type"),
				  	QString(TR("Type %1 for column %2 not known"))
						.arg(ftype)
						.arg(pSpec1->m_name),
					__ERRLOCN
				   )	;
			return	false	;
		}

		createSQL += QString(mapp->m_ora9iCreate)
				.arg(sep)
				.arg(pSpec1->m_name)
				.arg(pSpec1->m_length)
				.arg(pSpec1->m_prec)
				;


		if ((pSpec1->m_flags & KBFieldSpec::NotNull) != 0) 
			createSQL += " NOT NULL"   ;
		
		if ((pSpec1->m_flags & KBFieldSpec::Primary) != 0) 
			createSQL += " PRIMARY KEY";

		if ((pSpec1->m_flags & KBFieldSpec::Unique) != 0) 
			createSQL += " UNIQUE"	   ;

		sep	= ",\n"	;
	}

	createSQL += "\n)" ;

	OCIStmt	*stmt	;

	stmt	= executeQuery
		  (	createSQL,
		  	0, 0,
		  	QString("Error creating table (as %1)").arg(tabName)
		  )	;
	if (stmt == 0) return false ;
	OCIHandleFree ((dvoid *)stmt, OCI_HTYPE_STMT) ;

	stmt	= executeQuery
		  (	QString("ALTER TABLE %1 RENAME TO %2").arg(tabName).arg(tabSpec.m_name),
		  	0, 0,
			QString("Error renaming table (from %1)").arg(tabName)
		  )	;
	if (stmt == 0) return false ;
	OCIHandleFree ((dvoid *)stmt, OCI_HTYPE_STMT) ;

	if (!assoc)
		return	true	;


	if (pkSpec != 0)
	{
		createSQL = QString ("CREATE SEQUENCE %1_seq ORDER").arg(tabSpec.m_name) ;

		if (!executeQuery 
			(	createSQL,
				0, 0,
				"Unable to create associated sequence"
			))
				return false;
	}
	

	for
	(	QListIterator<KBFieldSpec> listIter2(tabSpec.m_fldList) ;
		listIter2.current() != 0 ;
		listIter2 += 1
	)
	{ 
		KBFieldSpec *pSpec2 = listIter2.current() ;

		if (pSpec2->m_typeName == "Primary Key")
			continue    ;
		if ((pSpec2->m_flags & KBFieldSpec::Primary) != 0)
			continue ;
		if ((pSpec2->m_flags & KBFieldSpec::Indexed) == 0)
			continue ;

		createSQL = QString ("CREATE INDEX %1_idx_%2 ON %3 (%4)")
				.arg(tabName)
				.arg(pSpec2->m_name)
				.arg(tabSpec.m_name)
				.arg(pSpec2->m_name) ;

		if (!executeQuery
			(	createSQL,
				0, 0,
				QString("Unable to create index on column %1").arg(pSpec2->m_name)
			))
				return false;
	}


	return true;
}

/*  KBOra10i								*/
/*  renameTable	: Rename a table					*/
/*  oldName	: cchar *	: Current table name			*/
/*  newName	: cchar *	: New table name			*/
/*  assoc	: bool		: Rename associated stuff		*/
/*  (returns)	: bool		: Success				*/

bool	KBOra10i::doRenameTable
	(	cchar		*oldName,
		cchar		*newName,
		bool 		assoc
	)
{
	QString renameSQL = QString("ALTER TABLE %1 RENAME TO %2"	 ).arg(oldName).arg(newName) ;
	QString renameSeq = QString("ALTER TABLE %1_seq RENAME TO %2_seq").arg(oldName).arg(newName) ;

	OCIStmt	*stmt	;
	stmt	= executeQuery
		  (	renameSQL,
			0, 0,
			QString("Failed to rename table \"%1\" as \"%2\"").arg(oldName).arg(newName)
		  )	;
	if (stmt == 0) return false ;
	OCIHandleFree ((dvoid *)stmt, OCI_HTYPE_STMT) ;

	if (!assoc)
		return true ;

	stmt	= executeQuery
		  (	renameSeq,
			0, 0,
			"Failed to rename associated sequence"
		  )	;
	if (stmt == 0) return false ;
	OCIHandleFree ((dvoid *)stmt, OCI_HTYPE_STMT) ;

	return	true	;
}

/*  KBOra10i								*/
/*  dropTable	: Drop a table						*/
/*  table	: cchar *	: Table name				*/
/*  assoc	: bool		: Drop associated stuff			*/
/*  (returns)	: bool		: Success				*/

bool	KBOra10i::doDropTable
	(	cchar		*table,
		bool		assoc
	)
{
	QString dropSQL = QString("DROP TABLE %1"	).arg(table) ;
	QString dropSeq = QString("DROP SEQUENCE %1_seq").arg(table) ;

	OCIStmt	*stmt	;
	stmt	= executeQuery
		  (	dropSQL,
			0, 0,
			QString("Failed to drop table \"%1\"").arg(table)
		  )	;
	if (stmt == 0) return false ;
	OCIHandleFree ((dvoid *)stmt, OCI_HTYPE_STMT) ;

	if (!assoc)
		return true ;

	stmt	= executeQuery
		  (	dropSeq,
		  	0, 0,
		  	"Failed to drop associated sequence"
		  )	;
	if (stmt == 0) return false ;
	OCIHandleFree ((dvoid *)stmt, OCI_HTYPE_STMT) ;

	return	true ;
}

/*  KBServer								*/
/*  listTypes	: Return a list of all column types			*/
/*  (returns)	: QString	: Types					*/

QString	KBOra10i::listTypes()
{
	static QString typeList ;

	if (typeList.isNull())
	{
		typeList = "Primary Key,0|Foreign Key,0";

		for (uint idx = 0 ; idx < sizeof(typeMap)/sizeof(Ora10iTypeMap) ; idx += 1)
		{
			Ora10iTypeMap* pMap = &typeMap[idx];

			if ((pMap->m_flags & FF_NOCREATE) != 0)
				continue;

			typeList += QString("|%1,%2")
					.arg((char*)pMap->m_kbName)
					.arg(pMap->m_flags)
					;
		}
	}

	return typeList ;
}

/*  KBODBC								*/
/*  bindParamaters							*/
/*		: Bind placeholder parameters to statement		*/
/*  stmt	: OCIStmt *	: Statement handle			*/
/*  nvals	: uint		: Number of substitution values		*/
/*  value	: KBValue *	: Substitution values			*/
/*  vList	: QList<KBOrai9iValue> &				*/
/*				: Substituted values list		*/
/*  (returns)	: bool		: Success				*/

bool	KBOra10i::bindParameters
	(	OCIStmt			*stmt,
		uint 			nvals,
		const KBValue		*values,
		QList<KBOra10iValue>	&vList
	)
{
	sword	ora10iRC	;

	for (uint idx = 0 ; idx < nvals ; idx += 1)
	{
		OCIBind		*pBindHandle	= 0 ;
		KBOra10iValue	*ora9Val	= new KBOra10iValue (values[idx]) ;

		vList.append	(ora9Val) ;

//		fprintf
//		(	stderr,
//			"KBOra10i::bindParameters: %2d: %s\n",
//			idx,
//			(cchar *)values[idx].getRawText().left(32)
//		)	;

		ora10iRC	= OCIBindByPos
				  (	stmt,
					&pBindHandle,
					m_pErrorHandle,
					idx + 1,
					ora9Val->valuePtr	(),
					ora9Val->valueSize	(),
					ora9Val->symbolicType	(),
					ora9Val->indPtr		(),
					0,
					0,
					0,
					0,
					OCI_DEFAULT
				  )	;


		if ((ora10iRC == 0) && (ora9Val->symbolicType() == SQLT_STR))
			OCIAttrSet
			(	(void*)pBindHandle,
				OCI_HTYPE_BIND,
				(void*)&CSID_NCHAR,
				0,
				OCI_ATTR_CHARSET_FORM,
				m_pErrorHandle
			)	;


		if (!checkOra10iOK (ora10iRC, "Error binding OCI parameter\n", m_lError))
		{
			fprintf
			(	stderr,
				"KBOra10i::bindParameters: bind error\n"
			)	;
			return	false	;
		}
	}

	return	true	;
}

int KBOra10i::errorNumber()
{
	sb4 errorCode;

	OCIErrorGet((dvoid*)m_pErrorHandle, (ub4)1, (text*)NULL, &errorCode, NULL, 0, OCI_HTYPE_ERROR);

	return errorCode;
}

QString KBOra10i::warning()
{
	unsigned char errorBuffer[512];
	sb4 errorCode;
	
	OCIErrorGet((dvoid*)m_pErrorHandle, (ub4)1, (text*)NULL, &errorCode, (text*)errorBuffer,
		(ub4)sizeof(errorBuffer), OCI_HTYPE_ERROR);

	return QString((char*)errorBuffer);
}

QString	KBOra10i::placeHolder
	(	uint		num
	)
{
	return QString(":%1").arg(num + 1) ;
}

/*  ------------------------------------------------------------------- */


/*  ColumnInfo								*/
/*  ColumnInfo	: Constructor for column information object		*/
/*  server	: KBOra10i *	: Server object				*/
/*  param	: OCIParam *	: Oracle paramater object		*/
/*  ok		: bool &	: Success return flag			*/
/*  pError	: KBError &	: Return error messgae			*/
/*  (returns)	: ColumnInfo	:					*/

ColumnInfo::ColumnInfo
	(	KBOra10i	*server,
		OCIParam	*param,
		bool		&ok,
		KBError		&pError
	)
	:
	m_pServer	(server)
{
	/* Instance of these objects are created when a select query is	*/
	/* executed in order to set up defines to recieve data from	*/
	/* the database and to convert it to its KBValue representation	*/
	int 		ora10iRC	= 0 ;

	text* 		colName 	= 0 ;
	ub4 		colNameLen 	= 0 ;

	/* Clear the object. Individual members are set up depending on	*/
	/* the data type. In all cases the "m_valuep" member will be	*/
	/* set to the value to pass to the OCIDefineByXXX call.		*/
	m_buffer	= 0		;
	m_dataSize	= 0		;
	m_ind		= 0		;
	m_define	= 0		;
	m_lobLocator	= 0		;
	m_rowid		= 0		;
	m_valuep	= 0		;
	m_datetime	= 0		;

	ora10iRC = OCIAttrGet
		   (	(dvoid*)param,
			(ub4   )OCI_DTYPE_PARAM,
			&m_colOraType,
			0, 
			(ub4)OCI_ATTR_DATA_TYPE,
			m_pServer->m_pErrorHandle
		   )	;

	if (ora10iRC == OCI_SUCCESS)
		ora10iRC = OCIAttrGet
			   (	(dvoid *)param,
			   	(ub4	)OCI_DTYPE_PARAM,
		   		(dvoid**)&colName, 
				(ub4   *)&colNameLen,
				(ub4	)OCI_ATTR_NAME,
				m_pServer->m_pErrorHandle
			   )	;
		
	if (ora10iRC == OCI_SUCCESS)
		ora10iRC = OCIAttrGet
			   (	(dvoid *)param,
				(ub4	)OCI_DTYPE_PARAM,
				&m_colDataSize,
				0, 
				(ub4    )OCI_ATTR_DATA_SIZE,
				m_pServer->m_pErrorHandle
			   )	;

	if (ora10iRC == OCI_SUCCESS)
		ora10iRC = OCIAttrGet
			   (	(dvoid *)param,
				(ub4    )OCI_DTYPE_PARAM,
				&m_colCharSize,
				0, 
				(ub4    )OCI_ATTR_CHAR_SIZE,
				m_pServer->m_pErrorHandle
			   )	;

	if (ora10iRC == OCI_SUCCESS)
		ora10iRC = OCIAttrGet
			   (	(dvoid *)param,
				(ub4	)OCI_DTYPE_PARAM,
				&m_colPrecision,
				0, 
				(ub4	)OCI_ATTR_PRECISION,
				m_pServer->m_pErrorHandle
			   )	;

	if (ora10iRC == OCI_SUCCESS)
		ora10iRC = OCIAttrGet
			   (	(dvoid *)param,
				(ub4	)OCI_DTYPE_PARAM,
				&m_colPrecision,
				0, 
				(ub4	)OCI_ATTR_SCALE,
				m_pServer->m_pErrorHandle
			   )	;

	if (ora10iRC == OCI_SUCCESS)
		ora10iRC = OCIAttrGet
			   (	(dvoid *)param,
				(ub4	)OCI_DTYPE_PARAM,
				(dvoid *)&m_colNullable,
				0, 
				(ub4	)OCI_ATTR_IS_NULL,
				m_pServer->m_pErrorHandle
			   )	;
	
	if (!m_pServer->checkOra10iOK (ora10iRC, "Failed to get column attributes", pError))
	{
		ok	= false ;
		return	;
	}

	if (m_colOraType == SQLT_BLOB)
		m_colDataSize = 0;

	m_colName	= (const char *)colName	;
	m_colName.truncate(colNameLen)	;

	/* We retrieve most values as character strings since this is	*/
	/* how Rekall will store them in the KBValue structure, however	*/
	/* some cases will change this below. Similarly, most are	*/
	/* fetch on OCI_DEFAULT mode.					*/
	m_colGetType	= SQLT_CHR	;
	m_mode		= OCI_DEFAULT	;

//	fprintf
//	(	stderr,
//		"ColumnInfo::ColumnInfo: %12s: nu=%d ot=%04x p=%d ds=%d cs=%d\n",
//		(cchar *)m_colName,
//		m_colNullable,
//		m_colOraType,
//		m_colPrecision,
//		m_colDataSize,
//		m_colCharSize
//	)	;

	switch (m_colOraType) 
	{
		case SQLT_CHR	:		// Varchar2
		case SQLT_STR	:		// null-term String
		case SQLT_LNG	:		// Long
		case SQLT_VCS	:		// Varchar	
		case SQLT_LVC	:		// Long Varchar
		case SQLT_AFC	:		// Char
		case SQLT_AVC	:		// Charz

			/* All these map to rekall string, no		*/
			/* complication.				*/
			m_colRklType	= KB::ITString	;
			m_buffer	= new char [m_colDataSize + 1] ;
			m_valuep	= m_buffer	;
			m_dataSize	= m_colDataSize	;
			break	;

		case SQLT_INT	:		// Integer
			m_colRklType	= KB::ITFixed	;
			m_buffer	= new char [m_colDataSize + 1] ;
			m_valuep	= m_buffer	;
			m_dataSize	= m_colDataSize	;
			break	;

		case SQLT_NUM	:		// Number
		case SQLT_VNU	:		// Varnum
			m_colRklType	= KB::ITFloat	;
			m_buffer	= new char [22]	;
			m_valuep	= m_buffer	;
			m_dataSize	= 22		;
			break	;

		case SQLT_FLT	:		// Float/Double
		case SQLT_BFLOAT:		// Native Float
		case SQLT_BDOUBLE :		// Native Double
		case SQLT_IBFLOAT:		// Canonical Float
		case SQLT_IBDOUBLE :		// Canonical Double
			m_colRklType	= KB::ITFloat	;
			m_buffer	= new char [m_colDataSize + 1] ;
			m_valuep	= m_buffer	;
			m_dataSize	= m_colDataSize	;
			break	;

		case SQLT_DAT	:
			m_colRklType	= KB::ITDateTime;
			m_buffer	= new char [m_colDataSize + 1] ;
			m_valuep	= m_buffer	;
			m_dataSize	= m_colDataSize	;
			break	;

		case SQLT_VBI	:
		case SQLT_BIN	:
		case SQLT_LBI	:
		case SQLT_LVB	:
			m_colRklType	= KB::ITBinary	;
			m_buffer	= new char [m_colDataSize + 1] ;
			m_valuep	= m_buffer	;
			m_dataSize	= m_colDataSize	;
			break	;

		case SQLT_CLOB	:		// Character long object
			m_colRklType	= KB::ITString	;
			m_buffer	= 0		;
			m_dataSize	= -1		;
			m_colGetType	= m_colOraType	;
			m_lobLocator	= createLobLocator() ;
			m_valuep	= &m_lobLocator	;
			break	;

		case SQLT_BLOB	:		// Binary long object
			m_colRklType	= KB::ITBinary	;
			m_buffer	= 0		;
			m_dataSize	= -1		;
			m_colGetType	= m_colOraType	;
			m_lobLocator	= createLobLocator() ;
			m_valuep	= &m_lobLocator	;
			break	;

		case SQLT_RDD	:		// ROWID
			m_colRklType	= KB::ITString	;
			m_buffer	= 0		;
			m_dataSize	= -1		;
			m_colGetType	= m_colOraType	;
			m_rowid		= createRowID()	;
			m_valuep	= &m_rowid	;
			break	;

		case SQLT_DATE		:
		case SQLT_TIMESTAMP	:
		case SQLT_TIMESTAMP_TZ	:
		case SQLT_TIMESTAMP_LTZ	:
			m_colRklType	= KB::ITDateTime;
			m_buffer	= 0		;
			m_dataSize	= -1		;
			m_colGetType	= m_colOraType	;
			m_datetime	= createDateTime() ;
			m_valuep	= &m_datetime	;
			break	;
#if	0
		case SQLT_VST	:
		case SQLT_INTERVAL_YM	:
		case SQLT_INTERVAL_DS	:

    
		case SQLT_UIN	:
		case SQLT_FILE	:
		case SQLT_NTY	:
		case SQLT_REF	:
		case SQLT_RID	:
			return	KB::ITBinary	;
    
		case SQLT_ODT	:
			return	KB::ITDateTime	;
#endif
		default:
			/* If the OCI type is not recognised then it is	*/
			/* really an error (we should cover all cases)	*/
			/* but map it to the internal unknown type.	*/
			fprintf
			(	stderr,
				"ColumnInfo::ColumnInfo: unknown oracle type %d\n",
				m_colOraType
			)	;
			m_colRklType	= KB::ITUnknown	;
			m_buffer	= new char [m_colDataSize + 1] ;
			m_valuep	= m_buffer	;
			m_dataSize	= m_colDataSize	;
			m_mode		= OCI_DEFAULT	;
			break	;
	}

	ok	= true	;
}

ColumnInfo::~ColumnInfo ()
{
	if (m_buffer != 0)
		delete [] m_buffer ;

	if (m_lobLocator != 0)
		OCIDescriptorFree (m_lobLocator, OCI_DTYPE_LOB) ;

	if (m_rowid != 0)
		OCIDescriptorFree (m_rowid, OCI_DTYPE_ROWID) ;

	if (m_datetime != 0)
		OCIDescriptorFree (m_datetime, OCI_DTYPE_DATE) ;
}

OCILobLocator
	*ColumnInfo::createLobLocator ()
{
	OCILobLocator	*pLob	;
	ub4		empty	= 0 ;

	OCIDescriptorAlloc
	(	(dvoid  *)m_pServer->m_pEnvHandle,
		(dvoid **)&pLob,
		OCI_DTYPE_LOB,
		0, 0
	)	;

	OCIAttrSet
	(	pLob,
		OCI_DTYPE_LOB,
		&empty,
		0,
		OCI_ATTR_LOBEMPTY,
		m_pServer->m_pErrorHandle
	)	;

	return	pLob	;
}

OCIRowid
	*ColumnInfo::createRowID ()
{
	OCIRowid	*pRowID	;

	OCIDescriptorAlloc
	(	(dvoid  *)m_pServer->m_pEnvHandle,
		(dvoid **)&pRowID,
		OCI_DTYPE_ROWID,
		0, 0
	)	;

	return	pRowID	;
}

OCIDate
	*ColumnInfo::createDateTime ()
{
	OCIDate		*pDateTime ;

	OCIDescriptorAlloc
	(	(dvoid  *)m_pServer->m_pEnvHandle,
		(dvoid **)&pDateTime,
		OCI_DTYPE_DATE,
		0, 0
	)	;

	return	pDateTime ;
}

KBValue	ColumnInfo::value
	(	KBType		*type
	)
{
	switch (m_colOraType)
	{
		case SQLT_BLOB	:		// Binary long object
		case SQLT_CLOB	:		// Character long object
		{
			ub4	len	;
			sword	ora10iRC;

			/* Get the length. If the value is null then	*/
			/* OCI seems to return an invalid handle error,	*/
			/* which is a bit odd?				*/
			ora10iRC = OCILobGetLength
				   (	m_pServer->m_pServerContextHandle,
					m_pServer->m_pErrorHandle,
					m_lobLocator,
					&len
				   )	;

			if (ora10iRC != OCI_SUCCESS)
				return	KBValue (type) ;


			KBValue	v (type) ;
			char	*buffer	 = v.preallocate(len) ;

			ub4 amt  =  len	;
			ora10iRC = OCILobRead
				   (	m_pServer->m_pServerContextHandle,
					m_pServer->m_pErrorHandle,
					m_lobLocator,
					&amt,
					1,
					buffer,
					len,
					0,
					0,
					0,
					SQLCS_IMPLICIT
				   )	;

			KBError	error	;
			if (!m_pServer->checkOra10iOK 
				(	ora10iRC,
					"Retrieve LOB data",
					error
				))
				error.DISPLAY() ;

			return	v	;
		}

		case SQLT_RDD	:
		{
			ub2	len	 = 32 ;
			sword	ora10iRC ;
			KBValue	v (type) ;
			char	*buffer	 = v.preallocate(len) ;

			ora10iRC = OCIRowidToChar
				   (	m_rowid,
				   	(OraText *)buffer,
				   	&len,
					m_pServer->m_pErrorHandle
				   )	;

			KBError	error	;
			if (!m_pServer->checkOra10iOK 
				(	ora10iRC,
					"Retrieve RowID",
					error
				))
				error.DISPLAY() ;

			return	v ;
		}

		case SQLT_DATE		:
		case SQLT_TIMESTAMP	:
		case SQLT_TIMESTAMP_TZ	:
		case SQLT_TIMESTAMP_LTZ	:
		{
			sb2	year	;
			ub1	month	;
			ub1	day	;
			ub1	hour	;
			ub1	min	;
			ub1	sec	;

			OCIDateGetDate	(m_datetime, &year, &month, &day) ;
			OCIDateGetTime	(m_datetime, &hour, &min,   &sec) ;

			fprintf
			(	stderr,
				"Column::value: datetime=(%d,%d,%d,%d,%d,%d)\n",
				year, month, day,
				hour, min,   sec
			)	;

			return	KBValue
				(	QDateTime
					(	QDate (year, month, day),
						QTime (hour, min,   sec)
				)	)	;
		}

		default	:
			break	;
	}

	return	KBValue (m_buffer, m_resultLen, type) ;
}

/*  ------------------------------------------------------------------- */

/*  KBOra10iQrySelect							*/
/*  KBOra10iQrySelect							*/
/*		: Constructor for select query object			*/
/*  server	: KBOra10i *	    : Server connection object		*/
/*  data	: bool		    : Fetching data			*/
/*  selectStr	: const QString &   : Select query text			*/
/*  update	: bool		    : Select for update			*/
/*  (returns)	: KBOra10iQrySelect :					*/

KBOra10iQrySelect::KBOra10iQrySelect
	(	KBOra10i	*server,
		bool		data,
		const QString	&selectStr,
		bool		update
	)	
	:
	KBSQLSelect	(server, data, selectStr),
	m_pServer	(server),
	m_update	(update)
{
	m_CRow 		= -1	;
	m_nRows 	= 0	;
	m_nFields 	= 0	;
	m_columns	= 0	;

	if (m_update)
		if (m_pServer->m_useTimeouts)
			m_rawQuery += QString(" wait %1").arg(m_pServer->m_lockTimeout) ;

	/* Prepare the query when it is created. If this fails then	*/
	/* the error will be picked up when it is executed.		*/
	m_stmt	= m_pServer->prepare (m_rawQuery) ;
	if (m_stmt == 0)
		m_lError = m_pServer->lastError() ;
}

/*  KBOra10iQrySelect							*/
/*  KBOra10iQrySelect							*/
/*		: Constructor for select query object			*/
/*  server	: KBOra10i *	    : Server connection object		*/
/*  data	: bool		    : Fetching data			*/
/*  selectStr	: const QString &   : Select query text			*/
/*  stmt	: OCIStmt *	    : Extant statement			*/
/*  (returns)	: KBOra10iQrySelect :					*/

KBOra10iQrySelect::KBOra10iQrySelect
	(	KBOra10i	*server,
		bool		data,
		const QString	&selectStr,
		OCIStmt		*stmt
	)	
	:
	KBSQLSelect	(server, data, selectStr),
	m_pServer	(server),
	m_stmt		(stmt)
{
	m_CRow 		= -1	;
	m_nRows 	= 0	;
	m_nFields 	= 0	;
	m_columns	= 0	;
}

/*  KBOra10iQrySelect							*/
/*  ~KBOra10iQrySelect							*/
/*		: Destructor for select query object			*/
/*  (returns)	:		:					*/

KBOra10iQrySelect::~KBOra10iQrySelect()
{

	if (m_columns != 0)
	{
		for (uint idx = 0 ; idx < m_nFields ; idx += 1)
			if (m_columns[idx] != 0)
				delete	m_columns[idx] ;
		delete	[] m_columns ;
	}

	if (m_stmt != 0)
		OCIHandleFree ((dvoid *)m_stmt, OCI_HTYPE_STMT) ;
}

/*  KBOra10iQrySelect							*/
/*  execute	: Execute the query					*/
/*  nvals	: uint		: Number of substitution values		*/
/*  value	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBOra10iQrySelect::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	sword		ora10iRC	;
	OCIParam	*pParamHandle	;

	m_CRow 		= -1	;
	m_nRows 	= 0	;

	/* If the statement handle is not set then there must have been	*/
	/* an error at the preparation stage, so fail. The error will	*/
	/* have been picked up at that time.				*/
	if (m_stmt == 0)
		return	false	;

	QList<KBOra10iValue> vList ;
	vList.setAutoDelete (true) ;

	if (nvals > 0)
		if (!m_pServer->bindParameters (m_stmt, nvals, values, vList))
		{
			m_lError = m_pServer->lastError() ;
			return	false	;
		}

	ora10iRC = OCIStmtExecute
		   (	m_pServer->m_pServerContextHandle,
			m_stmt,
			m_pServer->m_pErrorHandle,
			0,
			0,
			(CONST OCISnapshot*)NULL,
			(OCISnapshot*)NULL,
			OCI_DEFAULT
		    )	;

	if (!m_pServer->checkOra10iOK (ora10iRC, "select records", m_lError))
	{
		fprintf
		(	stderr,
			"KBOra10iQrySelect::execute: Failed to execute SQL query statement\n"
		)	;

		m_pServer->printQuery (m_rawQuery, nvals, values, false) ;
		return	false	;
	}

	m_pServer->printQuery (m_rawQuery, nvals, values, true) ;

	ub4	nRows	= 0 ;
	ub4	nFields = 0 ;

	ora10iRC = OCIAttrGet
		   (	(dvoid *)m_stmt,
			(ub4	)OCI_HTYPE_STMT,
			(dvoid *)&nRows,
			(ub4   *)0,
			(ub4	)OCI_ATTR_ROW_COUNT,
			m_pServer->m_pErrorHandle
		   )	;

	if (!m_pServer->checkOra10iOK (ora10iRC, "Failed to get row count", m_lError))
		return	false	;

	ora10iRC = OCIAttrGet
		   (	(dvoid *)m_stmt,
			(ub4	)OCI_HTYPE_STMT,
			(dvoid *)&nFields,
			(ub4   *)0,
			(ub4    )OCI_ATTR_PARAM_COUNT,
			m_pServer->m_pErrorHandle
		   )	;

	if (!m_pServer->checkOra10iOK(ora10iRC, "Failed to get column count", m_lError))
		return	false	;

	fprintf
	(	stderr,
		"KBOra10iQrySelect::execute: m_nRows=%d m_nFields=%d\n",
		nRows,
		nFields
	)	;

	m_nRows	  = KBSQLSelect::RowsUnknown ;
	m_nFields = nFields	;


	/* If the types list does not exist, this is the first time	*/
	/* that this select has been executed, so load the type		*/
	/* information. We assume that these do not change under our	*/
	/* feet during the lifetime of this object.			*/
	if (m_types == 0)
	{
		ub4	count = 0 ;

		m_types	  = new KBType	   *[m_nFields] ;
		m_columns = new ColumnInfo *[m_nFields] ;

		for (uint idx = 0 ; idx < m_nFields ; idx += 1)
		{
			m_types  [idx] = 0 ;
			m_columns[idx] = 0 ;
		}

		ora10iRC = OCIParamGet
			   (	(dvoid *)m_stmt,
				OCI_HTYPE_STMT, 
				m_pServer->m_pErrorHandle,
				(void **)&pParamHandle,
				1
			   )	;

		while (ora10iRC == OCI_SUCCESS)
		{
			bool	   ok	    ;
			ColumnInfo *colInfo = new ColumnInfo
						  (	m_pServer,
							pParamHandle,
							ok,
							m_lError
						  )	;

			if (!ok) return false ;

			ora10iRC = OCIDefineByPos
				   (	m_stmt,
					&colInfo->m_define,
					m_pServer->m_pErrorHandle,
					count + 1,
					colInfo->m_valuep,
					colInfo->m_dataSize,
					colInfo->m_colGetType,
					(dvoid *)&colInfo->m_ind,
					&colInfo->m_resultLen,
					(ub2   *)0,
					colInfo->m_mode
				   )	;

			m_columns[count] = colInfo ;
			m_types  [count] = new KBOra10iType
					   (	colInfo->m_colOraType,
						colInfo->m_colRklType,
						colInfo->m_colDataSize,
						colInfo->m_colNullable
					   )	;

			count 	+= 1 ;
			ora10iRC = OCIParamGet
				   (	(dvoid *)m_stmt,
					OCI_HTYPE_STMT, 
					m_pServer->m_pErrorHandle,
					(void **)&pParamHandle,
					count + 1
				   )	;
		}

		if (count != m_nFields)
		{
			m_lError = KBError
				   (	KBError::Error,
					TR("Failed to get select column information"),
					QString(TR("Got %1 columns of %2"))
						.arg(count)
						.arg(m_nFields),
					__ERRLOCN
				   )	;
			return	false	;
		}
	}

	m_CRow = -1 ;
	return true ;
}


/*  KBOra10iQrySelect							*/
/*  rowExists	: See if row exists					*/
/*  qrow	: uint		: Row number				*/
/*  		: bool		:					*/
/*  (returns)	: bool		: True if row exists			*/

bool	KBOra10iQrySelect::rowExists
	(	uint 		qrow,
		bool
	)
{
	/* If the number of rows is now known (ie., we have previously	*/
	/* retrieved all rows) then we can answer immediately.		*/
	if (m_nRows != KBSQLSelect::RowsUnknown)
		return	(int)qrow < m_nRows ;


	/* We will need to load data up to the required row since we	*/
	/* can only fetch data in a row-column-serial manner. Values	*/
	/* fetched are stored in the cache.				*/
	while (m_CRow < (int)qrow)
	{
		fprintf
		(	stderr,
			"KBOra10iQrySelect::rowExists: preload %p: %u\n",
			m_stmt,
			m_CRow + 1
		)	;

		sword	ora10iRC ;
		ora10iRC = OCIStmtFetch2
			   (	m_stmt,
				m_pServer->m_pErrorHandle,
				1, 
				OCI_FETCH_NEXT,
				(sb4)0,
				OCI_DEFAULT
			   )	;

		switch (ora10iRC)
		{
			case OCI_NEED_DATA :
				//***
				break	;

		
			case OCI_SUCCESS   :
			case OCI_NO_DATA   :
				break	;

			case OCI_SUCCESS_WITH_INFO :
				ora10iRC = OCI_SUCCESS ;
				break	;

			default	:
				m_pServer->checkOra10iOK (ora10iRC, "Failed to fetch record", m_lError) ;
				break	;
		}

		if (ora10iRC != OCI_SUCCESS)
		{
			fprintf
			(	stderr,
				"KBOra10iQrySelect::rowExists: out of data at %d\n",
				m_CRow
			)	;
			m_nRows	= m_CRow ;
			break	;
		}

		m_CRow	+= 1 ;

		for (uint qcol = 0 ; qcol < m_nFields ; qcol += 1)
		{	KBValue	v = m_columns[qcol]->value(m_types[qcol]) ;
			putInCache (m_CRow, qcol, v) ;
		}
	}

	return	(m_CRow >= 0) && ((int)qrow <= m_CRow)	;
}

/*  KBOra10iQrySelect							*/
/*  getField	: Get a specified field from the query results		*/
/*  qrow	: uint		: Row number				*/
/*  qcol	: uint		: Column number				*/
/*  (returns)	: KBValue	: Value					*/

KBValue	KBOra10iQrySelect::getField
	(	uint 		qrow,
		uint		qcol,
		KBValue::VTrans
	)
{
	if (rowExists (qrow))
	{
		KBValue	value	;
		return	getFromCache (qrow, qcol, value) ? value : KBValue() ;
	}

	return	KBValue	()	;
}

#if 0
int KBOra10iQrySelect::piecewiseRead()
{
	OCIDefine* pDef;
	ub4 typeHandle;
	ub1 ioType;
	ub4 numMultipleRows;
	ub4 arrayIdx;
	ub1 piecePtr;
	sword status;
	text colData[DYNAMIC_CHUNK_SIZE + 1];
	int colNum = -1;
	int ora10iRC = 0;
	bool isColNull;

	for(; ;)
	{
		ora10iRC = OCIStmtGetPieceInfo(m_pServer->m_pStmtHandle, m_pServer->m_pErrorHandle, (dvoid**)&pDef,
					&typeHandle, &ioType, &numMultipleRows, &arrayIdx, &piecePtr);
		colNum = colFromDefine(pDef);
		int chunkSize = DYNAMIC_CHUNK_SIZE;
		isColNull = false;

		ora10iRC = OCIStmtSetPieceInfo(pDef, OCI_HTYPE_DEFINE,  m_pServer->m_pErrorHandle, (void*)colData,
					(ub4*)&chunkSize, piecePtr, NULL, NULL);

		status = OCIStmtFetch(m_pServer->m_pStmtHandle, m_pServer->m_pErrorHandle, 1, 
					OCI_FETCH_NEXT, OCI_DEFAULT);

		if(status == -1)
		{
			sb4 errorCode;
			OCIErrorGet((dvoid*)m_pServer->m_pErrorHandle, (ub4)1, (text*)NULL, &errorCode, NULL,
				0, OCI_HTYPE_ERROR);

			switch(errorCode)
			{
				case 1405 : // NULL data

					isColNull = true;
					break;

				default:

					break;
			}
		}

		if(status == OCI_NO_DATA)
			break;

		if(isColNull || !chunkSize)
		{
			// Since we don't have any data ....
			m_Data.replace(colNum, (char*)0L);
			int* pLen = new int();
			*pLen = 0;
			m_Lengths.replace(colNum, pLen);
		}
		else
		{
			// We have data. Since some data may have been returned
			// we set up a QByteArray to the size of that data and 
			// copy the data into the byte array
			m_pByteArray = new QByteArray(length(colNum));
			memcpy(m_pByteArray, m_Data.at(colNum), sizeof(m_Data.at(colNum)));

			// Set up a temp char buffer equal to the chunksize + the byte array
			// NB temp is deleted by QByteArray in the class dtor (I hope)
			char* tempBuff = (char*)malloc(chunkSize + m_pByteArray->size());
			// Copy the contents of the byte array to the temp buffer
			memcpy(tempBuff, m_pByteArray->data(), m_pByteArray->size());
			// Append the contains of colData to temp buffer
			memcpy(tempBuff + m_pByteArray->size(), colData, chunkSize);
			// Replace the old buffer size with the new one
			int* pLen = new int();
			*pLen = sizeof(tempBuff);
			m_Lengths.replace(colNum, pLen);
			// Update the byte array
			*m_pByteArray = m_pByteArray->assign(tempBuff, chunkSize + m_pByteArray->size());
			// Replace contents of the column buffer with contents of the byte array
			m_Data.replace(colNum, m_pByteArray->data());
		}

		if(status == OCI_SUCCESS_WITH_INFO || status == OCI_NEED_DATA)
		{
			// Do nothing just keep on looping
		}
		else
			break;
	}
	return ora10iRC;
}
#endif

#if	0
int KBOra10iQrySelect::lobsRead()
{
	int ora10iRC = 0;
	OCILobLocator* pLob;
	ub4 lobLength;

	for(int i = 0; i < count(); ++i)
	{
		pLob = lobLocator(i);

		if(!pLob || isNull(i))
			continue;

		ora10iRC = OCILobGetLength(m_pServer->m_pServerContextHandle, 
					m_pServer->m_pErrorHandle, pLob, &lobLength);

		if(ora10iRC != 0)
			lobLength = 0;

		if(m_pByteArray)
			delete m_pByteArray;

		if(lobLength > 0)
		{
			m_pByteArray = 0;

			m_pByteArray = new QByteArray(lobLength);
			// Get the lob charset ID and tell the server to convert it to UTF8
			ub1 charSetID = 0;
			ora10iRC = OCILobCharSetForm(m_pServer->m_pEnvHandle, 
					m_pServer->m_pErrorHandle, pLob, &charSetID);

			if(ora10iRC != 0)
				charSetID = 0;

			ora10iRC = OCILobRead(m_pServer->m_pServerContextHandle, m_pServer->m_pErrorHandle, pLob, &lobLength,
						1, (void*)m_pByteArray->data(),  (ub4)m_pByteArray->size(), 0, 0, 0, charSetID);

			if(ora10iRC != 0)
				fprintf(stderr, "Could not read LOB: %s", m_pServer->warning().local8Bit().data());
			else
			{
				int* pLen = new int();
				*pLen = lobLength;
				m_Lengths.replace(i, pLen);
				m_Data.replace(i, m_pByteArray->data());
			}
			// May we could delete m_pByteArray here
		}

		if(ora10iRC != 0 || !lobLength)
		{
			m_Data.replace(i, (char*)0L);
			int* pLen = new int();
			*pLen = 0;
			m_Lengths.replace(i, pLen);
			ora10iRC = 0; // It is a non-fatal error so we let it go
		}
	}
	return ora10iRC;
}
#endif

/*  KBOra10iQrySelect							*/
/*  getFieldName: Get a specified field name from the query results	*/
/*  qcol	: uint		: Column number				*/
/*  (returns)	: QString	: name					*/

QString	KBOra10iQrySelect::getFieldName
	(	uint		qcol
	)
{
	/* First check that the request is within the range of rows	*/
	/* and fields returned by the query.				*/
	if (qcol >= m_nFields) 
		return QString();

	return	m_columns[qcol]->m_colName ;
}

/*  KBOra10iQryUpdate							*/
/*  KBOra10iQryUpdate							*/
/*		: Constructor for update query object			*/
/*  server	: KBOra10i *	  : Server connection object		*/
/*  query	: const QString & : Update query text			*/
/*  tabName	: const QString & : Table being updated			*/
/*  (returns)	: KBOra10iQryUpdate:					*/

KBOra10iQryUpdate::KBOra10iQryUpdate
	(	KBOra10i		*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLUpdate	(server, data, query, tabName),
	m_pServer	(server)
{
	/* Prepare the query when it is created. If this fails then	*/
	/* the error will be picked up when it is executed.		*/	
	m_stmt	= m_pServer->prepare (m_rawQuery) ;
	if (m_stmt == 0)
		m_lError = m_pServer->lastError();
}

/*  KBOra10iQryUpdate							*/
/*  ~KBOra10iQryUpdate							*/
/*		: Destructor for update query object			*/
/*  (returns)	:		:					*/

KBOra10iQryUpdate::~KBOra10iQryUpdate ()
{
	if (m_stmt != 0)
		OCIHandleFree ((dvoid *)m_stmt, OCI_HTYPE_STMT) ;
}

/*  KBOra10iQryUpdate							*/
/*  execute	: Execute the query					*/
/*  nvals	: uint		: Number of substitution values		*/
/*  value	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBOra10iQryUpdate::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	/* If the statement handle is not set then there must have been	*/
	/* an error at the preparation stage, so fail. The error will	*/
	/* have been picked up at that time.				*/
	if (m_stmt == 0)
		return	false	;

	return	m_pServer->executeQuery
			 	(	m_stmt,
					nvals,
					values,
					m_nRows,
					m_rawQuery,
					"update"
				)	;
}


static	sword	callBackNoData
	(
		dvoid		*,
		OCIBind		*,
		ub4		, 
		ub4		, 
		dvoid		**bufpp,
		ub4		*alenp,
		ub1		*piecep, 
		dvoid            **indpp
	)
{
	static	sb2	ind	;

	fprintf
	(	stderr,
		"callBackNoData: called .........\n"
	)	;

	ind		= -1		;
	*bufpp		= (void *)""	;
	*alenp		= 0		;
	*piecep		= OCI_ONE_PIECE ;
	*indpp		= &ind		;

	return	OCI_CONTINUE ;
}

static	sword	callBackGetRowID
	(	dvoid		*octxp,
		OCIBind		*,
		ub4		, 
		ub4		, 
		dvoid		**bufpp, 
		ub4		**alenpp,
		ub1		*piecep,
		dvoid		**indpp, 
		ub2		**rcodepp
	)
{
	KBOra10iQryInsert::RowIDResult *rid = (KBOra10iQryInsert::RowIDResult *)octxp ;

	fprintf
	(	stderr,
		"callBackGetRowID: called .........\n"
	)	;

	rid->m_len 	= sizeof(rid->m_text) ;

	*bufpp		= rid->m_text	;
	*alenpp		= &rid->m_len	;
	*indpp		= &rid->m_ind	;
	*rcodepp	= &rid->m_rc	;
	*piecep		= OCI_ONE_PIECE ;

	return	OCI_CONTINUE ;
}


/*  KBOra10iQryInsert							*/
/*  KBOra10iQryInsert							*/
/*		: Constructor for insert query object			*/
/*  server	: KBOra10i *	  : Server connection object		*/
/*  query	: const QString & : Insert query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  (returns)	 KBOra10iQryInsert :					*/

KBOra10iQryInsert::KBOra10iQryInsert
	(	KBOra10i	*server,
		bool		data,
		const QString	&query,
		const QString 	&tabName
	)	
	:
	KBSQLInsert (server, data, query, tabName),
	m_pServer   (server)
{
	sword	ora10iRC;
	QString	extqry	= QString("%1 RETURNING ROWID INTO :x_rowid")
				.arg(m_rawQuery)
				;

	m_useSeq	= -1 ;
	m_fetchKey	= 0  ;

	if ((m_stmt = m_pServer->prepare (extqry)) == 0)
	{
		m_lError = m_pServer->lastError() ;
		return ;
	}

	ora10iRC = OCIBindByName
			   (	m_stmt,
				&m_bind,
				m_pServer->m_pErrorHandle,
				(const OraText *)":x_rowid",
				8,
				0,
				sizeof(m_rowid.m_text),
				SQLT_CHR,
				&m_rowid.m_ind,
				0,
				0,
				0,
				0,
				OCI_DATA_AT_EXEC
			   )	;

	if (ora10iRC == OCI_SUCCESS)
		ora10iRC = OCIBindDynamic
			   (	m_bind,
				m_pServer->m_pErrorHandle,
				(dvoid *)&m_rowid,
				callBackNoData,
				(dvoid *)&m_rowid,
				callBackGetRowID
			   )	;

	if (!m_pServer->checkOra10iOK (ora10iRC, "Adding ROWID return", m_lError))
	{
		OCIHandleFree     ((dvoid *)m_stmt, OCI_HTYPE_STMT) ;
		m_stmt	= 0	;
	}
}

/*  KBOra10iQryInsert							*/
/*  ~KBOra10iQryInsert							*/
/*		: Destructor for insert query object			*/
/*  (returns)	:		:					*/

KBOra10iQryInsert::~KBOra10iQryInsert ()
{
	if (m_stmt != 0)
		OCIHandleFree ((dvoid *)m_stmt, OCI_HTYPE_STMT) ;
	if (m_fetchKey != 0)
		OCIHandleFree ((dvoid *)m_fetchKey, OCI_HTYPE_STMT) ;
}

/*  KBOra10iQryInsert							*/
/*  execute	: Execute the query					*/
/*  nvals	: uint		: Number of substitution values		*/
/*  value	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBOra10iQryInsert::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	/* If the statement handle is not set then there must have been	*/
	/* an error at the preparation stage, so fail. The error will	*/
	/* have been picked up at that time.				*/
	if (m_stmt == 0) return false ;

	m_rowid.m_len	  = 0 ;
	m_rowid.m_text[0] = 0 ;

	bool rc = m_pServer->executeQuery
				(	m_stmt,
					nvals,
					values,
					m_nRows,
					m_rawQuery,
					"insert"
				)	;

	fprintf
	(	stderr,
		"KBOra10iQryInsert::execute: rc=%d\n",
		rc
	)	;

	if (rc) m_rowid.m_text[m_rowid.m_len] = 0 ;
	return rc ;
}

/*  KBOra10i								*/
/*  getNewKey	: Get new primary key value				*/
/*  primary	: const QString & : Key column name			*/
/*  newKey	: KBValue &	  : New key value			*/
/*  prior	: bool		  : Pri-insert call			*/
/*  (returns)	: bool		  : Success				*/

bool	KBOra10iQryInsert::getNewKey
	(	const QString	&primary,
		KBValue		&newKey,
		bool		prior
	)
{
	sword	ora10iRC ;

	/* Before we execute the query for the first time, we need to	*/
	/* see the primary key column is sequence-derived. If so then	*/
	/* we can get the primary key ahead of the insert, if not then	*/
	/* we will retrieve it afterwards via the row Oid.		*/
	if (m_useSeq < 0)
	{
		KBTableSpec tabSpec (m_tabName) ;
		if (!m_server->listFields (tabSpec))
		{	m_lError = m_server->lastError() ;
			return	 false ;
		}

		if ( (tabSpec.m_prefKey >= 0) &&
		     (tabSpec.m_fldList.at(tabSpec.m_prefKey)->m_name.lower() == primary.lower()) )
		{
			bool exists  ;
			if (!m_pServer->objectExists (m_tabName + "_seq", KB::IsSequence, exists))
			{	m_lError = m_server->lastError() ;
				return	 false ;
			}

			m_useSeq  = exists ;
		}
		else	m_useSeq = false  ;

		QString	fetch	;

		if (m_useSeq)
			fetch	= QString("select %1_seq.nextval from sys.dual").arg(m_tabName) ;
		else	fetch	= QString("select %1 from %2 where rowid = :x_rowid").arg(primary).arg(m_tabName) ;

		fprintf
		(	stderr,
			"KBOra10iQryInsert::getNewKey: using [%s]\n",
			(cchar *)fetch
		)	;

		if ((m_fetchKey = m_pServer->prepare (fetch)) == 0)
		{
			m_lError = m_server->lastError() ;
			return	 false ;
		}

		char *b = m_newKey.preallocate (32) ;
		ora10iRC = OCIDefineByPos
			   (	m_fetchKey,
				&m_pkDefine,
				m_pServer->m_pErrorHandle,
				1,
				b,
				32,
				SQLT_CHR,
				(dvoid *)&m_pkInd,
				&m_pkResultLen,
				(ub2   *)0,
				OCI_DEFAULT
			   )	;
		if (!m_pServer->checkOra10iOK (ora10iRC, "Primary key retrieval", m_lError))
			return	false	;
	}

	if (prior && !m_useSeq)
	{
		newKey	= KBValue() ;
		return	true ;
	}

	if (!prior)
	{
		OCIBind	 *pBindHandle	= 0 ;
		ora10iRC = OCIBindByName
				  (	m_fetchKey,
					&pBindHandle,
					m_pServer->m_pErrorHandle,
					(const OraText *)":x_rowid",
					8,
					m_rowid.m_text,
					strlen(m_rowid.m_text),
					SQLT_CHR,
					&m_pkInd,
					0,
					0,
					0,
					0,
					OCI_DEFAULT
				  )	;

		fprintf
		(	stderr,
			"KBOra10iQryInsert::getNewKey: for rowid [%s]\n",
			m_rowid.m_text
		)	;
	}

	int	rows	;
	if (!m_pServer->executeQuery (m_fetchKey, 0, 0, rows, "retrieve primary key", "getNewKey"))
	{
		m_lError = m_server->lastError() ;
		return	 false ;
	}

	ora10iRC = OCIStmtFetch2
		   (	m_fetchKey,
			m_pServer->m_pErrorHandle,
			1, 
			OCI_FETCH_NEXT,
			(sb4)0,
			OCI_DEFAULT
		   )	;

	if (!m_pServer->checkOra10iOK (ora10iRC, "retrieve primary key", m_lError))
		return	false	;

	fprintf
	(	stderr,
		"KBOra10iQryInsert::getNewKey: [%s]\n",
		(cchar *)m_newKey.getRawText()
	)	;

	newKey	= m_newKey	;
	return	true	;
}

/*  KBOra10iQryDelete							*/
/*  KBOra10iQryDelete							*/
/*		: Constructor for delete query object			*/
/*  server	: KBOra10i *	   : Server connection object		*/
/*  query	: const QString &  : Delete query			*/
/*  tabName	: const QString &  : Table being updated		*/
/*  (returns)	: KBOra10iQryDelete :					*/

KBOra10iQryDelete::KBOra10iQryDelete
	(	KBOra10i	*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLDelete (server, data, query, tabName),
	m_pServer   (server)
{
	/* Prepare the query when it is created. If this fails then	*/
	/* the error will be picked up when it is executed.		*/	
	m_stmt	= m_pServer->prepare (m_rawQuery) ;
	if (m_stmt == 0)
		m_lError = m_pServer->lastError();
}

/*  KBOra10iQryDelete							*/
/*  ~KBOra10iQryDelete							*/
/*		: Destructor for delete query object			*/
/*  (returns)	:		:					*/

KBOra10iQryDelete::~KBOra10iQryDelete ()
{
	if (m_stmt != 0)
		OCIHandleFree ((dvoid *)m_stmt, OCI_HTYPE_STMT) ;
}

/*  KBOra10iQryDelete							*/
/*  execute	: Execute the query					*/
/*  nvals	: uint		: Number of substitution values		*/
/*  value	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBOra10iQryDelete::execute
	(	uint 		nvals,
		const KBValue	*values
	)
{
	/* If the statement handle is not set then there must have been	*/
	/* an error at the preparation stage, so fail. The error will	*/
	/* have been picked up at that time.				*/
	if (m_stmt == 0)
		return	false	;

	return	m_pServer->executeQuery
			 	(	m_stmt,
					nvals,
					values,
					m_nRows,
					m_rawQuery,
					"insert"
				)	;
}



bool	KBOra10i::command
	(	bool		data,
		const QString 	&command,
		uint		nvals,
		KBValue		*values,
		KBSQLSelect 	**select
	)
{
	fprintf
	(	stderr,
		"KBOra10i::command: [%s]\n",
		(cchar *)command
	)	;

	OCIStmt	*stmt	= executeQuery
			  (	command,
				nvals,
				values,
				"command execution"
			  )	;
	if (stmt == 0) return false ;

	sword	ora10iRC 	;
	ub2	stmtType = 0	;

	ora10iRC = OCIAttrGet
		  (	stmt,
			OCI_HTYPE_STMT,
			(dvoid*)&stmtType,
			NULL,
			OCI_ATTR_STMT_TYPE,
			m_pErrorHandle
		  )	;
	if (!checkOra10iOK (ora10iRC, "command execution", m_lError))
	{
		OCIHandleFree ((dvoid *)stmt, OCI_HTYPE_STMT) ;
		return	false ;
	}

	if (stmtType != OCI_STMT_SELECT)
	{
		OCIHandleFree ((dvoid *)stmt, OCI_HTYPE_STMT) ;
		if (select != 0) *select = 0 ;
		return	true	;
	}

	if (select == 0)
	{
		OCIHandleFree ((dvoid *)stmt, OCI_HTYPE_STMT) ;
		return	true	;
	}

	*select	= new KBOra10iQrySelect (this, data, command, stmt) ;
	return	true	;
}

KBFACTORY
(	KBOra10iFactory,
	"driver_ora10i"
)

KBFACTORYIMPL
(	KBOra10iFactory,
	driver_ora10i,
	"Rekall Oracle 10i driver",
	"Plugin",
	"0",
	"7",
	"0"
)


QObject	*KBOra10iFactory::create
	(	QObject			*parent,
		cchar 			*object,
		cchar 			*,
		const QStringList 	&args
	)
{
	if (dIdentToType.count() == 0)
	{
		for (uint idx = 0 ; idx < sizeof(typeMap)/sizeof(Ora10iTypeMap) ; idx += 1)
		{
			Ora10iTypeMap *m = &typeMap[idx] ;
			dIdentToType.insert(m->m_ora9iIdent, m) ;
		}
	}

	if ((parent != 0) && !parent->inherits("QWidget"))
	{
		fprintf
		(	stderr,
			"KBOra10iFactory: parent does not inherit QWidget\n"
		)	;
		return	0 ;
	}
	
	if (strcmp(object, "driver"  ) == 0) return new KBOra10i ();
	if (strcmp(object, "advanced") == 0) return new KBOra10iAdvanced () ;

	return	0 ;
}

cchar	*KBOra10iFactory::ident ()
{
	return	__KB_BUILD_IDENT	;
}
