/***************************************************************************
    file	         : kb_datetime.cpp
    copyright            : (C)	1999,2000,2001,2002,2003,2004,2005
				 by Mike Richardson
			   (C)	2001,2002,2003,2004,2005
				by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/


#include	<stdio.h>
#include	<stdlib.h>
#include	<ctype.h>

#include	<qintdict.h>
#include	<qlist.h>
#include 	<qregexp.h>
#include 	<qobject.h>


#define	__LOGGING	(0)

#if	__LOGGING
#define	LOGGING(x)	fprintf	(stderr, "KBDateTime::KBDateTime: ") ;	\
			fprintf x ;					\
			fprintf (stderr, "\n")			     ;
#else
#define	LOGGING(x)
#endif

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"


#if		__KB_TKC
class		KLocale	;
#endif		// __KB_TKC

#if		__KB_KDE

#include	<klocale.h>
#include	<kglobal.h>

#endif		// __KB_KDE


static	QStringList	weekDayNamesFull	;
static	QStringList	weekDayNamesAbbrev	;
static	QStringList	monthNamesFull		;
static	QStringList	monthNamesAbbrev	;


struct	LIBCOMMON_API	DTItem
{
	int	fchar	;	/* Format key character			*/
	int	offset	;	/* Offset into date/time array		*/
	int	rem	;	/* Display offset from data value	*/
	cchar	*fmt	;	/* Format string			*/
	int	len	;	/* Format output length			*/
}	;

#define	DT_YEAR		0
#define	DT_MONTH	1
#define	DT_DAY		2
#define	DT_HOUR		3
#define	DT_MIN		4
#define	DT_SEC		5
#define	DT_YDAY		6
#define	DT_WDAY		7
#define	DT_AMPM		8
#define	DT_SIZE		9

#define	DT_FORMAT	99

static	DTItem	dtItems[] =
{
{	'C',	DT_YEAR,	-100,	"%02d",			2	},
{	'd',	DT_DAY,		0,	"%02d",			2	},
{	'e',	DT_DAY,		0,	"%d",			0	},
{	'H',	DT_HOUR,	0,	"%02d",			2	},
{	'I',	DT_HOUR,	12,	"%02d",			2	},
{	'j',	DT_YDAY,	0,	"%03d",			2	},
{	'k',	DT_HOUR,	0,	"%2d",			2	},
{	'l',	DT_HOUR,	12,	"%2d",			2	},
{	'm',	DT_MONTH,	0,	"%02d",			2	},
{	'M',	DT_MIN,		0,	"%02d",			2	},
{	'S',	DT_SEC,		0,	"%02d",			2	},
{	'u',	DT_WDAY,	0,	"%d",			0	},
{	'y',	DT_YEAR,	100,	"%02d",			2	},
{	'Y',	DT_YEAR,	0,	"%04d",			4	},

/* The following cases are recursive ....				*/
{	'c',	DT_FORMAT,	0,	"%x %X",		-1	},
{	'D',	DT_FORMAT,	0,	"%m/%d/%y",		-1	},
{	'r',	DT_FORMAT,	0,	"%I:%M:%S %p",		-1	},
{	'R',	DT_FORMAT,	0,	"%H:%M",		-1	},
{	'T',	DT_FORMAT,	0,	"%H:%M:%S",		-1	},
{	'+',	DT_FORMAT,	0,	"%a %b %e %T %Z %Y",	-1	}
}	;

static	QIntDict<DTItem>	dtMap	;

/*  The following table lists the weekdays, the months, plus some other	*/
/*  things than can appear in dates. Note that we should do something	*/
/*  about making this language dependant.				*/

static	cchar	*names[] =
{	"sunday",
	"monday",
	"tuesday",
	"wednesday",
	"thursday",
	"friday",
	"saturday",
	"january",
	"february",
	"march",
	"april",
	"may",
	"june",
	"july",
	"august",
	"september",
	"october",
	"november",
	"december",
	"am",
	"pm",
	"gmt",
	"pst",
	"bst",
	"edt",
	"st",
	"nd",
	"rd",
	"th",
	"ut",
	0
}	;
#define	NUMDAYS	7
#define	NUMMONS	12
#define	NUMISAM	(NUMDAYS+NUMMONS+0)
#define	NUMISPM	(NUMDAYS+NUMMONS+1)

static	int	mDays [] =
{	31,    28,    31,    30,    31,    30,
	31,    31,    30,    31,    30,    31
}	;


static	KBDateTime::DateOrder	dateOrder = KBDateTime::DayMonthYear ;



class 	LIBCOMMON_API	DTToken
{
public	:
	int	sep	;
	QString	token	;
	bool	digit	;
	int	nidx	;
}	;


/*  getName	: Locate name entry for specified string		*/
/*  str		: cchar *	: String in question			*/
/*  (returns)	: int		: Index in name list or negative	*/

static	int	getName
	(	cchar		*str
	)
{
	int	slen	= qstrlen (str) ;
	cchar	**nptr	;

	for (nptr = &names[0] ; *nptr != 0 ; nptr += 1)
	{
		cchar	*nam	= *nptr	      ;
		int	nlen	= strlen(nam) ;

		/* If the name is less than three charactes then do a	*/
		/* length-dependant match. This distinguishes between,	*/
		/* say, "thursday" and "th".				*/
		if (nlen < 3)
		{	if (qstricmp (str, nam) == 0) break ;
			continue ;
		}

		if ((slen >= 3) && (qstrnicmp (str, nam, slen) == 0))
			break	;
	}

	return	(*nptr == 0) ? -1 : nptr - &names[0] ;
}

/*  fixYear	: Decode and fix year string				*/
/*  tokp	: DTToken	: Token					*/
/*  (returns)	: int		: Year as a number			*/

static	int	fixYear
	(	DTToken	*tokp
	)
{
	int	yno	= tokp->token.toInt() ;
	if (tokp->token.length() == 2)
		if	(yno >= 70) yno += 1900 ;
		else if (yno <  70) yno += 2000 ;
	return	yno	;
}


/*  getLocale	: Get local information					*/
/*  (returns)	: KLocale *	: Locale				*/

static	KLocale	*getLocale ()
{
	for (uint idx = 0 ; idx < sizeof(dtItems)/sizeof(DTItem) ; idx += 1)
		dtMap.insert (dtItems[idx].fchar, &dtItems[idx]) ;

#if	__KB_KDE
	static	KLocale	*locale	;

	/* Set up on the first call. Try to figure the default date	*/
	/* order. Format a date which can then be interpreted. If that	*/
	/* does not help then the US is MDY and everywhere else is	*/
	/* DMY.								*/
	if (locale == 0)
	{
		locale	= KGlobal::locale () ;

		QDate	dTest (2000, 10, 1)  ;
		QString	dText = locale->formatDate (dTest, true) ;
		int	dayno = dText.left(2).toInt() ;

		fprintf
		(	stderr,
			"getLocale: (2000,10,1)->[%s]->[%d] co=[%s]\n",
			dText.latin1(),
			dayno,
			locale->country().lower().latin1()
		)	;

		if	(dayno == 1 )
		{
			dateOrder = KBDateTime::DayMonthYear ;
		}
		else if (dayno == 10)
		{
			dateOrder = KBDateTime::MonthDayYear ;
		}
		else
		{
			dateOrder = locale->country().lower() == "us" ?
						KBDateTime::MonthDayYear :
						KBDateTime::DayMonthYear ;
		}

		fprintf
		(	stderr,
			"Date Format: [%s] [%s]\n",
			(cchar *)dText,
			dateOrder == KBDateTime::DayMonthYear ? "DMY" : "MDY"
		)	;

		for (uint day = 0 ; day <= 7 ; day += 1)
		{	weekDayNamesAbbrev.append (getLocale()->weekDayName (day, true )) ;
			weekDayNamesFull  .append (getLocale()->weekDayName (day, false)) ;
		}
		for (uint mon = 0 ; mon <= 12 ; mon += 1)
		{	monthNamesAbbrev  .append (getLocale()->monthName   (mon, true )) ;
			monthNamesFull    .append (getLocale()->monthName   (mon, false)) ;
		}
	}

	return	locale	;
#endif

#if	__KB_TKC
	static	bool	loaded	;

	if (!loaded)
	{
		loaded	= true	;
	
		static	const char	*dayFull  [] =
		{	0,
			"Monday",
			"Tuesday",
			"Wednesday",
			"Thursday",
			"Friday",
			"Saturday",
			"Sunday"
		}	;
		static	const char	*dayAbbrev[] =
		{	0,
			"Mon",
			"Tue",
			"Wed",
			"Thu",
			"Fri",
			"Sat",
			"Sun"
		}	;
		static	const char	*monFull   [] =
		{	0,
			"January",
			"February",
			"March",
			"April",
			"May",
			"June",
			"July",
			"August",
			"September",
			"October",
			"November",
			"December"
		}	;
		static	const char	*monAbbrev [] =
		{	0,
			"Jan",
			"Feb",
			"Mar",
			"Apr",
			"May",
			"Jun",
			"Jul",
			"Aug",
			"Sep",
			"Oct",
			"Nov",
			"Dec",
		}	;

		for (uint day = 0 ; day <= 7 ; day += 1)
		{	weekDayNamesAbbrev.append (dayAbbrev[day]) ;
			weekDayNamesFull  .append (dayFull  [day]) ;
		}
		for (uint mon = 0 ; mon <= 12 ; mon += 1)
		{	monthNamesAbbrev  .append (monAbbrev[mon]) ;
			monthNamesFull    .append (monFull  [mon]) ;
		}
	}

	return	0 ;
#endif
}

/*  weekDayName	: Get name of day of week				*/
/*  idx		: int		: Day number				*/
/*  abbrev	: bool		: Return abbreviated form		*/
/*  (returns)	: QString	: Name					*/

static	QString	weekDayName
	(	int		idx,
		bool		abbrev
	)
{
	if ((idx >= 0) && (idx <= 7))
		if (abbrev)
			return	weekDayNamesAbbrev[idx] ;
		else	return	weekDayNamesFull  [idx] ;

	return	"???"	;
}

/*  monthName	: Get name of month					*/
/*  idx		: int		: Month number				*/
/*  abbrev	: bool		: Return abbreviated form		*/
/*  (returns)	: QString	: Name					*/

static	QString	monthName
	(	int		idx,
		bool		abbrev
	)
{
	if ((idx >= 0) && (idx <= 12))
		if (abbrev)
			return	monthNamesAbbrev[idx] ;
		else	return	monthNamesFull  [idx] ;

	return	"???"	;
}

#if	__KB_KDE

/*  formatDate	: Get specified date in default format			*/
/*  date	: const QDate &	: Date					*/
/*  (returns)	: QString	: Formatted version			*/

static	QString	formatDate
	(	const QDate	&date
	)
{
	return	getLocale()->formatDate (date) ;
}

/*  formatTime	: Get specified time in default format			*/
/*  time	: const QTime &	: Time					*/
/*  (returns)	: QString	: Formatted version			*/

static	QString	formatTime
	(	const QTime	&time
	)
{
	return	getLocale()->formatTime (time) ;
}

#endif	// __KB_KDE

#if	__KB_TKC

/*  formatDate	: Get specified date in default format			*/
/*  date	: const QDate &	: Date					*/
/*  (returns)	: QString	: Formatted version			*/

static	QString	formatDate
	(	const QDate	&date
	)
{
	return	QString().sprintf
		(	"%04d-%02d-%02d",
			date.year  (),
			date.month (),
			date.day   ()
		)	;
}

/*  formatTime	: Get specified time in default format			*/
/*  time	: const QTime &	: Time					*/
/*  (returns)	: QString	: Formatted version			*/

static	QString	formatTime
	(	const QTime	&time
	)
{
	return	QString().sprintf
		(	"%02d:%02d:%02d",
			time.hour  (),
			time.minute(),
			time.second()
		)	;
}

#endif	// __KB_TKC

#define	__year	data[DT_YEAR ]
#define	__mon	data[DT_MONTH]
#define	__day	data[DT_DAY  ]
#define	__hour	data[DT_HOUR ]
#define	__mins	data[DT_MIN  ]
#define	__sec	data[DT_SEC  ]
#define	__ampm	data[DT_AMPM ]



/*  KBDateTime								*/
/*  doDeFormat	: Decode date/time string based on prior format		*/
/*  toknum	: uint		   : Token from which to process	*/
/*  tokens	: QList<DTToken> & : List of tokens			*/
/*  fmt		: const QString &  : Format string			*/
/*  data	: int[]		   : Array for results			*/
/*  (returns)	: int		   : Next token or negative on error	*/

int	KBDateTime::doDeFormat
	(	int		toknum,
		QList<DTToken>	&tokens,
		const QString	&fmt,
		int		data[]
	)
{
	int	offset	= 0	;

	for (uint d = 0 ; d < DT_SIZE ; d += 1) data[d] = -1 ;

	while ((toknum < (int)tokens.count()) && (offset = fmt.find('%', offset)) >= 0)
	{
		LOGGING
		((	stderr,
			"doDeFormat: [%s]",
			fmt.mid(offset, 2).latin1()
		))	;

		DTToken	*dt	= tokens.at(toknum) ;
		char	ftch	= fmt[offset+1] ;
		offset	+= 1	;

		switch (ftch)
		{
			case 'a' :
			case 'A' :
				/* Actual day name. Not enough info so	*/
				/* skip this token.			*/
				toknum	+= 1	;
				continue	;

			case 'C' :
				/* Century number is as much use as a	*/
				/* chocolate tea-pot.			*/
				toknum	+= 1	;
				continue	;

			case 'b' :
			case 'h' :
			case 'B' :
				/* Month name, store as the month	*/
				/* number and advance the tokens	*/
				data[DT_MONTH] = dt->nidx - NUMDAYS + 1 ;
				toknum	+= 1	;
				continue	;

			case 'n' :
			case 't' :
			case '%' :
				/* Ignore these, they are just escapes.	*/
				continue	;

			default	:
				break	;
		}

		/* Not a special case, so look for the format character	*/
		/* in the type map. If not found then silently ignore	*/
		/* it.							*/
		DTItem	*di	= dtMap.find (ftch) ;

		LOGGING
		((	stderr,
			"doDeFormat: [%c]->[%p] from [%s]",
			ftch,
			(void *)di,
			dt->token.latin1()
		))	;

		if (di == 0)
			continue ;

		/* Specical case here is DT_FORMAT, in which case we	*/
		/* recursively process the format string.		*/
		if (di->offset == DT_FORMAT)
		{
			toknum	= doDeFormat (toknum, tokens, di->fmt, data) ;
			if (toknum < 0) return toknum ;
			continue  ;
		}

		/* Get the numerical value. Then do some range checks	*/
		/* to make sure we have sensible values.		*/
		int	v = dt->token.toInt() ;
		switch (di->offset)
		{
			case DT_YEAR	:
				if (v >= 0)
					if	(v <=  50) v += 2000 ;
					else if	(v <  100) v += 1900 ;
				break	;

			case DT_HOUR	:
				v += di->rem ;
				break	;

			default	:
				break	;
		}

		data[di->offset] = v ;
		toknum	+= 1 ;
	}

	/* Note that the actual decoded values will be checked in the	*/
	/* ::decodeOK() method, so if we have decoded (say) a day value	*/
	/* 32, this will be picked up later.				*/
	return	toknum	;
}

/*  KBDateTime								*/
/*  doDecode	: Decode date/time without using a format		*/
/*  tokens	: QList<DTToken> & : List of tokens			*/
/*  data	: int[]		   : Array for results			*/
/*  (returns)	: bool		   : Success				*/

bool	KBDateTime::doDecode
	(	QList<DTToken>	&tokens,
		int		data[]
	)
{
#define	SF(f,v)	\
	{	if (f != -1) return false ;	\
		f = v ;				\
	}

	uint	tokp	= 0 ;
	int	left	;

	for (uint d = 0 ; d < DT_SIZE ; d += 1) data[d] = -1 ;

	while ((left = tokens.count() - tokp) > 0)
	{
		DTToken	*tok0	= tokens.at(tokp) ;
		DTToken	*tok1	= left > 1 ? tokens.at(tokp + 1) : 0 ;
		DTToken	*tok2	= left > 2 ? tokens.at(tokp + 2) : 0 ;
		DTToken	*tok3	= left > 3 ? tokens.at(tokp + 3) : 0 ;
		uint	drop	= 0 ;

		LOGGING	((stderr, " .... token [%s] with %d remaining", (cchar *)tok0->token, left)) ;

		/* First set of cases are where there are three tokens	*/
		/* available. Look for dates, and times where the	*/
		/* number of seconds is set.				*/
		if (tok2 != 0)
		{
			/* NN:NN:NN					*/
			/* Decode this as a time with the value order	*/
			/* of (hour, minute, second).			*/
			if	(tok0->digit &&  tok1->digit && tok2->digit
					&& (tok1->sep == ':') && (tok2->sep == ':'))
			{
				SF(__hour, tok0->nidx) ;
				SF(__mins, tok1->nidx) ;
				SF(__sec,  tok2->nidx) ;

				if ((tok3 != 0) && (tok3->sep == '.'))
					drop	= 4  ;
				else	drop	= 3  ;
				goto	advance	;
			}
			/* NN-CCC-NN					*/
			/* This is a date in order (day,month,year)	*/
			/* where the month has been named.		*/
			else if (tok0->digit && !tok1->digit && tok2->digit
					&& (tok1->sep == '-') && (tok2->sep == '-'))
			{
				
				SF(__day,  tok0->nidx   ) ;
				SF(__mon,  tok1->nidx - NUMDAYS + 1) ;
				SF(__year, fixYear(tok2)) ;
				drop	= 3	;
				goto	advance	;
			}
			/* NN-NN-NN					*/
			/* NN/NN/NN					*/
			/* NN.NN.NN					*/
			/* Decode thiese as a date. If the first token	*/
			/* is more then two characters long then treat	*/
			/* it as (year,month, day). Otherwise, treat as	*/
			/* (day,month,year) or (month,day,year)		*/
			/* depending on the default date ordering.	*/
			else if (tok0->digit &&  tok1->digit && tok2->digit
					&& (tok1->sep == tok2->sep)
					&& ( (tok1->sep == '-') ||
					     (tok1->sep == '/') ||
					     (tok1->sep == '.') ) )
			{
				if	(tok0->token.length() > 2)
				{
					SF(__year, fixYear(tok0)) ;
					SF(__mon,  tok1->nidx   ) ;
					SF(__day,  tok2->nidx   ) ;
				}
				else if	(dateOrder == KBDateTime::DayMonthYear)
				{
					SF(__year, fixYear(tok2)) ;
					SF(__mon,  tok1->nidx   ) ;
					SF(__day,  tok0->nidx   ) ;
				}
				else
				{
					SF(__year, fixYear(tok2)) ;
					SF(__mon,  tok0->nidx   ) ;
					SF(__day,  tok1->nidx   ) ;
				}

				drop	= 3	;
				goto	advance	;
			}
		}

		/* Not three tokens or nothing matched, next check for	*/
		/* a two-token case, which can only be a time with the	*/
		/* seconds unspecified (and interpreted as zero).	*/
		if (tok1 != 0)
			if	(tok0->digit && tok1->digit && (tok1->sep == ':'))
			{
				SF(__hour, tok0->nidx) ;
				SF(__mins, tok1->nidx) ;
				SF(__sec,  0) ;
				drop	= 2	;
				goto	advance ;
			}

		/* Allow three and four digit tokens which have a	*/
		/* leading + or -; these are GMT offsets. For the	*/
		/* moment, just skip them.				*/
		if ((tok0->token.length() >= 3) && (tok0->token.length() <= 4))
			if ((tok0->sep == '+') || (tok0->sep == '-'))
			{
				drop	= 1	;
				goto	advance	;
			}

		/* Only one token or again, no matches. Days are of	*/
		/* no use, but check for months or for AM/PM.		*/
		if (!tok0->digit)
		{
			if	(tok0->nidx <  NUMDAYS)
				;
			else if	(tok0->nidx <  NUMDAYS+NUMMONS)
				SF(__mon,  tok0->nidx - NUMDAYS + 1)
			else if (tok0->nidx == NUMISAM)
				SF(__ampm, NUMISAM)
			else if (tok0->nidx == NUMISPM)
				SF(__ampm, NUMISPM)

			drop	= 1	;
			goto	advance	;
		}

		/* Two digit numbers are interpreted as day numbers, so	*/
		/* we can handle things like '11 July'.			*/
		if (tok0->token.length() <= 2)
		{
			if ((tok0->sep != '+') && (tok0->sep != '-'))
				SF(__day,  tok0->nidx) ;
			drop	= 1	;
			goto	advance	;
		}

		/* Four digit tokens can only be years ...		*/
		if ((tok0->token.length() <= 4) && tok0->digit)
		{
			SF(__year, tok0->nidx) ;
			drop	= 1	;
			goto	advance	;
		}

		/* ... and anything else ends the scan.			*/
		LOGGING	((stderr, "fail on [%s]", (cchar *)tok0->token)) ;
		return	false	;

		advance	:
			tokp	+= drop ;
	}

	/* Note that the actual decoded values will be checked in the	*/
	/* ::decodeOK() method, so if we have decoded (say) a day value	*/
	/* 32, this will be picked up later.				*/
	return	true	;
}

/*  KBDateTime								*/
/*  decodeOK	: Check that the decoded values are OK			*/
/*  data	: int[]		   : Array of decoded results		*/
/*  (returns)	: bool		   : Success				*/

bool	KBDateTime::decodeOK
	(	int		data[]
	)
{
	/* This method does the actual checking. The decoder methods	*/
	/* are responsible for breaking up the date text; this method	*/
	/* checks that the results are plausible.			*/
	m_hasDate = false	;
	m_hasTime = false	;

	LOGGING
	((	stderr,
		" .... decoded: hour=%d ampm=%d",
		__hour,
		__ampm
	))	;

	if (__hour != -1)
	{
		if ((__ampm != -1) && (__hour >= 12))
			return	false	;

		if (__ampm == NUMISPM)
			__hour += 12	;
	}

	LOGGING
	((	stderr,
		" .... decoded: %d-%d-%d %d:%d:%d",
		__year, __mon,  __day,
		__hour, __mins, __sec
	))	;

	if ((__year != -1) && (__mon != -1) && (__day != -1))
	{
		if ((__year < 1752) || (__year > 8000)) return false ;
		if ((__mon  <    1) || (__mon  >   12)) return false ;
		if ( __day  <    1		      ) return false ;

		mDays[1] = __year % 4   != 0 ? 28 :
			   __year % 400 == 0 ? 29 :
			   __year % 100 == 0 ? 28 : 29 ;

		if ( __day  > mDays[__mon-1])
			return	false	;

		m_dt.setDate (QDate(__year, __mon, __day)) ;
		m_hasDate = true ;
	}
	else	m_dt.setDate (QDate()) ;

	if ((__hour != -1) && (__mins != -1) && (__sec != -1))
	{
		if (__hour >= 24) return false ;
		if (__mins >= 60) return false ;
		if (__sec  >= 60) return false ;

		m_dt.setTime (QTime(__hour, __mins, __sec)) ;
		m_hasTime = true ;
	}
	else	m_dt.setTime (QTime()) ;

	return	true		;
}

/*  KBDateTime								*/
/*  KBDateTime	: Constructor for date-time object			*/
/*  _dtstr	: const QString & : Date/Time string			*/
/*  fmt		: const QString & : Format				*/
/*  (returns)	: KBDateTime	  :					*/

KBDateTime::KBDateTime
	(	const QString	&_dtstr,
		const QString	&fmt
	)
{

	int		lch	= ' ' ;
	QList<DTToken>	tokens	;
	DTToken		*tokp	;

	/* Ensure that all date information is set up before we use any	*/
	/* The "getLocale()" routine does this as a side effect.	*/
	getLocale() ;

	LOGGING	((stderr, "[%s]", (cchar *)_dtstr)) ;

	/* Duplicate the date/time string and store it as the raw data.	*/
	/* This copy is also used in the parser below, but is left	*/
	/* unchanged.							*/
	const char 	*copy  = strdup (_dtstr)   ;
	const char	*dtstr = copy	    	   ;
	m_raw.assign	(copy, strlen (copy ))	;

	m_valid		= false	    ;
	m_hasDate	= false	    ;
	m_hasTime	= false	    ;
	tokens.setAutoDelete (true) ;

	while (*dtstr)
	{
		cchar	*tok	;

		if (!isalnum(*dtstr))
		{	lch	= *dtstr;
			dtstr    += 1	;
			continue	;
		}

		tokens.append (tokp = new DTToken) ;

		tokp->sep	= lch	;
		tokp->nidx	= 0	;
		tokp->digit	= isdigit(*dtstr)  ;
		tok		= dtstr	;

		while (tokp->digit ? isdigit(*dtstr) : isalpha(*dtstr))
			dtstr += 1 ;

		tokp->token	= QString(tok).left(dtstr - tok) ;

		LOGGING	((stderr, "token [%s]", (cchar *)tokp->token))

		if (!tokp->digit)
		{
			if ((tokp->nidx = getName((cchar *)tokp->token)) < 0)
			{
				LOGGING	((stderr, "failed on token [%s]", (cchar *)tokp->token)) ;
				return	;
			}
		}
		else	tokp->nidx = atol (tok) ;
	}

#if	__LOGGING
	LOGGING	((stderr, "[%s] -> %d tokens", (cchar *)_dtstr, tokens.count())) ;
	LITER
	(	DTToken,
		tokens,
		tokp,
		LOGGING	((stderr, ".... %c %s %d", tokp->sep, (cchar *)tokp->token, tokp->nidx)) ;
	)
#endif

	/* If there is a single token then it may be something like	*/
	/* "20050111...." comprising year-month-day and perhaps		*/
	/* hour-minute-second. This must not have any non-digits.	*/
	if (tokens.count() == 1)
	{
		const QString	&dstr	= tokens.at(0)->token ;
		int		data[DT_SIZE]	;
		for (uint d = 0 ; d < DT_SIZE ; d += 1) data[d] = -1 ;

		/* Assume all digits, but no year-month-day nor		*/
		/* hour-minute-second parts. Then scan the token to	*/
		/* see what we actually have.				*/
		bool		alld	= true	;
		bool		ymd	= false	;
		bool		hms	= false	;

		for (uint idx = 0 ; idx < dstr.length() ; idx += 1)
		{
			if (!dstr.at(idx).isDigit())
			{	alld	= false	;
				break	;
			}
			if (idx ==  7) ymd = true ;
			if (idx == 13) hms = true ;
		}

		/* Unless all characters are digits and we have a valid	*/
		/* year-month-day, then treat it as unknown.		*/
		if (!alld || !ymd)
		{
			LOGGING	((stderr, " .... single unknown token")) ;
			return	;
		}

		/* Decode the year-month-day, which we know we have,	*/
		/* and the hour-minute-second if we have that.		*/
		__year	= dstr.mid( 0, 4).toInt() ;
		__mon	= dstr.mid( 4, 2).toInt() ;
		__day	= dstr.mid( 6, 2).toInt() ;

		LOGGING	((stderr, " .... YMD %04d-%02d-%02d", __year, __mon, __day)) ;

		m_dt.setDate (QDate(__year, __mon, __day)) ;
		m_hasDate = true  ;

		if (hms)
		{	__hour	= dstr.mid( 8, 2).toInt() ;
			__mins	= dstr.mid(10, 2).toInt() ;
			__sec	= dstr.mid(12, 2).toInt() ;

			LOGGING	((stderr, " .... HMS %02d-%02d-%02d", __hour, __mins, __sec)) ;

			m_dt.setTime (QTime(__hour, __mins, __sec)) ;
			m_hasTime = true ;
		}

		m_valid	  = true  ;
		return	;
	}

	bool	ok	= false ;
	int	data[DT_SIZE]	;

	if (!fmt.isEmpty())
	{
		if (doDeFormat (0, tokens, fmt, data) >= 0)
			ok = decodeOK (data) ;
	}

	if (!ok)
	{
		if (doDecode   (tokens, data))
			ok = decodeOK (data) ;
	}

	if (!ok) return	;

	m_valid	= m_hasDate || m_hasTime ;
}

KBDateTime::KBDateTime
	(	const QDateTime	&dt
	)
	:
	m_dt	(dt),
	m_valid	(dt.isValid())
{
	m_raw	= (QCString)defFormat(KB::ITDateTime) ;
}

/*  KBDateTime								*/
/*  format	: Format into a text string				*/
/*  _fmt	: const QString & : Format specification		*/
/*  (returns)	: QString	  : Text string				*/
	
QString	KBDateTime::format
	(	const QString	&_fmt
	)
	const
{
	/* Ensure that all date information is set up before we use	*/
	/* any The "getLocale()" routine does this as a side effect.	*/
	getLocale() ;


	/* If the decoded date/time is marked as invalid then make do	*/
	/* with the original raw text.					*/
	if (!m_valid) return m_raw ;

	QString	res	;
	QString	tmp	;
	DTItem	*di	;
	int	idx	;
	char	fc	;

	int	dv[DT_SIZE] ;

	dv[DT_YEAR ]	= m_dt.date().year      () ;
	dv[DT_MONTH]	= m_dt.date().month     () ;
	dv[DT_DAY  ]	= m_dt.date().day       () ;	
	dv[DT_HOUR ]	= m_dt.time().hour      () ;	
	dv[DT_MIN  ]	= m_dt.time().minute    () ;	
	dv[DT_SEC  ]	= m_dt.time().second    () ;	
	dv[DT_YDAY ]	= m_dt.date().dayOfYear () ;		
	dv[DT_WDAY ]	= m_dt.date().dayOfWeek () ;

	for (cchar *fmt = _fmt ; *fmt != 0 ; fmt += 1)
	{
		if (*fmt != '%')
		{
			res.append (*fmt) ;
			continue ;
		}

		fmt	+= 1 ;
		if (*fmt == 0) break ;

		if ((di = dtMap.find (*fmt)) != 0)
		{
			if (di->offset == DT_FORMAT)
			{
				res	+= format (di->fmt) ;
				continue ;
			}

			int	v = dv[di->offset] ;
			if (di->rem > 0) v = v %  di->rem ;
			if (di->rem < 0) v = v / -di->rem ;
			tmp.sprintf (di->fmt, v) ;
			res += tmp ;
			continue   ;
		}

		switch (fc = *fmt)
		{
			case 0   :
				return	res	;

			case 'a' :
			case 'A' :
				idx	= m_dt.date().dayOfWeek () ;
				tmp	= weekDayName (idx, fc == 'a') ;
				break	;

			case 'b' :
			case 'h' :
			case 'B' :
				idx	= m_dt.date().month     () ;
				tmp    	= monthName   (idx, fc != 'B') ;
				break	;

//			case 'D' :
//				tmp	= format ("%m/%d/%y") ;
//				break	;

			case 'n' :
				tmp	= "\n"	;
				break	;

			case 'p' :
				tmp	= m_dt.time().hour() >= 12 ? "PM" : "AM" ;
				break	;
				
			case 'P' :
				tmp	= m_dt.time().hour() >= 12 ? "pm" : "am" ;
				break	;

			case 't' :
				tmp	= "\t"	;
				break	;

			case 'x' :
				tmp	= formatDate (m_dt.date()) ;
				break	;

			case 'X' :
				tmp    += formatTime (m_dt.time()) ;
				break	;

			case '%' :
				tmp	= "%"	;
				break	;

			default	 :
				tmp	= ""	;
				break	;
		}

		res	+= tmp ;
	}

	return	res	;
}


/*  KBDateTime								*/
/*  defFormat	: Get date and time in default format 			*/
/*  iType	: KB::IType	: Required internal type		*/
/*  (returns)	: QString	: Formatted date and time		*/

QString	KBDateTime::defFormat
	(	KB::IType	iType
	)
	const
{
	/* Default format is ISO-8601, or parts thereof depending on	*/
	/* what is wanted.						*/

	/* If we do not have a valid decoded date/time then make do	*/
	/* with the raw string. Maybe the server can make more sense	*/
	/* of it.							*/
	if (!m_valid) return m_raw ;

	if (iType == KB::ITDate)
		return	QString().sprintf ("%04d-%02d-%02d",
					   m_dt.date().year  (),
					   m_dt.date().month (),
					   m_dt.date().day   ()) ;

	if (iType == KB::ITTime)
		return	QString().sprintf ("%02d:%02d:%02d",
					   m_dt.time().hour  (),
					   m_dt.time().minute(),
					   m_dt.time().second()) ;

	if (iType == KB::ITDateTime)
		return	QString().sprintf ("%04d-%02d-%02d %02d:%02d:%02d",
					   m_dt.date().year  (),
					   m_dt.date().month (),
					   m_dt.date().day   (),
					   m_dt.time().hour  (),
					   m_dt.time().minute(),
					   m_dt.time().second()) ;

	/* Gak! Should not get here!					*/
	KBError::EError
	(	TR("Unexpected request to KBDateTime::defFormat"),
		QString(TR("KBDateTime::defFormat(%1)")).arg((int)iType),
		__ERRLOCN
	)	;
	return	m_raw	;
}



void	KBDateTime::setDateOrder
	(	DateOrder	_dateOrder
	)
{
	dateOrder = _dateOrder	;
}

