/***************************************************************************
    file	         : kb_macroeditor.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<qwidget.h>
#include	<qguardedptr.h>
#include	<qlayout.h>
#include	<qsplitter.h>


#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_database.h"
#include	"kb_gui.h"
#include	"kb_dbinfo.h"
#include	"kb_dblink.h"
#include	"kb_string.h"
#include	"kb_options.h"
#include	"kb_node.h"
#include	"kb_docroot.h"
#include	"kb_dbdociter.h"

#ifndef 	_WIN32
#include	"kb_macroeditor.moc"
#else
#include 	"kb_macroeditor.h"
#endif


/*  KBInstructions							*/
/*  KBInstructions							*/
/*		: Constructor for instruction list			*/
/*  editor	: KBMacroEditor * : Parent editor			*/
/*  parent	: QWidget *	  : Parent widget			*/
/*  (returns)	: KBInstructions  :					*/

KBInstructions::KBInstructions
	(	KBMacroEditor	*editor,
		QWidget		*parent
	)
	:
	KBEditListView	(true, parent),
	m_editor	(editor)
{
	addColumn   (TR("Index"   ),  50) ;
	addColumn   (TR("Macro"   ), 180) ;
	addColumn   (TR("Comments"), 400) ;
	setEditType (1, KBEditListView::EdComboBox) ;
}

/*  KBInstructions							*/
/*  fillCombo	: Fill combo box					*/
/*  comboBox	: QComboBox &	  : Combo box in question		*/
/*  		: uint		  : Column number			*/
/*  curVal	: const QString & : Current text value			*/
/*  (returns)	: void		  :					*/

void	KBInstructions::fillCombo
	(	QComboBox	&comboBox,
		uint		,
		const QString	&curVal
	)
{
	uint	at	= 0 ;

	comboBox.clear	    () ;
	comboBox.insertItem (QString::null) ;

	const QStringList &macroNames = KBMacroDef::getMacroNames() ;
	for (uint idx = 0 ; idx < macroNames.count() ; idx += 1)
	{
		comboBox.insertItem (macroNames[idx]) ;
		if (macroNames[idx] == curVal) at = idx + 1;
	}

	comboBox.setCurrentItem (at) ;
}

void	KBInstructions::itemClicked
	(	QListViewItem	*item,
		const QPoint	&pos,
		int		col
	)
{
	if (item != 0)
	{
		KBEditListView::itemClicked (item, pos, col) ;
		m_editor->itemCurrent ((KBInstructionItem *)item, false, true) ;
	}
}

KBEditListViewItem
	*KBInstructions::newItem
	(	QListViewItem	*after,
		QString		label1
	)
{
	return	new KBInstructionItem (this, after, label1, 0) ;
}

/*  ------------------------------------------------------------------  */

/*  KBInstructionItem							*/
/*  KBInstructionItem							*/
/*		: Constructor for instruction item			*/
/*  parent	: KBInstructions *	: Parent list view		*/
/*  after	: QListViewItem *	: Insert after this item	*/
/*  index	: QString		: Index in instruction list	*/
/*  macroInstr	: KBMacroInstr *	: Extant macro instruction	*/
/*  (returns)	: KBInstructionItem	:				*/

KBInstructionItem::KBInstructionItem
	(	KBInstructions	*parent,
		QListViewItem	*after,
		QString		index,
		KBMacroInstr	*macroInstr
	)
	:
	KBEditListViewItem
	(	parent,
		after,
		index
	)
{
	/* The macro instruction pointer is set at startup when the	*/
	/* viewer has parsed an extant macro.				*/
	if (macroInstr != 0)
	{
		setText	(1, macroInstr->action ()) ;
		setText (2, macroInstr->comment()) ;

		m_args	= macroInstr->args() ;
	}
}

/*  KBInstructionitem							*/
/*  args	: Get current argument list				*/
/*  (returns)	: const QStringList &	: List				*/

const	QStringList
	&KBInstructionItem::args ()
{
	return	m_args	;
}

/*  KBInstructionitem							*/
/*  saveSettings: Save currently edited settings			*/
/*  def		: KBMacroDef *	 : Definition of macro			*/
/*  page	: KBWizardPage * : Wizard page holding settings		*/
/*  (returns)	: bool		 : Values changed			*/

bool	KBInstructionItem::saveSettings
	(	KBMacroDef	*def,
		KBWizardPage	*page
	)
{
	bool	changed	= m_args.count() != def->m_args.count() ;

	if (!changed)
		for (uint idx = 0 ; idx < def->m_args.count() ; idx += 1)
			if (page->ctrlValue (idx) != m_args[idx])
			{	changed	= true	;
				break	;
			}

	m_args.clear () ;

	for (uint idx = 0 ; idx < def->m_args.count() ; idx += 1)
		m_args.append (page->ctrlValue (idx)) ;

	return	changed	;
}


/*  ------------------------------------------------------------------  */

/*  KBMacroEditor							*/
/*  KBMacroEditor							*/
/*		: Constructor for macro editor object			*/
/*  parent	: QWidget *	  : Parent widget			*/
/*  dbInfo	: KBDBInfo *	  : Database information		*/
/*  svName	: const QString & : Server name				*/	
/*  (returns)	: KBMacroEditor	  :					*/

KBMacroEditor::KBMacroEditor
	(	QWidget		*parent,
		KBDBInfo	*dbInfo,
		const QString	&svName
	)
	:
	QSplitter	(Qt::Vertical, parent),
	m_dbInfo	(dbInfo),
	m_svName	(svName)
{
	m_instrList	= new KBInstructions  (this, this) ;
	QWidget	  *details    = new QWidget   (this) ;

	QSplitter   *splitDetails = new QSplitter   (Qt::Horizontal, details) ;
	QVBoxLayout *layout	  = new QVBoxLayout (details) ;
	layout->addWidget (splitDetails) ;
	layout->setMargin (KBOptions::getDlgMargin()) ;

	m_pageStack	= new QWidgetStack   (splitDetails) ;
	m_pageInfo	= new QTextView	     (splitDetails) ;
	m_nullPage	= new QLabel	     (m_pageStack)    ;

	m_currItem	= 0		;
	m_currPage	= 0		;
	m_currDef	= 0		;
	m_changed	= false		;

	connect
	(	m_instrList,
		SIGNAL(    changed(KBEditListViewItem *, uint)),
		SLOT  (slotChanged(KBEditListViewItem *, uint))
	)	;
	connect
	(	m_instrList,
		SIGNAL(    deleted(KBEditListViewItem *)),
		SLOT  (slotDeleted(KBEditListViewItem *))
	)	;

	m_pageInfo->setTextFormat (Qt::RichText) ;
	m_nullPage->setTextFormat (Qt::RichText) ;
}

/*  KBMacroEditor							*/
/*  ~KBMacroEditor							*/
/*		: Destructor for component viewer			*/
/*  (returns)	: void		:					*/

KBMacroEditor::~KBMacroEditor ()
{
}

/*  KBMacroEditor							*/
/*  startup	: Startup macro viewer object				*/
/*  doc		: QByteArray &	: Macro definition			*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: KB::ShowRC	: Startup success			*/

KB::ShowRC
	KBMacroEditor::startup
	(	QByteArray	&doc,
		KBError		&pError
	)
{
	uint		count	= 0 ;
	QListViewItem	*last	= 0 ;

	if (!doc.isNull())
	{
		QDomDocument	dom	;
		if (!dom.setContent (doc))
		{
			pError	= KBError
				  (	KBError::Error,
					TR("Cannot parse macro definition"),
					QString::null,
					__ERRLOCN
				  )	;
			return	KB::ShowRCError	;
		}

		QDomElement	root	= dom.documentElement() ; 
		KBMacroExec	macro	(0, QString::null)	;
		if (!macro.load (root, pError))
			return	KB::ShowRCError	;


		const QList<KBMacroInstr> &ilist = macro.instructions() ;
		LITER
		(	KBMacroInstr,
			ilist,
			instr,

			last	= new KBInstructionItem
				  (	m_instrList,
					last,
					QString("%1").arg(count),
					instr
				  )	;
			count  += 1	;
		)
	}

	new KBInstructionItem	(m_instrList, last, QString("%1").arg(count), 0) ;

	m_instrList->show () ;
	return	KB::ShowRCOK ;
}

/*  KBMacroEditor							*/
/*  startup	: Startup macro viewer object				*/
/*  macro	: KBMacroExec *	: Extant macro				*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: KB::ShowRC	: Startup success			*/

KB::ShowRC
	KBMacroEditor::startup
	(	KBMacroExec	*macro,
		KBError		&
	)
{
	QListViewItem		  *last	 = 0 ;
	uint			  count  = 0 ;

	if (macro != 0)
	{
		const QList<KBMacroInstr> &ilist = macro->instructions () ;

		LITER
		(	KBMacroInstr,
			ilist,
			instr,

			last	= new KBInstructionItem
				  (	m_instrList,
					last,
					QString("%1").arg(count),
					instr
				  )	;
			count  += 1	;
		)
	}

	new KBInstructionItem	(m_instrList, last, QString("%1").arg(count), 0) ;

	m_instrList->show () ;
	return	KB::ShowRCOK ;
}

/*  KBMacroEditor							*/
/*  setMacroPage: Show macro instruction values in page			*/
/*  page	: KBWizardPage *	: Page on which to show		*/
/*  item	: KBInstructionItem *	: Instruction in question	*/
/*  clear	: bool			: True to clear arguments	*/
/*  (returns)	: void			:				*/

void	KBMacroEditor::setMacroPage
	(	KBWizardPage		*page,
		KBInstructionItem	*item,
		bool			clear
	)
{
	const QStringList &args = item->args()	;

	for (uint idx = 0 ; idx < page->numCtrls() ; idx += 1)
		page->setCtrl
		(	idx,
			clear ? QString::null : args[idx]
		)	;

	m_pageInfo->setText (page->blurb()) ;
}

/*  KBMacroEditor							*/
/*  syncCurrentPage							*/
/*		: Sync current page back to macro			*/
/*  (returns)	: void			:				*/

void	KBMacroEditor::syncCurrentPage ()
{
	if (m_currItem != 0)
		if ((m_currDef != 0) && (m_currPage != 0))
			if (m_currItem->saveSettings (m_currDef, m_currPage))
			{	m_changed = true ;
				emit changed ()	 ;
			}
}

bool	KBMacroEditor::addSpecialArg
	(	const KBMacroArgDef	&arg,
		KBWizardPage		*page
	)
{
	fprintf
	(	stderr,
		"KBMacroEditor::addSpecialArg: [%s]\n",
		(cchar *)arg.m_type
	)	;

	/* Special cases are denoted by the value having the form	*/
	/* "tag:ident[:ident....]", so start by splitting and rejecting	*/
	/* anything that does not fit at all.				*/
	QStringList	bits	= QStringList::split (":", arg.m_type) ;

	if (bits.count() < 2)
		return	false	;


	/* First case will result in a combobox showing objects of a	*/
	/* specified type. The identifier denotes the type of object	*/
	/* that we are interested in.					*/
	if (bits[0] == "object")
	{
		static	cchar	*objectList[] =
		{
			"form",		"form",		"frm",
			"report",	"report",	"rep",
			"query",	"query",	"qry",
			0
		}	;

		/* Scan the object list for the specified type. If not	*/
		/* matched then silently return failure.		*/
		cchar	*type	= 0 ;
		cchar	*extn	= 0 ;

		for (cchar **ap = &objectList[0] ; *ap != 0 ; ap += 3)
			if (*ap == bits[1])
			{	type	= ap[1] ;
				extn	= ap[2] ;
				break	;
			}

		fprintf
		(	stderr,
			"KBMacroEditor::addSpecialArg: [object:%s] -> [%s][%s]\n",
			(cchar *)bits[1],
			type,
			extn
		)	;

		if (type == 0)
			return	false	;


		QStringList	forms	;
		forms.append	("")	;

		/* Any bits of string following the "object" tag and	*/
		/* the type are added as-is. This is used, for instance	*/
		/* so that form macros can add "[Invoker]" to the list.	*/
		for (uint idx = 2 ; idx < bits.count() ; idx += 1)
			forms.append (bits[idx]) ;

		/* Run a document iterator for the specified type. If	*/
		/* this fails or yeilds no objects then just add a	*/
		/* default edit field.					*/
		KBDBDocIter	docIter	;
		KBError		error	;

		if (!docIter.init (m_dbInfo, m_svName, type, extn, error, false))
			return	false	;

		QString		name	;
		QString		stamp	;

		while (docIter.getNextDoc (stamp, name))
			forms.append (stamp) ;

		if (forms.count() == 0)
			return	false	;

		/* Success, create a choice field with the derived list	*/
		/* of objects.						*/
		page->addChoiceCtrl
		(	arg.m_arg,
			arg.m_arg,
			forms,
			QString::null,
			true
		)	;

		return	true	;
	}

	/* Tag unrecognised, silently fail. The caller will simply show	*/
	/* an edit field.						*/
	return	false	;
}

/*  KBMacroEditor							*/
/*  itemCurrent	: User makes item current				*/
/*  item	: KBInstructionItem *	: Instruction in question	*/
/*  clear	: bool			: True to clear arguments	*/
/*  sync	: bool			: Sync current page first	*/
/*  (returns)	: void			:				*/

void	KBMacroEditor::itemCurrent
	(	KBInstructionItem	*item,
		bool			clear,
		bool			sync
	)
{
	if (sync)
		syncCurrentPage ()  ;

	m_currItem	= 0 ;
	m_currDef	= 0 ;
	m_currPage	= 0 ;

	if (item == 0)
	{
		m_nullPage ->setText     (QString::null) ;
		m_pageInfo ->setText 	 (QString::null) ;
		m_pageStack->raiseWidget (m_nullPage) ;
		return	;
	}

	/* If the name is empty then show the null page without any	*/
	/* text, pending the user actually selecting a macro.		*/
	QString		name	= item->text (1) ;
	if (name.isEmpty())
	{
		m_nullPage ->setText     (QString::null) ;
		m_pageInfo ->setText 	 (QString::null) ;
		m_pageStack->raiseWidget (m_nullPage) ;

		m_currItem = item ;
		return	;
	}

	/* Next see if there is already a page for the particular	*/
	/* macro instruction. If so then we will reuse it.		*/
	KBWizardPage *page = m_pages.find(name) ;
	if (page != 0)
	{
		setMacroPage (page, item, clear) ;
		m_pageStack->raiseWidget  (page) ;

		m_currItem = item ;
		m_currDef  = KBMacroDef::getMacroDef (name) ;
		m_currPage = page ;
		return	;
	}

	/* If not then look for the macro definition. We should find	*/
	/* this since the definitions should match all instructions	*/
	/* that the user can create. If not show the null page.		*/
	KBMacroDef *def	= KBMacroDef::getMacroDef (name) ;
	if (def == 0)
	{
		m_nullPage ->setText     (QString(TR("No definition for %1")).arg(name)) ;
		m_pageInfo ->setText 	 (QString::null) ;
		m_pageStack->raiseWidget (m_nullPage) ;

		m_currItem = item ;
		return	;
	}

	/* Create a new page. The fields are arbitrarily named with	*/
	/* legend.							*/
	page	= new KBWizardPage (0, m_pageStack, QString::null) ;
	for (uint idx = 0 ; idx < def->m_args.count() ; idx += 1)
	{
		const KBMacroArgDef &arg = def->m_args[idx] ;

		if (addSpecialArg (arg, page))
			continue;

		if (arg.m_type == "choice")
		{
			page->addChoiceCtrl
			(	arg.m_arg,
				arg.m_arg,
				arg.m_choices,
				QString::null
			)	;
			continue;
		}

		page->addTextCtrl
		(	arg.m_arg,
			arg.m_arg,
			QString::null
		)	;
	}

	page->setBlurb (def->m_blurb) ;
	page->addedAll () ;

	setMacroPage   (page, item, clear) ;
	m_pages.insert (name, page) ;
	m_pageStack->raiseWidget (page) ;

	if (page->sizeHint().width() > m_pageStack->width())
		m_pageStack->setMinimumWidth (page->sizeHint().width()) ;

	m_currItem = item ;
	m_currDef  = def  ;
	m_currPage = page ;
}


/*  KBMacroEditor							*/
/*  slotChanged	: User changes a value					*/
/*  item	: KBEditListViewItem *	: Item				*/
/*  col		: uint			: Column number			*/
/*  (returns)	: void			:				*/

void	KBMacroEditor::slotChanged
	(	KBEditListViewItem	*item,
		uint			col
	)
{
	if (col != 1)
	{	m_changed = true ;
		emit changed ()  ;
		return	;
	}

	KBInstructionItem *instr = (KBInstructionItem *)item ;

	itemCurrent (instr, true, false) ;
	m_changed = true ;
	emit changed ()  ;
}

/*  KBMacroEditor							*/
/*  slotDeleted	: User deletes an instruction				*/
/*  item	: KBEditListViewItem *	: Item				*/
/*  (returns)	: void			:				*/

void	KBMacroEditor::slotDeleted
	(	KBEditListViewItem	*item
	)
{
	if (item == m_currItem)
		itemCurrent ((KBInstructionItem *)m_instrList->currentItem(), false, false) ;

	m_changed = true ;
	emit changed ()  ;
}

/*  KBMacroEditor							*/
/*  getChanged	: Report whether there are unsaved changes		*/
/*  (returns)	: bool		: True if changed			*/

bool	KBMacroEditor::getChanged ()
{
	syncCurrentPage ();
	return	m_changed ;
}

/*  KBMacroEditor							*/
/*  macro	: Get definition as macro				*/
/*  pError	: KBError &	: Error return				*/
/*  node	: KBNode *	: Owning node				*/
/*  (returns)	: KBMacroExec *	: Macro or null on error		*/

KBMacroExec
	*KBMacroEditor::macro
	(	KBError		&pError,
		KBNode		*node
	)
{
	syncCurrentPage ();

	/* Create a macro executor and add the instructions to it. We	*/
	/* skip any which have no action, typically this will be the	*/
	/* "new" entry at the end of the list.				*/
	KBLocation  locn = node == 0 ?
				KBLocation() :
				node->getRoot()->getDocRoot()->getLocation() ;
	KBMacroExec *m 	 = new KBMacroExec
				(	locn.dbInfo,
					locn.docLocn				)	;

	for
	(	KBInstructionItem *item = (KBInstructionItem *)m_instrList->firstChild() ;
		item != 0 ;
		item  = (KBInstructionItem *)item->nextSibling()
	)
		if (!item->text(1).isEmpty())
			if (!m->append
				(	item->text(1),
					item->args( ),
					item->text(2),
					pError
				))
			{
				pError.DISPLAY() ;
				delete	m ;
				return	0 ;
			}

	return	m	;
}

/*  KBMacroEditor							*/
/*  def		: Get definition as XML					*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: QString	: XML definition			*/

QString	KBMacroEditor::def
	(	KBError		&pError
	)
{
	KBMacroExec  *m = macro (pError) ;
	if (m == 0) return QString::null ;

	QDomDocument cXML ("macro") ;
	QDomElement  root ;

	cXML.appendChild
	(	cXML.createProcessingInstruction
		(	"xml",
			"version=\"1.0\" encoding=\"UTF-8\""
	)	)	;

	cXML.appendChild (root = cXML.createElement ("RekallMacro")) ;
	m->save	(root)	;
	delete	m	;

	return	cXML.toString() ;
}


