/***************************************************************************
    file	         : kb_object.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>

#include	<qapp.h>
#include	<qpalette.h>

#ifndef 	_WIN32
#include	"kb_object.moc"
#else
#include 	"kb_object.h"
#endif

#include	"kb_slot.h"
#include	"kb_ctrl.h"
#include	"kb_display.h"
#include	"kb_layout.h"
#include	"kb_formblock.h"
#include	"kb_options.h"
#include	"kb_font.h"
#include	"kb_docroot.h"
#include	"kb_location.h"
#include	"kb_script.h"
#include	"kb_config.h"
#include	"kb_override.h"
#include	"kb_complink.h"
#include	"kb_quicktext.h"
#include	"kb_directprop.h"
#include	"kb_deleter.h"
#include	"kb_ascii.h"


#if		! __KB_RUNTIME
#include	"kb_popupmenu.h"
#include	"kb_formcopier.h"
#include	"kb_sizer.h"
#include	"kb_noobjdlg.h"
#include	"kb_component.h"
#include	"kb_compaccessdlg.h"
#endif

#include	"tk_fontdialog.h"
#include	"tk_colordialog.h"
#include	"tk_messagebox.h"


#define	KAF_CONFIG	(KAF_GRPOTHER|KAF_CUSTOM|KAF_HIDDEN)
#define	KAF_SLOTS	(KAF_GRPEVENT|KAF_CUSTOM|KAF_HIDDEN|KAF_SYNTHETIC)

/*  KBObject								*/
/*  KBObject	: Constructor for object class				*/
/*  parent	: KBNode *	: Parent node				*/
/*  element	: cchar *	: Element tag				*/
/*  aList	: const QDict<QString> &				*/
/*				: List of attributes			*/
/*  (returns)	: KBObject	:					*/

KBObject::KBObject
	(	KBNode			*parent,
		cchar			*element,
		const QDict<QString>	&aList
	)
	:
	KBNode	(parent,   element, aList),
	x	(this,   "x", 	    aList),
	y	(this,   "y", 	    aList),
	w	(this,   "w", 	    aList),
	h	(this,   "h", 	    aList),
	xmode	(this,	 "xmode",   aList), 
	ymode	(this,	 "ymode",   aList), 
	name	(this,   "name",    aList)
{
	control		= 0	;
	display		= 0	;
	block		= parent == 0 ? 0 : parent->isBlock () ;
	display		= parent == 0 ? 0 : parent->isObject()->getContainer() ;
	curPal		= 0	;
	curFont		= 0	;

#if	! __KB_RUNTIME
	sizer		= 0	;
	m_quickText	= 0	;
#endif

	if (parent == 0)
	{	xmode.setValue(FMFixed) ;
		ymode.setValue(FMFixed) ;
	}

	oldGeom	= QRect
		  (	x.getIntValue(),
			y.getIntValue(),
			w.getIntValue(),
			h.getIntValue()
		  )	;

	m_scriptObj	= 0 ;
	m_attrConfig	= new KBAttrStr (this, "configs", "", KAF_CONFIG) ;
	m_attrSlots	= new KBAttrStr	(this, "slots",   "", KAF_SLOTS ) ;
	m_slotList.setAutoDelete (true) ;
}

/*  KBObject								*/
/*  KBObject	: Constructor for new object				*/
/*  parent	: KBObject *	: Parent object				*/
/*  element	: cchar *	: Element tag				*/
/*  rect	: const QRect &	: Size and position			*/
/*  (returns)	: KBObject	:					*/

KBObject::KBObject
	(	KBObject	*parent,
		cchar		*element,
		const QRect	&rect
	)
	:
	KBNode	(parent,   element),
	x	(this,     "x",     rect.x     ()),
	y	(this,     "y",     rect.y     ()),
	w	(this,     "w",     rect.width ()),
	h	(this,     "h",     rect.height()),
	xmode	(this,	   "xmode", FMFixed), 
	ymode	(this,	   "ymode", FMFixed), 
	name	(this,     "name",  "")
{
	control		= 0	;
	block		= parent->isBlock      () ;
	display		= parent->getContainer () ;
	curPal		= 0	;
	curFont		= 0	;
#if	! __KB_RUNTIME
	sizer		= 0 	;
	m_quickText	= 0	;
#endif

	if (parent == 0)
	{	xmode.setValue(FMFixed) ;
		ymode.setValue(FMFixed) ;
	}

	oldGeom	= QRect
		  (	x.getIntValue(),
			y.getIntValue(),
			w.getIntValue(),
			h.getIntValue()
		  )	;

	m_scriptObj	= 0 ;
	m_attrConfig	= new KBAttrStr (this, "configs", "", KAF_CONFIG) ;
	m_attrSlots	= new KBAttrStr	(this, "slots",   "", KAF_SLOTS ) ;
	m_slotList.setAutoDelete (true) ;
}

/*  KBObject								*/
/*  KBObject	: Constructor for new form object			*/
/*  _parent	: KBNode *	: Parent				*/
/*  _object	: KBObject *	: Extant object				*/
/*  (returns)	: KBObject	:					*/

KBObject::KBObject
	(	KBNode		*_parent,
		KBObject	*_object
	)
	:
	KBNode	(_parent,  _object),
	x	(this,     "x",     _object),
	y	(this,     "y",     _object),
	w	(this,     "w",     _object),
	h	(this,     "h",     _object),
	xmode	(this,	   "xmode", _object), 
	ymode	(this,	   "ymode", _object), 
	name	(this,     "name",  _object)
{
	control		= 0	;
	display		= 0	;
	block		= _parent == 0 ? 0 : _parent->isBlock() ;
	curPal		= 0	;
	curFont		= 0	;
#if	! __KB_RUNTIME
	sizer		= 0	;
	m_quickText	= 0	;
#endif

	oldGeom	= QRect
		  (	x.getIntValue(),
			y.getIntValue(),
			w.getIntValue(),
			h.getIntValue()
		  )	;

	m_scriptObj	= 0 ;
	m_attrConfig	= new KBAttrStr (this, "configs", "", KAF_CONFIG) ;
	m_attrSlots	= new KBAttrStr	(this, "slots",   "", KAF_SLOTS ) ;
	m_slotList.setAutoDelete (true) ;

	LITER
	(	KBSlot,
		_object->m_slotList,
		slot,
		new KBSlot (this, slot)
	)
}

/*  KBObject								*/
/*  ~KBObject	: Destructor for form object class			*/
/*  (returns)	 :		:					*/

KBObject::~KBObject ()
{
	DELOBJ	(m_scriptObj ) ;
	DELOBJ	(m_attrConfig) ;
	DELOBJ	(m_attrSlots ) ;
	DELOBJ	(control) ;
	DELOBJ	(curPal ) ;
	DELOBJ	(curFont) ;

#if	! __KB_RUNTIME
	DELOBJ	(m_quickText ) ;
	DELOBJ	(sizer  ) ;
#endif
}

#if	! __KB_RUNTIME
/*  KBObject								*/
/*  setSizer	: Set sizer						*/
/*  _sizer	: KBSizer *	: Sizer to set				*/
/*  (returns)	: void		:					*/

void	KBObject::setSizer
	(	KBSizer		*_sizer
	)
{
	DELOBJ	(sizer)	 ;
	sizer	= _sizer ;
}

#endif

/*  KBObject								*/
/*  showAs	: Switch in or out of design mode			*/
/*  mode	: KB::ShowAs	: Design mode				*/
/*  (returns)	: void		:					*/

void	KBObject::showAs
	(	KB::ShowAs	mode
	)
{
#if	! __KB_RUNTIME
	DELOBJ	(m_quickText ) ;
#endif
	if (control)
	{
#if	! __KB_RUNTIME
		if ((mode == KB::ShowAsDesign) && (sizer == 0))
		{
			sizer	= new KBSizer  (this, display, control) ;
		}
		if ((mode == KB::ShowAsData  ) && (sizer != 0))
		{	getLayout()->dropSizer (sizer) ;
			DELOBJ	(sizer)	;
		}
#endif
		control->showAs (mode) ;
	}

#if	! __KB_RUNTIME
	if ((mode == KB::ShowAsDesign) && showingData  ())
	{
		DELOBJ	(m_scriptObj) ;

		x.setValue (oldGeom.x	  ()) ;
		y.setValue (oldGeom.y	  ()) ;
		w.setValue (oldGeom.width ()) ;
		h.setValue (oldGeom.height()) ;

		curSize	= QSize () ;
	}

	if ((mode == KB::ShowAsData  ) && showingDesign())
	{
		DELOBJ	(m_scriptObj) ;

		oldGeom	= QRect
			  (	x.getIntValue(),
				y.getIntValue(),
				w.getIntValue(),
				h.getIntValue()
			  )	;

		curSize	= QSize () ;
	}

	if (mode == KB::ShowAsDesign)
		LITER
		(	KBAttr,
			attribs,
			attrib,
			if (attrib->isEvent() != 0)
				attrib->isEvent()->clearEmitter() ;
		)
#endif

	KBNode::showAs (mode) ;
}

/*  KBObject								*/
/*  setChanged	: Set changed state after design operation		*/
/*  (returns)	: void		:					*/

void	KBObject::setChanged ()
{
#if	! __KB_RUNTIME
	sizer->setBlobs () ;
#endif
	getLayout()->setChanged () ;
}

/*  KBObject								*/
/*  makeNewPopup: Make new object popup menu				*/
/*  cancel	: bool		: Show cancel item			*/
/*  _rect	: QRect		: New object area			*/
/*  (returns)	: KBPopupMenu *	: Menu					*/

KBPopupMenu *KBObject::makeNewPopup
	(	bool	,
		QRect
	)
{
	return	0	;
}

/*  KBObject								*/
/*  updateProps	: Update properties					*/
/*  (returns)	: void		:					*/

void	KBObject::updateProps ()
{
	/* This method handles the basic common properties. Subclasses	*/
	/* may override it.						*/
	setPalette   () ;
	setFont      () ;

	if (control != 0)
	{
		control->setGeometry (geometry()) ;
		redraw	() ;
	}

#if	! __KB_RUNTIME
	if (showingDesign())
	{
		if (control != 0)
			control->showName    () ;

		setChanged   () ;
		getLayout    ()->addSizer (getSizer(), false) ;
	}
#endif
}

#if	! __KB_RUNTIME

/*  KBObject								*/
/*  addNewObject: Add a new object					*/
/*  x		: int		: X-position				*/
/*  y		: int		: Y-position				*/
/*  w		: int		: Width					*/
/*  h		: int		: Height				*/
/*  (returns)	: bool		: Object (probably) added		*/

bool	KBObject::addNewObject
	(	int	x,
		int	y,
		int	w,
		int	h
	)
{
	extern	void	snapRect (int &, int &, int &, int &) ;

	/* Apply a minimum size contstraint, otherwise a left press and	*/
	/* release with no drag will try to create a zero-by-zero	*/
	/* object.							*/
	if ((w >= 2) && (h >= 2))
	{
		if (KBOptions::snappingOn()) snapRect (x, y, w, h) ;
		KBPopupMenu *pop = makeNewPopup (true, QRect(x, y, w, h)) ;
		pop->exec (QCursor::pos()) ;
		delete	pop  ;
		return	true ;
	}

	return	false	;
}

/*  KBObject								*/
/*  selectOverlap: Select any overlaped children			*/
/*  x, y, w, h	 : int, ...	: Position and size of area		*/
/*  (returns)	 : void		:					*/

void	KBObject::selectOverlap
	(	int	x,
		int	y,
		int	w,
		int	h
	)
{
	bool	multi	= false ;
	KBSizer	*sizer	;

	CITER
	(	Object,
		obj,
		if ((sizer = obj->overlaps (x, y, w, h)) != 0)
		{	getLayout()->addSizer (sizer, multi) ;
			multi = true ;
		}
	)
}

/*  KBObject								*/
/*  checkOverlap : Check for any overlaped children			*/
/*  x, y, w, h	 : int, ...	: Position and size of area		*/
/*  (returns)	 : void		: True if any overlapped		*/

bool	KBObject::checkOverlap
	(	int	x,
		int	y,
		int	w,
		int	h
	)
{
	CITER
	(	Object,
		obj,
		if (obj->overlaps (x, y, w, h) != 0) return true ;
	)

	return	false	;
}

/*  KBObject								*/
/*  overlaps	: Return sizer if object overlaps sepecified area	*/
/*  _x		: int		: X position				*/
/*  _y		: int		: Y position				*/
/*  _w		: int		: Width					*/
/*  _h		: int		: Height				*/
/*  (returns)	: KBSizer *	: Sizer or NULL				*/

KBSizer	*KBObject::overlaps
	(	int	_x,
		int	_y,
		int	_w,
		int	_h
	)
{
	QRect	rect	= geometry () ;

	if (_x >= rect.right ()) return 0 ;
	if (_y >= rect.bottom()) return 0 ;
	if (_x + _w < rect.x ()) return 0 ;
	if (_y + _h < rect.y ()) return 0 ;

//	fprintf	(stderr, "Overlap: %s::%s (%d,%d,%d,%d) over (%d,%d,%d,%d)\n",
//			 (cchar *)getElement(),
//			 (cchar *)name.getValue(),
//			 rect.x     (),
//			 rect.y     (),
//			 rect.width (),
//			 rect.height(), _x,_y,_w,_h) ;

	return	sizer	;
}

#endif

/*  KBObject								*/
/*  minGeometry	: Calculate minimum size of object			*/
/*  (returns)	: QSize		: Minimum size				*/

QSize	KBObject::minGeometry ()
{
	uint	_x	;
	uint	_y	;
	QSize	minSize	= getMinSize () ;

	switch ((FloatMode)xmode.getIntValue())
	{
		case FMFloat	:
			_x = x.getIntValue() ;
			break	;

		case FMStretch	:
			_x = x.getIntValue() + minSize.width () ;
			break	;

		default	:
			_x = x.getIntValue() + w.getIntValue () ;
			break	;
	}

	switch ((FloatMode)ymode.getIntValue())
	{
		case FMFloat	:
			_y = y.getIntValue() ;
			break	;

		case FMStretch	:
			_y = y.getIntValue() + minSize.height() ;
			break	;

		default	:
			_y = y.getIntValue() + h.getIntValue () ;
			break	;
	}

	return	QSize(_x, _y) ;
}

/*  KBObject								*/
/*  getMinSize	: Get minimum size allowable				*/
/*  (returns)	: QSize		: Size					*/

QSize	KBObject::getMinSize()
{
	int	w	= 0 ;
	int	h	= 0 ;

	CITER
	(	Object,
		obj,

		QSize	os = obj->minGeometry() ;

		if (os.width () > w) w = os.width () ;
		if (os.height() > h) h = os.height() ;
	)		

	return	QSize (w, h) ;
}

/*  KBObject								*/
/*  getBaseSize	: Get base size of object				*/
/*  (returns)	: QSize		: Base size				*/

QSize	KBObject::getBaseSize ()
{
#if	! __KB_RUNTIME
	if (showingData())
		return	QSize (oldGeom.width(), oldGeom.height()) ;
#endif

	return	QSize	(x.getIntValue(), y.getIntValue()) ;
}

/*  KBObject								*/
/*  setPalette	: Set palette						*/
/*  (returns)	: void		:					*/

void	KBObject::setPalette ()
{
	DELOBJ	(curPal) ;
	if (control != 0)
		control->setPalette (getPalette (false)) ;

	CITER
	(	Object,
		o,
		o->setPalette()
	)
}

/*  KBObject								*/
/*  setFont	: Set font						*/
/*  (returns)	: void		:					*/

void	KBObject::setFont ()
{
	DELOBJ	(curFont) ;
	if (control != 0)
		control->setFont (getFont (false)) ;
}

/*  KBObject								*/
/*  designPopup	: Display design menu					*/
/*  e		: QMouseEvent *	: Triggering mouse event		*/
/*  drow	: uint		: Display row				*/
/*  (returns)	: void		:					*/

void	KBObject::designPopup
	(	QMouseEvent	*,
		uint
	)
{
#if	! __KB_RUNTIME
	QPopupMenu pop ;
	pop.insertItem (TR("Cancel")) ;
	pop.insertItem (TR("C&ut"),		this, SLOT(cutObj         ())) ;
	pop.insertItem (TR("&Copy"),		this, SLOT(copyObj        ())) ;
	pop.insertItem (TR("&Delete"),		this, SLOT(deleteObj      ())) ;
	pop.insertItem (TR("Save as component"),this, SLOT(saveAsComponent())) ;
	pop.insertItem (TR("&Properties"),	this, SLOT(propertyDlg    ())) ;

	/* The structure below lists attribute shortcuts. Scan through	*/
	/* looking for attributes that appear in the object, and	*/
	/* create a submenu for each one found. The code is passed to	*/
	/* the menu item so that it will be passed to the handler slot.	*/
	struct	DirectMap
	{
		const char	*attr	;
		const char	*menu	;
		int		code	;
	}	;

	static	DirectMap	dmap[]	=
	{
		{	"font",		"&Font",	DP_FONT		},
		{	"fgcolor",	"&Foreground",	DP_FOREGROUND	},
		{	"bgcolor",	"&Background",	DP_BACKGROUND	},
		{	"text",		"&Text",	DP_TEXT		},
		{	0,		0,		0		}
	}	;

	bool	direct = false	;
	for (DirectMap *d = &dmap[0] ; d->attr != 0 ; d += 1)
		if (getAttr(d->attr) != 0)
		{	if (!direct)
			{	pop.insertSeparator() ;
				direct = true ;
			}
			pop.insertItem (TR(d->menu), this, SLOT(setPropDirect(int)), 0, d->code) ;
		}

	pop.exec       (QCursor::pos()) ;
#endif
}

/*  KBObject								*/
/*  setPropDirect: Handler for direct property setting			*/
/*  code	 : int		: Property code				*/
/*  (returns)	 : void		:					*/

void	KBObject::setPropDirect
	(	int		code
	)
{
#if	! __KB_RUNTIME
	KBAttr	*attr	= 0	;
	QString	value	;

	/* First step is to get the attribute. We should find it but	*/
	/* check for the sake if paranoia.				*/
	switch (code)
	{
		case DP_FONT		: attr	= getAttr("font"   ) ; break ;
		case DP_FOREGROUND	: attr	= getAttr("fgcolor") ; break ;
		case DP_BACKGROUND	: attr	= getAttr("bgcolor") ; break ;
		case DP_TEXT		: attr	= getAttr("text"   ) ; break ;
		default			: break	;
	}

	if (attr == 0)
	{
		fprintf
		(	stderr,
			"KBObject::setPropDirect: unknown for code %d\n",
			code
		)	;
		return	;
	}

	/* Got an attribute to pick up the current value and run the	*/
	/* appropriate dialog. If the user OK's then set the value back	*/
	/* and call the property update method.				*/
	value	= attr->getValue () ;

	switch (code)
	{
		case DP_FONT		:
			{
			TKFontDialog  fDialog (0, TR("Font"), false, true) ;
			fDialog.setFont (KBFont::specToFont (value)) ;
			if (!fDialog.exec()) return ;
			value	= KBFont::fontToSpec (fDialog.font()) ;
			}
			break	;

		case DP_FOREGROUND	:
		case DP_BACKGROUND	:
			{
			TKColorDialog cDialog (0, TR("Colour"), true) ;
			cDialog.setColor (QColor (value.toInt())) ;
			if (!cDialog.exec ()) return ;
			value.sprintf ("0x%06x", cDialog.color().rgb() & 0x00ffffff) ;
			}
			break	;

		case DP_TEXT	:
			{
			/* This is special. Create a quick text edit	*/
			/* and then leave it to run. The completion	*/
			/* event will be picked up in the event handler	*/
			/* below.					*/
			DELOBJ	(m_quickText) ;
			m_quickText = new KBQuickText
					  (	control->topWidget(),
						QRect
						(	0,
							0,
							control->topWidget()->width (),
							control->topWidget()->height()
						),
						value,
						this,
						(QEvent::Type)(QEvent::User + 1000)
					  )	;
			m_quickText->show     () ;
			m_quickText->setFocus () ;
			return	   ;
			}

		default	:
			return	;
	}

	attr->setValue	(value) ;
	updateProps	()	;
#endif
}

/*  KBObject								*/
/*  event	: Local event handler					*/
/*  e		: QEvent *	: Event in question			*/
/*  (returns)	: bool		: Event consumed			*/

bool	KBObject::event
	(	QEvent		*e
	)
{
#if	! __KB_RUNTIME
	/* Check for the quick text completion event. If it is *and*	*/
	/* there is a quick text widget (paranoia check) *and* there	*/
	/* is a text property (more paranoia) then update it.		*/
	if (e->type() == (QEvent::Type)(QEvent::User + 1000))
	{
		if (m_quickText != 0)
		{
			KBAttr	*attr	= getAttr("text") ;
			if (attr != 0)
			{	attr->setValue	(m_quickText->text()) ;
				updateProps	() ;
			}

			DELOBJ	(m_quickText) ;
		}

		return	true	;
	}
#endif
	return	KBNode::event (e) ;
}

/*  KBObject								*/
/*  contextMenu	: Display context menu					*/
/*  e		: QMouseEvent *	: Triggering mouse event		*/
/*  drow	: uint		: Display row				*/
/*  (returns)	: void		:					*/

void	KBObject::contextMenu
	(	QMouseEvent	*,
		uint
	)
{
}

/*  KBObject								*/
/*  moveFocusOK	: Check if OK to set focus				*/
/*  drow	: uint		: Display row				*/
/*  (returns)	: bool		: OK					*/

bool	KBObject::moveFocusOK
	(	uint	
	)
{
	return	showingDesign()	     ? true :
		getFormBlock () == 0 ? true :
				       getFormBlock()->focusOutOK (false) ;
}

void	KBObject::setFocus ()
{
	if (control != 0) control->setFocus () ;
}

/*  KBObject								*/
/*  focusInEvent: Handle focus in event					*/
/*		: uint		: Display row				*/
/*		: Reason	: Reason focus arrives			*/
/*  (returns)	: void		:					*/

void	KBObject::focusInEvent
	(	uint		,
		QFocusEvent::Reason
	)
{
	/* Nothing to do here. Derived classes will implement this as	*/
	/* neccessary.							*/
}

/*  KBObject								*/
/*  convGeometry: Convert geometry to/from absolute			*/
/*  geom	: const QRect &	: Geometry to convert			*/
/*  xm		: FloatMode	: X float mode				*/
/*  ym		: FloatMode	: Y float mode				*/
/*  (returns)	: QRect		: Converted geometry			*/

QRect	KBObject::convGeometry
	(	const	QRect	&geom,
		FloatMode	xm,
		FloatMode	ym
	)
{
	uint	x	= geom.x     () ;
	uint	y	= geom.y     () ;
	uint	w	= geom.width () ;
	uint	h	= geom.height() ;

	QRect	pr	= getParent() == 0 ?
				QRect() :
				getParent()->isObject()->geometry() ;

	switch (xm)
	{
		case FMFloat	: x = pr.width () - x	  ; break ;
		case FMStretch	: w = pr.width () - x - w ; break ;
		default		:			  ; break ;
	}
	switch (ym)
	{
		case FMFloat	: y = pr.height() - y	  ; break ;
		case FMStretch	: h = pr.height() - y - h ; break ;
		default		:			  ; break ;
	}

	return	QRect (x, y, w, h) ;
}

/*  KBObject								*/
/*  geometry	: Get geometry of this object				*/
/*  (returns)	: QRect		: Extent				*/

QRect	KBObject::geometry ()
{
	return	convGeometry
		(
			QRect
			(	x.getIntValue(),
				y.getIntValue(),
				w.getIntValue(),
				h.getIntValue()
			),
			(FloatMode)xmode.getIntValue(),
			(FloatMode)ymode.getIntValue()
		)	;
}

/*  KBObject								*/
/*  geometry	: Get geometry of this object with offset		*/
/*  offset	: const QPoint & : Offset to apply			*/
/*  (returns)	: QRect		 : Extent				*/

QRect	KBObject::geometry
	(	const QPoint	&offset
	)
{
	return	convGeometry
		(
			QRect
			(	x.getIntValue() + offset.x(),
				y.getIntValue() + offset.y(),
				w.getIntValue(),
				h.getIntValue()
			),
			(FloatMode)xmode.getIntValue(),
			(FloatMode)ymode.getIntValue()
		)	;
}

/*  KBObject								*/
/*  ctrlGone	: Note death of control					*/
/*  ctrl	: KBControl *	: The control				*/
/*  (returns)	: void		:					*/

void	KBObject::ctrlGone
	(	KBControl	*ctrl
	)
{
	if (ctrl == control) control = 0 ;
}

/*  KBObject								*/
/*  move	: Move the object in its parent				*/
/*  _x		: int		: New X position			*/
/*  _y		: int		: New Y position			*/
/*  (returns)	: void		:					*/

void	KBObject::move
	(	int	_x,
		int	_y
	)
{
	QRect	r = convGeometry
		    (	QRect(QPoint(_x, _y), geometry().size()),
		    	(FloatMode)xmode.getIntValue(),
		    	(FloatMode)ymode.getIntValue()
		    ) ;

	x.setValue (r.x	    ()) ;
	y.setValue (r.y	    ()) ;
	w.setValue (r.width ()) ;
	h.setValue (r.height()) ;

	if (control) control->setGeometry (geometry()) ;
#if	! __KB_RUNTIME
	if (sizer  ) sizer  ->setBlobs	  () ;
#endif
}

/*  KBObject								*/
/*  resize	: Resize the object in its parent			*/
/*  _w		: int		: New width				*/
/*  _h		: int		: New height				*/
/*  (returns)	: bool		: Size changed				*/

bool	KBObject::resize
	(	int	_w,
		int	_h
	)
{
	if ((curSize.width() != _w) || (curSize.height() != _h))
	{

		QRect	r = convGeometry	
			    (	QRect(geometry().topLeft(), QSize(_w, _h)),
			    	(FloatMode)xmode.getIntValue(),
		    		(FloatMode)ymode.getIntValue()
			    )	;

//		fprintf
//		(	stderr, "[%s] [%d,%d] -> [%d,%d]\n",
//				(cchar *)name.getValue(),
//				w.getIntValue(),
//				h.getIntValue(),
//				r.width	     (),
//				r.height     ()
//		)	;

		x.setValue (r.x	    ()) ;
		y.setValue (r.y	    ()) ;
		w.setValue (r.width ()) ;
		h.setValue (r.height()) ;

		if (control) control->setGeometry (geometry()) ;
#if	! __KB_RUNTIME
		if (sizer  ) sizer  ->setBlobs	  () ;
#endif
		curSize	= QSize	(_w, _h) ;
		return	 true	;
	}

	return	false	;
}

/*  KBObject								*/
/*  redraw	: Redraw control after design change			*/
/*  (returns)	: void		:					*/

void	KBObject::redraw ()
{
	if (control) control->redraw () ;
}

/*  KBObject								*/
/*  propertyDlg	: Default property dialog				*/
/*  (returns)	: void		:					*/

bool	KBObject::propertyDlg ()
{
	return	propertyDlg (0) ;
}

/*  KBObject								*/
/*  propertyDlg	: Default property dialog				*/
/*  iniAttr	: cchar *	: Initial attribute to display		*/
/*  (returns)	: void		:					*/

bool	KBObject::propertyDlg
	(	cchar	*iniAttr
	)
{
#if	! __KB_RUNTIME
	if (KBNode::propertyDlg (TR("Properties"), iniAttr))
	{
		getLayout()->addSizer (getSizer(), false) ;
		return	true	;
	}
#endif
	return	false	;
}

/*  KBObject								*/
/*  deleteObj	: Delete the object					*/
/*  (returns)	: void		:					*/

void	KBObject::deleteObj ()
{
#if	! __KB_RUNTIME
	if (getParent() == 0)
	{
		TKMessageBox::sorry
		(	0,
			QString(TR("Cannot delete the top-most object")),
			TR("Error deleting object")
		)	;
		return	;
	}

	getLayout()->setChanged  () ;
	KBDeleter::addObject (this) ;
//	delete	   this	   ;
#endif
}

/*  KBObject								*/
/*  saveAsComponent							*/
/*		: Save object as component				*/
/*  (returns)	: void		:					*/

void	KBObject::saveAsComponent ()
{
#if	! __KB_RUNTIME

	const KBLocation  &locn1   = getRoot()->getDocRoot()->getLocation() ;
	QString		  svName   = locn1.docLocn ;
	QString		  objName  ;
	QString		  comment  ;

	/* Run the component save-as dialog, which will return the	*/
	/* desired object name, server, and optionally a comment.	*/
	bool		   toFile ;
	KBComponentSaveDlg csDlg
			   (	objName,
				svName,
				comment,
				locn1.dbInfo,
				&toFile
			   )	;

	if (!csDlg.exec()) return ;


	/* We make a copy of the object, then adjust its position so	*/
	/* that the top-left one goes to (20,20). Note that this is	*/
	/* done to function correctly for stretching/floating objects.	*/
	KBObject *copy	= (KBObject *)replicate(0) ;
	QRect	 r	= copy->geometry() ;
	copy->KBObject::move   (20, 20) ;
	copy->KBObject::resize (r.width (), r.height()) ;

	/* OK, got a name and stuff, so stitch together the text with	*/
	/* the top level component settings ....			*/
	extern	QString	kbXMLEncoding   () ;

	QString	 text	= QString
			  (	"<?xml version=\"1.0\" encoding=\"%1\"?>\n"
				"<!DOCTYPE KBaseCompenent SYSTEM \"kbasecomponent.dtd\">\n"
			  	"<KBComponent w=\"%2\" h=\"%3\" type=\"%4\" notes=\"%5\">\n" 
			  )
			  .arg	(kbXMLEncoding())
			  .arg	(r.width () + 40)
			  .arg	(r.height() + 40)
			  .arg	((uint)objType())
			  .arg	(comment) ;


	copy->printNode (text, 2)  ;
	text += "</KBComponent>\n" ;

	fprintf	  (stderr, "Component:[[[\n%s\n]]]\n", (cchar *)text) ;

	/*  .... and save the component where requested. The special	*/
	/* case is if the used specified save-to-file.			*/
	if (toFile)
	{
		saveComponentToFile (objName, text) ;
		return	;
	}

	KBLocation	locn2
			(	locn1.dbInfo,
				"component",
				svName,
				objName
			)	;
	KBError		error	;

	if (!locn2.save (QString::null, QString::null, text, error))
		error.DISPLAY() ;
#endif
}

/*  KBObject								*/
/*  copyObj	: Copy the object					*/
/*  (returns)	: void		:					*/

void	KBObject::copyObj ()
{
#if	! __KB_RUNTIME
	KBFormCopier::self()->clearCopy () ;
	KBFormCopier::self()->addToCopy (replicate(0), objType()) ;
#endif
}

/*  KBObject								*/
/*  cutObj	: Cut the object					*/
/*  (returns)	: void		:					*/

void	KBObject::cutObj ()
{
#if	! __KB_RUNTIME
	if (getParent() == 0)
	{
		TKMessageBox::sorry
		(	0,
			QString(TR("Cannot cut the top-most object")),
			TR("Error cutting object")
		)	;
		return	;
	}

	copyObj   () ;
	deleteObj () ;

	getLayout ()->setChanged () ;
#endif
}

#if	! __KB_RUNTIME

/*  KBObjects								*/
/*  insertObjects							*/
/*		: Insert child objects					*/
/*  nodes	: QList<KBNode> & : List of objects			*/
/*  inDisplay	: KBDisplay *	  : Owning display for objects		*/
/*  ox		: int		  : X-offset to apply to positions	*/
/*  oy		: int		  : Y-offset to apply to positions	*/
/*  numRows	: uint		  : Number of rows of controls		*/
/*  dx		: int		  : Control X-step			*/
/*  dy		: int		  : Control Y-step			*/
/*  (returns)	: void		  :					*/

void	KBObject::insertObjects
	(	const QList<KBNode>	&nodes,
		KBDisplay		*inDisplay,
		int			ox,
		int			oy,
		uint			numRows,
		int			dx,
		int			dy
	)
{
	KBNode		*nNode	;
	KBObject	*nObj	;

	LITER
	(	KBNode,
		nodes,
		oNode,

		/* Replicate the node into this block. If it is not	*/
		/* also an object then we don't need to move it.	*/
		if ((nNode = oNode->replicate (this)) == 0)
			continue ;

		if ((nObj  = nNode->isObject ()) == 0)
			continue ;

		/* Set the position and size of the object and adjust	*/
		/* by the offsets. If snapping is enabled then snap the	*/
		/* result.						*/
		QRect	r  = nObj->geometry() ;
		int	x  = r.x     () + ox ;
		int	y  = r.y     () + oy ;
		int	w  = r.width () ;
		int	h  = r.height() ;

		extern	void	snapRect (int &, int &, int &, int &) ;

		if (KBOptions::snappingOn()) snapRect (x, y, w, h) ;

		/* Update the object position and then build its GUI.	*/
		/* Setting the design mode should always set design 	*/
		/* mode on, else we wouldn't be here.			*/
		nObj->buildDisplay (inDisplay) ;
		nObj->setGeometry  (QRect(x, y, w, h)) ;
		nObj->buildCtrls   (numRows, dx, dy) ;
		nObj->showAs   	   (isShowing()) ;

		if (nObj->isBlock() != 0)
			nObj->isBlock()->getContainer()->show() ;
	)

	/* Mark the form as changed, except in the case there were no	*/
	/* inserts (in which case we should never have been called).	*/
	if (nodes.count() > 0) getLayout()->setChanged () ;
}

/*  KBObject								*/
/*  insertObjects: Insert copied or cut objects				*/
/*  inDisplay	 : KBDisplay *	  : Associated display			*/
/*  (returns)	 : void		:					*/

void	KBObject::insertObjects
	(	KBDisplay	*inDisplay
	)
{
	/* This is the simple case. Paste in any copied objects at	*/
	/* the same positions as they were wherever they came from.	*/
	KBBlock	*block	= isBlock() != 0 ? isBlock() : getBlock () ;
	int	dx	= block->getAttrVal("dx").toInt() ;
	int	dy	= block->getAttrVal("dy").toInt() ;

	insertObjects
	(	KBFormCopier::self()->getCopied (objType()),
		inDisplay,
		0,
		0,
		block->rowsInBlock(),
		dx,
		dy
	)	;
//	KBError::EWarning
//	(	TR("Copied or cut objects can only be pasted into a block"),
//		QString::null,
//		__ERRLOCN
//	)	;
}

/*  KBObject								*/
/*  insertHere	: Insert copied or cut objects at specified position	*/
/*  nodes	: QList<KBNode> & : List of nodes to paste		*/
/*  inDisplay	: KBDisplay *	  : Associated display			*/
/*  insertAt	: QPoint	  : Position				*/
/*  (returns)	: void		  :					*/

void	KBObject::insertHere
	(	const QList<KBNode>	&nodes,
		KBDisplay		*inDisplay,
		QPoint			insertAt
	)
{
	/* Start by scanning for all objects to be inserted, and get	*/
	/* the minimum X- and Y- coordinates.				*/
	int	minx	;
	int	miny	;
	minPosition	(nodes, minx, miny) ;

//	LITER
//	(	KBNode,
//		nodes,
//		oNode,
//
//		if ((oObj = oNode->isObject()) == 0)
//			continue ;
//
//		QRect	r = oObj->geometry() ;
//		int 	x = r.x() ;
//		int	y = r.y() ;
//
//		if (x < minx) minx = x ;
//		if (y < miny) miny = y ;
//	)

	/* Insert the objects such that they appear at the position	*/
	/* noted when the popup-menu was displayed. Actually, the 	*/
	/* the effect is that this is position is the top-left corner	*/
	/* of a bounding box around the objects.			*/
	KBBlock	*block	= isBlock() != 0 ? isBlock() : getBlock () ;
	int	dx	= block->getAttrVal("dx").toInt() ;
	int	dy	= block->getAttrVal("dy").toInt() ;

	insertObjects
	(	nodes,
		inDisplay,
		insertAt.x() - minx,
		insertAt.y() - miny,
		block->rowsInBlock(),
		dx,
		dy
	)	;
//	KBError::EWarning
//	(	TR("Copied or cut objects can only be pasted into a block"),
//		QString::null,
//		__ERRLOCN
//	)	;
}

/*  KBObject								*/
/*  insertComponent							*/
/*		: Insert component object at specified position		*/
/*  inDisplay	: KBDisplay *	: Associated display			*/
/*  insertAt	: QPoint	: Position				*/
/*  newRect	: QRect		: New object size			*/
/*  paste	: bool		: Paste component, else link		*/
/*  (returns)	: void		:					*/

void	KBObject::insertComponent
	(	KBDisplay	*inDisplay,
		QPoint	 	insertAt,
		QRect		newRect,
		bool		paste
	)
{
	/* Start by running the document chooser so that the user can	*/
	/* select a set of components.					*/
	const KBLocation  &location = getRoot()->getDocRoot()->getLocation() ;

	fprintf
	(	stderr,
		"KBObject::pasteComponent: f=[%p] r=[%p]\n",
		(void *)getRoot()->isForm  (),
		(void *)getRoot()->isReport()
	)	;

	KBComponentLoadDlg clDlg
			   (	location.dbInfo,
			 	location.docLocn,
			   	getRoot()->getAttrVal ("language"),
				newRect.size(),
				paste,
				objType()
			   )	;

	if (!clDlg.exec()) return ;


	/* First case is pasting. Retrieve the component from the 	*/
	/* dialog, then insert all objects inside the component other	*/
	/* than the null query.	The component here can the be dropped.	*/
	if (paste)
	{
		KBError	    error ;
		KBComponent *component = clDlg.component (error) ;
		if (component == 0)
		{	error.DISPLAY () ;
			return	;
		}

		const QList<KBNode> &cList = component->getChildren() ;
		QList<KBNode>	     iList ;

		LITER
		(	KBNode,
			cList,
			child,
			if (child->isQryNull() == 0) iList.append (child) ;
		)

		insertHere (iList, inDisplay, insertAt) ;
		delete	component ;
		return	;
	}

	/* Second case is linking. Start by createing a new component	*/
	/* link object, which is initialised with the appropriate	*/
	/* geometry, plus the selected server and document.		*/
	KBAttrDict	aDict	;
	aDict.addValue	("x",	      insertAt.x()) ;
	aDict.addValue	("y",	      insertAt.y()) ;
	aDict.addValue	("w",	      newRect.width ()) ;
	aDict.addValue	("h",	      newRect.height()) ;
	aDict.addValue	("server",    clDlg.server  ()) ;
	aDict.addValue	("component", clDlg.document()) ;	

	bool		ok	;	
	KBCompLink	*clink	= new KBCompLink (this, aDict, &ok) ;
	if (!ok) return	;

//	/* If control returns OK from the constructor, then the linked	*/
//	/* component will have been loaded. At this point, ensure that	*/
//	/* the overrides match the configutation objects in the		*/
//	/* the component itself.					*/
//	clink->checkOverrides ();

	/* If control returns OK from the constructor, then the linked	*/
	/* component will have been loaded. At this point, gather up	*/
	/* the configuration objects and store copies immediately under	*/
	/* the component link. These will be used to update the linked	*/
	/* objects at run time.						*/
	QList<KBConfig>	configSet ;
	clDlg.getAllConfigs (clink, configSet, false, true) ;

	LITER
	(	KBConfig,
		configSet,
		c,

		new KBOverride
			(	clink,
				c->ident  (),
				c->path   (),
				c->attrib (),
				c->value  (),
				c->changed()
			)	;
	)

	/* We now go on to build the display so that the user can see	*/
	/* the linked objects.						*/
	KBBlock	*block	= isBlock() != 0 ? isBlock() : getBlock () ;
	int	dx	= block->getAttrVal("dx").toInt() ;
	int	dy	= block->getAttrVal("dy").toInt() ;

	clink->buildDisplay (inDisplay) ;
	clink->buildCtrls   (block->rowsInBlock(), dx, dy) ;
	clink->showAs       (KB::ShowAsDesign) ;
}

#endif

/*  KBObject								*/
/*  findAllConfigs							*/
/*		: Locate all configuration objects in document		*/
/*  paramSet	: QList<KBConfig> &	: List of found entries		*/
/*  path	: QString		: Path to this object		*/
/*  (returns)	: void			:				*/

void	KBObject::findAllConfigs
	(	QList<KBConfig>	&configSet,
		QString		path
	)
{
	path	= path.isEmpty() ? getName() : path + '/' + getName() ;

	LITER
	(	KBNode,
		getChildren(),
		child,

		KBConfig *config = child->isConfig() ;

		if (config != 0)
		{
			configSet.append(config) ;
			config->setPath	(path)   ;
			continue	;
		}

		KBObject *object = child->isObject() ;

		if (object != 0)
			object->findAllConfigs (configSet, path) ;
	)
}


/*  KBObject								*/
/*  pasteObjects: Paste in cut or copied objects			*/
/*  (returns)	: void		:					*/

void	KBObject::pasteObjects ()
{
	/* Placeholder only. Overridden in KBBlock, KBFramer and	*/
	/* KBComponent.							*/
}

/*  KBObject								*/
/*  calcNumRows	: Calculate number of rows of controls			*/
/*  bw		: uint		: Block display width			*/
/*  bh		: uint		: Block display height			*/
/*  dx		: int		: Control X-spacing			*/
/*  dy		: int		: Control X-spacing			*/
/*  (returns)	: uint		: Number of rows that can be displayed	*/

uint	KBObject::calcNumRows
	(	uint	bw,
		uint	bh,
		int	dx,
		int	dy
	)
{
	if (getRoot()->isReport() != 0) return 1 ;

//	fprintf	(stderr, "calcNumRows: dy=%2d bh=%3d y=%3d\n",
//			 dy,
//			 bh,
//			 y.getIntValue()) ;

	QRect	g	= geometry() ;

	/* Calculate dynamically how many rows can be displayed (but	*/
	/* always display at least one whatever).			*/
	int	xmax	= dx <= 0 ? 0 : (bw - g.width () - x.getIntValue()) / dx + 1 ;
	int	ymax	= dy <= 0 ? 0 : (bh - g.height() - y.getIntValue()) / dy + 1 ;

	int	numrows	= xmax == 0    ? ymax :
			  ymax == 0    ? xmax :
			  xmax >  ymax ? ymax :
					 xmax ;

//	fprintf	(stderr, "KBObject::calcNumRows => %d\n", numrows) ;
	return	numrows	<= 0 ? 1 : numrows    ;
}

/*  KBObject								*/
/*  calcNumRows	: Calculate number of rows of controls			*/
/*  dx		: int		: Control X-spacing			*/
/*  dy		: int		: Control X-spacing			*/
/*  mRows	: uint		: Maximum number of rows		*/
/*  (returns)	: uint		: Number of rows that can be displayed	*/

uint	KBObject::calcNumRows
	(	int	dx,
		int	dy,
		uint	mRows
	)
{
	uint	nRows	= calcNumRows (display->width(), display->height(), dx, dy) ;
	return	nRows > mRows ? mRows : nRows ;
}

/*  KBObject								*/
/*  buildDisplay: Build display for this item				*/
/*  parent	: KBDisplay *	: Parent display			*/
/*  (returns)	: void		:					*/

void	KBObject::buildDisplay
	(	KBDisplay	*parent
	)
{
	display = parent  ;
}

/*  KBObject								*/
/*  buildCtrlds	: Build controls for this item				*/
/*  numrows	: uint		: Number of rows in parent		*/
/*  dx		: int		: X offset between rows			*/
/*  dy		: int		: Y offset between rows			*/
/*  (returns)	: void		:					*/

void	KBObject::buildCtrls
	(	uint		,
		int		,
		int
	)
{
}

/*  KBObject								*/
/*  setGeometry	: Set object geometry					*/
/*  rect	: const QRect &	: New geometry				*/
/*  (returns)	: void		:					*/

void	KBObject::setGeometry
	(	const QRect	&rect
	)
{
	move	(rect.x	   (), rect.y	  ()) ;
	resize	(rect.width(), rect.height()) ;
}

/*  KBObject								*/
/*  getPalette	: Get palette for this object				*/
/*  useParent	: bool		: Use parent's palette			*/
/*  (returns)	: QPalette *	: The palette				*/

QPalette *KBObject::getPalette
	 (	bool	useParent
	 )
{
	static	QPalette *appPalette	= 0 ;
	static	QPalette *whitePalette	= 0 ;

	if (appPalette == 0)
	{
		appPalette	= new QPalette (QApplication::palette()) ;
		whitePalette	= new QPalette (QApplication::palette()) ;

		whitePalette->setColor (QColorGroup::Background, Qt::white) ;
	}

	if (curPal == 0)
	{

		QString	 fgcolor = getAttrVal ("fgcolor") ;
		QString	 bgcolor = getAttrVal ("bgcolor") ;

		QPalette pal	 = (getDisplay () != 0) && useParent ?
					getDisplay()->getDisplayWidget()->palette() :
				   getRoot()->isReport() ?
					*whitePalette :
					*appPalette   ;

		if (!fgcolor.isEmpty())
		{	const QColor fg = QColor (QRgb(strtol(fgcolor,0,0))) ;
			pal.setColor (QColorGroup::Text,       fg) ;
			pal.setColor (QColorGroup::ButtonText, fg) ;
			pal.setColor (QColorGroup::Foreground, fg) ;
		}
		if (!bgcolor.isEmpty())
		{	const QColor bg = QColor (QRgb(strtol(bgcolor,0,0))) ;
			pal.setColor (QColorGroup::Base,       bg) ;
			pal.setColor (QColorGroup::Button,     bg) ;
			pal.setColor (QColorGroup::Background, bg) ;
		}

		curPal	= new QPalette (pal) ;
	}

	return	curPal ;
}

/*  KBObject								*/
/*  getFont	: Get font for this object				*/
/*  useParent	: bool		: Use parent's font			*/
/*  (returns)	: QFont *	: The font				*/

QFont	*KBObject::getFont
	(	bool	useParent
	)
{
	if (curFont == 0)
	{
		QString	spec	= getAttrVal ("font") ;
		QFont	font	;

		if	(!spec.isEmpty())
			font	= KBFont::specToFont (spec) ;
		else if ((getDisplay () != 0) && useParent)
			font	= getDisplay()->getDisplayWidget()->font() ;
		else	font	= QApplication::font() ;

		curFont	= new QFont (font) ;
	}

	return	curFont	;
}

/*  KBObject								*/
/*  setControl	: Set the control to be used for sizing			*/
/*  _control	: KBControl *	: Control in question			*/
/*  (returns)	: void		:					*/

void	KBObject::setControl
	(	KBControl *_control
	)
{
#if	! __KB_RUNTIME
	if (control != _control) DELOBJ (sizer)	;
#endif
	control	= _control ;
}

/*  KBObject								*/
/*  getDisplayWidget							*/
/*		: Get widget used for display				*/
/*  (returns)	: QWidget *	: Display widget			*/

QWidget	*KBObject::getDisplayWidget ()
{
	return	display == 0 ? 0 : display->getDisplayWidget () ;
}

/*  KBObject								*/
/*  getTopWidget: Get top-most widget					*/
/*  (returns)	: QWidget *	: Top-most widget			*/

QWidget	*KBObject::getTopWidget ()
{
	return	display == 0 ? 0 : display->getTopWidget () ;
}

/*  KBObject								*/
/*  keyStroke	: Handle keystroke					*/
/*  e		: QKeyEvent *	: Event to be handled			*/
/*  (returns)	: bool		: True if event consumed		*/

bool	KBObject::keyStroke
	(	QKeyEvent	*
	)
{
	return	false	;
}

/*  KBObject								*/
/*  getBlock	: Get enclosing block for object			*/
/*  (returns)	: KBBlock *	: Block or null if none			*/

KBBlock	*KBObject::getBlock ()
{
	for (KBNode *p = getParent () ; p != 0 ; p = p->getParent())
		if (p->isBlock() != 0)
			return	p->isBlock() ;
	return	0 ;
}

/*  KBObject								*/
/*  getFormBlock: Get form block containing this object			*/
/*  (returns)	: KBFormBlock *	: Form block or null if none		*/

KBFormBlock *KBObject::getFormBlock ()
{
	for (KBNode *p = getParent () ; p != 0 ; p = p->getParent())
		if (p->isFormBlock() != 0)
			return	p->isFormBlock() ;
	return	0 ;
}

/*  KBObject								*/
/*  getReportBlock: Get report block containing this object		*/
/*  (returns)	  : KBReportBlock *	: Report block or null if none	*/

KBReportBlock *KBObject::getReportBlock ()
{
	for (KBNode *p = getParent () ; p != 0 ; p = p->getParent())
		if (p->isReportBlock() != 0)
			return	p->isReportBlock() ;
	return	0 ;
}

/*  KBObject								*/
/*  getNavigator: Get navigator containing this object			*/
/*  (returns)	: KBNavigator *	: Navigator or null if none		*/

KBNavigator *KBObject::getNavigator ()
{
	for (KBNode *p = getParent () ; p != 0 ; p = p->getParent())
		if (p->isNavigator() != 0)
			return	p->isNavigator() ;
	return	0 ;
}

/*  KBObject								*/
/*  write	: Write object to report writer				*/
/*  writer	: KBWriter *	: Writer object				*/
/*  offset	: QPoint	: Geometry offset			*/
/*  first	: bool		: First on page				*/
/*  extra	: init &	: Return extra space			*/
/*  prior	: bool		: Write prior value			*/
/*  (returns)	: void		:					*/

bool	KBObject::write
	(	KBWriter	*writer,
		QPoint		offset,
		bool		,
		int		&extra,
		bool		
	)
{
	/* The work is all done at the control level.			*/
	return	control->write
		(	writer,
			geometry(offset),
			KBValue(),
			false,
			extra
		)	;
}

/*  KBObject								*/
/*  setEnabled	: Set control enabled/disabled				*/
/*  enabled	: bool		: Enabled/disabled			*/
/*  (returns)	: void		:					*/

void	KBObject::setEnabled
	(	bool	enabled
	)
{
	if (control != 0) control->setEnabled (enabled) ;
}

/*  KBObject								*/
/*  setVisible	: Set control visibility				*/
/*  visible	: bool		: Visibility				*/
/*  (returns)	: void		:					*/

void	KBObject::setVisible
	(	bool	visible
	)
{
	if (control != 0) control->setVisible (visible) ;
}

/*  KBObject								*/
/*  isEnabled	: Get control enabled/disabled				*/
/*  (returns)	: bool		: Enabled/disabled			*/

bool	KBObject::isEnabled ()
{
	return	control == 0 ? false : control->isEnabled () ;
}

/*  KBObject								*/
/*  isVisible	: Get control visibility				*/
/*  (returns)	: bool		: Visibility				*/

bool	KBObject::isVisible ()
{
	return	control == 0 ? false : control->isVisible () ;
}

/*  KBObject								*/
/*  getMoveLimit: Get control movement limits				*/
/*  (returns)	: KBLimit	: Limit information			*/

KBLimit	KBObject::getMoveLimit ()
{
	if (getParent() == 0) return KBLimit () ;

	QRect	pRect	= getParent()->isObject()->geometry() ;
	QRect	mRect	= geometry () ;
 
	return	KBLimit
		(	-mRect.left(), pRect.width () - mRect.right (),
			-mRect.top (), pRect.height() - mRect.bottom()
		)	;
}

/*  KBObject								*/
/*  eventHook	 : Event execution hook					*/
/*  event	 : KBEvent &	: Event to execute			*/
/*  argc	 : uint		: Argument count			*/
/*  argv	 : KBValue *	: Argument vector			*/
/*  evRc	 : bool &	: Event return code			*/
/*  (returns)	 : bool		: No execution error			*/

bool	KBObject::eventHook
	(	KBEvent		&event,
		uint		argc,
		KBValue		*argv,
		bool		&evRc
	)
{
	KBValue	resval	;

	if (event.execute (resval, argc, argv) == KB::ScriptInlineError)
	{
		propertyDlg (event.getName()) ;
		return	false	;
	}

	evRc	= resval.isTrue() ;
	return	true	;
}

#if	! __KB_EMBEDDED

/*  KBObject								*/
/*  property	: Get QT widget property				*/
/*  name	: cchar *	: Property name				*/
/*  (returns)	: QVariant	: Property value			*/

QVariant
	KBObject::property
	(	cchar		*name
	)
{
	return	control == 0 ? QVariant() : control->property (name) ;
}

/*  KBObject								*/
/*  setProperty	: Set QT widget property				*/
/*  name	: cchar *	   : Property name			*/
/*  value	: const QVariant & : Value				*/
/*  (returns)	: bool		   : Success				*/

bool	KBObject::setProperty
	(	cchar		*name,
		const QVariant	&value
	)
{
	return	control == 0 ? false : control->setProperty (name, value) ;
}

#endif

/*  KBObject								*/
/*  getNamedObject							*/
/*		: Get named object relative to this one			*/
/*  name	: cchar *	: Name of object			*/
/*  showErr	: bool		: Show error dialog on error		*/
/*  (returns)	: KBObject *	: Object or null if not found		*/

KBObject*KBObject::getNamedObject
	(	QString		string,
		bool		showErr
	)
{
	QString	name	= string ;
	QString	rest	;
	int	idx	;

	/* Look for a separator. If this is the very first character	*/
	/* then work up to the root of the document, unless we are the	*/
	/* root.							*/
	if ((idx = name.find ('/')) == 0)
	{
		if ((getParent() != 0) && (getParent()->isObject() != 0))
			return	getParent()->isObject()->getNamedObject (name) ;

		name	= name.mid  (1)   ;
		idx	= name.find ('/') ;
	}

	/* If there is now a separator then split the string on it, so	*/
	/* that "name" is left with just the name at this level, and	*/
	/* "rest" contains everything after the separator.		*/
	if (idx > 0)
	{
		rest	= name.mid  (idx + 1)	;
		name	= name.left (idx)	;
	}

//	fprintf	(stderr, "getNamedObject: [%s][%d][%s]\n",
//			 (cchar *)name,
//			 idx,
//			 (cchar *)rest) ;


	/* Scan for the named object. There are a number of special	*/
	/* cases to be accounted for ...				*/
	/*	.		: Self   (as in Unix path names)	*/
	/*	..		: Parent (as in Unix path names)	*/
	/*	getRoot ()	: Document root				*/
	/*	getBlock()	: Enclosing block			*/
	/* NOTE:							*/
	/* The latter are not method applications, though the format	*/
	/* leaves scope for this later.					*/
	KBObject *object = 0 ;

	if	(name == "." )
	{
		object	= this	;
	}
	else if (name == "..")
	{
		if (getParent() != 0)
			object	= getParent()->isObject() ;
	}
	else if (name == "getRoot()" )
	{
		object	= getRoot  ()->isObject() ;
	}
	else if (name == "getBlock()")
	{
		object	= getBlock () ;
	}
	else
	{
		CITER
		(	Object,
			o,
			if (o->getName() == name)
			{	object	= o ;
				break	;
			}
		)
	}

	/* If have found an object and there are more components in the	*/
	/* path then work on down through them.				*/
	if (object != 0)
		if (!rest.isNull())
			object	= object->getNamedObject (rest, false) ;


	/* If we have not found an object and the show-error option is	*/
	/* set then show the no-object error dialog. If the user OK's	*/
	/* this then continue with the selected object, otherwise	*/
	/* really return not-found.					*/
#if	! __KB_RUNTIME
	if ((object == 0) && showErr)
	{
		KBNoObjDlg noObjDlg (this, string) ;

		if (noObjDlg.exec ()) object = noObjDlg.selectedObj() ;
	}
#endif

	return	object	;
}

/*  KBObject								*/
/*  getNamedObject							*/
/*		: Get named object relative to this one			*/
/*  name	: cchar *	: Name of object			*/
/*  pError	: KBError &	: Error message return			*/
/*  showErr	: bool		: Show error dialog on error		*/
/*  (returns)	: KBObject *	: Object or null if not found		*/

KBObject*KBObject::getNamedObject
	(	QString		string,
		KBError		&pError,
		bool		showErr
	)
{
	KBObject *object = getNamedObject (string, showErr) ;

	if (object == 0)
		pError	= KBError
			  (	KBError::Error,
				"Cannot find named object",
				string,
				__ERRLOCN
			  )	;

	return	object	;
}

/*  KBObject								*/
/*  connectLinks: Connect event/slot links				*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		: Success				*/

bool	KBObject::connectLinks
	(	KBError		&pError
	)
{
	LITER
	(	KBSlot,
		m_slotList,
		slot,
		if (!slot->connectLinks (pError)) return false ;
	)
	CITER
	(	Object,
		obj,
		if (!obj ->connectLinks (pError)) return false ;
	)

	return	true	;
}			

/*  KBObject								*/
/*  getEmitter	: Get signal emitter object for named event attribute	*/
/*  name	: const QString & : Event name				*/
/*  (returns)	: KBEmitter *	  : Emitter or null if none		*/

KBEmitter
	*KBObject::getEmitter
	(	const QString	&name
	)
{
	KBAttr	*attr	= getAttr (name) ;
	if (attr  == 0) return 0 ;

	KBEvent	*event	= attr->isEvent() ;
	if (event == 0) return 0 ;

	return	event->getEmitter () ;
}

/*  KBObject								*/
/*  addSlot	: Add a slot to this node				*/
/*  slot	: KBSlot *	: The slot in question			*/
/*  (returns)	: void		:					*/

void	KBObject::addSlot
	(	KBSlot		*slot
	)
{
	m_slotList.append (slot) ;

	fprintf
	(	stderr,
		"Node [%s] adds slot [%s]\n",
		(cchar *)getAttrVal("name"),
		(cchar *)slot->name()
	)	;
}

/*  KBObject								*/
/*  getSlots	: Get list of slots					*/
/*  (returns)	: const QList<KBSlot> & :				*/

const QList<KBSlot> &KBObject::getSlots ()
{
	return	m_slotList ;
}

/*  KBObject								*/
/*  clearSlots	: Clear current slot list				*/
/*  (returns)	: void		:					*/

void	KBObject::clearSlots ()
{
	m_slotList.clear () ;
}


void	KBObject::setTabOrder
	(	int
	)
{
}

int	KBObject::getTabOrder ()
{
	return	-1 ;
}


/*  KBObject								*/
/*  setScriptObject							*/
/*		: Set persistent script object				*/
/*  scriptObj	: KBScriptObject *	: Persistent script object	*/
/*  (returns)	: void			:				*/

void	KBObject::setScriptObject
	(	KBScriptObject	*scriptObj
	)
{
	m_scriptObj = scriptObj ;
}


/*  KBObject								*/
/*  scriptObject: Get persisistent script object			*/
/*  (returns)	: KBScriptObject *	: Persistent script object	*/

KBScriptObject
	*KBObject::scriptObject ()
{
	return	m_scriptObj ;
}


/*  KBObject								*/
/*  minPosition	: Get minimal x- and y- positions from a list of nodes	*/
/*  nodes	: const QList<KBNode> &	: List of notes			*/
/*  minx	: int &			: Return minimum x-position	*/
/*  miny	: int &			: Return minimum y-position	*/
/*  (returns)	: void			:				*/

void	KBObject::minPosition
	(	const QList<KBNode>	&nodes,
		int			&minx,
		int			&miny
	)
{
	KBObject *oObj	      ;
	minx	 = 0x7fffffff ;
	miny	 = 0x7fffffff ;

	LITER
	(	KBNode,
		nodes,
		oNode,

		if ((oObj = oNode->isObject()) == 0)
			continue ;

		if (oObj->isHidden() != 0)
			continue ;

		QRect	r = oObj->geometry() ;
		int 	x = r.x() ;
		int	y = r.y() ;

		if (x < minx) minx = x ;
		if (y < miny) miny = y ;
	)
}

QString	KBObject::getPath ()
{
	if (getParent() != 0)
	{
		return	getParent()->isObject()->getPath() + "/" + name.getValue() ;
	}

	return	name.getValue() ;
}
