/***************************************************************************
    file	         : kb_qrydesign.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<time.h>

#include	<qlist.h>
#include	<qintdict.h>


#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_location.h"
#include	"kb_docroot.h"

#include	"kb_qrydesign.h"
#include	"kb_block.h"
#include	"kb_notifier.h"
#include	"kb_nodereg.h"

#include	"tk_messagebox.h"


#define		QC_NONE		0
#define		QC_NAME		1
#define		QC_TYPE		2
#define		QC_PRIMARY	3
#define		QC_DESCR	4

#define		QC_NULLOK	5
#define		QC_LEN		6
#define		QC_INDEX	7
#define		QC_UNIQUE	8
#define		QC_PREC		9
#define		QC_COLNAM	99


/*  KBTabType								*/
/*  ---------								*/
/*  This class provides a type object specifically for the table	*/
/*  design query.							*/

class	KBTabType : public KBType
{
	uint	which	;
	bool	error	(KBError &, cchar *) ;

public	:

	KBTabType (uint) ;

	virtual	bool	isValid (const QByteArray &,
				 bool,
				 KBError &,
				 const QString    & = QString::null) ;
}	;



static	KBTabType	*designTypes[TI_COUNT] ;

/*  ------------------------------------------------------------------  */

/*  KBTabType								*/
/*  KBTabType	: Constructor for table design query type		*/
/*  which	: uint		: Notional column number		*/
/*  (returns)	: KBTabType	:					*/

KBTabType::KBTabType
	(	uint	which
	)
	:
	KBType	("Tab", KB::ITString, 0, 0, true),
	which	(which)
{
}

/*  KBTabType								*/
/*  error	: Set error code					*/
/*  pError	: KBError &	: Error object				*/
/*  text	: cchar *	: Error message				*/
/*  (returns)	: bool		: Always false				*/

bool	KBTabType::error
	(	KBError	&pError,
		cchar	*text
	)
{
	pError	= KBError 
		  (	KBError::Fault,
			text,
			"",
			__ERRLOCN
		  )	;
	return	false	;
}

/*  KBTabType								*/
/*  isValid	: Check validity of value				*/
/*  value	: const QByteArray &: Value to be checked		*/
/*  null	: bool		    : Value is null			*/
/*  pError	: KBError &	    : Error return			*/
/*  what	: const QString &   : Not used here			*/
/*  (returns)	: bool		    : Valid				*/
	
bool	KBTabType::isValid
	(	const QByteArray &,
		bool		 null,
		KBError		 &pError,
		const QString	 &
	)
{
	switch (which)
	{
		case QC_NAME	:
			if (null)
				return error (pError, TR("Column name must be set")) ;
			break	;

		case QC_TYPE	:
			if (null)
				return error (pError, TR("Column type must be set")) ;
			break	;

		default	:
			break	;
	}

	return	true	;
}


/*  ------------------------------------------------------------------  */

/*  KBQryDesign								*/
/*  KBQryDesign	: Constructor for form query node			*/
/*  parent	: KBNode *	: Parent node				*/
/*  aList	: const QDict<QString> &				*/
/*				: List of attributes			*/
/*		: bool *	:					*/
/*  (returns)	: KBQryDesign	:					*/

KBQryDesign::KBQryDesign
	(	KBNode			*parent,
		const QDict<QString>	&aList,
		bool			*
	)
	:
	KBQryBase	(parent, aList,	 	"KBQryDesign"),
	server		(this,	 "server",	aList),
	table		(this,   "table",	aList),
	create		(this,	 "create",	aList)
{
	fieldName	= 0 ;
	fieldType	= 0 ;
	fieldNullOK	= 0 ;
	fieldLen	= 0 ;
	fieldDescr 	= 0 ;
	fieldIndex	= 0 ;
	fieldUnique	= 0 ;
	isNew		= false ;
	tabInfo	 	= 0	;

	designInfo.setAutoDelete (true) ;
}

/*  KBQryDesign								*/
/*  ~KBQryDesign: Destructor for form query node			*/
/*  (returns)	:		:					*/

KBQryDesign::~KBQryDesign ()
{
}

/*  KBQryDesign								*/
/*  addItem	: Add a child form item					*/
/*  qlvl	: uint		: Query level				*/
/*  item	: KBItem *	: Child item				*/
/*  (returns)	: bool		: Fetches from database			*/

bool	KBQryDesign::addItem
	(	uint	qlvl,
		KBItem	*item
	)
{
	static	KBTabType	_kbTabNullOK(QC_NULLOK) ;
	static	KBTabType	_kbTabName  (QC_NAME  ) ;
	static	KBTabType	_kbTabType  (QC_TYPE  ) ;
	static	KBTabType	_kbTabLen   (QC_LEN   ) ;
	static	KBTabType	_kbTabPrec  (QC_PREC  ) ;
	static	KBTabType	_kbTabDescr (QC_DESCR ) ;
	static	KBTabType	_kbTabIndex (QC_INDEX ) ;

	/* An item specified as zero is used to indicate that the set	*/
	/* of items is to be reloaded. We use it here to clear the	*/
	/* current settings.						*/
	if (item == 0)
	{
		switch (qlvl)
		{
			case 0	:
				fieldName   = 0 ;
				fieldType   = 0 ;
				fieldNullOK = 0 ;
				fieldLen    = 0 ;
				fieldPrec   = 0 ;
				fieldDescr  = 0 ;
				fieldIndex  = 0 ;
				fieldUnique = 0 ;
				tblOuter.clear () ;
				break	;

			case 1	:
				tblInner.clear () ;
				break	;

			default	:
				break	;
		}

		return	true	;
	}

	const	QString	&name = item->getName () ;

	/* Query level is the main design stuff (ie., the field		*/
	/* specifications) plus the description (no prizes where that	*/
	/* comes from ....). These are handled specially.		*/
	if (qlvl == 0)
	{
		if (item->isRowMark() != 0)
		{	item->setQueryIdx (KBQryIdx(0,QC_NONE)) ;
			return	true   ;
		}
		tblOuter.append (item) ;

		/* Note the item on the basis of its name, and set its	*/
		/* notional column index. Note that we ignore the	*/
		/* expression.						*/
		if (name == "Name")
		{	fieldName  = item ;
			item->setQueryIdx  (KBQryIdx(0,QC_NAME)) ;
			item->setFieldType (&_kbTabName ) ;
			return	true	;
		}

		if (name == "Type")
		{	fieldType  = item ;
			item->setQueryIdx  (KBQryIdx(0,QC_TYPE)) ;
			item->setFieldType (&_kbTabType  ) ;
			return	true	;
		}

		if (name == "Description")
		{	fieldDescr  = item ;
			item->setQueryIdx  (KBQryIdx(0,QC_DESCR)) ;
			item->setFieldType (&_kbTabDescr) ;
			return	true	;
		}

#if	__KB_EMBEDDED
		if (name == "PK")
#else
		if (name == "PKey")
#endif
		{	fieldPrimary = item ;
			item->setQueryIdx  (KBQryIdx(0,QC_PRIMARY)) ;
			item->setFieldType (&_kbTabDescr) ;
			return	true	;
		}

		fprintf	(stderr, "Unexpected design name: %s\n", (cchar *)name) ;
		return	true	;
	}


	/* Query level one mostly the design dictionary stuff. All	*/
	/* this is picked up from the info sets array, et al. If there	*/
	/* turns out to be no design dictionary then the controls will	*/
	/* be disabled.							*/
	if (qlvl != 1)
	{
		fprintf	(stderr, "Unexpected design level: %d\n", qlvl) ;
		return	true	;
	}


	if (designTypes[0] == 0)
		for (uint idx = 0 ; idx < TI_COUNT ; idx += 1)
			designTypes[idx] = new KBTabType (idx | 0x8000) ;

	tblInner.append  (item) ;

	if (name == "NullOK" )
	{	fieldNullOK = item ;
		item->setQueryIdx  (KBQryIdx(0,QC_NULLOK)) ;
		item->setFieldType (&_kbTabNullOK) ;
		return	true	;
	}

	if (name == "Length" )
	{	fieldLen    = item ;
		item->setQueryIdx  (KBQryIdx(0,QC_LEN   )) ;
		item->setFieldType (&_kbTabLen  ) ;
		return	true	;
	}

	if (name == "Prec"   )
	{	fieldPrec   = item ;
		item->setQueryIdx  (KBQryIdx(0,QC_PREC  )) ;
		item->setFieldType (&_kbTabPrec ) ;
		return	true	;
	}

	if (name == "Indexed")
	{	fieldIndex  = item ;
		item->setQueryIdx  (KBQryIdx(0,QC_INDEX )) ;
		item->setFieldType (&_kbTabIndex) ;
		return	true	;
	}

	if (name == "Unique" )
	{	fieldUnique = item ;
		item->setQueryIdx  (KBQryIdx(0,QC_UNIQUE)) ;
		item->setFieldType (&_kbTabIndex) ;
		return	true	;
	}

	if (name == "Column" )
	{	
		item->setQueryIdx  (KBQryIdx(0,QC_COLNAM)) ;
		item->setFieldType (&_kbTabNullOK) ;
		return	true	;
	}


	int	ident	= -1 ;

	if 	(name == "Evalid" ) ident = TI_EVALID  ;
	else if (name == "Igncase") ident = TI_IGNCASE ;
	else if (name == "Defval" ) ident = TI_DEFAULT ;
	else if (name == "Format" ) ident = TI_FORMAT  ;
	else if (name == "Link"   ) ident = TI_LINK    ;

	if (ident < 0)
		KBError::EFault
		(	TR("Unexpected design field"),
			name,
			__ERRLOCN
		)	;


	item->setQueryIdx  (KBQryIdx(0,ident|0x8000)) ;
	item->setFieldType (designTypes[ident]) ;

	return	true	;
}

/*  KBQryDesign								*/
/*  remItem	: Remove a child form item				*/
/*  qlvl	: uint		: Query level				*/
/*  item	: KBItem *	: Child form item			*/
/*  (returns)	: void		:					*/

void	KBQryDesign::remItem
	(	uint	,
		KBItem	*
	)
{
}

/*  KBQryDesign								*/
/*  prepare	: Pre-execution setup					*/
/*  (returns)	: void		:					*/

void	KBQryDesign::prepare ()
{
	if (!linkServer (server.getValue()))
		lastError().DISPLAY() ;

	svrName	 = server.getValue     () ;
	tabName	 = table .getValue     () ;
	isNew	 = create.getBoolValue () ;
	tabInfo	 = 0	;

	KBDBInfo	*dbInfo  = getDocRoot ()->getDBInfo () ;
	KBServerInfo	*svInfo	 = dbInfo->findServer(svrName) ;

	if (svInfo != 0)
		tabInfo = svInfo->tableInfoSet()->getTableInfo (tabName) ;
}

/*  KBQryDesign								*/
/*  setLocation	: Set new location					*/
/*  newSvrName	: const QString & : New server name			*/
/*  newTabName	: const QString & : New table name			*/
/*  (returns)	: bool		  : Success				*/

bool	KBQryDesign::setLocation
	(	const QString	&newSvrName,
		const QString	&newTabName
	)
{
	/* First case is changing server. We ensure that we can get a	*/
	/* connection to the new server and that the target table does	*/
	/* not already exists there before changing anything.		*/
	if (newSvrName != svrName)
	{
		KBDBLink newLink	;
		bool	 newExists	;

		if (!newLink.connect (getDocRoot()->getDBInfo(), newSvrName))
		{
			newLink.lastError().DISPLAY() ;
			return	 false	;
		}
		if (!newLink.tableExists (newTabName, newExists))
		{
			newLink.lastError().DISPLAY() ;
			return	 false	;
		}

		if (newExists)
		{
			KBError::EWarning
			(	TR("Specified table already exists"),
				QString(TR("Server %1, Table %2")).arg(newSvrName).arg(newTabName),
				__ERRLOCN
			)	;
			return	false	;
		}
		if (!dbLink.copyLink (newLink, true))
		{
			dbLink .lastError().DISPLAY() ;
			return	 false	;
		}

		server .setValue (svrName = newSvrName) ;
		table  .setValue (tabName = newTabName) ;
		oldSpec.reset    (tabName) ;
		isNew	 = true ;

		/* The new server may or may not not have a design	*/
		/* so enable to disable the design items appropriately.	*/
		ddExists = (svrName == KBLocation::m_pFile) ||
			   dbLink.hasObjectTable() ;

		LITER
		(	KBItem,
			tblInner,
			item,

			if ((item->getQueryIdx() & 0x8000) != 0)
				item->setEnabled (0, ddExists) ;
		)

		return	true	 ;
	}

	/* Second case is just a change in the table name. We need only	*/
	/* ensure that the target table does not exist.			*/
	if (newTabName != tabName)
	{
		bool	 newExists	;

		if (!dbLink.tableExists (newTabName, newExists))
		{
			dbLink.lastError().DISPLAY() ;
			return	 false	;
		}

		if (newExists)
		{
			KBError::EWarning
			(	TR("Specified table already exists"),
				QString(TR("Server %1, Table %2")).arg(newSvrName).arg(newTabName),
				__ERRLOCN
			)	;
			return	false	;
		}

		table  .setValue (tabName = newTabName) ;
		oldSpec.reset    (tabName) ;
		isNew	= true	;
		return	true	;
	}

	return	true	;
}

/*  KBQryDesign								*/
/*  resetData	: Reset underlying data					*/
/*  qlvl	: uint		: Caller's query level			*/
/*  qrow	: uint		: Caller's query row number		*/
/*  (returns)	: bool		: Success				*/

void	KBQryDesign::resetData
	(	uint	,
		uint
	)
{
	/* Null operation in table design ...				*/
}

/*  KBQryDesign								*/
/*  loadItems	: Load items from query					*/
/*  qlvl	: uint		: Caller's query level			*/
/*  qrow	: uint		: Caller's query row number		*/
/*  (returns)	: bool		: Success				*/

bool	KBQryDesign::loadItems
	(	uint	qlvl,
		uint	qrow
	)
{
	QList<KBItem> items = qlvl == 0 ? tblOuter : tblInner ;

	LITER
	(	KBItem,
		items,
		item,

		item->setValue
		(	item->getBlock()->getCurQRow(),
			getField (qlvl, qrow, item->getQueryIdx())
		)
	)

	return	true	;
}

/*  KBQryDesign								*/
/*  clearItems	: Clear all items in query				*/
/*  qlvl	: uint		: Caller's query level			*/
/*  qrow	: uint		: Caller's query row number		*/
/*  (returns)	: bool		: Success				*/

bool	KBQryDesign::clearItems
	(	uint	qlvl,
		uint	
	)
{
	QList<KBItem> items = qlvl == 0 ? tblOuter : tblInner ;

	LITER
	(	KBItem,
		items,
		item,

		item->clearValue (item->getBlock()->getCurQRow(), true) ;
	)

	return	true	;
}

/*  KBQryDesign								*/
/*  getRowState	: Get state of specified query row			*/
/*  qlvl	: uint		: Query level				*/
/*  qrow	: uint		: Row number				*/
/*  (returns)	: KB::RState	: State					*/

KB::RState KBQryDesign::getRowState
	(	uint	,
		uint	qrow
	)
{
	return	qrow < newSpec.m_fldList.count() ?
		       newSpec.m_fldList.at(qrow)->m_state : KB::RSInSync ;
}

/*  KBQryDesign								*/
/*  getField	: Get field value					*/
/*  qlvl	: uint		: Query level				*/
/*  qrow	: uint		: Row number				*/
/*  qcol	: uint		: Column number				*/
/*  initial	: bool		: Get initial value			*/
/*  (returns)	: KBValue	: Value					*/

KBValue	KBQryDesign::getField
	(	uint	qlvl,
		uint	qrow,
		uint	qcol,
		bool
	)
{
	if (qlvl == 0)
	{
		KBFieldSpec *spec = newSpec.m_fldList.at(qrow) ;

		if (spec == 0) return KBValue ()   ;

		switch (qcol)
		{
			case QC_NAME	:
				return	KBValue (spec->m_name,     &_kbString) ;

			case QC_TYPE	:
				return	KBValue (spec->m_typeName, &_kbString) ;

			case QC_PRIMARY :
				return	KBValue ((spec->m_flags & KBFieldSpec::Primary) == 0 ? "0" : "1", &_kbString) ;

			default	:
				break	;
		}

		KBTableColumn *tc =  designInfo.at (qrow) ;
		if (tc == 0) return KBValue () ;
		
		switch (qcol)
		{
			case QC_DESCR	:
				return	KBValue(tc->designValue(TI_DESCR)) ;

			default	:
				break	;
		}

		return	KBValue () ;
	}

	KBFieldSpec *spec = newSpec.m_fldList.at(curQRow) ;

	if (spec != 0) switch (qcol)
	{
		case QC_NULLOK	:
			return	KBValue ((spec->m_flags & KBFieldSpec::NotNull) == 0 ? "Yes" : "No", &_kbString) ;

		case QC_INDEX	:
			return	KBValue ((spec->m_flags & KBFieldSpec::Indexed) == 0 ? "No" : "Yes", &_kbString) ;

		case QC_UNIQUE	:
			return	KBValue ((spec->m_flags & KBFieldSpec::Unique ) == 0 ? "No" : "Yes", &_kbString) ;

		case QC_LEN	:
			return	KBValue ((int)spec->m_length,  &_kbFixed) ;

		case QC_PREC	:
			return	KBValue ((int)spec->m_prec,    &_kbFixed) ;

		case QC_COLNAM	:
			return	KBValue	(spec->m_name) ;

		default	:
			break	;
	}

	KBTableColumn *tc = designInfo.at (curQRow) ;

	if (tc == 0) return KBValue ()   ;
	return	KBValue (tc->designValue (qcol & 0x7fff)) ;
}

/*  KBQryDesign								*/
/*  setField	: Set field value					*/
/*  qlvl	: uint		  : Query level				*/
/*  qrow	: uint		  : Row number				*/
/*  qcol	: uint		  : Column number			*/
/*  value	: const KBValue & : Value				*/
/*  (returns)	: void		  :					*/

void	KBQryDesign::setField
	(	uint		qlvl,
		uint		qrow,
		uint		qcol,
		const KBValue	&value
	)
{
	QString	text	= value.getRawText () ;

	if (qlvl == 0)
	{
		KBFieldSpec *spec = newSpec.m_fldList.at(qrow) ;
		if (spec == 0) return ;

		switch (qcol)
		{
			case QC_NAME	:
				fieldName   ->setValue (qrow, value) ;
				spec->m_name = text ;
				return	;

			case QC_TYPE	:
				fieldType   ->setValue (qrow, value) ;
				spec->m_typeName = text ;
				return	;

			case QC_PRIMARY :
				fieldPrimary->setValue (qrow, value) ;
				if (value.isTrue())
					spec->m_flags |=  KBFieldSpec::Primary ;
				else	spec->m_flags &= ~KBFieldSpec::Primary ;
				return	;

			default	:
				break	;
		}

		KBTableColumn *tc = designInfo.at (qrow) ;
		if (tc == 0) return ;

		switch (qcol)
		{
			case QC_DESCR	:
				fieldDescr->setValue  (qrow, value) ;
				tc->setDesignValue (TI_DESCR, text) ;

			default	:
				break	;
		}

		return	;
	}

	if (qrow != curQRow) return ;

	KBFieldSpec *spec = newSpec.m_fldList.at(curQRow) ;

	if (spec != 0) switch (qcol)
	{
		case QC_NULLOK	:
			fieldNullOK->setValue (0, value) ;
			if (value.isTrue())
				spec->m_flags |=  KBFieldSpec::NotNull ;
			else	spec->m_flags &= ~KBFieldSpec::NotNull ;
			return	;

		case QC_INDEX	:
			fieldIndex ->setValue (0, value) ;
			if (value.isTrue())
				spec->m_flags |=  KBFieldSpec::Indexed ;
			else	spec->m_flags &= ~KBFieldSpec::Indexed ;
			return	;

		case QC_UNIQUE	:
			fieldUnique->setValue (0, value) ;
			if (value.isTrue())
				spec->m_flags |=  KBFieldSpec::Unique  ;
			else	spec->m_flags &= ~KBFieldSpec::Unique  ;
			return	;

		case QC_LEN	:
			fieldLen   ->setValue (0, value) ;
			spec->m_length = text.toUInt() ;
			return	;

		case QC_PREC	:
			fieldPrec  ->setValue (0, value) ;
			spec->m_prec   = text.toUInt() ;
			return	;

		case QC_COLNAM	:
			return	;

		default	:
			break	;
	}

	KBTableColumn *tc = designInfo.at (curQRow) ;
	if (tc != 0) tc->setDesignValue (qcol & 0x7fff, text) ;
}

/*  KBQryDesign								*/
/*  qlvl	: uint		: Query level				*/
/*  getNumRows	: Get number of rows in query set			*/
/*  (returns)	: uint		: Number of rows			*/

uint	KBQryDesign::getNumRows
	(	uint	qlvl
	)
{
	return	qlvl == 0 ? newSpec.m_fldList.count () : 1 ;
}

/*  KBQryDesign								*/
/*  saveRow	: Save row						*/
/*  qlvl	: uint		: Query level				*/
/*  qrow	: uint		: Query row number			*/
/*  (returns)	: bool		: Success				*/

bool	KBQryDesign::saveRow
	(	uint	qlvl,
		uint	qrow
	)
{
	fprintf
	(	stderr, 
		"KBQryDesign::saveRow: %u,%u\n",
		qlvl,
		qrow
	)	;

	bool		changed	= false	;
	KBFieldSpec	*fspec	= 0	;
	KBTableColumn	*tc	= 0	;

	/* NOTE: As we save rows, we check for changes and set the row	*/
	/* state by hand. This is needed since the ::getField method	*/
	/* does not handle the initial value flag.			*/
	KBValue		value	;
	QString		text	;
	uint		flags	;
	uint		length	;
	uint		prec	;

	switch (qlvl)
	{
		case 0 :
			/* This is the outer level. Start by checking	*/
			/* that the outer level fields are all valid.	*/
			LITER
			(	KBItem,
				tblOuter,
				field,

				if (!field->isValid (qrow, false))
				{	setError (field->lastError()) ;
					return	 false ;
				}
			)


			/* If this is a new column then add a new	*/
			/* field specification to the table, and a new	*/
			/* design info structure.			*/
			if (qrow >= newSpec.m_fldList.count())
			{
				fspec = new KBFieldSpec (newSpec.m_fldList.count()) ;
				newSpec.m_fldList.append(fspec) ;
				designInfo.append  	(new KBTableColumn()) ;

				fspec->m_state = KB::RSInserted  ;
			}

			fspec	= newSpec.m_fldList.at (qrow) ;
			tc	= designInfo       .at (qrow) ;

			/* Save the field name, type and primary key	*/
			/* flag values into the field specification,	*/
			/* and retrieve a possibly changed description.	*/
			text	     = fieldName->getValue(qrow).getRawText() ;
			if (text  != fspec->m_name ) changed = true ;
			fspec->m_name  = text  ;

			text	     = fieldType->getValue(qrow).getRawText() ;
			if (text  != fspec->m_typeName) changed = true ;
			fspec->m_typeName = text  ;

			if (fieldPrimary->getValue(qrow).isTrue())
				flags = fspec->m_flags |  KBFieldSpec::Primary ;
			else	flags = fspec->m_flags & ~KBFieldSpec::Primary ;
			if (flags != fspec->m_flags) changed = true ;
			fspec->m_flags = flags ;

			value	= fieldDescr->getValue(qrow) ;
			if (value.getRawText() != tc->designValue(TI_DESCR)) changed = true ;
			tc->setDesignValue (TI_DESCR, value.getRawText()) ;

			/* Drop through to save the inner level ...	*/

		case 1	:
			/* This is the inner level, in which case use	*/
			/* the current outer row number to access the	*/
			/* data. There is a special check since we may	*/
			/* be called when the current outer row is the	*/
			/* notional empty row.				*/
			if (curQRow >= newSpec.m_fldList.count())
				return	true	;

			fspec	= newSpec.m_fldList.at (curQRow) ;
			tc	= designInfo       .at (curQRow) ;

			fprintf
			(	stderr,
				"KBQryDesign::saveRow: l=1 fspec=%p tc=%p\n",
				(void *)fspec,
				(void *)tc
			)	;
	
			LITER
			(	KBItem,
				tblInner,
				field,

				if (!field->isValid (0, false))
				{	setError (field->lastError()) ;
					return	 false ;
				}
			)

			/* The nullOK, index and unique fields are	*/
			/* actually part of the table specification.	*/
			/* All others are just design stuff.		*/
			flags	= fspec->m_flags ;

			if (fieldNullOK->getValue(0).getRawText() == "Yes")
				flags &= ~KBFieldSpec::NotNull ;
			else	flags |=  KBFieldSpec::NotNull ;

			if (fieldIndex ->getValue(0).getRawText() == "Yes")
				flags |=  KBFieldSpec::Indexed ;
			else	flags &= ~KBFieldSpec::Indexed ;

			if (fieldUnique->getValue(0).getRawText() == "Yes")
				flags |=  KBFieldSpec::Unique  ;
			else	flags &= ~KBFieldSpec::Unique  ;

			if (flags  != fspec->m_flags ) changed = true ;
			fspec->m_flags  = flags  ;

			length	= fieldLen ->getValue(0).getRawText().toUInt() ;
			if (length != fspec->m_length) changed = true ;
			fspec->m_length = length ;

			prec	= fieldPrec->getValue(0).getRawText().toUInt() ;
			if (prec   != fspec->m_prec  ) changed = true ;
			fspec->m_prec   = prec   ;

			if (tc == 0)
				break	;

			LITER
			(	KBItem,
				tblInner,
				field,

				uint colIdx  = field->getQueryIdx() ;
				if ((colIdx & 0x8000) == 0) continue ;
				colIdx &= 0x7fff ;

				value	= field->getValue (0) ;
				if (value.getRawText() != tc->designValue (colIdx)) changed = true ;

				fprintf
				(	stderr,
					"KBQryDesign::saveRow: save [%s] <- [%s]\n",
					(cchar *)field->getAttrVal("name"),
					(cchar *)field->getValue(0).getRawText()
				)	;

				tc->setDesignValue (colIdx, value.getRawText()) ;
			)

			break	;

		default	:
			/* Any other query level is a fatal error ....		*/
			KBError::EFault
			(	QString (TR("Unexpected query level %1")).arg(qlvl),
				QString::null,
				__ERRLOCN
			)	;
			break	;
	}

	/* The changed flag is set for any changes. If set then set	*/
	/* the specification state to changed manually since it will	*/
	/* not be picked up elsewhere.					*/
	if (changed && (fspec->m_state == KB::RSInSync))
		fspec->m_state = KB::RSChanged  ;

	// QString t 	;
	// newSpec.toXML  (t, 0, &designInfo) ;
	// fprintf (stderr, "%s", (cchar *)t) ;

	return	true	;
}

/*  KBQryDesign								*/
/*  deleteRow	: Delete row						*/
/*  qlvl	: uint		: Query level				*/
/*  qrow	: uint		: Query row number			*/
/*  (returns)	: bool		: Success				*/

bool	KBQryDesign::deleteRow
	(	uint	qlvl,
		uint	qrow
	)
{
	if (qlvl != 0) return false ;

	KBFieldSpec	*fSpec	= newSpec.m_fldList.at(qrow) ;

	if (fSpec->m_state != KB::RSInserted)
	{
		fSpec->m_state = KB::RSDeleted ;
		fSpec->m_dirty = true ;
		return	true ;
	}

	newSpec.m_fldList.remove (qrow) ;
	designInfo       .remove (qrow) ;

	for (uint idx = qrow ; idx < newSpec.m_fldList.count() ; idx += 1)
	{
		newSpec.m_fldList.at(idx)->m_dirty = true ;

		fieldDescr->setValue
		(	idx, 
			designInfo.at(idx)->designValue(QC_DESCR)
		)	;
	}

	LITER
	(	KBItem,
		tblInner,
		item,

		item->setValue
		(	qrow,
			getField (0, qrow, item->getQueryIdx())
		)
	)

	return	true	;
}

/*  KBQryDesign								*/
/*  insertRow	: Insert row						*/
/*  qlvl	: uint		: Query level				*/
/*  qrow	: uint		: Query row number			*/
/*  (returns)	: bool		: Success				*/

bool	KBQryDesign::insertRow
	(	uint	qlvl,
		uint	qrow
	)
{
	if (qlvl != 0) return false ;

	KBFieldSpec   *fs = new KBFieldSpec   (newSpec.m_fldList.count()) ;
	KBTableColumn *tc = new KBTableColumn () ;

	fs->m_state	= KB::RSInserted  ;

	newSpec.m_fldList.insert (qrow, fs) ;
	designInfo       .insert (qrow, tc) ;

	for (uint idx = qrow + 1 ; idx < newSpec.m_fldList.count() ; idx += 1)
	{
		newSpec.m_fldList.at(idx)->m_dirty = true ;

		fieldDescr->setValue
		(	idx, 
			designInfo.at(idx)->designValue(QC_DESCR)
		)	;
	}

	LITER
	(	KBItem,
		tblInner,
		item,
		item->clearValue (0, false)
	)

	return	true	;
}

/*  KBQryDesign								*/
/*  syncRow	: Synchronise row					*/
/*  qlvl	: uint		 : Query level				*/
/*  qrow	: uint		 : Query row number			*/
/*  pValue	: KBValue *	 : Parent value if any			*/
/*  cExpr	: const QString &: Child expression			*/
/*  block	: KBBlock *	 : Requesting block			*/
/*  oper	: KB::Action &	 : Operation performed			*/
/*  priKey	: KBValue &	 : Primary key return			*/
/*  (returns)	: bool		 : Success				*/

bool	KBQryDesign::syncRow
	(	uint		,
		uint		,
		KBValue		*,
		const QString	&,
		KBBlock		*,
		KB::Action	&,
		KBValue		&
	)
{
	return	false	;
}

/*  KBQryDesign								*/
/*  finish	: Issue change final notification			*/
/*  ok		: bool		: Success				*/
/*  (returns)	: void		:					*/

void	KBQryDesign::finish
	(	bool	ok
	)
{
	KBLocation location (getDocRoot()->getDBInfo(),
			     "query",
			     svrName,
			     tabName) ;
	KBNotifier::self()->nTablesChanged (location) ;
	ok = true ;
}

/*  KBQryDesign								*/
/*  copyOldData	: Copy old table data (and drop and rename)		*/
/*  (returns)	: bool		: Success				*/

bool	KBQryDesign::copyOldData ()
{

	QString		   select	;
	QString		   insert	;
	QString		   place	;
	uint		   slot		;
	cchar		   *sep		;

	/* Build the neccessary SQL statements. The "select" can	*/
	/* clearly be completely constructed; we build the insert with	*/
	/* placeholders.						*/
	select	= "select " ;
	insert	= "insert into " + dbLink.mapExpression (tmpName) + " (" ;
	place	= "" ;
	sep	= "" ;
	slot	= 0  ;

	for (uint nidx = 0 ; nidx < newSpec.m_fldList.count() ; nidx += 1)
	{
		KBFieldSpec *nSpec = newSpec.m_fldList.at(nidx) ;
		uint	    oColno ;

		if (nSpec->m_state == KB::RSDeleted ) continue  ;
		if (nSpec->m_state == KB::RSInserted) continue  ;

		if ((oColno = nSpec->m_colno) >= oldSpec.m_fldList.count ())
					      continue  ;

		KBFieldSpec *oSpec = oldSpec.m_fldList.at(oColno) ;

		select	+= sep + dbLink.mapExpression(oSpec->m_name) ;
		insert	+= sep + dbLink.mapExpression(nSpec->m_name) ;
		place	+= sep + dbLink.placeHolder  (slot) ;
		sep	 = ", " ;
		slot	+= 1	;
	}

	select	+= " from "	;
	select	+= dbLink.mapExpression(tabName) ;
	insert	+= ") values ("	;
	insert	+= place	;
	insert	+= ")"		;

	KBSQLSelect	   *qrySelect	;
	KBSQLInsert	   *qryInsert	;

	/* Rock and roll. Run the query statement to retrieve all the	*/
	/* values to be copied. Note that this will also give access	*/
	/* to a set of column types.					*/
	if ((qrySelect = dbLink.qrySelect (false, select)) == 0)
	{	setError (dbLink.lastError()) ;
		return	 false     ;
	}
	if ((qryInsert = dbLink.qryInsert (false, insert, tabName)) == 0)
	{	setError (dbLink.lastError()) ;
		delete	 qrySelect ;
		return	 false	   ;
	}
	if (!qrySelect->execute (0, 0))
	{	setError (qrySelect->lastError()) ;
		delete	 qrySelect ;
		delete	 qryInsert ;
		return	 false	   ;
	}

	uint	nvals	= qrySelect->getNumFields() ;
	KBValue*	values	= new KBValue[nvals] ;

	for (uint qrow = 0 ; qrySelect->rowExists (qrow) ; qrow += 1)
	{
		for (uint qcol = 0 ; qcol < nvals ; qcol += 1)
			values[qcol] = qrySelect->getField (qrow, qcol) ;

		if (!qryInsert->execute (nvals, values))
		{	setError (qryInsert->lastError ()) ;
			delete	 qrySelect ;
			delete	 qryInsert ;
			delete [] values;
			return	 false	   ;
		}
		if (qryInsert->getNumRows () != 1)
		{	setError
			(	KBError::Error,
				QString(TR("Unexpectedly inserted %1 rows")).arg(qryInsert->getNumRows()),
				qryInsert->getSubQuery(),
				__ERRLOCN
			)	;
			delete	 qryInsert ;
			delete	 qrySelect ;
			delete [] values;
			return	 false	   ;
		}
	}

	delete	qrySelect ;
	delete	qryInsert ;
	delete [] values;

	/* Time now to rename the tables. Rename the old table out of	*/
	/* the way, then rename the new table into place. This order	*/
	/* gives some sanity in the face of an error; if all is OK	*/
	/* then finally delete the original table.			*/
	if (!dbLink.renameTable ((cchar *)tabName, oldName, false))
	{	setError (dbLink.lastError()) ;
		return	 false   ;
	}
	if (!dbLink.renameTable (tmpName, (cchar *)tabName, false))
	{	setError (dbLink.lastError()) ;
		return	 false   ;
	}
	if (!dbLink.dropTable  (oldName, false))
	{	setError (dbLink.lastError()) ;
		return	 false   ;
	}

	return	true	;
}

/*  KBQryDesign								*/
/*  syncAll	: Synchronise all row					*/
/*  qlvl	: uint		: Query level				*/
/*  pValue	: KBValue *	 : Parent value if any			*/
/*  cExpr	: const QString &: Child expression			*/
/*  block	: KBBlock *	 : Requesting block			*/
/*  (returns)	: bool		 : Success				*/

bool	KBQryDesign::syncAll
	(	uint		qlvl,
		KBValue		*,
		const QString	&,
		KBBlock		*
	)
{

	/* ::SyncAll here maps to altering the existing table to match	*/
	/* the new specification. The strategy (unless we are creating	*/
	/* a completely new table) is to create a new table, copy the	*/
	/* contents of the old one over, and then to rename. Note that	*/
	/* some databases, such as MySQL, support this directly, but	*/
	/* (a) others do not and (b) doing it here gives us control	*/
	/* over type conversions.					*/
	if (qlvl != 0)
		return	true	;

	if (!saveRow (0, curQRow))
		return	false	;

	/* NOTE: Notwithstanding the above, we might want to have the	*/
	/*	 option to move this into the driver.			*/

	bool		sDelta	= newSpec.m_fldList.count() !=
				  oldSpec.m_fldList.count() ;

	KBTableSpec	create	(QString::null) ;

//	for (uint nidx = 0 ; nidx < newSpec.fldList.count() ; nidx += 1)
//	{	fprintf	(stderr, "%3d: ", nidx ) ;
//		newSpec.fldList.at(nidx)->Printf(stderr) ;
//		fprintf	(stderr, "\n") ;
//	}
	
	/* Scan the new specification table, generating a further	*/
	/* specification which skips those columns that have been	*/
	/* deleted. In the process, check:				*/
	/*	(a) All field names are set				*/
	/*	(b) All field types are set				*/
	/*	(c) There is exactly one primary key			*/
	int	pkeyCol	= -1 ;
	uint	nFields	= newSpec.m_fldList.count() ;

	for (uint nidx1 = 0 ; nidx1 < nFields ; nidx1 += 1)
	{
		KBFieldSpec *spec = newSpec.m_fldList.at (nidx1) ;

		if (spec->m_state == KB::RSDeleted)
		{	sDelta	 = true ;
			continue ;
		}

		/* See if this column has changed. Note that if the	*/
		/* number of columns has changed, sDelta will already	*/
		/* be true, so we don't have to worry about checking	*/
		/* the number of columns in the extant table.		*/
		if (!sDelta)
			if (! (*oldSpec.m_fldList.at(nidx1) == *spec))
				sDelta	= true ;

		bool	noName	= spec->m_name    .isEmpty() ;
		bool	noType	= spec->m_typeName.isEmpty() ;

		if (nidx1 == nFields - 1)
			if (noName && noType)
				break	;

		if (noName)
		{	setError
			(	KBError::Warning,
				TR("One or more columns does not have a name set"),
				__ERRLOCN
			)	;
			return	false ;
		}
		if (noType)
		{	setError
			(	KBError::Warning,
				TR("One or more columns does not have a type set"),
				__ERRLOCN
			)	;
			return	false ;
		}

		if ((spec->m_flags & KBFieldSpec::Primary) != 0)
		{
			if (pkeyCol >= 0)
			{	setError
				(	KBError::Warning,
					TR("Only one primary key can be specified"),
					__ERRLOCN
				)	;
				return	 false ;
			}
			pkeyCol	= nidx1 ;
		}

		create.m_fldList.append (new KBFieldSpec (*spec)) ;
	}

	if (create.m_fldList.count() == 0)
	{
		setError
		(	KBError::Warning,
			TR("No columns defined!"),
			__ERRLOCN
		)	;
		return	 false ;
	}

	if (pkeyCol < 0)
		if (TKMessageBox::questionYesNo
	   	(	0,
			TR("No primary key column: create/update table anyway?"),
			TR("No primary key")
		)
		!= TKMessageBox::Yes) return false ;


	/* We only ned to do all the create and copy stuff if the table	*/
	/* definition has changed. Otherwise, we just have changes to	*/
	/* the design information.					*/
	if (sDelta)
	{
		tmpName	= dbLink.rekallPrefix(QString("T_%1_%2").arg(time(0) % 1000000).arg(tabName)) ;
		oldName	= dbLink.rekallPrefix(QString("O_%1_%2").arg(time(0) % 1000000).arg(tabName)) ;

		dbLink.dropTable (tmpName, true) ;

		/* Create the table. In the event of an error, issue a	*/
		/* notification that there has been a general change	*/
		/* in the server tables, since we cannot be sure	*/
		/* exactly what the state is.				*/
		create.m_name    = isNew ? tabName : tmpName ;
		create.m_prefKey = pkeyCol ;

		if (!dbLink.createTable (create, isNew))
		{	setError (dbLink.lastError()) ;
			finish	 (false) ;
			return	 false   ;
		}


		/* If this is not a new table then we need to copy the	*/
		/* data from the original table and then to rename it	*/
		/* into place.						*/
		if (!isNew)
			if (!copyOldData ())
			{	finish	(false)	;
				return	false	;
			}

		/* OK, done: the table is now _definitely_ not new!	*/	
		isNew	= false	;
	}


	/* If there is any table information then update it with the	*/
	/* new set of columns, and save the document.			*/
	if (tabInfo != 0)
	{
		for (uint nidx1 = 0 ; nidx1 < newSpec.m_fldList.count() ; nidx1 += 1)
			designInfo.at(nidx1)->setColumnName(newSpec.m_fldList.at(nidx1)->m_name) ;

		designInfo.setAutoDelete (false) ;
		tabInfo->update (designInfo)	 ;
		designInfo.clear	 ()	 ;
		designInfo.setAutoDelete (true ) ;

		KBError	error ;
		if (!tabInfo->save
			(	getDocRoot()->getDBInfo(),
				server.getValue(),
				error,
				true
			))
		{
			setError (error) ;
			finish	 (true ) ;
			return	 false	 ;
		}
	}


	finish	(true)	;
	return	true	;
}


bool	KBQryDesign::getSelect
	(	uint		,
		KBSelect	&
	)
{
	return	true	;
}


/*  KBQryDesign								*/
/*  select	: Execute associated select statement			*/
/*  qlvl	: uint		: Query level				*/
/*  pValue	: KBValue *	: Parent value if any			*/
/*  cExpr	: const QString&: Child expression			*/
/*  filter	: const QString&: Filter if any				*/
/*  sorting	: const QString&: Sorting if any				*/
/*  query	: bool		: True to add query terms		*/
/*  qrow	: unsigned int	: Query row number			*/
/*  (returns)	: bool		: Success				*/

bool	KBQryDesign::select
	(	uint		qlvl,
		KBValue		*,
		const QString	&,
		const QString	&,
		const QString	&,
		bool		,
		unsigned int	
	)
{
	if (qlvl != 0) return true ;

	/* Get all the information about the table. Start by getting	*/
	/* a set of field specifications; note that we get two sets so	*/
	/* that we can sort out which columns are inserted, deleted or	*/
	/* changed.							*/
	oldSpec.reset	(tabName) ;
	newSpec.reset	(tabName) ;

	QStringList	typeList1 = QStringList::split ("|", dbLink.listTypes()) ;
	QStringList	typeList2 ;

	for (uint idx = 0 ; idx < typeList1.count() ; idx += 1)
	{	QString	type	= typeList1[idx]  ;
		int	sptr	= type.find (',') ;
		if (sptr >= 0) type = type.left (sptr) ;
		typeList2.append (type) ;
	}

	QString		typeList3 = typeList2.join("|") ;

	fieldType->setData (0, (void *)(cchar *)(typeList3)) ;
	curQRow	= 0	;

	if (isNew) return true ;

	if ( !dbLink.listFields (oldSpec) ||
	     !dbLink.listFields (newSpec) )
	{
		setError (dbLink.lastError()) ;
		return	 false	;
	}


	designInfo.clear () ;

	LITER
	(	KBFieldSpec,
		newSpec.m_fldList,
		fSpec,

		if (tabInfo != 0)
			designInfo.append (new KBTableColumn (tabInfo->getColumn (fSpec->m_name))) ;
		else	designInfo.append (new KBTableColumn) ;
	)

	return	true	;
}

/*  KBQryDesign								*/
/*  newRowEmpty	: Check if new row at end of query is empty		*/
/*  qlvl	: uint		: Query level				*/
/*  qrow	: uint		: Row being checked			*/
/*  (returns)	: bool		: Row is empty				*/

bool	KBQryDesign::newRowEmpty
	(	uint	,
		uint	qrow
	)
{
	if (!fieldName  ->isEmpty(qrow)) return false ;
	if (!fieldType  ->isEmpty(qrow)) return false ;
	if (!fieldNullOK->isEmpty(qrow)) return false ;
	if (!fieldLen   ->isEmpty(qrow)) return false ;
	return	true ;
}

/*  KBQryDesign								*/
/*  rowIsDirty	: See if specified row is dirty				*/
/*  qlvl	: uint		: Query level				*/
/*  qrow	: uint		: Query row number			*/
/*  reset	: bool		: True to reset dirty flag		*/
/*  (returns)	: bool		: Row is dirty				*/

bool	KBQryDesign::rowIsDirty
	(	uint	qlvl,
		uint	qrow,
		bool	reset
	)
{
	if (qlvl == 0)
	{
		KBFieldSpec *spec = newSpec.m_fldList.at(qrow) ;
		if (spec == 0) return false ;
		bool rc	= spec->m_dirty ;
		if (reset) spec->m_dirty = false ;
		return rc ;
	}

	return	true ;
}

/*  KBQryDesign								*/
/*  setCurrentRow: Set current query row				*/
/*  qlvl	 : uint		: Query level				*/
/*  qrow	 : uint		: Query row number			*/
/*  (returns)	 : uint		: Total rows here and below		*/

uint	KBQryDesign::setCurrentRow
	(	uint	qlvl,
		uint	qrow
	)
{
	if (qlvl == 0) curQRow = qrow ;
	return	1 ;
}

/*  KBQryDesign								*/
/*  getPermission: Get permission flags					*/
/*  qlvl	 : uint		: Query level number			*/
/*  (returns)	 : uint		: Permission flags			*/

uint	KBQryDesign::getPermission
	(	uint	qlvl
	)
{
	return	qlvl == 0 ? QP_SELECT|QP_INSERT|QP_UPDATE|QP_DELETE :
			    QP_SELECT|QP_UPDATE ;
}

/*  KBQryDesign								*/
/*  (returns)	: bool		: Success				*/

bool	KBQryDesign::propertyDlg ()
{
	return	false	;
}

/*  KBQryDesign								*/
/*  qlvl	: uint		: Query level				*/
/*  getFieldList: Get list of fields from database			*/
/*  fldList	: QList<KBFieldSpec>&					*/
/*				: Field list result			*/
/*  pKey	: int &		: Primary key index			*/
/*  (returns)	: bool		: Success				*/

bool	KBQryDesign::getFieldList
	(	uint			,
		QList<KBFieldSpec>	&,
		int			&
	)
{
	/* This method is used in a real query from the properties	*/
	/* dialogs, which we do not have for a table query. Hence, fail	*/
	/* always.							*/
	return	false	;
}


void	KBQryDesign::sortByColumn
	(	uint			,
		uint			,
		bool			,
		KBItem			*
	)
{
}

bool	KBQryDesign::startUpdate
	(	uint			,
		uint			,
		KBQryBase::Locking
	)
{
	return	true	;
}

bool	KBQryDesign::endUpdate
	(	uint			,
		bool
	)
{
	return	true	;
}

KBQryBase::Locking
	KBQryDesign::lockingState
	(	uint
	)
{
	return	KBQryBase::NoLocking ;
}

NEWNODE(QryDesign, (cchar *)0, KF_FORM) 
