# Copyright (c) 2002 Red Hat, Inc. All rights reserved.
#
# This software may be freely redistributed under the terms of the
# GNU General Public License.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# Cluter configuration dialogs
#
# Copyright 2001, 2002, Red Hat, Inc.
#
# Component of: Red Hat Database GUI Administration tool

package require Tk
package require Itcl
package require Itk
package require tkBits
package require clusterCollectionListbox
package require connectionConfigDialog
package require mainMisc

package provide clusterConfiguration 1.0

## public functions

# stand-alone add cluster
proc cluster_add {} {
    global clusterCollectionName
    global cluster_editor_result

    set w .configCluster

    create_cluster_editor $w "Add A New Cluster" $clusterCollectionName
    show_cluster_editor $w
    destroy_cluster_editor $w

    if {$cluster_editor_result(ok)} {
	# Add it
	$clusterCollectionName registerCluster \
	    $cluster_editor_result(newname) $cluster_editor_result(arglist)

	writeClusterSettings $clusterCollectionName
	refreshHierbox

	return 1
    }

    return 0
}

# stand-alone remove cluster; no confirmation is done
proc cluster_remove {clusterName} {
    global clusterCollectionName

    # If the cluster to delete exists and if the current view belongs
    # to that cluster, issue a hyperlink move request to clear the
    # view.
    if {! [$clusterCollectionName hasCluster $clusterName]} {
	return 0
    }

    # Issue a hyperlink move request to clear the selector
    hyperlinker selectorMoveRequest ""

    $clusterCollectionName unregisterCluster $clusterName
    writeClusterSettings $clusterCollectionName
    refreshHierbox

    return 1
}

# stand-alone remove cluster; user is asked to confirm removal via a
# dialog box.
proc cluster_remove_confirm {clusterName} {
    if {[confirm_remove $clusterName]} {
	cluster_remove $clusterName
    }
}

# stand-alone configure cluster
proc cluster_modify {clusterName} {
    global clusterCollectionName
    global cluster_editor_result

    # does the cluster exist?
    if {! [$clusterCollectionName hasCluster $clusterName]} {
	return 0
    }

    set w .configCluster

    create_cluster_editor $w "Configuring $clusterName" $clusterCollectionName $clusterName
    show_cluster_editor $w
    destroy_cluster_editor $w

    if {$cluster_editor_result(ok)} {
	set oldName $cluster_editor_result(oldname)
	set newName $cluster_editor_result(newname)
	set argList $cluster_editor_result(arglist)

	# Issue a hyperlink move request to clear the selector
	hyperlinker selectorMoveRequest ""

	# modify cluster
	$clusterCollectionName modifyCluster $oldName $newName $argList
	
	writeClusterSettings $clusterCollectionName
	refreshHierbox

	return 1
    }

    return 0
}

# configure clusters dialog
proc configure_clusters { } {
    global clusterCollectionName myClusterCollectionName

    # work on a copy of clusterCollection
    set myClusterCollectionName [getUniqueName]
    MyClusterCollection $myClusterCollectionName $clusterCollectionName
    
    toplevel .configList

    wm title .configList "Configure Clusters"
    wm protocol .configList WM_DELETE_WINDOW close_prefs_win

    frame .configList.selectPane -relief groove -borderwidth 1
    
    ClusterCollectionListbox .configList.selectPane.clusterListbox \
	    -clusterCollectionName $myClusterCollectionName
    .configList.selectPane.clusterListbox setSelectionCommand \
		"if {\[llength \[$myClusterCollectionName getClusterList\]\] > 0} {
             .configList.selectPane.paneButtons.configureButton configure -state normal;
             .configList.selectPane.paneButtons.deleteButton configure -state normal
         }"
    .configList.selectPane.clusterListbox setDoubleClickCommand {update idletasks; after 100; preferences_configure}
    .configList.selectPane.clusterListbox setOnRepopulateCommand \
	".configList.selectPane.paneButtons.configureButton configure -state disabled;
         .configList.selectPane.paneButtons.deleteButton configure -state disabled"

    pack .configList.selectPane.clusterListbox -side left -expand 1 -fill both

    frame .configList.selectPane.paneButtons

    button .configList.selectPane.paneButtons.addButton \
		-text "Add" \
		-underline 0 \
	-command "update idletasks; preferences_add"
    button .configList.selectPane.paneButtons.configureButton \
		-text "Configure" \
		-underline 3 \
		-state disabled \
		-command preferences_configure
    button .configList.selectPane.paneButtons.deleteButton \
		-text "Remove" \
		-underline 0 \
		-state disabled \
		-command preferences_delete

    pack .configList.selectPane.paneButtons -side right

    pack .configList.selectPane.paneButtons.addButton  \
	    .configList.selectPane.paneButtons.configureButton \
	    .configList.selectPane.paneButtons.deleteButton -side top -fill x

    pack .configList.selectPane.paneButtons -side right

    pack .configList.selectPane -side top -expand 1 -fill both

    frame .configList.bottomPane
    
    button .configList.bottomPane.okButton \
		-text "OK" \
		-underline 0 \
		-default active \
		-command preferences_ok

    button .configList.bottomPane.cancelButton \
		-text "Cancel" \
		-underline 0 \
		-command preferences_cancel
    pack .configList.bottomPane.cancelButton .configList.bottomPane.okButton \
	-side right -expand 1

    pack .configList.bottomPane -side right

    # bindings
    bind .configList <Alt-o>  {preferences_ok; break}
    bind .configList <Alt-c>  {preferences_cancel; break}
    bind .configList <Return> {preferences_ok; break}
    bind .configList <Escape> {preferences_cancel; break}
    bind .configList <Alt-a>  {preferences_add; break}
    bind .configList <Alt-f>  {preferences_configure; break}
    bind .configList <Alt-r>  {preferences_delete; break}

    # modal
    grab .configList
    wm transient .configList .
    wm deiconify .configList
}

### private functions

proc preferences_add {} {
    global cluster_editor_result
    global myClusterCollectionName

    set w .configCluster

    create_cluster_editor $w "Add A New Cluster" $myClusterCollectionName
    show_cluster_editor $w .configList
    destroy_cluster_editor $w

    if {$cluster_editor_result(ok)} {
	# Add it
	$myClusterCollectionName registerCluster \
	    $cluster_editor_result(newname) \
	    $cluster_editor_result(arglist)

	.configList.selectPane.clusterListbox repopulateListbox
    } else {
	.configList.selectPane.paneButtons.configureButton configure -state disabled
	.configList.selectPane.paneButtons.deleteButton configure -state disabled
    }

    # hack fix for leftover highlighting of Add button
    .configList.selectPane.paneButtons.addButton configure -state normal

    grab .configList
    focus .configList
    wm transient .configList .
    update
}

proc preferences_configure {} {
    global cluster_editor_result
    global myClusterCollectionName

    if {![.configList.selectPane.clusterListbox haveSelection]} {
	return
    }

    set targetCluster [.configList.selectPane.clusterListbox getSelectionName]

    set w .configCluster

    create_cluster_editor $w "Configuring $targetCluster" $myClusterCollectionName $targetCluster
    show_cluster_editor $w .configList
    destroy_cluster_editor $w

    if {$cluster_editor_result(ok)} {
	set oldName $cluster_editor_result(oldname)
	set newName $cluster_editor_result(newname)
	set argList $cluster_editor_result(arglist)

	$myClusterCollectionName modifyCluster $oldName $newName $argList
	
	.configList.selectPane.clusterListbox repopulateListbox
    } else {
	.configList.selectPane.paneButtons.configureButton configure -state disabled
	.configList.selectPane.paneButtons.deleteButton configure -state disabled
    }

    grab .configList
    focus .configList
    wm transient .configList .
    update
}

proc preferences_delete {} {
    global myClusterCollectionName

    # If nothing selected, jump out
    if {![.configList.selectPane.clusterListbox haveSelection]} {
	return
    }

    set clusterName [.configList.selectPane.clusterListbox getSelectionName]

    if {[confirm_remove $clusterName .configList]} {
	$myClusterCollectionName unregisterCluster $clusterName
	.configList.selectPane.clusterListbox repopulateListbox
    }

    # regrab the list
    grab .configList
    wm transient .configList .
}

proc preferences_ok {} {
    global myClusterCollectionName clusterCollectionName

    # Destroy cluster configuration window
    after idle destroy .configList
    
    set addList [$myClusterCollectionName getAddList]
    set delList [$myClusterCollectionName getDeleteList]
    array set modArr [$myClusterCollectionName getModifyArray]

    # Issue a hyperlink move request to clear the selector
    hyperlinker selectorMoveRequest ""

    # Do pending deletes
    foreach x $delList {
	$clusterCollectionName unregisterCluster $x
    }

    # Do pending modifications
    foreach {n o} [array get modArr] {
	$clusterCollectionName modifyCluster $o $n \
	    [$myClusterCollectionName getArgListByCluster $n]
    }

    # Do pending additions
    foreach x $addList {
	$clusterCollectionName registerCluster \
	    $x \
	    [$myClusterCollectionName getArgListByCluster $x]
    }

    # Destroy our working clusterCollection
    ::itcl::delete object $myClusterCollectionName

    # Write out config here
    writeClusterSettings $clusterCollectionName

    # Refresh
    refreshHierbox

    return
}

proc preferences_cancel { } {
    global myClusterCollectionName

    # Destroy cluster configuration window
    destroy .configList

    # Destroy our working clusterCollection
    ::itcl::delete object $myClusterCollectionName

    return
}

proc close_prefs_win { } {
    if {[winfo exists .configCluster]} {
	destroy .configCluster
    }
    preferences_cancel
}

proc create_cluster_editor {win title clusterCollection {clusterName {}}} {
    global cluster_editor_result

    if {[winfo exists $win]} {
	destroy_cluster_editor $win
    }

    toplevel $win
    wm withdraw $win
    wm title $win $title
    wm protocol $win WM_DELETE_WINDOW "set cluster_editor_result(ok) 0"

    set cluster_editor_result(ok) 0
    set cluster_editor_result(oldname) $clusterName

    if {[string length $clusterName] == 0} {
	set argList {}
    } else {
	set argList [$clusterCollection getArgListByCluster $clusterName]
    }

    frame $win.frame -relief groove -borderwidth 1
    eval ConnectionConfigDialog $win.frame.configDialog $argList
    .configCluster.frame.configDialog setClusterName $clusterName

    pack $win.frame.configDialog -side top -fill x
    pack $win.frame -side top -fill x

    frame $win.buttons
    button $win.buttons.okButton \
	-text "OK" \
	-underline 0 \
	-command "validate_cluster_editor $win $clusterCollection"
    button $win.buttons.cancelButton \
	-text "Cancel" \
	-underline 0 \
	-command {set cluster_editor_result(ok) 0}

    bind $win <Alt-o>  "validate_cluster_editor $win $clusterCollection; break"
    bind $win <Alt-c>  {set cluster_editor_result(ok) 0; break}
    bind $win <Return> "validate_cluster_editor $win $clusterCollection; break"
    bind $win <Escape> {set cluster_editor_result(ok) 0; break}

    pack $win.buttons.cancelButton $win.buttons.okButton \
	-side right -fill y -expand 1
    pack $win.buttons -side right
}

proc show_cluster_editor {win {parent .}} {
    global cluster_editor_result

    update idletasks
    after 100
    update

    placeWindow $win $parent
    
    set old [focus -displayof $win]

    catch {grab $win}
    $win.frame.configDialog setFocus
    update idletasks
    wm transient $win $parent
    wm deiconify $win

    tkwait variable cluster_editor_result(ok)

    catch {grab release $win}
    catch {wm withdraw $win}
    focus $old
}

proc destroy_cluster_editor {win} {
    destroy $win
}

proc validate_cluster_editor {win clusterCollection} {
    global cluster_editor_result

    set rv [$win.frame.configDialog validConfiguration]
    set oldName $cluster_editor_result(oldname)
    set newName [$win.frame.configDialog getClusterName]

    if {$rv == 1} {
	tk_messageBox -icon error -title "Missing Cluster Name" \
	    -message "You must specify a cluster name." \
	    -parent $win
	return
    } elseif {$rv == 2} {
	tk_messageBox -icon error -title "Missing Port" \
	    -message "You must specify a port number." \
	    -parent $win
	return
    } elseif {$rv == 4} {
	tk_messageBox -icon error -title "Invalid Cluster Name" \
	    -message "Cluster names can only contain letters, digits, '.', '_', and '-'." \
	    -parent $win
	return
    } elseif {$rv == 5} {
	tk_messageBox -icon error -title "Invalid Host Name" \
	    -message "Host names can only contain letters, digits, '.', '_', and '-'." \
	    -parent $win
	return
    } elseif {$rv == 6} {
	tk_messageBox -icon error -title "Missing Host Name" \
	    -message "You must specify a host name." \
	    -parent $win
	return
    } elseif {$rv == 7} {
	tk_messageBox -icon error -title "Invalid Cluster Name" \
	    -message "Cluster names may not begin with a digit or a '-'." \
	    -parent $win
	return
    } elseif {$rv == 8} {
	tk_messageBox -icon error -title "Invalid Unix Domain Socket Path" \
	    -message "Unix domain socket path is invalid." \
	    -parent $win
	return
    } elseif {$rv == 9} {
	tk_messageBox -icon error -title "Missing User Name" \
	    -message "You must specify a user name." \
	    -parent $win
	return
    }

    if {$newName != $oldName} {
	if {[$clusterCollection hasCluster $newName]} {
	    tk_messageBox -icon error -title "Can't set cluster name" \
		-message "A cluster named '$newName' already exists. Cluster names must be unique" \
		-parent $win
	    return
	}
    }

    # everything is ok
    set cluster_editor_result(newname) $newName
    set cluster_editor_result(arglist) [$win.frame.configDialog getArgumentString]
    set cluster_editor_result(ok) 1
}

# Bring up a removal confirmation dialog box. Return 1 if the user
# confirms the removal and 0 if they don't.
proc confirm_remove {clusterName {parent . }} {
    set sure [tk_messageBox \
		  -title "Remove Cluster $clusterName" \
		  -message "Are you sure you want to remove cluster $clusterName?" \
		  -icon question \
		  -type yesno \
		  -default no \
		  -parent $parent]

    switch -- $sure {
	yes {
	    return 1
	}

	no {
	    return 0
	}
	
	default {
	    return 0
	}

    }
}
    

# A collection of database clusters. Only meant for working with
# clusters when doing cluster configuration as semaphores aren't used
# and refreshDispatcher stuff is ignored.
itcl::class MyClusterCollection {
    constructor {args} {
	set addList {}
	set removeList {}
	array set modifyArray {}

	# Check for copy constructor passed in

	if {![catch {$args isa ClusterCollection}]} {
	    copy $args
	    return
	}

	# Otherwise parse the args
	
	foreach { option value } $args {
	    switch -- $option {
		"-refreshDispatcher" {
		    # ignore
		}

		default {
		    puts stderr "Invalid option to clusterCollection constructor: $option with value $value"
		}
	    }
	}
	
	set clusterList {}
	set argList {}
    }

    private method copy {object} {
	set argList [$object getArgList]
	set clusterList [$object getClusterList]
    }
    
    public method registerCluster {clusterName clusterArgs} {
	# Is there already a cluster by this name?
	set index [lsearch -exact $clusterList $clusterName]

	if {$index != -1} {
	    error "clusterCollection::registerCluster - Cluster already exists"
	}

	lappend clusterList $clusterName
	lappend argList $clusterArgs

	# Add new cluster to addList
	lappend addList $clusterName
    }

    public method unregisterCluster {clusterName} {
	# Does clusterName exist?
	set index [lsearch -exact $clusterList $clusterName]

	if {$index == -1} {
	    error "clusterCollection::unregisterCluster - Invalid name reference"
	}

	# Remove from arg and cluster lists
	set argList [lreplace $argList $index $index]
	set clusterList [lreplace $clusterList $index $index]

	# Is the cluster to remove a new one?
	# Yes - remove from addList
	# No  - add to removeList
	set index [lsearch -exact $addList $clusterName]
	if {$index != -1} {
	    set addList [lreplace $addList $index $index]
	} else {
	    # If we're removing an old array, check to see if it has
	    # been modified. If yes, remove from modifyArray.
	    set l [array get modifyArray $clusterName]
	    if {[llength $l] > 0} {
		set oldName [lindex $l 1]
		array unset modifyArray $clusterName
	    } else {
		set oldName $clusterName
	    }
	    
	    lappend removeList $oldName
	}
    }

    # Method to modify the name of a cluster

    public method modifyCluster {oldClusterName newClusterName clusterArgs} {
	set index [lsearch -exact $clusterList $oldClusterName]
	set clusterList [lreplace $clusterList $index $index $newClusterName]
	set argList [lreplace $argList $index $index $clusterArgs]

	# Is the cluster that was modified a new one?
	# Yes - Replace old name in addList with new name
	# No  - If the cluster has been modified before, remove the
	#       old mapping in modifyArray. Add a new
	#        newClusterName -> oldClusterName
	#      mapping to modifyArray.
	set index [lsearch -exact $addList $oldClusterName]
	if {$index != -1} {
	    set addList [lreplace $addList $index $index $newClusterName]
	} else {
	    set old [array get modifyArray $oldClusterName]
	    if {[llength $old] > 0} {
		set oldname [lindex $old 1]
		array unset modifyArray $oldClusterName
	    } else {
		set oldname $oldClusterName
	    }
	    array set modifyArray [list $newClusterName $oldname]
	}
    }

    public method hasCluster {clusterName} {
	set index [lsearch -exact $clusterList $clusterName]
	if {$index == -1} {
	    return 0
	} else {
	    return 1
	}
    }

    public method getClusterList {} {
	return $clusterList
    }

    public method getArgList {} {
	return $argList
    }

    public method getAddList {} {
	return $addList
    }

    public method getDeleteList {} {
	return $removeList
    }

    public method getModifyArray {} {
	return [array get modifyArray]
    }

    public method getArgListByCluster { clusterName } {
	set index [lsearch -exact $clusterList $clusterName]

	if {$index == -1} {
	    error "clusterCollection::getArgListByCluster - Invalid name reference"
	}

	return [lindex $argList $index]
    }

    public method getClusterObject {name} {
	set index [lsearch -exact $clusterList $name]

	if {$index == -1} {
	    error "clusterCollection::getClusterObject - Invalid name reference"
	}

	return [uplevel Cluster #auto $name $this [lindex $argList $index]]
    }

    private variable argList
    private variable clusterList
    private variable addList
    private variable removeList
    private variable modifyArray
}
