# Copyright (c) 2002 Red Hat, Inc. All rights reserved.
#
# This software may be freely redistributed under the terms of the
# GNU General Public License.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# TCL code to message box to confirm DELETE object and execute the DELETE
#
# Written for Red Hat Inc. by Permaine Cheung <pcheung@redhat.com>
#
# Component of: Red Hat Database GUI Administration tool

package require Tk
package require Itcl
package require Itk
package require Iwidgets
package require cluster
package require aggregateHandle
package require columnHandle
package require constraintHandle
package require databaseHandle
package require dataTypeHandle
package require functionHandle
package require groupHandle
package require indexHandle
package require ruleHandle
package require sequenceHandle
package require tableHandle
package require triggerHandle
package require userHandle
package require viewHandle
package require destructiveOperationHandler
package provide deleteMessageRaiser 1.0

proc deleteMessageRaise { args } {

    global setting_askConfirmations

    set expectedArgs \
	[list \
	-clusterCollectionName\
	-clusterName\
	-uniqueConstraintIdentifyingName\
	-databaseName\
	-tableName\
	-viewName\
	-deleteName \
	-objectType \
	-objectHandle \
	-refreshUpperLevelCommand \
	-refreshAnotherLevelCommand\
	-destructiveAction \
	-optionalIdentificationData \
	]

    foreach option $expectedArgs {
	set options($option) ""
    }

    foreach { option value } $args {
	if {[lsearch -exact $expectedArgs $option] == -1} {
	    error "Unexpected option \"$option\" - only expect $expectedArgs"
	}
	set options($option) "$value"
    }
 
    eval "DeleteOperationsHandler delOpHandle $args"

    # Check if we are supposed to ask for confirmations. If not, just assume it is a yes.

    if {$setting_askConfirmations == 1} {

        # Msg if user is deleteing an object

	if {$options(-objectType) != "dataType"} {
	    set objectType $options(-objectType)
	} else {
	    set objectType "type"
	}

	# In order to make sure we have the correct name, we always
	# pass it around in escaped form (special characters escaped).
	# However, when displaying the name in the title or as a 
	# message, this name should not appear as we pass it around
	# i.e. escaped. Unescape the name

	set deleteName [join [split $options(-deleteName) \\] ""]

        if {[string range $options(-destructiveAction) 0 5] == "delete"} {
	    set sure [tk_messageBox -icon question -type yesno -default no\
			  -title "DROP [string toupper $objectType] $deleteName" \
			  -message "Are you sure you want to drop $objectType $deleteName?"]

	} else {    # Msg if user is deleting some part of an object
	    set sure [tk_messageBox -icon question -type yesno -default no\
			  -title "DROP [string toupper $objectType] $deleteName" \
			  -message "Are you sure you want to drop the $objectType on $deleteName?"]
	}

    } else { # Else we shouldn't. Assume its a yes.
	set sure yes
    }

    # We detect errors from the backend by checking the return value. However
    # an internal tcl error might occur when deleting. Rather than showing a 
    # stack trace, we use a catch statement to get this error and show it to 
    # the user so that the error message is helpful.

    set errorLevel [catch {
	switch -- $sure {
	    yes {

		# Start the busy indicator

		status loadNotify 1
		
		switch -- $options(-destructiveAction) {
		    "deleteAggregate" {
			set errorText [delOpHandle dropAggregate]
		    }
		    "deleteConstraint" {
			set errorText [delOpHandle dropConstraint]
		    }
		    "deleteDatabase" {
			set errorText [delOpHandle dropDatabase]
		    }
		    "deleteDataType" {
			set errorText [delOpHandle dropDataType]
		    }
		    "deleteFunction" {
			set errorText [delOpHandle dropFunction]
		    }
		    "deleteGroup" {
			set errorText [delOpHandle dropGroup]
		    }
		    "deleteIndex" {
			set errorText [delOpHandle dropIndex]
		    }
		    "deleteLanguage" {
			set errorText [delOpHandle dropLanguage]
		    }
		    "deleteOperator" {
			set errorText [delOpHandle dropOperator]
		    }
		    "deleteRule" {
			set errorText [delOpHandle dropRule]
		    }
		    "deleteSequence" {
			set errorText [delOpHandle dropSequence]
		    }
		    "deleteTable" {
			set errorText [delOpHandle dropTable]
		    }
		    "deleteTrigger" {
			set errorText [delOpHandle dropTrigger]
		    }
		    "deleteUser" {
			set errorText [delOpHandle dropUser]
		    }
		    "deleteView" {
			set errorText [delOpHandle dropView]
		    }
		    "commentDrop" {
			set errorText [delOpHandle dropComment]
		    }
		    default {
			error "No associated function to drop $options(-objectType)"
		    }
		}

		# Lower the busy indicator

		status loadNotify 0
		
		# Check if everything went cleanly

		if {$errorText != "" } {

		    # No, so display error message

		    tk_messageBox -icon error -title "Error" \
 			-message "The backend returned the error:\n $errorText"
		}
	    }

	    no {
		# Do nothing
	    }  

	    default {
		error "Valid inputs are YES or NO"
	    }  
	}
    } internalErrorText]

    if {$errorLevel != 0} {
	tk_messageBox -title "Error" -icon error -message "An error occurred while performing the drop: $internalErrorText\nThe object was not dropped."
	
	# Set the error text variable to refresh doesn't get called.

	set errorText $internalErrorText
    }

     catch {
 	if {$errorText == "" } {
 	    # Refresh if everything went well
 	    eval eval $options(-refreshUpperLevelCommand)    
 	}
     }


    # No matter what hapened, make sure we delete what we created!
    itcl::delete object delOpHandle
}
