# Copyright (c) 2002 Red Hat, Inc. All rights reserved.
#
# This software may be freely redistributed under the terms of the
# GNU General Public License.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# Miscellaneous helper functions called by main program body and by menu
# routines
#
# Written for Red Hat Inc. by Neil Padgett <npadgett@redhat.com>
#
# Component of: Red Hat Database GUI Administration tool

package require preferences
package provide mainMisc 1.0

global configDir
set configDir "$env(HOME)/.rhdb"
set clusterInfoFilename "clusterCollectionSettings.cfg"

# Check whether the install dir exists
# Returns:
# 0 - Exists and is a directory
# 1 - Exists but isn't a directory
# 2 - Doesn't exist (or we can't see it)

proc configDirExists {} {
    global configDir

    # Check for the file

    if {[file exists $configDir]} {
	# We know the directory file exists

	if {[file isdirectory $configDir]} {
	    # It is a directory
	    
	    return 0
	} else {
	    return 1
	}
    }

    return 2
}

proc createConfigDir {} {
    global configDir

    if { [configDirExists] == 2 } {
	file mkdir $configDir
    } else {
	error "mainMisc createInstallDir: Didn't do create, since configDirExists reports a file collision."
    }
}

proc readClusterSettings { clusterCollectionName } {
    global configDir
    global clusterInfoFilename

    if { [$clusterCollectionName streamIn \
	    [file join $configDir $clusterInfoFilename]] == -1 } {
	
	# Read failed so we might not have a directory

	puts stderr "mainMisc readClusterSettings: streamIn failed -- this might be your first time running the program. \nTrying to resolve:"

	set dirStatus [configDirExists]

	if { $dirStatus == 1 } {
	    puts stderr "You have a file colliding with the config directory name. Please remove $configDir and try again."
	    exit
	} {
	    if { $dirStatus == 2} {
		puts stderr "No config dir -- creating one" 
		createConfigDir
		puts stderr "Now streaming an empty config file"
		$clusterCollectionName streamOut \
			[file join $configDir $clusterInfoFilename]
		puts stderr "Done."
	    } else {
		# Must be dir status zero
		puts stderr "Directory seems to be in order. Maybe there isn't a config file yet."
		puts stderr "Now streaming an empty config file"
		$clusterCollectionName streamOut \
			[file join $configDir $clusterInfoFilename]
		puts stderr "Done."
	    }
	}
    }
	
    # Settings read ok, so acquire the configuration lockfile, 
    # if possible

    if { ![grabLockFile] } {
	global applicationName

	tk_messageBox -icon warning -title "Warning" \
		-message "A lock file was found for $applicationName.\n\nThis may indicate that there is another running instance of $applicationName, or it could be an old lock file left over from an instance that exited prematurely.\nIf you believe the file remains from an instance of $applicationName then, after verifying that there are no running instances of $applicationName, you should delete the lock file found in [file join $configDir .rhdb-admin.lock].\n\nAs, due to the existing lock file, it did not acquire the lock file, this instance of $applicationName will not save configuration settings changes on exit."
    }
}

proc writeClusterSettings { clusterCollectionName } {
    global configDir
    global clusterInfoFilename

    # If we have the lockfile, write the configuration file

    if { [haveLockFile] } {
	if {[$clusterCollectionName streamOut \
		[file join $configDir $clusterInfoFilename]] == -1 } {
	    tk_messageBox -icon error \
		    -message "mainMisc writeClusterSettings: Error writing config file."
	}
    }
}

# Any random last minute clean-up on program exit should go here 

proc cleanUp {} {
    global applicationName

    # Create a replacement window to show while cleaning up
    
    toplevel .tchau
    wm withdraw .tchau
    wm title .tchau "$applicationName"
    label .tchau.byebyeText -text "$applicationName is shutting down..." \
        -height 5 -width 50
    pack .tchau.byebyeText -expand yes -fill both
    update idletasks
    set x [expr ([winfo screenwidth .]-[winfo reqwidth .tchau]) / 2]
    set y [expr ([winfo screenheight .]-[winfo reqheight .tchau]) / 2]
    wm geometry .tchau "+$x+$y"
    update

    # unmap the main window and show the place holder one
        
    wm withdraw .
    wm deiconify .tchau
    update
    raise .tchau

    # Save the current session's preferences to the
    # configuration file

    savePreferences

    releaseLockFile
    exit
}

proc refreshHierbox {} {

    # Do the refresh

    refresher refresh

}

# End of file




