# Copyright (c) 2002 Red Hat, Inc. All rights reserved.
#
# This software may be freely redistributed under the terms of the
# GNU General Public License.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# Save/Read user preferences to/from a configuration file
#
# Written for Red Hat Inc. by Benjamin Mar Kuck <bmarkuck@redhat.com>
#
# Component of: Red Hat Database GUI Administration tool

package require Tk
package require Itcl
package require Itk
package provide preferences 1.0

set preferencesFilename "rhdb-admin-preferences.cfg"

# Save the user's current session's preferences to a
# configuration file

proc savePreferences {} {
    global setting_showSystemObjects
    global setting_askConfirmations
    global setting_logSQLData
    global lockfile_haveLock
    global configDir
    global preferencesFilename

    # An indicator as to whether or not the configuration
    # file is formatted correctly

    global configFileOk

    # Only save the preferences if we have the lockfile
    # and the configuration file was read in ok at
    # program start-up

    if { ($lockfile_haveLock == 1) && ($configFileOk == 1) } {
        
	# Open the configuration file for output 

        if {[catch {open [file join $configDir $preferencesFilename] w 0600} fileId]} {
            puts stderr "Cannot open file [file join $configDir $preferencesFilename] for saving preferences: $fileId"
        
	} else {

            # Write out the preferences to a
            # configuration file
            # Note that the preferences are not intended
            # to be in any particular order

            # Write out the setting for showing
            # system objects

            puts $fileId "configure setting_showSystemObjects $setting_showSystemObjects"

            # Write out the setting for asking for
            # confirmations

            puts $fileId "configure setting_askConfirmations $setting_askConfirmations"

            # Write the setting for logging sql data

            puts $fileId "configure setting_logSQLData $setting_logSQLData"

            close $fileId
        }
    }
}

# Read the user's preferences from a configuration file

proc readPreferences {} {
    global leftcs
    global rightcs
    global setting_showSystemObjects
    global setting_askConfirmations
    global setting_logSQLData
    global configDir
    global preferencesFilename

    # An indicator as to whether or not the configuration
    # file is formatted correctly

    global configFileOk
    set configFileOk 1

    # If the configuration file cannot be read (e.g. this
    # is the user's first time starting rhdb-admin), start
    # up with the default preferences
    # Otherwise, start up with the user's preferences, as
    # saved in the configuration file

    if { [catch {open [file join $configDir $preferencesFilename] r} fileId] == 0 } {

        set preferences [read $fileId]
        close $fileId

        # Set up a safe interpreter to set the preferences

        set confInterp [interp create -safe]
        $confInterp alias configure setPreference

        # Catch improper format for the configuration file
        # Note that all error messages raised in
        # setPreference are subsequently caught here, so
        # only the error message brought up below will be
        # displayed

        if { [catch {$confInterp eval $preferences}] == 1 } {

            # The expected list of preferences that can be set

            set expectedPrefs \
                [list setting_showSystemObjects setting_askConfirmations setting_logSQLData]

            tk_messageBox -icon warning -title "Invalid Configuration File" \
                -message "The configuration file [file join $configDir $preferencesFilename] is formatted incorrectly.\nEach line must be of the following form:\n\nconfigure <preference name> <value>\n\nwhere <value> is either 0 or 1.\nThe supported preference names are:\n\n[join $expectedPrefs "\n"]\n\nDue to the invalid configuration file, some preference settings may not have been loaded, and the configuration settings will not be saved on exit."

             # Remember that configuration file not read
             # in ok so we don't overwrite the file upon
             # exit, and then kick out of setting the
             # rest of the preferences

             set configFileOk 0
             return
        }

        # Clean up the safe interpreter

        interp delete $confInterp
    }
}

# Set the preference setting to the specified value

proc setPreference { setting value } {
    global leftcs
    global rightcs
    global setting_showSystemObjects
    global setting_askConfirmations
    global setting_logSQLData
    global preferencesFilename

    # An indicator as to whether or not the configuration
    # file is formatted correctly

    global configFileOk

    # The expected list of preferences that can be set

    set expectedPrefs \
        [list setting_showSystemObjects setting_askConfirmations setting_logSQLData]

    # Handle the expected preferences

    if { [lsearch $expectedPrefs $setting] != -1 } {

        # Catch improper value for preference

        if { $value != 0 && $value != 1 } {

            tk_messageBox -icon warning -title "Invalid Configuration File" \
                -message "The configuration file [file join $configDir $preferencesFilename] is formatted incorrectly.\nEach line must be of the following form:\n\nconfigure <preference name> <value>\n\nwhere <value> is either 0 or 1.\nThe supported preference names are:\n\n[join $expectedPrefs "\n"]\n\nDue to the invalid configuration file, some preference settings may not have been loaded, and the configuration settings will not be saved on exit."

             # Remember that configuration file not read
             # in ok so we don't overwrite the file upon
             # exit, and then kick out of setting the
             # rest of the preferences

             set configFileOk 0
             return
        }

        # A switch is needed because not all settings can
        # be set by simply setting a global variable, such
        # as setting the preference for showing system
        # objects
        # Note that we only have to change the default if
        # a setting is set to 0 in the configuration file

        switch -- $setting {
            "setting_showSystemObjects" {
                if { $value == 0 } {
                    set $setting $value
                    $leftcs.box configure -showSystemObjects $value
                    $rightcs.megaView configure -showSystemObjects $value
                }
            }

            # Since setting_showSystemObjects is the only
            # setting that requires more than just setting a
            # global variable, all other expected settings
            # can be handled with the following default

            default {
                set $setting $value
            }
        }

    } else {

        # Catch an unrecognized preference

        tk_messageBox -icon warning -title "Invalid Configuration File" \
            -message "The configuration file [file join $configDir $preferencesFilename] contains an unrecognized preference ($setting).\nValid preferences are:\n\n[join $expectedPrefs "\n"]\n\nDue to the invalid configuration file, some preference settings may not have been loaded, and the configuration settings will not be saved on exit."

        # Remember that configuration file not read in
        # ok so we don't overwrite the file upon exit,
        # and then kick out of setting the rest of the
        # preferences

       set configFileOk 0
       return
    }
}

