# Copyright (c) 2002 Red Hat, Inc. All rights reserved.
#
# This software may be freely redistributed under the terms of the
# GNU General Public License.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# SQL Pretty-ifierTM 
#
# Written for Red Hat Inc. by Andrew Overholt <overholt@redhat.com>
#
# Component of: Red Hat Database GUI Administration tool

package provide prettyPrint 1.0

set stack {}
set indent 0
set pos 0

proc prettyPrintSELECT { inputString lineLength } {

	# Stack keeps track of "base" indentation levels in sub-queries

	global stack

	# Current indentation level

	global indent

	# Current cursor position

	global pos

	set indent 0
	set newLineFlag 0
	set newLineToken 0
	set pos 0
	set stack {}
	set outputString ""
	set oldToken ""
	set toReturn {}
regsub -all {"} $inputString {\"} inputString
	foreach token [split $inputString] {

		# Determine if we'll spill over our line width with this token

		set newLineFlag [tooLong $outputString $token $lineLength]

		# Will we put a new line with this token anyways?

		set newLineToken [regexp "^FROM$|^WHERE$|^HAVING$|^UNION$|^INTERSECT$|^EXCEPT$|^AND$|^OR$|^LIMIT$|^OFFSET$|^END$|^GROUP$|^WHEN$|^ELSE$|^END$|^ORDER$|^NATURAL$|^LEFT$|^RIGHT$|^FULL$|^CROSS$|^CASE$|^THEN$" $token]

		# Have we just put a new line down?

		set newLinePosition [string last "\n" $outputString]
		set alreadyNewLine [string compare "\n[indentation $indent]" \
			[string range $outputString $newLinePosition end]]

		# Start a new line if (the total length will be > lineLength)
		#                  && (we're not going to do it based on token)
		#		   && (we haven't just put a new line)

		if {$newLineFlag == 1 && 
		    $newLineToken != 1 &&
		    $alreadyNewLine != 0} {
			set outputString "$outputString\n[indentation $indent]"
		}

		# Special "case" needing logic outside of parseToken()

		if {$alreadyNewLine != 0 && $token == "CASE"} {
			set outputString "$outputString\n[indentation $indent]"
		}
		set outputString "$outputString[parseToken $token]"
		set newLinePosition [string last "\n" $outputString]
		set pos [expr [string length $outputString] - $newLinePosition\
		    - 1]
	}

	# Return the parsed SELECT query as a list to comply with other stuff

	set toReturn [wrapLines $outputString 1000000]

	return $toReturn
}


proc prettyPrintRULE { inputString lineLength } {

	# Stack keeps track of "base" indentation levels in sub-queries

	global stack

	# Current indentation level

	global indent

	set indent 7 
	set newLineFlag 0
	set newLineToken 0
	set pos 0
	set stack {}
	set outputString ""
	set oldToken ""
	set toReturn {}

	# Push the base indentation for a CREATE RULE statement

	set stack [push 7]
	foreach token [split $inputString] {
		set postNewLine 0

		# Determine if we'll spill over our line width with this token

		set newLineFlag [tooLong $outputString $token $lineLength]

		# Will we put a new line with this token anyways?

		set newLineToken [regexp "^FROM$|^WHERE$|^HAVING$|^UNION$|^INTERSECT$|^EXCEPT$|^AND$|^OR$|^LIMIT$|^OFFSET$|^END$|,$|^GROUP$|^WHEN$|^ELSE$|^END$|^ORDER$|^NATURAL$|^LEFT$|^RIGHT$|^FULL$|^CROSS$|^CASE$|^THEN$" $token]

		# Do we need to put down a new line now due to an "AS
		# ON" or a "DO INSTEAD" ?

		if {($oldToken == "AS" && $token == "ON")} { 
			set postNewLine 1
		} elseif {$oldToken == "DO" && ($token != "INSTEAD" && $token != "NOTHING")} {
			set outputString "$outputString\n[indentation $indent]"
		} 

		set oldToken $token

		# Have we just put a new line down?

		set newLinePosition [string last "\n" $outputString]
                set alreadyNewLine [string compare "\n[indentation $indent]" \
                        [string range $outputString $newLinePosition end]]

		# Start a new line if (the total length will be > lineLength)
		#                  && (we're not going to do it based on token)
		#		   && (we haven't just put a new line)

                if {$newLineFlag == 1 &&
                    $newLineToken != 1 &&
                    $alreadyNewLine != 0} {
			set outputString "$outputString\n[indentation $indent]"
		}

		# Special "case" needing logic outside of parseToken()

		if {$alreadyNewLine != 0 && $token == "CASE"} {
			set outputString "$outputString\n[indentation $indent]"
		}
		set outputString "$outputString[parseToken $token]"
		set newLinePosition [string last "\n" $outputString]
                set pos [expr [string length $outputString] - $newLinePosition\
                    - 1]

		if {$postNewLine == 1} {
			set outputString "$outputString\n[indentation $indent]"
		}
	}

	# Empty off the base indentation for the CREATE RULE

	pop

	# Return the parsed SELECT query as a list to comply with other stuff

	set toReturn [split $outputString "\n"]
	return $toReturn
}

# notSoPrettyPrintRULEQC:  splits up a RULE qualifying condition so that it is
# visible in the propTable.  It's "un"Pretty because it just linebreaks and
# doesn't do any special parsing.

proc notSoPrettyPrintRULEQC { inputString lineLength } {

	set newLineFlag 0
	set outputString ""
	set toReturn ""

	foreach token [split $inputString] {

	    # Determine if we'll spill over our line width with this token

	    set newLineFlag [tooLong $outputString $token $lineLength]

	    # Start a new line if (the total length will be > lineLength)

	    if { $newLineFlag == 1 } {
		set outputString "$outputString\n$token"
	    } else {
	    	set outputString "$outputString$token"
	    }
	}

	set toReturn [split $outputString "\n"]
	return $toReturn
}

proc parseToken { token } {

	global stack
	global indent
	global pos

	set outputString ""
	switch -regexp -- $token {
		"\\(" {

			# Push the current base indentation level

			set stack [push $indent]
			if {$pos > [peek]} {
				set indent [expr $indent + [string first "\(" \
				    $token] + [expr $pos - [peek]] + 1]
			} else {
				set indent [expr $indent + [string first "\(" \
				    $token] + 1]
			}

			set stack [push $indent]

			# Strip up to the first parenthesis

			set heldToken [string range $token 0 [string first \
			    "\(" $token]]

			# Set the remaining token

			set restOfToken [string range $token [expr [string \
			    first "\(" $token] + 1] "end"]

			# Parse the stripped token

			set outputString "$heldToken[parseToken $restOfToken]"
		} "\\)|\\);" {
			set numClosingParentheses 0
			set numOpeningParentheses 0

			# Determine how many "net" close brackets we have

			for {set i 0} {$i < [string length $token]} {incr i} {
				if {[string index $token $i] == "\)"} {
					incr numClosingParentheses
				} elseif {[string index $token $i] == "\("} {
					incr numOpeningParentheses
				}
			}

			# Reset the indentation level with each close bracket

			if {$numOpeningParentheses < $numClosingParentheses} {
				for {set i 0} {$i < [expr \
				    $numClosingParentheses - \
				    $numOpeningParentheses]} {incr i} {
					pop
					set indent [pop]
				}
			}

			# If the token happens to end in a comma, put a \n

			if {[string index $token [expr [string length $token] \
			    - 1]] == ","} {
				set outputString "$token\n[indentation $indent]"

			# Otherwise, just put the token itself

			} else {
				set outputString "$token "
			}
		} "^SELECT$" { 
                        set outputString "SELECT "

			# Set indentation to base indentation + length(SELECT)

                        set indent [expr 7 + $indent]
		} "^FROM$|^WHERE$|^HAVING$|^UNION$|^INTERSECT$|^EXCEPT$" {

			# Get current base indentation level

			set indent [peek]
			set outputString "\n[indentation $indent]$token "

			# Set indentation to base indentation + length(token)

			set indent [expr [expr [string length $token] + 1] + \
			    [peek]]
		} "^ORDER$|^GROUP$" {

			# Get current base indentation level

			set indent [peek]
			set outputString "\n[indentation $indent]$token "

			# Set indentation to base indentation + length(token BY)

			set indent [expr 9 + [peek]]
		} "^WHEN$|^ELSE$" {

			# Increase the indentation level (in case we spill over
			# a line)

			set outputString "\n[indentation $indent]$token "
		} "^THEN$" {

			# Decrease the indentation level again

			set outputString "\n[indentation $indent]$token "
		} "^END$" {
			set indent [expr $indent - 5]
			set outputString "\n[indentation $indent]$token "
		} "^JOIN$|^INNER$|^OUTER$" {
			set outputString "$token "
		} "^LEFT$|^RIGHT$|^FULL$|^CROSS$|^NATURAL$" {
			set indent [peek]
			set outputString "\n[indentation $indent]$token "
			set indent [expr $indent + [string length $token] + 1]
		} ",$" {
			set outputString "$token\n[indentation $indent]"
		} "^CASE$" {
			set indent [expr $indent + 5]
			set outputString "$token "
		} "^AS$|^ON$|^ALL$|^ASC$|^DESC$|^USING$|^DISTINCT$|^BY$" {
			set outputString "$token "
		} "^AND$|^OR$|^LIMIT$|^OFFSET$" {

			# Decrease indentation so sub-AND conditions line up

			set indent [expr $indent - [expr [string length \
			    $token] + 1]]

			# Begin new line with indentation

			set outputString "\n[indentation $indent]$token "

			# Reset indentation after sub-AND conditions

			set indent [expr $indent + [expr [string length \
			    $token] + 1]]
		} default {
			set outputString "$token "
		}
	}
	return $outputString
}

# Just peek at what's on the stack

proc peek { } {

	global stack

	if {[llength $stack] > 0} {
		set toreturn [lindex $stack 0]
	} else {
		set toreturn 0
	}
	return $toreturn
}

# Put the indentation in with spaces

proc indentation { indent } {
	set outputString ""
        for {set i 0} {$i < $indent} {incr i} {
                set outputString "$outputString "
        }
        return $outputString
}

# Push a value onto the stack

proc push { value } {

	global stack

        set stack [linsert $stack 0 $value]
        return $stack
}

# Wrap a list of lines.  These two functions are necessary only for lines 
# where the only token on the line falls on the lineLength or where the token
# is > lineLength itself

proc wrapLines { inLines lineLength } {
    set toReturn ""

    set lines [split $inLines "\n"]

    for { set i 0 } { $i < [llength $lines] } { incr i } {
	set line [lindex $lines $i]
	
	set leftOvers $line
	set newLines ""

	while { [string length $leftOvers] > $lineLength } { 
	    if { [string length $leftOvers] > $lineLength } {
		lappend newLines [string range $leftOvers 0 $lineLength]
		set leftOvers [string range $leftOvers [expr $lineLength + 1] end]
	    } 
	}
	

	lappend newLines $leftOvers
	    
	eval lappend toReturn $newLines
    }

    return $toReturn
}

# Pop a value from the stack - return 0 if nothing on the stack

proc pop { } {

	global stack

	if {[llength $stack] > 0} {
        	set toreturn [lindex $stack 0]
        	set stack [lreplace $stack 0 0]
	} else {
		set toreturn 0
	}
        return $toreturn
}

# Determine whether or not the linewidth will spill over lineLength

proc tooLong { outputString token lineLength } {

	global indent
	global pos

	if {[expr [string length $outputString] - \
	    [string last "\n" $outputString] + \
	    [string length $token] - 1] > $lineLength} {
		return 1
	}
	return 0
}
