# Copyright (c) 2002 Red Hat, Inc. All rights reserved.
#
# This software may be freely redistributed under the terms of the
# GNU General Public License.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# Modify the privileges of an object
#
# Written for Red Hat Inc. by Andrew Overholt <overholt@redhat.com>
#
# Component of:  Red Hat Database GUI Administration tool

package require Tk
package require Itcl
package require Itk
package require Iwidgets
package require cluster
package require databaseHandle
package require tableHandle
package require viewHandle
package require sequenceHandle
package require userHandle
package require groupHandle
package provide privileges 1.0

proc modifyPrivileges { args } {

    global clusterObject
    global databaseHandle
    global objHandle
    global numAffectedUsers

    set expectedArgs \
	[list \
	    -clusterCollectionName \
	    -clusterName \
	    -databaseName \
	    -objectList \
	    -newPrivileges \
	    -affectedUsers \
	    -numAffectedUsers
	]

    foreach option $expectedArgs {
	set options($option) ""
    }

    foreach { option value } $args {
	if {[lsearch -exact $expectedArgs $option] == -1} {
	    error "Unexpected option \"$option\" - only expect $expectedArgs"
	}
	set options($option) "$value"
    }

    set grantList {}
    set revokeList {}
    set currentPrivileges {}
    set toReturn ""

    set newPrivileges [split $options(-newPrivileges)]
    set numAffectedUsers $options(-numAffectedUsers)

    # Set up handles

    set clusterObject [$options(-clusterCollectionName) getClusterObject $options(-clusterName)]
    set databaseHandle [$clusterObject getDatabaseHandle $options(-databaseName)]

    # For each object we're changing.  This will be >= 1 in the case of
    # changing privileges for multiple users/groups and 1 in the case of
    # changing privileges for tables/views/sequences.

    foreach {objType objName} $options(-objectList) {

	# Determine and set up for which type of object we're modifying
	# the privileges.

	set objName [string trim $objName {"}]
	switch $objType {
	    "sequence" {
		set objHandle [$databaseHandle getSequenceHandle "$objName"]
	    } "table" {
		set objHandle [$databaseHandle getTableHandle "$objName"]
	    } "view" {
		set objHandle [$databaseHandle getViewHandle "$objName"]
	    }
	}

	# Parse the list of affected users.  This is necessary because
	# with PostgreSQL 7.1 it is not possible to alter privileges
	# for multiple users in one SQL statement but with >= 7.2 it is
	# possible.

	set affectedUsers [parseUsers $options(-affectedUsers)]

	# For each of the affected users, change the privileges on the
	# object.

	foreach affectedUser $affectedUsers {

	    # If there is more than one user or group, we don't
	    # care what the current privileges are, we just want to
	    # grant ones given and revoke all the rest.  Otherwise,
	    # we want to grant and revoke based upon current
	    # privileges.

	    set returned [constructGrantRevokeLists $affectedUser \
		$numAffectedUsers $newPrivileges $objHandle]

            # Do the action iff there is something to be done.

            if {[llength [lindex $returned 0]] != 0 || [llength [lindex $returned end]] != 0 } {
                set errorText [$objHandle setPrivileges $affectedUser $returned]
                if {$errorText != ""} {
		    return $errorText
	        }
            }
	}

	itcl::delete object $objHandle

    }

    itcl::delete object $databaseHandle
    itcl::delete object $clusterObject

    return
}

proc checkForPrivilegeChanges { args } {

    global clusterObject
    global databaseHandle
    global objHandle
    global numAffectedUsers

    set expectedArgs \
	[list \
	    -clusterCollectionName \
	    -clusterName \
	    -databaseName \
	    -objectList \
	    -newPrivileges \
	    -affectedUsers \
	    -numAffectedUsers
	]

    foreach option $expectedArgs {
	set options($option) ""
    }

    foreach { option value } $args {
	if {[lsearch -exact $expectedArgs $option] == -1} {
	    error "Unexpected option \"$option\" - only expect $expectedArgs"
	}
	set options($option) "$value"
    }

    set grantList {}
    set revokeList {}
    set currentPrivileges {}
    set toReturn ""

    set newPrivileges [split $options(-newPrivileges)]
    set numAffectedUsers $options(-numAffectedUsers)

    # Set up handles

    set clusterObject [$options(-clusterCollectionName) getClusterObject $options(-clusterName)]
    set databaseHandle [$clusterObject getDatabaseHandle $options(-databaseName)]

    # For each object we're changing.  This will be >= 1 in the case of
    # changing privileges for multiple users/groups and 1 in the case of
    # changing privileges for tables/views/sequences.

    foreach {objType objName} $options(-objectList) {

	# Determine and set up for which type of object we're modifying
	# the privileges.

	set objName [string trim $objName {"}]
	switch $objType {
	    "sequence" {
		set objHandle [$databaseHandle getSequenceHandle "$objName"]
	    } "table" {
		set objHandle [$databaseHandle getTableHandle "$objName"]
	    } "view" {
		set objHandle [$databaseHandle getViewHandle "$objName"]
	    }
	}

	# Parse the list of affected users.  This is necessary because
	# with PostgreSQL 7.1 it is not possible to alter privileges
	# for multiple users in one SQL statement but with >= 7.2 it is
	# possible.

	set affectedUsers [parseUsers $options(-affectedUsers)]

	# For each of the affected users, change the privileges on the
	# object.

	foreach affectedUser $affectedUsers {

	    # If there is more than one user or group, we don't
	    # care what the current privileges are, we just want to
	    # grant ones given and revoke all the rest.  Otherwise,
	    # we want to grant and revoke based upon current
	    # privileges.

	    set returned [constructGrantRevokeLists $affectedUser \
		$numAffectedUsers $newPrivileges $objHandle]
            if {[llength [lindex $returned 0]] != 0 || [llength [lindex $returned end]] != 0 } {
               return 1
            }

	}

	itcl::delete object $objHandle

    }

    itcl::delete object $databaseHandle
    itcl::delete object $clusterObject

    return 0
}

proc constructGrantRevokeLists { affectedUser numAffectedUsers newPrivileges objHandle } {

    global clusterObject

    set revokeList {}
    set grantList {}
    set currentPrivileges {}

    # If there is more than one affected user, we don't care what the
    # current privileges are.

    if {$numAffectedUsers == 1} {

	# If the affected "user" is a group

	if {[regexp -- "GROUP" $affectedUser] == 1} {
	    set groupPrivileges [parsePrivileges [string trim [$objHandle getPrivileges] "{}"] "g"]
	    set affectedGroup [string range $affectedUser [string first \" \
		$affectedUser] [expr [string length $affectedUser] - 1]]
	    set affectedUser "GROUP \"$affectedGroup\""

	    # If there are group privileges set

	    if {$groupPrivileges > 0} {
		set groups [lindex $groupPrivileges 0]
		set group_privs(0) {}
		set counter 0
		foreach temp_group $groups {
		    set group_privs($temp_group) [string trim [lindex [lindex \
			$groupPrivileges 1] $counter] "{}"]
		    incr counter
		}

		# If _this_ group has privileges set

		set trimmedAffectedGroup [string trim $affectedGroup {"}]
		if {[lsearch -exact $groups $trimmedAffectedGroup] != -1} {
		    set currentPrivileges $group_privs($trimmedAffectedGroup)
		}
	    }

	# If the affected "user" is `public'

	} elseif {$affectedUser == "PUBLIC"} {
	    set currentPrivileges [parsePrivileges [string trim [$objHandle getPrivileges] "{}"] "p"]

	# Otherwise, it's an actual user

	} else {
	    set userPrivileges [parsePrivileges [$objHandle getPrivileges] "u"]

	    # If there are user privileges set

	    if {$userPrivileges > 0} {
		set users [lindex $userPrivileges 0]
		set user_privs(0) {}
		set counter 0
		foreach temp_user $users {
		    set user_privs($temp_user) [string trim [lindex [lindex \
			$userPrivileges 1] $counter] "{}"]
		    incr counter
		}

		# If _this_ user has privileges set
		
		set trimmedAffectedUser [string trim $affectedUser {"}]
		if {[lsearch -exact $users $trimmedAffectedUser] != -1} {
		    set currentPrivileges $user_privs($trimmedAffectedUser)
		}
	    }
	}
    }


    # Set up list of privileges to be granted

    foreach privilege $newPrivileges {
	if {[lsearch -exact $currentPrivileges $privilege] == -1} {
	    lappend grantList $privilege
	}
    }

    # Set up list of privileges to be revoked

    foreach privilege $currentPrivileges {
	if {[lsearch -exact $newPrivileges $privilege] == -1} {
	    lappend revokeList $privilege
	}
    }

    if {[$clusterObject getBackendVersionSeries] >= 7.2} {
	set allPossiblePrivileges [list SELECT UPDATE INSERT DELETE RULE REFERENCES TRIGGER]
    } else {
	set allPossiblePrivileges [list SELECT UPDATE INSERT DELETE RULE]
    }

    # This last check of what privileges need to be revoked is necessary for
    # looping over multiple users in a higher level function

    if {$numAffectedUsers != 1} {
	foreach privilege $allPossiblePrivileges {

	    # If we should be revoking a privilege and it hasn't
	    # already been added to the revoke list

	    if {[lsearch -exact $newPrivileges $privilege] == -1 &&
		[lsearch -exact $revokeList $privilege] == -1 &&
		[lsearch -exact $currentPrivileges $privilege] == -1} {
		lappend revokeList $privilege
	    }
	}	
    }

    set grantList [join $grantList ","]
    set revokeList [join $revokeList ","]

    return [list $grantList $revokeList]
}

# Parse the list of affected users.  This is necessary because with PostgreSQL
# 7.1 it is not possible to alter privileges for multiple users in one SQL
# statement but with >= 7.2 it _is_ possible.

proc parseUsers { affectedUsers } {

    global clusterObject

    set tempUsers ""
    set parsedUsers {} 

    if {[$clusterObject getBackendVersionSeries] >= 7.2} {
	for {set i 0} {$i < [llength $affectedUsers]} {incr i} {
	    set affectedUser [lindex $affectedUsers $i]
	    if {[regexp -- "GROUP" $affectedUser] == 1} {
		set tempUsers "$tempUsers GROUP\
		    \"[join [lrange $affectedUser 1 end]]\","
	    } else {
		set tempUsers "$tempUsers $affectedUser,"
	    }
	}
	set tempUsers [string trim $tempUsers]
	set parsedUsers [linsert $parsedUsers 0 [string trim $tempUsers {,}]]
    } else {
	set tempUsers {}
	for {set i 0} {$i < [llength $affectedUsers]} {incr i} {
	    set affectedUser [lindex $affectedUsers $i]
	    if {[regexp -- "GROUP" $affectedUser] == 1} {
		lappend tempUsers \
		    "GROUP \"[join [lrange $affectedUser 1 end]]\""
	    } else {
		lappend tempUsers "$affectedUser"
	    }
	}
	set parsedUsers $tempUsers
    }
    return $parsedUsers
}

proc parsePrivileges { privilegeString desiredPrivilegeLevel } {

    set groups {}
    set users {}
    set publicPrivilegeList "" 
    set groupPrivilegeList {}
    set userPrivilegeList {}

    set privilegeList [split $privilegeString ","]

    for {set i 0} {$i < [llength $privilegeList]} {incr i} {

	# Break up privileges into public, group, and users.  Also, split group
	# and user privileges into two lists of names and privilege strings

	set tempList [split [string trim [lindex $privilegeList $i] {"}] "="]

	switch -regexp -- [lindex $tempList 0] {
	    "^$" { 
		set publicPrivilegeList [lindex $tempList 1] 
	    } "group " { 
	        set gname [lrange [lindex $tempList 0] 1 "end"]
		lappend groups "$gname"
		lappend groupPrivilegeList [lindex $tempList 1]
	    } default { 
		lappend users "[lindex $tempList 0]"
		lappend userPrivilegeList [lindex $tempList 1]
	    }
	}
    }

    set userPrivilegeLists {} 
    set groupPrivilegeLists {} 
    set publicPrivilegeLists {}

    # Set all user privileges initially to 0

    for {set i 0} {$i < [llength $users]} {incr i} {
	lappend userPrivilegeLists {}
    }

    # Create user permission lists and insert them into the lists

    for {set i 0} {$i < [llength $users]} {incr i} {
	set thisUserPrivilegeList {}
	set thisUserPrivilegeList [parsePrivilegeString \
	    [lindex $userPrivilegeList $i]]
	set userPrivilegeLists [lreplace $userPrivilegeLists $i $i \
	    $thisUserPrivilegeList]
    }

    # Set all group privileges initially to 0

    for {set i 0} {$i < [llength $groups]} {incr i} {
	lappend groupPrivilegeLists {} 
    }

    # Create group permission lists and insert them into the lists

    for {set i 0} {$i < [llength $groups]} {incr i} {
	set thisGroupPrivilegeList {}
	set thisGroupPrivilegeList [parsePrivilegeString \
	    [lindex $groupPrivilegeList $i]]
	set groupPrivilegeLists [lreplace $groupPrivilegeLists $i $i \
	    $thisGroupPrivilegeList]
    }

    # Set public privileges initially to 0

    set publicPrivilegeLists {}

    # Create public permission list

    set publicPrivilegeLists [parsePrivilegeString $publicPrivilegeList]

     switch -- $desiredPrivilegeLevel {
	"u" {
	     return [list $users $userPrivilegeLists]
	} "g" {
	     return [list $groups $groupPrivilegeLists]
	} "p" {
	     return $publicPrivilegeLists
	}
     }
}

# Helper function to parse PostgreSQL privilege strings

proc parsePrivilegeString { privilegeString } {

    set tempList {}

    # Order: SELECT UPDATE INSERT DELETE RULE REFERENCES TRIGGER 
    #          r      w      a      d     R       x         t

    for {set j 0} {$j < [string length $privilegeString]} {incr j} {
	switch -- [string index $privilegeString $j] {
	    "r" {
		lappend tempList "SELECT"
	    } "w" {
		lappend tempList "UPDATE"
	    } "a" {
		lappend tempList "INSERT"
	    } "d" {
		lappend tempList "DELETE"
	    } "R" {
		lappend tempList "RULE"
	    } "x" {
		lappend tempList "REFERENCES"
	    } "t" {
		lappend tempList "TRIGGER"
	    }
	}
    }

    return $tempList
}

# End of file
