///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "basis_symbolic.h"
using namespace std;
using namespace rheolef;
using namespace GiNaC;

basis_symbolic_nodal_on_geo::value_type 
basis_symbolic_nodal_on_geo::eval (
    const polynom_type&    p, 
    const basic_point<ex>& xi, 
    size_type              d) const
{
    ex expr = p;
    if (d > 0) expr = expr.subs(x == ex(xi[0]));
    if (d > 1) expr = expr.subs(y == ex(xi[1]));
    if (d > 2) expr = expr.subs(z == ex(xi[2]));
    return expr;
}
matrix
basis_symbolic_nodal_on_geo::vandermonde_matrix (
    const vector<ex>&          p,
    size_type                  d) const
{
    unsigned int n = _node.size();
    matrix   vdm (n, n);

    for (unsigned int i = 0; i < n; i++) {
    
        const basic_point<ex>& xi = _node[i];
        for (unsigned int j = 0; j < n; j++) {
          vdm(i,j) = eval (p[j], xi, d);
        }
    }
    return vdm;
}

/* 
 * Build the Lagrange basis, associated to nodes.
 *
 * input: nodes[n] and polynomial_basis[n]
 * output: node_basis[n]
 *   such that node_basis[j](node[i]) = kronecker[i][j]
 *
 * algorithm: let:
 *   b_i = \sum_i a_{i,j} p_j
 *  where:
 *       p_j = polynom"ial basis [1, x, y, ..]
 *       b_i = basis associated to node :
 *  we want:
 *   b_i(x_k) = delta_{i,k} 
 *   <=>
 *   a_{i,j} p_j(x_k) = \delta_{i,k}
 * Let A = (a_{k,j})_{i,j} and c_{i,j} = (p_j(x_i))
 * Then a_{i,j} c_{k,j} = delta_{i,k}
 *        <=>
 *    A = C^{-T}
 */

void
basis_symbolic_nodal_on_geo::make_node_basis()
{
  assert_macro (_node.size() == _poly.size(),
	"incompatible node set size (" << _node.size()
	<< ") and polynomial basis size (" << _poly.size() << ").");

  const size_type d = _hat_K.dimension();
  const size_type n = size();

#ifdef TO_CLEAN
  warning_macro ("node.size = " << _node.size());
  for (size_type i = 0; i < _node.size(); i++) {
      cerr << "node("<<i<<") = " << _node[i] << endl;
  }
  warning_macro ("poly.size = " << _poly.size());
  for (size_type i = 0; i < _poly.size(); i++) {
      cerr << "poly("<<i<<") = " << _poly[i] << endl;
  }
#endif // TO_CLEAN
  // Vandermonde matrix vdm(i,j) = pj(xi)
  matrix vdm = vandermonde_matrix (_poly, d);
  ex det_vdm = determinant(vdm);
  check_macro (det_vdm != 0, "basis unisolvence failed on element `" 
		  << _hat_K.name() << "'");
  matrix inv_vdm = vdm.inverse();
 
  // basis := trans(a)*poly
  _basis.resize(n);
  for (size_type i = 0; i < n; i++) {
    polynom_type s = 0;
    for (size_type j = 0; j < n; j++) {
      s += inv_vdm(j,i) * _poly[j];
    }
    s = expand(s);
    s = normal(s);
    _basis[i] = s;
  }
  warning_macro ("basis.size = " << _basis.size());
  for (size_type i = 0; i < _basis.size(); i++) {
      cerr << "basis("<<i<<") = " << _basis[i] << endl;
  }
  // check:
  matrix vdm_l = vandermonde_matrix (_basis, d);
  int ndigit10 = Digits;

  numeric tol = ex_to<numeric>(pow(10.,-ndigit10/2.));
  int status = 0;
  for (size_type i = 0; i < n; i++) {
    for (size_type j = 0; j < n; j++) {
      if ((i == j && abs(vdm_l(i,j) - 1) > tol) ||
          (i != j && abs(vdm_l(i,j))     > tol)    ) {
  	  error_macro ("Lagrange polynom check failed.");
      }
    }
  }
  // derivatives of the basis
  _grad_basis.resize(n);
  for (size_type i = 0; i < n; i++) {
    if (d > 0) _grad_basis [i][0] = _basis[i].diff(x).expand().normal();
    if (d > 1) _grad_basis [i][1] = _basis[i].diff(y).expand().normal();
    if (d > 2) _grad_basis [i][2] = _basis[i].diff(z).expand().normal();
  }
  warning_macro ("grad_basis.size = " << _grad_basis.size());
  for (size_type i = 0; i < _basis.size(); i++) {
      cerr << "grad_basis("<<i<<") = [";
      for (size_type j = 0; j < d; j++) {
          cerr << _grad_basis [i][j];
	  if (j == d-1) cerr << "]" << endl;
	  else cerr << ", ";
      }
  }
}
